/*
 *  Copyright (C) 2022-2024  The DOSBox Staging Team
 *  Copyright (C) 2002-2021  The DOSBox Team
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "int10.h"

#include "mem.h"
#include "inout.h"

// clang-format off
static const std::vector<uint8_t> static_functionality = {
    /* 0 */ 0xff,  // All modes supported #1
    /* 1 */ 0xff,  // All modes supported #2
    /* 2 */ 0x0f,  // All modes supported #3
    /* 3 */ 0x00, 0x00, 0x00, 0x00,  // Reserved
    /* 7 */ 0x07,  // 200, 350, 400 scan lines
    /* 8 */ 0x04,  // Total number of character blocks available in text modes
    /* 9 */ 0x02,  // Maximum number of active character blocks in text modes
    /* a */ 0xff,  // Misc Flags Everthing supported
    /* b */ 0x0e,  // Support for Display combination, intensity/blinking and video state saving/restoring
    /* c */ 0x00,  // Reserved
    /* d */ 0x00,  // Reserved
    /* e */ 0x00,  // Change to add new functions
    /* f */ 0x00   // Reserved
};

static const std::vector<uint16_t> map_offset = {
	0x0000, 0x4000, 0x8000, 0xc000,
	0x2000, 0x6000, 0xa000, 0xe000
};
// clang-format on

static void load_font(const PhysPt _font_data, const bool reload, const int num_chars,
                      const int first_char, const int _font_block,
                      const int char_height, const bool load_alternate_chars)
{
	constexpr auto BytesPerChar = 32;

	// Valid font block value range: EGA: 0-3; VGA: 0-7
	//
	// Source:
	// http://www.techhelpmanual.com/157-int_10h_1110h__load_and_activate_user_defined_font.html
	const auto font_block = (_font_block & 0x07);

	const PhysPt font_base_addr = PhysicalMake(
	        0xa000,
	        map_offset[font_block] + (uint16_t)(first_char * BytesPerChar));

	const uint16_t crtc_base = real_readw(BIOSMEM_SEG, BIOSMEM_CRTC_ADDRESS);

	// Put video adapter in planar mode

	// select plane 2 for writing
	IO_Write(0x3c4, 0x02);
	IO_Write(0x3c5, 0x04);

	// odd/even off in SEQ
	IO_Write(0x3c4, 0x04);
	IO_Write(0x3c5, 0x07);

	// select plane 2 for reading
	IO_Write(0x3ce, 0x04);
	IO_Write(0x3cf, 0x02);

	// write mode 0, odd/even off in GFX
	IO_Write(0x3ce, 0x05);
	IO_Write(0x3cf, 0x00);

	// CPU memory window A0000-AFFFF
	IO_Write(0x3ce, 0x06);
	IO_Write(0x3cf, 0x04);

	// Load character patterns
	auto font_data = _font_data;

	for (auto chr = 0; chr < num_chars; ++chr) {
		MEM_BlockCopy(font_base_addr + chr * BytesPerChar, font_data, char_height);
		font_data += char_height;
	}

	// Load alternate 9x14 or 9x16 character patterns on VGA
	if (load_alternate_chars) {
		while (auto chr = mem_readb(font_data++)) {
			MEM_BlockCopy(font_base_addr + chr * BytesPerChar,
			              font_data,
			              char_height);

			font_data += char_height;
		}
	}

	// Return to normal text mode

	// select planes 0&1 for writing
	IO_Write(0x3c4, 0x02);
	IO_Write(0x3c5, 0x03);

	// odd/even on in SEQ
	IO_Write(0x3c4, 0x04);
	IO_Write(0x3c5, 0x03);

	// select plane 0 for reading
	IO_Write(0x3ce, 0x04);
	IO_Write(0x3cf, 0x00);

	// write mode 0, odd/even on in GFX
	IO_Write(0x3ce, 0x05);
	IO_Write(0x3cf, 0x10);

	// Bx000-BxFFF, odd/even on
	IO_Write(0x3ce, 0x06);

	const bool monochrome = (crtc_base == VGAREG_MDA_CRTC_ADDRESS);
	IO_Write(0x3cf, monochrome ? 0x0a : 0x0e);

	// Reload tables and registers with new values based on this height
	if (reload) {
		// Max scanline
		IO_Write(crtc_base, 0x9);
		IO_Write(crtc_base + 1,
		         (IO_Read(crtc_base + 1) & 0xe0) | (char_height - 1));

		// Vertical display end
		const auto rows  = CurMode->sheight / char_height;
		const auto vdend = rows * char_height *
		                           ((CurMode->sheight == 200) ? 2 : 1) -
		                   1;

		IO_Write(crtc_base, 0x12);
		IO_Write(crtc_base + 1, (uint8_t)vdend);

		// Underline location
		if (CurMode->mode == 7) {
			IO_Write(crtc_base, 0x14);
			IO_Write(crtc_base + 1,
			         (IO_Read(crtc_base + 1) & ~0x1f) |
			                 (char_height - 1));
		}

		// Rows setting in BIOS segment
		real_writeb(BIOSMEM_SEG, BIOSMEM_NB_ROWS, rows - 1);
		real_writeb(BIOSMEM_SEG, BIOSMEM_CHAR_HEIGHT, (uint8_t)char_height);

		// Page size
		const int bios_pagesize = rows *
		                          real_readb(BIOSMEM_SEG, BIOSMEM_NB_COLS) *
		                          2;

		// BIOS adds extra on reload
		real_writew(BIOSMEM_SEG, BIOSMEM_PAGE_SIZE, bios_pagesize + 0x100);

		// Move up one line on 14+ line fonts
		auto cursor_height = char_height >= 14 ? (char_height - 1)
		                                       : char_height;

		INT10_SetCursorShape(cursor_height - 2, cursor_height - 1);
	}
}

void INT10_LoadFont(const PhysPt font_data, const bool reload,
                    const int num_chars, const int first_char,
                    const int font_block, const int char_height)
{
	constexpr auto LoadAlternateChars = false;
	load_font(font_data, reload, num_chars, first_char, font_block, char_height, LoadAlternateChars);
}

void INT10_ReloadFont()
{
	constexpr auto Reload    = false;
	constexpr auto NumChars  = 256;
	constexpr auto FirstChar = 0;
	constexpr auto FontBlock = 0;

	const auto char_height = CurMode->cheight;

	const auto font_data = RealToPhysical(
		char_height == 8  ? int10.rom.font_8_first :
		char_height == 14 ? int10.rom.font_14 :
		                    int10.rom.font_16
	);

	const auto is_vga_9dot_font = (IS_VGA_ARCH &&
	                               !vga.seq.clocking_mode.is_eight_dot_mode);

	const auto load_alternate_chars = (is_vga_9dot_font && char_height != 8);

	load_font(font_data, Reload, NumChars, FirstChar, FontBlock, char_height, load_alternate_chars);
}

void INT10_SetupRomMemory(void) {
/* This should fill up certain structures inside the Video Bios Rom Area */
	PhysPt rom_base=PhysicalMake(0xc000,0);
	Bitu i;
	int10.rom.used=3;
	if (IS_EGAVGA_ARCH) {
		// set up the start of the ROM
		phys_writew(rom_base+0,0xaa55);
		phys_writeb(rom_base+2,0x40);		// Size of ROM: 64 512-blocks = 32KB
		phys_writeb(rom_base+0x1e,0x49);	// IBM string
		phys_writeb(rom_base+0x1f,0x42);
		phys_writeb(rom_base+0x20,0x4d);
		phys_writeb(rom_base+0x21,0x20);

		if (IS_VGA_ARCH) {
			// SVGA card-specific ROM signatures
			switch (svgaCard) {
			case SVGA_S3Trio:
				phys_writeb(rom_base+0x003f,'S');
				phys_writeb(rom_base+0x0040,'3');
				phys_writeb(rom_base+0x0041,' ');
				phys_writeb(rom_base+0x0042,'8');
				phys_writeb(rom_base+0x0043,'6');
				phys_writeb(rom_base+0x0044,'C');
				phys_writeb(rom_base+0x0045,'7');
				phys_writeb(rom_base+0x0046,'6');
				phys_writeb(rom_base+0x0047,'4');
				break;
			case SVGA_TsengET4K:
			case SVGA_TsengET3K:
				phys_writeb(rom_base+0x0075,' ');
				phys_writeb(rom_base+0x0076,'T');
				phys_writeb(rom_base+0x0077,'s');
				phys_writeb(rom_base+0x0078,'e');
				phys_writeb(rom_base+0x0079,'n');
				phys_writeb(rom_base+0x007a,'g');
				phys_writeb(rom_base+0x007b,' ');
				break;
			case SVGA_ParadisePVGA1A:
				phys_writeb(rom_base+0x0048,' ');
				phys_writeb(rom_base+0x0049,'W');
				phys_writeb(rom_base+0x004a,'E');
				phys_writeb(rom_base+0x004b,'S');
				phys_writeb(rom_base+0x004c,'T');
				phys_writeb(rom_base+0x004d,'E');
				phys_writeb(rom_base+0x004e,'R');
				phys_writeb(rom_base+0x004f,'N');
				phys_writeb(rom_base+0x0050,' ');
				phys_writeb(rom_base+0x007d,'V');
				phys_writeb(rom_base+0x007e,'G');
				phys_writeb(rom_base+0x007f,'A');
				phys_writeb(rom_base+0x0080,'=');
				break;
			case SVGA_None:
				LOG_ERR("INT10: Invalid VGA card with the VGA machine type");
				break;
			}
		}
		int10.rom.used=0x100;
	}

	if (IS_VGA_ARCH && svgaCard==SVGA_S3Trio) INT10_SetupVESA();

	int10.rom.font_8_first=RealMake(0xC000,int10.rom.used);
	for (i=0;i<128*8;i++) {
		phys_writeb(rom_base+int10.rom.used++,int10_font_08[i]);
	}
	int10.rom.font_8_second=RealMake(0xC000,int10.rom.used);
	for (i=0;i<128*8;i++) {
		phys_writeb(rom_base+int10.rom.used++,int10_font_08[i+128*8]);
	}
	int10.rom.font_14=RealMake(0xC000,int10.rom.used);
	for (i=0;i<256*14;i++) {
		phys_writeb(rom_base+int10.rom.used++,int10_font_14[i]);
	}
	int10.rom.font_14_alternate=RealMake(0xC000,int10.rom.used);
	for (i=0;i<20*15+1;i++) {
		phys_writeb(rom_base+int10.rom.used++,int10_font_14_alternate[i]);
	}
	int10.rom.font_16=RealMake(0xC000,int10.rom.used);
	for (i=0;i<256*16;i++) {
		phys_writeb(rom_base+int10.rom.used++,int10_font_16[i]);
	}
	int10.rom.font_16_alternate=RealMake(0xC000,int10.rom.used);
	for (i=0;i<19*17+1;i++) {
		phys_writeb(rom_base+int10.rom.used++,int10_font_16_alternate[i]);
	}
	int10.rom.static_state=RealMake(0xC000,int10.rom.used);
	for (i=0;i<0x10;i++) {
		phys_writeb(rom_base+int10.rom.used++,static_functionality[i]);
	}
	for (i=0;i<128*8;i++) {
		phys_writeb(PhysicalMake(0xf000,0xfa6e)+i,int10_font_08[i]);
	}
	RealSetVec(0x1F,int10.rom.font_8_second);

	if (IS_EGAVGA_ARCH) {
		int10.rom.video_parameter_table=RealMake(0xC000,int10.rom.used);
		int10.rom.used+=INT10_SetupVideoParameterTable(rom_base+int10.rom.used);

		if (IS_VGA_ARCH) {
			int10.rom.video_dcc_table=RealMake(0xC000,int10.rom.used);
			phys_writeb(rom_base+int10.rom.used++,0x10);	// number of entries
			phys_writeb(rom_base+int10.rom.used++,1);		// version number
			phys_writeb(rom_base+int10.rom.used++,8);		// maximum display code
			phys_writeb(rom_base+int10.rom.used++,0);		// reserved
			// display combination codes
			phys_writew(rom_base+int10.rom.used,0x0000);	int10.rom.used+=2;
			phys_writew(rom_base+int10.rom.used,0x0100);	int10.rom.used+=2;
			phys_writew(rom_base+int10.rom.used,0x0200);	int10.rom.used+=2;
			phys_writew(rom_base+int10.rom.used,0x0102);	int10.rom.used+=2;
			phys_writew(rom_base+int10.rom.used,0x0400);	int10.rom.used+=2;
			phys_writew(rom_base+int10.rom.used,0x0104);	int10.rom.used+=2;
			phys_writew(rom_base+int10.rom.used,0x0500);	int10.rom.used+=2;
			phys_writew(rom_base+int10.rom.used,0x0502);	int10.rom.used+=2;
			phys_writew(rom_base+int10.rom.used,0x0600);	int10.rom.used+=2;
			phys_writew(rom_base+int10.rom.used,0x0601);	int10.rom.used+=2;
			phys_writew(rom_base+int10.rom.used,0x0605);	int10.rom.used+=2;
			phys_writew(rom_base+int10.rom.used,0x0800);	int10.rom.used+=2;
			phys_writew(rom_base+int10.rom.used,0x0801);	int10.rom.used+=2;
			phys_writew(rom_base+int10.rom.used,0x0700);	int10.rom.used+=2;
			phys_writew(rom_base+int10.rom.used,0x0702);	int10.rom.used+=2;
			phys_writew(rom_base+int10.rom.used,0x0706);	int10.rom.used+=2;

			int10.rom.video_save_pointer_table=RealMake(0xC000,int10.rom.used);
			phys_writew(rom_base+int10.rom.used,0x1a);	// length of table
			int10.rom.used+=2;
			phys_writed(rom_base+int10.rom.used,int10.rom.video_dcc_table);
			int10.rom.used+=4;
			phys_writed(rom_base+int10.rom.used,0);		// alphanumeric charset override
			int10.rom.used+=4;
			phys_writed(rom_base+int10.rom.used,0);		// user palette table
			int10.rom.used+=4;
			phys_writed(rom_base+int10.rom.used,0);		int10.rom.used+=4;
			phys_writed(rom_base+int10.rom.used,0);		int10.rom.used+=4;
			phys_writed(rom_base+int10.rom.used,0);		int10.rom.used+=4;
		}

		int10.rom.video_save_pointers=RealMake(0xC000,int10.rom.used);
		phys_writed(rom_base+int10.rom.used,int10.rom.video_parameter_table);
		int10.rom.used+=4;
		phys_writed(rom_base+int10.rom.used,0);		// dynamic save area pointer
		int10.rom.used+=4;
		phys_writed(rom_base+int10.rom.used,0);		// alphanumeric character set override
		int10.rom.used+=4;
		phys_writed(rom_base+int10.rom.used,0);		// graphics character set override
		int10.rom.used+=4;
		if (IS_VGA_ARCH) {
			phys_writed(rom_base+int10.rom.used,int10.rom.video_save_pointer_table);
		} else {
			phys_writed(rom_base+int10.rom.used,0);		// secondary save pointer table
		}
		int10.rom.used+=4;
		phys_writed(rom_base+int10.rom.used,0);		int10.rom.used+=4;
		phys_writed(rom_base+int10.rom.used,0);		int10.rom.used+=4;
	}

	INT10_SetupBasicVideoParameterTable();
	INT10_SetupRomMemoryChecksum();

	if (IS_TANDY_ARCH) {
		RealSetVec(0x44,RealMake(0xf000,0xfa6e));
	}
}

void INT10_ReloadRomFonts(void)
{
	// 16x8 font
	PhysPt font16pt = RealToPhysical(int10.rom.font_16);
	for (Bitu i = 0; i < 256 * 16; i++) {
		phys_writeb(font16pt + i, int10_font_16[i]);
	}
	phys_writeb(RealToPhysical(int10.rom.font_16_alternate), 0x1d);

	// 14x8 font
	PhysPt font14pt = RealToPhysical(int10.rom.font_14);
	for (Bitu i = 0; i < 256 * 14; i++) {
		phys_writeb(font14pt + i, int10_font_14[i]);
	}
	phys_writeb(RealToPhysical(int10.rom.font_14_alternate), 0x1d);

	// 8x8 fonts
	PhysPt font8pt = RealToPhysical(int10.rom.font_8_first);
	for (Bitu i = 0; i < 128 * 8; i++) {
		phys_writeb(font8pt + i, int10_font_08[i]);
	}

	font8pt = RealToPhysical(int10.rom.font_8_second);
	for (Bitu i = 0; i < 128 * 8; i++) {
		phys_writeb(font8pt + i, int10_font_08[i + 128 * 8]);
	}

	INT10_SetupRomMemoryChecksum();
}

void INT10_SetupRomMemoryChecksum(void)
{
	// EGA/VGA. Just to be safe
	if (IS_EGAVGA_ARCH) {
		// Sum of all bytes in rom module 256 should be 0
		uint8_t sum     = 0;
		PhysPt rom_base = PhysicalMake(0xc000, 0);

		// 32 KB romsize
		Bitu last_rombyte = 32 * 1024 - 1;

		for (Bitu i = 0; i < last_rombyte; i++) {
			// Overflow is okay
			sum += phys_readb(rom_base + i);
		}
		sum = (uint8_t)((256 - (Bitu)sum) & 0xff);

		phys_writeb(rom_base + last_rombyte, sum);
	}
}

// clang-format off
uint8_t int10_font_08[256 * 8] = {
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x7e, 0x81, 0xa5, 0x81, 0xbd, 0x99, 0x81, 0x7e,
  0x7e, 0xff, 0xdb, 0xff, 0xc3, 0xe7, 0xff, 0x7e,
  0x6c, 0xfe, 0xfe, 0xfe, 0x7c, 0x38, 0x10, 0x00,
  0x10, 0x38, 0x7c, 0xfe, 0x7c, 0x38, 0x10, 0x00,
  0x38, 0x7c, 0x38, 0xfe, 0xfe, 0x7c, 0x38, 0x7c,
  0x10, 0x10, 0x38, 0x7c, 0xfe, 0x7c, 0x38, 0x7c,
  0x00, 0x00, 0x18, 0x3c, 0x3c, 0x18, 0x00, 0x00,
  0xff, 0xff, 0xe7, 0xc3, 0xc3, 0xe7, 0xff, 0xff,
  0x00, 0x3c, 0x66, 0x42, 0x42, 0x66, 0x3c, 0x00,
  0xff, 0xc3, 0x99, 0xbd, 0xbd, 0x99, 0xc3, 0xff,
  0x0f, 0x07, 0x0f, 0x7d, 0xcc, 0xcc, 0xcc, 0x78,
  0x3c, 0x66, 0x66, 0x66, 0x3c, 0x18, 0x7e, 0x18,
  0x3f, 0x33, 0x3f, 0x30, 0x30, 0x70, 0xf0, 0xe0,
  0x7f, 0x63, 0x7f, 0x63, 0x63, 0x67, 0xe6, 0xc0,
  0x99, 0x5a, 0x3c, 0xe7, 0xe7, 0x3c, 0x5a, 0x99,
  0x80, 0xe0, 0xf8, 0xfe, 0xf8, 0xe0, 0x80, 0x00,
  0x02, 0x0e, 0x3e, 0xfe, 0x3e, 0x0e, 0x02, 0x00,
  0x18, 0x3c, 0x7e, 0x18, 0x18, 0x7e, 0x3c, 0x18,
  0x66, 0x66, 0x66, 0x66, 0x66, 0x00, 0x66, 0x00,
  0x7f, 0xdb, 0xdb, 0x7b, 0x1b, 0x1b, 0x1b, 0x00,
  0x3e, 0x63, 0x38, 0x6c, 0x6c, 0x38, 0xcc, 0x78,
  0x00, 0x00, 0x00, 0x00, 0x7e, 0x7e, 0x7e, 0x00,
  0x18, 0x3c, 0x7e, 0x18, 0x7e, 0x3c, 0x18, 0xff,
  0x18, 0x3c, 0x7e, 0x18, 0x18, 0x18, 0x18, 0x00,
  0x18, 0x18, 0x18, 0x18, 0x7e, 0x3c, 0x18, 0x00,
  0x00, 0x18, 0x0c, 0xfe, 0x0c, 0x18, 0x00, 0x00,
  0x00, 0x30, 0x60, 0xfe, 0x60, 0x30, 0x00, 0x00,
  0x00, 0x00, 0xc0, 0xc0, 0xc0, 0xfe, 0x00, 0x00,
  0x00, 0x24, 0x66, 0xff, 0x66, 0x24, 0x00, 0x00,
  0x00, 0x18, 0x3c, 0x7e, 0xff, 0xff, 0x00, 0x00,
  0x00, 0xff, 0xff, 0x7e, 0x3c, 0x18, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x30, 0x78, 0x78, 0x30, 0x30, 0x00, 0x30, 0x00,
  0x6c, 0x6c, 0x6c, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x6c, 0x6c, 0xfe, 0x6c, 0xfe, 0x6c, 0x6c, 0x00,
  0x30, 0x7c, 0xc0, 0x78, 0x0c, 0xf8, 0x30, 0x00,
  0x00, 0xc6, 0xcc, 0x18, 0x30, 0x66, 0xc6, 0x00,
  0x38, 0x6c, 0x38, 0x76, 0xdc, 0xcc, 0x76, 0x00,
  0x60, 0x60, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x30, 0x60, 0x60, 0x60, 0x30, 0x18, 0x00,
  0x60, 0x30, 0x18, 0x18, 0x18, 0x30, 0x60, 0x00,
  0x00, 0x66, 0x3c, 0xff, 0x3c, 0x66, 0x00, 0x00,
  0x00, 0x30, 0x30, 0xfc, 0x30, 0x30, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x30, 0x60,
  0x00, 0x00, 0x00, 0xfc, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x30, 0x00,
  0x06, 0x0c, 0x18, 0x30, 0x60, 0xc0, 0x80, 0x00,
  0x7c, 0xc6, 0xce, 0xde, 0xf6, 0xe6, 0x7c, 0x00,
  0x30, 0x70, 0x30, 0x30, 0x30, 0x30, 0xfc, 0x00,
  0x78, 0xcc, 0x0c, 0x38, 0x60, 0xcc, 0xfc, 0x00,
  0x78, 0xcc, 0x0c, 0x38, 0x0c, 0xcc, 0x78, 0x00,
  0x1c, 0x3c, 0x6c, 0xcc, 0xfe, 0x0c, 0x1e, 0x00,
  0xfc, 0xc0, 0xf8, 0x0c, 0x0c, 0xcc, 0x78, 0x00,
  0x38, 0x60, 0xc0, 0xf8, 0xcc, 0xcc, 0x78, 0x00,
  0xfc, 0xcc, 0x0c, 0x18, 0x30, 0x30, 0x30, 0x00,
  0x78, 0xcc, 0xcc, 0x78, 0xcc, 0xcc, 0x78, 0x00,
  0x78, 0xcc, 0xcc, 0x7c, 0x0c, 0x18, 0x70, 0x00,
  0x00, 0x30, 0x30, 0x00, 0x00, 0x30, 0x30, 0x00,
  0x00, 0x30, 0x30, 0x00, 0x00, 0x30, 0x30, 0x60,
  0x18, 0x30, 0x60, 0xc0, 0x60, 0x30, 0x18, 0x00,
  0x00, 0x00, 0xfc, 0x00, 0x00, 0xfc, 0x00, 0x00,
  0x60, 0x30, 0x18, 0x0c, 0x18, 0x30, 0x60, 0x00,
  0x78, 0xcc, 0x0c, 0x18, 0x30, 0x00, 0x30, 0x00,
  0x7c, 0xc6, 0xde, 0xde, 0xde, 0xc0, 0x78, 0x00,
  0x30, 0x78, 0xcc, 0xcc, 0xfc, 0xcc, 0xcc, 0x00,
  0xfc, 0x66, 0x66, 0x7c, 0x66, 0x66, 0xfc, 0x00,
  0x3c, 0x66, 0xc0, 0xc0, 0xc0, 0x66, 0x3c, 0x00,
  0xf8, 0x6c, 0x66, 0x66, 0x66, 0x6c, 0xf8, 0x00,
  0xfe, 0x62, 0x68, 0x78, 0x68, 0x62, 0xfe, 0x00,
  0xfe, 0x62, 0x68, 0x78, 0x68, 0x60, 0xf0, 0x00,
  0x3c, 0x66, 0xc0, 0xc0, 0xce, 0x66, 0x3e, 0x00,
  0xcc, 0xcc, 0xcc, 0xfc, 0xcc, 0xcc, 0xcc, 0x00,
  0x78, 0x30, 0x30, 0x30, 0x30, 0x30, 0x78, 0x00,
  0x1e, 0x0c, 0x0c, 0x0c, 0xcc, 0xcc, 0x78, 0x00,
  0xe6, 0x66, 0x6c, 0x78, 0x6c, 0x66, 0xe6, 0x00,
  0xf0, 0x60, 0x60, 0x60, 0x62, 0x66, 0xfe, 0x00,
  0xc6, 0xee, 0xfe, 0xfe, 0xd6, 0xc6, 0xc6, 0x00,
  0xc6, 0xe6, 0xf6, 0xde, 0xce, 0xc6, 0xc6, 0x00,
  0x38, 0x6c, 0xc6, 0xc6, 0xc6, 0x6c, 0x38, 0x00,
  0xfc, 0x66, 0x66, 0x7c, 0x60, 0x60, 0xf0, 0x00,
  0x78, 0xcc, 0xcc, 0xcc, 0xdc, 0x78, 0x1c, 0x00,
  0xfc, 0x66, 0x66, 0x7c, 0x6c, 0x66, 0xe6, 0x00,
  0x78, 0xcc, 0xe0, 0x70, 0x1c, 0xcc, 0x78, 0x00,
  0xfc, 0xb4, 0x30, 0x30, 0x30, 0x30, 0x78, 0x00,
  0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xfc, 0x00,
  0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0x78, 0x30, 0x00,
  0xc6, 0xc6, 0xc6, 0xd6, 0xfe, 0xee, 0xc6, 0x00,
  0xc6, 0xc6, 0x6c, 0x38, 0x38, 0x6c, 0xc6, 0x00,
  0xcc, 0xcc, 0xcc, 0x78, 0x30, 0x30, 0x78, 0x00,
  0xfe, 0xc6, 0x8c, 0x18, 0x32, 0x66, 0xfe, 0x00,
  0x78, 0x60, 0x60, 0x60, 0x60, 0x60, 0x78, 0x00,
  0xc0, 0x60, 0x30, 0x18, 0x0c, 0x06, 0x02, 0x00,
  0x78, 0x18, 0x18, 0x18, 0x18, 0x18, 0x78, 0x00,
  0x10, 0x38, 0x6c, 0xc6, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff,
  0x30, 0x30, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x78, 0x0c, 0x7c, 0xcc, 0x76, 0x00,
  0xe0, 0x60, 0x60, 0x7c, 0x66, 0x66, 0xdc, 0x00,
  0x00, 0x00, 0x78, 0xcc, 0xc0, 0xcc, 0x78, 0x00,
  0x1c, 0x0c, 0x0c, 0x7c, 0xcc, 0xcc, 0x76, 0x00,
  0x00, 0x00, 0x78, 0xcc, 0xfc, 0xc0, 0x78, 0x00,
  0x38, 0x6c, 0x60, 0xf0, 0x60, 0x60, 0xf0, 0x00,
  0x00, 0x00, 0x76, 0xcc, 0xcc, 0x7c, 0x0c, 0xf8,
  0xe0, 0x60, 0x6c, 0x76, 0x66, 0x66, 0xe6, 0x00,
  0x30, 0x00, 0x70, 0x30, 0x30, 0x30, 0x78, 0x00,
  0x0c, 0x00, 0x0c, 0x0c, 0x0c, 0xcc, 0xcc, 0x78,
  0xe0, 0x60, 0x66, 0x6c, 0x78, 0x6c, 0xe6, 0x00,
  0x70, 0x30, 0x30, 0x30, 0x30, 0x30, 0x78, 0x00,
  0x00, 0x00, 0xcc, 0xfe, 0xfe, 0xd6, 0xc6, 0x00,
  0x00, 0x00, 0xf8, 0xcc, 0xcc, 0xcc, 0xcc, 0x00,
  0x00, 0x00, 0x78, 0xcc, 0xcc, 0xcc, 0x78, 0x00,
  0x00, 0x00, 0xdc, 0x66, 0x66, 0x7c, 0x60, 0xf0,
  0x00, 0x00, 0x76, 0xcc, 0xcc, 0x7c, 0x0c, 0x1e,
  0x00, 0x00, 0xdc, 0x76, 0x66, 0x60, 0xf0, 0x00,
  0x00, 0x00, 0x7c, 0xc0, 0x78, 0x0c, 0xf8, 0x00,
  0x10, 0x30, 0x7c, 0x30, 0x30, 0x34, 0x18, 0x00,
  0x00, 0x00, 0xcc, 0xcc, 0xcc, 0xcc, 0x76, 0x00,
  0x00, 0x00, 0xcc, 0xcc, 0xcc, 0x78, 0x30, 0x00,
  0x00, 0x00, 0xc6, 0xd6, 0xfe, 0xfe, 0x6c, 0x00,
  0x00, 0x00, 0xc6, 0x6c, 0x38, 0x6c, 0xc6, 0x00,
  0x00, 0x00, 0xcc, 0xcc, 0xcc, 0x7c, 0x0c, 0xf8,
  0x00, 0x00, 0xfc, 0x98, 0x30, 0x64, 0xfc, 0x00,
  0x1c, 0x30, 0x30, 0xe0, 0x30, 0x30, 0x1c, 0x00,
  0x18, 0x18, 0x18, 0x00, 0x18, 0x18, 0x18, 0x00,
  0xe0, 0x30, 0x30, 0x1c, 0x30, 0x30, 0xe0, 0x00,
  0x76, 0xdc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x10, 0x38, 0x6c, 0xc6, 0xc6, 0xfe, 0x00,
  0x78, 0xcc, 0xc0, 0xcc, 0x78, 0x18, 0x0c, 0x78,
  0x00, 0xcc, 0x00, 0xcc, 0xcc, 0xcc, 0x7e, 0x00,
  0x1c, 0x00, 0x78, 0xcc, 0xfc, 0xc0, 0x78, 0x00,
  0x7e, 0xc3, 0x3c, 0x06, 0x3e, 0x66, 0x3f, 0x00,
  0xcc, 0x00, 0x78, 0x0c, 0x7c, 0xcc, 0x7e, 0x00,
  0xe0, 0x00, 0x78, 0x0c, 0x7c, 0xcc, 0x7e, 0x00,
  0x30, 0x30, 0x78, 0x0c, 0x7c, 0xcc, 0x7e, 0x00,
  0x00, 0x00, 0x78, 0xc0, 0xc0, 0x78, 0x0c, 0x38,
  0x7e, 0xc3, 0x3c, 0x66, 0x7e, 0x60, 0x3c, 0x00,
  0xcc, 0x00, 0x78, 0xcc, 0xfc, 0xc0, 0x78, 0x00,
  0xe0, 0x00, 0x78, 0xcc, 0xfc, 0xc0, 0x78, 0x00,
  0xcc, 0x00, 0x70, 0x30, 0x30, 0x30, 0x78, 0x00,
  0x7c, 0xc6, 0x38, 0x18, 0x18, 0x18, 0x3c, 0x00,
  0xe0, 0x00, 0x70, 0x30, 0x30, 0x30, 0x78, 0x00,
  0xc6, 0x38, 0x6c, 0xc6, 0xfe, 0xc6, 0xc6, 0x00,
  0x30, 0x30, 0x00, 0x78, 0xcc, 0xfc, 0xcc, 0x00,
  0x1c, 0x00, 0xfc, 0x60, 0x78, 0x60, 0xfc, 0x00,
  0x00, 0x00, 0x7f, 0x0c, 0x7f, 0xcc, 0x7f, 0x00,
  0x3e, 0x6c, 0xcc, 0xfe, 0xcc, 0xcc, 0xce, 0x00,
  0x78, 0xcc, 0x00, 0x78, 0xcc, 0xcc, 0x78, 0x00,
  0x00, 0xcc, 0x00, 0x78, 0xcc, 0xcc, 0x78, 0x00,
  0x00, 0xe0, 0x00, 0x78, 0xcc, 0xcc, 0x78, 0x00,
  0x78, 0xcc, 0x00, 0xcc, 0xcc, 0xcc, 0x7e, 0x00,
  0x00, 0xe0, 0x00, 0xcc, 0xcc, 0xcc, 0x7e, 0x00,
  0x00, 0xcc, 0x00, 0xcc, 0xcc, 0x7c, 0x0c, 0xf8,
  0xc3, 0x18, 0x3c, 0x66, 0x66, 0x3c, 0x18, 0x00,
  0xcc, 0x00, 0xcc, 0xcc, 0xcc, 0xcc, 0x78, 0x00,
  0x18, 0x18, 0x7e, 0xc0, 0xc0, 0x7e, 0x18, 0x18,
  0x38, 0x6c, 0x64, 0xf0, 0x60, 0xe6, 0xfc, 0x00,
  0xcc, 0xcc, 0x78, 0xfc, 0x30, 0xfc, 0x30, 0x30,
  0xf8, 0xcc, 0xcc, 0xfa, 0xc6, 0xcf, 0xc6, 0xc7,
  0x0e, 0x1b, 0x18, 0x3c, 0x18, 0x18, 0xd8, 0x70,
  0x1c, 0x00, 0x78, 0x0c, 0x7c, 0xcc, 0x7e, 0x00,
  0x38, 0x00, 0x70, 0x30, 0x30, 0x30, 0x78, 0x00,
  0x00, 0x1c, 0x00, 0x78, 0xcc, 0xcc, 0x78, 0x00,
  0x00, 0x1c, 0x00, 0xcc, 0xcc, 0xcc, 0x7e, 0x00,
  0x00, 0xf8, 0x00, 0xf8, 0xcc, 0xcc, 0xcc, 0x00,
  0xfc, 0x00, 0xcc, 0xec, 0xfc, 0xdc, 0xcc, 0x00,
  0x3c, 0x6c, 0x6c, 0x3e, 0x00, 0x7e, 0x00, 0x00,
  0x38, 0x6c, 0x6c, 0x38, 0x00, 0x7c, 0x00, 0x00,
  0x30, 0x00, 0x30, 0x60, 0xc0, 0xcc, 0x78, 0x00,
  0x00, 0x00, 0x00, 0xfc, 0xc0, 0xc0, 0x00, 0x00,
  0x00, 0x00, 0x00, 0xfc, 0x0c, 0x0c, 0x00, 0x00,
  0xc3, 0xc6, 0xcc, 0xde, 0x33, 0x66, 0xcc, 0x0f,
  0xc3, 0xc6, 0xcc, 0xdb, 0x37, 0x6f, 0xcf, 0x03,
  0x18, 0x18, 0x00, 0x18, 0x18, 0x18, 0x18, 0x00,
  0x00, 0x33, 0x66, 0xcc, 0x66, 0x33, 0x00, 0x00,
  0x00, 0xcc, 0x66, 0x33, 0x66, 0xcc, 0x00, 0x00,
  0x22, 0x88, 0x22, 0x88, 0x22, 0x88, 0x22, 0x88,
  0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa,
  0xdb, 0x77, 0xdb, 0xee, 0xdb, 0x77, 0xdb, 0xee,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0xf8, 0x18, 0x18, 0x18,
  0x18, 0x18, 0xf8, 0x18, 0xf8, 0x18, 0x18, 0x18,
  0x36, 0x36, 0x36, 0x36, 0xf6, 0x36, 0x36, 0x36,
  0x00, 0x00, 0x00, 0x00, 0xfe, 0x36, 0x36, 0x36,
  0x00, 0x00, 0xf8, 0x18, 0xf8, 0x18, 0x18, 0x18,
  0x36, 0x36, 0xf6, 0x06, 0xf6, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x00, 0x00, 0xfe, 0x06, 0xf6, 0x36, 0x36, 0x36,
  0x36, 0x36, 0xf6, 0x06, 0xfe, 0x00, 0x00, 0x00,
  0x36, 0x36, 0x36, 0x36, 0xfe, 0x00, 0x00, 0x00,
  0x18, 0x18, 0xf8, 0x18, 0xf8, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0xf8, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x1f, 0x00, 0x00, 0x00,
  0x18, 0x18, 0x18, 0x18, 0xff, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0xff, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x1f, 0x18, 0x18, 0x18,
  0x00, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00,
  0x18, 0x18, 0x18, 0x18, 0xff, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x1f, 0x18, 0x1f, 0x18, 0x18, 0x18,
  0x36, 0x36, 0x36, 0x36, 0x37, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x37, 0x30, 0x3f, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x3f, 0x30, 0x37, 0x36, 0x36, 0x36,
  0x36, 0x36, 0xf7, 0x00, 0xff, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xff, 0x00, 0xf7, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x37, 0x30, 0x37, 0x36, 0x36, 0x36,
  0x00, 0x00, 0xff, 0x00, 0xff, 0x00, 0x00, 0x00,
  0x36, 0x36, 0xf7, 0x00, 0xf7, 0x36, 0x36, 0x36,
  0x18, 0x18, 0xff, 0x00, 0xff, 0x00, 0x00, 0x00,
  0x36, 0x36, 0x36, 0x36, 0xff, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xff, 0x00, 0xff, 0x18, 0x18, 0x18,
  0x00, 0x00, 0x00, 0x00, 0xff, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x3f, 0x00, 0x00, 0x00,
  0x18, 0x18, 0x1f, 0x18, 0x1f, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x1f, 0x18, 0x1f, 0x18, 0x18, 0x18,
  0x00, 0x00, 0x00, 0x00, 0x3f, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0xff, 0x36, 0x36, 0x36,
  0x18, 0x18, 0xff, 0x18, 0xff, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0xf8, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x1f, 0x18, 0x18, 0x18,
  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
  0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff,
  0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0,
  0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f,
  0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x76, 0xdc, 0xc8, 0xdc, 0x76, 0x00,
  0x00, 0x78, 0xcc, 0xf8, 0xcc, 0xf8, 0xc0, 0xc0,
  0x00, 0xfc, 0xcc, 0xc0, 0xc0, 0xc0, 0xc0, 0x00,
  0x00, 0xfe, 0x6c, 0x6c, 0x6c, 0x6c, 0x6c, 0x00,
  0xfc, 0xcc, 0x60, 0x30, 0x60, 0xcc, 0xfc, 0x00,
  0x00, 0x00, 0x7e, 0xd8, 0xd8, 0xd8, 0x70, 0x00,
  0x00, 0x66, 0x66, 0x66, 0x66, 0x7c, 0x60, 0xc0,
  0x00, 0x76, 0xdc, 0x18, 0x18, 0x18, 0x18, 0x00,
  0xfc, 0x30, 0x78, 0xcc, 0xcc, 0x78, 0x30, 0xfc,
  0x38, 0x6c, 0xc6, 0xfe, 0xc6, 0x6c, 0x38, 0x00,
  0x38, 0x6c, 0xc6, 0xc6, 0x6c, 0x6c, 0xee, 0x00,
  0x1c, 0x30, 0x18, 0x7c, 0xcc, 0xcc, 0x78, 0x00,
  0x00, 0x00, 0x7e, 0xdb, 0xdb, 0x7e, 0x00, 0x00,
  0x06, 0x0c, 0x7e, 0xdb, 0xdb, 0x7e, 0x60, 0xc0,
  0x38, 0x60, 0xc0, 0xf8, 0xc0, 0x60, 0x38, 0x00,
  0x78, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0x00,
  0x00, 0xfc, 0x00, 0xfc, 0x00, 0xfc, 0x00, 0x00,
  0x30, 0x30, 0xfc, 0x30, 0x30, 0x00, 0xfc, 0x00,
  0x60, 0x30, 0x18, 0x30, 0x60, 0x00, 0xfc, 0x00,
  0x18, 0x30, 0x60, 0x30, 0x18, 0x00, 0xfc, 0x00,
  0x0e, 0x1b, 0x1b, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0xd8, 0xd8, 0x70,
  0x30, 0x30, 0x00, 0xfc, 0x00, 0x30, 0x30, 0x00,
  0x00, 0x76, 0xdc, 0x00, 0x76, 0xdc, 0x00, 0x00,
  0x38, 0x6c, 0x6c, 0x38, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00,
  0x0f, 0x0c, 0x0c, 0x0c, 0xec, 0x6c, 0x3c, 0x1c,
  0x78, 0x6c, 0x6c, 0x6c, 0x6c, 0x00, 0x00, 0x00,
  0x70, 0x18, 0x30, 0x60, 0x78, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x3c, 0x3c, 0x3c, 0x3c, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

uint8_t int10_font_14[256 * 14] = {
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x7e, 0x81, 0xa5, 0x81, 0x81, 0xbd, 0x99, 0x81,
  0x7e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7e, 0xff,
  0xdb, 0xff, 0xff, 0xc3, 0xe7, 0xff, 0x7e, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x6c, 0xfe, 0xfe,
  0xfe, 0xfe, 0x7c, 0x38, 0x10, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x10, 0x38, 0x7c, 0xfe, 0x7c,
  0x38, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x3c, 0x3c, 0xe7, 0xe7, 0xe7, 0x18, 0x18,
  0x3c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x3c,
  0x7e, 0xff, 0xff, 0x7e, 0x18, 0x18, 0x3c, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18,
  0x3c, 0x3c, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00,
  0xff, 0xff, 0xff, 0xff, 0xff, 0xe7, 0xc3, 0xc3,
  0xe7, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00,
  0x00, 0x00, 0x3c, 0x66, 0x42, 0x42, 0x66, 0x3c,
  0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff,
  0xc3, 0x99, 0xbd, 0xbd, 0x99, 0xc3, 0xff, 0xff,
  0xff, 0xff, 0x00, 0x00, 0x1e, 0x0e, 0x1a, 0x32,
  0x78, 0xcc, 0xcc, 0xcc, 0x78, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x3c, 0x66, 0x66, 0x66, 0x3c, 0x18,
  0x7e, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x3f, 0x33, 0x3f, 0x30, 0x30, 0x30, 0x70, 0xf0,
  0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7f, 0x63,
  0x7f, 0x63, 0x63, 0x63, 0x67, 0xe7, 0xe6, 0xc0,
  0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0xdb, 0x3c,
  0xe7, 0x3c, 0xdb, 0x18, 0x18, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x80, 0xc0, 0xe0, 0xf8, 0xfe, 0xf8,
  0xe0, 0xc0, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x02, 0x06, 0x0e, 0x3e, 0xfe, 0x3e, 0x0e, 0x06,
  0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x3c,
  0x7e, 0x18, 0x18, 0x18, 0x7e, 0x3c, 0x18, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x66, 0x66, 0x66, 0x66,
  0x66, 0x66, 0x00, 0x66, 0x66, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7f, 0xdb, 0xdb, 0xdb, 0x7b, 0x1b,
  0x1b, 0x1b, 0x1b, 0x00, 0x00, 0x00, 0x00, 0x7c,
  0xc6, 0x60, 0x38, 0x6c, 0xc6, 0xc6, 0x6c, 0x38,
  0x0c, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0xfe, 0xfe, 0xfe, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x18, 0x3c, 0x7e, 0x18,
  0x18, 0x18, 0x7e, 0x3c, 0x18, 0x7e, 0x00, 0x00,
  0x00, 0x00, 0x18, 0x3c, 0x7e, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x7e, 0x3c,
  0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x0c, 0xfe, 0x0c, 0x18, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x60,
  0xfe, 0x60, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0xc0, 0xc0,
  0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x28, 0x6c, 0xfe, 0x6c, 0x28, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10,
  0x38, 0x38, 0x7c, 0x7c, 0xfe, 0xfe, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0xfe, 0x7c,
  0x7c, 0x38, 0x38, 0x10, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x3c, 0x3c, 0x3c, 0x18, 0x18, 0x00, 0x18,
  0x18, 0x00, 0x00, 0x00, 0x00, 0x66, 0x66, 0x66,
  0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x6c, 0x6c, 0xfe, 0x6c,
  0x6c, 0x6c, 0xfe, 0x6c, 0x6c, 0x00, 0x00, 0x00,
  0x18, 0x18, 0x7c, 0xc6, 0xc2, 0xc0, 0x7c, 0x06,
  0x86, 0xc6, 0x7c, 0x18, 0x18, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xc2, 0xc6, 0x0c, 0x18, 0x30, 0x66,
  0xc6, 0x00, 0x00, 0x00, 0x00, 0x00, 0x38, 0x6c,
  0x6c, 0x38, 0x76, 0xdc, 0xcc, 0xcc, 0x76, 0x00,
  0x00, 0x00, 0x00, 0x30, 0x30, 0x30, 0x60, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x0c, 0x18, 0x30, 0x30, 0x30, 0x30,
  0x30, 0x18, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x30, 0x18, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x18,
  0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x66, 0x3c, 0xff, 0x3c, 0x66, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18,
  0x7e, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x18, 0x18, 0x30, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0xfe, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x02, 0x06, 0x0c, 0x18,
  0x30, 0x60, 0xc0, 0x80, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7c, 0xc6, 0xce, 0xde, 0xf6, 0xe6,
  0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x38, 0x78, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x7e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7c, 0xc6,
  0x06, 0x0c, 0x18, 0x30, 0x60, 0xc6, 0xfe, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x7c, 0xc6, 0x06, 0x06,
  0x3c, 0x06, 0x06, 0xc6, 0x7c, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x0c, 0x1c, 0x3c, 0x6c, 0xcc, 0xfe,
  0x0c, 0x0c, 0x1e, 0x00, 0x00, 0x00, 0x00, 0x00,
  0xfe, 0xc0, 0xc0, 0xc0, 0xfc, 0x06, 0x06, 0xc6,
  0x7c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x38, 0x60,
  0xc0, 0xc0, 0xfc, 0xc6, 0xc6, 0xc6, 0x7c, 0x00,
  0x00, 0x00, 0x00, 0x00, 0xfe, 0xc6, 0x06, 0x0c,
  0x18, 0x30, 0x30, 0x30, 0x30, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0x7c, 0xc6,
  0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x7c, 0xc6, 0xc6, 0xc6, 0x7e, 0x06, 0x06, 0x0c,
  0x78, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18,
  0x18, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00,
  0x00, 0x00, 0x18, 0x18, 0x30, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x06, 0x0c, 0x18, 0x30, 0x60, 0x30,
  0x18, 0x0c, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x7e, 0x00, 0x00, 0x7e, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0x30,
  0x18, 0x0c, 0x06, 0x0c, 0x18, 0x30, 0x60, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x7c, 0xc6, 0xc6, 0x0c,
  0x18, 0x18, 0x00, 0x18, 0x18, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7c, 0xc6, 0xc6, 0xde, 0xde, 0xde,
  0xdc, 0xc0, 0x7c, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x10, 0x38, 0x6c, 0xc6, 0xc6, 0xfe, 0xc6, 0xc6,
  0xc6, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfc, 0x66,
  0x66, 0x66, 0x7c, 0x66, 0x66, 0x66, 0xfc, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x3c, 0x66, 0xc2, 0xc0,
  0xc0, 0xc0, 0xc2, 0x66, 0x3c, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xf8, 0x6c, 0x66, 0x66, 0x66, 0x66,
  0x66, 0x6c, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00,
  0xfe, 0x66, 0x62, 0x68, 0x78, 0x68, 0x62, 0x66,
  0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0x66,
  0x62, 0x68, 0x78, 0x68, 0x60, 0x60, 0xf0, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x3c, 0x66, 0xc2, 0xc0,
  0xc0, 0xde, 0xc6, 0x66, 0x3a, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xc6, 0xc6, 0xc6, 0xc6, 0xfe, 0xc6,
  0xc6, 0xc6, 0xc6, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x3c, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x3c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1e, 0x0c,
  0x0c, 0x0c, 0x0c, 0x0c, 0xcc, 0xcc, 0x78, 0x00,
  0x00, 0x00, 0x00, 0x00, 0xe6, 0x66, 0x6c, 0x6c,
  0x78, 0x6c, 0x6c, 0x66, 0xe6, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xf0, 0x60, 0x60, 0x60, 0x60, 0x60,
  0x62, 0x66, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00,
  0xc6, 0xee, 0xfe, 0xfe, 0xd6, 0xc6, 0xc6, 0xc6,
  0xc6, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc6, 0xe6,
  0xf6, 0xfe, 0xde, 0xce, 0xc6, 0xc6, 0xc6, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x38, 0x6c, 0xc6, 0xc6,
  0xc6, 0xc6, 0xc6, 0x6c, 0x38, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xfc, 0x66, 0x66, 0x66, 0x7c, 0x60,
  0x60, 0x60, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x7c, 0xc6, 0xc6, 0xc6, 0xc6, 0xd6, 0xde, 0x7c,
  0x0c, 0x0e, 0x00, 0x00, 0x00, 0x00, 0xfc, 0x66,
  0x66, 0x66, 0x7c, 0x6c, 0x66, 0x66, 0xe6, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x7c, 0xc6, 0xc6, 0x60,
  0x38, 0x0c, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7e, 0x7e, 0x5a, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00, 0x00,
  0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
  0x7c, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc6, 0xc6,
  0xc6, 0xc6, 0xc6, 0xc6, 0x6c, 0x38, 0x10, 0x00,
  0x00, 0x00, 0x00, 0x00, 0xc6, 0xc6, 0xc6, 0xc6,
  0xd6, 0xd6, 0xfe, 0x7c, 0x6c, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xc6, 0xc6, 0x6c, 0x38, 0x38, 0x38,
  0x6c, 0xc6, 0xc6, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x66, 0x66, 0x66, 0x66, 0x3c, 0x18, 0x18, 0x18,
  0x3c, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0xc6,
  0x8c, 0x18, 0x30, 0x60, 0xc2, 0xc6, 0xfe, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x3c, 0x30, 0x30, 0x30,
  0x30, 0x30, 0x30, 0x30, 0x3c, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x80, 0xc0, 0xe0, 0x70, 0x38, 0x1c,
  0x0e, 0x06, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x3c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c,
  0x3c, 0x00, 0x00, 0x00, 0x10, 0x38, 0x6c, 0xc6,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x00,
  0x30, 0x30, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x78, 0x0c, 0x7c, 0xcc, 0xcc,
  0x76, 0x00, 0x00, 0x00, 0x00, 0x00, 0xe0, 0x60,
  0x60, 0x78, 0x6c, 0x66, 0x66, 0x66, 0x7c, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7c,
  0xc6, 0xc0, 0xc0, 0xc6, 0x7c, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x1c, 0x0c, 0x0c, 0x3c, 0x6c, 0xcc,
  0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x7c, 0xc6, 0xfe, 0xc0, 0xc6,
  0x7c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x38, 0x6c,
  0x64, 0x60, 0xf0, 0x60, 0x60, 0x60, 0xf0, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x76,
  0xcc, 0xcc, 0xcc, 0x7c, 0x0c, 0xcc, 0x78, 0x00,
  0x00, 0x00, 0xe0, 0x60, 0x60, 0x6c, 0x76, 0x66,
  0x66, 0x66, 0xe6, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x18, 0x00, 0x38, 0x18, 0x18, 0x18, 0x18,
  0x3c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x06,
  0x00, 0x0e, 0x06, 0x06, 0x06, 0x06, 0x66, 0x66,
  0x3c, 0x00, 0x00, 0x00, 0xe0, 0x60, 0x60, 0x66,
  0x6c, 0x78, 0x6c, 0x66, 0xe6, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x38, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0xec, 0xfe, 0xd6, 0xd6, 0xd6,
  0xc6, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0xdc, 0x66, 0x66, 0x66, 0x66, 0x66, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7c,
  0xc6, 0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xdc, 0x66, 0x66,
  0x66, 0x7c, 0x60, 0x60, 0xf0, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x76, 0xcc, 0xcc, 0xcc, 0x7c,
  0x0c, 0x0c, 0x1e, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0xdc, 0x76, 0x66, 0x60, 0x60, 0xf0, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7c,
  0xc6, 0x70, 0x1c, 0xc6, 0x7c, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x10, 0x30, 0x30, 0xfc, 0x30, 0x30,
  0x30, 0x36, 0x1c, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc,
  0x76, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x66, 0x66, 0x66, 0x66, 0x3c, 0x18, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc6,
  0xc6, 0xd6, 0xd6, 0xfe, 0x6c, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xc6, 0x6c, 0x38,
  0x38, 0x6c, 0xc6, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0xc6, 0xc6, 0xc6, 0xc6, 0x7e,
  0x06, 0x0c, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0xfe, 0xcc, 0x18, 0x30, 0x66, 0xfe, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x0e, 0x18, 0x18, 0x18,
  0x70, 0x18, 0x18, 0x18, 0x0e, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x18, 0x18, 0x18, 0x18, 0x00, 0x18,
  0x18, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x70, 0x18, 0x18, 0x18, 0x0e, 0x18, 0x18, 0x18,
  0x70, 0x00, 0x00, 0x00, 0x00, 0x00, 0x76, 0xdc,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x38,
  0x6c, 0xc6, 0xc6, 0xfe, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x3c, 0x66, 0xc2, 0xc0, 0xc0, 0xc2,
  0x66, 0x3c, 0x0c, 0x06, 0x7c, 0x00, 0x00, 0x00,
  0xcc, 0xcc, 0x00, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc,
  0x76, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x18, 0x30,
  0x00, 0x7c, 0xc6, 0xfe, 0xc0, 0xc6, 0x7c, 0x00,
  0x00, 0x00, 0x00, 0x10, 0x38, 0x6c, 0x00, 0x78,
  0x0c, 0x7c, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xcc, 0xcc, 0x00, 0x78, 0x0c, 0x7c,
  0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00, 0x60,
  0x30, 0x18, 0x00, 0x78, 0x0c, 0x7c, 0xcc, 0xcc,
  0x76, 0x00, 0x00, 0x00, 0x00, 0x38, 0x6c, 0x38,
  0x00, 0x78, 0x0c, 0x7c, 0xcc, 0xcc, 0x76, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3c, 0x66,
  0x60, 0x66, 0x3c, 0x0c, 0x06, 0x3c, 0x00, 0x00,
  0x00, 0x10, 0x38, 0x6c, 0x00, 0x7c, 0xc6, 0xfe,
  0xc0, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00, 0x00,
  0xcc, 0xcc, 0x00, 0x7c, 0xc6, 0xfe, 0xc0, 0xc6,
  0x7c, 0x00, 0x00, 0x00, 0x00, 0x60, 0x30, 0x18,
  0x00, 0x7c, 0xc6, 0xfe, 0xc0, 0xc6, 0x7c, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x66, 0x66, 0x00, 0x38,
  0x18, 0x18, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00,
  0x00, 0x18, 0x3c, 0x66, 0x00, 0x38, 0x18, 0x18,
  0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00, 0x60,
  0x30, 0x18, 0x00, 0x38, 0x18, 0x18, 0x18, 0x18,
  0x3c, 0x00, 0x00, 0x00, 0x00, 0xc6, 0xc6, 0x10,
  0x38, 0x6c, 0xc6, 0xc6, 0xfe, 0xc6, 0xc6, 0x00,
  0x00, 0x00, 0x38, 0x6c, 0x38, 0x00, 0x38, 0x6c,
  0xc6, 0xc6, 0xfe, 0xc6, 0xc6, 0x00, 0x00, 0x00,
  0x18, 0x30, 0x60, 0x00, 0xfe, 0x66, 0x60, 0x7c,
  0x60, 0x66, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xcc, 0x76, 0x36, 0x7e, 0xd8, 0xd8,
  0x6e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3e, 0x6c,
  0xcc, 0xcc, 0xfe, 0xcc, 0xcc, 0xcc, 0xce, 0x00,
  0x00, 0x00, 0x00, 0x10, 0x38, 0x6c, 0x00, 0x7c,
  0xc6, 0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xc6, 0xc6, 0x00, 0x7c, 0xc6, 0xc6,
  0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00, 0x60,
  0x30, 0x18, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0xc6,
  0x7c, 0x00, 0x00, 0x00, 0x00, 0x30, 0x78, 0xcc,
  0x00, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0x76, 0x00,
  0x00, 0x00, 0x00, 0x60, 0x30, 0x18, 0x00, 0xcc,
  0xcc, 0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xc6, 0xc6, 0x00, 0xc6, 0xc6, 0xc6,
  0xc6, 0x7e, 0x06, 0x0c, 0x78, 0x00, 0x00, 0xc6,
  0xc6, 0x38, 0x6c, 0xc6, 0xc6, 0xc6, 0xc6, 0x6c,
  0x38, 0x00, 0x00, 0x00, 0x00, 0xc6, 0xc6, 0x00,
  0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0x7c, 0x00,
  0x00, 0x00, 0x00, 0x18, 0x18, 0x3c, 0x66, 0x60,
  0x60, 0x66, 0x3c, 0x18, 0x18, 0x00, 0x00, 0x00,
  0x00, 0x38, 0x6c, 0x64, 0x60, 0xf0, 0x60, 0x60,
  0x60, 0xe6, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x66, 0x66, 0x3c, 0x18, 0x7e, 0x18, 0x7e, 0x18,
  0x18, 0x00, 0x00, 0x00, 0x00, 0xf8, 0xcc, 0xcc,
  0xf8, 0xc4, 0xcc, 0xde, 0xcc, 0xcc, 0xc6, 0x00,
  0x00, 0x00, 0x00, 0x0e, 0x1b, 0x18, 0x18, 0x18,
  0x7e, 0x18, 0x18, 0x18, 0x18, 0xd8, 0x70, 0x00,
  0x00, 0x18, 0x30, 0x60, 0x00, 0x78, 0x0c, 0x7c,
  0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00, 0x0c,
  0x18, 0x30, 0x00, 0x38, 0x18, 0x18, 0x18, 0x18,
  0x3c, 0x00, 0x00, 0x00, 0x00, 0x18, 0x30, 0x60,
  0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0xc6, 0x7c, 0x00,
  0x00, 0x00, 0x00, 0x18, 0x30, 0x60, 0x00, 0xcc,
  0xcc, 0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x76, 0xdc, 0x00, 0xdc, 0x66, 0x66,
  0x66, 0x66, 0x66, 0x00, 0x00, 0x00, 0x76, 0xdc,
  0x00, 0xc6, 0xe6, 0xf6, 0xfe, 0xde, 0xce, 0xc6,
  0xc6, 0x00, 0x00, 0x00, 0x00, 0x3c, 0x6c, 0x6c,
  0x3e, 0x00, 0x7e, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x38, 0x6c, 0x6c, 0x38, 0x00,
  0x7c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x30, 0x30, 0x00, 0x30, 0x30, 0x60,
  0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0xfe, 0xc0, 0xc0, 0xc0,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xfe, 0x06, 0x06, 0x06, 0x00, 0x00,
  0x00, 0x00, 0x00, 0xc0, 0xc0, 0xc6, 0xcc, 0xd8,
  0x30, 0x60, 0xdc, 0x86, 0x0c, 0x18, 0x3e, 0x00,
  0x00, 0xc0, 0xc0, 0xc6, 0xcc, 0xd8, 0x30, 0x66,
  0xce, 0x9e, 0x3e, 0x06, 0x06, 0x00, 0x00, 0x00,
  0x18, 0x18, 0x00, 0x18, 0x18, 0x3c, 0x3c, 0x3c,
  0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x36, 0x6c, 0xd8, 0x6c, 0x36, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xd8, 0x6c,
  0x36, 0x6c, 0xd8, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x11, 0x44, 0x11, 0x44, 0x11, 0x44, 0x11, 0x44,
  0x11, 0x44, 0x11, 0x44, 0x11, 0x44, 0x55, 0xaa,
  0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa,
  0x55, 0xaa, 0x55, 0xaa, 0xdd, 0x77, 0xdd, 0x77,
  0xdd, 0x77, 0xdd, 0x77, 0xdd, 0x77, 0xdd, 0x77,
  0xdd, 0x77, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0xf8,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0xf8, 0x18, 0xf8, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0xf6, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0xfe, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xf8, 0x18, 0xf8,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x36, 0x36,
  0x36, 0x36, 0x36, 0xf6, 0x06, 0xf6, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfe,
  0x06, 0xf6, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x36, 0xf6, 0x06, 0xfe,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x36, 0xfe, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x18, 0x18,
  0x18, 0xf8, 0x18, 0xf8, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0xf8, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x1f,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0xff, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0xff, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x1f, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0xff, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x1f, 0x18, 0x1f, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x37, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x37, 0x30, 0x3f,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x3f, 0x30, 0x37, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x36, 0xf7, 0x00, 0xff, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff,
  0x00, 0xf7, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x37, 0x30, 0x37,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x00, 0x00,
  0x00, 0x00, 0x00, 0xff, 0x00, 0xff, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x36, 0x36, 0x36, 0x36,
  0x36, 0xf7, 0x00, 0xf7, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x18, 0x18, 0x18, 0x18, 0x18, 0xff,
  0x00, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0xff,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0xff, 0x00, 0xff, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0xff, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x3f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x1f, 0x18, 0x1f,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x1f, 0x18, 0x1f, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x3f, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x36, 0xff, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x18, 0x18, 0x18, 0x18, 0x18, 0xff, 0x18, 0xff,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0xf8, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x1f, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff,
  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0xf0,
  0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0,
  0xf0, 0xf0, 0xf0, 0xf0, 0x0f, 0x0f, 0x0f, 0x0f,
  0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f,
  0x0f, 0x0f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
  0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x76, 0xdc, 0xd8,
  0xd8, 0xdc, 0x76, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7c, 0xc6, 0xfc, 0xc6, 0xc6, 0xfc,
  0xc0, 0xc0, 0x40, 0x00, 0x00, 0x00, 0xfe, 0xc6,
  0xc6, 0xc0, 0xc0, 0xc0, 0xc0, 0xc0, 0xc0, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0x6c,
  0x6c, 0x6c, 0x6c, 0x6c, 0x6c, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xfe, 0xc6, 0x60, 0x30, 0x18, 0x30,
  0x60, 0xc6, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x7e, 0xd8, 0xd8, 0xd8, 0xd8,
  0x70, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x66, 0x66, 0x66, 0x66, 0x7c, 0x60, 0x60, 0xc0,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x76, 0xdc,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7e, 0x18, 0x3c, 0x66, 0x66, 0x66,
  0x3c, 0x18, 0x7e, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x38, 0x6c, 0xc6, 0xc6, 0xfe, 0xc6, 0xc6, 0x6c,
  0x38, 0x00, 0x00, 0x00, 0x00, 0x00, 0x38, 0x6c,
  0xc6, 0xc6, 0xc6, 0x6c, 0x6c, 0x6c, 0xee, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x1e, 0x30, 0x18, 0x0c,
  0x3e, 0x66, 0x66, 0x66, 0x3c, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x7e, 0xdb, 0xdb,
  0x7e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x03, 0x06, 0x7e, 0xdb, 0xdb, 0xf3, 0x7e, 0x60,
  0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1c, 0x30,
  0x60, 0x60, 0x7c, 0x60, 0x60, 0x30, 0x1c, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x7c, 0xc6, 0xc6,
  0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0xfe, 0x00, 0x00, 0xfe, 0x00,
  0x00, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x18, 0x18, 0x7e, 0x18, 0x18, 0x00, 0x00,
  0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x18,
  0x0c, 0x06, 0x0c, 0x18, 0x30, 0x00, 0x7e, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x0c, 0x18, 0x30, 0x60,
  0x30, 0x18, 0x0c, 0x00, 0x7e, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x0e, 0x1b, 0x1b, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0xd8, 0xd8,
  0x70, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18,
  0x18, 0x00, 0x7e, 0x00, 0x18, 0x18, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x76, 0xdc,
  0x00, 0x76, 0xdc, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x38, 0x6c, 0x6c, 0x38, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x0f, 0x0c, 0x0c, 0x0c, 0x0c,
  0x0c, 0xec, 0x6c, 0x3c, 0x1c, 0x00, 0x00, 0x00,
  0x00, 0xd8, 0x6c, 0x6c, 0x6c, 0x6c, 0x6c, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x70,
  0xd8, 0x30, 0x60, 0xc8, 0xf8, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x7c, 0x7c, 0x7c, 0x7c, 0x7c, 0x7c, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

uint8_t int10_font_16[256 * 16] = {
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7e, 0x81, 0xa5, 0x81, 0x81, 0xbd,
  0x99, 0x81, 0x81, 0x7e, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7e, 0xff, 0xdb, 0xff, 0xff, 0xc3,
  0xe7, 0xff, 0xff, 0x7e, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x6c, 0xfe, 0xfe, 0xfe,
  0xfe, 0x7c, 0x38, 0x10, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x10, 0x38, 0x7c, 0xfe,
  0x7c, 0x38, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x18, 0x3c, 0x3c, 0xe7, 0xe7,
  0xe7, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x18, 0x3c, 0x7e, 0xff, 0xff,
  0x7e, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x3c,
  0x3c, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe7, 0xc3,
  0xc3, 0xe7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x3c, 0x66, 0x42,
  0x42, 0x66, 0x3c, 0x00, 0x00, 0x00, 0x00, 0x00,
  0xff, 0xff, 0xff, 0xff, 0xff, 0xc3, 0x99, 0xbd,
  0xbd, 0x99, 0xc3, 0xff, 0xff, 0xff, 0xff, 0xff,
  0x00, 0x00, 0x1e, 0x0e, 0x1a, 0x32, 0x78, 0xcc,
  0xcc, 0xcc, 0xcc, 0x78, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x3c, 0x66, 0x66, 0x66, 0x66, 0x3c,
  0x18, 0x7e, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x3f, 0x33, 0x3f, 0x30, 0x30, 0x30,
  0x30, 0x70, 0xf0, 0xe0, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7f, 0x63, 0x7f, 0x63, 0x63, 0x63,
  0x63, 0x67, 0xe7, 0xe6, 0xc0, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x18, 0x18, 0xdb, 0x3c, 0xe7,
  0x3c, 0xdb, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x80, 0xc0, 0xe0, 0xf0, 0xf8, 0xfe, 0xf8,
  0xf0, 0xe0, 0xc0, 0x80, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x02, 0x06, 0x0e, 0x1e, 0x3e, 0xfe, 0x3e,
  0x1e, 0x0e, 0x06, 0x02, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x18, 0x3c, 0x7e, 0x18, 0x18, 0x18,
  0x7e, 0x3c, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66,
  0x66, 0x00, 0x66, 0x66, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7f, 0xdb, 0xdb, 0xdb, 0x7b, 0x1b,
  0x1b, 0x1b, 0x1b, 0x1b, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x7c, 0xc6, 0x60, 0x38, 0x6c, 0xc6, 0xc6,
  0x6c, 0x38, 0x0c, 0xc6, 0x7c, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0xfe, 0xfe, 0xfe, 0xfe, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x18, 0x3c, 0x7e, 0x18, 0x18, 0x18,
  0x7e, 0x3c, 0x18, 0x7e, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x18, 0x3c, 0x7e, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x7e, 0x3c, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x0c, 0xfe,
  0x0c, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x60, 0xfe,
  0x60, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0xc0,
  0xc0, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x28, 0x6c, 0xfe,
  0x6c, 0x28, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x10, 0x38, 0x38, 0x7c,
  0x7c, 0xfe, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0xfe, 0xfe, 0x7c, 0x7c,
  0x38, 0x38, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x18, 0x3c, 0x3c, 0x3c, 0x18, 0x18,
  0x18, 0x00, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x66, 0x66, 0x66, 0x24, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x6c, 0x6c, 0xfe, 0x6c, 0x6c,
  0x6c, 0xfe, 0x6c, 0x6c, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x18, 0x7c, 0xc6, 0xc2, 0xc0, 0x7c, 0x06,
  0x06, 0x86, 0xc6, 0x7c, 0x18, 0x18, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0xc2, 0xc6, 0x0c, 0x18,
  0x30, 0x60, 0xc6, 0x86, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x38, 0x6c, 0x6c, 0x38, 0x76, 0xdc,
  0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x30, 0x30, 0x30, 0x60, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x0c, 0x18, 0x30, 0x30, 0x30, 0x30,
  0x30, 0x30, 0x18, 0x0c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x30, 0x18, 0x0c, 0x0c, 0x0c, 0x0c,
  0x0c, 0x0c, 0x18, 0x30, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x66, 0x3c, 0xff,
  0x3c, 0x66, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x7e,
  0x18, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x18, 0x18, 0x18, 0x30, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfe,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x02, 0x06, 0x0c, 0x18,
  0x30, 0x60, 0xc0, 0x80, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x38, 0x6c, 0xc6, 0xc6, 0xd6, 0xd6,
  0xc6, 0xc6, 0x6c, 0x38, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x18, 0x38, 0x78, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x7e, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7c, 0xc6, 0x06, 0x0c, 0x18, 0x30,
  0x60, 0xc0, 0xc6, 0xfe, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7c, 0xc6, 0x06, 0x06, 0x3c, 0x06,
  0x06, 0x06, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x0c, 0x1c, 0x3c, 0x6c, 0xcc, 0xfe,
  0x0c, 0x0c, 0x0c, 0x1e, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xfe, 0xc0, 0xc0, 0xc0, 0xfc, 0x06,
  0x06, 0x06, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x38, 0x60, 0xc0, 0xc0, 0xfc, 0xc6,
  0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xfe, 0xc6, 0x06, 0x06, 0x0c, 0x18,
  0x30, 0x30, 0x30, 0x30, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0x7c, 0xc6,
  0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0x7e, 0x06,
  0x06, 0x06, 0x0c, 0x78, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x00,
  0x00, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x00,
  0x00, 0x18, 0x18, 0x30, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x06, 0x0c, 0x18, 0x30, 0x60,
  0x30, 0x18, 0x0c, 0x06, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x7e, 0x00, 0x00,
  0x7e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x60, 0x30, 0x18, 0x0c, 0x06,
  0x0c, 0x18, 0x30, 0x60, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7c, 0xc6, 0xc6, 0x0c, 0x18, 0x18,
  0x18, 0x00, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x7c, 0xc6, 0xc6, 0xde, 0xde,
  0xde, 0xdc, 0xc0, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x10, 0x38, 0x6c, 0xc6, 0xc6, 0xfe,
  0xc6, 0xc6, 0xc6, 0xc6, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xfc, 0x66, 0x66, 0x66, 0x7c, 0x66,
  0x66, 0x66, 0x66, 0xfc, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x3c, 0x66, 0xc2, 0xc0, 0xc0, 0xc0,
  0xc0, 0xc2, 0x66, 0x3c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xf8, 0x6c, 0x66, 0x66, 0x66, 0x66,
  0x66, 0x66, 0x6c, 0xf8, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xfe, 0x66, 0x62, 0x68, 0x78, 0x68,
  0x60, 0x62, 0x66, 0xfe, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xfe, 0x66, 0x62, 0x68, 0x78, 0x68,
  0x60, 0x60, 0x60, 0xf0, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x3c, 0x66, 0xc2, 0xc0, 0xc0, 0xde,
  0xc6, 0xc6, 0x66, 0x3a, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xc6, 0xc6, 0xc6, 0xc6, 0xfe, 0xc6,
  0xc6, 0xc6, 0xc6, 0xc6, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x3c, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x1e, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c,
  0xcc, 0xcc, 0xcc, 0x78, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xe6, 0x66, 0x66, 0x6c, 0x78, 0x78,
  0x6c, 0x66, 0x66, 0xe6, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xf0, 0x60, 0x60, 0x60, 0x60, 0x60,
  0x60, 0x62, 0x66, 0xfe, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xc6, 0xee, 0xfe, 0xfe, 0xd6, 0xc6,
  0xc6, 0xc6, 0xc6, 0xc6, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xc6, 0xe6, 0xf6, 0xfe, 0xde, 0xce,
  0xc6, 0xc6, 0xc6, 0xc6, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
  0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xfc, 0x66, 0x66, 0x66, 0x7c, 0x60,
  0x60, 0x60, 0x60, 0xf0, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
  0xc6, 0xd6, 0xde, 0x7c, 0x0c, 0x0e, 0x00, 0x00,
  0x00, 0x00, 0xfc, 0x66, 0x66, 0x66, 0x7c, 0x6c,
  0x66, 0x66, 0x66, 0xe6, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7c, 0xc6, 0xc6, 0x60, 0x38, 0x0c,
  0x06, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7e, 0x7e, 0x5a, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
  0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
  0xc6, 0x6c, 0x38, 0x10, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xc6, 0xc6, 0xc6, 0xc6, 0xd6, 0xd6,
  0xd6, 0xfe, 0xee, 0x6c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xc6, 0xc6, 0x6c, 0x7c, 0x38, 0x38,
  0x7c, 0x6c, 0xc6, 0xc6, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x66, 0x66, 0x66, 0x66, 0x3c, 0x18,
  0x18, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xfe, 0xc6, 0x86, 0x0c, 0x18, 0x30,
  0x60, 0xc2, 0xc6, 0xfe, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x3c, 0x30, 0x30, 0x30, 0x30, 0x30,
  0x30, 0x30, 0x30, 0x3c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x80, 0xc0, 0xe0, 0x70, 0x38,
  0x1c, 0x0e, 0x06, 0x02, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x3c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c,
  0x0c, 0x0c, 0x0c, 0x3c, 0x00, 0x00, 0x00, 0x00,
  0x10, 0x38, 0x6c, 0xc6, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00,
  0x30, 0x30, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x78, 0x0c, 0x7c,
  0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xe0, 0x60, 0x60, 0x78, 0x6c, 0x66,
  0x66, 0x66, 0x66, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x7c, 0xc6, 0xc0,
  0xc0, 0xc0, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x1c, 0x0c, 0x0c, 0x3c, 0x6c, 0xcc,
  0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x7c, 0xc6, 0xfe,
  0xc0, 0xc0, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x38, 0x6c, 0x64, 0x60, 0xf0, 0x60,
  0x60, 0x60, 0x60, 0xf0, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x76, 0xcc, 0xcc,
  0xcc, 0xcc, 0xcc, 0x7c, 0x0c, 0xcc, 0x78, 0x00,
  0x00, 0x00, 0xe0, 0x60, 0x60, 0x6c, 0x76, 0x66,
  0x66, 0x66, 0x66, 0xe6, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x18, 0x18, 0x00, 0x38, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x06, 0x06, 0x00, 0x0e, 0x06, 0x06,
  0x06, 0x06, 0x06, 0x06, 0x66, 0x66, 0x3c, 0x00,
  0x00, 0x00, 0xe0, 0x60, 0x60, 0x66, 0x6c, 0x78,
  0x78, 0x6c, 0x66, 0xe6, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x38, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xec, 0xfe, 0xd6,
  0xd6, 0xd6, 0xd6, 0xc6, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xdc, 0x66, 0x66,
  0x66, 0x66, 0x66, 0x66, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x7c, 0xc6, 0xc6,
  0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xdc, 0x66, 0x66,
  0x66, 0x66, 0x66, 0x7c, 0x60, 0x60, 0xf0, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x76, 0xcc, 0xcc,
  0xcc, 0xcc, 0xcc, 0x7c, 0x0c, 0x0c, 0x1e, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xdc, 0x76, 0x66,
  0x60, 0x60, 0x60, 0xf0, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x7c, 0xc6, 0x60,
  0x38, 0x0c, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x10, 0x30, 0x30, 0xfc, 0x30, 0x30,
  0x30, 0x30, 0x36, 0x1c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xcc, 0xcc, 0xcc,
  0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x66, 0x66, 0x66,
  0x66, 0x66, 0x3c, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xc6, 0xc6, 0xd6,
  0xd6, 0xd6, 0xfe, 0x6c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xc6, 0x6c, 0x38,
  0x38, 0x38, 0x6c, 0xc6, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xc6, 0xc6, 0xc6,
  0xc6, 0xc6, 0xc6, 0x7e, 0x06, 0x0c, 0xf8, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0xcc, 0x18,
  0x30, 0x60, 0xc6, 0xfe, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x0e, 0x18, 0x18, 0x18, 0x70, 0x18,
  0x18, 0x18, 0x18, 0x0e, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x18, 0x18, 0x18, 0x18, 0x00, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x70, 0x18, 0x18, 0x18, 0x0e, 0x18,
  0x18, 0x18, 0x18, 0x70, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x76, 0xdc, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x10, 0x38, 0x6c, 0xc6,
  0xc6, 0xc6, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x3c, 0x66, 0xc2, 0xc0, 0xc0, 0xc0,
  0xc2, 0x66, 0x3c, 0x0c, 0x06, 0x7c, 0x00, 0x00,
  0x00, 0x00, 0xcc, 0x00, 0x00, 0xcc, 0xcc, 0xcc,
  0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x0c, 0x18, 0x30, 0x00, 0x7c, 0xc6, 0xfe,
  0xc0, 0xc0, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x10, 0x38, 0x6c, 0x00, 0x78, 0x0c, 0x7c,
  0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xcc, 0x00, 0x00, 0x78, 0x0c, 0x7c,
  0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x60, 0x30, 0x18, 0x00, 0x78, 0x0c, 0x7c,
  0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x38, 0x6c, 0x38, 0x00, 0x78, 0x0c, 0x7c,
  0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x3c, 0x66, 0x60, 0x60,
  0x66, 0x3c, 0x0c, 0x06, 0x3c, 0x00, 0x00, 0x00,
  0x00, 0x10, 0x38, 0x6c, 0x00, 0x7c, 0xc6, 0xfe,
  0xc0, 0xc0, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xc6, 0x00, 0x00, 0x7c, 0xc6, 0xfe,
  0xc0, 0xc0, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x60, 0x30, 0x18, 0x00, 0x7c, 0xc6, 0xfe,
  0xc0, 0xc0, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x66, 0x00, 0x00, 0x38, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x18, 0x3c, 0x66, 0x00, 0x38, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x60, 0x30, 0x18, 0x00, 0x38, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0xc6, 0x00, 0x10, 0x38, 0x6c, 0xc6, 0xc6,
  0xfe, 0xc6, 0xc6, 0xc6, 0x00, 0x00, 0x00, 0x00,
  0x38, 0x6c, 0x38, 0x00, 0x38, 0x6c, 0xc6, 0xc6,
  0xfe, 0xc6, 0xc6, 0xc6, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x30, 0x60, 0x00, 0xfe, 0x66, 0x60, 0x7c,
  0x60, 0x60, 0x66, 0xfe, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xcc, 0x76, 0x36,
  0x7e, 0xd8, 0xd8, 0x6e, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x3e, 0x6c, 0xcc, 0xcc, 0xfe, 0xcc,
  0xcc, 0xcc, 0xcc, 0xce, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x10, 0x38, 0x6c, 0x00, 0x7c, 0xc6, 0xc6,
  0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xc6, 0x00, 0x00, 0x7c, 0xc6, 0xc6,
  0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x60, 0x30, 0x18, 0x00, 0x7c, 0xc6, 0xc6,
  0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x30, 0x78, 0xcc, 0x00, 0xcc, 0xcc, 0xcc,
  0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x60, 0x30, 0x18, 0x00, 0xcc, 0xcc, 0xcc,
  0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xc6, 0x00, 0x00, 0xc6, 0xc6, 0xc6,
  0xc6, 0xc6, 0xc6, 0x7e, 0x06, 0x0c, 0x78, 0x00,
  0x00, 0xc6, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0xc6,
  0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0xc6, 0x00, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
  0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x18, 0x18, 0x3c, 0x66, 0x60, 0x60, 0x60,
  0x66, 0x3c, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x38, 0x6c, 0x64, 0x60, 0xf0, 0x60, 0x60,
  0x60, 0x60, 0xe6, 0xfc, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x66, 0x66, 0x3c, 0x18, 0x7e, 0x18,
  0x7e, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x00, 0xf8, 0xcc, 0xcc, 0xf8, 0xc4, 0xcc, 0xde,
  0xcc, 0xcc, 0xcc, 0xc6, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x0e, 0x1b, 0x18, 0x18, 0x18, 0x7e, 0x18,
  0x18, 0x18, 0x18, 0x18, 0xd8, 0x70, 0x00, 0x00,
  0x00, 0x18, 0x30, 0x60, 0x00, 0x78, 0x0c, 0x7c,
  0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x0c, 0x18, 0x30, 0x00, 0x38, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x18, 0x30, 0x60, 0x00, 0x7c, 0xc6, 0xc6,
  0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x18, 0x30, 0x60, 0x00, 0xcc, 0xcc, 0xcc,
  0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x76, 0xdc, 0x00, 0xdc, 0x66, 0x66,
  0x66, 0x66, 0x66, 0x66, 0x00, 0x00, 0x00, 0x00,
  0x76, 0xdc, 0x00, 0xc6, 0xe6, 0xf6, 0xfe, 0xde,
  0xce, 0xc6, 0xc6, 0xc6, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x3c, 0x6c, 0x6c, 0x3e, 0x00, 0x7e, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x38, 0x6c, 0x6c, 0x38, 0x00, 0x7c, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x30, 0x30, 0x00, 0x30, 0x30, 0x60,
  0xc0, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0xc0,
  0xc0, 0xc0, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0x06,
  0x06, 0x06, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0xc0, 0xc0, 0xc2, 0xc6, 0xcc, 0x18, 0x30,
  0x60, 0xdc, 0x86, 0x0c, 0x18, 0x3e, 0x00, 0x00,
  0x00, 0xc0, 0xc0, 0xc2, 0xc6, 0xcc, 0x18, 0x30,
  0x66, 0xce, 0x9e, 0x3e, 0x06, 0x06, 0x00, 0x00,
  0x00, 0x00, 0x18, 0x18, 0x00, 0x18, 0x18, 0x18,
  0x3c, 0x3c, 0x3c, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x36, 0x6c, 0xd8,
  0x6c, 0x36, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xd8, 0x6c, 0x36,
  0x6c, 0xd8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x11, 0x44, 0x11, 0x44, 0x11, 0x44, 0x11, 0x44,
  0x11, 0x44, 0x11, 0x44, 0x11, 0x44, 0x11, 0x44,
  0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa,
  0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa,
  0xdd, 0x77, 0xdd, 0x77, 0xdd, 0x77, 0xdd, 0x77,
  0xdd, 0x77, 0xdd, 0x77, 0xdd, 0x77, 0xdd, 0x77,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0xf8,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0xf8, 0x18, 0xf8,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0xf6,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfe,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xf8, 0x18, 0xf8,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x36, 0x36, 0x36, 0x36, 0x36, 0xf6, 0x06, 0xf6,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0x06, 0xf6,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x36, 0xf6, 0x06, 0xfe,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0xfe,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x18, 0x18, 0x18, 0x18, 0xf8, 0x18, 0xf8,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf8,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x1f,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0xff,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x1f,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0xff,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x1f, 0x18, 0x1f,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x37,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x37, 0x30, 0x3f,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x3f, 0x30, 0x37,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x36, 0xf7, 0x00, 0xff,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x00, 0xf7,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x37, 0x30, 0x37,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x00, 0xff,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x36, 0x36, 0x36, 0x36, 0x36, 0xf7, 0x00, 0xf7,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x18, 0x18, 0x18, 0x18, 0x18, 0xff, 0x00, 0xff,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0xff,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x00, 0xff,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x3f,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x1f, 0x18, 0x1f,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x1f, 0x18, 0x1f,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3f,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0xff,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x18, 0x18, 0x18, 0x18, 0x18, 0xff, 0x18, 0xff,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0xf8,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1f,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff,
  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
  0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0,
  0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0,
  0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f,
  0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f,
  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x76, 0xdc, 0xd8,
  0xd8, 0xd8, 0xdc, 0x76, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x78, 0xcc, 0xcc, 0xcc, 0xd8, 0xcc,
  0xc6, 0xc6, 0xc6, 0xcc, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xfe, 0xc6, 0xc6, 0xc0, 0xc0, 0xc0,
  0xc0, 0xc0, 0xc0, 0xc0, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0xfe, 0x6c, 0x6c, 0x6c,
  0x6c, 0x6c, 0x6c, 0x6c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0xfe, 0xc6, 0x60, 0x30, 0x18,
  0x30, 0x60, 0xc6, 0xfe, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x7e, 0xd8, 0xd8,
  0xd8, 0xd8, 0xd8, 0x70, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x66, 0x66, 0x66, 0x66,
  0x66, 0x7c, 0x60, 0x60, 0xc0, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x76, 0xdc, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x7e, 0x18, 0x3c, 0x66, 0x66,
  0x66, 0x3c, 0x18, 0x7e, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x38, 0x6c, 0xc6, 0xc6, 0xfe,
  0xc6, 0xc6, 0x6c, 0x38, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x38, 0x6c, 0xc6, 0xc6, 0xc6, 0x6c,
  0x6c, 0x6c, 0x6c, 0xee, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x1e, 0x30, 0x18, 0x0c, 0x3e, 0x66,
  0x66, 0x66, 0x66, 0x3c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x7e, 0xdb, 0xdb,
  0xdb, 0x7e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x03, 0x06, 0x7e, 0xdb, 0xdb,
  0xf3, 0x7e, 0x60, 0xc0, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x1c, 0x30, 0x60, 0x60, 0x7c, 0x60,
  0x60, 0x60, 0x30, 0x1c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0xc6,
  0xc6, 0xc6, 0xc6, 0xc6, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0xfe, 0x00, 0x00, 0xfe,
  0x00, 0x00, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x7e, 0x18,
  0x18, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x30, 0x18, 0x0c, 0x06, 0x0c,
  0x18, 0x30, 0x00, 0x7e, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x0c, 0x18, 0x30, 0x60, 0x30,
  0x18, 0x0c, 0x00, 0x7e, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x0e, 0x1b, 0x1b, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0xd8, 0xd8, 0xd8, 0x70, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x7e,
  0x00, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x76, 0xdc, 0x00,
  0x76, 0xdc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x38, 0x6c, 0x6c, 0x38, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18,
  0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x0f, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0xec,
  0x6c, 0x6c, 0x3c, 0x1c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0xd8, 0x6c, 0x6c, 0x6c, 0x6c, 0x6c, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x70, 0xd8, 0x30, 0x60, 0xc8, 0xf8, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x7c, 0x7c, 0x7c, 0x7c,
  0x7c, 0x7c, 0x7c, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

uint8_t int10_font_14_alternate[20 * 15 + 1] = {
  0x1d,
  0x00, 0x00, 0x00, 0x00, 0x24, 0x66, 0xff,
  0x66, 0x24, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x22,
  0x00, 0x63, 0x63, 0x63, 0x22, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x2b,
  0x00, 0x00, 0x00, 0x18, 0x18, 0x18, 0xff,
  0x18, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x2d,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x4d,
  0x00, 0x00, 0xc3, 0xe7, 0xff, 0xdb, 0xc3,
  0xc3, 0xc3, 0xc3, 0xc3, 0x00, 0x00, 0x00,
  0x54,
  0x00, 0x00, 0xff, 0xdb, 0x99, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00,
  0x56,
  0x00, 0x00, 0xc3, 0xc3, 0xc3, 0xc3, 0xc3,
  0xc3, 0x66, 0x3c, 0x18, 0x00, 0x00, 0x00,
  0x57,
  0x00, 0x00, 0xc3, 0xc3, 0xc3, 0xc3, 0xdb,
  0xdb, 0xff, 0x66, 0x66, 0x00, 0x00, 0x00,
  0x58,
  0x00, 0x00, 0xc3, 0xc3, 0x66, 0x3c, 0x18,
  0x3c, 0x66, 0xc3, 0xc3, 0x00, 0x00, 0x00,
  0x59,
  0x00, 0x00, 0xc3, 0xc3, 0xc3, 0x66, 0x3c,
  0x18, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00,
  0x5a,
  0x00, 0x00, 0xff, 0xc3, 0x86, 0x0c, 0x18,
  0x30, 0x61, 0xc3, 0xff, 0x00, 0x00, 0x00,
  0x6d,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xe6, 0xff,
  0xdb, 0xdb, 0xdb, 0xdb, 0x00, 0x00, 0x00,
  0x76,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xc3, 0xc3,
  0xc3, 0x66, 0x3c, 0x18, 0x00, 0x00, 0x00,
  0x77,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xc3, 0xc3,
  0xdb, 0xdb, 0xff, 0x66, 0x00, 0x00, 0x00,
  0x91,
  0x00, 0x00, 0x00, 0x00, 0x6e, 0x3b, 0x1b,
  0x7e, 0xd8, 0xdc, 0x77, 0x00, 0x00, 0x00,
  0x9b,
  0x00, 0x18, 0x18, 0x7e, 0xc3, 0xc0, 0xc0,
  0xc3, 0x7e, 0x18, 0x18, 0x00, 0x00, 0x00,
  0x9d,
  0x00, 0x00, 0xc3, 0x66, 0x3c, 0x18, 0xff,
  0x18, 0xff, 0x18, 0x18, 0x00, 0x00, 0x00,
  0x9e,
  0x00, 0xfc, 0x66, 0x66, 0x7c, 0x62, 0x66,
  0x6f, 0x66, 0x66, 0xf3, 0x00, 0x00, 0x00,
  0xf1,
  0x00, 0x00, 0x18, 0x18, 0x18, 0xff, 0x18,
  0x18, 0x18, 0x00, 0xff, 0x00, 0x00, 0x00,
  0xf6,
  0x00, 0x00, 0x18, 0x18, 0x00, 0x00, 0xff,
  0x00, 0x00, 0x18, 0x18, 0x00, 0x00, 0x00,
  0x00
};

uint8_t int10_font_16_alternate[19 * 17 + 1] = {
  0x1d,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x24, 0x66, 0xff,
  0x66, 0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x30,
  0x00, 0x00, 0x3c, 0x66, 0xc3, 0xc3, 0xdb, 0xdb,
  0xc3, 0xc3, 0x66, 0x3c, 0x00, 0x00, 0x00, 0x00,
  0x4d,
  0x00, 0x00, 0xc3, 0xe7, 0xff, 0xff, 0xdb, 0xc3,
  0xc3, 0xc3, 0xc3, 0xc3, 0x00, 0x00, 0x00, 0x00,
  0x54,
  0x00, 0x00, 0xff, 0xdb, 0x99, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00,
  0x56,
  0x00, 0x00, 0xc3, 0xc3, 0xc3, 0xc3, 0xc3, 0xc3,
  0xc3, 0x66, 0x3c, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x57,
  0x00, 0x00, 0xc3, 0xc3, 0xc3, 0xc3, 0xc3, 0xdb,
  0xdb, 0xff, 0x66, 0x66, 0x00, 0x00, 0x00, 0x00,
  0x58,
  0x00, 0x00, 0xc3, 0xc3, 0x66, 0x3c, 0x18, 0x18,
  0x3c, 0x66, 0xc3, 0xc3, 0x00, 0x00, 0x00, 0x00,
  0x59,
  0x00, 0x00, 0xc3, 0xc3, 0xc3, 0x66, 0x3c, 0x18,
  0x18, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00,
  0x5a,
  0x00, 0x00, 0xff, 0xc3, 0x86, 0x0c, 0x18, 0x30,
  0x60, 0xc1, 0xc3, 0xff, 0x00, 0x00, 0x00, 0x00,
  0x6d,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xe6, 0xff, 0xdb,
  0xdb, 0xdb, 0xdb, 0xdb, 0x00, 0x00, 0x00, 0x00,
  0x76,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xc3, 0xc3, 0xc3,
  0xc3, 0x66, 0x3c, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x77,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xc3, 0xc3, 0xc3,
  0xdb, 0xdb, 0xff, 0x66, 0x00, 0x00, 0x00, 0x00,
  0x78,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xc3, 0x66, 0x3c,
  0x18, 0x3c, 0x66, 0xc3, 0x00, 0x00, 0x00, 0x00,
  0x91,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x6e, 0x3b, 0x1b,
  0x7e, 0xd8, 0xdc, 0x77, 0x00, 0x00, 0x00, 0x00,
  0x9b,
  0x00, 0x18, 0x18, 0x7e, 0xc3, 0xc0, 0xc0, 0xc0,
  0xc3, 0x7e, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x9d,
  0x00, 0x00, 0xc3, 0x66, 0x3c, 0x18, 0xff, 0x18,
  0xff, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x9e,
  0x00, 0xfc, 0x66, 0x66, 0x7c, 0x62, 0x66, 0x6f,
  0x66, 0x66, 0x66, 0xf3, 0x00, 0x00, 0x00, 0x00,
  0xab,
  0x00, 0xc0, 0xc0, 0xc2, 0xc6, 0xcc, 0x18, 0x30,
  0x60, 0xce, 0x9b, 0x06, 0x0c, 0x1f, 0x00, 0x00,
  0xac,
  0x00, 0xc0, 0xc0, 0xc2, 0xc6, 0xcc, 0x18, 0x30,
  0x66, 0xce, 0x96, 0x3e, 0x06, 0x06, 0x00, 0x00,
  0x00
};
// clang-format on
