/*-------------------------------------------------------------------------
 * sgsub.c
 * 
 * SCSI Generic subroutines 
 *
 * Author:  Andy Cress   <arcress@users.sourceforge.net>
 * Copyright (c) 2001-2008  Intel Corp.
 * 
 * 04/30/01 ARCress - created
 * 05/18/01 ARCress - added send_scsicdb() routine
 * 07/09/01 ARCress - longer timeout (15 min) for scsi_format()
 * 08/31/01 ARCress - added noglist param to scsi_format()
 * 10/11/01 ARCress - added some debug output
 * 01/03/03 ARCress - added stop_unit function
 *-------------------------------------------------------------------------*/
/*-------------------------------------------------------------------------
Copyright (c) 2002, Intel Corporation
All rights reserved.

Redistribution and use in source and binary forms, with or without 
modification, are permitted provided that the following conditions are met:

  a.. Redistributions of source code must retain the above copyright notice, 
      this list of conditions and the following disclaimer. 
  b.. Redistributions in binary form must reproduce the above copyright notice,
      this list of conditions and the following disclaimer in the documentation 
      and/or other materials provided with the distribution. 
  c.. Neither the name of Intel Corporation nor the names of its contributors 
      may be used to endorse or promote products derived from this software 
      without specific prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE 
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR 
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON 
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
  -------------------------------------------------------------------------*/
#include <unistd.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <scsi/sg.h>
#include <scsi/scsi.h>

#include "sgsub.h"

extern int sg_chk_err(FILE * fdout, const char *leadin, int masked_status,
		      int host_status, int driver_status,
		      const unsigned char *sense_buffer, int sb_len);
/* now defined in sgsub.h */
//#define uchar  unsigned char
//#define ushort unsigned short
//#define ulong  unsigned long
//
// #define HDR    sizeof(struct sg_header)	/*=36 if 16 sense bytes*/
/* Define ioctl functions */
#define SG_GET_RESERVED_SIZE 0x2272
#define SG_SET_RESERVED_SIZE 0x2275
#define SG_GET_VERSION_NUM 0x2282
#define SG_NEXT_CMD_LEN 0x2283
/* Type values for scsi_reset():    */
#define SCSI_RESET_NOTHING   0	/*(check status) */
#define SCSI_RESET_DEVICE    1
#define SCSI_RESET_BUS       2
#define SCSI_RESET_HOST      3
/* return values for these subroutines, if >0 would be errno. */
#define SUNKN_ERR   -1
#define SCHECK_CND  -2
#define SWRITE_ERR  -3
#define SREAD_ERR   -4
#define SIOCTL_ERR  -5
#define INQ_CLEN        6
#define INQ_RLEN       96
#define WB_CMD_LEN     10
#define RCAP_CMD_LEN   10
#define RCAP_REPLY_LEN  8
#define MY_PACK_ID   1234
#define TUR_CMD_LEN     6
#define TUR_PACK_ID  1235
#define SER_LEN   16		/* 4 + serial number length */
char fshowerrs = 1;	/* global flag, 1 = show error messages, 0 = not */
char fdebugsub = 0;	/* should match fdebug in sg{main}.c */
FILE *dbgout = NULL;	/* should match fdlog  in sg{main}.c */
  	/* Use global cmdbuf to allow get_sense() later */
uchar cmdbuf[HDR + 250];  /*HDR+6+96 etc. */
struct sg_header *sgbufp = (struct sg_header *) cmdbuf;

static void showdbg(const char * format, ...)
{
    va_list vptr;
    FILE *fd;
    if (dbgout != NULL) fd = dbgout;
    else fd = stdout;
    va_start(vptr, format);
    vfprintf(fd, format, vptr);
    va_end(vptr);
}

int get_sense(int sts, uchar * buf)
{
    int ret;
    if (sts < 0 && sts != SIOCTL_ERR) {	/* buffer pointer is valid */
	/* buf must be at least SG_MAX_SENSE bytes */
	memcpy(buf, sgbufp->sense_buffer, SG_MAX_SENSE);
	ret = 0;
    } else
	ret = -1;
    return (ret);
}

int sense_report(int sts, const char *errmsg, uchar * bufp)
{
    int ret = 0;
    struct sg_header *hp;
    hp = (struct sg_header *) bufp;
    /* write to stderr first */
    ret = sg_chk_err(stderr, errmsg, hp->target_status, hp->host_status,
		     hp->driver_status, hp->sense_buffer, SG_MAX_SENSE);
    if (dbgout != NULL) {
	/* then write to log file */
	ret = sg_chk_err(dbgout, errmsg, hp->target_status, hp->host_status,
		       hp->driver_status, hp->sense_buffer, SG_MAX_SENSE);
    }
    /* Handle some sense errors here (2/4/2, etc) */
    if (ret == 0 && sts != 0) ret = sts;
    return (ret);
}

void i2h(uchar * chp, int len, char *str)
{
    uchar rgascii[] = { "0123456789abcdef" };
    int i, j;
    uchar k;
    for (i = 0, j = 0; i < len; i++) {
	k = (chp[i] & 0xf0) >> 4;
	str[j++] = rgascii[k];
	k = chp[i] & 0x0f;
	str[j++] = rgascii[k];
	str[j++] = ' ';
    }
    str[j] = 0;			/* stringify */
}				/*end i2h() */

static void dumpbuf(FILE * fdout, uchar * bufp, int mlen)
{
    int i;
    char abuf[5];
    if (fdout == NULL) fdout = stdout;
    if (fdebugsub) 
	fprintf(fdout, "len = %d, bufp = %p, ", mlen, bufp);
    fprintf(fdout, " Buffer: ");
    for (i = 0; i < mlen; i++) {
	if (i % 16 == 0)	/* start a new line */
	    fprintf(fdout, "\n%04x: ", i);
	i2h(&bufp[i], 1, abuf);
	fprintf(fdout, "%s", abuf);
    }
    fprintf(fdout, "\n");
    return;
}				/*end dumpbuf() */

int get_sernum(int sgfd, uchar * buf, int rlen)
{
    int ret, sts = 0;
    uchar inqserBlk[INQ_CLEN] = { 0x12, 0x01, 0x80, 0, SER_LEN, 0 };
    struct sg_header *isgbuf = (struct sg_header *) cmdbuf;
    int cmdlen, rsplen, plen;
    uchar *bufp;
    cmdlen = HDR + INQ_CLEN;
    rsplen = HDR + SER_LEN;
    inqserBlk[4] = SER_LEN;
    memset(cmdbuf, 0, HDR);	/* zeros out all defaults */
    memcpy(cmdbuf + HDR, inqserBlk, INQ_CLEN);
    isgbuf->reply_len = rsplen;
    isgbuf->sense_buffer[0] = 0;
    isgbuf->twelve_byte = 0;
    isgbuf->other_flags = 0;
    sgbufp = (struct sg_header *) cmdbuf;
    ret = write(sgfd, cmdbuf, cmdlen);
    if (ret < 0)
	sts = SWRITE_ERR;
    else {
	ret = read(sgfd, cmdbuf, rsplen);
	if (ret < 0)
	    sts = SREAD_ERR;
    }
    if (0 != isgbuf->sense_buffer[0])
	sts = SCHECK_CND;
    if (sts == 0) {
	bufp = cmdbuf + HDR + 8;
	plen = bufp[3];
	if (plen > rlen)
	    plen = rlen;
	memcpy(buf, bufp, plen);	/* this is safer than using ret */
	buf[plen] = 0;		/* stringify */
	if (fdebugsub)
	    showdbg( "sernum: len=%d, buf: %s\n", plen, buf);
	/* May need to debug the format of this. */
    } else if (fshowerrs) {	/* display the error messages */
	sense_report(sts, "get_sernum", cmdbuf);
    }
    return (sts);
}				/* end get_sernum() */

int scsi_inquiry(int sgfd, uchar * buf, int rlen)
{
    int ret, sts = 0;
    uchar inqCmdBlk[INQ_CLEN] = { 0x12, 0, 0, 0, INQ_RLEN, 0 };
    struct sg_header *isgbuf = (struct sg_header *) cmdbuf;
    int cmdlen, rsplen;
    int f;
    if (rlen > INQ_RLEN)
	rlen = INQ_RLEN;	/* throttle rlen if too big */
    cmdlen = HDR + INQ_CLEN;
    rsplen = HDR + INQ_RLEN;	/* must set full length to avoid driver hang */
#ifdef MAKEFAULT
    rsplen = HDR + rlen;	/* will cause driver hang with stock RH62 aic7xxx */
#endif
    if (fdebugsub)
	showdbg( "inq: HDR=%d, rlen=%d, RLEN=%d\n", HDR, rlen,
		INQ_RLEN);
    f = fcntl(sgfd, F_GETFL);
    f = f & (~O_NONBLOCK);	/* set to blocking mode */
    ret = fcntl(sgfd, F_SETFL, f);
    memset(cmdbuf, 0, rsplen);	/* zeros out all defaults */
    memcpy(cmdbuf + HDR, inqCmdBlk, INQ_CLEN);
    isgbuf->reply_len = rsplen;
    isgbuf->sense_buffer[0] = 0;
    isgbuf->twelve_byte = 0;
    isgbuf->other_flags = 0;
    sgbufp = (struct sg_header *) cmdbuf;
    if (fdebugsub) {
	int i;
	showdbg( "scsi_inquiry cdb (len=%d): ", INQ_CLEN);
	for (i = HDR; i < cmdlen; i++)
	    showdbg( "%02x ", cmdbuf[i]);
	showdbg( "\n");
    }
    ret = write(sgfd, cmdbuf, cmdlen);
    if (ret < 0)
	sts = SWRITE_ERR;
    else {
	ret = read(sgfd, cmdbuf, rsplen);
	if (fdebugsub)
	    showdbg( "scsi_inquiry: read ret = %d\n", ret);
	if (ret < 0)
	    sts = SREAD_ERR;
    }
    if (sts == 0) {
	memcpy(buf, cmdbuf + HDR, rlen);
	if (fdebugsub) dumpbuf(dbgout,cmdbuf+HDR,rlen);
	}
    if (0 != isgbuf->sense_buffer[0])
	sts = SCHECK_CND;
    if (sts != 0 && (fshowerrs || fdebugsub)) {	/* display the error messages */
	sense_report(sts, "scsi_inquiry", cmdbuf);
    }
    return (sts);
}				/* end scsi_inquiry() */

int test_unit_ready(int sgfd, char fshowmsg)
{
    uchar turCmdBlk[TUR_CMD_LEN] = { 0x00, 0, 0, 0, 0, 0 };
    struct sg_header *sghp = (struct sg_header *) cmdbuf;
    int sts = 0;
    int ret = 0;
    int k;
    int cmdlen, rsplen;
    cmdlen = HDR + TUR_CMD_LEN;
    rsplen = HDR;
    k = TUR_PACK_ID;		/* can be used as a sequence number */
    sghp->reply_len = rsplen;
    sghp->twelve_byte = 0;
    sghp->other_flags = 0;	/* some apps assume this is done ?!? */
    sghp->sense_buffer[0] = 0;	/* only needed for old sg driver */
    sghp->pack_id = k;
    memcpy(cmdbuf + HDR, turCmdBlk, TUR_CMD_LEN);
    sgbufp = (struct sg_header *) cmdbuf;
    ret = write(sgfd, cmdbuf, cmdlen);
    if (ret < 0)
	sts = SWRITE_ERR;
    else {
	ret = read(sgfd, cmdbuf, rsplen);
	if (ret < 0)
	    sts = SREAD_ERR;
    }
    if (fdebugsub && sts == 0) {
	if (k != sghp->pack_id)	/* this shouldn't happen */
	    showdbg(
		    "test_unit_ready pack_id mismatch, wanted=%d, got=%d\n",
		    k, sghp->pack_id);
    }
    if (0 != sghp->sense_buffer[0])
	sts = SCHECK_CND;
    if (sts != 0 && fshowmsg) {	/* display the error messages */
	sense_report(sts, "test_unit_ready", cmdbuf);
    }
    return (sts);
}				/*end test_unit_ready() */

/* 
 * read_capacity
 * This function returns the number of even 0x100-byte chunks in dsize. 
 * Divide by 4 to get the value in 1K increments.
 */
int read_capacity(int sgfd, ulong * dsize)
{
    int ret, sts = 0;
    uchar rcapCmdBlk[RCAP_CMD_LEN] = { 0x25, 0, 0, 0, 0, 0, 0, 0, 0, 0 };
    struct sg_header *sghp = (struct sg_header *) cmdbuf;
    uchar *buffp = cmdbuf + HDR;
    int cmdlen, rsplen;		// int clen;
    rsplen = HDR + RCAP_REPLY_LEN;
    sghp->reply_len = rsplen;
    sghp->pack_id = MY_PACK_ID;
    sghp->twelve_byte = 0;
    sghp->other_flags = 0;	/* some apps assume this is done ?!? */
    sghp->sense_buffer[0] = 0;
    cmdlen = HDR + RCAP_CMD_LEN;
    memcpy(cmdbuf + HDR, rcapCmdBlk, RCAP_CMD_LEN);
    sgbufp = (struct sg_header *) cmdbuf;
    ret = write(sgfd, cmdbuf, cmdlen);
    if (ret < 0) sts = SWRITE_ERR;
    else {
	ret = read(sgfd, cmdbuf, rsplen);
	if (ret < 0) sts = SREAD_ERR;
    }
    if (sts == 0) {		/* decode capacity to num 0x100 chunks */
	ulong last_blk_addr, nblks;
	ulong block_size, b;
	last_blk_addr = 0xff & buffp[3];
	last_blk_addr |= (0xff & buffp[2]) << 8;
	last_blk_addr |= (0xff & buffp[1]) << 16;
	last_blk_addr |= (0xff & buffp[0]) << 24;
	nblks = last_blk_addr + 1;
	block_size = 0xff & buffp[7];
	block_size |= (0xff & buffp[6]) << 8;
	block_size |= (0xff & buffp[5]) << 16;
	block_size |= (0xff & buffp[4]) << 24;
	b = block_size / 0x100;	/* avoid overflow for >8G */
	*dsize = (nblks * b);
    }
    if (sghp->sense_buffer[0] != 0) sts = SCHECK_CND;
    if (sts != 0 && fshowerrs) {	/* display the error messages */
	sense_report(sts, "read_capacity", cmdbuf);
    }
    return (sts);
}				/* end read_capacity() */

/*-----------------------------------------------
  write_buffer 
     Send a write buffer command with the firmware
     image in it to update the disk firmware.
     It will be big and take a longer time.
  For aic7xxx: 
  scsi0, channel=0, device=0, lun=0,  scsi_type=0
  SG sg_tablesize=128, timeout=6000, version_num=20136
  number of sectors=17916240, sector size=512                   
  So res_size default of ~9MB should be large enough.
  But timeout of 60 seconds needs to increase.
 ------------------------------------------------*/
int write_buffer(int sg_fd, uchar * buf, ulong len, uchar mod, uchar bufid)
{
    uchar wbCmdBlk[WB_CMD_LEN] = { 0x3B, 0x05, 0x00, 0, 0, 0, 0, 0, 0, 0 };
    uchar *bigbuf;
    struct sg_header *sghp;
    int ret, sts = 0;
    int timeout, waittime, k;
    ulong res_size, sg_tablesize, maxbufsz;
    ulong cmdlen;
    int rsplen;
    ulong start;

    start = 0L;
    cmdlen = HDR + WB_CMD_LEN + len;
    if (fdebugsub) {
        if (dbgout == NULL) dbgout = stdout;
	showdbg(
	     "write_buffer: HDR=%d, clen=%d, buflen=%ld, total=%ld, bufid=%d\n",
	     HDR, WB_CMD_LEN, len, cmdlen, bufid);
    }
    /* Set the timeout longer. */
    k = 100;
    timeout = ioctl(sg_fd, SG_GET_TIMEOUT, &k);
    if (timeout < 0)
	return (SIOCTL_ERR);
    /* timeout is in jitters (10ms increments) */
    /* time-to-wait should be 120sec = 120000ms = 12000 jitters */
    waittime = 12000;
    if (timeout < waittime) {
	sts = ioctl(sg_fd, SG_SET_TIMEOUT, &waittime);
	if (fdebugsub) {
	    showdbg(
		    "writebuffer: timeout was = %d, set to %d, sts = %d\n",
		    timeout, waittime, sts);
	}
	if (sts < 0)
	    return (SIOCTL_ERR);
    }
    /* Get the Reserved buffer size  */
    if (ioctl(sg_fd, SG_GET_RESERVED_SIZE, &res_size) < 0)
	return (SIOCTL_ERR);
    /* Get the number of Scatter-Gather entries  */
    if (ioctl(sg_fd, SG_GET_SG_TABLESIZE, &sg_tablesize) < 0)
	return (SIOCTL_ERR);
    if (sg_tablesize < 1)
	maxbufsz = res_size;
    else
	maxbufsz = res_size * sg_tablesize;
    /* Set the reserved buffer large enough. */
    if (maxbufsz < len) {	/* too small, need to change it */
	ulong newrsize;
	if (sg_tablesize < 1)
	    newrsize = len;
	else
	    newrsize = (len / sg_tablesize) + 1;
	if (fdebugsub) {
	    showdbg(
		    "writebuffer: rbuffer size = %ld, s/g table = %ld, wb size = %ld, newrbuf = %ld\n",
		    res_size, sg_tablesize, len, newrsize);
	}
	if (ioctl(sg_fd, SG_SET_RESERVED_SIZE, &newrsize) < 0)
	    return (SIOCTL_ERR);
    }
    /* Perform the write buffer command. */
    bigbuf = malloc(cmdlen);
    if (bigbuf == NULL)
	return (errno);
    sghp = (struct sg_header *) bigbuf;
    rsplen = HDR;		// + WB_CMD_LEN;
    sghp->pack_len = 0;		/* don't care */
    sghp->reply_len = rsplen;
    sghp->twelve_byte = 0;
    sghp->result = 0;
    sghp->pack_id = k;
    sghp->other_flags = 0;
    sghp->sense_buffer[0] = 0;
    wbCmdBlk[1] = mod;		/* set download mode from param  */
    wbCmdBlk[2] = bufid;	/* buffer id */
    wbCmdBlk[3] = 0xff & (start >> 16);
    wbCmdBlk[4] = 0xff & (start >> 8);
    wbCmdBlk[5] = 0xff & start;	/* cdb[3,4,5] = buffer offset (start) */
    wbCmdBlk[6] = 0xff & (len >> 16);
    wbCmdBlk[7] = 0xff & (len >> 8);
    wbCmdBlk[8] = 0xff & len;
    memcpy(bigbuf + HDR, wbCmdBlk, WB_CMD_LEN);
    memcpy(bigbuf + HDR + WB_CMD_LEN, buf, len);
    sgbufp = (struct sg_header *) bigbuf;
    if (fdebugsub)
	showdbg(
		"writebuffer cdb: %02x %02x %02x %02x %02x %02x %02x %02x %02x %02x\n",
		wbCmdBlk[0], wbCmdBlk[1], wbCmdBlk[2], wbCmdBlk[3],
		wbCmdBlk[4], wbCmdBlk[5], wbCmdBlk[6], wbCmdBlk[7],
		wbCmdBlk[8], wbCmdBlk[9]);
    ret = write(sg_fd, bigbuf, cmdlen);
    if (ret < 0)
	sts = SWRITE_ERR;
    else {
	ret = read(sg_fd, bigbuf, rsplen);
	if (ret < 0)
	    sts = SREAD_ERR;
    }
    /* check for errors */
    if (sts == 0 && sghp->sense_buffer[0] != 0)
	sts = SCHECK_CND;
    if (sts != 0 && fshowerrs) {	/* display the error messages */
	int slen;
	slen = cmdlen;
	if (slen > 80)
	    slen = 80;		/* limit how much of the buffer to show */
	sense_report(sts, "write_buffer", bigbuf);
	if (fdebugsub) {
	    dumpbuf(stdout, bigbuf, slen);
	    if (dbgout != NULL)
		dumpbuf(dbgout, bigbuf, slen);
	}
    }
    free(bigbuf);
    if (timeout < waittime) {
	waittime = 6000;	/* normal */
	ret = ioctl(sg_fd, SG_SET_TIMEOUT, &waittime);
	if (fdebugsub) {
	    showdbg(
		    "writebuffer: set timeout back to %d, sts = %d\n",
		    waittime, ret);
	}
    }
    return (sts);
}				/* end write_buffer() */

#ifdef TEST
int read_safte(int sgfd, int mode, uchar * buf, int len)
{
    uchar cdb[10] = { 0x3C, 0x01, 0x00, 0, 0, 0, 0, 0x00, 0x50, 0 };
    int ret, sts = 0;
    int rsplen, cmdlen;
    uchar saftebuf[HDR+6 + 200];
    struct sg_header *sghp = (struct sg_header *) saftebuf;
    int max_data = 200;
    if (len > max_data) len = max_data;
    rsplen = HDR + len;
    cmdlen = HDR + sizeof(cdb);
    sghp->reply_len = rsplen;
    sghp->twelve_byte = 0;
    sghp->other_flags = 0;	/* some apps assume this is done ?!? */
    sghp->sense_buffer[0] = 0;	/* only needed for old sg driver */
    sghp->pack_id = 12397;
    memcpy(saftebuf + HDR, cdb, sizeof(cdb));
    saftebuf[HDR + 2] = mode;
    saftebuf[HDR + 7] = (len & 0xff00) >> 8;
    saftebuf[HDR + 8] = len & 0x0ff;
    sgbufp = (struct sg_header *) saftebuf;
    ret = write(sgfd, saftebuf, cmdlen);
    if (ret < 0) sts = SWRITE_ERR;
    else {
	ret = read(sgfd, saftebuf, rsplen);
	if (ret < 0) sts = SREAD_ERR;
    }
    if (sts == 0) {
	if (fdebugsub)
	    printf("read_safte: requested %d, sent = %d, returned = %d\n",
		   len, rsplen, ret);
	if (ret > len) ret = len;
	memcpy(buf, saftebuf + HDR, ret);
    }
    if (0 != sghp->sense_buffer[0])
	sts = SCHECK_CND;
    if (sts != 0 && fshowerrs) {	/* display the error messages */
	sense_report(sts, "read_safte", saftebuf);
    }
    return (sts);
}				/* end read_safte() */
#endif

uchar dlistbuf[HDR + 10 + 0xFFFF];	/* big defect list buffer */

int get_defects(int sgfd, uchar * buf, int len, char fplist)
{
    uchar cdb[10] = { 0x37, 0, 0x15, 0, 0, 0, 0, 0xFF, 0xFF, 0 };
    int ret, sts = 0;
    int rsplen, cmdlen;
    struct sg_header *sghp = (struct sg_header *) dlistbuf;
    uchar param;
    int max_data = 0xFFFF;
    if (fplist)
	param = 0x15;		/* get plist in Phys DefListFmt */
    else
	param = 0x0d;		/* get glist in Phys DefListFmt */
    if (len > max_data)
	len = max_data;
    rsplen = HDR + len;
    cmdlen = HDR + 10;
    sghp->reply_len = rsplen;
    sghp->twelve_byte = 0;
    sghp->other_flags = 0;	/* some apps assume this is done ?!? */
    sghp->sense_buffer[0] = 0;	/* only needed for old sg driver */
    sghp->pack_id = 12397;
    memcpy(dlistbuf + HDR, cdb, 10);
    /* cdb[2] values:   */
    /*    0x10 = PList only  */
    /*    0x08 = GList only  */
    /*    0x00 = Block DefListFmt (4 bytes) LBA */
    /*    0x05 = Phys DefListFmt  (8 bytes) Cyl, Head, Sector */
    dlistbuf[HDR + 2] = param;
    dlistbuf[HDR + 7] = (len & 0xff00) >> 8;
    dlistbuf[HDR + 8] = len & 0x0ff;
    sgbufp = (struct sg_header *) dlistbuf;
    ret = write(sgfd, dlistbuf, cmdlen);
    if (ret < 0)
	sts = SWRITE_ERR;
    else {
	ret = read(sgfd, dlistbuf, rsplen);
	if (ret < 0)
	    sts = SREAD_ERR;
    }
    if (sts == 0) {
	if (fdebugsub)
	    printf("get_defects: requested %d, sent = %d, returned = %d\n",
		   len, rsplen, ret);
	if (ret > len)
	    ret = len;
	memcpy(buf, dlistbuf + HDR, ret);
    }
    if (0 != sghp->sense_buffer[0])
	sts = SCHECK_CND;
    if (sts != 0 && fshowerrs) {	/* display the error messages */
	sense_report(sts, "get_defects", dlistbuf);
    }
    return (sts);
}				/* end get_defects() */

int mode_sense(int sgfd, uchar page, uchar * buf)
{
    uchar cdb[6] = { 0x1A, 0, 0x3F, 0, 0x9F, 0 };
    int ret, sts = 0;
    int rsplen;
    uchar modebuf[HDR + 250];	/*HDR+6+240 */
    struct sg_header *sghp = (struct sg_header *) modebuf;
    int max_data = 240;
    /* Typical data len for all mode pages is 0x9f = 159. */
    /* max len for each page is   4 + 8 + 2 + 0x16 = 36. */
    rsplen = HDR + max_data;
    sghp->reply_len = rsplen;
    sghp->twelve_byte = 0;
    sghp->other_flags = 0;	/* some old apps assume this is done */
    sghp->sense_buffer[0] = 0;	/* only needed for old sg driver */
    sghp->pack_id = 12367;
    memcpy(modebuf + HDR, cdb, 6);
    /* 
     * page control, 3F means get all active pages 
     *   [PC(bits7,6)=00b, current value] 
     */
    modebuf[HDR + 2] = page;
    modebuf[HDR + 4] = max_data;
    sgbufp = (struct sg_header *) modebuf;
    ret = write(sgfd, modebuf, HDR + 6);
    if (ret < 0)
	sts = SWRITE_ERR;
    else {
	ret = read(sgfd, modebuf, rsplen);
	if (ret < 0)
	    sts = SREAD_ERR;
    }
    if (sts == 0) {
	if (ret > max_data)
	    ret = max_data;
	memcpy(buf, modebuf + HDR, ret);
    }
    if (0 != sghp->sense_buffer[0])
	sts = SCHECK_CND;
    if (sts != 0 && fshowerrs) {	/* display the error messages */
	sense_report(sts, "mode_sense", modebuf);
    }
    return (sts);
}				/* end mode_sense() */

/* 
 * mode_select
 * #define CMD_SIZE(opcode) scsi_csize[((opcode) >> 5) & 7]
 * const unsigned char scsi_csize[8] = { 6, 10, 10, 12, 12, 12, 10, 10 };
 * CMD_SIZE(0x15) = 6
 * CMD_SIZE(0x3B) = 10
 */
int mode_select(int sgfd, uchar * buf, uchar len)
{
    uchar cdb[6] = { 0x15, 0x11, 0, 0, 0x1C, 0 };
    int max_data = 64;  /* max len for each page is usu: 4+8+2+ 0x16 = 36. */
    uchar modebuf[HDR + 70];  /* HDR + 6 + max_data */
    struct sg_header *sghp = (struct sg_header *) modebuf;
    int ret, sts = 0;
    int cmdlen;

    if (len > max_data) {
        if (fdebugsub) 
	   showdbg( "mode_select: len=%d truncated to max %d\n", len,max_data);
	len = max_data;
    }
    cmdlen = HDR + 6 + len;
    sghp->reply_len = HDR;
    sghp->twelve_byte = 0;
    sghp->other_flags = 0;	/* some apps assume this is done ?!? */
    sghp->sense_buffer[0] = 0;	/* only needed for old sg driver */
    sghp->pack_id = 12368;
    // cdb[1]=0x11;  /* 0x10=PF (std fmt), 0x01=SP save pages */
    cdb[4] = len;
    memcpy(modebuf + HDR, cdb, 6);
    memcpy(modebuf + HDR + 6, buf, len);
    sgbufp = (struct sg_header *) modebuf;
    if (fdebugsub) {
	showdbg( "mode_select: len=%d page=0x%02x\n", len, buf[12]);
	dumpbuf(dbgout, modebuf, cmdlen);
    }
    ret = write(sgfd, modebuf, cmdlen);
    if (ret < 0)
	sts = SWRITE_ERR;
    else {
	ret = read(sgfd, modebuf, HDR);
	if (ret < 0)
	    sts = SREAD_ERR;
    }
    if (fdebugsub)
	printf("mode_select: write/read done, sts = %d\n", sts);
    if (sts == 0 && sghp->sense_buffer[0] != 0)
	sts = SCHECK_CND;
    if (sts != 0 && fshowerrs) {	/* display the error messages */
	if (fdebugsub) {
	    showdbg( "mode_select: sts = %d\n", sts);
	    dumpbuf(dbgout, modebuf, HDR);
	}
	sense_report(sts, "mode_select", modebuf);
    }
    return (sts);
}				/* end mode_select() */

int start_unit(int sgfd)
{
    uchar cdb[6] = { 0x1B, 0, 0, 0, 0x01, 0 };
    struct sg_header *sghp = (struct sg_header *) cmdbuf;
    int ret, sts = 0;
    sghp->reply_len = HDR;
    sghp->twelve_byte = 0;
    sghp->other_flags = 0;	/* some apps assume this is done ?!? */
    sghp->sense_buffer[0] = 0;	/* only needed for old sg driver */
    sghp->pack_id = 12359;
    memcpy(cmdbuf + HDR, cdb, 6);
    sgbufp = (struct sg_header *) cmdbuf;
    ret = write(sgfd, cmdbuf, HDR + 6);
    if (ret < 0) sts = SWRITE_ERR;
    else {
	ret = read(sgfd, cmdbuf, HDR);
	if (ret < 0)
	    sts = SREAD_ERR;
    }
    if (sghp->sense_buffer[0] != 0) sts = SCHECK_CND;
    if (sts != 0 && fshowerrs) {	/* display the error messages */
	sense_report(sts, "start_unit", cmdbuf);
    }
    return (sts);
}				/*end start_unit() */

int stop_unit(int sgfd)
{
    uchar cdb[6] = { 0x1B, 0, 0, 0, 0x00, 0 };
    struct sg_header *sghp = (struct sg_header *) cmdbuf;
    int ret, sts = 0;
    sghp->reply_len = HDR;
    sghp->twelve_byte = 0;
    sghp->other_flags = 0;	/* some apps assume this is done */
    sghp->sense_buffer[0] = 0;	/* only needed for old sg driver */
    sghp->pack_id = 12359;
    memcpy(cmdbuf + HDR, cdb, 6);
    sgbufp = (struct sg_header *) cmdbuf;
    ret = write(sgfd, cmdbuf, HDR + 6);
    if (ret < 0) sts = SWRITE_ERR;
    else {
	ret = read(sgfd, cmdbuf, HDR);
	if (ret < 0)
	    sts = SREAD_ERR;
    }
    if (sghp->sense_buffer[0] != 0) sts = SCHECK_CND;
    if (sts != 0 && fshowerrs) {	/* display the error messages */
	sense_report(sts, "stop_unit", cmdbuf);
    }
    return (sts);
}				/*end stop_unit() */

/******************************************************
 * scsi_reset(fd, type) 
 * type values below:
 *    SCSI_RESET_NOTHING   0    (check status)
 *    SCSI_RESET_DEVICE    1
 *    SCSI_RESET_BUS       2
 *    SCSI_RESET_HOST      3
 *****************************************************/
int scsi_reset(int sgfd, int type)
{				/* check if reset supported */
    int k, j, res;
    int do_wait = 1;
    switch (type) {
    case 0:			/* SCSI Reset Nothing (check status) */
	k = type;
	break;
    case 1:			/* SCSI Device Reset */
	k = type;
	break;
    case 2:			/* SCSI Bus Reset  */
	k = type;
	break;
    case 3:			/* SCSI Host Reset */
	k = type;
	break;
    default:
	k = SCSI_RESET_BUS;	/* SCSI Bus Reset  (default) */
    }
    res = ioctl(sgfd, SG_SCSI_RESET, &k);
    if (res < 0) {
	if (EBUSY == errno)
	    printf("sg_reset: BUSY, may be resetting now\n");
	else if (EIO == errno)
	    printf
		("sg_reset: requested type of reset may not be available\n");
	else if (EACCES == errno)
	    printf("sg_reset: to do a reset needs root permission\n");
	else
	    printf("sg_reset: SG_SCSI_RESET not supported (%d)\n", errno);
	return 1;
    }
    if (k == SCSI_RESET_NOTHING)
	printf("sg_reset: did nothing, device is normal mode\n");
    else {
	if (SCSI_RESET_DEVICE == k)
	    printf("sg_reset: started device reset\n");
	else if (SCSI_RESET_BUS == k)
	    printf("sg_reset: started bus reset\n");
	else if (SCSI_RESET_HOST == k)
	    printf("sg_reset: started host reset\n");
	if (do_wait) {
	    printf("waiting for the reset to complete...\n");
	    j = 0;
	    k = SCSI_RESET_NOTHING;
	    do {
		if (j != 0)
		    sleep(1);
		res = ioctl(sgfd, SG_SCSI_RESET, &k);
		++j;
	    } while ((res < 0) && (errno == EBUSY));
	    printf("  ... reset returned\n");
	}
    }				/*end else */
    return (res);
}				/* end scsi_reset() */

int scsi_format(int sgfd, uchar patt, int timeout, int noglist)
{
    uchar cdb[6] = { 0x04, 0, 0, 0, 0, 0 };
    uchar fmtbuf[HDR + 20];
    struct sg_header *sghp = (struct sg_header *) fmtbuf;
    int ret, sts = 0;
    int wlen;
    int waittime;
    wlen = HDR + 6 + 10;	/* HDR=36, CDB=6, params=10 */
    memset(fmtbuf, 0, wlen);	/* zeros out cdb & data */
    sghp->reply_len = HDR;
    sghp->twelve_byte = 0;
    sghp->other_flags = 0;	/* some apps assume this is done ?!? */
    sghp->sense_buffer[0] = 0;	/* only needed for old sg driver */
    sghp->pack_id = 12350;
    memcpy(fmtbuf + HDR, cdb, 6);
    if (noglist == 1)		/* format, throwing away the G-List */
	fmtbuf[HDR + 1] = 0x18;	/* FMTDATA, CMPLST, Defects=block fmt */
    /* Note that defect list length is still zero. */
  /**********************************
   * SCSI Format Parameters Buffer
   * Offset Description
   *  0    Reserved   
   *  1    Defect Options (InitPat & Immed) = 0x8A
   *  2    Defect List Length MSB 
   *  3    Defect List Length LSB 
   *  4    Init Pattern Modifier (0=no modify)
   *  5    Init Pattern Type (repeat pattern)  = 0x01
   *  6    Init Pattern Length MSB 
   *  7    Init Pattern Length LSB   = 2
   *  8    Initialization Pattern byte  = 0xF6
   *  9    Initialization Pattern byte  = 0xF6
   * 10+   Defect Descriptors (4 bytes each for block format)
   **********************************/
    fmtbuf[HDR + 6 + 1] = 0x8A;
    fmtbuf[HDR + 6 + 5] = 0x01;
    fmtbuf[HDR + 6 + 7] = 2;
    fmtbuf[HDR + 6 + 8] = patt;	/* usu. 0xF6 */
    fmtbuf[HDR + 6 + 9] = patt;
    sgbufp = (struct sg_header *) fmtbuf;
    if (fdebugsub) {
	showdbg( "scsi_format: patt = %x timeout = %d, nog?=%d\n", 
		patt, timeout, noglist);
	dumpbuf(dbgout, fmtbuf, wlen);
	}
    /* set timeout to user param, should be around 15 minutes (>= 7 min). */
    // waittime = 90000; /* 15 min = 900 sec = 90000 jitters */
    waittime = timeout;
    {
	sts = ioctl(sgfd, SG_SET_TIMEOUT, &waittime);
	if (fdebugsub) {
	    printf("scsi_format: timeout set to %d, sts = %d\n",
		   waittime, sts);
	}
	if (sts < 0)
	    return (SIOCTL_ERR);
    }
    /* now do the SCSI format command */
    ret = write(sgfd, fmtbuf, wlen);
    if (ret < 0)
	sts = SWRITE_ERR;
    else {
	ret = read(sgfd, fmtbuf, HDR);
	if (ret < 0)
	    sts = SREAD_ERR;
    }
    if (sts == 0 && sghp->sense_buffer[0] != 0)
	sts = SCHECK_CND;
    if (sts != 0 && fshowerrs) {	/* display the error messages */
	sense_report(sts, "scsi_format", fmtbuf);
    }
    /* This often results in 06/29/02 errors. */
    /* set timeout back to normal */
    if (!fdebugsub) {
	waittime = 6000;	/* normal 60 sec */
	ret = ioctl(sgfd, SG_SET_TIMEOUT, &waittime);
	if (fdebugsub) {
	    printf("scsi_format: timeout back to %d, sts = %d\n",
		   waittime, ret);
	}
    }
    return (sts);
}				/* end scsi_format() */

int seagate_inquiry(int sgfd, uchar * buf, int rlen)
{				/* Special SCSI inquiry for Seagate disks. */
    uchar scsi_cdb[16];
    uchar seagbuf[HDR + 6 + 0x84];
    struct sg_header *sghp = (struct sg_header *) seagbuf;
    int cmdlen, rsplen;
    int ret;
    int sts = 0;
    cmdlen = HDR + INQ_CLEN;
    rsplen = HDR + INQ_RLEN;	/* must set full length to avoid driver hang */
    if (rlen > 0x84)
	rlen = 0x84;		/* 132 bytes max */
    if (rsplen > rlen)
	rsplen = rlen;
    /* This command returns ~32 bytes:   */
    /*   004: SCSI FW (10180906)         */
    /*   020: Servo ROM   (4045) version */
    /*  Fill in SCSI INQUIRY Command.    */
    scsi_cdb[0] = 0x12;		/* SCSI Inquiry Command */
    scsi_cdb[1] = 0x01;		/* LUN bits 7:5, EVPD bit 0 */
    scsi_cdb[2] = 0xc0;		/* Page code, nonzero only if EVPD set */
    scsi_cdb[3] = 0;
    scsi_cdb[4] = rlen;		/* length */
    scsi_cdb[5] = 0;
    sghp = (struct sg_header *) seagbuf;
    sghp->reply_len = rsplen;
    sghp->twelve_byte = 0;
    sghp->other_flags = 0;	/* some apps assume this is done ?!? */
    sghp->sense_buffer[0] = 0;	/* only needed for old sg driver */
    sghp->pack_id = 12350;
    memcpy(seagbuf + HDR, scsi_cdb, INQ_CLEN);
    sgbufp = (struct sg_header *) seagbuf;
    if (fdebugsub) {
	int i;
	showdbg( "seagate_inquiry cdb (len=%d): ", INQ_CLEN);
	for (i = HDR; i < cmdlen; i++)
	    showdbg( "%02x ", seagbuf[i]);
	showdbg( "\n");
    }
    ret = write(sgfd, seagbuf, cmdlen);
    if (fdebugsub)
	showdbg( "seagate_inquiry: write ret = %d, rlen=%d\n", ret,
		rlen);
    if (ret < 0)
	sts = SWRITE_ERR;
    else {
	ret = read(sgfd, seagbuf, rsplen);
	if (fdebugsub)
	    showdbg( "seagate_inquiry: read ret = %d\n", ret);
	if (ret < 0)
	    sts = SREAD_ERR;
    }
    if (sts == 0) {
	memcpy(buf, seagbuf + HDR, rsplen);
    }
    if (0 != sghp->sense_buffer[0]) sts = SCHECK_CND;
    if (sts != 0 && fshowerrs) {	/* display the error messages */
	sense_report(sts, "seagate_inquiry", seagbuf);
    }
    return (sts);
}				/*end seagate_inquiry() */

int sn_inquiry(int sgfd, uchar pgcode, uchar * buf, int rlen)
{				/* Special SCSI inquiry for Seagate disks. */
    uchar scsi_cdb[16];
    uchar snbuf[HDR + 6 + 0x30];
    struct sg_header *sghp = (struct sg_header *) snbuf;
    int cmdlen, rsplen;
    int ret;
    int sts = 0;
    if (rlen > 0x30)
	rlen = 0x30;
    // pgcode = 0x80;  
    /*  Fill in SCSI INQUIRY Command.    */
    scsi_cdb[0] = 0x12;		/* SCSI Inquiry Command */
    scsi_cdb[1] = 0x01;		/* LUN bits 7:5, EVPD bit 0 */
    scsi_cdb[2] = pgcode;	/* Page code, nonzero only if EVPD set */
    scsi_cdb[3] = 0;
    scsi_cdb[4] = rlen;		/* length */
    scsi_cdb[5] = 0;
    cmdlen = HDR + INQ_CLEN;
    rsplen = HDR + rlen;
    sghp = (struct sg_header *) snbuf;
    sghp->reply_len = rsplen;
    sghp->twelve_byte = 0;
    sghp->other_flags = 0;	/* some apps assume this is done ?!? */
    sghp->sense_buffer[0] = 0;	/* only needed for old sg driver */
    sghp->pack_id = 12350;
    memcpy(snbuf + HDR, scsi_cdb, INQ_CLEN);
    sgbufp = (struct sg_header *) snbuf;
    ret = write(sgfd, snbuf, cmdlen);
    if (ret < 0)
	sts = SWRITE_ERR;
    else {
	ret = read(sgfd, snbuf, rsplen);
	if (ret < 0)
	    sts = SREAD_ERR;
    }
    if (sts == 0) {
	if (rsplen > rlen) rsplen = rlen;
	memcpy(buf, snbuf + HDR, rsplen);
    }
    if (0 != sghp->sense_buffer[0]) sts = SCHECK_CND;
    if (sts != 0 && fshowerrs) {	/* display the error messages */
	sense_report(sts, "sn_inquiry", snbuf);
    }
    return (sts);
}				/*end sn_inquiry() */

int send_scsicdb(int sgfd, uchar * wbuf, int wlen, uchar * rbuf, int rlen)
{
    struct sg_header *sghp;
    int ret, sts = 0;
    /* Assumes that caller has already set up the CDB at offset HDR, 
       and any write data beyond that in wbuf.  */
    sghp = (struct sg_header *) wbuf;
    sghp->reply_len = rlen;
    sghp->twelve_byte = 0;
    sghp->other_flags = 0;	/* some apps assume this is done ?!? */
    sghp->sense_buffer[0] = 0;	/* only needed for old sg driver */
    sghp->pack_id = 12399;
    sgbufp = (struct sg_header *) wbuf;
    ret = write(sgfd, wbuf, wlen);
    if (ret < 0) sts = SWRITE_ERR;
    else {
	ret = read(sgfd, wbuf, rlen);
	if (ret < 0) sts = SREAD_ERR;
    }
    if (0 != sghp->sense_buffer[0]) sts = SCHECK_CND;
    if (sts != 0 && fshowerrs) {	/* display the error messages */
	sense_report(sts, "send_scsicdb", wbuf);
    }
    /* if (sts == 0 && rlen > 0) memcpy(rbuf,wbuf,rlen);  */
    return (sts);
}				/* end send_scsicdb() */

#ifdef OLD
/* now defined in sgsub.h */
#define SG_SENSE1 18
struct sg_hdr1 {
    int pack_len;
    int reply_len;
    int pack_id;
    int result;
    unsigned int twelve_byte:1;
    unsigned int target_status:5;
    unsigned int host_status:8;
    unsigned int driver_status:8;
    unsigned int other_flags:10;
    unsigned char sense_buffer[SG_SENSE1];
};				/* This structure is 38 bytes long on i386 */
#endif

int send_scsicdb1(int sgfd, uchar * wbuf, int wlen, uchar * rbuf, int rlen)
{
    struct sg_hdr1 *sghp;
    uchar *sensebuf;
    int ret, sts = 0;
    /* Assumes that caller has already set up the CDB at offset HDR, 
       and any write data beyond that in wbuf.  */
    sghp = (struct sg_hdr1 *) wbuf;
    sghp->reply_len = rlen;
    sghp->twelve_byte = 0;
    sghp->other_flags = 0;	/* some apps assume this is done ?!? */
    sghp->sense_buffer[0] = 0;	/* only needed for old sg driver */
    sghp->pack_id = 12398;
    sgbufp = (struct sg_header *) wbuf;
    ret = write(sgfd, wbuf, wlen);
    if (ret < 0) sts = SWRITE_ERR;
    else {
	sgbufp = (struct sg_header *) rbuf;
	ret = read(sgfd, rbuf, rlen);	/* Note: rbuf != wbuf */
	if (ret < 0) sts = SREAD_ERR;
    }
    if (0 != sghp->sense_buffer[0]) sts = SCHECK_CND;
    if (sts != 0 && fshowerrs) {	/* display the error messages */
	if (sts == SREAD_ERR) sensebuf = rbuf;
	else		      sensebuf = wbuf;
	sense_report(sts, "send_scsicdb1", sensebuf);
    }
    /* if (sts == 0 && rlen > 0) memcpy(rbuf,wbuf,rlen);  */
    return (sts);
}				/* end send_scsicdb1() */

int set_sg_debug(int sgfd, int dbglvl)
{
    int sts;
    /* set the SG debug level, default = 0, usually set = 10. */
    sts = ioctl(sgfd, SG_SET_DEBUG, &dbglvl);
    if (sts < 0) return (sts);
    else 	 return (0);
}

/* end of sgsub.c */
