/*++

Copyright (C) 2018 Autodesk Inc. (Original Author)

All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice, this
list of conditions and the following disclaimer.
2. Redistributions in binary form must reproduce the above copyright notice,
this list of conditions and the following disclaimer in the documentation
and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

--*/

//////////////////////////////////////////////////////////////////////////////////////////////////////
// buildimplementationpascal .go
// functions to generate Pascal interface classes, implementation stubs and wrapper code that maps to
// the Pascal header.
//////////////////////////////////////////////////////////////////////////////////////////////////////

package main

import (
	"errors"
	"fmt"
	"log"
	"math/rand"
	"path"
	"strings"
)

// BuildImplementationPascal builds Pascal interface classes, implementation stubs and wrapper code that maps to the Pascal header
func BuildImplementationPascal(component ComponentDefinition, outputFolder string, stubOutputFolder string, projectOutputFolder string, implementation ComponentDefinitionImplementation) error {
	//doJournal := len (component.Global.JournalMethod) > 0;
	forceRecreation := false

	NameSpace := component.NameSpace
	libraryname := component.LibraryName
	baseName := component.BaseName

	indentString := getIndentationString(implementation.Indentation)
	stubIdentifier := ""
	if len(implementation.StubIdentifier) > 0 {
		stubIdentifier = "_" + strings.ToLower(implementation.StubIdentifier)
	}

	if stubIdentifier == "" {
		return errors.New("pascal Stub Identifier must not be empty")
	}

	IntfWrapperTypesName := path.Join(outputFolder, baseName+"_types.pas")
	log.Printf("Creating \"%s\"", IntfWrapperTypesName)
	typesWrapperfile, err := CreateLanguageFile(IntfWrapperTypesName, indentString)
	if err != nil {
		return err
	}
	typesWrapperfile.WritePascalLicenseHeader(component,
		fmt.Sprintf("This is an autogenerated Pascal type definition file in order to allow easy\ndevelopment of %s. The functions in this file need to be implemented. It needs to be generated only once.", libraryname),
		true)
	buildPascalTypeDefinition(component, typesWrapperfile, NameSpace, baseName)

	IntfWrapperExceptionName := path.Join(outputFolder, baseName+"_exception.pas")
	log.Printf("Creating \"%s\"", IntfWrapperExceptionName)
	exceptionWrapperfile, err := CreateLanguageFile(IntfWrapperExceptionName, indentString)
	if err != nil {
		return err
	}
	exceptionWrapperfile.WritePascalLicenseHeader(component,
		fmt.Sprintf("This is an autogenerated Pascal exception class definition file in order to allow easy\ndevelopment of %s. The functions in this file need to be implemented. It needs to be generated only once.", libraryname),
		true)
	buildPascalExceptionDefinition(component, exceptionWrapperfile, NameSpace, baseName)

	IntfWrapperInterfaceName := path.Join(outputFolder, baseName+"_interfaces.pas")
	log.Printf("Creating \"%s\"", IntfWrapperInterfaceName)
	interfaceWrapperfile, err := CreateLanguageFile(IntfWrapperInterfaceName, indentString)
	if err != nil {
		return err
	}
	interfaceWrapperfile.WritePascalLicenseHeader(component,
		fmt.Sprintf("This is an autogenerated Pascal interface definition file in order to allow easy\ndevelopment of %s. The functions in this file need to be implemented. It needs to be generated only once.", libraryname),
		true)
	buildPascalInterfaceDefinition(component, interfaceWrapperfile, NameSpace, baseName)

	IntfWrapperExportName := path.Join(outputFolder, baseName+"_exports.pas")
	log.Printf("Creating \"%s\"", IntfWrapperExportName)
	exportWrapperfile, err := CreateLanguageFile(IntfWrapperExportName, indentString)
	if err != nil {
		return err
	}
	exportWrapperfile.WritePascalLicenseHeader(component,
		fmt.Sprintf("This is an autogenerated Pascal export implementation file in order to allow easy\ndevelopment of %s. The functions in this file need to be implemented. It needs to be generated only once.", libraryname),
		true)

	buildPascalExportsDefinition(component, exportWrapperfile, NameSpace, baseName, stubIdentifier, implementation.ClassIdentifier)

	var defaultImplementation []string
	defaultImplementation = append(defaultImplementation, fmt.Sprintf("raise E%sException.Create(%s_ERROR_NOTIMPLEMENTED);", NameSpace, strings.ToUpper(NameSpace)))

	IntfWrapperStubName := path.Join(stubOutputFolder, baseName+stubIdentifier+".pas")
	if forceRecreation || (!FileExists(IntfWrapperStubName)) {
		log.Printf("Creating \"%s\"", IntfWrapperStubName)
		templatefile, err := CreateLanguageFile(IntfWrapperStubName, indentString)
		if err != nil {
			return err
		}
		templatefile.WritePascalLicenseHeader(component,
			fmt.Sprintf("This is an autogenerated Pascal implementation file in order to allow easy\ndevelopment of %s. It needs to be generated only once.", libraryname),
			true)

		err = buildStubImplementation(component, templatefile, NameSpace, baseName, stubIdentifier, defaultImplementation)
		if err != nil {
			return err
		}
	} else {
		log.Printf("Omitting recreation of Stub implementation \"%s\"", IntfWrapperStubName)
	}

	IntfWrapperLPIName := path.Join(projectOutputFolder, baseName+".lpi")
	if forceRecreation || (!FileExists(IntfWrapperLPIName)) {
		log.Printf("Creating \"%s\"", IntfWrapperLPIName)
		lpifile, err := CreateLanguageFile(IntfWrapperLPIName, indentString)
		if err != nil {
			return err
		}
		err = buildLPIImplementation(component, lpifile, NameSpace, baseName)
		if err != nil {
			return err
		}
	} else {
		log.Printf("Omitting recreation of Stub implementation \"%s\"", IntfWrapperLPIName)
	}

	IntfWrapperLPRName := path.Join(outputFolder, baseName+".lpr")
	if forceRecreation || (!FileExists(IntfWrapperLPRName)) {
		log.Printf("Creating \"%s\"", IntfWrapperLPRName)
		lprfile, err := CreateLanguageFile(IntfWrapperLPRName, indentString)
		if err != nil {
			return err
		}
		lprfile.WritePascalLicenseHeader(component,
			fmt.Sprintf("This is an autogenerated Pascal project file in order to allow easy\ndevelopment of %s.", libraryname),
			true)
		err = buildLPRImplementation(component, lprfile, NameSpace, baseName)
		if err != nil {
			return err
		}
	} else {
		log.Printf("Omitting recreation of Stub implementation \"%s\"", IntfWrapperLPRName)
	}

	defFileName := path.Join(projectOutputFolder, baseName+".def")
	log.Printf("Creating \"%s\"", defFileName)
	defFile, err := CreateLanguageFile(defFileName, "")
	if err != nil {
		return err
	}
	err = buildDefFile(component, defFile, NameSpace)
	if err != nil {
		return err
	}

	err = buildPascalStub(component, NameSpace, implementation.ClassIdentifier, baseName, stubOutputFolder, indentString, stubIdentifier, forceRecreation, defaultImplementation)
	if err != nil {
		return err
	}

	return nil
}

// buildDefFile writes a module definition file with all exported functions
func buildDefFile(component ComponentDefinition, w LanguageWriter, NameSpace string) error {
	w.Writeln("EXPORTS")
	global := component.Global
	for _, method := range global.Methods {
		w.Writeln("%s_%s", strings.ToLower(NameSpace), strings.ToLower(method.MethodName))
	}
	for _, class := range component.Classes {
		for _, method := range class.Methods {
			w.Writeln("%s_%s_%s", strings.ToLower(NameSpace), strings.ToLower(class.ClassName), strings.ToLower(method.MethodName))
		}
	}
	return nil
}

func buildPascalTypeDefinition(component ComponentDefinition, w LanguageWriter, NameSpace string, BaseName string) error {

	w.Writeln("{$MODE DELPHI}")
	w.Writeln("unit %s_types;", BaseName)
	w.Writeln("")
	w.Writeln("interface")
	w.Writeln("")
	w.Writeln("uses")

	w.Writeln("  Classes,")
	w.Writeln("  sysutils;")
	w.Writeln("")

	err := writePascalBaseTypeDefinitions(component, w, NameSpace, BaseName)
	if err != nil {
		return err
	}

	w.Writeln("implementation")
	w.Writeln("")

	w.Writeln("end.")

	return nil

}

func buildPascalExceptionDefinition(component ComponentDefinition, w LanguageWriter, NameSpace string, BaseName string) error {

	w.Writeln("{$MODE DELPHI}")
	w.Writeln("unit %s_exception;", BaseName)
	w.Writeln("")
	w.Writeln("interface")
	w.Writeln("")
	w.Writeln("uses")

	w.Writeln("  %s_types,", BaseName)
	w.Writeln("  %s_interfaces,", BaseName)
	w.Writeln("  Classes,")
	w.Writeln("  sysutils;")
	w.Writeln("")
	w.Writeln("type")
	w.Writeln("  E%sException = class(Exception)", NameSpace)
	w.Writeln("  private")
	w.Writeln("    FErrorCode: T%sResult;", NameSpace)
	w.Writeln("    FCustomMessage: String;")
	w.Writeln("  public")
	w.Writeln("    property ErrorCode: T%sResult read FErrorCode;", NameSpace)
	w.Writeln("    property CustomMessage: String read FCustomMessage;")
	w.Writeln("    constructor Create(AErrorCode: T%sResult);", NameSpace)
	w.Writeln("    constructor CreateCustomMessage(AErrorCode: T%sResult; AMessage: String);", NameSpace)
	w.Writeln("  end;")
	w.Writeln("")
	w.Writeln("")

	w.Writeln("(*************************************************************************************************************************")
	w.Writeln(" Definition of exception handling functionality for %s", NameSpace)
	w.Writeln("**************************************************************************************************************************)")
	w.Writeln("")
	w.Writeln("function Handle%sException(A%sObject: TObject; E: E%sException): T%sResult;", NameSpace, NameSpace, NameSpace, NameSpace)
	w.Writeln("function HandleStdException(A%sObject: TObject; E: Exception): T%sResult;", NameSpace, NameSpace)
	w.Writeln("function HandleUnhandledException(A%sObject: TObject): T%sResult;", NameSpace, NameSpace)
	w.Writeln("")
	w.Writeln("")

	w.Writeln("implementation")
	w.Writeln("")
	w.Writeln("  constructor E%sException.Create(AErrorCode: T%sResult);", NameSpace, NameSpace)
	w.Writeln("  var")
	w.Writeln("    ADescription: String;")
	w.Writeln("  begin")
	w.Writeln("    FErrorCode := AErrorCode;")
	w.Writeln("    case FErrorCode of")

	for _, error := range component.Errors.Errors {
		w.Writeln("      %s_ERROR_%s: ADescription := '%s';", strings.ToUpper(NameSpace), error.Name, error.Description)
	}

	w.Writeln("      else")
	w.Writeln("        ADescription := 'unknown';")
	w.Writeln("    end;")
	w.Writeln("")
	w.Writeln("    inherited Create(Format('%s Error - %%s (#%%d)', [ ADescription, AErrorCode ]));", component.LibraryName)
	w.Writeln("  end;")
	w.Writeln("")
	w.Writeln("  constructor E%sException.CreateCustomMessage(AErrorCode: T%sResult; AMessage: String);", NameSpace, NameSpace)
	w.Writeln("  begin")
	w.Writeln("    FCustomMessage := AMessage;")
	w.Writeln("    FErrorCode := AErrorCode;")
	w.Writeln("    inherited Create(Format('%%s(%%d)', [FCustomMessage, AErrorCode]));")
	w.Writeln("  end;")
	w.Writeln("")

	registerErrorMethod := RegisterErrorMessageMethod()
	pascalBaseInterfaceName := "I" + NameSpace + component.baseClass().ClassName
	w.Writeln("(*************************************************************************************************************************")
	w.Writeln(" Implementation of exception handling functionality for %s", NameSpace)
	w.Writeln("**************************************************************************************************************************)")
	w.Writeln("")
	w.Writeln("function Handle%sException(A%sObject: TObject; E: E%sException): T%sResult;", NameSpace, NameSpace, NameSpace, NameSpace)
	w.Writeln("begin")
	w.Writeln("  result := E.ErrorCode;")
	w.Writeln("  if Supports(A%sObject, %s) then begin", NameSpace, pascalBaseInterfaceName)
	w.Writeln("    (A%sObject as %s).%s(E.CustomMessage)", NameSpace, pascalBaseInterfaceName, registerErrorMethod.MethodName)
	w.Writeln("  end;")
	w.Writeln("end;")
	w.Writeln("function HandleStdException(A%sObject: TObject; E: Exception): T%sResult;", NameSpace, NameSpace)
	w.Writeln("begin")
	w.Writeln("  Result := %s_ERROR_GENERICEXCEPTION;", strings.ToUpper(NameSpace))
	w.Writeln("  if Supports(A%sObject, %s) then begin", NameSpace, pascalBaseInterfaceName)
	w.Writeln("    (A%sObject as %s).%s(E.Message)", NameSpace, pascalBaseInterfaceName, registerErrorMethod.MethodName)
	w.Writeln("  end;")
	w.Writeln("end;")

	w.Writeln("function HandleUnhandledException(A%sObject: TObject): T%sResult;", NameSpace, NameSpace)
	w.Writeln("begin")
	w.Writeln("  Result := %s_ERROR_GENERICEXCEPTION;", strings.ToUpper(NameSpace))
	w.Writeln("  if Supports(A%sObject, %s) then begin", NameSpace, pascalBaseInterfaceName)
	w.Writeln("    (A%sObject as %s).%s('Unhandled Exception')", NameSpace, pascalBaseInterfaceName, registerErrorMethod.MethodName)
	w.Writeln("  end;")
	w.Writeln("end;")

	w.Writeln("end.")

	return nil

}

func createRandomUUID() (string, error) {

	u := make([]byte, 16)

	_, err := rand.Read(u)
	if err != nil {
		return "", err
	}

	u[8] = (u[8] | 0x80) & 0xBF // what does this do?
	u[6] = (u[6] | 0x40) & 0x4F // what does this do?

	uuid := fmt.Sprintf("%X-%X-%X-%X-%X", u[0:4], u[4:6], u[6:8], u[8:10], u[10:])

	return uuid, nil

}

func buildPascalInterfaceDefinition(component ComponentDefinition, w LanguageWriter, NameSpace string, BaseName string) error {

	w.Writeln("{$MODE DELPHI}")
	w.Writeln("{$INTERFACES CORBA}")

	w.Writeln("unit %s_interfaces;", BaseName)
	w.Writeln("")
	w.Writeln("interface")
	w.Writeln("")
	w.Writeln("uses")
	w.Writeln("  %s_types,", BaseName)
	writeSubComponentUses(component, w, false)
	w.Writeln("  Classes,")
	w.Writeln("  sysutils;")
	w.Writeln("")

	writeEnumConversionInterface(component, w, NameSpace)

	w.Writeln("")
	w.Writeln("type")

	pascalBaseInterfaceName := "I" + NameSpace + component.baseClass().ClassName
	for i := 0; i < len(component.Classes); i++ {
		class := component.Classes[i]

		w.Writeln("")
		w.Writeln("(*************************************************************************************************************************")
		w.Writeln(" Interface definition for %s", class.ClassName)
		w.Writeln("**************************************************************************************************************************)")
		w.Writeln("")

		parentClassName := class.ParentClass
		if parentClassName == "" {
			parentClassName = pascalBaseInterfaceName
		} else {
			parentClassName = "I" + NameSpace + parentClassName
		}

		if component.isBaseClass(class) {
			w.Writeln("I%s%s = interface", NameSpace, class.ClassName)
		} else {
			w.Writeln("I%s%s = interface(%s)", NameSpace, class.ClassName, parentClassName)
		}

		uuid, err := createRandomUUID()
		if err != nil {
			return err
		}
		w.Writeln("  ['{%s}']", uuid)
		w.Writeln("")

		w.AddIndentationLevel(1)
		if component.isBaseClass(class) {
			var methods [5]ComponentDefinitionMethod
			methods[0] = GetLastErrorMessageMethod()
			methods[1] = ClearErrorMessageMethod()
			methods[2] = RegisterErrorMessageMethod()
			methods[3] = IncRefCountMethod()
			methods[4] = DecRefCountMethod()

			for _, method := range methods {
				err := writePascalImplClassMethodDefinition(method, w, NameSpace, class.ClassName, false)
				if err != nil {
					return err
				}
			}
		}

		for j := 0; j < len(class.Methods); j++ {
			method := class.Methods[j]
			err := writePascalImplClassMethodDefinition(method, w, NameSpace, class.ClassName, false)
			if err != nil {
				return err
			}
		}
		w.AddIndentationLevel(-1)

		w.Writeln("end;")
		w.Writeln("")
	}

	w.Writeln("implementation")
	w.Writeln("")

	if len(component.Enums) > 0 {
		// Unit contains enums conversion that may raise exceptions
		w.Writeln("uses")
		w.Writeln("  %s_exception;", BaseName)
		w.Writeln("")
	}

	writeEnumConversionImplementation(component, w, NameSpace)

	w.Writeln("end.")

	return nil

}

func writePascalExportDefinition(component ComponentDefinition, method ComponentDefinitionMethod, w LanguageWriter, ClassName string, isGlobal bool, doComment bool) error {
	NameSpace := component.NameSpace

	PascalExportName := GetCExportName(NameSpace, ClassName, method, isGlobal)

	parameters := ""
	if !isGlobal {
		parameters = fmt.Sprintf("p%s: T%sHandle", ClassName, NameSpace)
	}

	if doComment {
		w.Writeln("(**")
		w.Writeln("* %s", method.MethodDescription)
		w.Writeln("*")
		if !isGlobal {
			w.Writeln("* @param[in] p%s - %s instance.", ClassName, ClassName)
		}
	}

	for k := 0; k < len(method.Params); k++ {
		param := method.Params[k]

		pascalParams, err := generatePlainPascalParameter(param, ClassName, method.MethodName, NameSpace)
		if err != nil {
			return err
		}

		for _, pascalParam := range pascalParams {
			if doComment {
				w.Writeln(pascalParam.ParamComment)
			}
			if parameters != "" {
				parameters = parameters + "; "
			}
			parameters = parameters + pascalParam.ParamName + ": " + pascalParam.ParamTypeNoConvention
		}

	}

	if doComment {
		w.Writeln("* @return error code or 0 (success)")
		w.Writeln("*)")
	}

	w.Writeln("function %s(%s): T%sResult; cdecl;", PascalExportName, parameters, NameSpace)

	return nil
}

func generatePrePostCallPascalFunctionCode(component ComponentDefinition, method ComponentDefinitionMethod, ClassIdentifier string, ClassName string) ([]string, []string, []string, []string, string, string, error) {
	NameSpace := component.NameSpace

	variableDefinitions := make([]string, 0)
	preCallCode := make([]string, 0)
	postCallCode := make([]string, 0)
	checkInputCode := make([]string, 0)
	callParameters := ""
	resultVariable := ""

	for k := 0; k < len(method.Params); k++ {
		param := method.Params[k]

		pascalParams, err := generatePlainPascalParameter(param, ClassName, method.MethodName, NameSpace)
		if err != nil {
			return make([]string, 0), make([]string, 0), make([]string, 0), make([]string, 0), "", "", err
		}

		switch param.ParamPass {
		case "in":
			if callParameters != "" {
				callParameters = callParameters + ", "
			}
			switch param.ParamType {
			case "uint8", "uint16", "uint32", "uint64", "int8", "int16", "int32", "int64", "single", "double", "pointer":
				callParameters = callParameters + pascalParams[0].ParamName
			case "bool":
				callParameters = callParameters + pascalParams[0].ParamName + " <> 0"
			case "enum":
				callParameters = callParameters + "convertConstTo" + param.ParamClass + "(" + pascalParams[0].ParamName + ")"
			case "struct":
				checkInputCode = append(checkInputCode, fmt.Sprintf("if not Assigned(%s) then", pascalParams[0].ParamName))
				checkInputCode = append(checkInputCode, fmt.Sprintf("  raise E%sException.Create(%s_ERROR_INVALIDPARAM);", NameSpace, strings.ToUpper(NameSpace)))

				callParameters = callParameters + pascalParams[0].ParamName + "^"

			case "basicarray", "structarray":
				checkInputCode = append(checkInputCode, fmt.Sprintf("if ((not Assigned(%s)) and (%s>0)) then", pascalParams[1].ParamName, pascalParams[0].ParamName))
				checkInputCode = append(checkInputCode, fmt.Sprintf("  raise E%sException.Create(%s_ERROR_INVALIDPARAM);", NameSpace, strings.ToUpper(NameSpace)))

				callParameters = callParameters + fmt.Sprintf("%s, %s", pascalParams[0].ParamName, pascalParams[1].ParamName)

			case "class", "optionalclass":
				paramNameSpace, paramClassName, _ := decomposeParamClassName(param.ParamClass)
				if len(paramNameSpace) > 0 {
					theSubWrapper := fmt.Sprintf("T%sWrapper.%sWrapper", NameSpace, paramNameSpace)
					acqurireMethod := component.ImportedComponentDefinitions[paramNameSpace].Global.AcquireMethod
					variableDefinitions = append(variableDefinitions, fmt.Sprintf("Object%s: T%s%s;", param.ParamName, paramNameSpace, paramClassName))
					checkInputCode = append(checkInputCode, fmt.Sprintf("Object%s := T%s%s.Create(%s, %s);", param.ParamName, paramNameSpace, paramClassName, theSubWrapper, pascalParams[0].ParamName))
					checkInputCode = append(checkInputCode, fmt.Sprintf("%s.%s(Object%s);", theSubWrapper, acqurireMethod, param.ParamName))
				} else {
					variableDefinitions = append(variableDefinitions, fmt.Sprintf("Object%s: TObject;", param.ParamName))

					checkInputCode = append(checkInputCode, fmt.Sprintf("Object%s := TObject(%s);", param.ParamName, pascalParams[0].ParamName))
					if (param.ParamType == "class") {
						checkInputCode = append(checkInputCode, fmt.Sprintf("if (not Supports(Object%s, I%s%s)) then", param.ParamName, NameSpace, param.ParamClass))
						checkInputCode = append(checkInputCode, fmt.Sprintf("  raise E%sException.Create(%s_ERROR_INVALIDCAST);", NameSpace, strings.ToUpper(NameSpace)))
					}
				}

				checkInputCode = append(checkInputCode, "")

				callParameters = callParameters + fmt.Sprintf("Object%s", param.ParamName)

			case "string":
				checkInputCode = append(checkInputCode, fmt.Sprintf("if (not Assigned(%s)) then", pascalParams[0].ParamName))
				checkInputCode = append(checkInputCode, fmt.Sprintf("  raise E%sException.Create(%s_ERROR_INVALIDPARAM);", NameSpace, strings.ToUpper(NameSpace)))

				callParameters = callParameters + "StrPas(" + pascalParams[0].ParamName + ")"

			case "functiontype":
				callParameters = callParameters + pascalParams[0].ParamName

			default:
				return make([]string, 0), make([]string, 0), make([]string, 0), make([]string, 0), "", "", fmt.Errorf("method parameter type \"%s\" of param pass \"%s\" is not implemented for %s::%s(%s) )", param.ParamType, param.ParamPass, ClassName, method.MethodName, param.ParamName)
			}

		case "out":
			if callParameters != "" {
				callParameters = callParameters + ", "
			}

			switch param.ParamType {

			case "uint8", "uint16", "uint32", "uint64", "int8", "int16", "int32", "int64", "single", "double", "pointer", "struct":
				checkInputCode = append(checkInputCode, fmt.Sprintf("if (not Assigned(%s)) then", pascalParams[0].ParamName))
				checkInputCode = append(checkInputCode, fmt.Sprintf("  raise E%sException.Create(%s_ERROR_INVALIDPARAM);\n", NameSpace, strings.ToUpper(NameSpace)))

				callParameters = callParameters + pascalParams[0].ParamName + "^"

			case "enum":
				checkInputCode = append(checkInputCode, fmt.Sprintf("if not Assigned(%s) then", pascalParams[0].ParamName))
				checkInputCode = append(checkInputCode, fmt.Sprintf("  raise E%sException.Create(%s_ERROR_INVALIDPARAM);", NameSpace, strings.ToUpper(NameSpace)))

				variableDefinitions = append(variableDefinitions, fmt.Sprintf("Result%s: T%s%s;", param.ParamName, NameSpace, param.ParamClass))

				callParameters = callParameters + fmt.Sprintf("Result%s", param.ParamName)

				postCallCode = append(postCallCode, fmt.Sprintf("%s^ := convert%sToConst(Result%s);", pascalParams[0].ParamName, param.ParamClass, param.ParamName))

			case "bool":
				checkInputCode = append(checkInputCode, fmt.Sprintf("if not Assigned(%s) then", pascalParams[0].ParamName))
				checkInputCode = append(checkInputCode, fmt.Sprintf("  raise E%sException.Create(%s_ERROR_INVALIDPARAM);", NameSpace, strings.ToUpper(NameSpace)))

				variableDefinitions = append(variableDefinitions, fmt.Sprintf("Result%s: Boolean;", param.ParamName))

				callParameters = callParameters + fmt.Sprintf("Result%s", param.ParamName)

				postCallCode = append(postCallCode, fmt.Sprintf("%s^ := Ord(Result%s);", pascalParams[0].ParamName, param.ParamName))

			case "basicarray", "structarray":
				checkInputCode = append(checkInputCode, fmt.Sprintf("if ((not Assigned(%s)) and (not Assigned(%s))) then", pascalParams[1].ParamName, pascalParams[2].ParamName))
				checkInputCode = append(checkInputCode, fmt.Sprintf("  raise E%sException.Create(%s_ERROR_INVALIDPARAM);", NameSpace, strings.ToUpper(NameSpace)))

				callParameters = callParameters + fmt.Sprintf("%s, %s, %s", pascalParams[0].ParamName, pascalParams[1].ParamName, pascalParams[2].ParamName)

			case "string":
				checkInputCode = append(checkInputCode, fmt.Sprintf("if ((not Assigned(%s)) and (not Assigned(%s))) then", pascalParams[2].ParamName, pascalParams[1].ParamName))
				checkInputCode = append(checkInputCode, fmt.Sprintf("  raise E%sException.Create(%s_ERROR_INVALIDPARAM);", NameSpace, strings.ToUpper(NameSpace)))

				variableDefinitions = append(variableDefinitions, fmt.Sprintf("Result%s: String;", param.ParamName))
				variableDefinitions = append(variableDefinitions, fmt.Sprintf("Len%s: Cardinal;", param.ParamName))

				callParameters = callParameters + "Result" + param.ParamName

				postCallCode = append(postCallCode, fmt.Sprintf("Len%s := Length(Result%s);", param.ParamName, param.ParamName))
				postCallCode = append(postCallCode, fmt.Sprintf("if Assigned(%s) then", pascalParams[1].ParamName))
				postCallCode = append(postCallCode, fmt.Sprintf("  %s^ := Len%s + 1;", pascalParams[1].ParamName, param.ParamName))

				postCallCode = append(postCallCode, fmt.Sprintf("if Assigned(%s) then begin", pascalParams[2].ParamName))
				postCallCode = append(postCallCode, fmt.Sprintf("  if (Len%s >= %s) then", param.ParamName, pascalParams[0].ParamName))
				postCallCode = append(postCallCode, fmt.Sprintf("    raise E%sException.Create(%s_ERROR_BUFFERTOOSMALL);", NameSpace, strings.ToUpper(NameSpace)))
				postCallCode = append(postCallCode, fmt.Sprintf("  Move(PAnsiChar(Result%s)^, %s^, Len%s);", param.ParamName, pascalParams[2].ParamName, param.ParamName))
				postCallCode = append(postCallCode, fmt.Sprintf("  %s[Len%s] := Char(0);", pascalParams[2].ParamName, param.ParamName))
				postCallCode = append(postCallCode, fmt.Sprintf("end;"))

			case "class", "optionalclass":
				checkInputCode = append(checkInputCode, fmt.Sprintf("if not Assigned(p%s) then", param.ParamName))
				checkInputCode = append(checkInputCode, fmt.Sprintf("  raise E%sException.Create(%s_ERROR_INVALIDPARAM);", NameSpace, strings.ToUpper(NameSpace)))

				paramNameSpace, paramClassName, _ := decomposeParamClassName(param.ParamClass)
				if len(paramNameSpace) > 0 {
					theSubWrapper := fmt.Sprintf("T%sWrapper.%sWrapper", NameSpace, paramNameSpace)
					variableDefinitions = append(variableDefinitions, fmt.Sprintf("Out%s: T%s%s;", param.ParamName, paramNameSpace, paramClassName))

					acqurireMethod := component.ImportedComponentDefinitions[paramNameSpace].Global.AcquireMethod
					postCallCode = append(postCallCode, fmt.Sprintf("%s.%s(Out%s);", theSubWrapper, acqurireMethod, param.ParamName))
					postCallCode = append(postCallCode, fmt.Sprintf("%s^ := Out%s.TheHandle;", pascalParams[0].ParamName, param.ParamName))
				} else {
					variableDefinitions = append(variableDefinitions, fmt.Sprintf("Out%s: TObject;", param.ParamName))
					postCallCode = append(postCallCode, fmt.Sprintf("%s^ := Out%s;", pascalParams[0].ParamName, param.ParamName))
				}
				callParameters = callParameters + "Out" + param.ParamName

			default:
				return make([]string, 0), make([]string, 0), make([]string, 0), make([]string, 0), "", "", fmt.Errorf("method parameter type \"%s\" of param pass \"%s\" is not implemented for %s::%s(%s) )", param.ParamType, param.ParamPass, ClassName, method.MethodName, param.ParamName)
			}

		case "return":

			switch param.ParamType {

			case "uint8", "uint16", "uint32", "uint64", "int8", "int16", "int32", "int64", "single", "double", "pointer":
				checkInputCode = append(checkInputCode, fmt.Sprintf("if not Assigned(%s) then", pascalParams[0].ParamName))
				checkInputCode = append(checkInputCode, fmt.Sprintf("  raise E%sException.Create(%s_ERROR_INVALIDPARAM);", NameSpace, strings.ToUpper(NameSpace)))

				variableDefinitions = append(variableDefinitions, fmt.Sprintf("Result%s: %s;", param.ParamName, pascalParams[0].ParamType))

				postCallCode = append(postCallCode, fmt.Sprintf("%s^ := Result%s;", pascalParams[0].ParamName, param.ParamName))

				resultVariable = fmt.Sprintf("Result%s", param.ParamName)

			case "bool":
				checkInputCode = append(checkInputCode, fmt.Sprintf("if not Assigned(%s) then", pascalParams[0].ParamName))
				checkInputCode = append(checkInputCode, fmt.Sprintf("  raise E%sException.Create(%s_ERROR_INVALIDPARAM);", NameSpace, strings.ToUpper(NameSpace)))

				variableDefinitions = append(variableDefinitions, fmt.Sprintf("Result%s: Boolean;", param.ParamName))

				postCallCode = append(postCallCode, fmt.Sprintf("%s^ := Ord(Result%s);", pascalParams[0].ParamName, param.ParamName))

				resultVariable = fmt.Sprintf("Result%s", param.ParamName)

			case "enum":
				checkInputCode = append(checkInputCode, fmt.Sprintf("if not Assigned(%s) then", pascalParams[0].ParamName))
				checkInputCode = append(checkInputCode, fmt.Sprintf("  raise E%sException.Create(%s_ERROR_INVALIDPARAM);", NameSpace, strings.ToUpper(NameSpace)))

				variableDefinitions = append(variableDefinitions, fmt.Sprintf("Result%s: T%s%s;", param.ParamName, NameSpace, param.ParamClass))

				postCallCode = append(postCallCode, fmt.Sprintf("%s^ := convert%sToConst(Result%s);", pascalParams[0].ParamName, param.ParamClass, param.ParamName))

				resultVariable = fmt.Sprintf("Result%s", param.ParamName)

			case "struct":
				checkInputCode = append(checkInputCode, fmt.Sprintf("if not Assigned(%s) then", pascalParams[0].ParamName))
				checkInputCode = append(checkInputCode, fmt.Sprintf("  raise E%sException.Create(%s_ERROR_INVALIDPARAM);", NameSpace, strings.ToUpper(NameSpace)))

				variableDefinitions = append(variableDefinitions, fmt.Sprintf("Result%s: T%s%s;", param.ParamName, NameSpace, param.ParamClass))

				postCallCode = append(postCallCode, fmt.Sprintf("%s^ := Result%s;", pascalParams[0].ParamName, param.ParamName))

				resultVariable = fmt.Sprintf("Result%s", param.ParamName)

			case "string":
				checkInputCode = append(checkInputCode, fmt.Sprintf("if ((not Assigned(%s)) and (not Assigned(%s))) then", pascalParams[2].ParamName, pascalParams[1].ParamName))
				checkInputCode = append(checkInputCode, fmt.Sprintf("  raise E%sException.Create(%s_ERROR_INVALIDPARAM);", NameSpace, strings.ToUpper(NameSpace)))

				variableDefinitions = append(variableDefinitions, fmt.Sprintf("Result%s: String;", param.ParamName))
				variableDefinitions = append(variableDefinitions, fmt.Sprintf("Len%s: Cardinal;", param.ParamName))

				resultVariable = fmt.Sprintf("Result%s", param.ParamName)

				postCallCode = append(postCallCode, fmt.Sprintf("Len%s := Length(Result%s);", param.ParamName, param.ParamName))
				postCallCode = append(postCallCode, fmt.Sprintf("if Assigned(%s) then", pascalParams[1].ParamName))
				postCallCode = append(postCallCode, fmt.Sprintf("  %s^ := Len%s + 1;", pascalParams[1].ParamName, param.ParamName))

				postCallCode = append(postCallCode, fmt.Sprintf("if Assigned(%s) then begin", pascalParams[2].ParamName))
				postCallCode = append(postCallCode, fmt.Sprintf("  if (Len%s >= %s) then", param.ParamName, pascalParams[0].ParamName))
				postCallCode = append(postCallCode, fmt.Sprintf("    raise E%sException.Create(%s_ERROR_BUFFERTOOSMALL);", NameSpace, strings.ToUpper(NameSpace)))
				postCallCode = append(postCallCode, fmt.Sprintf("  Move(PAnsiChar(Result%s)^, %s^, Len%s);", param.ParamName, pascalParams[2].ParamName, param.ParamName))
				postCallCode = append(postCallCode, fmt.Sprintf("  %s[Len%s] := Char(0);", pascalParams[2].ParamName, param.ParamName))
				postCallCode = append(postCallCode, fmt.Sprintf("end;"))

			case "class", "optionalclass":
				checkInputCode = append(checkInputCode, fmt.Sprintf("if not Assigned(p%s) then", param.ParamName))
				checkInputCode = append(checkInputCode, fmt.Sprintf("  raise E%sException.Create(%s_ERROR_INVALIDPARAM);", NameSpace, strings.ToUpper(NameSpace)))

				paramNameSpace, paramClassName, _ := decomposeParamClassName(param.ParamClass)
				if len(paramNameSpace) > 0 {
					theSubWrapper := fmt.Sprintf("T%sWrapper.%sWrapper", NameSpace, paramNameSpace)
					variableDefinitions = append(variableDefinitions, fmt.Sprintf("Result%s: T%s%s;", param.ParamName, paramNameSpace, paramClassName))

					acqurireMethod := component.ImportedComponentDefinitions[paramNameSpace].Global.AcquireMethod
					postCallCode = append(postCallCode, fmt.Sprintf("%s.%s(Result%s);", theSubWrapper, acqurireMethod, param.ParamName))
					postCallCode = append(postCallCode, fmt.Sprintf("%s^ := Result%s.TheHandle;", pascalParams[0].ParamName, param.ParamName))
				} else {
					variableDefinitions = append(variableDefinitions, fmt.Sprintf("Result%s: TObject;", param.ParamName))
					postCallCode = append(postCallCode, fmt.Sprintf("%s^ := Result%s;", pascalParams[0].ParamName, param.ParamName))
				}

				resultVariable = fmt.Sprintf("Result%s", param.ParamName)
			default:
				return make([]string, 0), make([]string, 0), make([]string, 0), make([]string, 0), "", "", fmt.Errorf("method parameter type \"%s\" of param pass \"%s\" is not implemented for %s::%s(%s) )", param.ParamType, param.ParamPass, ClassName, method.MethodName, param.ParamName)
			}

		default:
			return make([]string, 0), make([]string, 0), make([]string, 0), make([]string, 0), "", "", fmt.Errorf("invalid method parameter passing \"%s\" for %s.%s (%s)", param.ParamPass, ClassName, method.MethodName, param.ParamName)
		}
	}

	return variableDefinitions, checkInputCode, preCallCode, postCallCode, callParameters, resultVariable, nil
}

func writePascalClassExportImplementation(component ComponentDefinition, method ComponentDefinitionMethod, w LanguageWriter, defaultDefinitionLines []string, defaultImplementationLines []string, ClassName string, isGlobal bool, ClassIdentifier string) error {
	NameSpace := component.NameSpace
	variableDefinitions, parameterChecks, preCallCode, postCallCode, callParameters, resultVariable, err := generatePrePostCallPascalFunctionCode(component, method, ClassIdentifier, ClassName)
	if err != nil {
		return err
	}

	if !isGlobal {
		// Define variables
		variableDefinitions = append(variableDefinitions, fmt.Sprintf("Object%s: TObject;", ClassName))
		variableDefinitions = append(variableDefinitions, fmt.Sprintf("Intf%s: I%s%s;", ClassName, NameSpace, ClassName))

		parameterChecks = append(parameterChecks, fmt.Sprintf("if not Assigned(p%s) then", ClassName))
		parameterChecks = append(parameterChecks, fmt.Sprintf("  raise E%sException.Create(%s_ERROR_INVALIDPARAM);", NameSpace, strings.ToUpper(NameSpace)))
	}

	if len(defaultDefinitionLines)+len(defaultImplementationLines) > 0 {
		if len(defaultDefinitionLines) > 0 {
			w.Writeln("var")
			w.Writelns("  ", defaultDefinitionLines)
		}
	} else {
		if len(variableDefinitions) > 0 {
			w.Writeln("var")
			w.Writelns("  ", variableDefinitions)
		}
	}

	w.Writeln("begin")
	w.Writeln("  try")
	w.Writelns("    ", parameterChecks)

	w.Writeln("")
	if !isGlobal {
		w.Writeln("    Object%s := TObject(p%s);", ClassName, ClassName)
		w.Writeln("    if Supports(Object%s, I%s%s) then begin", ClassName, NameSpace, ClassName)
		w.Writeln("      Intf%s := Object%s as I%s%s;", ClassName, ClassName, NameSpace, ClassName)
	}

	w.Writelns("", preCallCode)

	resultVariableAssignment := ""
	if resultVariable != "" {
		resultVariableAssignment = resultVariable + " := "
	}

	classInstanceSpacing := "      "
	classInstanceToCall := "Intf" + ClassName
	postCallCodeSpacing := "      "
	if isGlobal {
		classInstanceToCall = "T" + NameSpace + "Wrapper"
		classInstanceSpacing = "    "
		postCallCodeSpacing = "    "
	}

	if len(defaultImplementationLines) > 0 {
		w.Writelns(classInstanceSpacing, defaultImplementationLines)
	} else {
		w.Writeln(classInstanceSpacing+"%s%s.%s(%s);", resultVariableAssignment, classInstanceToCall, method.MethodName, callParameters)
		w.Writeln("")
		w.Writelns(postCallCodeSpacing, postCallCode)
	}

	if !isGlobal {
		w.Writeln("    end else")
		w.Writeln("      raise E%sException.Create(%s_ERROR_INVALIDCAST);", NameSpace, strings.ToUpper(NameSpace))
		w.Writeln("")
	}

	w.Writeln("    Result := %s_SUCCESS;", strings.ToUpper(NameSpace))
	w.Writeln("  except")
	if !isGlobal {
		w.Writeln("    On E: E%sException do begin", NameSpace)
		w.Writeln("      Result := Handle%sException(Object%s , E);", NameSpace, ClassName)
		w.Writeln("    end;")
		w.Writeln("    On E: Exception do begin")
		w.Writeln("      Result := HandleStdException(Object%s , E);", ClassName)
		w.Writeln("    end")
		w.Writeln("    else begin")
		w.Writeln("      Result := HandleUnhandledException(Object%s);", ClassName)
		w.Writeln("    end;")
	} else {
		w.Writeln("    On E: E%sException do begin", NameSpace)
		w.Writeln("      Result := E.ErrorCode;")
		w.Writeln("    end")
		w.Writeln("    else begin")
		w.Writeln("      Result := %s_ERROR_GENERICEXCEPTION;", strings.ToUpper(NameSpace))
		w.Writeln("    end")
	}
	w.Writeln("  end;")
	w.Writeln("end;")
	w.Writeln("")

	return nil
}

func buildPascalGetSymbolAddressMethod(component ComponentDefinition, w LanguageWriter, DeclarationOnly bool) {
	NameSpace := component.NameSpace
	w.Writeln("")
	w.Writeln("(*************************************************************************************************************************")
	w.Writeln(" Function table lookup implementation")
	w.Writeln("**************************************************************************************************************************)")
	w.Writeln("")

	w.Writeln("function _%s_getprocaddress_internal(pProcName: PAnsiChar; out ppProcAddress: Pointer): T%sResult cdecl;", strings.ToLower(NameSpace), NameSpace)
	w.Writeln("")
	if DeclarationOnly {
		return
	}
	w.Writeln("begin")
	w.AddIndentationLevel(1)
	w.Writeln("result := %s_SUCCESS;", strings.ToUpper(NameSpace))
	w.Writeln("ppProcAddress := nil;")
	w.Writeln("")

	isFirst := true
	beginString := "if"
	global := component.Global
	for i := 0; i < len(component.Classes); i++ {
		class := component.Classes[i]
		for j := 0; j < len(class.Methods); j++ {
			method := class.Methods[j]
			procName := strings.ToLower(NameSpace + "_" + class.ClassName + "_" + method.MethodName)

			w.Writeln("%s (pProcName = '%s') then", beginString, procName)
			w.Writeln("  ppProcAddress := @%s", procName)
			if isFirst {
				isFirst = false
				beginString = "else if"
			}
		}
	}
	for j := 0; j < len(global.Methods); j++ {
		method := global.Methods[j]
		procName := strings.ToLower(NameSpace + "_" + method.MethodName)
		w.Writeln("%s (pProcName = '%s') then", beginString, procName)
		w.Writeln("  ppProcAddress := @%s", procName)
		if isFirst {
			isFirst = false
			beginString = "else if"
		}
	}
	w.Writeln("else")
	w.Writeln("  result := %s_ERROR_COULDNOTFINDLIBRARYEXPORT;", strings.ToUpper(NameSpace))

	w.AddIndentationLevel(-1)
	w.Writeln("end;")
}

func buildPascalExportsDefinition(component ComponentDefinition, w LanguageWriter, NameSpace string, BaseName string, stubIdentifier string, ClassIdentifier string) error {

	global := component.Global

	w.Writeln("{$MODE DELPHI}")
	w.Writeln("unit %s_exports;", BaseName)
	w.Writeln("")
	w.Writeln("interface")
	w.Writeln("")
	w.Writeln("uses")
	w.Writeln("  %s%s,", BaseName, stubIdentifier)
	w.Writeln("  %s_types,", BaseName)
	w.Writeln("  %s_interfaces,", BaseName)
	w.Writeln("  %s_exception,", BaseName)
	w.Writeln("  Classes,")
	writeSubComponentUses(component, w, false)
	w.Writeln("  sysutils;")
	w.Writeln("")

	for i := 0; i < len(component.Classes); i++ {
		class := component.Classes[i]

		w.Writeln("(*************************************************************************************************************************")
		w.Writeln(" Class export definition of %s ", class.ClassName)
		w.Writeln("**************************************************************************************************************************)")
		w.Writeln("")

		for j := 0; j < len(class.Methods); j++ {
			method := class.Methods[j]

			err := writePascalExportDefinition(component, method, w, class.ClassName, false, true)
			if err != nil {
				return err
			}

			w.Writeln("")
		}
	}

	w.Writeln("(*************************************************************************************************************************")
	w.Writeln(" Global function export definition")
	w.Writeln("**************************************************************************************************************************)")
	w.Writeln("")

	for j := 0; j < len(global.Methods); j++ {
		method := global.Methods[j]

		err := writePascalExportDefinition(component, method, w, "Wrapper", true, true)
		if err != nil {
			return err
		}
		w.Writeln("")
	}

	buildPascalGetSymbolAddressMethod(component, w, true)

	w.Writeln("implementation")
	w.Writeln("")

	for i := 0; i < len(component.Classes); i++ {
		class := component.Classes[i]

		for j := 0; j < len(class.Methods); j++ {

			method := class.Methods[j]

			err := writePascalExportDefinition(component, method, w, class.ClassName, false, false)
			if err != nil {
				return err
			}

			err = writePascalClassExportImplementation(component, method, w, make([]string, 0), make([]string, 0), class.ClassName, false, ClassIdentifier)
			if err != nil {
				return err
			}
		}

	}

	for j := 0; j < len(global.Methods); j++ {
		method := global.Methods[j]

		err := writePascalExportDefinition(component, method, w, "Wrapper", true, false)
		if err != nil {
			return err
		}

		thisMethodDefaultImpl := []string{}
		thisMethodDefinitionLines := []string{}

		isSpecialFunction, err := CheckHeaderSpecialFunction(method, global)
		if err != nil {
			return err
		}
		if isSpecialFunction == eSpecialMethodSymbolLookup {
			var symbolLookupImplementation []string
			symbolLookupImplementation = append(symbolLookupImplementation,
				fmt.Sprintf("p%s^ := @_%s_getprocaddress_internal;", method.Params[0].ParamName, strings.ToLower(NameSpace)))
			thisMethodDefaultImpl = symbolLookupImplementation
		} else if isSpecialFunction == eSpecialMethodJournal {
			var definitionLines []string
			thisMethodDefinitionLines = definitionLines

			var journalImplementation []string
			journalImplementation = append(journalImplementation, fmt.Sprintf(""))
			thisMethodDefaultImpl = journalImplementation
		} else if isSpecialFunction == eSpecialMethodInjection {
			var definitionLines []string
			definitionLines = append(definitionLines, "ANameSpaceFound: boolean;")
			thisMethodDefinitionLines = definitionLines

			var injectionImplementation []string
			sParamName := "StrPas(p" + method.Params[0].ParamName + ")"
			for _, subComponent := range component.ImportedComponentDefinitions {
				theNameSpace := subComponent.NameSpace

				injectionImplementation = append(injectionImplementation, fmt.Sprintf("ANameSpaceFound := False;"))
				injectionImplementation = append(injectionImplementation, fmt.Sprintf("if (%s = '%s') then begin", sParamName, theNameSpace))
				injectionImplementation = append(injectionImplementation, fmt.Sprintf("  T%sWrapper.%sWrapper := T%sWrapper.CreateFromSymbolLookupMethod(p%s);", NameSpace, theNameSpace, theNameSpace, method.Params[1].ParamName))
				injectionImplementation = append(injectionImplementation, fmt.Sprintf("  ANameSpaceFound := True;"))
				injectionImplementation = append(injectionImplementation, fmt.Sprintf("end;"))
			}
			injectionImplementation = append(injectionImplementation, fmt.Sprintf("if not ANameSpaceFound then"))
			injectionImplementation = append(injectionImplementation, fmt.Sprintf("  raise E%sException.Create(%s_ERROR_COULDNOTLOADLIBRARY);", NameSpace, strings.ToUpper(NameSpace)))
			thisMethodDefaultImpl = injectionImplementation
		}

		err = writePascalClassExportImplementation(component, method, w, thisMethodDefinitionLines, thisMethodDefaultImpl, "Wrapper", true, ClassIdentifier)
		if err != nil {
			return err
		}

	}
	w.Writeln("")
	buildPascalGetSymbolAddressMethod(component, w, false)
	w.Writeln("")
	w.Writeln("end.")
	w.Writeln("")

	return nil

}

func buildLPRImplementation(component ComponentDefinition, w LanguageWriter, NameSpace string, BaseName string) error {

	global := component.Global

	w.Writeln("{$MODE DELPHI}")
	w.Writeln("library %s;", BaseName)
	w.Writeln("")
	w.Writeln("uses")
	w.Writeln("{$IFDEF UNIX}")
	w.Writeln("  cthreads,")
	w.Writeln("{$ENDIF UNIX}")
	w.Writeln("  syncobjs,")
	w.Writeln("  %s_types,", BaseName)
	w.Writeln("  %s_exports,", BaseName)
	w.Writeln("  Classes,")
	w.Writeln("  sysutils;")
	w.Writeln("")
	w.Writeln("exports")

	for i := 0; i < len(component.Classes); i++ {
		class := component.Classes[i]

		for j := 0; j < len(class.Methods); j++ {

			commaString := ","
			if (j == (len(class.Methods) - 1)) && (len(global.Methods) == 0) {
				commaString = ";"
			}

			method := class.Methods[j]
			w.Writeln("  %s%s", GetCExportName(NameSpace, class.ClassName, method, false), commaString)
		}

	}

	for j := 0; j < len(global.Methods); j++ {
		method := global.Methods[j]
		commaString := ","
		if j == (len(global.Methods) - 1) {
			commaString = ";"
		}

		w.Writeln("  %s%s", GetCExportName(NameSpace, "Wrapper", method, true), commaString)
	}

	w.Writeln("")
	w.Writeln("begin")
	w.Writeln("")
	w.Writeln("end.")

	return nil
}

func buildLPIImplementation(component ComponentDefinition, w LanguageWriter, NameSpace string, BaseName string) error {

	otherUnitFiles := ""
	for _, subComponent := range component.ImportedComponentDefinitions {
		otherUnitFiles = otherUnitFiles + fmt.Sprintf(";..\\..\\..\\%s_Component\\Bindings\\Pascal", subComponent.NameSpace)
	}

	w.Writeln("<?xml version=\"1.0\" encoding=\"UTF-8\"?>")
	w.Writeln("<CONFIG>")
	w.Writeln("  <ProjectOptions>")
	w.Writeln("    <Version Value=\"%d\"/>", 10)

	w.Writeln("    <PathDelim Value=\"\\\"/>")
	w.Writeln("    <General>")
	w.Writeln("      <Flags>")
	w.Writeln("        <MainUnitHasCreateFormStatements Value=\"False\" />")
	w.Writeln("        <MainUnitHasTitleStatement Value=\"False\" />")
	w.Writeln("        <MainUnitHasScaledStatement Value=\"False\" />")
	w.Writeln("      </Flags>")
	w.Writeln("      <SessionStorage Value=\"InProjectDir\" />")
	w.Writeln("      <MainUnit Value=\"%d\"/>", 0)
	w.Writeln("      <Title Value=\"%s\" />", NameSpace)
	w.Writeln("      <UseAppBundle Value=\"False\" />")
	w.Writeln("      <ResourceType Value=\"res\" />")
	w.Writeln("    </General>")
	w.Writeln("    <BuildModes Count=\"%d\">", 2)
	w.Writeln("      <Item1 Name=\"Release\" Default=\"True\"/>")
	w.Writeln("      <Item2 Name=\"Debug\">")
	w.Writeln("        <CompilerOptions>")
	w.Writeln("          <Version Value=\"11\" />")
	w.Writeln("          <PathDelim Value=\"\\\"/>")
	w.Writeln("          <Target>")
	w.Writeln("            <Filename Value=\"bin\\$(TargetCPU)-$(TargetOS)\\Release\\project%s\"/>", BaseName)
	w.Writeln("          </Target>")
	w.Writeln("          <SearchPaths>")
	w.Writeln("            <IncludeFiles Value=\"$(ProjOutDir)\"/>")
	w.Writeln("            <OtherUnitFiles Value=\"Stub;Interfaces%s\"/>", otherUnitFiles)
	w.Writeln("            <UnitOutputDirectory Value=\"lib\\$(TargetCPU)-$(TargetOS)\"/>")
	w.Writeln("          </SearchPaths>")
	w.Writeln("          <Parsing>")
	w.Writeln("            <SyntaxOptions>")
	w.Writeln("              <IncludeAssertionCode Value=\"True\"/>")
	w.Writeln("            </SyntaxOptions>")
	w.Writeln("          </Parsing>")
	w.Writeln("          <CodeGeneration>")
	w.Writeln("            <RelocatableUnit Value=\"True\" />")
	w.Writeln("          </CodeGeneration>")
	w.Writeln("          <Linking>")
	w.Writeln("            <Debugging>")
	w.Writeln("              <UseExternalDbgSyms Value=\"True\"/>")
	w.Writeln("            </Debugging>")
	w.Writeln("            <Options>")
	w.Writeln("              <ExecutableType Value=\"Library\"/>")
	w.Writeln("            </Options>")
	w.Writeln("          </Linking>")
	w.Writeln("        </CompilerOptions>")
	w.Writeln("      </Item2>")
	w.Writeln("    </BuildModes>")
	w.Writeln("    <PublishOptions>")
	w.Writeln("      <Version Value=\"%d\"/>", 2)
	w.Writeln("    </PublishOptions>")
	w.Writeln("    <RunParams>")
	w.Writeln("      <local>")
	w.Writeln("        <FormatVersion Value=\"1\"/>")
	w.Writeln("      </local>")
	w.Writeln("    </RunParams>")
	w.Writeln("    <Units Count=\"%d\">", 2)
	w.Writeln("      <Unit0>")
	w.Writeln("        <Filename Value=\"Interfaces\\%s.lpr\"/>", BaseName)
	w.Writeln("        <IsPartOfProject Value=\"True\"/>")
	w.Writeln("      </Unit0>")
	w.Writeln("      <Unit1>")
	w.Writeln("        <Filename Value=\"Stub\\%s.pas\"/>", BaseName)
	w.Writeln("        <IsPartOfProject Value=\"True\"/>")
	w.Writeln("      </Unit1>")
	w.Writeln("    </Units>")
	w.Writeln("  </ProjectOptions>")
	w.Writeln("  <CompilerOptions>")
	w.Writeln("    <Version Value=\"%d\"/>", 11)
	w.Writeln("    <PathDelim Value=\"\\\"/>")
	w.Writeln("    <Target>")
	w.Writeln("      <Filename Value=\"bin\\$(TargetCPU)-$(TargetOS)\\Release\\%s\"/>", BaseName)
	w.Writeln("    </Target>")
	w.Writeln("    <SearchPaths>")
	w.Writeln("      <IncludeFiles Value=\"$(ProjOutDir)\"/>")
	w.Writeln("      <OtherUnitFiles Value=\"Stub;Interfaces%s\"/>", otherUnitFiles)
	w.Writeln("      <UnitOutputDirectory Value=\"lib\\$(TargetCPU)-$(TargetOS)\"/>")
	w.Writeln("    </SearchPaths>")
	w.Writeln("    <Parsing>")
	w.Writeln("      <SyntaxOptions>")
	w.Writeln("        <IncludeAssertionCode Value=\"True\"/>")
	w.Writeln("      </SyntaxOptions>")
	w.Writeln("    </Parsing>")
	w.Writeln("    <CodeGeneration>")
	w.Writeln("      <RelocatableUnit Value=\"True\"/>")
	w.Writeln("    </CodeGeneration>")
	w.Writeln("    <Linking>")
	w.Writeln("      <Debugging>")
	w.Writeln("        <StripSymbols Value=\"True\"/>")
	w.Writeln("        <UseExternalDbgSyms Value=\"True\"/>")
	w.Writeln("      </Debugging>")
	w.Writeln("      <Options>")
	w.Writeln("        <ExecutableType Value=\"Library\"/>")
	w.Writeln("      </Options>")
	w.Writeln("    </Linking>")
	w.Writeln("  </CompilerOptions>")
	w.Writeln("  <Debugging>")
	w.Writeln("    <Exceptions Count=\"%d\">", 3)
	w.Writeln("      <Item1>")
	w.Writeln("        <Name Value=\"EAbort\"/>")
	w.Writeln("      </Item1>")
	w.Writeln("      <Item2>")
	w.Writeln("        <Name Value=\"ECodetoolError\"/>")
	w.Writeln("      </Item2>")
	w.Writeln("      <Item3>")
	w.Writeln("        <Name Value=\"EFOpenError\"/>")
	w.Writeln("      </Item3>")
	w.Writeln("    </Exceptions>")
	w.Writeln("  </Debugging>")
	w.Writeln("</CONFIG>")
	w.Writeln("")

	return nil

}

func getPascalImplClassParameters(method ComponentDefinitionMethod, NameSpace string, ClassName string, isGlobal bool, isImplementation bool) (string, string, error) {
	parameters := ""
	returnType := ""

	for k := 0; k < len(method.Params); k++ {
		param := method.Params[k]
		ParamTypeName, err := getPascalParameterType(param.ParamType, NameSpace, param.ParamClass, false, isImplementation)
		if err != nil {
			return "", "", err
		}

		switch param.ParamPass {
		case "in":
			switch param.ParamType {
			case "basicarray", "structarray":
				if parameters != "" {
					parameters = parameters + "; "
				}
				parameters = parameters + "const A" + param.ParamName + "Count: QWord"
				parameters = parameters + "; const A" + param.ParamName + ": " + ParamTypeName
			case "class", "optionalclass":
				if parameters != "" {
					parameters = parameters + "; "
				}
				parameters = parameters + "A" + param.ParamName + ": " + ParamTypeName
			default:
				if parameters != "" {
					parameters = parameters + "; "
				}
				parameters = parameters + "const A" + param.ParamName + ": " + ParamTypeName
			}

		case "out":
			switch param.ParamType {
			case "basicarray", "structarray":
				if parameters != "" {
					parameters = parameters + "; "
				}
				parameters = parameters + "const A" + param.ParamName + "Count: QWord"
				parameters = parameters + "; P" + param.ParamName + "NeededCount: PQWord"
				parameters = parameters + "; A" + param.ParamName + ": " + ParamTypeName
			default:
				if parameters != "" {
					parameters = parameters + "; "
				}
				parameters = parameters + "out A" + param.ParamName + ": " + ParamTypeName
			}

		case "return":
			if param.ParamType == "basicarray" || param.ParamType == "structarray" {
				return "", "", fmt.Errorf("return value \"%s\" is not supported for Pascal method \"%s\"", param.ParamName, method.MethodName)
			}
			if returnType != "" {
				return "", "", fmt.Errorf("duplicate return value \"%s\" for Pascal method \"%s\"", param.ParamName, method.MethodName)
			}
			returnType = ParamTypeName
		}
	}

	return parameters, returnType, nil
}

func writePascalClassMethodDummyStub(method ComponentDefinitionMethod, w LanguageWriter, NameSpace string, ClassName string, outClassName string, isGlobal bool, customImplementation []string) error {
	parameters, returnType, err := getPascalImplClassParameters(method, NameSpace, ClassName, isGlobal, true)
	if err != nil {
		return err
	}

	classPrefix := ""
	if isGlobal {
		classPrefix = "class "
	}

	if returnType == "" {
		w.Writeln("%sprocedure %s.%s(%s);", classPrefix, outClassName, method.MethodName, parameters)
	} else {
		w.Writeln("%sfunction %s.%s(%s): %s;", classPrefix, outClassName, method.MethodName, parameters, returnType)
	}

	w.Writeln("begin")
	w.Writelns("  ", customImplementation)
	w.Writeln("end;")
	w.Writeln("")

	return nil
}

func buildPascalStub(component ComponentDefinition, NameSpace string, ClassIdentifier string, BaseName string, outputFolder string, indentString string, stubIdentifier string, forceRecreation bool, defaultImplementation []string) error {

	pascalBaseClassName := "T" + ClassIdentifier + NameSpace + component.baseClass().ClassName

	var baseClassMethods [5]ComponentDefinitionMethod
	baseClassMethods[0] = GetLastErrorMessageMethod()
	baseClassMethods[1] = ClearErrorMessageMethod()
	baseClassMethods[2] = RegisterErrorMessageMethod()
	baseClassMethods[3] = IncRefCountMethod()
	baseClassMethods[4] = DecRefCountMethod()

	var baseClassMethodImplementation [5][]string
	baseClassMethodImplementation[0] = append(baseClassMethodImplementation[0], "result := (FMessages.Count>0);")
	baseClassMethodImplementation[0] = append(baseClassMethodImplementation[0], "if (result) then")
	baseClassMethodImplementation[0] = append(baseClassMethodImplementation[0], "  AErrorMessage := FMessages[FMessages.Count-1];")
	baseClassMethodImplementation[1] = append(baseClassMethodImplementation[1], "FMessages.Clear();")
	baseClassMethodImplementation[2] = append(baseClassMethodImplementation[2], "FMessages.Add(AErrorMessage);")

	baseClassMethodImplementation[3] = append(baseClassMethodImplementation[3], "inc(FReferenceCount);")

	baseClassMethodImplementation[4] = append(baseClassMethodImplementation[4], "dec(FReferenceCount);")
	baseClassMethodImplementation[4] = append(baseClassMethodImplementation[4], "if (FReferenceCount = 0) then begin")
	baseClassMethodImplementation[4] = append(baseClassMethodImplementation[4], "  self.Destroy();")
	baseClassMethodImplementation[4] = append(baseClassMethodImplementation[4], "  result := true;")
	baseClassMethodImplementation[4] = append(baseClassMethodImplementation[4], "end")
	baseClassMethodImplementation[4] = append(baseClassMethodImplementation[4], "else")
	baseClassMethodImplementation[4] = append(baseClassMethodImplementation[4], "  result := false;")

	for i := 0; i < len(component.Classes); i++ {
		class := component.Classes[i]
		outClassName := "T" + ClassIdentifier + NameSpace + class.ClassName

		outparentClassName := ""

		if class.ParentClass != "" {
			outparentClassName = "T" + ClassIdentifier + NameSpace + class.ParentClass
		} else {
			outparentClassName = pascalBaseClassName
		}

		StubFileName := path.Join(outputFolder, BaseName+stubIdentifier+"_"+strings.ToLower(class.ClassName)+".pas")
		if !forceRecreation && (FileExists(StubFileName)) {
			log.Printf("Omitting recreation of Stub implementation for \"%s\"", outClassName)
			continue
		}

		log.Printf("Creating \"%s\"", StubFileName)
		w, err := CreateLanguageFile(StubFileName, indentString)
		if err != nil {
			return err
		}
		w.WritePascalLicenseHeader(component,
			fmt.Sprintf("This is the class declaration of %s", outClassName),
			false)

		w.Writeln("{$MODE DELPHI}")
		w.Writeln("unit %s%s_%s;", BaseName, stubIdentifier, strings.ToLower(class.ClassName))
		w.Writeln("")
		w.Writeln("interface")
		w.Writeln("")
		w.Writeln("uses")
		w.Writeln("  %s_types,", BaseName)
		w.Writeln("  %s_interfaces,", BaseName)
		w.Writeln("  %s_exception,", BaseName)

		if class.ParentClass != "" {
			w.Writeln("  %s%s_%s,", BaseName, stubIdentifier, strings.ToLower(class.ParentClass))
		}

		writeSubComponentUses(component, w, true)

		w.Writeln("  Classes,")
		w.Writeln("  sysutils;")
		w.Writeln("")

		w.Writeln("type")
		if component.isBaseClass(class) {
			w.Writeln("  %s = class(TObject, I%s%s)", outClassName, NameSpace, class.ClassName)
		} else {
			w.Writeln("  %s = class(%s, I%s%s)", outClassName, outparentClassName, NameSpace, class.ClassName)
		}

		w.Writeln("    private")
		if component.isBaseClass(class) {
			w.Writeln("      FMessages: TStringList;")
			w.Writeln("      FReferenceCount: integer;")
		}
		w.Writeln("")
		w.Writeln("    protected")
		w.Writeln("")
		w.Writeln("    public")

		if component.isBaseClass(class) {
			w.Writeln("      constructor Create();")
			w.Writeln("      destructor Destroy(); override;")
		}

		w.AddIndentationLevel(3)
		if component.isBaseClass(class) {
			for _, method := range baseClassMethods {
				err := writePascalImplClassMethodDefinition(method, w, NameSpace, class.ClassName, false)
				if err != nil {
					return err
				}
			}
		}

		for j := 0; j < len(class.Methods); j++ {
			method := class.Methods[j]
			err := writePascalImplClassMethodDefinition(method, w, NameSpace, class.ClassName, false)
			if err != nil {
				return err
			}
		}
		w.AddIndentationLevel(-3)

		w.Writeln("  end;")
		w.Writeln("")

		w.Writeln("implementation")
		w.Writeln("")

		if component.isBaseClass(class) {
			w.Writeln("constructor %s.Create();", pascalBaseClassName)
			w.Writeln("begin")
			w.Writeln("  inherited Create();")
			w.Writeln("  FMessages := TStringList.Create();")
			w.Writeln("  FReferenceCount := 1;")
			w.Writeln("end;")
			w.Writeln("")
			w.Writeln("destructor %s.Destroy();", pascalBaseClassName)
			w.Writeln("begin")
			w.Writeln("  FreeAndNil(FMessages);")
			w.Writeln("  inherited Destroy();")
			w.Writeln("end;")
			w.Writeln("")
		}

		if component.isBaseClass(class) {
			for i, method := range baseClassMethods {
				err := writePascalClassMethodDummyStub(method, w, NameSpace, class.ClassName, outClassName, false, baseClassMethodImplementation[i])
				if err != nil {
					return err
				}
			}
		}

		for j := 0; j < len(class.Methods); j++ {
			method := class.Methods[j]
			err := writePascalClassMethodDummyStub(method, w, NameSpace, class.ClassName, outClassName, false, defaultImplementation)
			if err != nil {
				return err
			}
		}

		w.Writeln("end.")
	}

	return nil
}

func writePascalImplClassMethodDefinition(method ComponentDefinitionMethod, w LanguageWriter, NameSpace string, ClassName string, isGlobal bool) error {

	parameters, returnType, err := getPascalImplClassParameters(method, NameSpace, ClassName, isGlobal, true)
	if err != nil {
		return err
	}

	classPrefix := ""
	if isGlobal {
		classPrefix = "class "
	}

	if returnType == "" {
		w.Writeln("%sprocedure %s(%s);", classPrefix, method.MethodName, parameters)
	} else {
		w.Writeln("%sfunction %s(%s): %s;", classPrefix, method.MethodName, parameters, returnType)
	}

	return nil
}

func writeSubComponentUses(component ComponentDefinition, w LanguageWriter, onlyUsedOnes bool) {
	// TODO: check if component is used
	if len(component.ImportedComponentDefinitions) > 0 {
		for _, subComponent := range component.ImportedComponentDefinitions {
			w.Writeln("  Unit_%s,", subComponent.NameSpace)
		}
	}
}

func buildStubImplementation(component ComponentDefinition, w LanguageWriter, NameSpace string, BaseName string, stubIdentifier string, defaultImplementation []string) error {

	global := component.Global

	w.Writeln("{$MODE DELPHI}")
	w.Writeln("Unit %s%s;", BaseName, stubIdentifier)
	w.Writeln("")
	w.Writeln("interface")
	w.Writeln("")
	w.Writeln("uses")
	w.Writeln("  %s_types,", BaseName)
	w.Writeln("  %s_exception,", BaseName)
	w.Writeln("  %s_interfaces,", BaseName)
	w.Writeln("  Classes,")
	writeSubComponentUses(component, w, false)
	w.Writeln("  sysutils;")
	w.Writeln("")
	w.Writeln("type")
	w.Writeln("  T%sWrapper = class(TObject)", NameSpace)
	if len(component.ImportedComponentDefinitions) > 0 {
		w.Writeln("    private")
		for _, subComponent := range component.ImportedComponentDefinitions {
			w.Writeln("      class var G%sWrapper: T%sWrapper;", subComponent.NameSpace, subComponent.NameSpace)
		}
		for _, subComponent := range component.ImportedComponentDefinitions {
			w.Writeln("      class function Get%sWrapper: T%sWrapper; static;", subComponent.NameSpace, subComponent.NameSpace)
			w.Writeln("      class procedure Set%sWrapper(A%sWrapper: T%sWrapper); static;", subComponent.NameSpace, subComponent.NameSpace, subComponent.NameSpace)
		}
	}

	w.Writeln("    public")
	for _, subComponent := range component.ImportedComponentDefinitions {
		w.Writeln("      class property %sWrapper: T%sWrapper read Get%sWrapper write Set%sWrapper;", subComponent.NameSpace, subComponent.NameSpace, subComponent.NameSpace, subComponent.NameSpace)
	}
	w.AddIndentationLevel(3)
	for j := 0; j < len(global.Methods); j++ {
		method := global.Methods[j]
		isSpecialFunction, err := CheckHeaderSpecialFunction(method, global)
		if err != nil {
			return err
		}
		if (isSpecialFunction == eSpecialMethodJournal) || (isSpecialFunction == eSpecialMethodInjection) ||
			(isSpecialFunction == eSpecialMethodSymbolLookup) {
			continue
		}
		err = writePascalImplClassMethodDefinition(method, w, NameSpace, "Wrapper", true)
		if err != nil {
			return err
		}
	}
	w.AddIndentationLevel(-3)

	w.Writeln("  end;")
	w.Writeln("")
	w.Writeln("")
	w.Writeln("implementation")
	w.Writeln("")

	for _, subComponent := range component.ImportedComponentDefinitions {
		theNameSpace := subComponent.NameSpace
		w.Writeln("class procedure T%sWrapper.Set%sWrapper(A%sWrapper: T%sWrapper);", NameSpace, theNameSpace, theNameSpace, theNameSpace)
		w.Writeln("  begin")
		w.Writeln("  if assigned(G%sWrapper) then", theNameSpace)
		w.Writeln("    raise E%sException.Create(%s_ERROR_COULDNOTLOADLIBRARY);", NameSpace, strings.ToUpper(NameSpace))
		w.Writeln("  G%sWrapper := A%sWrapper;", theNameSpace, theNameSpace)
		w.Writeln("end;")
		w.Writeln("")
		w.Writeln("class function T%sWrapper.Get%sWrapper(): T%sWrapper;", NameSpace, theNameSpace, theNameSpace)
		w.Writeln("begin")
		w.Writeln("  if not assigned(G%sWrapper) then", theNameSpace)
		w.Writeln("    raise E%sException.Create(%s_ERROR_COULDNOTLOADLIBRARY);", NameSpace, strings.ToUpper(NameSpace))
		w.Writeln("  result := G%sWrapper;", theNameSpace)
		w.Writeln("end;")
		w.Writeln("")
	}

	for j := 0; j < len(global.Methods); j++ {
		method := global.Methods[j]
		thisMethodDefaultImpl := defaultImplementation

		isSpecialFunction, err := CheckHeaderSpecialFunction(method, global)
		if err != nil {
			return err
		}
		if isSpecialFunction == eSpecialMethodRelease {
			var releaseImplementation []string
			releaseImplementation = append(releaseImplementation,
				fmt.Sprintf("(A%s as I%s%s).DecRefCount(); ", method.Params[0].ParamName, NameSpace, global.BaseClassName))
			thisMethodDefaultImpl = releaseImplementation
		}
		if isSpecialFunction == eSpecialMethodAcquire {
			var acquireImplementation []string
			acquireImplementation = append(acquireImplementation,
				fmt.Sprintf("(A%s as I%s%s).IncRefCount(); ", method.Params[0].ParamName, NameSpace, global.BaseClassName))
			thisMethodDefaultImpl = acquireImplementation
		}
		if (isSpecialFunction == eSpecialMethodJournal) || (isSpecialFunction == eSpecialMethodInjection) ||
			(isSpecialFunction == eSpecialMethodSymbolLookup) {
			continue
		}
		if isSpecialFunction == eSpecialMethodVersion {
			var versionImplementation []string
			versionImplementation = append(versionImplementation,
				fmt.Sprintf("A%s := %s_VERSION_MAJOR;", method.Params[0].ParamName, strings.ToUpper(NameSpace)),
				fmt.Sprintf("A%s := %s_VERSION_MINOR;", method.Params[1].ParamName, strings.ToUpper(NameSpace)),
				fmt.Sprintf("A%s := %s_VERSION_MICRO;", method.Params[2].ParamName, strings.ToUpper(NameSpace)))
			thisMethodDefaultImpl = versionImplementation
		}

		err = writePascalClassMethodDummyStub(method, w, NameSpace, "Wrapper", "T"+NameSpace+"Wrapper", true, thisMethodDefaultImpl)
		if err != nil {
			return err
		}
	}
	w.Writeln("")

	w.Writeln("end.")

	return nil
}
