/* SPDX-FileCopyrightText: 2025 Greenbone AG
 *
 * SPDX-License-Identifier: AGPL-3.0-or-later
 */

import {describe, test, expect} from '@gsa/testing';
import {rendererWith, screen} from 'web/testing';
import Permission from 'gmp/models/permission';
import PermissionDetails from 'web/pages/permissions/PermissionDetails';

describe('PermissionDetails tests', () => {
  test('should render comment', () => {
    const permission = new Permission({
      comment: 'Test permission comment',
      name: 'get_tasks',
    });
    const {render} = rendererWith({capabilities: true});
    render(<PermissionDetails entity={permission} />);
    expect(screen.getByText('Comment')).toBeVisible();
    expect(screen.getByText('Test permission comment')).toBeVisible();
  });

  test('should not render comment row when comment is undefined', () => {
    const permission = new Permission({
      name: 'get_tasks',
    });
    const {render} = rendererWith({capabilities: true});
    render(<PermissionDetails entity={permission} />);
    expect(screen.queryByText('Comment')).not.toBeInTheDocument();
  });

  test('should render description', () => {
    const permission = new Permission({
      name: 'get_tasks',
      comment: 'Test comment',
    });
    const {render} = rendererWith({capabilities: true});
    render(<PermissionDetails entity={permission} />);
    expect(screen.getByText('Description')).toBeVisible();
    // The description should be generated by permissionDescription function
    expect(screen.getByText('Has read access to unknown')).toBeVisible();
  });

  test('should render resource information', () => {
    const permission = Permission.fromElement({
      name: 'get_tasks',
      comment: 'Test comment',
      resource: {
        _id: 'resource-123',
        type: 'task',
      },
    });
    const {render} = rendererWith({capabilities: true});
    render(<PermissionDetails entity={permission} />);
    expect(screen.getByText('Resource')).toBeVisible();
    expect(screen.getByText('Task')).toBeVisible();
    // The resource might not have a name or it could be empty
  });

  test('should not render resource row when resource is undefined', () => {
    const permission = new Permission({
      name: 'get_tasks',
      comment: 'Test comment',
    });
    const {render} = rendererWith({capabilities: true});
    render(<PermissionDetails entity={permission} />);
    expect(screen.queryByText('Resource')).not.toBeInTheDocument();
  });

  test('should render subject information', () => {
    const permission = Permission.fromElement({
      name: 'get_tasks',
      comment: 'Test comment',
      subject: {
        _id: 'user-123',
        type: 'user',
      },
    });
    const {render} = rendererWith({capabilities: true});
    render(<PermissionDetails entity={permission} />);
    expect(screen.getByText('Subject')).toBeVisible();
    expect(screen.getByText('User')).toBeVisible();
    // The subject might not have a name or it could be empty
  });

  test('should not render subject row when subject is undefined', () => {
    const permission = new Permission({
      name: 'get_tasks',
      comment: 'Test comment',
    });
    const {render} = rendererWith({capabilities: true});
    render(<PermissionDetails entity={permission} />);
    expect(screen.queryByText('Subject')).not.toBeInTheDocument();
  });

  test('should render with both resource and subject', () => {
    const permission = Permission.fromElement({
      name: 'get_tasks',
      comment: 'Full permission test',
      resource: {
        _id: 'task-123',
        type: 'task',
      },
      subject: {
        _id: 'role-123',
        type: 'role',
      },
    });
    const {render} = rendererWith({capabilities: true});
    render(<PermissionDetails entity={permission} />);

    expect(screen.getByText('Comment')).toBeVisible();
    expect(screen.getByText('Full permission test')).toBeVisible();
    expect(screen.getByText('Description')).toBeVisible();
    expect(screen.getByText('Resource')).toBeVisible();
    expect(screen.getByText('Task')).toBeVisible();
    expect(screen.getByText('Subject')).toBeVisible();
    expect(screen.getByText('Role')).toBeVisible();
  });

  test('should render with empty comment', () => {
    const permission = new Permission({
      name: 'get_tasks',
      comment: '',
    });
    const {render} = rendererWith({capabilities: true});
    render(<PermissionDetails entity={permission} />);
    expect(screen.getByText('Comment')).toBeVisible();
    const commentRow = screen.getByText('Comment').closest('tr');
    expect(commentRow).toBeVisible();
  });

  test('should render Super permission description', () => {
    const permission = new Permission({
      name: 'Super',
      comment: 'Super user permission',
    });
    const {render} = rendererWith({capabilities: true});
    render(<PermissionDetails entity={permission} />);
    expect(screen.getByText('Description')).toBeVisible();
    expect(
      screen.getByText('Has super access to all resources of unknown'),
    ).toBeVisible();
  });

  test('should render permission with resource but without subject', () => {
    const permission = Permission.fromElement({
      name: 'modify_config',
      comment: 'Modify config permission',
      resource: {
        _id: 'config-123',
        type: 'config',
      },
    });
    const {render} = rendererWith({capabilities: true});
    render(<PermissionDetails entity={permission} />);

    expect(screen.getByText('Comment')).toBeVisible();
    expect(screen.getByText('Modify config permission')).toBeVisible();
    expect(screen.getByText('Resource')).toBeVisible();
    expect(screen.getByText('Scan Config')).toBeVisible();
    expect(screen.queryByText('Subject')).not.toBeInTheDocument();
  });

  test('should render permission with subject but without resource', () => {
    const permission = Permission.fromElement({
      name: 'authenticate',
      comment: 'Login permission',
      subject: {
        _id: 'group-123',
        type: 'group',
      },
    });
    const {render} = rendererWith({capabilities: true});
    render(<PermissionDetails entity={permission} />);

    expect(screen.getByText('Comment')).toBeVisible();
    expect(screen.getByText('Login permission')).toBeVisible();
    expect(screen.getByText('Subject')).toBeVisible();
    expect(screen.getByText('Group')).toBeVisible();
    expect(screen.queryByText('Resource')).not.toBeInTheDocument();
  });

  test('should render minimal permission without comment, resource, or subject', () => {
    const permission = new Permission({
      name: 'authenticate',
    });
    const {render} = rendererWith({capabilities: true});
    render(<PermissionDetails entity={permission} />);

    expect(screen.queryByText('Comment')).not.toBeInTheDocument();
    expect(screen.getByText('Description')).toBeVisible();
    expect(screen.getByText('May login')).toBeVisible();
    expect(screen.queryByText('Resource')).not.toBeInTheDocument();
    expect(screen.queryByText('Subject')).not.toBeInTheDocument();
  });

  test('should render description with resource information when resource is provided', () => {
    const permission = Permission.fromElement({
      name: 'get_tasks',
      resource: {
        _id: 'task-456',
        type: 'task',
      },
    });
    const {render} = rendererWith({capabilities: true});
    render(<PermissionDetails entity={permission} />);

    expect(screen.getByText('Description')).toBeVisible();
    // The description should include the resource in the permission description
    expect(screen.getByText(/Has read access to/)).toBeVisible();
  });

  test('should render description with subject information when subject is provided', () => {
    const permission = Permission.fromElement({
      name: 'get_tasks',
      subject: {
        _id: 'user-789',
        type: 'user',
      },
    });
    const {render} = rendererWith({capabilities: true});
    render(<PermissionDetails entity={permission} />);

    expect(screen.getByText('Description')).toBeVisible();
    // When subject is provided, description should include subject information
    expect(screen.getByText(/User.*has read access/)).toBeVisible();
  });
});
