/*
 * Copyright (c) 2010-2011, Jan Friesse <honzaf@users.sourceforge.net>
 *
 * Permission to use, copy, modify, and/or distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <sys/types.h>

#include <sys/queue.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <sys/uio.h>

#include <arpa/inet.h>
#include <net/if.h>
#include <netinet/in.h>

#include <err.h>
#include <ifaddrs.h>
#define __STDC_FORMAT_MACROS
#include <inttypes.h>
#include <limits.h>
#include <errno.h>
#include <netdb.h>
#include <poll.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>

#include "odhcploc.h"
#include "opt.h"

/*
 * Defines
 */

/*
 * Standard ports
 */
#define BOOTPC_PORT	68
#define BOOTPS_PORT	67

/*
 * My hostname for DHCP message
 */
#define HOSTNAME	"ROQUE"

/*
 * Maximum DHCP message length
 */
#define MSG_LEN		576

/*
 * Macros
 */
#define STR_VALUE(val)		#val
#define QUOTE_VALUE(val)	STR_VALUE(val)

#ifndef INFTIM
#define INFTIM -1
#endif

/*
 * Enums
 */

/*
 * DHCP message fields
 */
enum {
	DHCP_FP_OP	= 0,
	DHCP_FP_HTYPE	= DHCP_FP_OP	 + 1,
	DHCP_FP_HLEN	= DHCP_FP_HTYPE	 + 1,
	DHCP_FP_HOPS	= DHCP_FP_HLEN	 + 1,
	DHCP_FP_XID	= DHCP_FP_HOPS	 + 1,
	DHCP_FP_SECS	= DHCP_FP_XID	 + 4,
	DHCP_FP_FLAGS	= DHCP_FP_SECS	 + 2,
	DHCP_FP_CIADDR	= DHCP_FP_FLAGS	 + 2,
	DHCP_FP_YIADDR	= DHCP_FP_CIADDR + 4,
	DHCP_FP_SIADDR	= DHCP_FP_YIADDR + 4,
	DHCP_FP_GIADDR	= DHCP_FP_SIADDR + 4,
	DHCP_FP_CHADDR	= DHCP_FP_GIADDR + 4,
	DHCP_FP_SNAME	= DHCP_FP_CHADDR + 16,
	DHCP_FP_FILE	= DHCP_FP_SNAME	 + 64,
	DHCP_FP_OPTIONS	= DHCP_FP_FILE	 + 128
};

/*
 * Structures
 */

/*
 * Local interface informations
 */
struct local_if {
	struct sockaddr_in broad_addr;
	struct sockaddr_in local_addr;
	char *device_name;
	const char *local_ip_s;
	unsigned int if_index;
};

/*
 * CLI parameters
 */
struct params {
	unsigned int	count;
	unsigned int	interval;
	unsigned int	suppress;
	unsigned int	verbose;
	struct local_if lif;
	unsigned int	validipc;
	char		**validipv;
};

/*
 * Valid IP list item
 */
struct validip_item {
	struct addrinfo			*ai;
	SLIST_ENTRY(validip_item)	entries;
};

/*
 * List of valid IPs
 */
SLIST_HEAD(validip_item_head, validip_item) validip_list;

/*
 * Cli parsed params
 */
struct params cli_params;

/*
 * Prototypes
 */
static unsigned int	 abs_time_diff(const struct timeval *tv1,
    const struct timeval *tv2);

static int		 build_validip_list(int validipc, char *validipv[]);

static int		 create_socket(const struct local_if *lif);

static void		 decode_dhcpmsg(const char *msg, size_t msg_len,
    enum opt_msg_type *msg_type, int verbose);

static void		 decode_dhcpmsg_opts(const char *msg, size_t msg_len,
    enum opt_msg_type *msg_type, enum opt_overload *overload, int verbose);

static size_t		 dhcpmsg_fill(char *msg, size_t msg_len);

static int		 find_local_if(const char *local_ip,
    struct local_if *lif);

static void		 free_validip_list(void);

static int		 is_valid_ip(const struct sockaddr *sa);

static int		 parse_cmdline(int argc, char *argv[],
    struct params *params);

static void		 print_bootp_info(const char *msg, size_t msg_len);

static void		 print_from_info(
    const struct sockaddr_storage *addr_from);

static void		 print_reply_info(
    const struct sockaddr_storage *addr_from, const char *msg, size_t msg_len,
    int verbose);

static void		 read_reply(int sock, int interval, int verbose,
    int suppress);

static struct addrinfo  *return_addr_info(const char *hostname,
    const char *port);

static void		 send_request(int sock, const struct local_if *lif);

static void		 show_version(void);

static void		 sigint_handler(int sig);

static void		 usage(void);

/*
 * Functions
 */

/*
 * Entry point of ODHCPloc. The odhcploc is program to locate and display
 * active DHCP servers on subnet. Program sends standard DHCP discover
 * messages. If it detects any unauthorized server (server not included
 * in valid_dhcp list), it displays it in special way. It's also possible
 * to display only unauthorized servers. Received DHCP message may be printed
 * in short format or in full format with whole packet decoded.
 */
int
main(int argc, char *argv[])
{
	int sock;

	memset(&cli_params, 0, sizeof(cli_params));

	signal(SIGINT, sigint_handler);

	parse_cmdline(argc, argv, &cli_params);

	build_validip_list(cli_params.validipc, cli_params.validipv);

	sock = create_socket(&cli_params.lif);

	while (cli_params.count > 0) {
		send_request(sock, &cli_params.lif);
		read_reply(sock, cli_params.interval, cli_params.verbose,
		    cli_params.suppress);

		cli_params.count--;
	}

	sigint_handler(SIGINT);

	return (0);
}

/*
 * Static functions
 */

/*
 * Return absolute value of difference between two timestamps tv1 and tv2 in
 * milliseconds.
 */
static unsigned int
abs_time_diff(const struct timeval *tv1, const struct timeval *tv2)
{
	uint64_t utv1, utv2, tmp;

	utv1 = tv1->tv_sec * 1000 + tv1->tv_usec / 1000;
	utv2 = tv2->tv_sec * 1000 + tv2->tv_usec / 1000;
	if (utv1 > utv2) {
		tmp = utv1;
		utv1 = utv2;
		utv2 = tmp;
	}

	return ((unsigned int )(utv2 - utv1));
}

/*
 * Create list of valid IP addresses. validipc is number of addresses in string
 * format stored in arrat validipv.
 * Return 0 on success
 */
static int
build_validip_list(int validipc, char *validipv[])
{
	struct addrinfo *ai;
	struct validip_item *vip;
	int i;

	SLIST_INIT(&validip_list);

	for (i = 0; i < validipc; i++) {
		vip = (struct validip_item *)malloc(sizeof(*vip));
		if (vip == NULL) {
			errx(1, "Low memory!");
		}

		ai = return_addr_info(validipv[i], NULL);

		vip->ai = ai;

		SLIST_INSERT_HEAD(&validip_list, vip, entries);
	}

	return (0);
}

/*
 * Create socket for sending DHCP requests and replies. Socket is also binded
 * and correct socket options are set. lif is local interface structure pointer
 * filled by informations about local interface.
 * Returns socket file descriptor, otherwise -1.
 */
static int
create_socket(const struct local_if *lif)
{
	struct sockaddr_in addr_any;
	int optval;
	int res;
	int sock;

	sock = socket(AF_INET, SOCK_DGRAM, 0);
	if (sock == -1) {
		err(1, "Can't create socket");
	}

	optval = 1;

	res = setsockopt(sock, SOL_SOCKET, SO_BROADCAST, &optval,
	    sizeof(optval));
	if (res == -1)
		goto sockopt_err;

	res = setsockopt(sock, SOL_SOCKET, SO_REUSEADDR, &optval,
	    sizeof(optval));
	if (res == -1)
		goto sockopt_err;

#ifdef SO_REUSEPORT
	res = setsockopt(sock, SOL_SOCKET, SO_REUSEPORT, &optval,
	    sizeof(optval));
	if (res == -1)
		goto sockopt_err;
#endif

	optval = 16;
	res = setsockopt(sock, IPPROTO_IP, IP_TTL, &optval, sizeof(optval));
	if (res == -1)
		goto sockopt_err;

#ifdef SO_BINDTODEVICE
	setsockopt(sock, SOL_SOCKET, SO_BINDTODEVICE, lif->device_name,
	    sizeof(lif->device_name));

	/*
	 * Ignore errors because (for example) bridge may not considered as
	 * valid device
	 */
#endif

#ifdef IP_ONESBCAST
	optval = 1;

	res = setsockopt(sock, IPPROTO_IP, IP_ONESBCAST, &optval,
	    sizeof(optval));
	if (res == -1)
		goto sockopt_err;
#endif

#ifdef IP_PKTINFO
	optval = 1;

	res = setsockopt(sock, IPPROTO_IP, IP_PKTINFO, &optval, sizeof(optval));
	if (res == -1)
		goto sockopt_err;
#endif

	memset((char *)&addr_any, 0, sizeof(addr_any));
	addr_any.sin_family = AF_INET;
#ifdef __CYGWIN__
	/*
	 * Cygwin binds to local IP address
	 */
	addr_any.sin_addr = lif->local_addr.sin_addr;
#else
	/*
	 * Other platforms binds to INADDR_ANY
	 */
	addr_any.sin_addr.s_addr = htonl(INADDR_ANY);
#endif
	addr_any.sin_port = htons(BOOTPC_PORT);

	res = bind(sock, (struct sockaddr *)&addr_any, sizeof(addr_any));
	if (res == -1)
		err(1, "Can't bind socket");

	return (sock);

sockopt_err:
	err(1, "Can't set sock option");
}

/*
 * Decode received DHCP reply. msg is actual received DHCP reply packet with
 * msg_len length. msg_type is filled by type of DHCP message given from message
 * and verbose is boolean flag giving verbosity (display decoded DHCP options)
 */
static void
decode_dhcpmsg(const char *msg, size_t msg_len, enum opt_msg_type *msg_type,
    int verbose)
{
	const char *msg_p;
	enum opt_overload overload;
	uint8_t mc[4];

	if (verbose) {
		print_bootp_info(msg, msg_len);
	}

	if (msg_len < DHCP_FP_OPTIONS + 4) {
		if (verbose) {
			printf("Message is pure BOOTP\n");
		}

		return ;
	}

	mc[0] = 0x63; mc[1] = 0x82; mc[2] = 0x53; mc[3] = 0x63;

	if (memcmp(msg + DHCP_FP_OPTIONS, mc, sizeof(mc)) != 0) {
		if (verbose) {
			printf("Invalid magic cookie\n");
		}

		return ;
	} else {
		if (verbose)
			printf("Valid magic cookie\n");
	}

	overload = OPT_OVERLOAD_NONE;
	*msg_type = OPT_MSG_TYPE_NONE;

	msg_p = msg + DHCP_FP_OPTIONS + sizeof(mc);

	decode_dhcpmsg_opts(msg_p, msg_len, msg_type, &overload, verbose);

	if (overload & OPT_OVERLOAD_FILE) {
		decode_dhcpmsg_opts(msg + DHCP_FP_FILE, 128, msg_type, NULL,
		    verbose);
	};

	if (overload & OPT_OVERLOAD_SNAME) {
		decode_dhcpmsg_opts(msg + DHCP_FP_SNAME, 64, msg_type, NULL,
		    verbose);
	};
}

/*
 * Decode DHCP message options. msg is received packet with msg_len length.
 * msg_type is filled according to decoded packet and donates type of message.
 * overload is also filled according to decoded packet and donates type of
 * overload used in DHCP message. verbose is boolean flag used for printing
 * verbose informations (decoded options)
 */
static void
decode_dhcpmsg_opts(const char *msg, size_t msg_len,
    enum opt_msg_type *msg_type, enum opt_overload *overload, int verbose)
{
	const char *msg_p;
	int end_reached;
	uint8_t opt;
	uint8_t opt_len;

	msg_p = msg;

	end_reached = 0;

	while ((size_t)(msg_p - msg) < msg_len && !end_reached) {
		opt = *msg_p++;

		if (opt != 0 && opt != 255) {
			opt_len = *msg_p++;
		} else {
			opt_len = 0;
		}

		if ((size_t)(msg_p - msg + opt_len) > msg_len) {
			if (verbose)
				printf("invalid len\n");
			return ;
		}

		if (verbose)
			printf("Option(t=%3u", opt);

		if (opt_len > 0) {
			if (verbose)
				printf(", l=%3u", opt_len);
		}

		if (verbose)
			printf("): ");

		opt_decode(opt, opt_len, msg_p, &end_reached, msg_type,
		    overload, verbose);

		if (verbose)
			printf("\n");

		msg_p += opt_len;
	}
}

/*
 * Fill DHCP request message. Message is stored in msg with maximum length
 * msg_len.
 * Return number of bytes really filled
 */
static size_t
dhcpmsg_fill(char *msg, size_t msg_len)
{
	char *orig_msg;
	int i;
	uint16_t u16;

	orig_msg = msg;

	memset(msg, 0, msg_len);

	*msg++ = 0x01; /* op - BOOTREQUEST */
	*msg++ = 0x01; /* htype - 10mb ethernet */
	*msg++ = 0x06; /* hlen - 10mb ethernet */
	msg++; /* Hops */
	msg += 4; /* xid */

	/* secs */
	u16 = htons(60);
	memcpy(msg, &u16, sizeof(u16)); msg += sizeof(u16);

	/* flags - Set BROADCAST flag */
	u16 = htons(0x8000);
	memcpy(msg, &u16, sizeof(u16)); msg += sizeof(u16);

	msg += 4; /* ciaddr */
	msg += 4; /* yiaddr */
	msg += 4; /* siaddr */
	msg += 4; /* giaddr */

	/* chaddr */
	for (i = 0; i < 6; i++) {
		*msg++ = 0x31 + i;
	}
	msg += 16 - 6;

	msg += 64; /* sname */
	msg += 128; /* file */

	/*
	 * Set Magic Cookie
	 */
	*msg++ = 0x63; *msg++ = 0x82; *msg++ = 0x53; *msg++ = 0x63;

	/*
	 * Set DHCP Request type to DHCP Discover
	 */
	*msg++ = 53; *msg++ = 0x01; *msg++ = 0x01;

	/*
	 * Set Hostname
	 */
	*msg++ = 12; *msg++ = strlen(HOSTNAME) + 1;
	memcpy(msg, HOSTNAME, strlen(HOSTNAME) + 1);
	msg += strlen(HOSTNAME) + 1;

	/*
	 * Client identifier
	 */
	*msg++ = 61; *msg++ = 7; *msg++ = 0x01;

	for (i = 0; i < 6; i++) {
		*msg++ = 0x31 + i;
	}

	/*
	 * End option
	 */
	*msg++ = 0xFF;

	return (msg - orig_msg);
}

/*
 * Try to find local network interface with local_ip address. local_if structure
 * is filled after successful return with device_name, broad and local addr and
 * if_index. local_ip_s pointer is set to local_ip.
 * Function returns 0 on success, otherwise -1.
 */
static int
find_local_if(const char *local_ip, struct local_if *lif)
{
	struct addrinfo *ai;
	struct addrinfo *ai_i;
	struct ifaddrs *ifa;
	struct ifaddrs *ifa_i;
	struct in_addr ia1, ia2;
	struct sockaddr *local_addr;
	struct sockaddr *ifa_addr;
	int res;

	res = getifaddrs(&ifa);
	if (res != 0) {
		err(1, "Can't get local addresses");
		/* NOTREACHED */
	}

	ai = return_addr_info(local_ip, NULL);

	res = 0;
	local_addr = NULL;
	ifa_addr = NULL;

	for (ifa_i = ifa; ifa_i != NULL; ifa_i = ifa_i->ifa_next) {
		if (ifa_i->ifa_addr == NULL ||
		    ifa_i->ifa_addr->sa_family != AF_INET) {
			continue ;
		}

		ifa_addr = ifa_i->ifa_addr;

		for (ai_i = ai; ai_i != NULL; ai_i = ai_i->ai_next) {
			local_addr = ai_i->ai_addr;

			if (local_addr->sa_family != AF_INET) {
				continue ;
			}

			ia1 = ((struct sockaddr_in *)local_addr)->sin_addr;
			ia2 = ((struct sockaddr_in *)ifa_addr)->sin_addr;
			if (ia1.s_addr == ia2.s_addr) {
				res = 1;
				break ;
			}
		}

		if (res == 1) {
			break ;
		}
	}

	if (res && local_addr != NULL && ifa_addr != NULL) {
		lif->device_name = (char *)malloc(strlen(ifa_i->ifa_name) + 1);
		if (lif->device_name == NULL) {
			errx(1, "Low memory!");
		}
		strcpy(lif->device_name, ifa_i->ifa_name);

		res = if_nametoindex(lif->device_name);
		if (res == 0) {
			errx(1, "Can't convert if name %s to index!",
			    lif->device_name);
		}
		lif->if_index = res;

		if (ifa_i->ifa_broadaddr == NULL) {
			errx(1, "Can't get broadcast address for interface %s!",
			    lif->device_name);
		}
		memcpy(&lif->broad_addr, ifa_i->ifa_broadaddr,
		    sizeof(lif->broad_addr));

		memcpy(&lif->local_addr, local_addr, sizeof(lif->local_addr));
		lif->local_ip_s = local_ip;

		res = 0;
	} else {
		res = -1;
	}

	freeifaddrs(ifa);
	freeaddrinfo(ai);

	return (res);
}

/*
 * Free list of valid IPs. All informations stored in this list are also freed
 * (this include addrinfo and items itself).
 */
static void
free_validip_list(void)
{
	struct validip_item *vip;

	while (!SLIST_EMPTY(&validip_list)) {
		vip = SLIST_FIRST(&validip_list);
		SLIST_REMOVE_HEAD(&validip_list, entries);
		if (vip->ai != NULL) {
			freeaddrinfo(vip->ai);
		}
		free(vip);
     }
}

/*
 * Function return non 0 if address sa is in valid IPs list, otherwise 0.
 */
static int
is_valid_ip(const struct sockaddr *sa)
{
	struct validip_item *vip;
	struct addrinfo *ipp;
	struct sockaddr *addr;
	struct in_addr ia1, ia2;
	int res;

	res = 0;

	SLIST_FOREACH(vip, &validip_list, entries) {
		for (ipp = vip->ai; ipp != NULL; ipp = ipp->ai_next) {
			addr = ipp->ai_addr;

			if (addr->sa_family == AF_INET) {
				ia1 = ((struct sockaddr_in *)sa)->sin_addr;
				ia2 = ((struct sockaddr_in *)addr)->sin_addr;
				if (ia1.s_addr == ia2.s_addr) {
					res = 1;
					break;
				}
			}
		}
	}

	return (res);
}

/*
 * Parse command line. argc is number of arguments, actual arguments are stored
 * in argv array. These two variables should be directly passed from main.
 * params structure is filled with decoded informations.
 * Return 0 on success.
 */
static int
parse_cmdline(int argc, char *argv[], struct params *params)
{
	char *ep;
	int ch, num;
	int res;

	params->count = 1;
	params->interval = 30;
	params->suppress = 0;
	params->verbose = 0;

	while ((ch = getopt(argc, argv, "pVvc:i:")) != -1) {
		switch (ch) {
		case 'p':
			params->suppress = 1;
			break;
		case 'V':
			show_version();
			/* NOTREACHED */
			break;
		case 'v':
			params->verbose++;
			break;
		case 'i':
			num = strtol(optarg, &ep, 10);
			if (num < 1 || num * 1000.0 >= INT_MAX ||
			    *ep != '\0') {
				warnx("illegal number, -i argument -- %s",
				    optarg);
				usage();
			}

			params->interval = num;
			break;
		case 'c':
			errno = 0;
			num = strtol(optarg, &ep, 10);
			if (num < 1 || errno != 0 || *ep != '\0') {
				warnx("illegal number, -c argument -- %s",
				    optarg);
				usage();
			}

			params->count = num;
			break;
		default:
			usage();
			/* NOTREACHED */
		}
	}

	argc -= optind;
	argv += optind;

	if (argc < 1) {
		warnx("IP address of local NIC must be entered");
		usage();
		exit(1);
	}

	res = find_local_if(argv[0], &params->lif);
	if (res != 0) {
		warnx("Can't find local NIC with given IP address");
		usage();
		exit(1);
	}

	params->validipc = argc - 1;
	params->validipv = argv + 1;

	return (0);
}

/*
 * Display bootp packet information. msg is received packet to decode with
 * msg_len size.
 */
static void
print_bootp_info(const char *msg, size_t msg_len)
{
	char ipaddr_s[INET_ADDRSTRLEN];
	uint32_t u32;
	int i;
	uint16_t u16;
	uint8_t u8;

	u8 = *msg;
	printf("Message type: %u", u8);

	switch (u8) {
	case 1:
		printf(" Boot Request");
		break;
	case 2:
		printf(" Boot reply");
		break;
	default:
		printf(" Invalid type");
	}
	printf("\n");

	u8 = *(msg + DHCP_FP_HTYPE);
	printf("Hardware type: %"PRIu8"\n", u8);

	u8 = *(msg + DHCP_FP_HLEN);
	printf("Hardware addr len: %"PRIu8"\n", u8);

	u8 = *(msg + DHCP_FP_HOPS);
	printf("Hops: %"PRIu8"\n", u8);

	u32 = *(msg + DHCP_FP_XID);
	u32 = ntohl(u32);
	printf("Transaction ID: 0x%08"PRIx32"\n", u32);

	u16 = *(msg + DHCP_FP_SECS);
	u16 = ntohl(u16);
	printf("Seconds elapsed: %"PRIu16"\n", u16);

	u16 = *(msg + DHCP_FP_FLAGS);
	u16 = ntohl(u16);
	printf("Flags: 0x%04"PRIx16"\n", u16);

	inet_ntop(AF_INET, msg + DHCP_FP_CIADDR, ipaddr_s, INET_ADDRSTRLEN);
	printf("Client IP address: %s\n", ipaddr_s);

	inet_ntop(AF_INET, msg + DHCP_FP_YIADDR, ipaddr_s, INET_ADDRSTRLEN);
	printf("Your (client) IP address: %s\n", ipaddr_s);

	inet_ntop(AF_INET, msg + DHCP_FP_SIADDR, ipaddr_s, INET_ADDRSTRLEN);
	printf("Next server IP address: %s\n", ipaddr_s);

	inet_ntop(AF_INET, msg + DHCP_FP_GIADDR, ipaddr_s, INET_ADDRSTRLEN);
	printf("Relay agent IP address: %s\n", ipaddr_s);

	printf("Client MAC address: ");
	for (i = 0; i < 6; i++) {
		if (i != 0)
			printf(":");
		printf("%02x", (unsigned char)*(msg + DHCP_FP_CHADDR + i));
	}
	printf("\n");

	printf("Server host name ");
	if (*(msg + DHCP_FP_SNAME) == 0x0) {
		printf("not given\n");
	} else {
		printf("%s\n", msg + DHCP_FP_SNAME);
	}

	printf("Boot file name ");
	if (*(msg + DHCP_FP_SNAME) == 0x0) {
		printf("not given\n");
	} else {
		printf("%s\n", msg + DHCP_FP_FILE);
	}
}

/*
 * Display address of DHCP server from whose DHCP reply was received. addr_from
 * is address of DHCP server.
 */
static void
print_from_info(const struct sockaddr_storage *addr_from)
{
	char ipaddr_s[INET_ADDRSTRLEN];

	if (addr_from->ss_family == AF_INET) {
		inet_ntop(AF_INET,
		    &((const struct sockaddr_in *)addr_from)->sin_addr,
		    ipaddr_s, INET_ADDRSTRLEN);

		printf(" (S)%-*s", (4 * 3 + 3) + 5, ipaddr_s);
		if (!is_valid_ip((const struct sockaddr *)addr_from)) {
			printf("***");
		}
	} else {
		printf(" (S) ipv6");
	}
	printf("\n");
}

/*
 * Display DHCP reply info. addr_from is address of DHCP server from whom is
 * reply. msg is DHCP reply packet with msg_len length. verbose is boolean flag
 * giving level of verbosity (display decoded DHCP options).
 */
static void
print_reply_info(const struct sockaddr_storage *addr_from, const char *msg,
    size_t msg_len, int verbose)
{
	char ipaddr_s[INET_ADDRSTRLEN];
	char tmp_buf[256];
	const char *dhcp_mt[] = {
		"NONE",		/* 0 */
		"DISCOVER",	/* 1 */
		"OFFER",	/* 2 */
		"REQUEST",	/* 3 */
		"DECLINE",	/* 4 */
		"ACK",		/* 5 */
		"NAK",		/* 6 */
		"RELEASE",	/* 7 */
		"UNKNOWN",	/* 8 */
	};
	enum opt_msg_type msg_type;
	time_t tres;

	/*
	 * Display time
	 */
	tres = time(NULL);

	strftime(tmp_buf, sizeof(tmp_buf), "%T", localtime(&tres));
	printf("%s", tmp_buf);

	if (verbose) {
		print_from_info(addr_from);
	}

	if (msg_len < DHCP_FP_OPTIONS) {
		printf(" Too short message");

		if (!verbose) {
			print_from_info(addr_from);
		}

		return ;
	}

	decode_dhcpmsg(msg, msg_len, &msg_type, verbose);

	if (!verbose) {
		/*
		 * Display client IP
		 */
		inet_ntop(AF_INET, msg + DHCP_FP_YIADDR, ipaddr_s,
		    INET_ADDRSTRLEN);

		printf(" (IP)%-*s", (4 * 3 + 3) + 1, ipaddr_s);

		/*
		 * Display type
		 */
		printf("%-9s", dhcp_mt[msg_type]);
		/*
		 * Display from info
		 */
		print_from_info(addr_from);
	}
}

/*
 * Read reply from DHCP server. sock is socket used for replies listening,
 * interval is time in seconds to wait for answers. verbose is boolean flag
 * giving verbosity of output. suppress is boolean flag donating if replies from
 * servers in valid IP list should be shown or not
 */
static void
read_reply(int sock, int interval, int verbose, int suppress)
{
	char msg[MSG_LEN];
	struct pollfd pfd;
	struct sockaddr_storage addr_from;
	struct timeval start_time;
	struct timeval tv;
	ssize_t recv_size;
	socklen_t len_from;
	int timeout;
	int res;

	if ((res = gettimeofday(&start_time, NULL)) != 0) {
		err(1, "gettimeofday error");
	}

	do {
		pfd.fd = sock;
		pfd.events = POLLIN;
		pfd.revents = 0;

		if ((res = gettimeofday(&tv, NULL)) != 0) {
			err(1, "gettimeofday error");
		}

		timeout = (interval * 1000) - abs_time_diff(&start_time, &tv);
		if (timeout <= 0) {
			return ;
		}

		res = poll(&pfd, 1, timeout);

		if (res == -1 && errno == EINTR) {
			return ;
		};

		if (res == -1 || (pfd.revents & (POLLERR|POLLHUP|POLLNVAL))) {
			err (1, "poll error");
		};

		if (res != 0) {
			memset(&addr_from, 0, sizeof(addr_from));
			len_from = sizeof(addr_from);

			recv_size = recvfrom(sock, (void *)&msg, sizeof(msg), 0,
			    (struct sockaddr *)&addr_from, &len_from);

			if (recv_size == -1) {
				err(1, "recvfrom");
			}

			if (!suppress ||
			    !is_valid_ip((const struct sockaddr *)&addr_from)) {
				print_reply_info(&addr_from, msg, recv_size,
				    verbose);
			}
		}
	} while (res != 0);
}

/*
 * Wrapper on top of getaddrinfo call. hostname is IP address or host name to
 * find and port is port number.
 * Returns addrinfo list with address(es) of hostname
 */
static struct addrinfo *
return_addr_info(const char *hostname, const char *port)
{
	struct addrinfo ai_hints, *res;
	int error;

	memset(&ai_hints, 0, sizeof(ai_hints));
	ai_hints.ai_family = AF_INET;
	ai_hints.ai_socktype = SOCK_DGRAM;
	ai_hints.ai_protocol = IPPROTO_UDP;
	ai_hints.ai_flags = AI_NUMERICSERV;

	error = getaddrinfo(hostname, port, &ai_hints, &res);
	if (error != 0) {
		errx(1, "%s", gai_strerror(error));
	}

	return (res);
}

/*
 * Send DHCP request message on socket sock. lif is local interface structure
 * pointer filled by informations about local interface.
 */
static void
send_request(int sock, const struct local_if *lif)
{
	char msg[MSG_LEN];
	struct iovec iov;
	struct msghdr msgh;
	struct sockaddr_in br_addr;
	size_t msg_len;
#ifdef IP_PKTINFO
	char cmsg_buf[CMSG_SPACE(sizeof(struct in_pktinfo))];
	struct cmsghdr *cmsg;
	struct in_pktinfo pkt_info;
#endif

	memset(&br_addr, 0, sizeof(br_addr));
	br_addr.sin_family = AF_INET;
	br_addr.sin_port = htons(BOOTPS_PORT);

#ifdef IP_ONESBCAST
	br_addr.sin_addr = lif->broad_addr.sin_addr;
#else
	br_addr.sin_addr.s_addr = htonl(INADDR_BROADCAST);
#endif

	msg_len = dhcpmsg_fill(msg, sizeof(msg));

	memset(&iov, 0, sizeof(iov));
	iov.iov_base = msg;
	iov.iov_len = sizeof(msg);

	memset(&msgh, 0, sizeof(msgh));

	msgh.msg_name = &br_addr;
	msgh.msg_namelen = sizeof(br_addr);
	msgh.msg_iov = &iov;
	msgh.msg_iovlen = 1;

#ifdef IP_PKTINFO
	memset(cmsg_buf, 0, sizeof(cmsg_buf));
	msgh.msg_control = cmsg_buf;
	msgh.msg_controllen = CMSG_LEN(sizeof(pkt_info));
	cmsg = CMSG_FIRSTHDR(&msgh);
	cmsg->cmsg_len = CMSG_LEN(sizeof(pkt_info));
	cmsg->cmsg_level = IPPROTO_IP;
	cmsg->cmsg_type = IP_PKTINFO;

	memset(&pkt_info, 0, sizeof(pkt_info));
	pkt_info.ipi_ifindex = lif->if_index;

#ifndef __CYGWIN__
	memcpy(&pkt_info.ipi_spec_dst, &lif->local_addr.sin_addr,
	    sizeof(lif->local_addr.sin_addr));
#endif

	memcpy(CMSG_DATA(cmsg), &pkt_info, sizeof(pkt_info));
#endif
	if (sendmsg(sock, &msgh, 0) == -1) {
		err(1, "Can't send packet");
	}
}

/*
 * Display version of ODHCPLoc together with compiled features. Function calls
 * exit, so it never returns.
 */
static void
show_version(void)
{

	printf("ODHCPLoc version %s compiled with", ODHCPLOC_VER);
#ifdef SO_BROADCAST
	printf(" SO_BROADCAST");
#endif
#ifdef SO_BINDTODEVICE
	printf(" SO_BINDTODEVICE");
#endif
#ifdef IP_PKTINFO
	printf(" IP_PKTINFO");
#endif
#ifdef IP_ONESBCAST
	printf(" IP_ONESBCAST");
#endif
#ifdef __CYGWIN__
	printf(" CYGWIN");
#endif
	printf("\n");

	exit(0);
}

/*
 * Handler for INT signal. Frees all allocated memory and exits with error code
 * 0.
 */
static void
sigint_handler(int sig)
{
	free_validip_list();
	free(cli_params.lif.device_name);

	exit(0);
}

/*
 * Display usage of ODHCPLoc. Function never returns and exit with error code 1.
 */
static void
usage(void)
{
	extern char *__progname;

	fprintf(stderr, "usage: %s [-pVv] [-c count] [-i interval] my_ip "
	    "[valid_dhcp...]\n", __progname);

	exit(1);
}
