/*
  Copyright (C) 2004-2005 Tommi Tervonen, Petteri Klemola, Pasi Orovuo

  This file is part of Kajaani Kombat.

  Kajaani Kombat is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  Kajaani Kombat is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with Kajaani Kombat; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef __PARTICLE_EFECTS_H
#define __PARTICLE_EFECTS_H

#include "msg.h"
#include "game_obj.h"
#include "coord.h"
#include <cstdlib>
#include "SDL.h"
#include <stdio.h>

#define PI 3.141592654

Uint32 multiplyRGB (Uint32 col, double am);

/**
 * A class for single particle to be used in the efects.
 *
 * \author Tommi Tervonen <tommi@medusa.tutka.net>
 */
class particle
{
 public:
  /**
   * Creates a particle in position pos with random direction.
   *
   * \param pos Pixel-coordinate position of the particle.
   */
  particle(const coord & pos);

  particle (const coord & pos, double dirx, double diry);

  particle (const coord & pos, double dirx, double diry, Uint32 col);

  /**
   * Default constructor.
   */
  particle();

  /**
   * Moves the particle for given time-units. Updates color etc at the same time.
   */
  virtual void move_decel(Uint32 time, double rate = 5.0);
  /**
   * Moves the particle for given time-units. Decelerates.
   */
  virtual void move_accel(Uint32 time, const coord &ssize);

  virtual void move_const(Uint32 time, const coord &ssize);

  void move(Uint32 time);
  void decelerate(Uint32 time);
  void fade(Uint32 time);

  /**
   * Checks whether the particle is dead or not.
   *
   * \return True if is dead, false otherwise.
   */
  bool is_dead();
  /**
   * Gives the color of the particle.
   *
   * \return Color of the particle.
   */
  inline Uint32 get_color() const
  { return color; }
  /**
   * Sets the color of the particle.
   *
   * \param col of the particle.
   */
  inline void set_color(Uint32 col)
    { color = col; }
    
  /**
   * Gives the position of the particle.
   * 
   * \return Position in pixel-coordinates.
   */
  coord get_position() const;

  /**
   * Sets the position.
   *
   * \param pos new position.
   */
  void set_position (const coord & pos);
  /**
   * Sets the direction.
   *
   * \param new direction
   */
  void set_direction (double dirx, double diry);
 protected:
  double posx;
  double posy;
  double dirx;
  double diry;

  bool dead;
  Uint32 color;
};

/**
 * Abstract base-class for basic particle-efect object.
 *
 * \author Tommi Tervonen <tommi@medusa.tutka.net>
 */
class particle_efect : public game_obj
{
 public:
  /**
   * Creates a new particle efect.
   *
   * \param center Pixel-coordinate center of the efect.
   * \param num_particles Number of particles in the efect.
   */
  particle_efect (const coord &center, int num_particles = 200);
  /**
   * Destructor.
   */
  virtual ~particle_efect();

  /**
   * Draws the particle efect on specifies surface which has same coordinate-space as the efect.
   *
   * \param surf Surface to be drawn into.
   */
  virtual void draw (SDL_Surface *surf);

  virtual void move(Uint32 time) = 0;
   
 protected:
  particle *ptc;
  int num_particles;
};

/**
 * Class for particle-explosion efect.
 *
 * \author Tommi Tervonen <tommi@medusa.tutka.net>
 */
class explosion_efect : public particle_efect
{
 public:
  explosion_efect (const coord &center, SDL_Surface *surf);
  virtual void move(Uint32 time);
  virtual bool is_finished() const;
 protected:
  bool done;
};

/**
 * Class for starfield efect.
 *
 * \author Tommi Tervonen <tommi@medusa.tutka.net>
 */
class starfield_efect : public particle_efect
{
 public:
  starfield_efect(const coord &center, const coord &scrsize, int no_stars = 50);
  virtual void move (Uint32 time);
  virtual bool is_finished() const;
 protected:

  particle spawn_star (const coord & pos);
  particle spawn_center_star ();
  coord scrsize;
  int speed;
  int no_stars;
};

class real_explosion_efect : public particle_efect
{
 public:
  real_explosion_efect (const coord &center, int num_part = 50, double speed = 5.0);
  virtual void move(Uint32 time);
  virtual bool is_finished() const;
  void set_colormap_yellowred();
  void set_colormap_yellowredgray();
  virtual void draw (SDL_Surface *surf);

  static void create_colormaps();

 protected:
  bool done;
  Uint32 *cmap;
  double speed;

  static Uint32 colormap_yr[256];
  static Uint32 colormap_yrg[256];
  static bool cmap_inited;
};

#endif
