/*
  Copyright (C) 2004-2005 Tommi Tervonen, Petteri Klemola, Pasi Orovuo

  This file is part of Kajaani Kombat.

  Kajaani Kombat is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  Kajaani Kombat is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with Kajaani Kombat; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef __GUI_H
#define __GUI_H

#include "SDL.h"
#include "coord.h"
#include "globals.h"
#include "SDL_ttf.h"
#include <vector>
#include <typeinfo>
#include <string>
#include "gfx_functions.h"
#include <cctype>
#include "game_obj.h"
#include "gfx.h"
#include "particle_efects.h"
#include "audio.h"
#include "rotozoom.h"
#include "lookup.h"
#include "mosaic.h"
#include "config.h"

#define GUI_BUTTON_DOWNTIME 150
#define SCREEN_ZOOMTIME 1000

class gui_screen;

class gui_component
{
 public:
  gui_component(const coord &_pos)
    { pos = _pos; hidden = false;}

  virtual ~gui_component()
    { }

  void set_position(const coord &_pos) { pos = _pos; }
  const coord & get_position() const { return pos; }
  virtual void draw (SDL_Surface *screen) = 0;
  virtual coord get_size() = 0;
  virtual bool get_hidden() const { return hidden; }
  virtual void hide() { hidden = true;}
  virtual void unhide() { hidden = false;}

 protected:
  coord pos;
  bool hidden;
};

class gui_focus_component : public gui_component
{
 public:
  gui_focus_component (const coord &pos)
    : gui_component(pos)
    {
      act = false;
      focus = false;
    }

  virtual ~gui_focus_component()
    { }

  virtual void set_focus (bool foc)
    { focus = foc; }

  virtual void set_act(bool _act)
    { act = _act; }

  virtual void action(SDLKey sym, SDLMod mod) = 0;
  virtual coord get_size() = 0;

 protected:
  bool focus;
  bool act;
  gui_screen *transfer_screen;
};

class gui_button : public gui_focus_component
{
 public:
  // NOTE! all 3 images must be of same size!
  gui_button (const coord &pos, SDL_Surface *_nofoc_surf, SDL_Surface *_focus_surf, SDL_Surface *_act_surf);

  virtual void draw (SDL_Surface *screen);
  virtual void action(SDLKey sym, SDLMod mod) throw (int);

  virtual coord get_size();

  virtual void set_act_screen(gui_screen *s, int sign = 0)
    { transfer_screen = s; sig = sign; }

  // precond: set_act_screen() set ..
  virtual void set_act_callback( void (*func) (gui_screen *s))
    { callb = func; }

  virtual ~gui_button()
    { }

 protected:
  int sig;
  SDL_Surface *nofoc_surf;
  SDL_Surface *focus_surf;
  SDL_Surface *act_surf;
  gui_screen *transfer_screen;
  void (*callb) (gui_screen *s);
};

class gui_editbox : public gui_focus_component
{
 public:
  gui_editbox(const coord &pos, TTF_Font *font, Uint32 blink_time = 100);

  virtual void draw(SDL_Surface *scr);
  virtual void action (SDLKey sym, SDLMod mod);
  virtual coord get_size();
  virtual void set_focus(bool foc);

  virtual void set_text (string &text);
  virtual const string & get_text();

 protected:
  unsigned int curs_pos;
  TTF_Font *font;
  string txt;
  SDL_Color col;
  bool curs_on;
  Uint32 blink_time;
  Uint32 last_blinked;
};

class gui_textbox : public gui_component
{
  public:
    gui_textbox( const char *text, const coord& pos, TTF_Font *font );

    virtual void draw( SDL_Surface *screen );
    virtual void set_text( string& text );
    virtual const string& get_text();
    virtual void set_font( TTF_Font *font )
    { string s = get_text(); this->font = font; set_text( s ); }
    virtual const TTF_Font *get_font()
    { return ( font ); }
    virtual void set_color( SDL_Color color )
    { this->color.r = color.r; this->color.g = color.g; this->color.b = color.b; }
    virtual SDL_Color get_color()
    { return ( color ); }
    virtual coord get_size()
    { return ( coord::pc( surf->w, surf->h ) ); }

  private:
    SDL_Surface   *surf;
    string        text;
    SDL_Color     color;
    TTF_Font      *font;
};

class gui_textbutton : public gui_button
{
  public:
    gui_textbutton( const char *text, const coord& pos, TTF_Font *font );

    virtual void draw( SDL_Surface *screen );
    virtual void set_text( string& text );
    virtual const string& get_text()
    { return ( text ); }
    virtual coord get_size()
    { return ( ( nofoc_surf != 0 ) ? gui_button::get_size() : coord::pc( 0, 0 ) ); }

  private:
    string      text;
    TTF_Font    *font;
    SDL_Color   color;
};

class gui_numchooser : public gui_focus_component
{
 public:
  gui_numchooser(const coord &pos, TTF_Font *font, int num_alts, int min_num, Uint32 blink_time = 100);
  virtual ~gui_numchooser();

  virtual void set_focus(bool foc);
  virtual void draw(SDL_Surface *scr);
  virtual void action (SDLKey sym, SDLMod mod);
  virtual coord get_size();

  virtual int get_choice() const;
  virtual void set_choice(int number);

  virtual void set_numbers(int minnum, int numalts);
 protected:
  SDL_Surface *nsurf;
  int num_alts;
  int min_num;
  int chooser_pos;
  TTF_Font *font;
  SDL_Color col;
  bool curs_on;
  Uint32 blink_time;
  Uint32 last_blinked;
  int nwid;

  virtual void create_surface();
};

class gui_quit_button : public gui_button
{
 public:
  gui_quit_button (const coord &pos, SDL_Surface *_nofoc_surf, SDL_Surface *_focus_surf, SDL_Surface *_act_surf)
    : gui_button(pos, _nofoc_surf, _focus_surf, _act_surf)
    { }

  virtual void action(SDLKey sym, SDLMod mod) throw(int)
    { if (sym == SDLK_RETURN) throw 1; }

};

class gui_image : public gui_component
{
 public:
  gui_image (const coord &pos, SDL_Surface *img);

  virtual void draw (SDL_Surface *screen);
  virtual coord get_size();

 protected:
  SDL_Surface *img;
};

class gui_screen
{
 public:
  gui_screen(SDL_Surface *screen, TTF_Font *font, gfx *graphix, starfield_efect *starf, audio *aud, config *cfg);
  
  void add_component(gui_component *g);
  virtual void re_init(bool reverse);
  virtual void focus_down();
  virtual void focus_up();
  virtual void activate(int val) = 0;
  virtual void draw ();
  virtual void click(gui_focus_component *comp);
  virtual ~gui_screen();
  virtual void focus_first_component();
  virtual void clear_focus();

  void back_one_screen();

  void remove_text_objects();
  void reset_last_drawn();
  void update_config( config *newcfg )
  { *conf = *newcfg; }

 protected:
  SDL_Surface *screen;
  vector<gui_component *> comps;
  int cur_focus;
  virtual void set_focus (int focus_index);
  TTF_Font *font;
  vector<game_obj *> objs;
  gfx *graphix;
  Uint32 last_drawn;
  starfield_efect *starfield;
  audio *aud;
  config  *conf;

  virtual void process_objects();
  virtual void move_objects(Uint32 time);

  // extremely slow and not tested - do not use
  void rotozoom_screen_in();

};

#endif
