/*
  Copyright (C) 2004-2005 Tommi Tervonen, Petteri Klemola, Pasi Orovuo

  This file is part of Kajaani Kombat.

  Kajaani Kombat is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  Kajaani Kombat is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with Kajaani Kombat; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "gui.h"

gui_image::gui_image(const coord &pos, SDL_Surface *_img)
  : gui_component (pos)
{
  img = _img;
}

void gui_image::draw (SDL_Surface *screen)
{
  SDL_Rect dest;
  dest.x = pos.getPX();
  dest.y = pos.getPY();

  SDL_BlitSurface (img, NULL, screen, &dest);
}

gui_button::gui_button (const coord &pos, SDL_Surface *_nofoc_surf, SDL_Surface *_focus_surf, SDL_Surface *_act_surf)
  : gui_focus_component(pos)
{
  nofoc_surf = _nofoc_surf;
  focus_surf = _focus_surf;
  act_surf = _act_surf;
  transfer_screen = NULL;
  callb = NULL;
}

void gui_button::draw (SDL_Surface *screen)
{
  SDL_Surface *ds = 0;
  
  if (act)
    ds = act_surf;
  else if (focus)
    ds = focus_surf;
  else ds = nofoc_surf;

  SDL_Rect dest;
  dest.x = pos.getPX();
  dest.y = pos.getPY();
    
  //  printf ("Blitting to %d %d\n", dest.x, dest.y);
  SDL_BlitSurface (ds, NULL, screen, &dest);
}

void gui_screen::focus_up()
{

  aud->play_gui_navigation_sound();

  int curind = cur_focus;
  for (unsigned int i=0;i<comps.size();i++)
    {
      curind--;
      if (curind <0) curind = comps.size()-1;
      if (dynamic_cast<gui_focus_component *>(comps[curind]) != 0 && (!comps[curind]->get_hidden()))
	{
	  set_focus(curind);
	  return;
	}
    }
}

void gui_screen::focus_down()
{

  aud->play_gui_navigation_sound();

  int curind = cur_focus;
  for (unsigned int i=0;i<comps.size();i++)
    {
      curind++;
      if (curind >= (int) comps.size()) curind = 0;
      if (dynamic_cast<gui_focus_component *>(comps[curind]) != 0 && (!comps[curind]->get_hidden()))
	{
	  set_focus(curind);
	  return;
	}
    }
}

void gui_screen::reset_last_drawn()
{
  last_drawn = 0;
}

void gui_screen::draw()
{
  SDL_FillRect (screen, NULL, 0);

  Uint32 tnow = SDL_GetTicks();
  if (last_drawn != 0)
    move_objects (tnow - last_drawn);
  process_objects();
  last_drawn = tnow;
    
  graphix->draw_objects(objs);

  for (unsigned int i=0;i<comps.size();i++)
    if (!comps[i]->get_hidden())
      comps[i]->draw(screen);



  SDL_Flip(screen);
}

gui_screen::gui_screen(SDL_Surface *scr, TTF_Font *_font, gfx *_graphix, starfield_efect *se, audio *_aud, config *_cfg)
{ 
  aud = _aud;
  last_drawn = 0;
  cur_focus = -1; 
  screen = scr;
  font = _font;
  graphix = _graphix;
  starfield = se;
  conf = _cfg;
  objs.push_back(se);

}

void gui_screen::set_focus(int index)
{
  if (cur_focus != -1)
    {
      gui_focus_component *oldf = dynamic_cast<gui_focus_component *>(comps[cur_focus]);
      oldf->set_focus(false);
    }

  if (index != -1)
    {
      gui_focus_component *newf = dynamic_cast<gui_focus_component *>(comps[index]);
      newf->set_focus(true);
    }

  cur_focus = index;
}

void gui_screen::add_component(gui_component *g)
{       
  comps.push_back(g);   
}

void gui_screen::focus_first_component()
{
  for (unsigned int i=0;i<comps.size();i++)
    {
      if (dynamic_cast<gui_focus_component *>(comps[i]) != 0)
	{
	  set_focus (i);
	  return;
	}
    }
  set_focus(-1);


}

void gui_screen::re_init(bool reverse)
{

  /* we are using only the reverse mosaic at the moment
  if(!reverse)
    {
      mosaic *moz1 = new mosaic(screen, screen,SDL_GetTicks(),false);
      while(!moz1->is_finished())
	{
	  moz1->update(SDL_GetTicks());
	  graphix->update();
	}
      delete moz1;
    }
  */

 SDL_Surface *temp_scr= SDL_CreateRGBSurface (SDL_SWSURFACE, screen->w, screen->h, screen->format->BitsPerPixel,
				screen->format->Rmask, screen->format->Gmask,
				screen->format->Bmask, screen->format->Amask);
 SDL_BlitSurface (screen, NULL, temp_scr, NULL);
  

  SDL_FillRect (temp_scr, NULL, 0);

  Uint32 tnow = SDL_GetTicks();
  if (last_drawn != 0)
    move_objects (tnow - last_drawn);
  process_objects();
  last_drawn = tnow;

  //graphix->draw_objects(objs);

  for (unsigned int i=0;i<comps.size();i++)
    if (!comps[i]->get_hidden())
      comps[i]->draw(temp_scr);

  //SDL_BlitSurface (temp_scr, NULL, screen, NULL);



  mosaic *moz2 = new mosaic(temp_scr, screen,SDL_GetTicks(),true);
  while(!moz2->is_finished())
    {
      moz2->update(SDL_GetTicks());
      graphix->update();
    }
  delete moz2;
  SDL_FreeSurface(temp_scr);

  //clear event queue
  SDL_Event ev;
  while (SDL_PollEvent(&ev))
    { }

  
  //SDL_Flip(screen);







}

void gui_screen::click(gui_focus_component *button)
{

  aud->play_gui_select_sound();

  Uint32 start_t = SDL_GetTicks();
  
  gui_button *b = dynamic_cast<gui_button *>(button);
  if (b == NULL)
    return;
  button->set_act(true);
  draw();
  while (SDL_GetTicks() - start_t < GUI_BUTTON_DOWNTIME)
    draw();
  button->set_act(false);
  draw();
}


void gui_screen::back_one_screen()
{
  /* we are using only the reverse mosaic at the moment
  mosaic *moz2 = new mosaic(screen, screen,SDL_GetTicks(),false);
  while(!moz2->is_finished())
    {
      moz2->update(SDL_GetTicks());
      graphix->update();
    }
  delete moz2;
  */

  aud->play_gui_select_sound();

  throw 1;//pois
}

gui_screen::~gui_screen()
{
  for (unsigned int i=0;i<objs.size();i++)
    if (typeid (*objs[i]) != typeid(starfield_efect))
      delete objs[i];
  objs.clear();
  for (unsigned int i=0;i<comps.size();i++)
    delete comps[i];
  comps.clear();
}

void gui_button::action(SDLKey sym, SDLMod mod) throw (int)
{ 
  if (sym == SDLK_RETURN)
    {
      if (callb) callb(transfer_screen);
      else
	transfer_screen->activate(sig); 
    }
}

gui_editbox::gui_editbox(const coord &pos, TTF_Font *fon, Uint32 _btime)
  : gui_focus_component(pos)
{
  curs_pos = 0;
  font = fon;
  col.r = 0xFF;
  col.g = 0xFF;
  col.b = 0xFF;
  curs_on = true;
  blink_time = _btime;
  last_blinked = 0;
}

void gui_editbox::set_focus(bool foc)
{
  gui_focus_component::set_focus(foc);
  if (foc) 
    {
      last_blinked = SDL_GetTicks();
      curs_on = true;
    }
}

void gui_editbox::draw(SDL_Surface *scr)
{




  string txt2;
  Uint32 tim = SDL_GetTicks();
  if (focus && tim-last_blinked >= blink_time)
    {
      last_blinked = tim;
      curs_on = !curs_on;
    }
  if (curs_on && focus)
    txt2 = string(txt + "|");
  else txt2 = txt;

  if (txt2.length() > 0)
    draw_text_baseline (scr, font, txt2.c_str(), pos, col);
}

void gui_editbox::action (SDLKey sym, SDLMod mod)
{
  switch (sym)
    {
    case SDLK_RETURN:
      break;
    case SDLK_BACKSPACE:
      if (curs_pos)
	txt.erase(--curs_pos, 1);
      break;
    case SDLK_RSHIFT:
    case SDLK_LSHIFT:
      break;
    default:
      if ( isprint( sym ) )
	txt.insert( curs_pos++, 1, ( mod & ( KMOD_LSHIFT | KMOD_RSHIFT ) ) ? toupper( sym ) : sym );
    }
  
}

gui_textbox::gui_textbox( const char *text, const coord& pos, TTF_Font *font )
: gui_component( pos )
{
  surf = 0;
  color.r = 0xFF;
  color.g = 0x00;
  color.b = 0xFF;

  this->font = font;

  string    s = text;
  set_text( s );
}

void gui_textbox::draw( SDL_Surface *screen ) {
  if ( !surf )
    return;

  SDL_Rect dest;
  dest.x = pos.getPX();
  dest.y = pos.getPY();

  SDL_BlitSurface( surf, NULL, screen, &dest );
}

void gui_textbox::set_text( string& newtext ) {
  if ( newtext == text ) return;
  
  if ( surf ) {
    SDL_FreeSurface( surf );
    surf = 0;
  }
  text = "";

  if ( newtext.length() && font ) {
    surf = TTF_RenderText_Blended( font, newtext.c_str(), color );
    if ( !surf )
      fprintf( stderr, "gui_textbox::set_text(): Error rendering text\n" );

    text = newtext;
  }
}

const string& gui_textbox::get_text() {
  return ( text );
}

gui_textbutton::gui_textbutton( const char *text, const coord& pos, TTF_Font *font )
: gui_button( pos, NULL, NULL, NULL )
{
  color.r = 0xFF;
  color.g = 0x00;
  color.b = 0xFF;

  this->font = font;
  this->text = "";

  if ( text ) {
    string  s = text;
    set_text( s );
  }
}

void gui_textbutton::set_text( string& text ) {
  if ( text == this->text ) return;

  if ( nofoc_surf ) {
    SDL_FreeSurface( nofoc_surf );
    nofoc_surf = 0;
  }
  if ( focus_surf ) {
    SDL_FreeSurface( focus_surf );
    focus_surf = 0;
  }
  if ( act_surf ) {
    SDL_FreeSurface( act_surf );
    act_surf = 0;
  }
  this->text = "";

  if ( text.length() && font != NULL ) {
    SDL_Color   cfoc = color;
    SDL_Color   cact = color;

    cact.g = cfoc.g = 0xFF;

    nofoc_surf = TTF_RenderText_Blended( font, text.c_str(), color );
    focus_surf = TTF_RenderText_Blended( font, text.c_str(), cfoc );
    act_surf = TTF_RenderText_Blended( font, text.c_str(), cact );

    if ( !nofoc_surf || !focus_surf || !act_surf ) {
      fprintf( stderr, "gui_textbutton::set_text(): Text rendering failed for either one or more of nofoc_surf, "
          "foc_surf or/and act_surf\nWe'll probably come crashing down any time soon...\n" );
    }

    this->text = text;
  }
}

void gui_textbutton::draw( SDL_Surface *screen ) {
  if ( nofoc_surf && focus_surf && act_surf )
    gui_button::draw( screen );
}

gui_numchooser::gui_numchooser(const coord &pos, TTF_Font *fon, int _num_alts, int _min_num, Uint32 _btime)
  : gui_focus_component(pos)
{
  nsurf = 0;
  curs_on = true;
  blink_time = _btime;
  font = fon;
  col.r = 0xFF;
  col.g = 0xFF;
  col.b = 0xFF;
  num_alts = _num_alts;
  min_num = _min_num;
  chooser_pos = 0; // chooser position
  create_surface();
}

void gui_numchooser::set_numbers(int minnum, int numalts)
{
  min_num = minnum;
  num_alts = numalts;
  create_surface();
}

void gui_numchooser::create_surface()
{
  char buf[10];
  if (nsurf != NULL)
    SDL_FreeSurface(nsurf);
  sprintf (buf, "%d", min_num+num_alts);
  SDL_Surface *tmp = TTF_RenderText_Blended (font, buf, col);
  int h = tmp->h + 10;
  int w = h;
  nwid = w;
  SDL_FreeSurface (tmp);
  SDL_Surface *tsurf = SDL_CreateRGBSurface (SDL_HWSURFACE|SDL_SRCALPHA, w * num_alts, h, 32, 0,0,0,0);
  nsurf = SDL_DisplayFormat(tsurf);
  SDL_FreeSurface(tsurf);
  // now blit all the numbers
  for (int i=0;i<num_alts;i++)
    {
      sprintf (buf, "%d", min_num+i);
      SDL_Surface *fsurf = TTF_RenderText_Blended(font, buf, col);
      SDL_Rect drect;
      drect.x = w * i + (int) ((w - fsurf->w) * 0.5);
      drect.y = (int) ((h - fsurf->h));
      SDL_BlitSurface (fsurf, NULL, nsurf, &drect);
      SDL_FreeSurface (fsurf);
    }
}

gui_numchooser::~gui_numchooser()
{
  SDL_FreeSurface(nsurf);
}

int gui_numchooser::get_choice() const
{
  return min_num + chooser_pos;
}

void gui_numchooser::set_choice(int num)
{
  assert (num >= min_num && num <= min_num + num_alts);
  chooser_pos = num - min_num;
}

void gui_numchooser::set_focus(bool foc)
{
  gui_focus_component::set_focus(foc);
  if (foc) 
    {
      last_blinked = SDL_GetTicks();
      curs_on = true;
    }
  else
    curs_on = true;
}

void gui_numchooser::draw(SDL_Surface *scr)
{
  string txt2;
  Uint32 tim = SDL_GetTicks();
  if (focus && tim-last_blinked >= blink_time)
    {
      last_blinked = tim;
      curs_on = !curs_on;
    }
  SDL_Rect drect;
  drect.x = pos.getPX();
  drect.y = pos.getPY() - nsurf->h;
  SDL_BlitSurface (nsurf, NULL, scr, &drect);
  if (curs_on)
    {
      coord spos = pos + coord::pc (chooser_pos * nwid, 5);
      coord dpos = pos + coord::pc (((chooser_pos+1) * nwid) - 1, nsurf->h-1 + 5);
      spos.subPY(nsurf->h);
      dpos.subPY(nsurf->h);
      Uint32 c = SDL_MapRGBA (nsurf->format, col.r, col.g, col.b, 0xFF);
      draw_rectangle (scr, spos, dpos, c);
    }
}

void gui_numchooser::action (SDLKey sym, SDLMod mod)
{
  switch (sym)
    {
    case SDLK_LEFT:
      chooser_pos--;
      if (chooser_pos < 0) chooser_pos = num_alts-1;
      break;
    case SDLK_RIGHT:
      chooser_pos++;
      chooser_pos %= num_alts; 
      break;
    default:
      break;
    }
  
}

coord gui_image::get_size()
{
  return coord::pc(img->w, img->h);
}

coord gui_button::get_size()
{
  return coord::pc(nofoc_surf->w, nofoc_surf->h);
}

coord gui_editbox::get_size()
{
  string txt2 = txt + "|";
  SDL_Surface *surf = TTF_RenderText_Blended (font, txt2.c_str(), col);
  coord r = coord::pc (surf->w, surf->h);
  SDL_FreeSurface (surf);
  return r;
}

coord gui_numchooser::get_size()
{
  coord r = coord::pc (nsurf->w, nsurf->h);
  r.incPX();
  r.incPY();
  return r;
}

void gui_screen::clear_focus()
{
  if (cur_focus != -1)
    {
      gui_focus_component *f = dynamic_cast<gui_focus_component *>(comps[cur_focus]);
      f->set_focus(false);
      cur_focus = -1;
    }
}

void gui_editbox::set_text (string &_text)
{
  txt = _text;
  curs_pos = txt.length();
}

const string & gui_editbox::get_text()
{
  return txt;
}

void gui_screen::move_objects(Uint32 time)
{
  vector<game_obj *>::iterator it = objs.begin();
  
  while (it != objs.end())
    {
      game_obj *o = *it++;
      o->move(time);
    }
}

void gui_screen::process_objects()
{
  vector<game_obj*>::iterator it = objs.begin();
  while (it != objs.end())
    {
      game_obj *o = *it++;
      assert(o != NULL); // musn't be null
      
      if (!o->is_finished())
        continue;

      // delete object
      it = objs.erase(it - 1);


      if (typeid (*o) == typeid(text_object)) // is of type text_object
	{
	}
      // delete the actual object.
      delete o;
    }  
}

void gui_screen::rotozoom_screen_in()
{
  lookup *lut = new lookup();
  rotozoom *rz = new rotozoom(lut);

  Uint32 start = SDL_GetTicks();

  SDL_Surface *buf = deep_copy_surface(screen);

  coord cent = coord::pc(SCR_WIDTH / 2, SCR_HEIGHT / 2);

  while (SDL_GetTicks() - start < SCREEN_ZOOMTIME)
    {
      float scale = 1.0f + ((float) (SDL_GetTicks() - start)) / (float) SCREEN_ZOOMTIME;
      int angle = SDL_GetTicks() - start;
      rz->rotatescale_blit(scale, angle, screen, buf, cent);
      SDL_Flip(screen);
    }
  delete rz;
  delete lut;
}

void gui_screen::remove_text_objects()
{
  vector<game_obj *>::iterator it = objs.begin();
  while (it != objs.end())
    {
      game_obj *o = *it++;
      if (typeid (*o) == typeid(text_object))
	{
	  it = objs.erase(it -1);
	  delete o;
	}
    }
  
}
