/*
  Copyright (C) 2004-2005 Tommi Tervonen, Petteri Klemola, Pasi Orovuo

  This file is part of Kajaani Kombat.

  Kajaani Kombat is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  Kajaani Kombat is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with Kajaani Kombat; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef __BLOCK_H_
#define __BLOCK_H_

#include <vector>
#include "coord.h"

#define NUM_BLOCK_TYPES 7

using namespace std;

/**
 * A class for block-objects.
 *
 * \author Tommi Tervonen <tommi@medusa.tutka.net>
 */
class block
{
 public:
  typedef enum block_type {
	TYPE_SINGLE = 0,
	TYPE_L,
	TYPE_U,
	TYPE_CORNER,
	TYPE_CROSS,
	TYPE_I,
	TYPE_E
  };

 public:
  /**
   * Default constructor. Creates a random block with centers coordinates (-1,-1) and random orientation.
   */
  block();
  /**
   * Constructor. Creates a random block with random orientation.
   * \param center Coordinates of the center of the block.
   */
  block(const coord &center);
  /**
   * Constructor. Creates a block with random orientation.
   * \param center Coordinates of the center of the block.
   * \param type Type of the block.
   */
  block(const coord &center, block_type type);
  /**
   * Constructor. Creates a block with random orientation.
   * \param center Coordinates of the center of the block.
   * \param type Type of the block.
   */
  block(const coord &center, int type)
	{ block (center, (block_type) type); }
  /**
   * Constructor.
   * \param center Coordinates of the center of the block.
   * \param type Type of the block.
   * \param orientation Orientation of the block.
   */
  block(const coord &center, block_type type, int orientation);

  /**
   * Gives the type of the block.
   * \return Blocks type.
   */
  inline block_type get_type()
	{ return type; }
  /**
   * Gives the orientation of the block.
   * \return Blocks orientation.
   */
  inline int get_orientation()
	{ return orientation; }
  /**
   * Sets the center coordinates of the block.
   * \param cen New coordinates for the center.
   */
  inline void set_center (const coord &cen)
	{ center = cen; }
  /**
   * Changes the type of the block.
   * \param type New type of the block.
   */
  void set_type (int type);
  /**
   * Changes the orientation of the block.
   * Precond. 0 <= orientation < 4.
   * \param orientation New orientation.
   */
  void set_orientation (int orientation);
  /**
   * Rotates the block 90 degrees clockwise.
   */
  void rotate();
  /**
   * Gives the center of the block.
   * \return Coordinates of the center.
   */
  inline const coord& get_center() const
    { return center; }
  /**
   * Gives the vector of coordinates of the tiles of the block.
   * \return Vector of coordinates of the tiles of the block.
   */
  inline const vector<coord>& get_tiles() const
    { return tiles; }

  /**
   * Moves the block. Doesn't move it if it would move the block over the screen boundaries.
   *
   * \return true if the block was moved, false if not.
   */
  bool move(const coord &dir);


 protected:
  /**
   * Fills the blocks data structure with the tiles of its type.
   */
  void fill_block();

  /** Orientation of the block (0-3). */
  int orientation;
  /** Type of the block. */
  block_type type;
  /** Tiles of the block (positioned relative to the center). */
  vector<coord> tiles;
  /** Center coordinates of the block. */
  coord center;
};

#endif
