/*
 * GQradio
 * (C) 2004 John Ellis
 *
 * Author: John Ellis
 *
 * This software is released under the GNU General Public License (GNU GPL).
 * Please read the included file COPYING for more information.
 * This software comes with no warranty of any kind, use at your own risk!
 */

/*
 * Currently supported radio interfaces:
 * 
 * Linux    using video4linux
 * FreeBSD  using bktr
 *
 * note: If you implement support for one of the systems that is listed as removed in mixer.c,
 *       please follow the instructions at the top of that file to pull support for the mixer
 *       devices back into GQradio from GQmpeg.
 */
 
/*
 * bktr Interface: Serdar Ozler <sozler@sitebest.com>
 *
 * Special thanks to the developer of xmradio.
 * I looked at the code of xmradio to implement the bktr interface.
 */


#include "gqradio.h"
#include "io_radio.h"

#include "ui_fileops.h"


#if defined(linux) && defined(HAVE_VIDEO4LINUX)
  #include <fcntl.h>
  #include <sys/ioctl.h>
  #include <linux/videodev.h>
  #include <errno.h>
#elif defined(HAVE_IOCTL_BT848)
  #include <fcntl.h>
  #include <sys/ioctl.h>
  #include <machine/ioctl_bt848.h>
  #include <errno.h>
#endif


/* ----------------------------------------------------------
   input / output interface to a radio tuner

   playlist entry format:
   ======================

   radio stations are stored as a file pathname in the format:

     radio:XX.X:<TITLE>

   radio:   identifies this as a radio station entry
   XX.X     the frequency (in MHz, entry of AM KHz is not defined)
   <TITLE>  the text description (optional)

   for no title the second colon can be omitted: radio:XX.X

   Currently suported platforms:
   =============================
     Linux (using the video4linux /dev/radio interface)

   ----------------------------------------------------------*/

/* these are set here, but should be overwritten in radio_real_test() */
static guint32 radio_limit_lower = RADIO_FREQ_MIN;
static guint32 radio_limit_upper = RADIO_FREQ_MAX;

/* whether or not to use the min/max frequencies reported by the driver */
static gint radio_limit_use_card = FALSE;

/* current radio device */
static gchar *radio_device = NULL;

gint radio_volume_boost = FALSE;


/*
 *-----------------------------------------------------------------------------
 * radio controls (ioctl & system specific)
 *-----------------------------------------------------------------------------
 */

#if defined(linux) && defined(HAVE_VIDEO4LINUX)

/*
 * video4linux interface
 */

static gint radio_enabled = TRUE;

#ifndef RADIO_DEVICE
  #define RADIO_DEVICE "/dev/radio"
#endif

static gint v4l_enabled = FALSE;	/* found, ioctl seems to work */
static gint v4l_fd = -1;		/* /dev/radio fd */

static void radio_error_message(void)
{
	printf("Error talking (ioctl) to %s, %s\n", RADIO_DEVICE, strerror(errno));
}

static guint radio_calc_steps(struct video_tuner *t)
{
	if ( (t->flags & VIDEO_TUNER_LOW) )
		{
		return 16000;	/* field is KHz */
		}
	else
		{
		return 16;	/* field is MHz */
		}
}

static gint radio_control_on(void)
{
	if (v4l_fd != -1) return TRUE;

	if (!radio_device) radio_device = g_strdup(RADIO_DEVICE);

	if ((v4l_fd = open(radio_device, O_RDONLY)) == -1)
		{
		printf("failed to open radio device %s, %s\n", radio_device, strerror(errno));
		v4l_enabled = FALSE;
		v4l_fd = -1;
		return FALSE;
		}

	return TRUE;
}

static gint radio_control_off(void)
{
	if (v4l_fd == -1) return TRUE;

	close(v4l_fd);
	v4l_fd = -1;

	return TRUE;
}

static gint radio_control_set_mute(gint mute)
{
	struct video_audio av;

	if (v4l_fd == -1) return FALSE;

	if (ioctl(v4l_fd, VIDIOCGAUDIO, &av) != 0)
		{
		radio_error_message();
		return FALSE;
		}

	if (mute)
		{
		av.flags |= VIDEO_AUDIO_MUTE;
		}
	else
		{
		if (av.volume == 0 || radio_volume_boost) av.volume = 65535;
		av.flags &= ~VIDEO_AUDIO_MUTE;
		}

	if (ioctl(v4l_fd, VIDIOCSAUDIO, &av) != 0)
		{
		radio_error_message();
		return FALSE;
		}

	return TRUE;
}

static gint radio_control_set_freq(guint32 freq)
{
	struct video_tuner t;
	guint32 f;
	guint m;

	if (v4l_fd == -1) return FALSE;

	t.tuner = 0;

	if (ioctl(v4l_fd, VIDIOCGTUNER, &t) == 0)
		{
		m = radio_calc_steps(&t);
		}
	else
		{
		radio_error_message();
		m = 16;			/* dunno */
		}

	f = (guint32)((gfloat)freq / 1000000.0 * m);
	if (ioctl(v4l_fd, VIDIOCSFREQ, &f) != 0)
		{
		radio_error_message();
		return FALSE;
		}

	return TRUE;
}

static void radio_error(void)
{
	radio_control_set_mute(TRUE);
	radio_control_off();
}

static gint radio_real_startup(guint32 freq)
{
	if (!radio_control_on()) return FALSE;

	if (freq > 0)
		{
		if (!radio_control_set_freq(freq) ||
		    !radio_control_set_mute(FALSE))
			{
			radio_error();
			return FALSE;
			}
		}

	return TRUE;
}

static gint radio_real_shutdown(void)
{
	radio_control_set_mute(TRUE);
	return radio_control_off();
}

static gint radio_real_test(void)
{
	struct video_tuner t;

	if (!radio_device) radio_device = g_strdup(RADIO_DEVICE);

	if (!isname(radio_device))
		{
		printf(_("Failed to find %s\n"), radio_device);
		v4l_enabled = FALSE;
		return FALSE;
		}

	if (!radio_control_on())
		{
		v4l_enabled = FALSE;
		return FALSE;
		}

	t.tuner = 0;

	if (ioctl(v4l_fd, VIDIOCGTUNER, &t) == 0)
		{
		guint g;

		g = radio_calc_steps(&t);
		radio_limit_lower = t.rangelow / g * 1000000;
		radio_limit_upper = t.rangehigh / g * 1000000;
		v4l_enabled = TRUE;
		}
	else
		{
		radio_error_message();
		v4l_enabled = FALSE;
		}

	radio_control_off();

	return v4l_enabled;
}

/* stereo is 0 or 1, strength is 0 to 100 (%) */
static gint radio_real_status(gint *stereo, gint *strength)
{
	struct video_tuner t;
	struct video_audio av;

	if (v4l_fd == -1 || !stereo || !strength) return FALSE;

	/* stereo */

	if (ioctl(v4l_fd, VIDIOCGAUDIO, &av) != 0)
		{
		radio_error_message();
		return FALSE;
		}
	if ( (av.mode & VIDEO_SOUND_STEREO) )
		{
		*stereo = TRUE;
		}
	else
		{
		*stereo = FALSE;
		}

	/* strength */

	t.tuner = 0;
	if (ioctl(v4l_fd, VIDIOCGTUNER, &t) != 0)
		{
		radio_error_message();
		return FALSE;
		}
	*strength = (double)t.signal / 65535.0 * 100.0;	/* 16bit scale */

	return TRUE;
}

static gint radio_real_set_freq(guint32 freq)
{
	return radio_control_set_freq(freq);
}

static gint radio_real_set_mute(gint mute)
{
	return radio_control_set_mute(mute);
}


#elif defined(HAVE_IOCTL_BT848)

/*
 * ioctl_bt848 interface
 */

static gint radio_enabled = TRUE;

#ifndef RADIO_DEVICE
  #define RADIO_DEVICE "/dev/tuner"
#endif

static gint bt848_enabled = FALSE;	/* found, ioctl seems to work */
static gint bt848_fd = -1;		/* /dev/tuner fd */

static void radio_error_message(void)
{
	printf("Error talking (ioctl) to %s, %s\n", RADIO_DEVICE, strerror(errno));
}

static guint radio_calc_steps(int *t)
{
	return 100;	/* field is KHz */
}

static gint radio_control_on(void)
{
	int av;

	if (bt848_fd != -1) return TRUE;

	if (!radio_device) radio_device = g_strdup(RADIO_DEVICE);

	if ((bt848_fd = open(radio_device, O_RDONLY)) == -1)
		{
		printf(_("Failed to open radio device %s, %s\n"), radio_device, strerror(errno));
		bt848_enabled = FALSE;
		bt848_fd = -1;
		return FALSE;
		}

	av = 3;

	if (ioctl(bt848_fd, TVTUNER_SETCHNL, &av) != 0)
		{
		radio_error_message();
		return FALSE;
		}

	av = CHNLSET_WEUROPE;

	if (ioctl(bt848_fd, TVTUNER_SETTYPE, &av) != 0)
		{
		radio_error_message();
		return FALSE;
		}

	if (ioctl(bt848_fd, RADIO_GETMODE, &av) != 0)
		{
		radio_error_message();
		return FALSE;
		}

	av &= ~RADIO_AFC;

	if (ioctl(bt848_fd, RADIO_SETMODE, &av) != 0)
		{
		radio_error_message();
		return FALSE;
		}

	av = AUDIO_INTERN;
	
	if (ioctl(bt848_fd, BT848_SAUDIO, &av) != 0)
		{
		radio_error_message();
		return FALSE;
		}

	return TRUE;
}

static gint radio_control_off(void)
{
	if (bt848_fd == -1) return TRUE;

	close(bt848_fd);
	bt848_fd = -1;

	return TRUE;
}

static gint radio_control_set_mute(gint mute)
{
	int av;

	if (bt848_fd == -1) return FALSE;

	if (ioctl(bt848_fd, BT848_GAUDIO, &av) != 0)
		{
		radio_error_message();
		return FALSE;
		}

	if (mute)
		{
		av = AUDIO_MUTE;
		}
	else
		{
		av = AUDIO_UNMUTE;
		}

	if (ioctl(bt848_fd, BT848_SAUDIO, &av) != 0)
		{
		radio_error_message();
		return FALSE;
		}

	return TRUE;
}

static gint radio_control_set_freq(guint32 freq)
{
	int t;
	guint32 f;
	guint m;

	if (bt848_fd == -1) return FALSE;

	if (ioctl(bt848_fd, TVTUNER_GETSTATUS, &t) == 0)
		{
		m = radio_calc_steps(&t);
		}
	else
		{
		radio_error_message();
		m = 100;		/* dunno */
		}

	f = (guint32)((gfloat)freq / 1000000.0 * m);
	if (ioctl(bt848_fd, RADIO_SETFREQ, &f) != 0)
		{
		radio_error_message();
		return FALSE;
		}

	return TRUE;
}

static void radio_error(void)
{
	radio_control_set_mute(TRUE);
	radio_control_off();
}

static gint radio_real_startup(guint32 freq)
{
	if (!radio_control_on()) return FALSE;

	if (freq > 0)
		{
		if (!radio_control_set_freq(freq) ||
		    !radio_control_set_mute(FALSE))
			{
			radio_error();
			return FALSE;
			}
		}

	return TRUE;
}

static gint radio_real_shutdown(void)
{
	radio_control_set_mute(TRUE);
	return radio_control_off();
}

static gint radio_real_test(void)
{
	int t;

	if (!radio_device) radio_device = g_strdup(RADIO_DEVICE);

	if (!isname(radio_device))
		{
		printf(_("Failed to find %s\n"), radio_device);
		bt848_enabled = FALSE;
		return FALSE;
		}

	if (!radio_control_on())
		{
		bt848_enabled = FALSE;
		return FALSE;
		}

	if (ioctl(bt848_fd, TVTUNER_GETSTATUS, &t) == 0)
		{
		guint g;

		g = radio_calc_steps(&t);
		radio_limit_lower = 87.50 / g * 1000000;
		radio_limit_upper = 108.00 / g * 1000000;
		bt848_enabled = TRUE;
		}
	else
		{
		radio_error_message();
		bt848_enabled = FALSE;
		}

	radio_control_off();

	return bt848_enabled;
}

/* stereo is 0 or 1, strength is 0 to 100 (%) */
static gint radio_real_status(gint *stereo, gint *strength)
{
	int t;
	int av;

	if (bt848_fd == -1 || !stereo || !strength) return FALSE;

	/* stereo */

	if (ioctl(bt848_fd, RADIO_GETMODE, &av) != 0)
		{
		radio_error_message();
		return FALSE;
		}
	if ( !(av & RADIO_MONO) )
		{
		*stereo = TRUE;
		}
	else
		{
		*stereo = FALSE;
		}

	/* strength */

	if (ioctl(bt848_fd, TVTUNER_GETSTATUS, &t) != 0)
		{
		radio_error_message();
		return FALSE;
		}
	*strength = (double)(t & 0x07) / 7.0 * 100.0;	/* 16bit scale */

	return TRUE;
}

static gint radio_real_set_freq(guint32 freq)
{
	return radio_control_set_freq(freq);
}

static gint radio_real_set_mute(gint mute)
{
	return radio_control_set_mute(mute);
}


#else


/*
 * OS does not support radio, so give dummy funcs
 */

static gint radio_enabled = FALSE;


static gint radio_real_startup(guint32 freq)
{
	return TRUE;
}

static gint radio_real_shutdown(void)
{
	return TRUE;
}

static gint radio_real_is_active(void)
{
	return FALSE;
}

static gint radio_real_test(void)
{
	return FALSE;
}

static gint radio_real_status(gint *stereo, gint *strength)
{
	return FALSE;
}

static gint radio_real_set_freq(guint32 freq)
{
	return TRUE;
}

static gint radio_real_set_mute(gint mute)
{
	return TRUE;
}


#endif


/*
 *-----------------------------------------------------------------------------
 * wrappers to real stuff, plus update routines
 *-----------------------------------------------------------------------------
 */

/* radio test is only called when needed - and then only once,
 * in other words not at start-up so it does not touch the
 * device - which on linux, will load kernel modules on app start. (slows start)
 * added recheck to allow re-detection (only used from preferences)
 */
gint radio_test(gint recheck)
{
	static gint checked = FALSE;
	static gint found = FALSE;

	if (!radio_enabled) return FALSE;

	if (recheck)
		{
		checked = FALSE;
		found = FALSE;
		}

	if (checked) return found;
	checked = TRUE;

	if (debug_mode) printf("Checking for radio tuner...\n");

	found = radio_real_test();

	if (!found) printf(_("Radio device control test failed\n"));

	return found;
}

void radio_set_device(const gchar *device)
{
	g_free(radio_device);
	radio_device = g_strdup(device);
}

gint radio_init(void)
{
	return radio_test(FALSE);
}

gint radio_status(gint *stereo, gint *strength)
{
	return radio_real_status(stereo, strength);
}

gint radio_startup(guint32 freq)
{
	if (!radio_test(FALSE)) return FALSE;

	return radio_real_startup(freq);
}

gint radio_shutdown(void)
{
	return radio_real_shutdown();
}

gint radio_set_freq(guint32 freq)
{
	freq = CLAMP(freq, radio_limit_get_lower(), radio_limit_get_upper());

	return radio_real_set_freq(freq);
}

gint radio_set_mute(gint mute)
{
	return radio_real_set_mute(mute);
}

guint32 radio_limit_get_lower(void)
{
	radio_test(FALSE);
	return (radio_limit_use_card ? radio_limit_lower : RADIO_FREQ_MIN);
}

guint32 radio_limit_get_upper(void)
{
	radio_test(FALSE);
	return (radio_limit_use_card ? radio_limit_upper : RADIO_FREQ_MAX);
}

void radio_limit_by_card_set(gint use)
{
	radio_limit_use_card = use;
}

gint radio_limit_by_card_get(void)
{
	return radio_limit_use_card;
}

/*
 *-----------------------------------------------------------------------------
 * misc
 *-----------------------------------------------------------------------------
 */

gint radio_freq_get_major(guint32 freq)
{
	return (freq / 1000000);
}

gint radio_freq_get_minor(guint32 freq)
{
	return ((freq % 1000000) / 10000);
}

gchar *radio_freq_to_text(guint32 freq)
{
	gint maj, min;

	maj = radio_freq_get_major(freq);
	min = radio_freq_get_minor(freq);

	return g_strdup_printf("%d.%s%d", maj, (min < 10) ? "0" : "", min);
}

guint32 radio_freq_from_text(const gchar *text)
{
	const gchar *ptr;
	gchar *buf;
	gint maj, min;
	gint n;

	n = 0;
	ptr = text;
	while (*ptr != '.' && *ptr != '\0')
		{
		n++;
		ptr++;
		}
	if (n == 0) return 0;

	buf = g_strndup(text, n);
	maj = strtol(text, NULL, 10);
	g_free(buf);

	if (*ptr == '.')
		{
		gchar *end;

		ptr++;

		buf = g_strndup(ptr, 2);
		min = strtol(buf, &end, 10);
		ptr = buf;
		ptr++;
		if (ptr == end) min = min * 10;
		g_free(buf);
		}
	else
		{
		min = 0;
		}

	return (maj * 1000000) + (min * 10000);
}

guint32 radio_freq_clamp_to_increment(guint32 freq, guint32 step)
{
	guint32 low, high;
	guint32 scale;

	if (step == 0) return freq;

	if (step == 200000) 
		{
		/* special case (US freqs), convert to odd */
		scale = freq / step;

		if (freq % 200000 >= 100000) scale++;

		low = scale * step - 100000;
		high = scale * step + 100000;

		if (freq - low < high - freq) return low;
		return high;
		}
	else
		{
		if (freq % step == 0) return freq;

		scale = freq / step;

		low = scale * step;
		high = (scale + 1) * step;
		}

	if (freq - low < high - freq) return low;
	return high;
}

/*
 *-----------------------------------------------------------------------------
 * path parsing generating (radio:0.0:title)
 *-----------------------------------------------------------------------------
 */

gint radio_path_is_radio(const gchar *path)
{

	if (!path || path[0] != 'r') return FALSE;
	if (strncmp(path, "radio:", 6) == 0) return TRUE;

	return FALSE;
}

gchar *radio_freq_to_path(guint32 freq, const gchar *title)
{
	gchar *text;
	gchar *ret;

	text = radio_freq_to_text(freq);

	if (title && strlen(title) > 0)
		{
		ret = g_strdup_printf("radio:%s:%s", text, title);
		}
	else
		{
		ret = g_strdup_printf("radio:%s", text);
		}
	g_free(text);

	return ret;
}

guint32 radio_path_to_freq(const gchar *path)
{
	const gchar *ptr;

	if (!radio_path_is_radio(path)) return 0;

	ptr = path;
	ptr += 6;

	return radio_freq_from_text(ptr);
}

gchar *radio_path_to_title(const gchar *path)
{
	const gchar *ptr;

	if (!path) return NULL;

	ptr = path;
	while (*ptr !=':' && *ptr != '\0') ptr++;
	if (*ptr == '\0') return NULL;
	ptr++;
	while (*ptr !=':' && *ptr != '\0') ptr++;
	if (*ptr == '\0') return NULL;
	ptr++;
	return g_strdup(ptr);
}

