/* File:      persistent_tables.P
** Author(s): David S. Warren
** Contact:   xsb-contact@cs.stonybrook.edu
** 
** Copyright (C) David S. Warren 2017
** 
** XSB is free software; you can redistribute it and/or modify it under the
** terms of the GNU Library General Public License as published by the Free
** Software Foundation; either version 2 of the License, or (at your option)
** any later version.
** 
** XSB is distributed in the hope that it will be useful, but WITHOUT ANY
** WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
** FOR A PARTICULAR PURPOSE.  See the GNU Library General Public License for
** more details.
** 
** You should have received a copy of the GNU Library General Public License
** along with XSB; if not, write to the Free Software Foundation,
** Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**
** $Id: persistent_tables.P,v 1.6 2010-08-19 15:03:38 warren Exp $
** 
*/

/* to do:

1. fix table_index implementation in cp_opt.P to throw an error if a
call does not have a specified mode.  (To better find errors! Maybe
optionally?)

2. Make table_persistent(p(_,_,_),..).
AND table_index(p(_,_,_),..). work together.  And table alone.
and put table_persistent into compiler...

*/

:- comment(title,"Persistent Tables").
:- comment(subtitle,"Persistent Tables on Disk").
:- comment(author,"David S. Warren").
:- comment(copyright,"Copyright 2017 David S. Warren, XSB, Inc.").

:- comment(summary, "This package supports the generation and
maintenance of persistent tables stored in data files on disk (in a
choice of formats.)  Persistent tables store tuples that are computed
answers of subgoals, just as internal XSB tables do.  Persistent
tables allow tables to be shared among concurrent processes or between
related processes over time.  XSB programmers can declare a predicate
to be persistently tabled, and the system will then, when a subgoal
for the predicate is called, look to see if the corresponding table
exists on disk, and, if it does, read the tuples that are answers for
the subgoal on demand from the data file.  If the persistent table for
the subgoal does not exist, the XSB subgoal will be called and the
tuples that are returned as answers will be stored on disk, and then
returned to the call.  Persistent tables cannot be recursively
self-dependent, unlike internal XSB tables.  Normally the tables are
subsumptive tables and abstracted from the original call.  They act
like (internal) subsumptive tables with call abstraction.

A persistent table can serve to communicate between two XSB processes:
a process that requests the evaluation of a subgoal and a sub-process
that evaluates that subgoal.  This is done by declaring a persistently
tabled predicate to have its subgoals be evaluated by a subprocess.
In this case, when a persistent table for a subgoal needs to be
created, a subprocess will be spawned to compute and save the subgoal
answers in the persistent table.  The calling process will wait for the
table to be computed and filled and, when the table is completed, will
continue by reading and returning the tuples from the generated
persistent table to the initial calling subgoal.

Persistent tables and internal tables (i.e., normal XSB tables) are
independent: a predicate may be persistently tabled but not (internally)
tabled, tabled but not persistently tabled, neither or both.  In many
cases one will want to (internally) table a persistently tabled
predicate, but not always.

Persistent tables provide a declarative mechanism for accessing data
files.  A data file can be used to define a persistent table.  I.e., a
data file can be used to define a persistent table for a desired goal.
The data file format must conform to the format declared for the
persistent table for its goal.  When this is done, simply invoking the
goal will access the persistent table, i.e., the data from the data
file.  One may, or may not, want to (internally) table this goal.  If
it is (internally) tabled then this will act similarly to a load_dyn
of the original data.

In a similar way, persistent tables can serve as a communications
mechanism among processes defined by different programming languages,
with the tables accessed and/or generated by the various processes.

This module supports using persistent tables in a ""view generation
framework.""  This is done by:

@begin{enumerate}

@item Defining a module that contains persistent tabled predicates
that correspond to the desired (stored) views.

@item Using @pred{pt_need/1} declarations (see below) to declare table
dependencies to support concurrent table evaluation.

@item Running a view-generation process (@pred{pt_fill/1/2}) to
compute the desired views by calling XSB processes.  The
view-generation process will ""pre""-compute the requred tables in a
bottom-up order, using multiple concurrent processes as appropriate
(and declared.)  Since no XSB persistently tabled predicate will be
called until after all the persistent tables that it depends on have
been computed, all XSB predicates will run using those precomputed
persistent tables, without blocking and without having to re-compute
any of them.

@end{enumerate}
	   ").

:- comment(module, "The @tt{persistent_tables} subsystem maintains
persistent tables in directories and files in a subdirectory of the
directory containing the source code for a module that defines
persistently tabled predicates.  The subdirectory is named
@file{xsb_persistent_tables/}.  Only predicates defined in a
(non-usermod) module can be persistently tabled.  For each module with
declared persistent tables, there is a subdirectory (whose name is the
module name) of @file{xsb_persistent_tables/} that contains the contents
of its tables.  In such a subdirectory there is a file, named
@file{PT_Directory.P}, that contains information on all existent
persistent tables (stored or proposed.)  The subdirectory also contains
all the files that store the contents of persistent tables for the given
module.

Currently the way a predicate is declared to be persistently tabled is
somewhat verbose and redundant.  This is because, at this time, there
is no XSB compiler (or preprocessor) support for persistent tables, and
therefore the user must define explicitly all the predicates necessary
for the implementation.  In the future, if this facility proves to be
useful, and used, we will extend the compiler (or add a preprocessor)
to simplify the necessary declarations.

The following packaging and import statements, and predicate
definition, are needed once in any module that uses persistent tables:

@begin{verbatim}
:- packaging:bootstrap_package('persistent_tables','persistent_tables').
:- import table_persistent/5, pt_call/1 from persistent_tables.
:- export ensure_<Module>_loaded/0.
ensure_<Module>_loaded.
@end{verbatim}

The specific ensure_<Module>_loaded/0 predicate (where <Module> is the
actual name of the module) is called by the system when it is required
that the module be loaded.

A persistent table for predicate @tt{Pred/K} is declared and defined as
follows:

@begin{verbatim}
:- export Pred/K, Pred_ptdef/K.
:- table_persistent(PredSkel,ModeList,TableInfo,ProcessSpec,DemandGoal).
PredSkel :- pt_call(PredSkel).
Pred_ptdef(....) :- ... definition of Pred/K ....
@end{verbatim}
	   
@tt{PredSkel} indicates a most-general goal for the predicate
@pred{Pred/K}.

As can be seen, the user must define an auxiliary predicate of the
same arity as the persistently tabled predicate, whose name is the
original predicate name with ""@tt{_ptdef}"" appended.  This predicate is
defined using the clauses intended to define @pred{Pred/K}.
@pred{Pred/K} itself is defined by the single clause that calls the
persistent-tabling meta-predicate @pred{pt_call/1}.  This meta-predicate
will generate subgoals for @pred{Pred_undef/K} and call them as is
required.

The arguments of the @pred{table_persistent/5} declaration are as
follows:

@begin{itemize}

@item{@var{PredSkel}}: is the goal whose instances are to be persistently
tabled.  Its arguments must be distinct variables.

@item{@var{ModeList}}: a list of mode-lists (or a single mode-list.)
A mode-list is a list of constants, @tt{+}, @tt{t}, @tt{-}, and
@tt{-+} with a length equal to the arity of @var{Goal}.  The mode
indicates constraints on the state of its corresponding argument in a
subgoal call.  A ""@tt{-}"" mode indicates that the corresponding
position of a call to this goal may be bound or free and is to be
abstracted when filling the persistent table; a ""@tt{+}"" mode
indicates that the corresponding position must be bound and is not
abstracted, and so a separate persistent table will be kept for each
call bound to any specific constant in this argument position; a
""@tt{t}"" mode indicates that this argument must be bound to a
""timestamp"" value.  I.e., it must be bound to an integer obtained
from the persistent tabling system that indicates the snapshot of this
table to use.  (See @pred{add_new_table/2} for details on using
timestamps.)  A ""@tt{-+}"" mode indicates that the corresponding
argument may be bound or free, but on first call, it will be
abstracted and a separate table will be constructed for each value
that this argument may take on.  So it is similar to a ""@tt{-}"" mode
in that it is abstracted, but differs in that it generates multiple
tables, one for each distinct value this argument takes on.  This can
be used to split data into separate files to be processed
concurrently.  (A future implementation may use a single indexed file
to store the generated set of tables, indexed on the @tt{-+}
argument.)

There may be multiple such mode-lists and the first one that a
particular call of @var{Goal} matches will be used to determine the
table to be generated and persistently stored.  A call does @em{not}
match a mode-list if the call has a variable in a position that is a
""+"" in that mode-list.  If a call does not match any mode-list, an
error is thrown.  Clearly if any mode list contains a @tt{t} mode, all
must contain one in the same position.  (Note: I have not as yet found
much need for multiple mode lists.)

@item{@var{TableInfo}}: a term that describes the type and format of
the persistent tables for this predicate.  It currently has only the
following possibilities:

@begin{itemize}

@item{@var{canonical}}: indicates that the persistent table will be
stored in a file as lists of field values in XSB canonical form.
These files support answers that contain variables.  (Except, answers
to goals with modes of @tt{-+} must be ground.)

@item{@var{delimited(OPTS)}}: indicates that the persistent table will
be stored in a file as delimited fields, where OPTS is a list of
options specifying the separator (and other properties) as described
as options for the predicate @pred{read_dsv/3} defined in the XSB lib
module @tt{proc_files}.  Goal answers stored in these files must be
ground.

@end{itemize}

@item{@var{ProcessSpec}}: a term that describes how the table is to
be computed.  It can be one of the following forms:

@begin{itemize}

@item{@var{xsb}}: indicating that the persistent table will be filled by
calling the goal in the current xsb process.

@item{@var{spawn_xsb}}: indicating that the persistent table will be
filled by spawning an xsb process to evaluate the goal and fill the table.

@end{itemize}

@item{@var{DemandGoal}}: a goal that will be called just before the
main persistently tabled goal is called to compute and fill a
persistent table.  The main use of this goal is to invoke
@pred{pt_need/1} commands (see below) to indicate to the system that
the persistent tables that this goal depends on are needed.  This
allows tables that will be needed by this computation to be computed
concurrently by other processes.  This is the way that parallel
computation of a complex query is supported.

[Note: A future extension may try to automatically generate these goals from
the source program, or from a previous execution of the program.  The
details remain to be designed and implemented...]

@end{itemize}

The file named @file{PT_Directory.P} is maintained by the subsystem to
keep track of the state of persistent tables for its associated module.
It contains facts for two predicates: @pred{table_instance/8} and
@pred{table_instance_cnt/2}.

The predicate @pred{table_instance(TId, Goal, Module, Status,
GoalArgs, AnsVars, TableInfo, FileName)} contains information on a
particular persistent table, as follows:

@begin{itemize}	   

@item{@var{TId}}: a unique id for the persistent table.  It is unique
for the module.  It is generated by concatenating the predicate name
and a unique number (generated using the fact in
@pred{table_instance_cnt/2}.)  (The predicate name is actually
unnecessary, since the number uniquely identifies the table.  The name
is included to make it easier for a user to see the goal a table file
is associated with.)

@item{@var{Goal}}: the goal of the persistently tabled predicate that
generates this table.

@item{@var{Module}}: the module of the persistently tabled
predicate. (Maybe should eliminate this field, since it is not
necessary, the super-directory is now the module, so we need to know
it to get here...)

@item{@var{Status}}: the status of this persistent table.  It can be:

@begin{itemize}							 

@item{@var{generated(DateTimeGen,DateTimeUsed)}}: indicating that the
table is completed and available for use. @var{DateTimeGen} is the
date and time it was generated.  @var{DateTimeUsed} is the date and
time it was most recently used.  (Updating this date means writing the
PT_Directory.P file more often.  Is it worth that overhead to keep
this value?)

@item{@var{group_generated(DateTimeGen,DateTimeUsed)}}: indicating
that this goal generated a group of files, based on a @tt{-+} mode.
This fact does not describe a single table but stands for group of
tables.

@item{@var{being_generated(Pid,DateTime)}}: indicating that the table
is in the process of being generated by process with process ID
@var{Pid}, and started generation at @var{DateTime}.

@item{@var{invalid(DateTime)}}: indicating that the generation of this
table failed or aborted in some way, at time @var{DateTime}, and so is
not valid.  (There is work to do to maintained this field, by catching
errors, associating them with the correct Tid, and updating this
value to allow propagation of failure.)

@item{@var{needs_generation(DateTime)}}: indicating that the table was
requested to be generated (at @var{DateTime}), but no process is
currently in the process of generating it.  This status is set by
@pred{pt_need/1} and is used to support concurrent generaton of
persistent tables.

@end{itemize}							 

@item{@var{GoalArgs}}: a list, @var{Arity} long, of arguments to
@var{Goal}, that generates this persistent table.  These are exactly the
arguments of @var{Goal}.

@item{@var{GoalVars}}: the list of variables in @var{Goal}.  These
generate the answer tuples and correspond to the fields in the
persistent table.

@item{@var{TableInfo}}: the table info for this table, as described
above for @pred{table_persistent/5}.

@item{@var{FileName}}: the name of the file containing the table data.

@end{itemize}	   

The predicate @pred{table_instance_cnt/2} has one fact that defines
two system values: 1) the last number used to name a unique table
file, when generating a @var{TId} for a persistent table.  This is
incremented every time a new persistent table file is created and used
in the name of that file.  And 2) a non-negative integer represeting
the most-recent version time-stamp used.  (See the predicates below
for how time-stamps can be used.)

The contents of the persistent tables that are described in
@pred{table_instance/8} are stored in files in the same directory as
its @file{PT_Directory.P} file.  The files are named ""table_<TId>.P""
(for files containing canonical terms, and .txt for delimited files
containing separated values.)


@section{Methodology for Defining View Systems}

As mentioned above, persistent tables can be used to construct view
systems, i.e., DAGs representing expressions over functions on
relations.  A relational function is a basic view definition.  An
expression over such functions is a view system.  The leaf relations
in the expression are the base relations, and every sub-expression
defines a view.  A view expression can be evaluated bottom up, given
values for every base relation.  Independent subexpressions can be
evaluated in parallel.  Failing computations can be corrected, and
only those views depending on a failed computation need to be
re-computed.

Sometimes view systems are required to be ""incremental"".  That is,
given a completely computed view system, in which the base relations
are given and all derived relations have been computed, we are given
tuples to add to (and maybe delete from) the given base relations, and
we want to compute all the new derived view contents.  In many systems
such incremental changes to the base relations result in incremental
changes to the derived relations, and those new derived relations can
be computed in much less time than would be required to recompute all
the derived relations starting from scratch with the new (updated)
base relations.

To implement a view system in XSB using persistent tables, each view
definition is provided by the definition of a persistently tabled
predicate.  Then given table instances for the base relations, each
view goal can be called to create a persistent table representing the
contents of the corresponding derived view.

The following describes, at a high level, a methodology for
implementing a given view system in XSB using persistent tables.

@begin{enumerate}

@item Define the top-level view relations, just thinking Prolog, in a
single XSB module.  A top-level relation is the ultimate desired
output of a view system, i.e., a relation that is normally not used in
the definition of another view.  Define supporting relations as seems
reasonable.  Don't worry about efficiency. Use Prolog intuitions for
defining relations.  Don't worry about incrementality; just get the
semantics defined correctly.

@item Now think about bottom-up evaluation.  I.e., we use subsumptive
tables, so goals will be called (mostly) open, with variables as
arguments.  Decide what relations will be stored intermediate views.
Restructure if necessary to get reasonable stored views.

@item Now make it so the stored views can be correctly evaluated
bottom-up, i.e., with an open call.  This will mean that the Prolog
intuition of passing bound values downward into called predicates
needs to be rethought.  For bottom-up evalution, all head variables
have to be bound by some call in the body.  So some definitions may
need new body calls, to provide a binding for variables whose values
had been assumed to be passed in by the caller.

@item Declare the stored views as table_persistent, and test on
relatively small input data. For each table_persistent, decide initially
whether to compute it in the given environment or to spawn a process
to evaluate in a new process environment.

@item If you don't need incrementality (i.e., given relatively small
additions/deletions to the base relations, compute the new derived
relations without recomputing results for old unchanged data): then
tune (maybe adding split-compute-join concurrency, using the @tt{-+}
mode, as appropriate.)  And you're done.

@item If you *do* need incrementality: In principle, the system ought to
be able automatically to transform the program given thus far into an
incremental version.  (See Annie Liu's research.)  But at this point,
I don't know how to do this ensuring that the reslting performance is
close to optimal.  (Maybe Annie does, but...)  So we will transform
the existing program by hand, and we will give ""rules-of-thumb"" to
help in this process.

@item To begin, we will assume that we are only adding new contents to
the existing views.  Now, for every stored view predicate P in the
existing definition, we make two predicates: old_P and delta_P.  The
predicate old_P will contain the tuples of the existing ... (to be
continued...)

@end{enumerate}

@section{Using Timestamps (or version numbers)}

The persistent table package provides some support for integer
timestamps for versioning of tables.  The programmer can define view
predicates with an argument whose value is a version number.  The
version number must be bound on all calls to persistently tabled goals
that contan them.  Normally a subgoal of a persistently tabled
predicate with a given version number will depend on other subgoals
with the same version.  This allows the programmer to keep earlier
versions of tables for view systems, in order to back out changes or
to keep a history of uses of the view system.  So normally a new set
of base tables will get a new version number, and then all subgoals
depending of those base tables will have that same version number.

The @pred{pt_add_table/3} predicate will add base tables and give them
a new version number, returning that new version number.  This allows
the programmer to use that version number in subsequent calls to
@pred{pt_fill} to fill the tables with the correct version.  Also,
when calling the predicate @pred{pt_eval_viewsys/5} the @var{Time}
variable can be used in the subgoals in the @var{FillList} to invoke
the correctly versioned subgoals.

A particularly interestng use of versions is in the implementation of
incremental view systems.  Recall that in an incremental view system,
one has a table that contains the accumulated records named, say,
old_records/5, and receives a base table of new records to process
named, say, new_records/5.  The incremental view system will define an
updated record file named, say, all_records/5, which will contain the
updated records after processing and includng the new_records.  It is
natural to use versions here, and make each predicate old_record/5,
new_record/5, and old_record/5 have a version argument, say the first
argument.  Then note that we can define old_records in terms of the
previous version of all_records, as follows:

@begin{verbatim}
old_records(Time,....) :-
     Time > 1,
     PrevTime is Time - 1,
     all_records(PrevTime,...).
@end{verbatim}

Note that the version numbers, being always bound on call (and treated
according to a @tt{+} mode), will not appear in any stored table.  The
numbers will appear only in the called subgoals that are stored in the
@pred{table_instance/8} predicate in the @file{PT_Directory.P} file.
So using version numbers does not make the persistent tables any
larger.

").

:- export table_persistent/5.	% declare persistent table
:- export pt_call/1. 		% evaluate persistently tabled goal, returning answers
:- export pt_fill/1.		% generate persistent table, if nec.
:- export pt_fill/2.		% generate persistent table, if nec, using N procs
:- export pt_need/1. 		% schedule ets; add needs_generation facts to PT_Directory.P
:- export pt_eval_viewsys/5.	% add base tables and fill dependent views
:- export pt_move_tables/1.	% move persistent tables
:- export pt_reset/1.		% delete failed and being_generated table_inst entries
:- export pt_remove_unused_tables/1. % delete files containing tables not referenced in table_instance.
:- export pt_abolish_subgoals/1. % remove tables for all given subgoals
:- export pt_add_table/2, pt_add_table/3. % add a table for a goal.  To add base tables from outside.
:- export pt_add_tables/2, pt_add_tables/3. % add tables for goals.
:- export pt_delete_table/1.		    % delete tables for goals.
:- export pt_spawn_call/1.    % for top-level handling of spawned goal
:- export pt_delete_later/2.		% to delete timestamped tables later than a given time
:- export pt_delete_earlier/2.  % to delete timestamped tables earlier than a given time

:- export pt_generate_table/10.

:- dynamic persistent_tables/5.

:- import save_pt_directory/2, lock_tables/2, unlock_tables/3,
	unlock_tables/2, load_pt_directory/2,
	ensure_directories_exist/2, write_in_fmt/3,
	get_table_filename/5, new_tableid/2, new_timestamp/1,
	get_stdout_filename/4, reset_timestamp/1,
	tabdir_time/3
	from pt_utilities.
:- import get_pt_dir_dir/3 from pt_utilities.
:- import call_and_distribute/8 from pt_grouper.

:- import term_new_mod/3 from machine.
:- import term_psc/2, psc_name/2, psc_data/2 from machine.
:- import datime/1, call_c/1 from standard.
:- import sys_pid/1, process_control/2 from shell.
:- import concat_atom/2, term_to_atom/3 from string.
:- import member/2, ith/3, length/2 from basics.
:- import read_dsv/3 from proc_files.
:- import xsb_configuration/2 from xsb_configuration.
:- import search_module/6 from consult.
:- import list_directory/2, rm/1 from shell.
:- import misc_error/1 from error_handler.

:- comment(pt_call/1," @pred{pt_call(+Goal)} assumes that @var{Goal}
is persistently tabled and calls it.  This predicate is normally used
only in the definition of the @tt{_ptdef} version of the persistently
tabled predicate, as described above.

If the table for @var{Goal} exists, it reads the table file and
returns its answers.  If the table file is being generated, it waits
until it is generated and then reads and returns its answers.  If the
table file doesn't exist and is not in the process of being generated,
it generates the table and then returns its results.  If the persistent
table process declaration indicates @tt{spawn_xsb}, it spawns a
process to generate the table and reads and returns those answers when
the process is completed.  If the process indication is @tt{xsb}, it
calls the goal and fills the table if necessary, and returns the
answers.  ").

:- import excess_vars/4 from setof.
%% call persistently tabled goal, generate table if nec and return answers
pt_call(Goal) :-
	ensure_mod_loaded(Goal,Goal1,Module,ModuleDir),
	ensure_table_available(Goal1,Module,ModuleDir), % lock and unlock
	%% use table_instance left from ensure_table_loaded to get table(s) to use,
	(get_usable_table(Goal,Module,generated(_,_),AnsArgs,TableId,FileName,TableInfo)
	 ->	true
	 ; get_usable_table(Goal,Module,group_generated(_,_),AnsArgs,_TableId,_FileName,_TableInfo)
	 ->	excess_vars(Goal,AnsArgs,[],GrpVars),
		GrpVars \== [], % fail if group value has no answers.
		get_group_instance_tables(Goal,Module,AnsArgs,TableId,FileName,TableInfo)
	 ;	misc_error(('[PERSISTENT TABLES: pt_call] ERROR: internal error; table not generated for: ',Goal))
	),
	gensym('_ctr_',Ctr),
	(fmt_write(userout,'Reading Persistent Table: %s for %S\n',args(FileName,Goal)),
	 conset(Ctr,0)
	 ;
	 conget(Ctr,NumRead),
	 fmt_write(userout,'Read Persistent Table: %d records from %s for %S\n',args(NumRead,FileName,Goal)),
	 fail
	 ),
	 get_persistent_table_data(TableInfo,AnsArgs,FileName),
	 coninc(Ctr).

:- comment(pt_fill/1,"The predicate @pred{pt_fill(+GoalList)} checks
if the persistent table for each persistently tabled @var{Goal} in
@var{GoalList} exists and creates it if not.  It should always succeed
(once, unless it throws an error) and the table will then exist.  If
the desired table is already generated, it immediately succeeds.  If
the desired table is being generated, it looks to see if there is
another table that is marked as @tt{needs_generating} and, if so,
invokes the @pred{pt_fill/1} operation for that table.  It continues
this until it finds that @var{Goal} is marked as @tt{generated}, at
which time it returns successfully.  If no table for @var{Goal} exists
or is being generated, it generates it.  ").

pt_fill([]) :- !.
pt_fill([Goal|Goals]) :- !,
	ensure_mod_loaded(Goal,Goal1,Module,ModuleDir),
	ensure_table_available(Goal1,Module,ModuleDir),
	pt_fill(Goals).
pt_fill(Goal) :- 
	ensure_mod_loaded(Goal,Goal1,Module,ModuleDir),
	ensure_table_available(Goal1,Module,ModuleDir).

:- comment(pt_fill/2," @pred{pt_fill(+Goal,+NumProcs)} is similar to
@pred{pt_fill/1} except that it starts @var{NumProcs} processes to
ensure that the table for @var{Goal} is generated.  Note that filling
the table for @var{Goal} may require filling many other tables.  And
those table may become marked as @tt{needs_generation}, in which case
multiple processes can work concurrently to fill the required tables.
").

pt_fill(Goals,NProcs) :-
	(NProcs =< 1
	 ->	pt_fill(Goals)
	 ;	(explicit_mod_list(Goals,MGoals)
		 ->	spawn_fill_goals(NProcs,MGoals)
		 ;	misc_error('[PERSISTENT TABLES: pt_fill] ERROR: All goals to pt_fill/1 must be in same module.')
		)
	).

explicit_mod_list([],[]) :- !.
explicit_mod_list([Goal|Goals],[MGoal|MGoals]) :- !,
	explicit_mod(Goal,MGoal,_ModuleDir,_Module),
	explicit_mod_list(Goals,MGoals).
explicit_mod_list(Goal,MGoal) :-
	explicit_mod(Goal,MGoal,_ModuleDir,_Module).

explicit_mod(Goal,MGoal,ModuleDir,Module) :-
	(Goal = Module:_G
	 ->	MGoal = Goal
	 ;	ensure_mod_loaded(Goal,_Goal1,Module,ModuleDir),
		MGoal = Module:Goal
	).

:- comment(pt_need/1," @pred{pt_need(+Goals)} creates table entries in
the @file{PT_Directory.P} file for each persistently tabled @var{Goal}
in the list of goals @var{Goals}.  (@var{Goals} alternatively may be a
single persistently tabled goal.}  The new entry is given status
@tt{needs_generation}.  This predicate is intended to be used in a
goal that appears as the 5th argument of a @pred{table_persistent/5}
declaration.  It is used to indicate other goals that are required for
the computation of the goal in the first argument of its
@pred{table_persistent/5} declaration.  By marking them as ""needed"",
other processes (started by a call to @pred{pt_fill/2}) can begin
computing them concurrently.  Note that these @var{Goals} can share
variables with the main @var{Goal} of the declaration, and thus
appropriate instances of the subgoals can be generated.  For example,
if time stamps are used, the needed subgoals should have the same
variable as the main goal in the corresponding ""time"" positions.

Note that a call to @pred{pt_need/1} should appear @bf{only} in the
final argument of a @pred{table_persistent/5} declaration.  Its correct
execution requires a lock to be held and predicates to be loaded,
which are ensured when that goal is called, but cannot be correctly
ensured by any other call(s) to the @tt{persistent_tables} subsystem.
").

%% lock must be held and table_instance loaded.
pt_need([]) :- !.
pt_need([Goal|Goals]) :- !,
	schedule_table(Goal),
	pt_need(Goals).
pt_need(Goal) :- 
	schedule_table(Goal).

schedule_table(Goal0) :-
	ensure_mod_loaded(Goal0,Goal,Module,ModuleDir),
	(get_table_variant(Goal,Module,_Status,_GoalArgs,_AnsArgs,TableId,TableInfo,_FileName)
	 ->	true
	 ;	gen_new_instance_info(Goal,ModuleDir,Module,_GoalArgs,AbsGoal,_AbsArgs,GrpArgs,AnsArgs,TableInfo,_ProcSpec,_Need),
		add_table_instance(needs_generation,AbsGoal,Module,ModuleDir,GrpArgs,AnsArgs,TableInfo,TableId,_FileName)
	).

ensure_mod_loaded(Goal0,Goal,Module,ModuleDir) :-
	(Goal0 = Module:Goal1
	 ->	term_new_mod(Module,Goal1,Goal)
		%%term_psc(Goal,PSC)
	 ;	Goal = Goal0,
		term_psc(Goal,PSC),
		psc_modname(PSC,Module0),
		Module = Module0
	),
	ensure_mod_loaded(Module,ModuleDir).

ensure_mod_loaded(Module) :-
	concat_atom([ensure_,Module,'_loaded'],LoadModCmd),
	call_c(Module:LoadModCmd).

ensure_mod_loaded(Module,ModuleDir) :-
	ensure_mod_loaded(Module),
	search_module(Module,ModuleDir0,_ModName,_Ext,_Base,_Obj),
	xsb_filename(ModuleDir0,ModuleDir).

psc_modname(Psc,ModName) :-
	psc_data(Psc,ModPscOrFile),
	(integer(ModPscOrFile)
	 ->	(ModPscOrFile =:= 0
		 ->	ModName = usermod
		 ;	psc_name(ModPscOrFile,ModName)
		)
	 ;	ModName = usermod
	).

%% if FileName is a variable, it will be given the default name.
add_table_instance(StatFrm,AbsGoal,Module,ModuleDir,GrpArgs,AnsArgs,TableInfo,TableId,FileName) :- 
	datime(DTime),
	(StatFrm == being_generated
	 ->	sys_pid(Pid),
		Status = being_generated(Pid,DTime)
	 ; StatFrm == needs_generation
	 ->	(GrpArgs == []
		 ->	Status = needs_generation(DTime)
		 ;	Status = needs_generation(DTime) % need?? eliminate test if not...
		)
	),
	AbsGoal =.. [Pred|AbsArgs],
	(var(TableId)
	 ->	new_tableid(Pred,TableId)
	 ;	retractall(table_instance(TableId,_,_,_,_,_,_,_))
	),
	(var(FileName)
	 ->	get_table_filename(ModuleDir,Module,TableId,TableInfo,FileName)
	 ;	true
	),
	assert(table_instance(TableId,AbsGoal,Module,Status,AbsArgs,
			      AnsArgs,TableInfo,FileName)).

%% returns TableId and all info of table to use
ensure_table_available(Goal,Module,ModuleDir) :-
	lock_tables(ModuleDir,Module),
	load_pt_directory(ModuleDir,Module),
	(get_usable_table(Goal,Module,Status0,AnsArgs,TableId,FileName,TableInfo)
	 ->	(Status0 = needs_generation(_)
		 ->	%% if AnsArgs not variant of GoalArgs?, wrong table, not right
			fmt_write(userout,'Generating Table for %s:%S, (%s)\n',args(Module,Goal,TableId)),
			update_instance_and_gen_table(ModuleDir,Goal,Module,TableInfo,TableId,FileName,_AnsArgs,Status)
		 ;	Status = Status0 %%, AnsArgs = AnsArgs0
		)
	 ;	fmt_write(userout,'Generating Table for %s:%S\n',args(Module,Goal)),
		update_instance_and_gen_table(ModuleDir,Goal,Module,TableInfo,TableId,_FileName,AnsArgs,Status)
	),
	wait_till_table_ready(Status,ModuleDir,Module,TableId,NewStatus),
	%% only need following to keep last used date for generated, more efficient if we dont
	(retract(table_instance(TableId,UGoal,Module,_Status,GoalArgs,AArgs,_NTableInfo,NFileName))
	 do_all
	 assert(table_instance(TableId,UGoal,Module,NewStatus,GoalArgs,AArgs,TableInfo,NFileName))
	),
	save_pt_directory(ModuleDir,Module), % only needed to keep last use date
	unlock_tables(ModuleDir,Module).

%% tables locked and table_instance loaded
update_instance_and_gen_table(ModuleDir,Goal,Module,TableInfo,TableId,FileName,AnsArgs,Status) :-
	gen_new_instance_info(Goal,ModuleDir,Module,GoalArgs,AbsGoal,AbsArgs,GrpArgs,AnsArgs,TableInfo,ProcSpec,Need),
	add_table_instance(being_generated,AbsGoal,Module,ModuleDir,GrpArgs,AnsArgs,TableInfo,TableId,FileName),
	(do_all
	 call(Need) % update needed, by calling goal in last arg of table_persistent
	),
	save_pt_directory(ModuleDir,Module),
	unlock_tables(ModuleDir,Module),
	pt_generate_table(Goal,Module,TableInfo,ProcSpec,AbsArgs,GrpArgs,AnsArgs,ModuleDir,TableId,FileName),
	lock_tables(ModuleDir,Module),
	load_pt_directory(ModuleDir,Module),
	GoalArgs = AbsArgs,
	datime(Datime),
	(GrpArgs == []
	 ->	Status = generated(Datime,Datime)
	 ;	Status = group_generated(Datime,Datime)
	).


%% tables locked.
get_usable_table(Goal,Module,Status,AnsArgs,TableId,FileName,TableInfo) :-
	term_variables(Goal,GoalVars),
	term_new_mod(usermod,Goal,UGoal),
	once((table_instance(TableId,UGoal,Module,Status,GoalArgs,AnsArgs,TableInfo,FileName),
	      is_most_general_term(GoalVars))), % first at least as general
	Goal =.. [_|GoalArgs].

get_group_instance_tables(Goal,Module,AnsArgs,TableId,FileName,TableInfo) :-
	term_new_mod(usermod,Goal,UGoal),
	table_instance(TableId,UGoal,Module,generated(_,_),GoalArgs,AnsArgs,TableInfo,FileName),
	Goal =.. [_|GoalArgs].

get_table_variant(Goal,Module,Status,GoalArgs,AnsArgs,TableId,TableInfo,FileName) :-
	term_variables(Goal,GoalVars),
	term_new_mod(usermod,Goal,UGoal),
	once((table_instance(TableId,UGoal,Module,Status,GoalArgs,AnsArgs,TableInfo,FileName),
	      is_most_general_term(GoalVars),
	      is_most_general_term(AnsArgs)
	     )),
	Goal =.. [_|GoalArgs].


%% must hold lock on entry and exit (unless abort...)
wait_till_table_ready(generated(GenTime,_),_ModuleDir,_Module,_TableId,generated(GenTime,Datime)) :- !,
	datime(Datime).
wait_till_table_ready(group_generated(GenTime,_),_ModuleDir,_Module,_TableId,group_generated(GenTime,Datime)) :- !,
	datime(Datime).
wait_till_table_ready(invalid(_),ModuleDir,Module,TableId,_) :- !,
	unlock_tables(ModuleDir,Module,_FileTime),
	misc_error(('[PERSISTENT TABLES: pt_fill] ERROR: Encountered INVALID table: ',TableId,' in ',ModuleDir)).
wait_till_table_ready(being_generated(_,_),ModuleDir,Module,TableId,NewStatus) :-
	(find_needs_generation_and_fill(ModuleDir)
	 ->	true
	 ; may_still_be_something_to_do
	 ->	unlock_tables(ModuleDir,Module,FileTime),
		wait_till_tabdir_changes(0,ModuleDir,Module,FileTime),
		lock_tables(ModuleDir,Module),
		load_pt_directory(ModuleDir,Module)
	 ;	true
	),
	(table_instance(TableId,_Goal,_Module,NStatus,_GoalArgs,_AnsArgs,_TableInfo,_FileName)
	 ->	wait_till_table_ready(NStatus,ModuleDir,Module,TableId,NewStatus)
	 ;	unlock_tables(ModuleDir,Module,_),
		misc_error(('[PERSISTENT TABLES: pt_fill] ERROR: Table ',TableId,' disappeared!?!?'))
	).

wait_till_tabdir_changes(Cnt,ModuleDir,Module,FileTime) :-
	sleep(1),
	tabdir_time(ModuleDir,Module,CurFileTime),
	((FileTime @< CurFileTime ; Cnt > 20)
	 ->	true, writeln(userout,file_changed_or_timeout)
	 ;	Cnt1 is Cnt+1,
		wait_till_tabdir_changes(Cnt1,ModuleDir,Module,FileTime)
	).

may_still_be_something_to_do :-
	table_instance(_TableId,_Goal,_Module,NStatus,_GoalArgs,_AnsArgs,_TableInfo,_FileName),
	(NStatus = being_generated(_,_)
	 ->	true
	 ;	NStatus = needs_generation
	).

%% enter and leave locked
find_needs_generation_and_fill(ModuleDir) :-
	%%writeln(userout,try_compute_ahead),
	(table_instance(TableId,Goal0,Module,needs_generation(_),_GoalArgs,_AnsArgs,TableInfo,FileName)
	 ->	term_new_mod(Module,Goal0,Goal),
		%%writeln(userout,took_table_to_gen(TableId,Goal0)),
		update_instance_and_gen_table(ModuleDir,Goal,Module,TableInfo,TableId,FileName,__AnsArgs,Status),
		wait_till_table_ready(Status,ModuleDir,Module,TableId,NewStatus),
		%% only need following to keep last used data for generated, more efficient if we dont
		(retract(table_instance(TableId,UGoal,Module,_Status,GoalArgs,AArgs,_NTableInfo,NFileName))
		 do_all
		 assert(table_instance(TableId,UGoal,Module,NewStatus,GoalArgs,AArgs,TableInfo,NFileName))
		)
	 ;      fail
	).

gen_new_instance_info(Goal,ModuleDir,Module,GoalArgs,AbsGoal,AbsArgs,GrpArgs,AnsArgs,TableInfo,ProcSpec,Need) :-
	Goal =.. [Pred|GoalArgs],
	(persistent_tables(Goal,ModeLists,TableInfo,ProcSpec,Need), % orig spec for persistent table
	 member(ModeList,ModeLists),
	 is_usable_mode(ModeList,GoalArgs,AbsArgs,GrpArgs,AnsArgs)
	 ->	true
	 ;	unlock_tables(ModuleDir,Module,_),
		(persistent_tables(Goal,ModeLists,_TableInfo,_ProcSpec,_Need)
		 ->	misc_error(('[PERSISTENT TABLES: pt_need] ERROR: No appropriate mode for ',
				    Goal,' from ',ModeLists))
		 ;	misc_error(('[PERSISTENT TABLES: pt_need] ERROR: ',Goal,' is not persistently tabled.'))
		)
	),
	AbsGoal =.. [Pred|AbsArgs].


is_usable_mode([],[],[],[],[]).
is_usable_mode([Mode|ModeList],[Arg|Args],[AbsArg|AbsArgs],GrpArgs,AnsArgs) :-
	((Mode == (+) ; Mode == t)
	 ->	(ground(Arg)
		 ->	AbsArg = Arg,
			GrpArgs = GrpArgs1,
			AnsArgs = AnsArgs1
		 ;	fail
		)
	 ; Mode == (-)
	 ->	(var(Arg)
		 ->	AbsArg = Arg
		 ;	true
		),
		GrpArgs = GrpArgs1,
		AnsArgs = [AbsArg|AnsArgs1]
	 ; Mode == (-+)
	 ->	GrpArgs = [AbsArg|GrpArgs1],
		AnsArgs = AnsArgs1
	 ;	fail
	),
	is_usable_mode(ModeList,Args,AbsArgs,GrpArgs1,AnsArgs1).

:- comment(pt_generate_table/10,"This is an internal predicate used by the
@tt{persistent_tables} system.  It needs to be exported because it is
used across process boundaries.").

:- index pt_generate_table/10-4.
pt_generate_table(SubGoal,Module,FileFmt,xsb,AbsArgs,GrpArgs,AnsArgs,ModuleDir,TableId,FileName) :-
	(SubGoal = Module1:SubGoal1
	 ->	(Module1 == Module
		 ->	true
		 ;	misc_error(('[PERSISTENT TABLES: pt_generate_table] ERROR: Inconsistent module specs in call to pt_generate_table: ',
				    Module,' and ',Module1))
		)
	 ;	SubGoal1 = SubGoal
	),
	SubGoal1 =.. [Pred|_GoalArgs],
	AbsGoal =.. [Pred|AbsArgs],
	term_new_mod(Module,AbsGoal,MAbsGoal),

	(GrpArgs == []
	 ->	call_and_write_in_fmt(FileFmt,MAbsGoal,FileName,AnsArgs),
		lock_tables(ModuleDir,Module),
		load_pt_directory(ModuleDir,Module),
		datime(Datime),
		(retract(table_instance(TableId,NSubGoal,Module,_,TVars,AArgs,TableInfo,FileName))
		 do_all
		 assert(table_instance(TableId,NSubGoal,Module,generated(Datime,Datime),TVars,AArgs,TableInfo,FileName))
		)
	 ;      %%writeln(userout,calling_call_and_distribute),
		construct_ptdef_goal(MAbsGoal,MAbsGoalDef),
		call_and_distribute(MAbsGoal,MAbsGoalDef,Module,ModuleDir,FileFmt,GrpArgs,AnsArgs,Groups),
		lock_tables(ModuleDir,Module),
		load_pt_directory(ModuleDir,Module),
		%% update table_instance for all groups, and for group generator.
		update_table_instance_for_groups(TableId,MAbsGoal,AbsArgs,GrpArgs,AnsArgs,ModuleDir,Module,FileFmt,Groups)
	),
	save_pt_directory(ModuleDir,Module),
	unlock_tables(ModuleDir,Module).
pt_generate_table(SubGoal,Module,FileFmt,spawn_xsb,AbsArgs,GrpArgs,AnsArgs,ModuleDir,TableId,FileName) :-
	%%writeln(userout,gentable_spawn(SubGoal,AbsArgs,ModuleDir,TableId)),
	(SubGoal = Module1:_Goal
	 ->	(Module1 == Module
		 ->	true
		 ;	misc_error(('[PERSISTENT TABLES: pt_generate_table] ERROR: Inconsistent module specs in call to pt_generate_table: ',
				    Module,' and ',Module1))
		)
	 ;	true
	),
	xsb_configuration(config_bindir,XSBBin),
	concat_atom([XSBBin,'/xsb'],XSBexe0),
	xsb_filename(XSBexe0,XSBexe),
	term_to_atom(([persistenttables],
		      persistent_tables:
		      pt_spawn_call(pt_generate_table(SubGoal,Module,FileFmt,xsb,AbsArgs,
						   GrpArgs,AnsArgs,ModuleDir,TableId,FileName))),
		     GoalString,[quoted(true)]),
	concat_atom(['"',XSBexe,'" -e "',GoalString,'."'],Command),
	fmt_write(userout,'Spawn: %S\n',args(Command)),

	get_stdout_filename(ModuleDir,Module,TableId,StdOutFile),
	%%concat_atom([ModuleDir,'/xsb_persistent_tables/stdout_for_',TableId,'.txt'],StdOutFile),
	open(StdOutFile,write,SStr),
	spawn_process(Command,none,SStr,SStr,Pid),
	wait_till_procs_finished([pid(Pid,SStr,Command)]).

:- import gensym/2, conset/2, coninc/1, conget/2 from gensym.
call_and_write_in_fmt(Fmt,SubGoal,TableFileName,AnsVars) :- !,
	conset('_ansctr',0),
	construct_ptdef_goal(SubGoal,SubGoalDef),
	open(TableFileName,write,OStr),
	fmt_write(userout,'Creating Persistent Table %s for %S\n',args(TableFileName,SubGoalDef)),
	(do_all
	 call(SubGoalDef),
	 coninc('_ansctr'),
	 write_in_fmt(Fmt,OStr,AnsVars)
	),
	conget('_ansctr',NumAns),
	close(OStr),
	fmt_write(userout,'Wrote Persistent Table: %d records to %s for %S\n',args(NumAns,TableFileName,SubGoalDef)).

update_table_instance_for_groups(TableId,GGoal,AbsArgs,GrpArgs,AnsArgs,ModuleDir,Module,TableInfo,Groups) :-
	%%writeln(userout,update_table_instance_for_groups(TableId,GGoal,AbsArgs,GrpArgs,AnsArgs,ModuleDir,Module,TableInfo,Groups)),
	datime(Datime),
	(retract(table_instance(TableId,NSubGoal,Module,_,TVars,AArgs,TableInfo1,FileName))
	 do_all
	 assert(table_instance(TableId,NSubGoal,Module,group_generated(Datime,Datime),TVars,AArgs,TableInfo1,FileName))
	),
	(do_all
	 member(grp(GrpArgs,GTableId),Groups),
	 get_table_filename(ModuleDir,Module,GTableId,TableInfo,FileName),
	 %%writeln(userout,table_instance(GTableId,GGoal,Module,generated(Datime,Datime),AbsArgs,AnsArgs,TableInfo,FileName)),
	 assert(table_instance(GTableId,GGoal,Module,generated(Datime,Datime),AbsArgs,AnsArgs,TableInfo,FileName))
	),
	writeln(userout,completed_group_generation(GGoal)).


construct_ptdef_goal(SubGoal0,SubGoalDef) :-
	ensure_mod_loaded(SubGoal0,SubGoal,Module,_),
	SubGoal =.. [Pred|Args],
	concat_atom([Pred,'_ptdef'],PredDef),
	SubGoalDefU =.. [PredDef|Args],
	term_new_mod(Module,SubGoalDefU,SubGoalDef).

%% read persistent table
get_persistent_table_data(canonical,AnsArgs,TableFileName) :- !,
	open(TableFileName,read,IStr,[new]),
	repeat,
	read_canonical(IStr,Values),
	(Values == end_of_file
	 ->	!,
		close(IStr),
		fail
	 ;	Values = AnsArgs
	).
get_persistent_table_data(delimited(Opts),AnsArgs,TableFileName) :-
	Term =.. [tuple|AnsArgs],
	read_dsv(TableFileName,Opts,Term).

/********************************************************/
/* Predicates defined in table files.			*/
/********************************************************/

:- dynamic table_instance/8.
:- import table_instance/8 from usermod.
:- index(table_instance/8,[1,2+3]).

:- dynamic table_instance_cnt/2.
:- import table_instance_cnt/2 from usermod.

:- comment(table_persistent/5, "This predicate (used as a directive)
declares a predicate to be persistently tabled.  The form is
@pred{table_persistent(+Goal, +Modes, +TableInfo, +ProcessSpec,
+DemandGoal)}, where:

@begin{itemize}

@item{@var{Goal}}: is the goal whose instances are to be persistently
tabled.  Its arguments must be distinct variables.
@var{Goal} must be defined by the single clause:

@begin{verbatim}
Goal :- pt_fill(Goal).
@end{verbatim}

Clauses to define the tuples of @var{Goal} must be associated with
another predicate (of the same arity), whose name is obtained from
@var{Goal}'s predicate name by appending @tt{_ptdef}.

@item{@var{ModeList}}: a list of mode-lists (or a single mode-list.)
A mode-list is a list of constants, @tt{+}, @tt{t}, @tt{-}, and
@tt{-+} with a length equal to the arity of @var{Goal}.  The mode
indicates puts constraints on the state of corresponding argument in a
subgoal call.  A ""@tt{-}"" mode indicates that the corresponding
position of the goal is to be abstracted for the persistent table; a
""@tt{+}"" mode indicates that the corresponding position is not
abstracted and a separate persistent table will be kept for each call
bound to any specific constant in this argument position; a ""@tt{t}""
mode indicates that this argument will have a ""timestamp"".  I.e., it
will be bound to an integer obtained from the persistent tabling system
that indicates the snapshot of this table to use.  (See
@pred{add_new_table/2} for details on using timestamps.)  A mode of
""@tt{-+}"" is similar to a ""@tt{-}"" mode in that the associated
argument is abstracted.  The difference is that instead of all the
answers being stored in a single table, there are multiple tables, one
for each value of this argument for which there are answers.

There may be multiple such mode-lists and the first one that a
particular call of @var{Goal} matches will be used to determine the
table to be generated and persistently stored.  A call does @em{not}
match a mode-list if the call has a variable in a position that is a
""+"" in that mode-list.  If a call does not match any mode-list, an
error is thrown.  If any mode list contains a @tt{t} mode, all must
contain one in the same position.

@item{@var{TableInfo}} is a term that describes the type and format of the
persistent tables for this predicate.  It may have the following forms,
with the described meanings:

@begin{itemize}

@item{@var{file(canonical)}}: indicates that the persistent table will be
stored in a file as lists of field values in XSB canonical form.

@item{@var{file(delimited(OPTS))}}: indicates that the persistent table
will be stored in a file as delimited fields, where OPTS is a list of
options specifying the separator (and other properties) as described
as options for the predicate @pred{read_dsv/3} in the XSB lib module
@tt{proc_files}.

@end{itemize}

@item{@var{ProcessSpec}} is a term that describes how the table is to
be computed.  It can be one of the following forms:

@begin{itemize}

@item{@var{xsb}}: indicating that the persistent table will be filled by
calling the goal in the current xsb process.

@item{@var{spawn_xsb}}: indicating that the persistent table will be
filled by spawning an xsb process to evaluate the goal and fill the table.

@end{itemize}

@item{@var{DemandGoal}}: a goal that will be called just before the
main persistently tabled goal is called to compute and fill a persistent
table.  The main use of this goal is to invoke @pred{pt_need/1}
commands (see below) to indicate to the system that the persistent
tables that this goal depends on are indeed needed.  This allows
tables that will be needed by this computation to be computed by other
processes.  This is the way that parallel computation of a complex
query is supported.

@end{itemize}
	   ").

table_persistent(Goal,Modes0,TableInfo,ProcSpec,Needed) :-
	retractall(persistent_tables(Goal,_,_,_,_)),
	(Modes0 = [F|_],atomic(F) % allow a single mode
	 ->	Modes = [Modes0]
	 ;	Modes = Modes0
	),
	functor(Goal,_Pred,Arity),
	(do_all
	 member(Mode,Modes),
	 (length(Mode,Arity)
	  ->	 \+ valid_mode(Mode),
		 misc_error(('[PERSISTENT TABLES: table_persistent] ERROR: Invalid mode: ',Mode,' for ',Goal))
	  ;	 misc_error(('[PERSISTENT TABLES: table_persistent] ERROR: Wrong length mode: ',Mode,' for ',Goal))
	 )
	),
	assert(persistent_tables(Goal,Modes,TableInfo,ProcSpec,Needed)).

valid_mode([]).
valid_mode([M|Ms]) :-
	valid_m(M),
	valid_mode(Ms).

valid_m(+).
valid_m(-).
valid_m(t).
valid_m(-+).

:- comment(pt_abolish_subgoals/1, " This predicate
@pred{pt_abolish_subgoals(+GoalList)} abolishes the persistent tables
for all goals in @var{GoalList} by removing the corresponding facts in
table_instance. The table files containing the data remain, and can be
cleaned up using @pred{pt_remove_unused_tables/1}.  ").

pt_abolish_subgoals(Goals) :-
	Goals = [FirstGoal|_],
	ensure_mod_loaded(FirstGoal,_,Module,ModuleDir),
	lock_tables(ModuleDir,Module),
	load_pt_directory(ModuleDir,Module),
	(do_all
	 member(Goal0,Goals),
	 ensure_mod_loaded(Goal0,Goal,Module,ModuleDir), % ignore if different module
	 term_variables(Goal,GoalVars),
	 table_instance(TableId,Goal,Module,_,_,AnsArgs,_,_),
	 is_most_general_term(AnsArgs),
	 is_most_general_term(GoalVars),
	 retractall(table_instance(TableId,_,_,_,_,_,_,_)),
	 fmt_write(userout,'Removed: %S table (%s)\n',
		   args(Goal,TableId))
	 
	),
	save_pt_directory(ModuleDir,Module),
	unlock_tables(ModuleDir,Module).

:- comment(pt_move_tables/1, "@pred{pt_move_tables(+MoveList)} moves
persistent tables.  @var{MoveList} is a list of pairs of goals of the
form @tt{FromGoal > ToGoal}, where @var{FromGoal} and @var{ToGoal} are
persistently tabled goals and their persistent tables have been
filled.  For each such pair the table file for @var{ToGoal} is set to
the file containing the table for @var{FromGoal}.  The table files
must be of the same format.  @var{FromGoal} has its table_instance
fact removed.  This predicate may be useful for updating new and old
tables when implementing incremental view systems.  ").

pt_move_tables(MoveList) :- 
	MoveList = [FirstGoal0>_|_],
	ensure_mod_loaded(FirstGoal0,_,Module,ModuleDir),	
	lock_tables(ModuleDir,Module),
	load_pt_directory(ModuleDir,Module),
	(do_all
	 member(FromGoal0>ToGoal0,MoveList),
	 ensure_mod_loaded(FromGoal0,FromGoal,Module,ModuleDir), % ignored if not in same module
	 ensure_mod_loaded(ToGoal0,ToGoal,Module,ModuleDir),
	 term_variables(FromGoal,FromVars),
	 table_instance(FromTableId,FromGoal,Module,_,_,FromAnsArgs,_,FromFileName),
	 is_most_general_term(FromAnsArgs),
	 is_most_general_term(FromVars),
	 term_variables(ToGoal,ToVars),
	 table_instance(ToTableId,ToGoal,Module,ToStatus,ToAbsArgs,ToAnsArgs,ToTableInfo,_),
	 is_most_general_term(ToAnsArgs),
	 is_most_general_term(ToVars),
	 retractall(table_instance(FromTableId,_,_,_,_,_,_,_)),
	 retractall(table_instance(ToTableId,_,_,_,_,_,_,_)),
	 assert(table_instance(ToTableId,ToGoal,Module,ToStatus,ToAbsArgs,ToAnsArgs,ToTableInfo,FromFileName)),
	 fmt_write(userout,'Renamed: %S table (%s) now defined in %s\n',
		   args(ToGoal,ToTableId,FromFileName))
	),
	save_pt_directory(ModuleDir,Module),
	unlock_tables(ModuleDir,Module).

:- comment(pt_remove_unused_tables/1," This predicate cleans up unused
files from the directory that stores persistent
tables. @pred{pt_remove_unused_tables(+Module)} looks through the
@file{PT_Directory.P} file for the indicated module and removes all
files with names of the form @file(table_<Tid>.P) (or .txt) for which
there is no table id of @tt{<Tid>}.  So a user may delete (or abolish)
a persistent table by simply editing the @file{PT_Directory.P} file
(when no one is using it!) and deleting its table_instance fact.  Then
periodically running this predicate will clean up the storage for
unnecessary tables.  ").
%% too hard-coded; make more parameterized, so can change just by changing pt_utilities preds.

pt_remove_unused_tables(Module) :-
	search_module(Module,ModuleDir0,_,_,_,_),
	xsb_filename(ModuleDir0,ModuleDir),
	get_pt_dir_dir(ModuleDir,Module,TableDir),
	lock_tables(ModuleDir,Module),
	load_pt_directory(ModuleDir,Module),
	(do_all
	 list_directory(TableDir,File),
	 (concat_atom([table_,TableId,'.',Suff],File),
	  (Suff == 'txt' -> true ; Suff == 'P')
	  ;	 
	  concat_atom(['stdout_for_',TableId,'.txt'],File)
	 ),
	 concat_atom([_PredName,'_',Digits],TableId),
	 atom_all_digits(Digits),
	 \+ table_instance(TableId,_,_,_,_,_,_,_),
	 concat_atom([TableDir,'/',File],FullFile),
	 (rm(FullFile)
	  ->	 fmt_write(userout,'Removed file: %s\n',args(FullFile))
	  ;	 fmt_write(userout,'Could not remove file: %s\n',args(FullFile))
	 )
	),
	unlock_tables(ModuleDir,Module).

atom_all_digits(Atom) :-
	atom_codes(Atom,Codes),
	forall(member(Code,Codes),(Code >= 0'0, Code =< 0'9)).

:- comment(pt_reset/1," @pred{pt_reset(+Module)} processes the
@file{PT_Directory.P} file and deletes all table_instance records for
tables that have status @tt{being_generated}.  This will cause them to
be re-evaluated when necessary.  This is appropriate to call if all
processes computing these tables have been aborted and were not able
to update the directory.  It may also be useful if for some reason all
processes are waiting for something to be done and no progress is
being made.  ").

pt_reset(Module) :-
	ensure_mod_loaded(Module,ModuleDir),
	lock_tables(ModuleDir,Module),
	load_pt_directory(ModuleDir,Module),
	(table_instance(TableId,Goal,_,Status,_,_,_,_),
	 (Status = being_generated(_,_) -> true ; Status = invalid(_))
	 do_all
	 retractall(table_instance(TableId,_,_,_,_,_,_,_)),
	 fmt_write(userout,'Reset: %S (%s) from %S\n',args(Goal,TableId,Status))
	),
	save_pt_directory(ModuleDir,Module),
	unlock_tables(ModuleDir,Module).


:- comment(pt_delete_later/2,"
@pred{pt_delete_later(Module,TimeStamp)} delete all tables that have a
timestamp larger than @var{Timestamp}.  It keeps the tables of the
TimeStamp snapshot.  It deletes the corresponding table records from
the PT_Directory, and removes the corresponding files that store the tuples.  ").

pt_delete_later(Module,TimeStamp) :- 
	pt_delete_tables_by_time(Module,_All,later,TimeStamp).

:- comment(pt_delete_earlier/2,"
@pred{pt_delete_earlier(Module,TimeStamp)} delete all tables that have a
timestamp smaller than @var{Timestamp}.  It keeps the tables of the
TimeStamp snapshot.  It deletes the corresponding table records from
the PT_Directory, and removes the corresponding files that store the tuples. ").

pt_delete_earlier(Module,TimeStamp) :-
	pt_delete_tables_by_time(Module,_All,earlier,TimeStamp).

%% deletes all tables with timestamps older than TimeStamp
pt_delete_tables_by_time(Module,GoalList0,When,TimeStamp) :-
	ensure_list(GoalList0,GoalList),
	goals_to_usermod(GoalList,UGoalList),
	ensure_mod_loaded(Module,ModuleDir), % to be sure persistent_tables/5 is defined
	lock_tables(ModuleDir,Module),
	load_pt_directory(ModuleDir,Module),
	findall(p(TableId,Fmt),
		(persistent_tables(AGoal,ModeLists,Fmt,_ProcSpec,_Need), % orig spec 
		 term_new_mod(usermod,AGoal,UGoal),
		 once(member(UGoal,UGoalList)),
		 table_instance(TableId,UGoal,Module,_,_,_,_,_),
		 compare_goal_time_ts(When,UGoal,ModeLists,TimeStamp)
		),
		TableIds),
	(do_all
	 member(p(TableId,_),TableIds),
	 retractall(table_instance(TableId,AGoal,Module,_,_,_,_,_))
	),
	(When \== earlier % if deleting later, reset timestamp
	 ->	reset_timestamp(TimeStamp)
	 ;	true
	),
	save_pt_directory(ModuleDir,Module),
	unlock_tables(ModuleDir,Module),
	delete_files_for(TableIds,ModuleDir,Module).

:- comment(pt_delete_table/1, " @pred{pt_delete_table(+Goal)} deletes
the table for @var{Goal} in its @file{PT_Directory.P} file, so it will
need to be regenerated when next invoked.  The actual file containing
the table data is @em{not} removed.  (It may be a file in another
directory that defines the table via a call to @pred{pt_add_table/2}
or friend.)  To remove a local file that contains the tabled data, use
@pred{pt_remove_unused_tables/1}.  ").

pt_delete_table(Goal0) :-
	ensure_mod_loaded(Goal0,Goal,Module,ModuleDir),
	term_new_mod(usermod,Goal,UGoal),
	term_variables(Goal,GoalVars),
	lock_tables(ModuleDir,Module),
	load_pt_directory(ModuleDir,Module),
	(table_instance(TableId,UGoal,Module,_Status,_GoalArgs,AnsArgs,_TableInfo,_FileName),
	 is_most_general_term(GoalVars), % first at least as general
	 is_most_general_term(AnsArgs)	 % variant
	 ->	retractall(table_instance(TableId,_,_,_,_,_,_,_)),
	 	fmt_write(userout,'Removed table for : %s:%S (%s)\n',args(Module,UGoal,TableId)),
		save_pt_directory(ModuleDir,Module)
	 ;	fmt_write(userout,'Unable to Remove table for: %s:%S\n',args(Module,UGoal))
	),
	unlock_tables(ModuleDir,Module).

goals_to_usermod([],[]).
goals_to_usermod([Goal|GoalList],[UGoal|UGoalList]) :-
	term_new_mod(usermod,Goal,UGoal),
	goals_to_usermod(GoalList,UGoalList).

ensure_list(GoalList0,GoalList) :-
	(is_list(GoalList0)
	 ->	GoalList = GoalList0
	 ;	GoalList = [GoalList0]
	).

compare_goal_time_ts(When,Goal,[ModeList|_],TimeStamp) :-
	ith(I,ModeList,t),
	Goal =.. [_P|Args],
	ith(I,Args,TS),
	(When == earlier
	 ->	TS < TimeStamp
	 ;	TS > TimeStamp
	).

delete_files_for([],_,_).
delete_files_for([p(TableId,Fmt)|TableIds],ModuleDir,Module) :-
	get_table_filename(ModuleDir,Module,TableId,Fmt,TableFileName),
	(rm(TableFileName)
	 ->	fmt_write(userout,'Removed: %s\n',args(TableFileName))
	 ;	fmt_write(userout,'Unable to remove: %s\n',args(TableFileName))
	),
	get_stdout_filename(ModuleDir,Module,TableId,StdoutFileName),
	(rm(StdoutFileName)
	 ->	fmt_write(userout,'Removed: %s\n',args(StdoutFileName))
	 ;	true
	),
	delete_files_for(TableIds,ModuleDir,Module).

spawn_fill_goals(NProcs,MGoals) :-
	start_n_procs(NProcs,MGoals,PidList),
	wait_till_procs_finished(PidList).

start_n_procs(N,MGoals,Pids) :-
	(N > 0
	 ->	xsb_configuration(config_bindir,XSBBin),
		concat_atom([XSBBin,'/xsb'],XSBexe),
		term_to_atom(([persistenttables],
			      pt_spawn_call(pt_fill(MGoals)
					   )),
			     GoalString,[quoted(true)]),
		concat_atom(['"',XSBexe,'" -e "',GoalString,'."'],Command),
		sys_pid(MyPid),
		concat_atom(['./xsb_persistent_tables/stdout_fill_goal_',MyPid,'_',N,'.txt'],StdOutFile),
		ensure_directories_exist(StdOutFile,userout),
		open(StdOutFile,write,SStr),
		spawn_process(Command,none,SStr,SStr,Pid),
		Pids = [pid(Pid,SStr,Command)|Pids0],
		N1 is N-1,
		start_n_procs(N1,MGoals,Pids0)
	 ;	Pids = []
	).

wait_till_procs_finished([]) :-
	writeln(userout,'All processes completed').
wait_till_procs_finished([pid(Pid,SStr,Command)|Pids]) :-
	fmt_write(userout,'Waiting for spawned process %d to complete: %S\n',args(Pid,Command)),
	(process_control(Pid,wait(RetCode))
	 ->	(RetCode =:= 0
		 ->	fmt_write(userout,'SUCCESS: Pid %d\n',args(Pid))
		 ;	fmt_write(userout,'FAILED: Pid %d, RC %d\n',args(Pid,RetCode))
		)
	 ;	fmt_write(userout,'[PERSISTENT TABLES: pt_fill] ERROR: Process %d disappeared\n',args(Pid,Command))
	),
	close(SStr),
	wait_till_procs_finished(Pids).
	

:- comment(pt_add_table/2," @pred{pt_add_table(+Goal,+FileName)} uses
the file @var{FileName} to create a persistent table for @var{Goal}.
@var{Goal} must be persistently tabled. It creates a new
table_instance record in the @file{PT_Directory.P} file and points it
to the given file.  The file is not checked for having a format
consistent with that declared for the persistently tabled predicate,
i.e., that it is correctly formated to represent the desired tuples.
The user is responsible for ensuring this.  ").

%%pt_add_table(new_callout(3,demo,CO,SE,TA),'../test_data/....txt'))
pt_add_table(Goal0,FileName) :-
	pt_add_table(Goal0,FileName,none).

:- comment(pt_add_table/3,"
@pred{pt_add_table(+Goal,+FileName,?TimeStamp)} uses the
file @var{FileName} to create a persistent table for @var{Goal}, which
must be persistently tabled.  It returns in @var{TimeStamp} a new (the
next) time stamp for this module (obtained from the fact for predicate
@pred{table_instance_cnt/2}} in the ET Directory.)  It is assumed that
@var{Goal} has a time argument and the returned value will be used in
its eventual call.

This predicate creates a new table_instance record in the
@file{PT_Directory.P} file and sets its defining file to be the value
of @var{FileName}.  The file is not checked for consistency, that it
is correctly formated to represent the desired tuples.  The user is
responsible for insuring this.  ").

pt_add_table(Goal0,FileName,Time) :-
	ensure_mod_loaded(Goal0,Goal,Module,ModuleDir),
	Goal =.. [_|Vars],
	functor(Goal,Pred,_Arity),
	lock_tables(ModuleDir,Module),
	load_pt_directory(ModuleDir,Module),
	(var(Time) -> new_timestamp(Time) ; true),
	(get_usable_table(Goal,Module,_,_,_,_,_)
	 ->	unlock_tables(ModuleDir,Module),
		misc_error(('[PERSISTENT TABLES: pt_add_table] ERROR: table for ',Goal0,' already exists.'))
	 ;	true
	),
	(persistent_tables(Goal,_,TableInfo,_,_)
	 ->	true
	 ;	unlock_tables(ModuleDir,Module),
		misc_error(('[PERSISTENT TABLES: pt_add_table] ERROR: ',Goal0,' not persistent tabled.'))
	),
	new_tableid(Pred,TableId),
	term_variables(Goal,AnsVars),
	NStatus = generated(NTime,NTime),
	datime(NTime),
	assert(table_instance(TableId,Goal,Module,NStatus,Vars,AnsVars,TableInfo,FileName)),
	save_pt_directory(ModuleDir,Module),
	unlock_tables(ModuleDir,Module).

:- comment(pt_add_tables/2," @pred{pt_add_tables(+GoalList,+FileList)}
is similar to @pred{pt_add_table/2} but takes a list of goals and a
corresponding list of files, and defines the tables of the goals using
the files.  ").

pt_add_tables(Goals,Files) :-
	pt_add_tables(Goals,Files,none,[]).

:- comment(pt_add_tables/3,"
@pred{pt_add_tables(+GoalList,+FileList,-Time)} is similar to
@pred{pt_add_table/3} but takes a list of goals and a corresponding
list of files, and defines the tables of the goals using the files,
returning the snapshot time in @var{Time}.  ").

pt_add_tables(Goals,Files,Time) :-
	pt_add_tables(Goals,Files,Time,[]).

%% have to handle time better: pass var for first in module (to get new ts), then 
pt_add_tables([],[],_,_).
pt_add_tables([Goal|Goals],[File|Files],Time,Modules) :-
	ensure_mod_loaded(Goal,_,Module,_),
	(member(Module,Modules)
	 ->	pt_add_table(Goal,File,Time),
		pt_add_tables(Goals,Files,Time,Modules)
	 ;	/*(var(Time)
		 ->	pt_add_table(Goal,File,Time) % need to bind Time in Goal, if there
		 ;	pt_add_table(Goal,File,Time0),
			(Time0 =:= Time
			 ->	true
			 ;	misc_error(('[PERSISTENT TABLES: pt_add_tables] ERROR: inconsistent timestamp in module ',Module))
			)
		),*/
		pt_add_table(Goal,File,Time), % need to bind Time in Goal, if there	
		pt_add_tables(Goals,Files,Time,[Module|Modules])
	).

:- comment(pt_spawn_call/1,"This is an internal predicate used by the
@tt{persistent_tables} system.  It needs to be exported because it is
used across process boundaries.").

pt_spawn_call(Goal) :-
	(catch(Goal,Ball,pt_proc_ball(Goal,Ball))
	 ->	fmt_write(userout,'SUCCESSFUL Completion of goal %S\n',args(Goal)),
		halt
	 ;	fmt_write(userout,'[PERSISTENT TABLES: pt_spawn_call] ERROR: Spawned goal failed: %S\n',args(Goal)),
		set_invalid(Goal),
		halt(4)
	).

pt_proc_ball(Goal,Ball) :-
	fmt_write(userout,'[PERSISTENT TABLES: pt_spawn_call] ERROR: Spawned goal: %S threw an error\n',args(Goal)),
	default_error_handler(Ball),
	set_invalid(Goal),
	halt(4).

%% Mark tabled goal to be invalid.
set_invalid(Goal0) :-
	ensure_mod_loaded(Goal0,Goal,Module,ModuleDir),
	term_new_mod(usermod,Goal,UGoal),
	term_variables(Goal,GoalVars),
	lock_tables(ModuleDir,Module),
	load_pt_directory(ModuleDir,Module),
	datime(Datime),
	(table_instance(TableId,UGoal,Module,_Status,GoalArgs,AnsArgs,TableInfo,FileName),
	 is_most_general_term(GoalVars), % first at least as general
	 is_most_general_term(AnsArgs)	 % variant
	 ->	retractall(table_instance(TableId,UGoal,Module,_,_,_,_,_)),
		assert(table_instance(TableId,UGoal,Module,invalid(Datime),GoalArgs,AnsArgs,TableInfo,FileName))
	 ;	fmt_write(userout,'[PERSISTENT TABLES: pt_spawn_call] ERROR: Tabled goal not found: %s:%S',
			  args(Module,UGoal))
	),
	save_pt_directory(ModuleDir,Module),
	unlock_tables(ModuleDir,Module).

:- comment(pt_eval_viewsys/5," The predicate
@pred{pt_eval_viewsys(+GoalList, +FileList, -Time, +FillList, +NProcs)}
adds user files containing base tables to a persistent tabling system
and invokes the computing and filling of dependent tables.
@var{GoalList} is a list of subgoals that correspond to the base
tables of the view system. @var{FileList} is the corresponding list of
files that contain the data for the base tables.  They must be
formated as the @tt{table_persistent} declarations of their
corresponding subgoals specify.  @var{Time} is a variable that will be
set to the timestamp, if the base goals of @var{GoalList} contain time
stamp arguments.  @var{FillList} is a list of persistently tabled
subgoals to be filled (using @pred{pt_fill/1/2}.)  @var{NProcs} is an
integer indicating the maximum number of processes to use to evaluate
the view system.  This predicate provides a simple interface to
@pred{pt_add_tables/3} and @pred{pt_fill/2}.
	   ").


pt_eval_viewsys(GoalList,FileList,Time,FillList,NProcs) :-
	pt_add_tables(GoalList,FileList,Time),
	pt_fill(FillList,NProcs).


