/* File:      tp_index.P
** Author(s): Saumya Debray, Terrance Swift
** Contact:   xsb-contact@cs.sunysb.edu
** 
** Copyright (C) The Research Foundation of SUNY, 1986, 1993-1998
** Copyright (C) ECRC, Germany, 1990
** 
** XSB is free software; you can redistribute it and/or modify it under the
** terms of the GNU Library General Public License as published by the Free
** Software Foundation; either version 2 of the License, or (at your option)
** any later version.
** 
** XSB is distributed in the hope that it will be useful, but WITHOUT ANY
** WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
** FOR A PARTICULAR PURPOSE.  See the GNU Library General Public License for
** more details.
** 
** You should have received a copy of the GNU Library General Public License
** along with XSB; if not, write to the Free Software Foundation,
** Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**
** $Id: tp_index.P,v 1.12 2010-08-19 15:03:36 spyrosh Exp $
** 
*/


/*======================================================================*/
/*  makeindex(+Sym, +Clauses, -Pil, -PilR, +PredPreg, +Tabled)  	*/
/* 	Generates try/retry/trust and index instructions for clauses	*/
/*	of the predicate/function.					*/
/*======================================================================*/

/*----------------------------------------------------------------------*/
/* single clause predicate cases: no try/index instruction is needed.	*/
/*----------------------------------------------------------------------*/
:- mode makeindex(?,?,?,?,?,+).
makeindex(Sym, [clause(_,_,crec(_,Label))], Pil, Pilr, _, Tabled) :- !,
	( Tabled == 0 -> Pil = Pilr
	; Tabled = tabled(Ind,_), sym_name(Sym, _, Arity),
	  Pil = [tabletrysingle(Arity,Label,Ind)|Pilr]
	).

/*----------------------------------------------------------------------*/
/* 0-ary predicates generate try's only and are NOT indexed.		*/
/*----------------------------------------------------------------------*/

makeindex(Sym, Clauses, Pil, Pilr, _Prag, Tabled) :-
	sym_name(Sym, _, Arity),
	Arity =:= 0,
	!,
	makeNOindex(Clauses, Arity, Pil, Pilr, Tabled).

/*----------------------------------------------------------------------*
 * all other cases                                                      *
 *----------------------------------------------------------------------*/

makeindex(Sym, Clauses, Pil, Pilr, Prag, Tabled) :-
	sym_name(Sym, _, Arity),
	(member(index(Narg), Prag), integer(Narg)
	 ->	true
	 ; var(Narg)
	 ->	Narg = 1		% Default index on arg 1
	 ;	warning(('Illegal index specification: ',Narg)),
		fail
	),
	( Narg =:= 0 ->
	    makeNOindex(Clauses, Arity, Pil, Pilr, Tabled)
	; index_argpats(Clauses, Narg, Tabled, ArgList),
	  index_split(ArgList, ArgListGroup),
	  ( ArgListGroup = [_One] -> NeedLabel = 0, Pil2 = Pil
          ; index_extract_label_list(ArgListGroup, LabelList),
	    index_gentrys(Arity, LabelList, Pil, Pil2, Tabled),
	    NeedLabel = 1
	  ),
	  index_groups(ArgListGroup, Narg, Pil2, Pilr,
		       NeedLabel, Arity, Tabled)
	).

makeNOindex(Clauses, Arity, Pil, Pilr, Tabled) :-
	index_label_list(Clauses, LabelList),
	index_gentrys(Arity, LabelList, Pil, Pilr, Tabled).

/*======================================================================*/
/* index_groups(+ArgListGroup, +ArgNum, -Pil, -PilRest, +NeedLabel,	*/
/*		+Arity, +Tabled)			                */
/*======================================================================*/

index_groups([], _, Pil, Pil, _, _, _).
index_groups([arglabel(T,Group,L)|Rest], Narg, Pil, Pilr,
	     NeedLabel, Arity, Tabled) :-
	( T = g ->
	   ( NeedLabel =:= 0 -> Pil = Pil1
	   ; gennum(N), L = (inde,N), Pil = [label(inde,L)|Pil1]
           ),
	   index_group(Group, Narg, Pil1, Pil2, Arity, Tabled)
	; Pil = Pil2
	),
	index_groups(Rest, Narg, Pil2, Pilr, NeedLabel, Arity, Tabled).

/*======================================================================*/
/* index_group(+Group, +Narg, -Pil, -PilRest, +Arity, +Tabled)  	*/
/*	Generates index and try/retry/trust instructions for a group.	*/
/*======================================================================*/

index_group([arglabel(T1,_,L1),arglabel(T2,_,L2)],
		Narg, Pil, Pilr, Arity, Tabled) :-
        ( index_sot_inst(T1,T2,Narg,L1,L2,Inst) -> Pil = [Inst|Pil1]
	; T1 = l, T2 = l -> Pil = Pil1
	),
	index_gentrys(Arity, [L1,L2], Pil1, Pilr, Tabled).
index_group(Group, Narg, Pil, Pilr, Arity, Tabled) :-
	length(Group, Size),
	gennum(N), L = (inde,N),
	Pil = [cases(L, Arity, Size, _)|Pil1],
	index_genarglabel(Group, Pil1, Pil2),
	Pil2 = [switchonbound(Narg, L, Size)|Pil3],
	index_extract_label_list(Group, LabelList),
	index_gentrys(Arity, LabelList, Pil3, Pilr, Tabled).

index_genarglabel([], Pil, Pil).
index_genarglabel([ArgLabel|Rest], [ArgLabel|Pil], Pilr) :-
	index_genarglabel(Rest, Pil, Pilr).

/*======================================================================*/
/* index_gentrys(+Arity, +LabelList, -Pil, -PilR, +Tabled)      	*/
/*======================================================================*/

index_gentrys(Arity, [Label|Rest], Pil, Pilr, Tabled) :-
	( Tabled == 0 -> Pil = [try(Arity,Label)|Pilm]
	; Tabled = tabled(Ind,_), Pil = [tabletry(Arity,Label,Ind)|Pilm]
	),
	index_genretrys(Arity, Rest, Pilm, Pilr, Tabled).

index_genretrys(Arity, [L1], Pil, Pilr, Tabled) :-
	( Tabled \== 0 -> Pil = [tabletrust(Arity,L1)|Pilr]
	;                 Pil = [trust(Arity,L1)|Pilr]
	).
index_genretrys(Arity, [L1|Rest], Pil, Pilr, Tabled) :-
	Rest = [_|_],
	( Tabled \== 0 -> Pil = [tableretry(Arity,L1)|Pilm]
	;                 Pil = [retry(Arity,L1)|Pilm]
	),
	index_genretrys(Arity, Rest, Pilm, Pilr, Tabled).

/*======================================================================*/
/* special switchonbound cases: two clauses, one is atomic type and the	*/
/* other is compound type.						*/
/*======================================================================*/

index_sot_inst(i,l,N,L1,L2,switchonterm(N,L1,L2)).
index_sot_inst(i,s,N,L1,L2,switchonterm(N,L1,L2)).
index_sot_inst(n,l,N,L1,L2,switchonterm(N,L1,L2)).
index_sot_inst(n,s,N,L1,L2,switchonterm(N,L1,L2)).
index_sot_inst(c,l,N,L1,L2,switchonterm(N,L1,L2)).
index_sot_inst(c,s,N,L1,L2,switchonterm(N,L1,L2)).
index_sot_inst(l,i,N,L1,L2,switchonterm(N,L2,L1)).
index_sot_inst(l,n,N,L1,L2,switchonterm(N,L2,L1)).
index_sot_inst(l,c,N,L1,L2,switchonterm(N,L2,L1)).
index_sot_inst(s,n,N,L1,L2,switchonterm(N,L2,L1)).
index_sot_inst(s,i,N,L1,L2,switchonterm(N,L2,L1)).
index_sot_inst(s,c,N,L1,L2,switchonterm(N,L2,L1)).

/*======================================================================*/
/* index_label_list(+Clauses, -LabelList)				*/
/*	Obtain the list of labels of all clauses. The labels are	*/
/*	generated (later) by compile_clause/5.				*/
/*======================================================================*/

index_label_list([],[]).
index_label_list([clause(_,_,crec(_,Label))|ClRest],[Label|LRest]) :-
	index_label_list(ClRest,LRest).

/*======================================================================*/
/* index_extract_label_list(+ArgPatterns, -LabelList)			*/
/*	Same as index_label_list/2 but the input is a list of arg	*/
/*	patterns.							*/
/*======================================================================*/

index_extract_label_list([], []).
index_extract_label_list([arglabel(_,_,Label)|Rest], [Label|LRest]) :-
	index_extract_label_list(Rest, LRest).

/*======================================================================*/
/* index_split(+ArgPatterns, +ArgGroups)				*/
/*	Split the argpatterns into a list of arg patterns (ArgGroup)	*/
/*	so that each group contains either a "v" arg or a list of	*/
/*	non-"v" (bound) args.						*/
/*	Precisely, ArgGroups is a list of arglabel(T,Value,Label) 	*/
/*	and when T = g, Value is again a list of arg patterns.		*/
/*======================================================================*/
/* index_split/2 is also used in cp_opt.P.  				*/
/*======================================================================*/

:- mode index_split(?,?).
index_split(ArgPatterns, ArgGroups) :-
	index_split0(ArgPatterns, ArgGroups0),
	index_simplify_singleton(ArgGroups0, ArgGroups).

index_split0([], []).
index_split0([One|Rest], [arglabel(g,Group,_L)|GRest]) :-
	( One = arglabel(v, _, _) -> Group = [One], NRest = Rest
	; index_split1([One|Rest], NRest, Group)
	),
	index_split0(NRest, GRest).

index_split1([], [], []).
index_split1([One|Rest], NRest, NA) :-
	( One = arglabel(v, _, _) -> NRest = [One|Rest], NA = []
	; NA = [One|NNA], index_split1(Rest, NRest, NNA)
	).

index_simplify_singleton([], []).
index_simplify_singleton([Group|Rest], [NGroup|NRest]) :-
	Group = arglabel(g, Patterns, _L),
	( Patterns = [One] -> NGroup = One ; NGroup = Group ),
	index_simplify_singleton(Rest, NRest).

/*======================================================================*/
/* index_argpats(+Clauses, +IndexArg, +Tabled, -ArgPatList)		*/
/*	IndexArg: the argument number to be indexed			*/
/*	ArgPatList: a list of arglabel(Pattern, HashVal, Label)		*/
/*	  where Label is the label for the clause			*/
/*		Pattern 	HashVal					*/
/*		n (nil symbol)	[]					*/
/*		i (integer)	an integer				*/
/*		c (constant)	a Symbol entry				*/
/*		v (variable)	v					*/
/*		l (".")		l					*/
/*		s (structure)	a Symbol entry				*/
/*======================================================================*/
/* code like index_argpats/3 is used in cp_opt/2.  Please make sure that*/
/* any major indexing changes which affect index_argpats/3 does not also*/
/* affect cpo_index_argpats/3						*/
/*======================================================================*/

index_argpats([], _, _, []).
index_argpats([clause(Arglist,Body,crec(_,L))|CRest], N, Tabled,
	      [arglabel(Pattern,NewArg,L)|APatternRest]) :-
    nthmember(Arg, Arglist, N),
    (Tabled == 0, % seems to work only for untabled preds, should figure out why
     Arg = varocc(VN),
     first_goal(Body,inlinegoal(=,2,[varocc(VN),Arg1]))
     ->	index_argpat(Arg1,Pattern,NewArg)
        %%,standard:writeln(index_thru_var(VN,Arg1))
     ;	index_argpat(Arg, Pattern, NewArg)
    ),
    index_argpats(CRest, N, Tabled, APatternRest).

first_goal(and(Goal,_),Goal) :- !.
first_goal(Goal,Goal).

index_argpat(constant(Sym), n, []) :- sym_name(Sym, [], 0).
index_argpat(integer(X), i, X).
%%index_argpat(real(X), f, X). %% dsw changed from i...
index_argpat(real(X), d, X). %% dsw changed from f for double.....
index_argpat(constant(C), c, C).
index_argpat(varocc(_), v, v).
index_argpat(string(_), l, l).
index_argpat(structure(Sym,_), l, l) :- sym_name(Sym, '.', 2).
index_argpat(structure(Sym,_), s, Sym).


force_tp_index_loaded.
