//
//  Copyright (c) 2018, Joyent, Inc. All rights reserved.
//
//  This Source Code Form is subject to the terms of the Mozilla Public
//  License, v. 2.0. If a copy of the MPL was not distributed with this
//  file, You can obtain one at http://mozilla.org/MPL/2.0/.
//

package md

import (
	"fmt"
	"os"
	"path"
	"path/filepath"
	"strings"
	"time"

	"github.com/joyent/triton-go/cmd/internal/command"
	"github.com/joyent/triton-go/cmd/internal/config"
	"github.com/pkg/errors"
	"github.com/rs/zerolog/log"
	"github.com/spf13/cobra"
	"github.com/spf13/cobra/doc"
	"github.com/spf13/viper"
)

const template = `---
date: %s
title: "%s"
slug: %s
url: %s
---
`

var Cmd = &command.Command{
	Cobra: &cobra.Command{
		Use:   "doc",
		Short: "Generates and installs triton cli documentation in markdown",
		Long: `Generate Markdown documentation for the Triton CLI.

It creates one Markdown file per command `,

		PreRunE: func(cmd *cobra.Command, args []string) error {
			return nil
		},
		RunE: func(cmd *cobra.Command, args []string) error {
			mdDir := viper.GetString(config.KeyDocMarkdownDir)

			if _, err := os.Stat(mdDir); os.IsNotExist(err) {
				if err := os.MkdirAll(mdDir, 0777); err != nil {
					return errors.Wrapf(err, "unable to make mddir %q", mdDir)
				}
			}

			log.Info().Str(config.KeyDocMarkdownDir, mdDir).Msg("Installing markdown documentation")

			now := time.Now().UTC().Format(time.RFC3339)
			prefix := viper.GetString(config.KeyDocMarkdownURLPrefix)
			prepender := func(filename string) string {
				name := filepath.Base(filename)
				base := strings.TrimSuffix(name, path.Ext(name))
				url := prefix + path.Join("/", strings.ToLower(base), "/")
				return fmt.Sprintf(template, now, strings.Replace(base, "_", " ", -1), base, url)
			}

			linkHandler := func(name string) string {
				base := strings.TrimSuffix(name, path.Ext(name))
				return prefix + path.Join("/", strings.ToLower(base), "/")
			}

			doc.GenMarkdownTreeCustom(cmd.Root(), mdDir, prepender, linkHandler)

			log.Info().Msg("Installation completed successfully.")

			return nil
		},
	},
	Setup: func(parent *command.Command) error {

		{
			const (
				key          = config.KeyDocMarkdownDir
				longName     = "dir"
				shortName    = "d"
				description  = "Specify the directory for generated Markdown files"
				defaultValue = config.DefaultMarkdownDir
			)

			flags := parent.Cobra.Flags()
			flags.StringP(longName, shortName, defaultValue, description)
			viper.BindPFlag(key, flags.Lookup(longName))
			viper.SetDefault(key, defaultValue)
		}

		{
			const (
				key          = config.KeyDocMarkdownURLPrefix
				longName     = "url-prefix"
				shortName    = "p"
				description  = "Specify the prefix for links generated by Markdown"
				defaultValue = config.DefaultMarkdownURLPrefix
			)

			flags := parent.Cobra.Flags()
			flags.String(longName, defaultValue, description)
			viper.BindPFlag(key, flags.Lookup(longName))
			viper.SetDefault(key, defaultValue)
		}

		return nil
	},
}
