require 'spec_helper'

describe "ChildLabor::Task" do
  it "runs the subprocess" do
    ChildLabor.subprocess("echo 'hello world' > #{tmp}/hello_world.txt") { }
    tmp("hello_world.txt").should exist
  end

  it "sets up STDOUT correctly" do
    out = nil
    ChildLabor.subprocess("echo 'hello world'") { |t| out = t.read }
    out.should == "hello world\n"
  end

  it "sets up STDIN correctly" do
    out = nil
    ChildLabor.subprocess("cat") do |t|
      t.write "hello world"
      t.close_write
      out = t.read
    end
    out.should == "hello world"
  end

  it "sets up STDERR correctly" do
    out = nil
    ChildLabor.subprocess("echo 'hello world' 1>&2") { |t| out = t.read_stderr }
    out.should == "hello world\n"
  end

  it "provides the exit status" do
    task = ChildLabor.subprocess("echo 'hi'") { }
    task.exit_status.should == 0
  end

  it "correctly returns that it is running" do
    t = ChildLabor.subprocess("sh -c 'read'")
    t.running?.should be_true
    t.write "hello\n"
    t.close_write
    sleep 0.5
    t.running?.should be_false
  end

  it "doesn't raise ECHILD when checking for running" do
    t = ChildLabor.subprocess("sh -c 'read'")
    t.running?.should be_true
    t.write "hello\n"
    t.close_write
    t.wait
    t.running?.should be_false
  end

  it "correctly returns that it is terminated" do
    t = ChildLabor.subprocess("sh -c 'read'")
    t.terminated?.should be_false
    t.write "hello\n"
    t.close_write
    sleep 0.5
    t.terminated?.should be_true
  end

  it "should not return true for terminated? if unlaunched" do
    t = ChildLabor::Task.new("echo hello")
    t.terminated?.should be_false
  end

  it "doesn't raise ECHILD when checking for terminated" do
    t = ChildLabor.subprocess("sh -c 'read'")
    t.terminated?.should be_false
    t.write "hello\n"
    t.close_write
    t.wait
    t.terminated?.should be_true
  end

  it "can send signals" do
    out = nil
    ChildLabor.subprocess(%[echo 'sync' ; trap "echo hello; exit" USR1 ; sleep 1]) do |t|
      t.read(5) # Read "sync\n"
      t.signal 'USR1'
      out = t.read
    end

    out.should == "hello\n"
  end
end
