// starcolors.cpp
//
// Copyright (C) 2004, Chris Laurel <claurel@shatters.net>
//
// Tables of star colors, indexed by temperature.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.

#include "starcolors.h"

#include <algorithm>
#include <array>
#include <iterator>
#include <memory>

#include <Eigen/Core>
#include <Eigen/Dense>

namespace
{

constexpr std::size_t BlackbodyTableEntries = 401;
constexpr float MaxTemperature = 40000.0f;
constexpr float TemperatureStep = MaxTemperature / static_cast<float>(BlackbodyTableEntries - 1);

// Temperature of the color table bucket containing the Sun
const float SolarTemperatureBucket = std::nearbyint(5772.0f / TemperatureStep) * TemperatureStep;
const float VegaTemperatureBucket = std::nearbyint(9602.0f / TemperatureStep) * TemperatureStep;

// D65 whitepoint and SRGB primary chromaticities
const Eigen::Vector3d D65_XYZ(0.95047, 1.00000, 1.08883);
const Eigen::Vector2d SRGB_R_xy(0.64, 0.33);
const Eigen::Vector2d SRGB_G_xy(0.30, 0.60);
const Eigen::Vector2d SRGB_B_xy(0.15, 0.06);


// Approximate colors used by older versions of Celestia
constexpr std::array<Color, 41> StarColors_Enhanced
{
    Color(0.00f, 0.00f, 0.00f), // T = 0K
    Color(0.75f, 0.20f, 0.20f), // T = 1000K
    Color(1.00f, 0.40f, 0.40f), // T = 2000K
    Color(1.00f, 0.70f, 0.70f), // T = 3000K
    Color(1.00f, 0.90f, 0.70f), // T = 4000K
    Color(1.00f, 1.00f, 0.75f), // T = 5000K
    Color(1.00f, 1.00f, 0.88f), // T = 6000K
    Color(1.00f, 1.00f, 0.95f), // T = 7000K
    Color(1.00f, 1.00f, 1.00f), // T = 8000K
    Color(0.95f, 0.98f, 1.00f), // T = 9000K
    Color(0.90f, 0.95f, 1.00f), // T = 10000K
    Color(0.85f, 0.93f, 1.00f), // T = 11000K
    Color(0.80f, 0.90f, 1.00f), // T = 12000K
    Color(0.79f, 0.89f, 1.00f), // T = 13000K
    Color(0.78f, 0.88f, 1.00f), // T = 14000K
    Color(0.77f, 0.87f, 1.00f), // T = 15000K
    Color(0.76f, 0.86f, 1.00f), // T = 16000K
    Color(0.75f, 0.85f, 1.00f), // T = 17000K
    Color(0.74f, 0.84f, 1.00f), // T = 18000K
    Color(0.73f, 0.83f, 1.00f), // T = 19000K
    Color(0.72f, 0.82f, 1.00f), // T = 20000K
    Color(0.71f, 0.81f, 1.00f), // T = 21000K
    Color(0.70f, 0.80f, 1.00f), // T = 22000K
    Color(0.69f, 0.79f, 1.00f), // T = 23000K
    Color(0.68f, 0.78f, 1.00f), // T = 24000K
    Color(0.67f, 0.77f, 1.00f), // T = 25000K
    Color(0.66f, 0.76f, 1.00f), // T = 26000K
    Color(0.65f, 0.75f, 1.00f), // T = 27000K
    Color(0.65f, 0.75f, 1.00f), // T = 28000K
    Color(0.64f, 0.74f, 1.00f), // T = 29000K
    Color(0.64f, 0.74f, 1.00f), // T = 30000K
    Color(0.63f, 0.73f, 1.00f), // T = 31000K
    Color(0.63f, 0.73f, 1.00f), // T = 32000K
    Color(0.62f, 0.72f, 1.00f), // T = 33000K
    Color(0.62f, 0.72f, 1.00f), // T = 34000K
    Color(0.61f, 0.71f, 1.00f), // T = 35000K
    Color(0.61f, 0.71f, 1.00f), // T = 36000K
    Color(0.60f, 0.70f, 1.00f), // T = 37000K
    Color(0.60f, 0.70f, 1.00f), // T = 38000K
    Color(0.60f, 0.70f, 1.00f), // T = 39000K
    Color(0.60f, 0.70f, 1.00f), // T = 40000K
};


struct CIEPoint
{
    int wavelength;
    std::array<float, 3> xyz;
};


// CIE 2018, CIE 1931 color matching functions, 2 degree observer (data table),
// International Commission on Illumination (CIE), Vienna, Austria
// DOI:10.25039/CIE.DS.xvudnb9b
constexpr std::array<CIEPoint, 471> CIEFunctions
{
    CIEPoint { 360, { 0.000129900000f, 0.0000039170000f, 0.000606100000f } },
    CIEPoint { 361, { 0.000145847000f, 0.0000043935810f, 0.000680879200f } },
    CIEPoint { 362, { 0.000163802100f, 0.0000049296040f, 0.000765145600f } },
    CIEPoint { 363, { 0.000184003700f, 0.0000055321360f, 0.000860012400f } },
    CIEPoint { 364, { 0.000206690200f, 0.0000062082450f, 0.000966592800f } },
    CIEPoint { 365, { 0.000232100000f, 0.0000069650000f, 0.001086000000f } },
    CIEPoint { 366, { 0.000260728000f, 0.0000078132190f, 0.001220586000f } },
    CIEPoint { 367, { 0.000293075000f, 0.0000087673360f, 0.001372729000f } },
    CIEPoint { 368, { 0.000329388000f, 0.0000098398440f, 0.001543579000f } },
    CIEPoint { 369, { 0.000369914000f, 0.0000110432300f, 0.001734286000f } },
    CIEPoint { 370, { 0.000414900000f, 0.0000123900000f, 0.001946000000f } },
    CIEPoint { 371, { 0.000464158700f, 0.0000138864100f, 0.002177777000f } },
    CIEPoint { 372, { 0.000518986000f, 0.0000155572800f, 0.002435809000f } },
    CIEPoint { 373, { 0.000581854000f, 0.0000174429600f, 0.002731953000f } },
    CIEPoint { 374, { 0.000655234700f, 0.0000195837500f, 0.003078064000f } },
    CIEPoint { 375, { 0.000741600000f, 0.0000220200000f, 0.003486000000f } },
    CIEPoint { 376, { 0.000845029600f, 0.0000248396500f, 0.003975227000f } },
    CIEPoint { 377, { 0.000964526800f, 0.0000280412600f, 0.004540880000f } },
    CIEPoint { 378, { 0.001094949000f, 0.0000315310400f, 0.005158320000f } },
    CIEPoint { 379, { 0.001231154000f, 0.0000352152100f, 0.005802907000f } },
    CIEPoint { 380, { 0.001368000000f, 0.0000390000000f, 0.006450001000f } },
    CIEPoint { 381, { 0.001502050000f, 0.0000428264000f, 0.007083216000f } },
    CIEPoint { 382, { 0.001642328000f, 0.0000469146000f, 0.007745488000f } },
    CIEPoint { 383, { 0.001802382000f, 0.0000515896000f, 0.008501152000f } },
    CIEPoint { 384, { 0.001995757000f, 0.0000571764000f, 0.009414544000f } },
    CIEPoint { 385, { 0.002236000000f, 0.0000640000000f, 0.010549990000f } },
    CIEPoint { 386, { 0.002535385000f, 0.0000723442100f, 0.011965800000f } },
    CIEPoint { 387, { 0.002892603000f, 0.0000822122400f, 0.013655870000f } },
    CIEPoint { 388, { 0.003300829000f, 0.0000935081600f, 0.015588050000f } },
    CIEPoint { 389, { 0.003753236000f, 0.0001061361000f, 0.017730150000f } },
    CIEPoint { 390, { 0.004243000000f, 0.0001200000000f, 0.020050010000f } },
    CIEPoint { 391, { 0.004762389000f, 0.0001349840000f, 0.022511360000f } },
    CIEPoint { 392, { 0.005330048000f, 0.0001514920000f, 0.025202880000f } },
    CIEPoint { 393, { 0.005978712000f, 0.0001702080000f, 0.028279720000f } },
    CIEPoint { 394, { 0.006741117000f, 0.0001918160000f, 0.031897040000f } },
    CIEPoint { 395, { 0.007650000000f, 0.0002170000000f, 0.036210000000f } },
    CIEPoint { 396, { 0.008751373000f, 0.0002469067000f, 0.041437710000f } },
    CIEPoint { 397, { 0.010028880000f, 0.0002812400000f, 0.047503720000f } },
    CIEPoint { 398, { 0.011421700000f, 0.0003185200000f, 0.054119880000f } },
    CIEPoint { 399, { 0.012869010000f, 0.0003572667000f, 0.060998030000f } },
    CIEPoint { 400, { 0.014310000000f, 0.0003960000000f, 0.067850010000f } },
    CIEPoint { 401, { 0.015704430000f, 0.0004337147000f, 0.074486320000f } },
    CIEPoint { 402, { 0.017147440000f, 0.0004730240000f, 0.081361560000f } },
    CIEPoint { 403, { 0.018781220000f, 0.0005178760000f, 0.089153640000f } },
    CIEPoint { 404, { 0.020748010000f, 0.0005722187000f, 0.098540480000f } },
    CIEPoint { 405, { 0.023190000000f, 0.0006400000000f, 0.110200000000f } },
    CIEPoint { 406, { 0.026207360000f, 0.0007245600000f, 0.124613300000f } },
    CIEPoint { 407, { 0.029782480000f, 0.0008255000000f, 0.141701700000f } },
    CIEPoint { 408, { 0.033880920000f, 0.0009411600000f, 0.161303500000f } },
    CIEPoint { 409, { 0.038468240000f, 0.0010698800000f, 0.183256800000f } },
    CIEPoint { 410, { 0.043510000000f, 0.0012100000000f, 0.207400000000f } },
    CIEPoint { 411, { 0.048995600000f, 0.0013620910000f, 0.233692100000f } },
    CIEPoint { 412, { 0.055022600000f, 0.0015307520000f, 0.262611400000f } },
    CIEPoint { 413, { 0.061718800000f, 0.0017203680000f, 0.294774600000f } },
    CIEPoint { 414, { 0.069212000000f, 0.0019353230000f, 0.330798500000f } },
    CIEPoint { 415, { 0.077630000000f, 0.0021800000000f, 0.371300000000f } },
    CIEPoint { 416, { 0.086958110000f, 0.0024548000000f, 0.416209100000f } },
    CIEPoint { 417, { 0.097176720000f, 0.0027640000000f, 0.465464200000f } },
    CIEPoint { 418, { 0.108406300000f, 0.0031178000000f, 0.519694800000f } },
    CIEPoint { 419, { 0.120767200000f, 0.0035264000000f, 0.579530300000f } },
    CIEPoint { 420, { 0.134380000000f, 0.0040000000000f, 0.645600000000f } },
    CIEPoint { 421, { 0.149358200000f, 0.0045462400000f, 0.718483800000f } },
    CIEPoint { 422, { 0.165395700000f, 0.0051593200000f, 0.796713300000f } },
    CIEPoint { 423, { 0.181983100000f, 0.0058292800000f, 0.877845900000f } },
    CIEPoint { 424, { 0.198611000000f, 0.0065461600000f, 0.959439000000f } },
    CIEPoint { 425, { 0.214770000000f, 0.0073000000000f, 1.039050100000f } },
    CIEPoint { 426, { 0.230186800000f, 0.0080865070000f, 1.115367300000f } },
    CIEPoint { 427, { 0.244879700000f, 0.0089087200000f, 1.188497100000f } },
    CIEPoint { 428, { 0.258777300000f, 0.0097676800000f, 1.258123300000f } },
    CIEPoint { 429, { 0.271807900000f, 0.0106644300000f, 1.323929600000f } },
    CIEPoint { 430, { 0.283900000000f, 0.0116000000000f, 1.385600000000f } },
    CIEPoint { 431, { 0.294943800000f, 0.0125731700000f, 1.442635200000f } },
    CIEPoint { 432, { 0.304896500000f, 0.0135827200000f, 1.494803500000f } },
    CIEPoint { 433, { 0.313787300000f, 0.0146296800000f, 1.542190300000f } },
    CIEPoint { 434, { 0.321645400000f, 0.0157150900000f, 1.584880700000f } },
    CIEPoint { 435, { 0.328500000000f, 0.0168400000000f, 1.622960000000f } },
    CIEPoint { 436, { 0.334351300000f, 0.0180073600000f, 1.656404800000f } },
    CIEPoint { 437, { 0.339210100000f, 0.0192144800000f, 1.685295900000f } },
    CIEPoint { 438, { 0.343121300000f, 0.0204539200000f, 1.709874500000f } },
    CIEPoint { 439, { 0.346129600000f, 0.0217182400000f, 1.730382100000f } },
    CIEPoint { 440, { 0.348280000000f, 0.0230000000000f, 1.747060000000f } },
    CIEPoint { 441, { 0.349599900000f, 0.0242946100000f, 1.760044600000f } },
    CIEPoint { 442, { 0.350147400000f, 0.0256102400000f, 1.769623300000f } },
    CIEPoint { 443, { 0.350013000000f, 0.0269585700000f, 1.776263700000f } },
    CIEPoint { 444, { 0.349287000000f, 0.0283512500000f, 1.780433400000f } },
    CIEPoint { 445, { 0.348060000000f, 0.0298000000000f, 1.782600000000f } },
    CIEPoint { 446, { 0.346373300000f, 0.0313108300000f, 1.782968200000f } },
    CIEPoint { 447, { 0.344262400000f, 0.0328836800000f, 1.781699800000f } },
    CIEPoint { 448, { 0.341808800000f, 0.0345211200000f, 1.779198200000f } },
    CIEPoint { 449, { 0.339094100000f, 0.0362257100000f, 1.775867100000f } },
    CIEPoint { 450, { 0.336200000000f, 0.0380000000000f, 1.772110000000f } },
    CIEPoint { 451, { 0.333197700000f, 0.0398466700000f, 1.768258900000f } },
    CIEPoint { 452, { 0.330041100000f, 0.0417680000000f, 1.764039000000f } },
    CIEPoint { 453, { 0.326635700000f, 0.0437660000000f, 1.758943800000f } },
    CIEPoint { 454, { 0.322886800000f, 0.0458426700000f, 1.752466300000f } },
    CIEPoint { 455, { 0.318700000000f, 0.0480000000000f, 1.744100000000f } },
    CIEPoint { 456, { 0.314025100000f, 0.0502436800000f, 1.733559500000f } },
    CIEPoint { 457, { 0.308884000000f, 0.0525730400000f, 1.720858100000f } },
    CIEPoint { 458, { 0.303290400000f, 0.0549805600000f, 1.705936900000f } },
    CIEPoint { 459, { 0.297257900000f, 0.0574587200000f, 1.688737200000f } },
    CIEPoint { 460, { 0.290800000000f, 0.0600000000000f, 1.669200000000f } },
    CIEPoint { 461, { 0.283970100000f, 0.0626019700000f, 1.647528700000f } },
    CIEPoint { 462, { 0.276721400000f, 0.0652775200000f, 1.623412700000f } },
    CIEPoint { 463, { 0.268917800000f, 0.0680420800000f, 1.596022300000f } },
    CIEPoint { 464, { 0.260422700000f, 0.0709110900000f, 1.564528000000f } },
    CIEPoint { 465, { 0.251100000000f, 0.0739000000000f, 1.528100000000f } },
    CIEPoint { 466, { 0.240847500000f, 0.0770160000000f, 1.486111400000f } },
    CIEPoint { 467, { 0.229851200000f, 0.0802664000000f, 1.439521500000f } },
    CIEPoint { 468, { 0.218407200000f, 0.0836668000000f, 1.389879900000f } },
    CIEPoint { 469, { 0.206811500000f, 0.0872328000000f, 1.338736200000f } },
    CIEPoint { 470, { 0.195360000000f, 0.0909800000000f, 1.287640000000f } },
    CIEPoint { 471, { 0.184213600000f, 0.0949175500000f, 1.237422300000f } },
    CIEPoint { 472, { 0.173327300000f, 0.0990458400000f, 1.187824300000f } },
    CIEPoint { 473, { 0.162688100000f, 0.1033674000000f, 1.138761100000f } },
    CIEPoint { 474, { 0.152283300000f, 0.1078846000000f, 1.090148000000f } },
    CIEPoint { 475, { 0.142100000000f, 0.1126000000000f, 1.041900000000f } },
    CIEPoint { 476, { 0.132178600000f, 0.1175320000000f, 0.994197600000f } },
    CIEPoint { 477, { 0.122569600000f, 0.1226744000000f, 0.947347300000f } },
    CIEPoint { 478, { 0.113275200000f, 0.1279928000000f, 0.901453100000f } },
    CIEPoint { 479, { 0.104297900000f, 0.1334528000000f, 0.856619300000f } },
    CIEPoint { 480, { 0.095640000000f, 0.1390200000000f, 0.812950100000f } },
    CIEPoint { 481, { 0.087299550000f, 0.1446764000000f, 0.770517300000f } },
    CIEPoint { 482, { 0.079308040000f, 0.1504693000000f, 0.729444800000f } },
    CIEPoint { 483, { 0.071717760000f, 0.1564619000000f, 0.689913600000f } },
    CIEPoint { 484, { 0.064580990000f, 0.1627177000000f, 0.652104900000f } },
    CIEPoint { 485, { 0.057950010000f, 0.1693000000000f, 0.616200000000f } },
    CIEPoint { 486, { 0.051862110000f, 0.1762431000000f, 0.582328600000f } },
    CIEPoint { 487, { 0.046281520000f, 0.1835581000000f, 0.550416200000f } },
    CIEPoint { 488, { 0.041150880000f, 0.1912735000000f, 0.520337600000f } },
    CIEPoint { 489, { 0.036412830000f, 0.1994180000000f, 0.491967300000f } },
    CIEPoint { 490, { 0.032010000000f, 0.2080200000000f, 0.465180000000f } },
    CIEPoint { 491, { 0.027917200000f, 0.2171199000000f, 0.439924600000f } },
    CIEPoint { 492, { 0.024144400000f, 0.2267345000000f, 0.416183600000f } },
    CIEPoint { 493, { 0.020687000000f, 0.2368571000000f, 0.393882200000f } },
    CIEPoint { 494, { 0.017540400000f, 0.2474812000000f, 0.372945900000f } },
    CIEPoint { 495, { 0.014700000000f, 0.2586000000000f, 0.353300000000f } },
    CIEPoint { 496, { 0.012161790000f, 0.2701849000000f, 0.334857800000f } },
    CIEPoint { 497, { 0.009919960000f, 0.2822939000000f, 0.317552100000f } },
    CIEPoint { 498, { 0.007967240000f, 0.2950505000000f, 0.301337500000f } },
    CIEPoint { 499, { 0.006296346000f, 0.3085780000000f, 0.286168600000f } },
    CIEPoint { 500, { 0.004900000000f, 0.3230000000000f, 0.272000000000f } },
    CIEPoint { 501, { 0.003777173000f, 0.3384021000000f, 0.258817100000f } },
    CIEPoint { 502, { 0.002945320000f, 0.3546858000000f, 0.246483800000f } },
    CIEPoint { 503, { 0.002424880000f, 0.3716986000000f, 0.234771800000f } },
    CIEPoint { 504, { 0.002236293000f, 0.3892875000000f, 0.223453300000f } },
    CIEPoint { 505, { 0.002400000000f, 0.4073000000000f, 0.212300000000f } },
    CIEPoint { 506, { 0.002925520000f, 0.4256299000000f, 0.201169200000f } },
    CIEPoint { 507, { 0.003836560000f, 0.4443096000000f, 0.190119600000f } },
    CIEPoint { 508, { 0.005174840000f, 0.4633944000000f, 0.179225400000f } },
    CIEPoint { 509, { 0.006982080000f, 0.4829395000000f, 0.168560800000f } },
    CIEPoint { 510, { 0.009300000000f, 0.5030000000000f, 0.158200000000f } },
    CIEPoint { 511, { 0.012149490000f, 0.5235693000000f, 0.148138300000f } },
    CIEPoint { 512, { 0.015535880000f, 0.5445120000000f, 0.138375800000f } },
    CIEPoint { 513, { 0.019477520000f, 0.5656900000000f, 0.128994200000f } },
    CIEPoint { 514, { 0.023992770000f, 0.5869653000000f, 0.120075100000f } },
    CIEPoint { 515, { 0.029100000000f, 0.6082000000000f, 0.111700000000f } },
    CIEPoint { 516, { 0.034814850000f, 0.6293456000000f, 0.103904800000f } },
    CIEPoint { 517, { 0.041120160000f, 0.6503068000000f, 0.096667480000f } },
    CIEPoint { 518, { 0.047985040000f, 0.6708752000000f, 0.089982720000f } },
    CIEPoint { 519, { 0.055378610000f, 0.6908424000000f, 0.083845310000f } },
    CIEPoint { 520, { 0.063270000000f, 0.7100000000000f, 0.078249990000f } },
    CIEPoint { 521, { 0.071635010000f, 0.7281852000000f, 0.073208990000f } },
    CIEPoint { 522, { 0.080462240000f, 0.7454636000000f, 0.068678160000f } },
    CIEPoint { 523, { 0.089739960000f, 0.7619694000000f, 0.064567840000f } },
    CIEPoint { 524, { 0.099456450000f, 0.7778368000000f, 0.060788350000f } },
    CIEPoint { 525, { 0.109600000000f, 0.7932000000000f, 0.057250010000f } },
    CIEPoint { 526, { 0.120167400000f, 0.8081104000000f, 0.053904350000f } },
    CIEPoint { 527, { 0.131114500000f, 0.8224962000000f, 0.050746640000f } },
    CIEPoint { 528, { 0.142367900000f, 0.8363068000000f, 0.047752760000f } },
    CIEPoint { 529, { 0.153854200000f, 0.8494916000000f, 0.044898590000f } },
    CIEPoint { 530, { 0.165500000000f, 0.8620000000000f, 0.042160000000f } },
    CIEPoint { 531, { 0.177257100000f, 0.8738108000000f, 0.039507280000f } },
    CIEPoint { 532, { 0.189140000000f, 0.8849624000000f, 0.036935640000f } },
    CIEPoint { 533, { 0.201169400000f, 0.8954936000000f, 0.034458360000f } },
    CIEPoint { 534, { 0.213365800000f, 0.9054432000000f, 0.032088720000f } },
    CIEPoint { 535, { 0.225749900000f, 0.9148501000000f, 0.029840000000f } },
    CIEPoint { 536, { 0.238320900000f, 0.9237348000000f, 0.027711810000f } },
    CIEPoint { 537, { 0.251066800000f, 0.9320924000000f, 0.025694440000f } },
    CIEPoint { 538, { 0.263992200000f, 0.9399226000000f, 0.023787160000f } },
    CIEPoint { 539, { 0.277101700000f, 0.9472252000000f, 0.021989250000f } },
    CIEPoint { 540, { 0.290400000000f, 0.9540000000000f, 0.020300000000f } },
    CIEPoint { 541, { 0.303891200000f, 0.9602561000000f, 0.018718050000f } },
    CIEPoint { 542, { 0.317572600000f, 0.9660074000000f, 0.017240360000f } },
    CIEPoint { 543, { 0.331438400000f, 0.9712606000000f, 0.015863640000f } },
    CIEPoint { 544, { 0.345482800000f, 0.9760225000000f, 0.014584610000f } },
    CIEPoint { 545, { 0.359700000000f, 0.9803000000000f, 0.013400000000f } },
    CIEPoint { 546, { 0.374083900000f, 0.9840924000000f, 0.012307230000f } },
    CIEPoint { 547, { 0.388639600000f, 0.9874182000000f, 0.011301880000f } },
    CIEPoint { 548, { 0.403378400000f, 0.9903128000000f, 0.010377920000f } },
    CIEPoint { 549, { 0.418311500000f, 0.9928116000000f, 0.009529306000f } },
    CIEPoint { 550, { 0.433449900000f, 0.9949501000000f, 0.008749999000f } },
    CIEPoint { 551, { 0.448795300000f, 0.9967108000000f, 0.008035200000f } },
    CIEPoint { 552, { 0.464336000000f, 0.9980983000000f, 0.007381600000f } },
    CIEPoint { 553, { 0.480064000000f, 0.9991120000000f, 0.006785400000f } },
    CIEPoint { 554, { 0.495971300000f, 0.9997482000000f, 0.006242800000f } },
    CIEPoint { 555, { 0.512050100000f, 1.0000000000000f, 0.005749999000f } },
    CIEPoint { 556, { 0.528295900000f, 0.9998567000000f, 0.005303600000f } },
    CIEPoint { 557, { 0.544691600000f, 0.9993046000000f, 0.004899800000f } },
    CIEPoint { 558, { 0.561209400000f, 0.9983255000000f, 0.004534200000f } },
    CIEPoint { 559, { 0.577821500000f, 0.9968987000000f, 0.004202400000f } },
    CIEPoint { 560, { 0.594500000000f, 0.9950000000000f, 0.003900000000f } },
    CIEPoint { 561, { 0.611220900000f, 0.9926005000000f, 0.003623200000f } },
    CIEPoint { 562, { 0.627975800000f, 0.9897426000000f, 0.003370600000f } },
    CIEPoint { 563, { 0.644760200000f, 0.9864444000000f, 0.003141400000f } },
    CIEPoint { 564, { 0.661569700000f, 0.9827241000000f, 0.002934800000f } },
    CIEPoint { 565, { 0.678400000000f, 0.9786000000000f, 0.002749999000f } },
    CIEPoint { 566, { 0.695239200000f, 0.9740837000000f, 0.002585200000f } },
    CIEPoint { 567, { 0.712058600000f, 0.9691712000000f, 0.002438600000f } },
    CIEPoint { 568, { 0.728828400000f, 0.9638568000000f, 0.002309400000f } },
    CIEPoint { 569, { 0.745518800000f, 0.9581349000000f, 0.002196800000f } },
    CIEPoint { 570, { 0.762100000000f, 0.9520000000000f, 0.002100000000f } },
    CIEPoint { 571, { 0.778543200000f, 0.9454504000000f, 0.002017733000f } },
    CIEPoint { 572, { 0.794825600000f, 0.9384992000000f, 0.001948200000f } },
    CIEPoint { 573, { 0.810926400000f, 0.9311628000000f, 0.001889800000f } },
    CIEPoint { 574, { 0.826824800000f, 0.9234576000000f, 0.001840933000f } },
    CIEPoint { 575, { 0.842500000000f, 0.9154000000000f, 0.001800000000f } },
    CIEPoint { 576, { 0.857932500000f, 0.9070064000000f, 0.001766267000f } },
    CIEPoint { 577, { 0.873081600000f, 0.8982772000000f, 0.001737800000f } },
    CIEPoint { 578, { 0.887894400000f, 0.8892048000000f, 0.001711200000f } },
    CIEPoint { 579, { 0.902318100000f, 0.8797816000000f, 0.001683067000f } },
    CIEPoint { 580, { 0.916300000000f, 0.8700000000000f, 0.001650001000f } },
    CIEPoint { 581, { 0.929799500000f, 0.8598613000000f, 0.001610133000f } },
    CIEPoint { 582, { 0.942798400000f, 0.8493920000000f, 0.001564400000f } },
    CIEPoint { 583, { 0.955277600000f, 0.8386220000000f, 0.001513600000f } },
    CIEPoint { 584, { 0.967217900000f, 0.8275813000000f, 0.001458533000f } },
    CIEPoint { 585, { 0.978600000000f, 0.8163000000000f, 0.001400000000f } },
    CIEPoint { 586, { 0.989385600000f, 0.8047947000000f, 0.001336667000f } },
    CIEPoint { 587, { 0.999548800000f, 0.7930820000000f, 0.001270000000f } },
    CIEPoint { 588, { 1.009089200000f, 0.7811920000000f, 0.001205000000f } },
    CIEPoint { 589, { 1.018006400000f, 0.7691547000000f, 0.001146667000f } },
    CIEPoint { 590, { 1.026300000000f, 0.7570000000000f, 0.001100000000f } },
    CIEPoint { 591, { 1.033982700000f, 0.7447541000000f, 0.001068800000f } },
    CIEPoint { 592, { 1.040986000000f, 0.7324224000000f, 0.001049400000f } },
    CIEPoint { 593, { 1.047188000000f, 0.7200036000000f, 0.001035600000f } },
    CIEPoint { 594, { 1.052466700000f, 0.7074965000000f, 0.001021200000f } },
    CIEPoint { 595, { 1.056700000000f, 0.6949000000000f, 0.001000000000f } },
    CIEPoint { 596, { 1.059794400000f, 0.6822192000000f, 0.000968640000f } },
    CIEPoint { 597, { 1.061799200000f, 0.6694716000000f, 0.000929920000f } },
    CIEPoint { 598, { 1.062806800000f, 0.6566744000000f, 0.000886880000f } },
    CIEPoint { 599, { 1.062909600000f, 0.6438448000000f, 0.000842560000f } },
    CIEPoint { 600, { 1.062200000000f, 0.6310000000000f, 0.000800000000f } },
    CIEPoint { 601, { 1.060735200000f, 0.6181555000000f, 0.000760960000f } },
    CIEPoint { 602, { 1.058443600000f, 0.6053144000000f, 0.000723680000f } },
    CIEPoint { 603, { 1.055224400000f, 0.5924756000000f, 0.000685920000f } },
    CIEPoint { 604, { 1.050976800000f, 0.5796379000000f, 0.000645440000f } },
    CIEPoint { 605, { 1.045600000000f, 0.5668000000000f, 0.000600000000f } },
    CIEPoint { 606, { 1.039036900000f, 0.5539611000000f, 0.000547866700f } },
    CIEPoint { 607, { 1.031360800000f, 0.5411372000000f, 0.000491600000f } },
    CIEPoint { 608, { 1.022666200000f, 0.5283528000000f, 0.000435400000f } },
    CIEPoint { 609, { 1.013047700000f, 0.5156323000000f, 0.000383466700f } },
    CIEPoint { 610, { 1.002600000000f, 0.5030000000000f, 0.000340000000f } },
    CIEPoint { 611, { 0.991367500000f, 0.4904688000000f, 0.000307253300f } },
    CIEPoint { 612, { 0.979331400000f, 0.4780304000000f, 0.000283160000f } },
    CIEPoint { 613, { 0.966491600000f, 0.4656776000000f, 0.000265440000f } },
    CIEPoint { 614, { 0.952847900000f, 0.4534032000000f, 0.000251813300f } },
    CIEPoint { 615, { 0.938400000000f, 0.4412000000000f, 0.000240000000f } },
    CIEPoint { 616, { 0.923194000000f, 0.4290800000000f, 0.000229546700f } },
    CIEPoint { 617, { 0.907244000000f, 0.4170360000000f, 0.000220640000f } },
    CIEPoint { 618, { 0.890502000000f, 0.4050320000000f, 0.000211960000f } },
    CIEPoint { 619, { 0.872920000000f, 0.3930320000000f, 0.000202186700f } },
    CIEPoint { 620, { 0.854449900000f, 0.3810000000000f, 0.000190000000f } },
    CIEPoint { 621, { 0.835084000000f, 0.3689184000000f, 0.000174213300f } },
    CIEPoint { 622, { 0.814946000000f, 0.3568272000000f, 0.000155640000f } },
    CIEPoint { 623, { 0.794186000000f, 0.3447768000000f, 0.000135960000f } },
    CIEPoint { 624, { 0.772954000000f, 0.3328176000000f, 0.000116853300f } },
    CIEPoint { 625, { 0.751400000000f, 0.3210000000000f, 0.000100000000f } },
    CIEPoint { 626, { 0.729583600000f, 0.3093381000000f, 0.000086133330f } },
    CIEPoint { 627, { 0.707588800000f, 0.2978504000000f, 0.000074600000f } },
    CIEPoint { 628, { 0.685602200000f, 0.2865936000000f, 0.000065000000f } },
    CIEPoint { 629, { 0.663810400000f, 0.2756245000000f, 0.000056933330f } },
    CIEPoint { 630, { 0.642400000000f, 0.2650000000000f, 0.000049999990f } },
    CIEPoint { 631, { 0.621514900000f, 0.2547632000000f, 0.000044160000f } },
    CIEPoint { 632, { 0.601113800000f, 0.2448896000000f, 0.000039480000f } },
    CIEPoint { 633, { 0.581105200000f, 0.2353344000000f, 0.000035720000f } },
    CIEPoint { 634, { 0.561397700000f, 0.2260528000000f, 0.000032640000f } },
    CIEPoint { 635, { 0.541900000000f, 0.2170000000000f, 0.000030000000f } },
    CIEPoint { 636, { 0.522599500000f, 0.2081616000000f, 0.000027653330f } },
    CIEPoint { 637, { 0.503546400000f, 0.1995488000000f, 0.000025560000f } },
    CIEPoint { 638, { 0.484743600000f, 0.1911552000000f, 0.000023640000f } },
    CIEPoint { 639, { 0.466193900000f, 0.1829744000000f, 0.000021813330f } },
    CIEPoint { 640, { 0.447900000000f, 0.1750000000000f, 0.000020000000f } },
    CIEPoint { 641, { 0.429861300000f, 0.1672235000000f, 0.000018133330f } },
    CIEPoint { 642, { 0.412098000000f, 0.1596464000000f, 0.000016200000f } },
    CIEPoint { 643, { 0.394644000000f, 0.1522776000000f, 0.000014200000f } },
    CIEPoint { 644, { 0.377533300000f, 0.1451259000000f, 0.000012133330f } },
    CIEPoint { 645, { 0.360800000000f, 0.1382000000000f, 0.000010000000f } },
    CIEPoint { 646, { 0.344456300000f, 0.1315003000000f, 0.000007733333f } },
    CIEPoint { 647, { 0.328516800000f, 0.1250248000000f, 0.000005400000f } },
    CIEPoint { 648, { 0.313019200000f, 0.1187792000000f, 0.000003200000f } },
    CIEPoint { 649, { 0.298001100000f, 0.1127691000000f, 0.000001333333f } },
    CIEPoint { 650, { 0.283500000000f, 0.1070000000000f, 0.000000000000f } },
    CIEPoint { 651, { 0.269544800000f, 0.1014762000000f, 0.000000000000f } },
    CIEPoint { 652, { 0.256118400000f, 0.0961886400000f, 0.000000000000f } },
    CIEPoint { 653, { 0.243189600000f, 0.0911229600000f, 0.000000000000f } },
    CIEPoint { 654, { 0.230727200000f, 0.0862648500000f, 0.000000000000f } },
    CIEPoint { 655, { 0.218700000000f, 0.0816000000000f, 0.000000000000f } },
    CIEPoint { 656, { 0.207097100000f, 0.0771206400000f, 0.000000000000f } },
    CIEPoint { 657, { 0.195923200000f, 0.0728255200000f, 0.000000000000f } },
    CIEPoint { 658, { 0.185170800000f, 0.0687100800000f, 0.000000000000f } },
    CIEPoint { 659, { 0.174832300000f, 0.0647697600000f, 0.000000000000f } },
    CIEPoint { 660, { 0.164900000000f, 0.0610000000000f, 0.000000000000f } },
    CIEPoint { 661, { 0.155366700000f, 0.0573962100000f, 0.000000000000f } },
    CIEPoint { 662, { 0.146230000000f, 0.0539550400000f, 0.000000000000f } },
    CIEPoint { 663, { 0.137490000000f, 0.0506737600000f, 0.000000000000f } },
    CIEPoint { 664, { 0.129146700000f, 0.0475496500000f, 0.000000000000f } },
    CIEPoint { 665, { 0.121200000000f, 0.0445800000000f, 0.000000000000f } },
    CIEPoint { 666, { 0.113639700000f, 0.0417587200000f, 0.000000000000f } },
    CIEPoint { 667, { 0.106465000000f, 0.0390849600000f, 0.000000000000f } },
    CIEPoint { 668, { 0.099690440000f, 0.0365638400000f, 0.000000000000f } },
    CIEPoint { 669, { 0.093330610000f, 0.0342004800000f, 0.000000000000f } },
    CIEPoint { 670, { 0.087400000000f, 0.0320000000000f, 0.000000000000f } },
    CIEPoint { 671, { 0.081900960000f, 0.0299626100000f, 0.000000000000f } },
    CIEPoint { 672, { 0.076804280000f, 0.0280766400000f, 0.000000000000f } },
    CIEPoint { 673, { 0.072077120000f, 0.0263293600000f, 0.000000000000f } },
    CIEPoint { 674, { 0.067686640000f, 0.0247080500000f, 0.000000000000f } },
    CIEPoint { 675, { 0.063600000000f, 0.0232000000000f, 0.000000000000f } },
    CIEPoint { 676, { 0.059806850000f, 0.0218007700000f, 0.000000000000f } },
    CIEPoint { 677, { 0.056282160000f, 0.0205011200000f, 0.000000000000f } },
    CIEPoint { 678, { 0.052971040000f, 0.0192810800000f, 0.000000000000f } },
    CIEPoint { 679, { 0.049818610000f, 0.0181206900000f, 0.000000000000f } },
    CIEPoint { 680, { 0.046770000000f, 0.0170000000000f, 0.000000000000f } },
    CIEPoint { 681, { 0.043784050000f, 0.0159037900000f, 0.000000000000f } },
    CIEPoint { 682, { 0.040875360000f, 0.0148371800000f, 0.000000000000f } },
    CIEPoint { 683, { 0.038072640000f, 0.0138106800000f, 0.000000000000f } },
    CIEPoint { 684, { 0.035404610000f, 0.0128347800000f, 0.000000000000f } },
    CIEPoint { 685, { 0.032900000000f, 0.0119200000000f, 0.000000000000f } },
    CIEPoint { 686, { 0.030564190000f, 0.0110683100000f, 0.000000000000f } },
    CIEPoint { 687, { 0.028380560000f, 0.0102733900000f, 0.000000000000f } },
    CIEPoint { 688, { 0.026344840000f, 0.0095333110000f, 0.000000000000f } },
    CIEPoint { 689, { 0.024452750000f, 0.0088461570000f, 0.000000000000f } },
    CIEPoint { 690, { 0.022700000000f, 0.0082100000000f, 0.000000000000f } },
    CIEPoint { 691, { 0.021084290000f, 0.0076237810000f, 0.000000000000f } },
    CIEPoint { 692, { 0.019599880000f, 0.0070854240000f, 0.000000000000f } },
    CIEPoint { 693, { 0.018237320000f, 0.0065914760000f, 0.000000000000f } },
    CIEPoint { 694, { 0.016987170000f, 0.0061384850000f, 0.000000000000f } },
    CIEPoint { 695, { 0.015840000000f, 0.0057230000000f, 0.000000000000f } },
    CIEPoint { 696, { 0.014790640000f, 0.0053430590000f, 0.000000000000f } },
    CIEPoint { 697, { 0.013831320000f, 0.0049957960000f, 0.000000000000f } },
    CIEPoint { 698, { 0.012948680000f, 0.0046764040000f, 0.000000000000f } },
    CIEPoint { 699, { 0.012129200000f, 0.0043800750000f, 0.000000000000f } },
    CIEPoint { 700, { 0.011359160000f, 0.0041020000000f, 0.000000000000f } },
    CIEPoint { 701, { 0.010629350000f, 0.0038384530000f, 0.000000000000f } },
    CIEPoint { 702, { 0.009938846000f, 0.0035890990000f, 0.000000000000f } },
    CIEPoint { 703, { 0.009288422000f, 0.0033542190000f, 0.000000000000f } },
    CIEPoint { 704, { 0.008678854000f, 0.0031340930000f, 0.000000000000f } },
    CIEPoint { 705, { 0.008110916000f, 0.0029290000000f, 0.000000000000f } },
    CIEPoint { 706, { 0.007582388000f, 0.0027381390000f, 0.000000000000f } },
    CIEPoint { 707, { 0.007088746000f, 0.0025598760000f, 0.000000000000f } },
    CIEPoint { 708, { 0.006627313000f, 0.0023932440000f, 0.000000000000f } },
    CIEPoint { 709, { 0.006195408000f, 0.0022372750000f, 0.000000000000f } },
    CIEPoint { 710, { 0.005790346000f, 0.0020910000000f, 0.000000000000f } },
    CIEPoint { 711, { 0.005409826000f, 0.0019535870000f, 0.000000000000f } },
    CIEPoint { 712, { 0.005052583000f, 0.0018245800000f, 0.000000000000f } },
    CIEPoint { 713, { 0.004717512000f, 0.0017035800000f, 0.000000000000f } },
    CIEPoint { 714, { 0.004403507000f, 0.0015901870000f, 0.000000000000f } },
    CIEPoint { 715, { 0.004109457000f, 0.0014840000000f, 0.000000000000f } },
    CIEPoint { 716, { 0.003833913000f, 0.0013844960000f, 0.000000000000f } },
    CIEPoint { 717, { 0.003575748000f, 0.0012912680000f, 0.000000000000f } },
    CIEPoint { 718, { 0.003334342000f, 0.0012040920000f, 0.000000000000f } },
    CIEPoint { 719, { 0.003109075000f, 0.0011227440000f, 0.000000000000f } },
    CIEPoint { 720, { 0.002899327000f, 0.0010470000000f, 0.000000000000f } },
    CIEPoint { 721, { 0.002704348000f, 0.0009765896000f, 0.000000000000f } },
    CIEPoint { 722, { 0.002523020000f, 0.0009111088000f, 0.000000000000f } },
    CIEPoint { 723, { 0.002354168000f, 0.0008501332000f, 0.000000000000f } },
    CIEPoint { 724, { 0.002196616000f, 0.0007932384000f, 0.000000000000f } },
    CIEPoint { 725, { 0.002049190000f, 0.0007400000000f, 0.000000000000f } },
    CIEPoint { 726, { 0.001910960000f, 0.0006900827000f, 0.000000000000f } },
    CIEPoint { 727, { 0.001781438000f, 0.0006433100000f, 0.000000000000f } },
    CIEPoint { 728, { 0.001660110000f, 0.0005994960000f, 0.000000000000f } },
    CIEPoint { 729, { 0.001546459000f, 0.0005584547000f, 0.000000000000f } },
    CIEPoint { 730, { 0.001439971000f, 0.0005200000000f, 0.000000000000f } },
    CIEPoint { 731, { 0.001340042000f, 0.0004839136000f, 0.000000000000f } },
    CIEPoint { 732, { 0.001246275000f, 0.0004500528000f, 0.000000000000f } },
    CIEPoint { 733, { 0.001158471000f, 0.0004183452000f, 0.000000000000f } },
    CIEPoint { 734, { 0.001076430000f, 0.0003887184000f, 0.000000000000f } },
    CIEPoint { 735, { 0.000999949300f, 0.0003611000000f, 0.000000000000f } },
    CIEPoint { 736, { 0.000928735800f, 0.0003353835000f, 0.000000000000f } },
    CIEPoint { 737, { 0.000862433200f, 0.0003114404000f, 0.000000000000f } },
    CIEPoint { 738, { 0.000800750300f, 0.0002891656000f, 0.000000000000f } },
    CIEPoint { 739, { 0.000743396000f, 0.0002684539000f, 0.000000000000f } },
    CIEPoint { 740, { 0.000690078600f, 0.0002492000000f, 0.000000000000f } },
    CIEPoint { 741, { 0.000640515600f, 0.0002313019000f, 0.000000000000f } },
    CIEPoint { 742, { 0.000594502100f, 0.0002146856000f, 0.000000000000f } },
    CIEPoint { 743, { 0.000551864600f, 0.0001992884000f, 0.000000000000f } },
    CIEPoint { 744, { 0.000512429000f, 0.0001850475000f, 0.000000000000f } },
    CIEPoint { 745, { 0.000476021300f, 0.0001719000000f, 0.000000000000f } },
    CIEPoint { 746, { 0.000442453600f, 0.0001597781000f, 0.000000000000f } },
    CIEPoint { 747, { 0.000411511700f, 0.0001486044000f, 0.000000000000f } },
    CIEPoint { 748, { 0.000382981400f, 0.0001383016000f, 0.000000000000f } },
    CIEPoint { 749, { 0.000356649100f, 0.0001287925000f, 0.000000000000f } },
    CIEPoint { 750, { 0.000332301100f, 0.0001200000000f, 0.000000000000f } },
    CIEPoint { 751, { 0.000309758600f, 0.0001118595000f, 0.000000000000f } },
    CIEPoint { 752, { 0.000288887100f, 0.0001043224000f, 0.000000000000f } },
    CIEPoint { 753, { 0.000269539400f, 0.0000973356000f, 0.000000000000f } },
    CIEPoint { 754, { 0.000251568200f, 0.0000908458700f, 0.000000000000f } },
    CIEPoint { 755, { 0.000234826100f, 0.0000848000000f, 0.000000000000f } },
    CIEPoint { 756, { 0.000219171000f, 0.0000791466700f, 0.000000000000f } },
    CIEPoint { 757, { 0.000204525800f, 0.0000738580000f, 0.000000000000f } },
    CIEPoint { 758, { 0.000190840500f, 0.0000689160000f, 0.000000000000f } },
    CIEPoint { 759, { 0.000178065400f, 0.0000643026700f, 0.000000000000f } },
    CIEPoint { 760, { 0.000166150500f, 0.0000600000000f, 0.000000000000f } },
    CIEPoint { 761, { 0.000155023600f, 0.0000559818700f, 0.000000000000f } },
    CIEPoint { 762, { 0.000144621900f, 0.0000522256000f, 0.000000000000f } },
    CIEPoint { 763, { 0.000134909800f, 0.0000487184000f, 0.000000000000f } },
    CIEPoint { 764, { 0.000125852000f, 0.0000454474700f, 0.000000000000f } },
    CIEPoint { 765, { 0.000117413000f, 0.0000424000000f, 0.000000000000f } },
    CIEPoint { 766, { 0.000109551500f, 0.0000395610400f, 0.000000000000f } },
    CIEPoint { 767, { 0.000102224500f, 0.0000369151200f, 0.000000000000f } },
    CIEPoint { 768, { 0.000095394450f, 0.0000344486800f, 0.000000000000f } },
    CIEPoint { 769, { 0.000089023900f, 0.0000321481600f, 0.000000000000f } },
    CIEPoint { 770, { 0.000083075270f, 0.0000300000000f, 0.000000000000f } },
    CIEPoint { 771, { 0.000077512690f, 0.0000279912500f, 0.000000000000f } },
    CIEPoint { 772, { 0.000072313040f, 0.0000261135600f, 0.000000000000f } },
    CIEPoint { 773, { 0.000067457780f, 0.0000243602400f, 0.000000000000f } },
    CIEPoint { 774, { 0.000062928440f, 0.0000227246100f, 0.000000000000f } },
    CIEPoint { 775, { 0.000058706520f, 0.0000212000000f, 0.000000000000f } },
    CIEPoint { 776, { 0.000054770280f, 0.0000197785500f, 0.000000000000f } },
    CIEPoint { 777, { 0.000051099180f, 0.0000184528500f, 0.000000000000f } },
    CIEPoint { 778, { 0.000047676540f, 0.0000172168700f, 0.000000000000f } },
    CIEPoint { 779, { 0.000044485670f, 0.0000160645900f, 0.000000000000f } },
    CIEPoint { 780, { 0.000041509940f, 0.0000149900000f, 0.000000000000f } },
    CIEPoint { 781, { 0.000038733240f, 0.0000139872800f, 0.000000000000f } },
    CIEPoint { 782, { 0.000036142030f, 0.0000130515500f, 0.000000000000f } },
    CIEPoint { 783, { 0.000033723520f, 0.0000121781800f, 0.000000000000f } },
    CIEPoint { 784, { 0.000031464870f, 0.0000113625400f, 0.000000000000f } },
    CIEPoint { 785, { 0.000029353260f, 0.0000106000000f, 0.000000000000f } },
    CIEPoint { 786, { 0.000027375730f, 0.0000098858770f, 0.000000000000f } },
    CIEPoint { 787, { 0.000025524330f, 0.0000092173040f, 0.000000000000f } },
    CIEPoint { 788, { 0.000023793760f, 0.0000085923620f, 0.000000000000f } },
    CIEPoint { 789, { 0.000022178700f, 0.0000080091330f, 0.000000000000f } },
    CIEPoint { 790, { 0.000020673830f, 0.0000074657000f, 0.000000000000f } },
    CIEPoint { 791, { 0.000019272260f, 0.0000069595670f, 0.000000000000f } },
    CIEPoint { 792, { 0.000017966400f, 0.0000064879950f, 0.000000000000f } },
    CIEPoint { 793, { 0.000016749910f, 0.0000060486990f, 0.000000000000f } },
    CIEPoint { 794, { 0.000015616480f, 0.0000056393960f, 0.000000000000f } },
    CIEPoint { 795, { 0.000014559770f, 0.0000052578000f, 0.000000000000f } },
    CIEPoint { 796, { 0.000013573870f, 0.0000049017710f, 0.000000000000f } },
    CIEPoint { 797, { 0.000012654360f, 0.0000045697200f, 0.000000000000f } },
    CIEPoint { 798, { 0.000011797230f, 0.0000042601940f, 0.000000000000f } },
    CIEPoint { 799, { 0.000010998440f, 0.0000039717390f, 0.000000000000f } },
    CIEPoint { 800, { 0.000010253980f, 0.0000037029000f, 0.000000000000f } },
    CIEPoint { 801, { 0.000009559646f, 0.0000034521630f, 0.000000000000f } },
    CIEPoint { 802, { 0.000008912044f, 0.0000032183020f, 0.000000000000f } },
    CIEPoint { 803, { 0.000008308358f, 0.0000030003000f, 0.000000000000f } },
    CIEPoint { 804, { 0.000007745769f, 0.0000027971390f, 0.000000000000f } },
    CIEPoint { 805, { 0.000007221456f, 0.0000026078000f, 0.000000000000f } },
    CIEPoint { 806, { 0.000006732475f, 0.0000024312200f, 0.000000000000f } },
    CIEPoint { 807, { 0.000006276423f, 0.0000022665310f, 0.000000000000f } },
    CIEPoint { 808, { 0.000005851304f, 0.0000021130130f, 0.000000000000f } },
    CIEPoint { 809, { 0.000005455118f, 0.0000019699430f, 0.000000000000f } },
    CIEPoint { 810, { 0.000005085868f, 0.0000018366000f, 0.000000000000f } },
    CIEPoint { 811, { 0.000004741466f, 0.0000017122300f, 0.000000000000f } },
    CIEPoint { 812, { 0.000004420236f, 0.0000015962280f, 0.000000000000f } },
    CIEPoint { 813, { 0.000004120783f, 0.0000014880900f, 0.000000000000f } },
    CIEPoint { 814, { 0.000003841716f, 0.0000013873140f, 0.000000000000f } },
    CIEPoint { 815, { 0.000003581652f, 0.0000012934000f, 0.000000000000f } },
    CIEPoint { 816, { 0.000003339127f, 0.0000012058200f, 0.000000000000f } },
    CIEPoint { 817, { 0.000003112949f, 0.0000011241430f, 0.000000000000f } },
    CIEPoint { 818, { 0.000002902121f, 0.0000010480090f, 0.000000000000f } },
    CIEPoint { 819, { 0.000002705645f, 0.0000009770578f, 0.000000000000f } },
    CIEPoint { 820, { 0.000002522525f, 0.0000009109300f, 0.000000000000f } },
    CIEPoint { 821, { 0.000002351726f, 0.0000008492513f, 0.000000000000f } },
    CIEPoint { 822, { 0.000002192415f, 0.0000007917212f, 0.000000000000f } },
    CIEPoint { 823, { 0.000002043902f, 0.0000007380904f, 0.000000000000f } },
    CIEPoint { 824, { 0.000001905497f, 0.0000006881098f, 0.000000000000f } },
    CIEPoint { 825, { 0.000001776509f, 0.0000006415300f, 0.000000000000f } },
    CIEPoint { 826, { 0.000001656215f, 0.0000005980895f, 0.000000000000f } },
    CIEPoint { 827, { 0.000001544022f, 0.0000005575746f, 0.000000000000f } },
    CIEPoint { 828, { 0.000001439440f, 0.0000005198080f, 0.000000000000f } },
    CIEPoint { 829, { 0.000001341977f, 0.0000004846123f, 0.000000000000f } },
    CIEPoint { 830, { 0.000001251141f, 0.0000004518100f, 0.000000000000f } },
};


class XYZRGBConverter
{
public:
    XYZRGBConverter(const Eigen::Vector3d&,
                    const Eigen::Vector2d&,
                    const Eigen::Vector2d&,
                    const Eigen::Vector2d&);
    Eigen::Vector3f convert(const Eigen::Vector3d&) const;
private:
    Eigen::Matrix3d transform;
    Eigen::Vector3f whitepointRGB;

    Eigen::Vector3f convertUnnormalized(const Eigen::Vector3d&) const;
};


// Compute the XYZ->RGB transformation matrix from the whitepoint and the
// chromaticities of the RGB primaries. Implements the formulae from
// http://www.brucelindbloom.com/index.html?Eqn_Spect_to_XYZ.html
XYZRGBConverter::XYZRGBConverter(const Eigen::Vector3d& whitepointXYZ,
                                 const Eigen::Vector2d& redxy,
                                 const Eigen::Vector2d& greenxy,
                                 const Eigen::Vector2d& bluexy)
{
    auto xr = redxy[0] / redxy[1];
    auto zr = (1.0 - redxy[0] - redxy[1]) / redxy[1];
    auto xg = greenxy[0] / greenxy[1];
    auto zg = (1.0 - greenxy[0] - greenxy[1]) / greenxy[1];
    auto xb = bluexy[0] / bluexy[1];
    auto zb = (1.0 - bluexy[0] - bluexy[1]) / bluexy[1];
    Eigen::Matrix3d mat;
    mat << xr,  xg,  xb,
           1.0, 1.0, 1.0,
           zr,  zg,  zb;
    Eigen::RowVector3d Srgb = (mat.inverse() * whitepointXYZ).transpose();
    mat.array().rowwise() *= Srgb.array();
    transform = mat.inverse();
    // Linear RGB color of the white-point used for fixing out-of-gamut colors
    whitepointRGB = convertUnnormalized(whitepointXYZ);
}


Eigen::Vector3f
XYZRGBConverter::convert(const Eigen::Vector3d& xyz) const
{
    Eigen::Vector3f result = convertUnnormalized(xyz);
    Eigen::Index index;
    auto minValue = result.minCoeff(&index);
    if (minValue >= 0.0)
        return result;

    // Handle out-of-gamut colors (occurs for ultracool objects). For
    // consistency with previous versions of Celestia, follow the approach
    // taken by Mitchell Charity, where the color is mixed with the white
    // point in linear RGB space until the color is inside the gamut.
    // http://www.vendian.org/mncharity/dir3/blackbody/
    float p = whitepointRGB[index] / (whitepointRGB[index] - minValue);
    return whitepointRGB + p * (result - whitepointRGB);
}


Eigen::Vector3f
XYZRGBConverter::convertUnnormalized(const Eigen::Vector3d& xyz) const
{
    Eigen::Vector3d rgb = transform * xyz;
    return (rgb / rgb.maxCoeff()).cast<float>();
}


// Blackbody spectrum, lambda in nanometers, temp in kelvins
double planck(double lambda, double temp)
{
    // Planck's constant * speed of light / Boltzmann constant * 1e9
    constexpr double hc_k = 0.01438777353827720266e9;
    return std::pow(lambda / 1000.0, -5.0) / std::expm1(hc_k / (lambda * temp));
}


using BlackbodyTable = std::array<Color, BlackbodyTableEntries>;


Eigen::Vector3d temperatureToXYZ(float temp)
{
    Eigen::Vector3d xyz = Eigen::Vector3d::Zero();
    for (const CIEPoint& cie : CIEFunctions)
    {
        double spectrum = planck(cie.wavelength, temp);
        xyz += Eigen::Map<const Eigen::Vector3f>(cie.xyz.data()).cast<double>() * spectrum;
    }

    return xyz / xyz.sum();
}


const Eigen::Vector3d SunXYZ = temperatureToXYZ(SolarTemperatureBucket);
const Eigen::Vector3d VegaXYZ = temperatureToXYZ(VegaTemperatureBucket);


void
createBlackbodyTable(const Eigen::Vector3d& whitepoint,
                     float& tempScale,
                     std::vector<Color>& colors)
{
    tempScale = (BlackbodyTableEntries - 1) / MaxTemperature;
    colors.clear();
    colors.reserve(BlackbodyTableEntries);
    colors.emplace_back(0.0f, 0.0f, 0.0f);

    XYZRGBConverter converter(whitepoint, SRGB_R_xy, SRGB_G_xy, SRGB_B_xy);

    for (std::size_t i = 1; i < BlackbodyTableEntries; ++i)
    {
        float teff = static_cast<float>(i) * TemperatureStep;
        Eigen::Vector3d xyz = temperatureToXYZ(teff);
        Eigen::Vector3f rgb = converter.convert(xyz);
        colors.emplace_back(rgb);
    }
}

} // end unnamed namespace


ColorTemperatureTable::ColorTemperatureTable(ColorTableType _type)
{
    if (!setType(_type))
        setType(ColorTableType::Enhanced);
}


bool
ColorTemperatureTable::setType(ColorTableType _type)
{
    tableType = _type;
    switch (tableType)
    {
    case ColorTableType::Enhanced:
        colors.clear();
        colors.reserve(StarColors_Enhanced.size());
        std::copy(StarColors_Enhanced.cbegin(), StarColors_Enhanced.cend(), std::back_inserter(colors));
        tempScale = static_cast<float>(StarColors_Enhanced.size() - 1) / MaxTemperature;
        return true;

    case ColorTableType::Blackbody_D65:
        createBlackbodyTable(D65_XYZ, tempScale, colors);
        return true;

    case ColorTableType::SunWhite:
        createBlackbodyTable(SunXYZ, tempScale, colors);
        return true;

    case ColorTableType::VegaWhite:
        createBlackbodyTable(VegaXYZ, tempScale, colors);
        return true;

    default:
        return false;
    }
}
