#include "bool.h"
#ifdef HAVE_SYSLOG
# include <syslog.h>
#endif
#include <sys/fcntl.h>
#include <sys/stat.h>
#include <sys/param.h>
#include <sys/resource.h>
#include <sys/wait.h>
#ifdef TIME_WITH_SYS_TIME
# include <sys/time.h>
# include <time.h>
#else
# ifdef HAVE_SYS_TIME_H
#  include <sys/time.h>
# else
#  include <time.h>
# endif
#endif
#include <errno.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <stdarg.h>
#include <stdio.h>
#include "util.h"
#include "child.h"
#include "main.h"

static bool create_file(char *file, int options);
static void print_date_pid(void);
static bool become_child(void);

/* Never pass a user generated buffer directly to one of the notice
 * functions. e.g. do -not- do
 * 	notice(foodata);
 * Instead do
 * 	notice("%s", foodata);
 */
/* TODO: levels need checking (e.g LOG_NOTICE) */
void notice(const char *format, ...)
{
	va_list ap;
	va_start(ap, format);
#ifdef HAVE_SYSLOG
	if (MYLOG_OPTIONS==MYLOG_SYSLOG) {
		if (-1==vsnprintf(error_messages, MAX_LOG_LENGTH, format, ap))
			syslog(LOG_NOTICE, "message too big for buffer\n");
		else
			syslog(LOG_NOTICE, "%s", error_messages);
	} else {
		print_date_pid();
		vfprintf(stderr, format, ap);
		fflush(stderr);
	}
#else
	print_date_pid();
	vfprintf(stderr, format, ap);
	fflush(stderr);
#endif
	va_end(ap);
	return;
}

void notice_err(const char *format, ...)
{
	va_list ap;
	va_start(ap, format);
#ifdef HAVE_SYSLOG
	if (MYLOG_OPTIONS==MYLOG_SYSLOG) {
		if (-1==vsnprintf(error_messages, MAX_LOG_LENGTH, format, ap))
			syslog(LOG_NOTICE, "message too big for buffer\n");
		else
			syslog(LOG_NOTICE, "%s: %m\n", error_messages);
	} else {
		print_date_pid();
		vfprintf(stderr, format, ap);
		fprintf(stderr, ": %s\n", strerror(errno));
		fflush(stderr);
	}
#else
	print_date_pid();
	vfprintf(stderr, format, ap);
	fprintf(stderr, ": %s\n", strerror(errno));
	fflush(stderr);
#endif
	va_end(ap);
	return;
}

void notice_errdebug(const char *format, ...)
{
	va_list ap;
	
	if (!set.debug)
		return;

	va_start(ap, format);
#ifdef HAVE_SYSLOG
	if (MYLOG_OPTIONS==MYLOG_SYSLOG) {
		if (-1==vsnprintf(error_messages, MAX_LOG_LENGTH, format, ap))
			syslog(LOG_NOTICE, "message too big for buffer\n");
		else
			syslog(LOG_DEBUG, "%s: %m\n", error_messages);
	} else {
		print_date_pid();
		vfprintf(stderr, format, ap);
		fprintf(stderr, ": %s\n", strerror(errno));
		fflush(stderr);
	}
#else
	print_date_pid();
	vfprintf(stderr, format, ap);
	fprintf(stderr, ": %s\n", strerror(errno));
	fflush(stderr);
#endif
	va_end(ap);
	return;
}

void notice_debug(const char *format, ...)
{
	va_list ap;

	if (!set.debug)
		return;

	va_start(ap, format);
#ifdef HAVE_SYSLOG
	if (MYLOG_OPTIONS==MYLOG_SYSLOG) {
		if (-1==vsnprintf(error_messages, MAX_LOG_LENGTH, format, ap))
			syslog(LOG_NOTICE, "message too big for buffer\n");
		else
			syslog(LOG_DEBUG, "%s", error_messages);
	} else {
		print_date_pid();
		vfprintf(stderr, format, ap);
		fflush(stderr);
	}
#else
	print_date_pid();
	vfprintf(stderr, format, ap);
	fflush(stderr);
#endif
	va_end(ap);
	return;
}

/* To start each line to a file/stderr with the date and our pid.
 * (For MYLOG_OPTIONS==MYLOG_STDERR_DATEPID) */
void print_date_pid(void)
{
#ifdef HAVE_STRFTIME
	time_t curr;
	if (MYLOG_OPTIONS!=MYLOG_STDERR_DATEPID)
		return;
	curr = time(NULL);
	if (0==strftime(error_messages, MAX_LOG_LENGTH, "%c", localtime(&curr)))
		return;
	fputs(error_messages, stderr);
	fprintf(stderr, " [%lu]: ", (u_long)getpid());
#endif
	return;
}

bool become_child(void)
{
	pid_t pid = fork();
	if (pid==-1) {
		notice_err("fork failed");
		return TRUE;
	}
	if (pid > 0)
		_exit(0);
	return FALSE;
}

bool become_daemon(void)
{
	FILE *pidfp;

	if (become_child() || become_child())
		return TRUE;
	if (-1==setsid()) {
		notice_err("setsid failed");
		return TRUE;
	}

	MYLOG_OPTIONS = MYLOG_STDERR_DATEPID;
#ifdef HAVE_SYSLOG
	if (!set.logfile) {
		openlog("dwun", LOG_PID, LOG_DAEMON);
		MYLOG_OPTIONS = MYLOG_SYSLOG;
	}
#endif
	if (set.logfile) {
		/* Created in rcfile if it didn't exist */
		if (redirect_std(STDERR_FILENO, set.logfile, 
					O_WRONLY | O_APPEND)) {
			notice_err("opening logfile '%s' for writing failed", 
					set.logfile);
			return TRUE;
		}
	} else {
		if (redirect_std(STDERR_FILENO, "/dev/null", O_WRONLY)) {
			notice_err("redirecting standard error failed");
			return TRUE;
		}
	}
	if (redirect_std(STDOUT_FILENO, "/dev/null", O_WRONLY)) {
		notice_err("redirecting standard output failed");
		return TRUE;
	}
	if (redirect_std(STDIN_FILENO, "/dev/null", O_RDONLY)) {
		notice_err("redirecting standard input failed");
		return TRUE;
	}

	/* Created in rcfile or set_string_defaults if it didn't exist. */
	if (!(pidfp = fopen(set.pidfile, "w"))) {
		notice_err("opening pid file '%s' for writing failed", 
				set.pidfile);
		return TRUE;
	}
	
	/* must be after daemon or pid will have changed */
	fprintf(pidfp, "%lu\n", (u_long)getpid());
	if (EOF==fclose(pidfp)) {
		notice_err("closing pid file '%s' failed", set.pidfile);
		return TRUE;
	}
	chdir("/");
	return FALSE;
}

bool redirect_std(int fd, const char *filename, int flags)
{
	int tmp = open(filename, flags, S_IRUSR | S_IWUSR); /* -rw------ */
	if (-1==tmp)
		return TRUE;
	fd = dup2(tmp, fd);
	close(tmp);
	if (-1==fd)
		return TRUE;
	return FALSE;
}
	
/* xstrdup */
bool alloc_copy(char **s, char *msg)
{
	if (!(*s = strdup(msg))) {
		notice_err("malloc failed");
		return TRUE;
	}
	return FALSE;
}

/* Put the realpath of msg in *s so we can chdir("/") safely.
 * [Important for set.authfile, the others either use "/bin/sh -c" or are
 * done before the chdir.]
 *
 * If the options bitmask includes...
 * MY_CREATE then we will create the file if it doesn't exist.
 * MY_ALLREAD then if we create the file, it will be -rw-r--r-- instead of
 * 	the default -rw-------.
 */
bool realfile(char **s, char *msg, int options)
{
	if (!(*s = xmalloc(MAXPATHLEN)))
		return TRUE;
	if (realpath(msg, *s))
		return FALSE;
	if (errno==ENOENT && (options & MY_CREATE)) {
		if (create_file(msg, options)) {
			notice_err("creating file '%s' failed", msg);
			return TRUE;
		}
		if (realpath(msg, *s))
			return FALSE;
	}
	notice_err("accessing file '%s' failed", msg);
	return TRUE;
}

bool create_file(char *file, int options)
{
	int fd;
	mode_t xmode = 0;
	if (options & MY_ALLREAD)
		xmode = S_IRGRP | S_IROTH;
	fd = open(file, O_WRONLY | O_CREAT | O_EXCL, S_IRUSR | S_IWUSR | xmode);
	if (fd==-1)
		return TRUE; /* maintaining errno */
	close(fd);
	return FALSE;
}

/* like strncpy, but the destination is always null-terminated.
 * char foo[FOO_LEN+1];
 * char bar[FOO_LEN+1];
 *
 * ...
 * term_strncpy(bar, foo, FOO_LEN)
 */
void term_strncpy(char *dest, const char *src, size_t n)
{
	size_t len = strlen(src);
	size_t smallest = len > n ? n : len;
	strncpy(dest, src, n);
	*(dest + smallest) = '\0';
	return;
}

/* strtoFOO wrapper with error checking and using base 10 with a NULL endptr.
 * If there is an error, *conv is not altered */
#define mystrtonum(str, type, numpt, func, min, max) 		\
{								\
	type ret = (func)(str, NULL, 10);			\
	if ((ret==(min) || ret==(max)) && errno==ERANGE)	\
		return TRUE;					\
	*numpt = ret;						\
	return FALSE;						\
}

bool mystrtol(const char *nptr, long *conv)
{
	mystrtonum(nptr, long, conv, strtol, LONG_MIN, LONG_MAX);
}
bool mystrtoi(const char *nptr, int *conv)
{
	mystrtonum(nptr, int, conv, strtol, INT_MIN, INT_MAX);
}
bool mystrtouc(const char *nptr, u_char *conv)
{
	mystrtonum(nptr, u_char, conv, strtol, 0, UCHAR_MAX);
}
bool mystrtoul(const char *nptr, u_long *conv)
{
	mystrtonum(nptr, u_long, conv, strtoul, 0, ULONG_MAX);
}
bool mystrtous(const char *nptr, unsigned short *conv)
{
	mystrtonum(nptr, unsigned short, conv, strtoul, 0, USHRT_MAX);
}

/* Blocks all signals, forks, then returns to previous block mask in parent.
 * Child calls unblock_all_signals when they've ready to receive signals */
pid_t xfork(void)
{
	pid_t pid;
	sigset_t old;
	int fork_errno;

	block_all_signals(&old);
	pid = fork();
	fork_errno = errno;
	if (pid!=0) /* child chooses when to do unblock_all_signals(NULL) */
		sigprocmask(SIG_SETMASK, &old, NULL);
	errno = fork_errno;
	return pid;
}

/* Remove inherited signals from parent. Do after an xfork(). After we have
 * called cleanup_signals() we can safely do unblock_all_signals(NULL). 
 *
 * (Since all signals are blocked, we don't bother with EINTR.) */
bool cleanup_signals(void)
{
	struct sigaction cursig;
	int sigs[] = { SIGHUP, SIGUSR1, SIGUSR2, CONNECT_FAILED_SIGNAL, SIGCHLD,
		SIGSEGV, SIGTERM, SIGPIPE, 0 };
	int i;
	
	memset(&cursig, 0, sizeof(struct sigaction));
	cursig.sa_handler = SIG_DFL;

	alarm(0); /* remove if present */
	for (i=0; sigs[i]; ++i) {
		if (-1==sigaction(sigs[i], &cursig, NULL)) {
			notice_err("cleanup of signal %d failed", sigs[i]);
			return TRUE;
		}
	}
	return FALSE;
}

/* Macros for block_all_signals() and unblock_all_signals() in util.h. */
void all_signals(int how, sigset_t *old)
{
	sigset_t full;
	sigfillset(&full);
	sigprocmask(how, &full, old);
	return;
}

/* fd should be sig_atomic_t. It's useful to do
 * 	#define xsigwrite(x) sigwrite(some_write_pipe, (x))
 * in the signal handler.
 *
 * It should be safe to ignore the return status of sigwrite and sigread -
 * they should never fail. (Except if we close the signal pipe without
 * (XXX: without what?)  e.g. if we are about to exit, and then receive a
 * signal).
 */
bool sigwrite(int fd, int c)
{
	int ret;
	do {
		ret = write(fd, &c, sizeof(c));
	} while (ret==-1 && errno==EINTR);
	if (ret==-1) {
		notice_errdebug("sigwrite failed");
		return TRUE;
	}
	return FALSE;
}

bool sigread(int fd, int *c)
{
	int ret;
	do {
		ret = read(fd, c, sizeof(*c));
	} while (ret==-1 && errno==EINTR);
	if (ret==-1) {
		notice_errdebug("sigread failed");
		return TRUE;
	}
	return FALSE;
}

pid_t waitpid_noint(pid_t pid, int *status, int options)
{
	int ret;
	do {
		ret = waitpid(pid, status, options);
	} while (ret==-1 && errno==EINTR);
	return ret;
}

bool cleanup_fds_except(int leave)
{
	int fd;
	int fdlimit;
	struct rlimit rl;
	
	if (-1==getrlimit(RLIMIT_NOFILE, &rl)) {
		fdlimit = sysconf(_SC_OPEN_MAX);
		if (fdlimit==-1)
			fdlimit = 1024;
	} else {
		fdlimit = rl.rlim_cur;
	}
	for (fd=0; fd < fdlimit; ++fd) {
		if (fd!=leave)
			close(fd);
	}
	if (		redirect_std(STDOUT_FILENO, "/dev/null", O_WRONLY) ||
			redirect_std(STDERR_FILENO, "/dev/null", O_WRONLY) ||
			redirect_std(STDIN_FILENO, "/dev/null", O_RDONLY)) {
		notice_err("redirecting standard streams to /dev/null failed");
		return TRUE;
	}
	return FALSE;
}

void *xmalloc(size_t size)
{
	void *p = malloc(size);
	if (!p) 
		notice_err("malloc failed");
	return p;
}
