#include "bool.h"
#include <stdio.h>
#include <limits.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/socket.h>
#include <netinet/in.h>
#ifdef HAVE_ARPA_INET_H
#include <arpa/inet.h>
#endif
#ifdef HAVE_SYS_TIME_H
# include <sys/time.h>
#else
# include <time.h>
#endif
#include <errno.h>
#include <string.h>
#include <signal.h>
#include <stdlib.h>
#ifdef HAVE_SYSLOG
# include <syslog.h>
#endif
#ifdef HAVE_SYS_SELECT_H
# include <sys/select.h>
#endif
#include "shared.h"
#include "rets.h"
#include "network.h"
#include "child.h"
#include "socket.h"
#include "buffer.h"
#include "item.h"
#include "exec.h"
#include "util.h"
#include "state.h"
#include "disconnect.h"
#include "connect.h"
#include "ipc.h"
#include "main.h"

static bool create_listener(const char *ip);
static int parent(int *listeners);

/* Socket functions. */
static bool client_add(int sd, int sokcount, mmap_area authfile,
		struct clients *first, struct clients **end, int child_pwrite,
		fd_set *readersbak);
static bool sendtopid(pid_t pid, char *buf, struct clients *first);

/* IPC with child() processes. */
static bool do_child_msg(int pread, struct clients *start, time_t connect_time,
		struct pid_s *pid, struct state_s *state, int *time_to_hangup,
		int sokcount);
static bool time_send(int pwrite, time_t connect_time);
static bool pipe_send(int pwrite, struct pipe_msg_buf msg);
static bool pipe_recv(int pread, struct pipe_msg_buf *msg);

/* Internal/external functions. */
static SOK *open_external(void);
static SOK *open_internal(void);
static int sendtoall_pipe(SOK *stream, struct clients *start, const char *type);

/* Signal functions. */
static bool sigsetup(void);
static void sighandler(int sig);

static volatile sig_atomic_t sighandler_write_pipe; /* sighandler */
int internal_write_fd = -1; /* extern'ed in exec.h for run_fd3open(). */

int server(void)
{
	int i;
	int sd;
	int ret;
	int *listeners;
	static bool starting = TRUE;
	if (starting) {
		notice("starting\n");
		starting = FALSE;
	}

	/* bind and listen to all given IP addresses */
	if (!(listeners = xmalloc((set.num_ips+1) * sizeof(*listeners))))
		return QSHUTDOWN;
	for (i=0; set.bind_ips[i]; ++i) {
		sd = create_listener(set.bind_ips[i]);
		if (-1==sd) {
			/* TODO: close the sd's thus far. */
			free(listeners);
			return QSHUTDOWN;
		}
		listeners[i] = sd;
	}
	listeners[i] = -1;
	ret = parent(listeners);
	for (i=0; listeners[i]!=-1; ++i)
		close(listeners[i]);
	free(listeners);
	return ret;
}

bool create_listener(const char *ip)
{
#define BACKLOG 10
	int sd;
	struct sockaddr_in address;
	int optval;

	sd = socket(PF_INET, SOCK_STREAM, 0);
	if(-1==sd) {
		notice_err("socket failed");
		return -1;
        }
	address.sin_family = AF_INET;
	address.sin_port = htons(set.port);
	memset(&address.sin_zero, 0, sizeof(address.sin_zero));
	if (!inet_aton(ip, &address.sin_addr)) {
		notice("cannot bind to invalid ip '%s'\n", ip);
		return -1;
	}
	
	/* See RFC 793 for TIME_WAIT. We want to be able to bind to the 
	 * server socket immediately after the server is terminated, so
	 * we enable SO_REUSEADDR. */
	optval = 1;
	if (-1==setsockopt(sd, SOL_SOCKET, SO_REUSEADDR, &optval, 
				sizeof(optval))) {
		notice_err("setsockopt failed");
		close(sd);
		return -1;
	}
	if (-1==bind(sd, (struct sockaddr *)&address, sizeof(address))) {
		notice_err("bind failed for '%s'", ip);
		close(sd);
		return -1;
	}
	if(-1==listen(sd, BACKLOG)) {
		notice_err("listen failed for '%s'", ip);
		close(sd);
		return -1;
	}
	/* accept() may fail, we don't want it to block */
	if (-1==fcntl(sd, F_SETFL, O_NONBLOCK)) {
		notice_err("setting socket non-blocking failed");
		close(sd);
    		return -1;
	}
	return sd;
}

/* The main loop. */
int parent(int *listeners)
{
	/* state */
	int time_to_hangup = QSHUTDOWN;
	struct state_s state = { TRUE, CONS_DISCONNECTED, 0 };
	struct fail_s fail = { 0, 0 };
	int sokcount = 0;	 	/* number of clients */
	time_t connect_time = -1;
	struct pid_s pid = { NOT_RUN, NOT_RUN, NOT_RUN,	NOT_RUN, NOT_RUN,
		NOT_RUN, NOT_RUN };
	/* tmp vars */
	int lcount;			/* dealing with all listeners */
	int ret;
	int numready;
	bool maybe_adjust;
	bool waiting_for_death = FALSE;
	/* for select() */
	fd_set readersbak;
	fd_set readers;
	/* clients linked list */
	struct clients *start = NULL;
	struct clients *end;
	/* IPC */
	int pipe_parent[2] = { -1, -1 };
	int highest_fd;
	mmap_area authfile = { NULL, -1 };
	SOK *external_read = NULL; /* external messages from FIFO */
	SOK *internal_read = NULL; /* fd3 for messages from executed programs */
	SOK *streams[3];    /* NULL termed list of all streams we select on */
	/* signals */
	int pipe_signals[2] = { -1, -1 };
	int event;

	/* for select */
	FD_ZERO(&readersbak);
	for (lcount=0; listeners[lcount]!=-1; ++lcount)
		FD_SET(listeners[lcount], &readersbak);

	/* initialize list */
	if (!(start = xmalloc(sizeof(*start))))
		goto end;
	memset(start, 0, sizeof(struct clients)); /* unnecessary */
	end = start;
	end->next = NULL;
	
	/* child->parent communication. Write pipe is shared between all
	 * children so send of pipe_msg_buf must be atomic. */
	if (make_pipes(pipe_parent) || smallpipe(pipe_parent[1])) 
		goto end;
	FD_SET(pipe_parent[0], &readersbak);

	/* signal handling */
	if (make_pipes(pipe_signals))
		goto end;
	sighandler_write_pipe = pipe_signals[1];
	if (sigsetup())
		goto end;
	FD_SET(pipe_signals[0], &readersbak);
	
	/* authfile mmapped mem */
	if (map_authfile(&authfile))
		goto end;
	
	/* messages from named pipe */
	if (set.external) {
		external_read = open_external();
		if (!external_read)
			goto end;
		FD_SET(external_read->sd, &readersbak);
	}

	/* messages from {pre,post}_commandon */
	internal_read = open_internal();
	if (!internal_read)
		goto end;
	FD_SET(internal_read->sd, &readersbak);
	highest_fd = internal_read->sd;

	/* (Note: order is important because external_read -may- be NULL) */
	streams[0] = internal_read;
	streams[1] = external_read;
	streams[2] = NULL;

	time_to_hangup = QNORMAL;
	while (time_to_hangup==QNORMAL || QIS_RUNNING(pid.pre_commandon) || 
			QIS_RUNNING(pid.commandon) ||
			QIS_RUNNING(pid.post_commandon)) {
		/* waiting_for_death, act_on_signal() and do_child_msg() are
		 * the only things which can make calling adjust() necessary.
		 * Getting them to update a variable to set when adjust() is
		 * *really* necessary is, IMHO, more effort than it's worth. */
		maybe_adjust = FALSE;

		if (!waiting_for_death && time_to_hangup!=QNORMAL) {
			waiting_for_death = TRUE;
			/* Get commandon to die: don't accept(), ignore
			 * child requests and disconnect immediately. */
			for (lcount=0; listeners[lcount]!=-1; ++lcount)
				FD_CLR(listeners[lcount], &readersbak);
			FD_CLR(pipe_parent[0], &readersbak);
			/* FD_CLR(pipe_signals[0], &readersbak); */
			setall_disconnected(start->next, &pid.last_ondisc);
			state.pon = 0;
			pid.can_disconnect = HAVE_RUN;
			if (time_to_hangup==QRESTART) {
				sendtousers("%all", "#Server restarting\n",
						start->next);
			} else {
				sendtousers("%all", "#Server shutting down\n",
						start->next);
			}
			maybe_adjust = TRUE;
			numready = 0;
		} else {
			do {
				memcpy(&readers, &readersbak, sizeof(readers)); 
				ret = SOK_select(highest_fd+1, &readers, NULL,
						NULL, NULL, streams);
			} while (ret==-1 && errno==EINTR);
			if (-1==ret) {
				notice_err("select failed");
				time_to_hangup = QSHUTDOWN;
				continue;
			}
			numready = ret;
		}

		/* signals. */
		if (numready && FD_ISSET(pipe_signals[0], &readers)) {
			(void)sigread(pipe_signals[0], &event);
			act_on_signal(event, &pid, &state, &fail, &sokcount,
					start, &end, &connect_time,
					&time_to_hangup);
			maybe_adjust = TRUE;
			--numready;
		}
		/* message from child. */
		if (numready && FD_ISSET(pipe_parent[0], &readers)) {
			if (!do_child_msg(pipe_parent[0], start, connect_time,
					&pid, &state, &time_to_hangup,
					sokcount))
				maybe_adjust = TRUE;
			--numready;
		}
		/* adjust connection state. */
		if (maybe_adjust && set.commandon) {
			if (adjust(&pid, &state, start->next))
				raise(CONNECT_FAILED_SIGNAL); /* XXX */
		}
		/* someone trying to connect()? */
		for (lcount=0; listeners[lcount]!=-1; ++lcount) {
			if (FD_ISSET(listeners[lcount], &readers)) {
				if (!client_add(listeners[lcount], sokcount, 
							authfile, start->next,
							&end, pipe_parent[1],
							&readersbak)) {
					++sokcount;
				}
				if (!--numready)
					continue;
			}
		}
		/* message from external named pipe */
		if (external_read && FD_ISSET(external_read->sd, &readers)) {
			if (-1==sendtoall_pipe(external_read, start,
						"external")) {
				notice("ignoring messages from '%s'\n",
						set.external);
				FD_CLR(external_read->sd, &readersbak);
			}
			if (!--numready)
				continue;
		}
		/* message from internal pipe */
		if (FD_ISSET(internal_read->sd, &readers))
			(void)sendtoall_pipe(internal_read, start, "internal");
	} /* big while loop */
	
	kill_remove_all(start, &end); /* don't waitpid */
	kill(0, SIGTERM); /* rest of them. */
	while (waitpid_noint(-1, NULL, 0) > 0);

end: /* For fd's which are -1, close will just return -1 and errno==EBADF.
	unmap_authfile will return TRUE if authfile.data==NULL. sclose will
	return TRUE for a NULL SOK *. This is all fine and dandy. */
	free(start);
	close(pipe_parent[0]); close(pipe_parent[1]);
	block_all_signals(NULL);
	close(pipe_signals[0]); close(pipe_signals[1]);
	unmap_authfile(authfile);
	sclose(external_read);
	close(internal_write_fd);
	sclose(internal_read);
	return time_to_hangup;
}

/* Socket functions.
 *
 * client_add		- do everything from accept()ing, item_add()ing,
 * 			  fork()ing and calling child(). Also checks client
 * 			  against hostmask.
 * sendtousers		- send a message to all clients with this username.
 * 			  (or meta-users such as "%con".)
 * sendconnectedtousers	- like sendtousers but explicitly sends "#Connected\n".
 * sendtopid		- send a message to the client which is being served
 * 			  by a child with this pid.
 * sendalert		- send a message to the client with this socket.
 */

bool client_add(int sd, int sokcount, mmap_area authfile, struct clients *first,
		struct clients **end, int child_pwrite, fd_set *readersbak)
{
	struct sockaddr_in address;
	int addrlen = sizeof(struct sockaddr_in);
	pid_t pid;
	int sok = -1;
	int pipe_child[2] = { -1, -1 };
	char *name = NULL;
	
	do {
		sok = accept(sd, (struct sockaddr *)&address, &addrlen);
	} while (sok==-1 && errno==EINTR);
	if (sok==-1)
		return TRUE; /* reset between select and accept */
	if (!(name = makepeername_alloc(sok)))
		return TRUE;

	if (max_exceeded(sokcount+1, set.maxcons)) {
		notice("max connections exceeded, denied '%s'\n", name);
		goto error;
	}

	if (ip_out_of_range(name)) {
		notice("connection denied to %s\n", name);
		goto error;
	}
#if 0
	if (set.max_from_one!=-1) {
		if (max_per_host_equalled(name, first)) {
			notice("max connections exceeded for one host by %s\n",
					name);
			goto error;
		}
	}
#endif
	if (make_pipes(pipe_child))
		goto error;
	
	pid = xfork();
	if (pid==-1) {
	        notice_err("fork failed");
		goto error;
	} else if (pid==0) {
		if (cleanup_signals()) {
			notice("cleanup failed: not adding client %s\n", name);
			_exit(EXIT_FAILURE);
		}
		unblock_all_signals(NULL); /* broken? */
		close(sd);
		close(pipe_child[1]);
		close_all_items(first);
		if (child(sok, child_pwrite, pipe_child[0], authfile, name)) 
			_exit(EXIT_FAILURE);
		_exit(EXIT_SUCCESS);
	}
	close(pipe_child[0]); pipe_child[0] = -1;
	if (item_add(end, pid, sok, pipe_child[1], name))
		goto error;
	FD_SET(pipe_child[1], readersbak);
	return FALSE;
error:
	free(name);
	close(sok);
	close(pipe_child[0]); close(pipe_child[1]);
	return TRUE;
}

bool sendtousers_raw(char *user, char *msg, struct clients *first,
		bool connectedmsg)
{
#define BROADCAST(x)							\
	do {								\
		while (first) {						\
			if ((x))					\
				(void)sendalert(msg, first->sok);	\
			first = first->next;				\
		}							\
	} while (0)

	if (0==strcmp(user, "%con")) {
		if (set.onconnect_wait && connectedmsg) {
			BROADCAST(first->connected &&
					QHAVE_RUN(first->onconnect));
		} else {
			BROADCAST(first->connected);
		}
	} else if (0==strcmp(user, "%all")) {
		BROADCAST(TRUE);
	} else {
		BROADCAST(0==strcmp(first->username, user));
	}
	return FALSE;
}

bool sendtopid(pid_t pid, char *buf, struct clients *first)
{
	first = item_from_pid(first, pid);
	if (!first) {
		notice_debug("sendtopid: pid %lu not found\n", (u_long)pid);
		return TRUE;
	}
	return sendalert(buf, first->sok);
}

/* FIXME: Both sendalert and sendline in child.c can send to the same sok at
 * the same time - messages can become intermixed.
 *
 * mutex? have sendalert get the child process to relay? have us relay for
 * the child process? have a new process just for sending data?
 */
bool sendalert(char *msg, int sok)
{
	struct timeval tv;
	int ret;
	char *name;
	
	if (-1==sok)
		return FALSE; /* We're just waiting for their onconnect to
				 die so we can ondisconnect for them. */
	tv.tv_sec = SOK_SEND_TIMEOUT_SECS;
	tv.tv_usec = 0;
	
	ret = sok_send(sok, &tv, msg, strlen(msg));
	if (ret==SOK_SUCCESS)
		return FALSE;
	if (!(name = makepeername_alloc(sok))) {
		notice_debug("sendalert failed for unknown peer\n");
		return TRUE;
	}
	switch(ret) {
	case SOK_FAILED:
	        notice_debug("sendalert failed for %s\n", name);
		break;
	case SOK_TIMEOUT:
		notice_debug("sendalert timeout for %s\n", name);
		break;
	}
	free(name);
	return TRUE;
}

/* IPC with child() processes.
 *
 * do_child_msg		- gets and handles message from child.
 * time_send		- send child info for UPTIME.
 * pipe_send		- send pipe_msg_buf to child.
 * pipe_recv		- receive pipe_msg_buf from child.
 */

bool do_child_msg(int pread, struct clients *start, time_t connect_time,
		struct pid_s *pid, struct state_s *state, int *time_to_hangup,
		int sokcount)
{
	struct pipe_msg_buf msg;
	struct clients *item;
	pid_t *firstlastp = NULL;
	
	if (pipe_recv(pread, &msg))
		return TRUE;
	item = item_from_pid(start->next, msg.from);
	if (!item)
		return TRUE;
	switch(msg.command) {
	case cSTATUS:
		msg.from = 1;
		msg.command = state->con;
		if (pipe_send(item->pwrite, msg)) {
			notice("sending STATUS message failed\n");
			return TRUE;
		}
		break;
	case cCONNECT:
		if (state->pon==0)
			firstlastp = &pid->first_onconnect;
		if (!item_con(msg.from, firstlastp, start, TRUE))
			state->pon++;
		msg.from = 1;
		msg.command = state->con;
		if (set.onconnect_wait && !QHAVE_RUN(item->onconnect)) {
			if (msg.command==CONS_CONNECTED)
				msg.command=CONS_CONNECTING;
		}
		if (pipe_send(item->pwrite, msg)) {
			notice("sending cCONNECT message failed\n");
			return TRUE;
		}
		break;
    	case cDISCONNECT:
		if (state->pon==1)
			firstlastp = &pid->last_ondisc;
		if (!item_con(msg.from, firstlastp, start, FALSE))
			state->pon--;
		break;
	case cAUTH:
		item_auth(msg.from, msg.username, start);
		break;
	case cDEAUTH:
		item_auth(msg.from, "%unknown", start);
		break;
	case cLIST:
		if (send_clients_list(item->pwrite, sokcount, start->next)) {
			notice("sending LIST message failed\n");
			return TRUE;
		}
		break;
	case cRESTART:
		*time_to_hangup = QRESTART;
		break;
	case cUPTIME:
		(void)time_send(item->pwrite, connect_time);
		break;
    	default: 
		notice_debug("do_child_msg: message %d invalid\n", msg.command);
		return TRUE;
	}
	return FALSE;
}

bool time_send(int pwrite, time_t connect_time)
{
	struct timeval tv;
	tv.tv_sec = 5;
	tv.tv_usec = 0;
	
	switch(sok_send(pwrite, &tv, &connect_time, sizeof(connect_time)))
	{
	case SOK_SUCCESS: break;
	case SOK_FAILED:
		notice_debug("time_send failed\n");
		return TRUE;
	case SOK_TIMEOUT:
		notice_debug("time_send timeout\n");
		return TRUE;
	}
	return FALSE;
}

bool pipe_send(int pwrite, struct pipe_msg_buf msg)
{
	struct timeval tv;
	tv.tv_sec = 5;
	tv.tv_usec = 0;
	
	switch(sok_send(pwrite, &tv, &msg, sizeof(struct pipe_msg_buf))) {
	case SOK_SUCCESS: break;
	case SOK_FAILED:
		notice_debug("pipe_send failed\n");
		return TRUE;
	case SOK_TIMEOUT:
		notice_debug("pipe_send timeout\n");
		return TRUE;
	}
	return FALSE;
}

bool pipe_recv(int pread, struct pipe_msg_buf *msg)
{
	struct timeval tv;
	tv.tv_sec = 5;
	tv.tv_usec = 0;
	switch(sok_recv(pread, &tv, msg, sizeof(struct pipe_msg_buf))) {
	case SOK_SUCCESS: break;
	case SOK_FAILED:
		 notice_debug("pipe_recv failed\n");
		 return TRUE;
	case SOK_TIMEOUT:
		 notice_debug("pipe_recv timeout\n");
		 return TRUE;
	}
	return FALSE;
}

#if 0
 /* FIXME: This is an extremely slow way of comparing dotted quads. */
bool max_per_host_equalled(struct clients *first, const char *peername)
{
	int c = 0;
	if (set.max_from_one==-1)
		return FALSE; /* unlimited */
	while (first) {
		if (c >= set.max_from_one)
			return TRUE;
		if (0==strcmp(first->peername, peername))
			++c;
		first = first->next;
	}
	return FALSE;
}
#endif

/* Internal/external functions.
 *
 * (These allow programs to send dwun messages which are then sent to
 * specified clients.)
 *
 * open_external	- open the set.external FIFO.
 * open_internal	- setup the pipes to be passed as fd3 to some executed
 * 			  programs.
 * sendtoall_pipe	- get a message from internal or external and send it
 * 			  to the specified clients.
 */

SOK *open_external(void)
{
	SOK *stream;
	int fd;
	/* Opening a FIFO O_RDWR is undefined in POSIX but it's okay on most
	 * *nix systems (where is it not?).
	 *
	 * If we didn't do this, we'd have to close then open the fifo 
	 * everytime we received a message from things like echo. (And
	 * potentially lose messages). */
	fd = open(set.external, O_RDWR | O_NONBLOCK);
	if (-1==fd) {
		notice_err("opening external '%s' for reading and writing "
				"failed", set.external);
		return NULL;
	}
	stream = sdopen(fd);
	stream->out.bufsize = 0; /* disable ssend */
	return stream;
}

SOK *open_internal(void)
{
	SOK *stream;
	int filedes[2];

	if (make_pipes(filedes))
		return NULL;
	internal_write_fd = filedes[1];
	stream = sdopen(filedes[0]);
	stream->out.bufsize = 0; /* disable ssend */
	return stream;
}

/* type = "external" or "internal" */
/* XXX: change fn name */
int sendtoall_pipe(SOK *stream, struct clients *start, const char *type)
{
	char buf[MAXLINE];
	char *message;
	char *to;
	u_long pid;

	switch(srecv(stream, buf, MAXLINE-1)) {
	case SOK_SUCCESS: 
		strcat(buf, "\n"); /* srecv removes \n, we want it back */
		/* format is "user:message\n" or "pid:message\n" */
		if (!(message = strchr(buf, ':'))) {
			notice("skipping %s message missing colon\n", type);
			notice_debug("failed %s message: %s", type, buf);
			return 0;
		}
		*message++ = '\0';
		to = buf;
#if 1
		if (was_a_username(to)) {
			notice("warning: you should now use '%s' instead of "
					"'%s'\n", to+1, to);
			(void)sendtousers(to+1, message, start->next); 
#endif
		} else if (is_a_username(to[0])) {
			(void)sendtousers(to, message, start->next); 
		} else {
			if (string_isdigit((unsigned char *)to) ||
				mystrtoul(to, &pid)) {
				notice_debug("failed %s message to '%s': "
						"not a username or PID\n",
						type, to);
				return 0;
			}
			(void)sendtopid((pid_t)pid, message, start->next);
		}
		return 0;
	case SOK_TIMEOUT:
		notice("timeout reading from %s\n", type);
		return 0;
	case SOK_TOOLONG:
		notice("skipping %s message longer than %d characters\n", type,
				MAXLINE-2); /* don't include \n\0 in count */
		if (discard(stream, buf, (ssize_t)MAXLINE)) {
			notice_debug("partial %s message left in buffer\n",
					type);
		}
		return 0;
	default:
		notice("reading from %s failed\n", type);
		return -1; /* disable */
	}
}

/* Signal functions.
 *
 * sigsetup	- associate sighandler with signals and ignore SIGPIPE.
 * sighandler	- handles signals when they come.
 */

bool sigsetup(void)
{
	struct sigaction ignore;
	struct sigaction cursig;
	int i;
	int handled[] = { SIGHUP, CONNECT_FAILED_SIGNAL, SIGUSR1, SIGUSR2,
		SIGCHLD, SIGALRM, SIGSEGV, 0 };
		
	memset(&ignore, 0, sizeof(ignore));
	memset(&cursig, 0, sizeof(cursig));
	
	ignore.sa_handler = SIG_IGN;
	cursig.sa_handler = sighandler;
#ifndef DEBUG_GENERAL
	/* Since my dev platform supports SA_RESTART, I still want to catch
	 * any calls I've missed for the sake of other platforms. */
# ifdef SA_RESTART
	/* We still check for EINTR from appropriate calls. */
	cursig.sa_flags = SA_RESTART;
# endif
#endif
	block_all_signals(NULL);
	for (i=0; handled[i]; ++i) {
		if (-1==sigaction(handled[i], &cursig, NULL)) {
			notice_err("setting up signal handlers failed");
			return TRUE;
		}
	}
	if (-1==sigaction(SIGPIPE, &ignore, NULL) ||
			/* So we can do kill(0, SIGTERM) */
			-1==sigaction(SIGTERM, &ignore, NULL)) {
		notice_err("ignoring signals failed");
		return TRUE;
	}
	unblock_all_signals(NULL);
	return FALSE;
}

void sighandler(int sig)
{
	/* Not that the kernel may "combine" several of the same signal into
	 * one. (e.g. one SIGCHLD, many children dead.) */
	struct handled_s {
		int sig;
		int c;
	} handled[] = {
		{ SIGHUP, ACT_HANGUP },
		{ CONNECT_FAILED_SIGNAL, ACT_FATAL },
		{ SIGUSR1, ACT_CONNECTED },
		{ SIGUSR2, ACT_RECONNECT },
		{ SIGCHLD, ACT_CHILDDEAD },
		{ SIGALRM, ACT_CANDIAL },
		{ 0, 0 }
	};
	int i;
	for (i=0; handled[i].sig; ++i) {
		if (handled[i].sig==sig) {
			sigwrite(sighandler_write_pipe, handled[i].c);
			return;
		}
	}
	if (sig==SIGSEGV) {
		/* notice() may not be safe in a signal handler, but since
		 * we're segfaulting :) */
		notice("exiting with SIGSEGV\n");
		kill(0, SIGTERM);
		_exit(EXIT_FAILURE);
	}
#ifdef DEBUG_GDB
	notice_debug("sighandler: unknown signal caught\n");
#endif
	return;
}
