#include "bool.h"
#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <sys/stat.h>
#include <unistd.h>
#include <sys/mman.h>
#include <fcntl.h>
#include <string.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <limits.h>
#include <errno.h>
#include <time.h>
#include "shared.h"
#include "ipc.h"
#include "socket.h"
#include "main.h"
#include "util.h"

static bool pipe_send(int pwrite, struct pipe_msg_buf msg);
static bool pipen_send(int pwrite, void *msg, ssize_t size);

bool map_authfile(mmap_area *authfile)
{
	struct stat buf;
	int fd = open(set.authfile, O_RDWR);
	if (-1==fd) {
		notice_err("opening authfile '%s' for reading and writing "
				"failed", set.authfile); 
		return TRUE;
	}
	if (-1==fstat(fd, &buf)) {
		notice("getting authfile '%s' status failed\n", set.authfile);
		(void)close(fd);
		return TRUE;
	}
	if ((int)buf.st_size==0) {
		notice("reading authfile '%s' failed: file length is 0\n", 
				set.authfile);
		(void)close(fd);
		return TRUE;
	}
	authfile->len = buf.st_size;
	authfile->data = (char *)mmap(0, authfile->len, PROT_READ, MAP_SHARED,
			fd, 0);
	close(fd);
	if (authfile->data==(char *)-1) {
		notice_err("mapping authfile '%s' into memory failed",
				set.authfile);
		return TRUE;
	}
	if (invalid_authfile(*authfile))
		return TRUE;
	return FALSE;
}

/* unmap from memory, updating file. */
bool unmap_authfile(mmap_area authfile)
{
	bool error = FALSE;
	if (!authfile.data)
		return TRUE;
	/* XXX: Is this alignment necessary? */
	if (authfile.len % getpagesize() != 0)
		authfile.len += getpagesize() - (authfile.len % getpagesize());

	if (-1==msync(authfile.data, authfile.len, MS_ASYNC)) {
		notice_err("updating authfile failed");
		error = TRUE;
	}
	if (-1==munmap(authfile.data, authfile.len)) {
		notice_err("unmapping authfile from memory failed");
		error = TRUE;
	}
	return error;
}

/* set/unset authfile writeable */
bool writeable_authfile(bool writeable, mmap_area authfile)
{
	int flags = PROT_READ;
	if (writeable)
		flags |= PROT_WRITE;
	return -1==mprotect(authfile.data, authfile.len, flags);
}

bool send_clients_list(int pwrite, int sokcount, struct clients *first)
{
	struct pipe_msg_buf msg;
	struct clientlist tosend;

	/* Send number of clients */
	memset(&msg, 0, sizeof(msg));
	msg.from = getpid();
	msg.command = sokcount;
	if (pipe_send(pwrite, msg))
		return TRUE;

	/* Send client list */
	while (first) {
		if (first->sok==-1) {
			first = first->next;
			continue;
		}
		tosend.pid = first->pid;
		tosend.connected = first->connected;
		term_strncpy(tosend.username, first->username, USER_LEN);
		term_strncpy(tosend.peername, first->peername, PEER_LEN);
		if (pipen_send(pwrite, &tosend, sizeof(tosend)))
			return TRUE;
		first = first->next;
	}
	return FALSE;
}

/* Creates non-blocking pipes. */
bool make_pipes(int filedes[2])
{
	int i;
	if (-1==pipe(filedes)) {
		notice_err("creating pipes failed");
		return TRUE;
	}
	for (i=0; i<2; ++i) {
		if (-1==fcntl(filedes[i], F_SETFL, O_NONBLOCK)) {
			notice_err("setting pipe non-blocking failed");
			close(filedes[0]); close(filedes[1]);
			return TRUE;
		}
	}
	return FALSE;
}

bool pipen_send(int pwrite, void *msg, ssize_t size)
{
	struct timeval tv;
	tv.tv_sec = IPC_SEND_TIMEOUT_SECS;
	tv.tv_usec = 0;
	switch(sok_send(pwrite, &tv, msg, size)) {
	case SOK_SUCCESS: break;
	case SOK_FAILED: 
		notice_debug("pipe_send failed\n");
		return TRUE;
	case SOK_TIMEOUT: 
		notice_debug("pipe_send timeout\n");
		return TRUE;
	}
	return FALSE;
}

bool pipe_send(int pwrite, struct pipe_msg_buf msg)
{
	return pipen_send(pwrite, &msg, sizeof(struct pipe_msg_buf));
}

/* A single pair of pipes is used for all communication between child.c
 * children and the network.c parent.
 *
 * pipe_msg_buf and clientlist are the only things which are sent over these
 * pipes. We must check that these can be sent atomically.
 */
bool smallpipe(int filedes)
{
	size_t maxatomic;
	size_t len1 = sizeof(struct pipe_msg_buf);
	size_t len2 = sizeof(struct clientlist);
#ifdef PIPE_BUF
	if(filedes){}; /* avoid compiler warning */
	maxatomic = (size_t)PIPE_BUF;
#else
	if (-1==(maxatomic = (size_t)fpathconf(filedes, _PC_PIPE_BUF))) {
		notice_err("fpathconf failed");
		return TRUE;
	}
#endif
	if (len1 > maxatomic || len2 > maxatomic) {
		/* Since the other members are insignificant. (And PEER_LEN is
		 * not user-configurable.) */
		notice("USER_LEN in ipc.h is too large\n");
		return TRUE;
	}
	return FALSE;
}
