/*                                                     -*- linux-c -*-
    Copyright (C) 2004 Tom Szilagyi

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: editor.c,v 1.15 2004/06/17 18:12:07 tszilagyi Exp $
*/


#include <errno.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <gtk/gtk.h>
#include <jack/jack.h>

#include "common.h"
#include "file_io.h"
#include "gui_ext.h"
#include "sound.h"
#include "ports.h"
#include "export.h"
#include "about.h"
#include "editor.h"


/* size of combs & allpass filters graph */
#define DRAW_WIDTH     400
#define DRAW_HEIGHT    150


REVTYPE * curr = NULL;
REVTYPE * root = NULL;


extern float decay;
extern float drylevel;
extern float wetlevel;
extern int combs_en; /* on/off */
extern int allps_en; /* on/off */
extern int bandps_en; /* on/off */
extern int stereo_en; /* on/off */
extern int bypass; /* on/off */

/* to track if we need to recalculate DSP parameters on next buffer process(): */
extern int changed_settings;

int client_running = 0;

int current_comb = 0;
int current_allp = 0;

float impresp_data[IMPRESP_MAXLEN * MAX_SAMPLERATE / 1000];
long int impresp_time = 1000;


extern int port_dialog_open;
extern GtkWidget * ports_window;


GList * revtype_list = NULL;
GtkWidget * main_window;
GtkWidget * notebook;
GtkWidget * combo_revtype;
GtkObject * adj_combs_delay;
GtkObject * adj_combs_feedback;
GtkObject * adj_combs_freqresp;
GtkObject * adj_allps_delay;
GtkObject * adj_allps_feedback;
GtkObject * adj_bandps_hi;
GtkObject * adj_bandps_lo;
GtkWidget * draw_combs;
GtkWidget * draw_allps;
GtkWidget * draw_impresp;
GtkObject * adj_impresp_time;
GtkWidget * ports_button;
GtkWidget * export_button;
GtkWidget * about_button;
GtkWidget * toggle_combs;
GtkWidget * toggle_allps;
GtkWidget * toggle_bandps;
GtkWidget * toggle_enhst;
GtkWidget * toggle_bypass;

static GdkPixmap * pixmap_combs = NULL;
static GdkPixmap * pixmap_allps = NULL;
static GdkPixmap * pixmap_impresp = NULL;

gint tag_running;

void update_combo(REVTYPE * root);
void display_revtype(void);
void redraw_combs(void);
void redraw_allps(void);
void redraw_impresp(long int nframes);



int
main_window_close(GtkWidget *widget, gpointer * data) {

	if (confirm_exit_box("Quit",
			     "   Are you sure you want to quit?   \n\n"
			     "   If you have unsaved Reverb Types,   \n"
			     "   you may go back and save them now.   ",
			     main_window, GTK_MESSAGE_QUESTION)) {
		return 1;
	} else {
		gtk_main_quit();
		return 0;
	}
}



void
toggle_combs_callback(GtkWidget *widget, gpointer data) {

	GdkColor color;

	if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON (widget))) {

		color.red = G_ACTIVE_R;
		color.green = G_ACTIVE_G;
		color.blue = G_ACTIVE_B;
		gtk_widget_modify_bg(GTK_WIDGET(toggle_combs), GTK_STATE_ACTIVE, &color);

		color.red = G_PRELIGHT_R;
		color.green = G_PRELIGHT_G;
		color.blue = G_PRELIGHT_B;
		gtk_widget_modify_bg(GTK_WIDGET(toggle_combs), GTK_STATE_PRELIGHT, &color);

		combs_en = 1;
    	} else {

		color.red = PRELIGHT_R;
		color.green = PRELIGHT_G;
		color.blue = PRELIGHT_B;
		gtk_widget_modify_bg(GTK_WIDGET(toggle_combs), GTK_STATE_PRELIGHT, &color);

		combs_en = 0;
	}
}

void
toggle_allps_callback(GtkWidget *widget, gpointer data) {

	GdkColor color;

	if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON (widget))) {

		color.red = G_ACTIVE_R;
		color.green = G_ACTIVE_G;
		color.blue = G_ACTIVE_B;
		gtk_widget_modify_bg(GTK_WIDGET(toggle_allps), GTK_STATE_ACTIVE, &color);

		color.red = G_PRELIGHT_R;
		color.green = G_PRELIGHT_G;
		color.blue = G_PRELIGHT_B;
		gtk_widget_modify_bg(GTK_WIDGET(toggle_allps), GTK_STATE_PRELIGHT, &color);

		allps_en = 1;
    	} else {

		color.red = PRELIGHT_R;
		color.green = PRELIGHT_B;
		color.blue = PRELIGHT_B;
		gtk_widget_modify_bg(GTK_WIDGET(toggle_allps), GTK_STATE_PRELIGHT, &color);

		allps_en = 0;
	}
}

void
toggle_bandps_callback(GtkWidget *widget, gpointer data) {

	GdkColor color;

	if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON (widget))) {

		color.red = G_ACTIVE_R;
		color.green = G_ACTIVE_G;
		color.blue = G_ACTIVE_B;
		gtk_widget_modify_bg(GTK_WIDGET(toggle_bandps), GTK_STATE_ACTIVE, &color);

		color.red = G_PRELIGHT_R;
		color.green = G_PRELIGHT_G;
		color.blue = G_PRELIGHT_B;
		gtk_widget_modify_bg(GTK_WIDGET(toggle_bandps), GTK_STATE_PRELIGHT, &color);

		bandps_en = 1;
    	} else {

		color.red = PRELIGHT_R;
		color.green = PRELIGHT_G;
		color.blue = PRELIGHT_B;
		gtk_widget_modify_bg(GTK_WIDGET(toggle_bandps), GTK_STATE_PRELIGHT, &color);

		bandps_en = 0;
	}
}

void
toggle_enhst_callback(GtkWidget *widget, gpointer data) {

	GdkColor color;

	if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON (widget))) {

		color.red = G_ACTIVE_R;
		color.green = G_ACTIVE_G;
		color.blue = G_ACTIVE_B;
		gtk_widget_modify_bg(GTK_WIDGET(toggle_enhst), GTK_STATE_ACTIVE, &color);

		color.red = G_PRELIGHT_R;
		color.green = G_PRELIGHT_G;
		color.blue = G_PRELIGHT_B;
		gtk_widget_modify_bg(GTK_WIDGET(toggle_enhst), GTK_STATE_PRELIGHT, &color);

		stereo_en = 1;
    	} else {

		color.red = PRELIGHT_R;
		color.green = PRELIGHT_G;
		color.blue = PRELIGHT_B;
		gtk_widget_modify_bg(GTK_WIDGET(toggle_enhst), GTK_STATE_PRELIGHT, &color);

		stereo_en = 0;
	}

	changed_settings = 1;
}

void
toggle_bypass_callback(GtkWidget *widget, gpointer data) {

	GdkColor color;

	if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON (widget))) {

		color.red = R_ACTIVE_R;
		color.green = R_ACTIVE_G;
		color.blue = R_ACTIVE_B;
		gtk_widget_modify_bg(GTK_WIDGET(toggle_bypass), GTK_STATE_ACTIVE, &color);

		color.red = R_PRELIGHT_R;
		color.green = R_PRELIGHT_G;
		color.blue = R_PRELIGHT_B;
		gtk_widget_modify_bg(GTK_WIDGET(toggle_bypass), GTK_STATE_PRELIGHT, &color);

		bypass = 1;
    	} else {

		color.red = PRELIGHT_R;
		color.green = PRELIGHT_G;
		color.blue = PRELIGHT_B;
		gtk_widget_modify_bg(GTK_WIDGET(toggle_bypass), GTK_STATE_PRELIGHT, &color);

		bypass = 0;
	}

	sound_init();
	changed_settings = 1;
}

void
changed_decay(GtkAdjustment *adj, gpointer data) {

	decay = gtk_adjustment_get_value(adj);
	changed_settings = 1;
}

void
changed_drylevel(GtkAdjustment *adj, gpointer data) {

	drylevel = gtk_adjustment_get_value(adj);
}

void
changed_wetlevel(GtkAdjustment *adj, gpointer data) {

	wetlevel = gtk_adjustment_get_value(adj);
}

void
changed_revtype(GtkWidget *widget, gpointer * data) {

	const gchar * string;
	REVTYPE * item;

	string = gtk_entry_get_text(GTK_ENTRY(widget));

	if (string[0] == '\0')
		return;

	if ((item = get_revtype_by_name(root, string)) != NULL) {
		curr = item;
		display_revtype();
		changed_settings = 1;
	}
}

void
clicked_ports(GtkWidget *widget, gpointer * data) {

	gtk_widget_set_sensitive(GTK_WIDGET(ports_button), FALSE);
	port_setup_dialog();
}

void
clicked_export(GtkWidget *widget, gpointer * data) {

	gtk_widget_set_sensitive(GTK_WIDGET(export_button), FALSE);
	export_setup_dialog();
}

void
clicked_about(GtkWidget *widget, gpointer * data) {

	gtk_widget_set_sensitive(GTK_WIDGET(about_button), FALSE);
	about_dialog();
}


void
clicked_new(GtkWidget *widget, gpointer * data) {

	char str[MAXLEN];
	REVTYPE * item = NULL;
	REVTYPE * prev = NULL;
	int i = 0;

	if (!input_box("New Reverb Type",
		       "   Please enter the name of the new Reverb Type:   ", "", str,
		       main_window, GTK_MESSAGE_QUESTION))
		return;

	if (str[0] == '\0')
		return;

	if (get_revtype_by_name(root, str)) {
		message_box("Notice",
			    "   There is an existing Reverb Type with this name.   ",
			    main_window, GTK_MESSAGE_WARNING);
		return;
	}

	/* create a new reverb type */
	if ((item = malloc(sizeof(REVTYPE))) == NULL) {
		fprintf(stderr, "editor.c: malloc failed.\n");
		return;
	}
	item->next = NULL;
	strcpy(item->name, str);
	item->num_combs = 1;
	item->num_allps = 1;
	item->combs_data[0] = 1.0f;
	item->combs_data[1] = 75.0f;
	item->combs_data[2] = 0.5f;
	item->allps_data[0] = 1.0f;
	item->allps_data[1] = 75.0f;
	item->bandps_lo = 100.0f;
	item->bandps_hi = 10000.0f;

	/* insert item into list */
	if (root == NULL)
		return;
	prev = root;
	while ((prev->next != NULL) && (strcmp(prev->next->name, str) < 0)) {
		prev = prev->next;
		i++;
	}
	item->next = prev->next;
	prev->next = item;

	update_combo(root);
	gtk_list_unselect_all(GTK_LIST((GTK_COMBO(combo_revtype))->list));
	gtk_list_select_item(GTK_LIST((GTK_COMBO(combo_revtype))->list), i);
	curr = item;
	display_revtype();
}

void
clicked_copy(GtkWidget *widget, gpointer * data) {

	char str[MAXLEN];
	REVTYPE * item = NULL;
	REVTYPE * prev = NULL;
	int i = 0, j = 0;

	if (!curr)
		return;

	if (!input_box("Copy Reverb Type",
		       "   Please enter the name for the new copy of this Reverb Type:   ",
		       curr->name, str,
		       main_window, GTK_MESSAGE_QUESTION))
		return;

	if (strcmp(curr->name, str) == 0) {
		message_box("Notice",
			    "   You have to specify a different name to copy to.   ",
			    main_window, GTK_MESSAGE_WARNING);
		return;
	}

	if (get_revtype_by_name(root, str)) {
		message_box("Notice",
			    "   There is an existing Reverb Type with this name.   \n"
			    "   If you would like to overwrite it, please delete it first.   ",
			    main_window, GTK_MESSAGE_WARNING);
		return;
	}

	/* create a new reverb type & copy the current one to it */
	if ((item = malloc(sizeof(REVTYPE))) == NULL) {
		fprintf(stderr, "editor.c: malloc failed.\n");
		return;
	}
	item->next = NULL;
	strcpy(item->name, str);
	item->num_combs = curr->num_combs;
	item->num_allps = curr->num_allps;
	for (i = 0; i < curr->num_combs; i++) {
		item->combs_data[3 * i] = curr->combs_data[3 * i];
		item->combs_data[3 * i + 1] = curr->combs_data[3 * i + 1];
		item->combs_data[3 * i + 2] = curr->combs_data[3 * i + 2];
	}
	for (i = 0; i < curr->num_allps; i++) {
		item->allps_data[2 * i] = curr->allps_data[2 * i];
		item->allps_data[2 * i + 1] = curr->allps_data[2 * i + 1];
	}
	item->bandps_lo = curr->bandps_lo;
	item->bandps_hi = curr->bandps_hi;

	/* insert item into list */
	if (root == NULL)
		return;
	prev = root;
	while ((prev->next != NULL) && (strcmp(prev->next->name, str) < 0)) {
		prev = prev->next;
		j++;
	}
	item->next = prev->next;
	prev->next = item;

	update_combo(root);
	gtk_list_unselect_all(GTK_LIST((GTK_COMBO(combo_revtype))->list));
	gtk_list_select_item(GTK_LIST((GTK_COMBO(combo_revtype))->list), j);
	curr = item;
	display_revtype();
}

void
clicked_rename(GtkWidget *widget, gpointer * data) {

	char str[MAXLEN];
	REVTYPE * item = NULL;
	REVTYPE * prev = NULL;
	int i = 0;

	if (!input_box("Rename Reverb Type",
		       "   Please enter the new name for this Reverb Type:   ", curr->name, str,
		       main_window, GTK_MESSAGE_QUESTION))
		return;

	if (str[0] == '\0')
		return;

	if (get_revtype_by_name(root, str)) {
		message_box("Notice",
			    "   There is an existing Reverb Type with this name.   ",
			    main_window, GTK_MESSAGE_WARNING);
		return;
	}

	item = curr;
	strcpy(item->name, str);

	/* remove item from list */
	if (root == NULL)
		return;
	prev = root;
	while (prev->next != item)
		prev = prev->next;
	prev->next = item->next;

	/* insert item into list */
	prev = root;
	while ((prev->next != NULL) && (strcmp(prev->next->name, str) < 0)) {
		prev = prev->next;
		i++;
	}
	item->next = prev->next;
	prev->next = item;

	update_combo(root);
	gtk_list_unselect_all(GTK_LIST((GTK_COMBO(combo_revtype))->list));
	gtk_list_select_item(GTK_LIST((GTK_COMBO(combo_revtype))->list), i);
	curr = item;
	display_revtype();
}

void
clicked_del(GtkWidget *widget, gpointer * data) {

	REVTYPE * item = NULL;
	REVTYPE * prev = NULL;

	if (confirm_box("Delete Reverb Type",
			"   Are you sure you want to delete this Reverb Type?   ",
			main_window, GTK_MESSAGE_WARNING))
		return;

	item = curr;

	if ((item == root->next) && (item->next == NULL)) {
		message_box("Notice",
			    "   This is the only Reverb Type. We won't delete it.   ",
			    main_window, GTK_MESSAGE_WARNING);
		return;
	}

	/* remove item from list */
	if (root == NULL)
		return;
	prev = root;
	while (prev->next != item)
		prev = prev->next;
	prev->next = item->next;

	free(item);

	update_combo(root);
	gtk_list_unselect_all(GTK_LIST((GTK_COMBO(combo_revtype))->list));
	gtk_list_select_item(GTK_LIST((GTK_COMBO(combo_revtype))->list), 0);
	curr = root->next;
	display_revtype();
}


void
changed_combs_delay(GtkAdjustment *adj, gpointer data) {

	if (curr == NULL)
		return;
	curr->combs_data[3 * current_comb] = gtk_adjustment_get_value(adj);
	redraw_combs();
	changed_settings = 1;
}

void
changed_combs_feedback(GtkAdjustment *adj, gpointer data) {

	if (curr == NULL)
		return;
	curr->combs_data[3 * current_comb + 1] = gtk_adjustment_get_value(adj);
	redraw_combs();
	changed_settings = 1;
}

void
changed_combs_freqresp(GtkAdjustment *adj, gpointer data) {

	if (curr == NULL)
		return;
	curr->combs_data[3 * current_comb + 2] = gtk_adjustment_get_value(adj);
	redraw_combs();
	changed_settings = 1;
}

void
changed_allps_delay(GtkAdjustment *adj, gpointer data) {

	if (curr == NULL)
		return;
	curr->allps_data[2 * current_allp] = gtk_adjustment_get_value(adj);
	redraw_allps();
	changed_settings = 1;
}

void
changed_allps_feedback(GtkAdjustment *adj, gpointer data) {

	if (curr == NULL)
		return;
	curr->allps_data[2 * current_allp + 1] = gtk_adjustment_get_value(adj);
	redraw_allps();
	changed_settings = 1;
}

void
changed_bandps_lo(GtkAdjustment *adj, gpointer data) {

	if (curr == NULL)
		return;
	curr->bandps_lo = gtk_adjustment_get_value(adj);
	changed_settings = 1;
}

void
changed_bandps_hi(GtkAdjustment *adj, gpointer data) {

	if (curr == NULL)
		return;
	curr->bandps_hi = gtk_adjustment_get_value(adj);
	changed_settings = 1;
}

void
combs_new_clicked(GtkWidget *widget, gpointer * data) {

	if (curr->num_combs == MAX_COMBS) {
		message_box("Notice",
			    "   Maximum number of comb filters reached, can't add more.   ",
			    main_window, GTK_MESSAGE_WARNING);
		return;
	}

	curr->num_combs++;
	current_comb = curr->num_combs - 1;
	curr->combs_data[3 * current_comb] = 1.0f;
	curr->combs_data[3 * current_comb + 1] = 75.0f;
	curr->combs_data[3 * current_comb + 2] = 0.5f;
	gtk_adjustment_set_value(GTK_ADJUSTMENT(adj_combs_delay),
				 curr->combs_data[3 * current_comb]);
	gtk_adjustment_set_value(GTK_ADJUSTMENT(adj_combs_feedback),
				 curr->combs_data[3 * current_comb + 1]);
	gtk_adjustment_set_value(GTK_ADJUSTMENT(adj_combs_freqresp),
				 curr->combs_data[3 * current_comb + 2]);
	redraw_combs();
	changed_settings = 1;
}

void
combs_del_clicked(GtkWidget *widget, gpointer * data) {

	int i;

	if (curr->num_combs == 1) {
		message_box("Notice",
			    "   Can't remove last comb filter.   ",
			    main_window, GTK_MESSAGE_WARNING);
		return;
	}

	for (i = current_comb; i < curr->num_combs - 1; i++) {
		curr->combs_data[3 * i] = curr->combs_data[3 * (i + 1)];
		curr->combs_data[3 * i + 1] = curr->combs_data[3 * (i + 1) + 1];
		curr->combs_data[3 * i + 2] = curr->combs_data[3 * (i + 1) + 2];
	}
	
	curr->num_combs--;
	if (current_comb == curr->num_combs)
		current_comb--;

	gtk_adjustment_set_value(GTK_ADJUSTMENT(adj_combs_delay),
				 curr->combs_data[3 * current_comb]);
	gtk_adjustment_set_value(GTK_ADJUSTMENT(adj_combs_feedback),
				 curr->combs_data[3 * current_comb + 1]);
	gtk_adjustment_set_value(GTK_ADJUSTMENT(adj_combs_freqresp),
				 curr->combs_data[3 * current_comb + 2]);
	redraw_combs();
	changed_settings = 1;
}


void
combs_next_clicked(GtkWidget *widget, gpointer * data) {

	current_comb++;
	if (current_comb == curr->num_combs)
		current_comb = 0;
	gtk_adjustment_set_value(GTK_ADJUSTMENT(adj_combs_delay),
				 curr->combs_data[3 * current_comb]);
	gtk_adjustment_set_value(GTK_ADJUSTMENT(adj_combs_feedback),
				 curr->combs_data[3 * current_comb + 1]);
	gtk_adjustment_set_value(GTK_ADJUSTMENT(adj_combs_freqresp),
				 curr->combs_data[3 * current_comb + 2]);
	redraw_combs();
}

void
combs_prev_clicked(GtkWidget *widget, gpointer * data) {

	if (current_comb == 0)
		current_comb = curr->num_combs - 1;
	else
		current_comb--;
	gtk_adjustment_set_value(GTK_ADJUSTMENT(adj_combs_delay),
				 curr->combs_data[3 * current_comb]);
	gtk_adjustment_set_value(GTK_ADJUSTMENT(adj_combs_feedback),
				 curr->combs_data[3 * current_comb + 1]);
	gtk_adjustment_set_value(GTK_ADJUSTMENT(adj_combs_freqresp),
				 curr->combs_data[3 * current_comb + 2]);
	redraw_combs();
}

void
allps_new_clicked(GtkWidget *widget, gpointer * data) {

	if (curr->num_allps == MAX_ALLPS) {
		message_box("Notice",
			    "   Maximum number of allpass filters reached, can't add more.   ",
			    main_window, GTK_MESSAGE_WARNING);
		return;
	}

	curr->num_allps++;
	current_allp = curr->num_allps - 1;
	curr->allps_data[2 * current_allp] = 1.0f;
	curr->allps_data[2 * current_allp + 1] = 75.0f;
	gtk_adjustment_set_value(GTK_ADJUSTMENT(adj_allps_delay),
				 curr->allps_data[2 * current_allp]);
	gtk_adjustment_set_value(GTK_ADJUSTMENT(adj_allps_feedback),
				 curr->allps_data[2 * current_allp + 1]);
	redraw_allps();
	changed_settings = 1;
}

void
allps_del_clicked(GtkWidget *widget, gpointer * data) {

	int i;

	if (curr->num_allps == 1) {
		message_box("Notice",
			    "   Can't remove last allpass filter.   ",
			    main_window, GTK_MESSAGE_WARNING);
		return;
	}

	for (i = current_allp; i < curr->num_allps - 1; i++) {
		curr->allps_data[2 * i] = curr->allps_data[2 * (i + 1)];
		curr->allps_data[2 * i + 1] = curr->allps_data[2 * (i + 1) + 1];
	}
	
	curr->num_allps--;
	if (current_allp == curr->num_allps)
		current_allp--;

	gtk_adjustment_set_value(GTK_ADJUSTMENT(adj_allps_delay),
				 curr->allps_data[2 * current_allp]);
	gtk_adjustment_set_value(GTK_ADJUSTMENT(adj_allps_feedback),
				 curr->allps_data[2 * current_allp + 1]);
	redraw_allps();
	changed_settings = 1;
}

void
allps_next_clicked(GtkWidget *widget, gpointer * data) {

	current_allp++;
	if (current_allp == curr->num_allps)
		current_allp = 0;
	gtk_adjustment_set_value(GTK_ADJUSTMENT(adj_allps_delay),
				 curr->allps_data[2 * current_allp]);
	gtk_adjustment_set_value(GTK_ADJUSTMENT(adj_allps_feedback),
				 curr->allps_data[2 * current_allp + 1]);
	redraw_allps();
}

void
allps_prev_clicked(GtkWidget *widget, gpointer * data) {

	if (current_allp == 0)
		current_allp = curr->num_allps - 1;
	else
		current_allp--;
	gtk_adjustment_set_value(GTK_ADJUSTMENT(adj_allps_delay),
				 curr->allps_data[2 * current_allp]);
	gtk_adjustment_set_value(GTK_ADJUSTMENT(adj_allps_feedback),
				 curr->allps_data[2 * current_allp + 1]);
	redraw_allps();
}

void
impresp_recalc_clicked(GtkWidget *widget, gpointer * data) {

	long int i;
	long int nframes = impresp_time * sample_rate / 1000;

	impresp_data[0] = 1.0f;
	for (i = 1; i < nframes; i++)
		impresp_data[i] = 0.0f;

	process_impresp(impresp_data, nframes);
	redraw_impresp(nframes);
}

void
changed_impresp_time(GtkAdjustment *adj, gpointer data) {

	impresp_time = gtk_adjustment_get_value(adj);
}


static
gint
configure_combs(GtkWidget *widget, GdkEventConfigure *event) {

	GdkColor color;
	GdkGC * gc;

	if (pixmap_combs)
		gdk_pixmap_unref(pixmap_combs);

	pixmap_combs = gdk_pixmap_new(widget->window,
				widget->allocation.width,
				widget->allocation.height,
				-1);
	color.red = 0;
	color.green = 0;
	color.blue = 0;
	gc = gdk_gc_new(pixmap_combs);
	gdk_gc_set_rgb_fg_color(gc, &color);
	gdk_draw_rectangle(pixmap_combs,
			   gc,
			   TRUE,
			   0, 0,
			   widget->allocation.width,
			   widget->allocation.height);
	return TRUE;
}

static
gint
expose_combs(GtkWidget *widget, GdkEventExpose *event) {

	gdk_draw_pixmap(widget->window,
			widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
			pixmap_combs,
			event->area.x, event->area.y,
			event->area.x, event->area.y,
			event->area.width, event->area.height);
	return FALSE;
}

static
gint
configure_allps(GtkWidget *widget, GdkEventConfigure *event) {

	GdkColor color;
	GdkGC * gc;

	if (pixmap_allps)
		gdk_pixmap_unref(pixmap_allps);

	pixmap_allps = gdk_pixmap_new(widget->window,
				widget->allocation.width,
				widget->allocation.height,
				-1);
	color.red = 0;
	color.green = 0;
	color.blue = 0;
	gc = gdk_gc_new(pixmap_allps);
	gdk_gc_set_rgb_fg_color(gc, &color);
	gdk_draw_rectangle(pixmap_allps,
			   gc,
			   TRUE,
			   0, 0,
			   widget->allocation.width,
			   widget->allocation.height);
	return TRUE;
}

static
gint
expose_allps(GtkWidget *widget, GdkEventExpose *event) {

	gdk_draw_pixmap(widget->window,
			widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
			pixmap_allps,
			event->area.x, event->area.y,
			event->area.x, event->area.y,
			event->area.width, event->area.height);
	return FALSE;
}

static
gint
configure_impresp(GtkWidget *widget, GdkEventConfigure *event) {

	GdkColor color;
	GdkGC * gc;

	if (pixmap_impresp)
		gdk_pixmap_unref(pixmap_impresp);

	pixmap_impresp = gdk_pixmap_new(widget->window,
				widget->allocation.width,
				widget->allocation.height,
				-1);
	color.red = 0;
	color.green = 0;
	color.blue = 0;
	gc = gdk_gc_new(pixmap_impresp);
	gdk_gc_set_rgb_fg_color(gc, &color);
	gdk_draw_rectangle(pixmap_impresp,
			   gc,
			   TRUE,
			   0, 0,
			   widget->allocation.width,
			   widget->allocation.height);
	return TRUE;
}

static
gint
expose_impresp(GtkWidget *widget, GdkEventExpose *event) {

	gdk_draw_pixmap(widget->window,
			widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
			pixmap_impresp,
			event->area.x, event->area.y,
			event->area.x, event->area.y,
			event->area.width, event->area.height);
	return FALSE;
}



void display_editor(void) {

	GdkColor color;

	GtkWidget * main_table;
	GtkWidget * menu_table;
	GtkWidget * upper_table;
	GtkWidget * main_vbox;

	GtkWidget * label_decay;
	GtkWidget * label_drylevel;
	GtkWidget * label_wetlevel;
	GtkObject * adj_decay;
	GtkObject * adj_drylevel;
	GtkObject * adj_wetlevel;
	GtkWidget * scale_decay;
	GtkWidget * scale_drylevel;
	GtkWidget * scale_wetlevel;
	GtkWidget * spin_decay;
	GtkWidget * spin_drylevel;
	GtkWidget * spin_wetlevel;

	GtkWidget * table_middle;
	GtkWidget * label_revtype;
	GtkWidget * button_new;
	GtkWidget * button_copy;
	GtkWidget * button_rename;
	GtkWidget * button_del;

	GtkWidget * vbox_combs;
	GtkWidget * table_combs;
	GtkWidget * table2_combs;
	GtkWidget * label_combs;
	GtkWidget * vbox_allps;
	GtkWidget * table_allps;
	GtkWidget * table2_allps;
	GtkWidget * label_allps;
	GtkWidget * vbox_bandps;
	GtkWidget * table_bandps;
	GtkWidget * label_bandps;
	GtkWidget * table_impresp;
	GtkWidget * label_impresp;

	GtkWidget * button_combs_new;
	GtkWidget * button_combs_del;
	GtkWidget * button_combs_next;
	GtkWidget * button_combs_prev;
	GtkWidget * label_combs_delay;
	GtkWidget * label_combs_feedback;
	GtkWidget * label_combs_freqresp;
	GtkWidget * scale_combs_delay;
	GtkWidget * scale_combs_feedback;
	GtkWidget * scale_combs_freqresp;
	GtkWidget * spin_combs_delay;
	GtkWidget * spin_combs_feedback;
	GtkWidget * spin_combs_freqresp;
	GtkWidget * viewp_combs;

	GtkWidget * button_allps_new;
	GtkWidget * button_allps_del;
	GtkWidget * button_allps_next;
	GtkWidget * button_allps_prev;
	GtkWidget * label_allps_delay;
	GtkWidget * label_allps_feedback;
	GtkWidget * scale_allps_delay;
	GtkWidget * scale_allps_feedback;
	GtkWidget * spin_allps_delay;
	GtkWidget * spin_allps_feedback;
	GtkWidget * viewp_allps;

	GtkWidget * label_bandps_hi;
	GtkWidget * label_bandps_lo;
	GtkWidget * scale_bandps_hi;
	GtkWidget * scale_bandps_lo;
	GtkWidget * spin_bandps_hi;
	GtkWidget * spin_bandps_lo;

	GtkWidget * label_impresp_time;
	GtkWidget * spin_impresp_time;
	GtkWidget * scale_impresp_time;
	GtkWidget * button_impresp_recalc;
	GtkWidget * viewp_impresp;


	/* Main window */
	main_window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
        color.red = WINDOW_R;
        color.green = WINDOW_G;
        color.blue = WINDOW_B;
        gtk_widget_modify_bg(GTK_WIDGET(main_window), GTK_STATE_NORMAL, &color);
	gtk_window_set_title(GTK_WINDOW(main_window), "TAP Reverb Editor");
	gtk_window_set_position(GTK_WINDOW(main_window), GTK_WIN_POS_CENTER);
	g_signal_connect(G_OBJECT(main_window), "delete_event", G_CALLBACK(main_window_close), NULL);

	/* A main_vbox to put the menu and the main table in: */
	main_vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(main_window), main_vbox);
	gtk_widget_show(main_vbox);

	/* Create the main table */
	main_table = gtk_table_new(3, 1, FALSE);
	gtk_box_pack_end(GTK_BOX(main_vbox), main_table, TRUE, TRUE, 2);
	gtk_table_set_row_spacing(GTK_TABLE(main_table), 0, 10);
	gtk_table_set_row_spacing(GTK_TABLE(main_table), 1, 10);
	gtk_widget_show(main_table);

	/* Create the upper table */
	upper_table = gtk_table_new(4, 4, FALSE);
	gtk_table_attach(GTK_TABLE(main_table), upper_table, 0, 1, 0, 1,
			 GTK_FILL, GTK_FILL, 2, 2);
	gtk_table_set_col_spacing(GTK_TABLE(upper_table), 0, 10);
	gtk_widget_show(upper_table);


	/* Create the menu_table to hold the menu buttons and add it to our upper table */
	menu_table = gtk_table_new(1, 3, FALSE);
	gtk_table_attach(GTK_TABLE(upper_table), menu_table, 1, 3, 0, 1,
			 GTK_FILL, GTK_FILL, 0, 0);
	gtk_widget_show(menu_table);

	/* Create menu buttons */
	ports_button = gtk_button_new_with_label("Port Setup");
	g_signal_connect(G_OBJECT(ports_button), "clicked", G_CALLBACK(clicked_ports), NULL);
	gtk_table_attach(GTK_TABLE(menu_table), ports_button, 0, 1, 0, 1,
			 GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, 5, 5);
	gtk_widget_show(ports_button);

	export_button = gtk_button_new_with_label("Save / Export");
	g_signal_connect(G_OBJECT(export_button), "clicked", G_CALLBACK(clicked_export), NULL);
	gtk_table_attach(GTK_TABLE(menu_table), export_button, 1, 2, 0, 1,
			 GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, 5, 5);
	gtk_widget_show(export_button);

	about_button = gtk_button_new_with_label("About");
	g_signal_connect(G_OBJECT(about_button), "clicked", G_CALLBACK(clicked_about), NULL);
	gtk_table_attach(GTK_TABLE(menu_table), about_button, 2, 3, 0, 1,
			 GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, 5, 5);
	gtk_widget_show(about_button);



        /* Create toggle buttons */
	toggle_combs = gtk_toggle_button_new_with_label("Comb Filters");
	g_signal_connect(G_OBJECT(toggle_combs), "toggled", G_CALLBACK(toggle_combs_callback), NULL);
	gtk_table_attach(GTK_TABLE(upper_table), toggle_combs, 0, 1, 0, 1,
			 GTK_FILL, GTK_FILL | GTK_EXPAND, 5, 5);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(toggle_combs), TRUE);
	gtk_widget_show(toggle_combs);

	toggle_allps = gtk_toggle_button_new_with_label("Allpass Filters");
	g_signal_connect(G_OBJECT(toggle_allps), "toggled", G_CALLBACK(toggle_allps_callback), NULL);
	gtk_table_attach(GTK_TABLE(upper_table), toggle_allps, 0, 1, 1, 2,
			 GTK_FILL, GTK_FILL | GTK_EXPAND, 5, 5);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(toggle_allps), TRUE);
	gtk_widget_show(toggle_allps);

	toggle_bandps = gtk_toggle_button_new_with_label("Bandpass Filter");
	g_signal_connect(G_OBJECT(toggle_bandps), "toggled", G_CALLBACK(toggle_bandps_callback), NULL);
	gtk_table_attach(GTK_TABLE(upper_table), toggle_bandps, 0, 1, 2, 3,
			 GTK_FILL, GTK_FILL | GTK_EXPAND, 5, 5);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(toggle_bandps), TRUE);
	gtk_widget_show(toggle_bandps);

	toggle_enhst = gtk_toggle_button_new_with_label("Enhanced Stereo");
	g_signal_connect(G_OBJECT(toggle_enhst), "toggled", G_CALLBACK(toggle_enhst_callback), NULL);
	gtk_table_attach(GTK_TABLE(upper_table), toggle_enhst, 0, 1, 3, 4,
			 GTK_FILL, GTK_FILL | GTK_EXPAND, 5, 5);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(toggle_enhst), TRUE);
	gtk_widget_show(toggle_enhst);

	toggle_bypass = gtk_toggle_button_new_with_label("BYPASS");
	g_signal_connect(G_OBJECT(toggle_bypass), "toggled", G_CALLBACK(toggle_bypass_callback), NULL);
	gtk_table_attach(GTK_TABLE(upper_table), toggle_bypass, 3, 4, 0, 1,
			 GTK_FILL, GTK_FILL | GTK_EXPAND, 5, 5);
	gtk_widget_show(toggle_bypass);


	/* Create scale/spin widgets */

	label_decay = gtk_label_new("Decay [ms]");
	adj_decay = gtk_adjustment_new(2500, 0, 10000, 1, 100, 0);
	scale_decay = gtk_hscale_new(GTK_ADJUSTMENT(adj_decay));
	gtk_scale_set_digits(GTK_SCALE(scale_decay), 0);
	gtk_scale_set_draw_value(GTK_SCALE(scale_decay), FALSE);
	spin_decay = gtk_spin_button_new(GTK_ADJUSTMENT(adj_decay), 0.5f, 0);
	gtk_spin_button_set_wrap(GTK_SPIN_BUTTON(spin_decay), FALSE);
	gtk_widget_set_size_request(GTK_WIDGET(spin_decay), 70, -1);
	gtk_table_attach(GTK_TABLE(upper_table), label_decay, 1, 2, 1, 2,
			 GTK_FILL, GTK_FILL | GTK_EXPAND, 5, 5);
	gtk_table_attach(GTK_TABLE(upper_table), scale_decay, 2, 3, 1, 2,
			 GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, 5, 5);
	gtk_table_attach(GTK_TABLE(upper_table), spin_decay, 3, 4, 1, 2,
			 GTK_FILL, GTK_FILL | GTK_EXPAND, 5, 5);
	gtk_widget_show(label_decay);
        gtk_widget_modify_bg(GTK_WIDGET(scale_decay), GTK_STATE_PRELIGHT, &color);
        gtk_widget_modify_bg(GTK_WIDGET(spin_decay), GTK_STATE_PRELIGHT, &color);
	gtk_widget_show(scale_decay);
	gtk_widget_show(spin_decay);

	label_drylevel = gtk_label_new("Dry Level [dB]");
	adj_drylevel = gtk_adjustment_new(0.0f, -70.0f, 10.0f, 0.1f, 2.0f, 0);
	scale_drylevel = gtk_hscale_new(GTK_ADJUSTMENT(adj_drylevel));
	gtk_scale_set_digits(GTK_SCALE(scale_drylevel), 1);
	gtk_scale_set_draw_value(GTK_SCALE(scale_drylevel), FALSE);
	spin_drylevel = gtk_spin_button_new(GTK_ADJUSTMENT(adj_drylevel), 0.5f, 1);
	gtk_spin_button_set_wrap(GTK_SPIN_BUTTON(spin_drylevel), FALSE);
	gtk_widget_set_size_request(GTK_WIDGET(spin_drylevel), 70, -1);
	gtk_table_attach(GTK_TABLE(upper_table), label_drylevel, 1, 2, 2, 3,
			 GTK_FILL, GTK_FILL | GTK_EXPAND, 5, 5);
	gtk_table_attach(GTK_TABLE(upper_table), scale_drylevel, 2, 3, 2, 3,
			 GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, 5, 5);
	gtk_table_attach(GTK_TABLE(upper_table), spin_drylevel, 3, 4, 2, 3,
			 GTK_FILL, GTK_FILL | GTK_EXPAND, 5, 5);
	gtk_widget_show(label_drylevel);
	gtk_widget_show(scale_drylevel);
	gtk_widget_show(spin_drylevel);

	label_wetlevel = gtk_label_new("Wet Level [dB]");
	adj_wetlevel = gtk_adjustment_new(0.0f, -70.0f, 10.0f, 0.1f, 2.0f, 0);
	scale_wetlevel = gtk_hscale_new(GTK_ADJUSTMENT(adj_wetlevel));
	gtk_scale_set_digits(GTK_SCALE(scale_wetlevel), 1);
	gtk_scale_set_draw_value(GTK_SCALE(scale_wetlevel), FALSE);
	spin_wetlevel = gtk_spin_button_new(GTK_ADJUSTMENT(adj_wetlevel), 0.5f, 1);
	gtk_spin_button_set_wrap(GTK_SPIN_BUTTON(spin_wetlevel), FALSE);
	gtk_widget_set_size_request(GTK_WIDGET(spin_wetlevel), 70, -1);
	gtk_table_attach(GTK_TABLE(upper_table), label_wetlevel, 1, 2, 3, 4,
			 GTK_FILL, GTK_FILL | GTK_EXPAND, 5, 5);
	gtk_table_attach(GTK_TABLE(upper_table), scale_wetlevel, 2, 3, 3, 4,
			 GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, 5, 5);
	gtk_table_attach(GTK_TABLE(upper_table), spin_wetlevel, 3, 4, 3, 4,
			 GTK_FILL, GTK_FILL | GTK_EXPAND, 5, 5);
	gtk_widget_show(label_wetlevel);
	gtk_widget_show(scale_wetlevel);
	gtk_widget_show(spin_wetlevel);

	g_signal_connect(G_OBJECT(adj_decay), "value_changed", G_CALLBACK(changed_decay), NULL);
	g_signal_connect(G_OBJECT(adj_drylevel), "value_changed", G_CALLBACK(changed_drylevel), NULL);
	g_signal_connect(G_OBJECT(adj_wetlevel), "value_changed", G_CALLBACK(changed_wetlevel), NULL);


	/* Fill the middle cell of the main table */
        table_middle = gtk_table_new(1, 6, FALSE);
	gtk_table_attach(GTK_TABLE(main_table), table_middle, 0, 1, 1, 2,
			 GTK_FILL, GTK_FILL, 0, 0);
	gtk_widget_show(table_middle);
	label_revtype = gtk_label_new(" Reverb Type");
	gtk_table_attach(GTK_TABLE(table_middle), label_revtype, 0, 1, 0, 1,
			 GTK_FILL, GTK_FILL, 3, 3);
	gtk_widget_show(label_revtype);
	combo_revtype = gtk_combo_new();
	gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(combo_revtype)->entry), FALSE);
	gtk_combo_set_use_arrows_always(GTK_COMBO(combo_revtype), TRUE);
	g_signal_connect(G_OBJECT(GTK_COMBO(combo_revtype)->entry), "changed",
			 G_CALLBACK(changed_revtype), NULL);
	gtk_table_attach(GTK_TABLE(table_middle), combo_revtype, 1, 2, 0, 1,
			 GTK_FILL | GTK_EXPAND, GTK_FILL, 3, 3);
	gtk_widget_show(combo_revtype);

	button_new = gtk_button_new_with_label("New");
	gtk_table_attach(GTK_TABLE(table_middle), button_new, 2, 3, 0, 1,
			 GTK_FILL, GTK_FILL, 3, 3);
	g_signal_connect(G_OBJECT(button_new), "clicked", G_CALLBACK(clicked_new), NULL);
	gtk_widget_show(button_new);

	button_copy = gtk_button_new_with_label("Copy");
	gtk_table_attach(GTK_TABLE(table_middle), button_copy, 3, 4, 0, 1,
			 GTK_FILL, GTK_FILL, 3, 3);
	g_signal_connect(G_OBJECT(button_copy), "clicked", G_CALLBACK(clicked_copy), NULL);
	gtk_widget_show(button_copy);

	button_rename = gtk_button_new_with_label("Rename");
	gtk_table_attach(GTK_TABLE(table_middle), button_rename, 4, 5, 0, 1,
			 GTK_FILL, GTK_FILL, 3, 3);
	g_signal_connect(G_OBJECT(button_rename), "clicked", G_CALLBACK(clicked_rename), NULL);
	gtk_widget_show(button_rename);

	button_del = gtk_button_new_with_label("Del");
	gtk_table_attach(GTK_TABLE(table_middle), button_del, 5, 6, 0, 1,
			 GTK_FILL, GTK_FILL, 3, 3);
	g_signal_connect(G_OBJECT(button_del), "clicked", G_CALLBACK(clicked_del), NULL);
	gtk_widget_show(button_del);



	/* setup the notebook */
	notebook = gtk_notebook_new();
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(notebook), GTK_POS_TOP);

	/* setup the comb filters page */
	vbox_combs = gtk_vbox_new(FALSE, 0);
	gtk_widget_show(vbox_combs);
	label_combs = gtk_label_new("Comb Filters");
	gtk_widget_show(label_combs);
	table_combs = gtk_table_new(4, 2, FALSE);
	gtk_box_pack_start(GTK_BOX(vbox_combs), table_combs, FALSE, FALSE, 2);
	gtk_widget_show(table_combs);

	viewp_combs = gtk_viewport_new(NULL, NULL);
	gtk_viewport_set_shadow_type(GTK_VIEWPORT(viewp_combs), GTK_SHADOW_IN);
	gtk_table_attach(GTK_TABLE(table_combs), viewp_combs, 0, 1, 0, 4,
			 GTK_FILL, GTK_FILL, 3, 3);
        gtk_widget_show(viewp_combs);

	draw_combs = gtk_drawing_area_new();
	gtk_drawing_area_size(GTK_DRAWING_AREA(draw_combs), DRAW_WIDTH, DRAW_HEIGHT);
	g_signal_connect(G_OBJECT(draw_combs), "configure_event", G_CALLBACK(configure_combs), NULL);
	g_signal_connect(G_OBJECT(draw_combs), "expose_event", G_CALLBACK(expose_combs), NULL);
        gtk_container_add(GTK_CONTAINER(viewp_combs), draw_combs);
	gtk_widget_show(draw_combs);

	button_combs_new = gtk_button_new_with_label("New");
	g_signal_connect(G_OBJECT(button_combs_new), "clicked", G_CALLBACK(combs_new_clicked), NULL);
	gtk_table_attach(GTK_TABLE(table_combs), button_combs_new, 1, 2, 0, 1,
			 GTK_FILL | GTK_EXPAND, GTK_FILL, 3, 3);
	gtk_widget_show(button_combs_new);

	button_combs_del = gtk_button_new_with_label("Del");
	g_signal_connect(G_OBJECT(button_combs_del), "clicked", G_CALLBACK(combs_del_clicked), NULL);
	gtk_table_attach(GTK_TABLE(table_combs), button_combs_del, 1, 2, 1, 2,
			 GTK_FILL | GTK_EXPAND, GTK_FILL, 3, 3);
	gtk_widget_show(button_combs_del);

	button_combs_next = gtk_button_new_with_label("Next");
	g_signal_connect(G_OBJECT(button_combs_next), "clicked", G_CALLBACK(combs_next_clicked), NULL);
	gtk_table_attach(GTK_TABLE(table_combs), button_combs_next, 1, 2, 2, 3,
			 GTK_FILL | GTK_EXPAND, GTK_FILL, 3, 3);
	gtk_widget_show(button_combs_next);

	button_combs_prev = gtk_button_new_with_label("Prev");
	g_signal_connect(G_OBJECT(button_combs_prev), "clicked", G_CALLBACK(combs_prev_clicked), NULL);
	gtk_table_attach(GTK_TABLE(table_combs), button_combs_prev, 1, 2, 3, 4,
			 GTK_FILL | GTK_EXPAND, GTK_FILL, 3, 3);
	gtk_widget_show(button_combs_prev);

	/* lower table of combs notebook page */
	table2_combs = gtk_table_new(3, 3, FALSE);
	gtk_box_pack_start(GTK_BOX(vbox_combs), table2_combs, FALSE, FALSE, 2);
	gtk_widget_show(table2_combs);

	label_combs_delay = gtk_label_new("Delay [ms]");
	adj_combs_delay = gtk_adjustment_new(100, 1, 250, 0.1f, 10, 0);
	scale_combs_delay = gtk_hscale_new(GTK_ADJUSTMENT(adj_combs_delay));
	gtk_scale_set_digits(GTK_SCALE(scale_combs_delay), 1);
	gtk_scale_set_draw_value(GTK_SCALE(scale_combs_delay), FALSE);
	spin_combs_delay = gtk_spin_button_new(GTK_ADJUSTMENT(adj_combs_delay), 0.5f, 1);
	gtk_spin_button_set_wrap(GTK_SPIN_BUTTON(spin_combs_delay), FALSE);
	gtk_widget_set_size_request(GTK_WIDGET(spin_combs_delay), 70, -1);
	gtk_table_attach(GTK_TABLE(table2_combs), label_combs_delay, 0, 1, 0, 1,
			 GTK_FILL, GTK_FILL | GTK_EXPAND, 5, 5);
	gtk_table_attach(GTK_TABLE(table2_combs), scale_combs_delay, 1, 2, 0, 1,
			 GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, 5, 5);
	gtk_table_attach(GTK_TABLE(table2_combs), spin_combs_delay, 2, 3, 0, 1,
			 GTK_FILL, GTK_FILL | GTK_EXPAND, 5, 5);
	gtk_widget_show(label_combs_delay);
	gtk_widget_show(scale_combs_delay);
	gtk_widget_show(spin_combs_delay);

	label_combs_feedback = gtk_label_new("Feedback [%]");
	adj_combs_feedback = gtk_adjustment_new(50, 1, 100, 0.1f, 10, 0);
	scale_combs_feedback = gtk_hscale_new(GTK_ADJUSTMENT(adj_combs_feedback));
	gtk_scale_set_digits(GTK_SCALE(scale_combs_feedback), 1);
	gtk_scale_set_draw_value(GTK_SCALE(scale_combs_feedback), FALSE);
	spin_combs_feedback = gtk_spin_button_new(GTK_ADJUSTMENT(adj_combs_feedback), 0.25f, 1);
	gtk_spin_button_set_wrap(GTK_SPIN_BUTTON(spin_combs_feedback), FALSE);
	gtk_table_attach(GTK_TABLE(table2_combs), label_combs_feedback, 0, 1, 1, 2,
			 GTK_FILL, GTK_FILL | GTK_EXPAND, 5, 5);
	gtk_table_attach(GTK_TABLE(table2_combs), scale_combs_feedback, 1, 2, 1, 2,
			 GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, 5, 5);
	gtk_table_attach(GTK_TABLE(table2_combs), spin_combs_feedback, 2, 3, 1, 2,
			 GTK_FILL, GTK_FILL | GTK_EXPAND, 5, 5);
	gtk_widget_show(label_combs_feedback);
	gtk_widget_show(scale_combs_feedback);
	gtk_widget_show(spin_combs_feedback);

	label_combs_freqresp = gtk_label_new("Freq. Response");
	adj_combs_freqresp = gtk_adjustment_new(0.5f, 0.0f, 1.0f, 0.01f, 0.1f, 0);
	scale_combs_freqresp = gtk_hscale_new(GTK_ADJUSTMENT(adj_combs_freqresp));
	gtk_scale_set_digits(GTK_SCALE(scale_combs_freqresp), 2);
	gtk_scale_set_draw_value(GTK_SCALE(scale_combs_freqresp), FALSE);
	spin_combs_freqresp = gtk_spin_button_new(GTK_ADJUSTMENT(adj_combs_freqresp), 0.1f, 2);
	gtk_spin_button_set_wrap(GTK_SPIN_BUTTON(spin_combs_freqresp), FALSE);
	gtk_table_attach(GTK_TABLE(table2_combs), label_combs_freqresp, 0, 1, 2, 3,
			 GTK_FILL, GTK_FILL | GTK_EXPAND, 5, 5);
	gtk_table_attach(GTK_TABLE(table2_combs), scale_combs_freqresp, 1, 2, 2, 3,
			 GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, 5, 5);
	gtk_table_attach(GTK_TABLE(table2_combs), spin_combs_freqresp, 2, 3, 2, 3,
			 GTK_FILL, GTK_FILL | GTK_EXPAND, 5, 5);
	gtk_widget_show(label_combs_freqresp);
	gtk_widget_show(scale_combs_freqresp);
	gtk_widget_show(spin_combs_freqresp);

	g_signal_connect(G_OBJECT(adj_combs_delay), "value_changed",
			 G_CALLBACK(changed_combs_delay), NULL);
	g_signal_connect(G_OBJECT(adj_combs_feedback), "value_changed",
			 G_CALLBACK(changed_combs_feedback), NULL);
	g_signal_connect(G_OBJECT(adj_combs_freqresp), "value_changed",
			 G_CALLBACK(changed_combs_freqresp), NULL);


	/* setup the allpass filters page */
	vbox_allps = gtk_vbox_new(FALSE, 0);
	gtk_widget_show(vbox_allps);
	label_allps = gtk_label_new("Allpass Filters");
	gtk_widget_show(label_allps);
	table_allps = gtk_table_new(4, 2, FALSE);
	gtk_box_pack_start(GTK_BOX(vbox_allps), table_allps, FALSE, FALSE, 2);
	gtk_widget_show(table_allps);

	viewp_allps = gtk_viewport_new(NULL, NULL);
	gtk_viewport_set_shadow_type(GTK_VIEWPORT(viewp_allps), GTK_SHADOW_IN);
	gtk_table_attach(GTK_TABLE(table_allps), viewp_allps, 0, 1, 0, 4,
			 GTK_FILL, GTK_FILL, 3, 3);
        gtk_widget_show(viewp_allps);

	draw_allps = gtk_drawing_area_new();
	gtk_drawing_area_size(GTK_DRAWING_AREA(draw_allps), DRAW_WIDTH, DRAW_HEIGHT);
	g_signal_connect(G_OBJECT(draw_allps), "configure_event", G_CALLBACK(configure_allps), NULL);
	g_signal_connect(G_OBJECT(draw_allps), "expose_event", G_CALLBACK(expose_allps), NULL);
        gtk_container_add(GTK_CONTAINER(viewp_allps), draw_allps);
	gtk_widget_show(draw_allps);

	button_allps_new = gtk_button_new_with_label("New");
	g_signal_connect(G_OBJECT(button_allps_new), "clicked", G_CALLBACK(allps_new_clicked), NULL);
	gtk_table_attach(GTK_TABLE(table_allps), button_allps_new, 1, 2, 0, 1,
			 GTK_FILL | GTK_EXPAND, GTK_FILL, 3, 3);
	gtk_widget_show(button_allps_new);

	button_allps_del = gtk_button_new_with_label("Del");
	g_signal_connect(G_OBJECT(button_allps_del), "clicked", G_CALLBACK(allps_del_clicked), NULL);
	gtk_table_attach(GTK_TABLE(table_allps), button_allps_del, 1, 2, 1, 2,
			 GTK_FILL | GTK_EXPAND, GTK_FILL, 3, 3);
	gtk_widget_show(button_allps_del);

	button_allps_next = gtk_button_new_with_label("Next");
	g_signal_connect(G_OBJECT(button_allps_next), "clicked", G_CALLBACK(allps_next_clicked), NULL);
	gtk_table_attach(GTK_TABLE(table_allps), button_allps_next, 1, 2, 2, 3,
			 GTK_FILL | GTK_EXPAND, GTK_FILL, 3, 3);
	gtk_widget_show(button_allps_next);

	button_allps_prev = gtk_button_new_with_label("Prev");
	g_signal_connect(G_OBJECT(button_allps_prev), "clicked", G_CALLBACK(allps_prev_clicked), NULL);
	gtk_table_attach(GTK_TABLE(table_allps), button_allps_prev, 1, 2, 3, 4,
			 GTK_FILL | GTK_EXPAND, GTK_FILL, 3, 3);
	gtk_widget_show(button_allps_prev);

	/* lower table of allpass notebook page */
	table2_allps = gtk_table_new(3, 3, FALSE);
	gtk_box_pack_start(GTK_BOX(vbox_allps), table2_allps, FALSE, FALSE, 2);
	gtk_widget_show(table2_allps);

	label_allps_delay = gtk_label_new("Delay [ms]");
	adj_allps_delay = gtk_adjustment_new(10, 1, 20, 0.1f, 1, 0);
	scale_allps_delay = gtk_hscale_new(GTK_ADJUSTMENT(adj_allps_delay));
	gtk_scale_set_digits(GTK_SCALE(scale_allps_delay), 1);
	gtk_scale_set_draw_value(GTK_SCALE(scale_allps_delay), FALSE);
	spin_allps_delay = gtk_spin_button_new(GTK_ADJUSTMENT(adj_allps_delay), 0.5f, 1);
	gtk_spin_button_set_wrap(GTK_SPIN_BUTTON(spin_allps_delay), FALSE);
	gtk_widget_set_size_request(GTK_WIDGET(spin_allps_delay), 70, -1);
	gtk_table_attach(GTK_TABLE(table2_allps), label_allps_delay, 0, 1, 0, 1,
			 GTK_FILL, GTK_FILL | GTK_EXPAND, 5, 5);
	gtk_table_attach(GTK_TABLE(table2_allps), scale_allps_delay, 1, 2, 0, 1,
			 GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, 5, 5);
	gtk_table_attach(GTK_TABLE(table2_allps), spin_allps_delay, 2, 3, 0, 1,
			 GTK_FILL, GTK_FILL | GTK_EXPAND, 5, 5);
	gtk_widget_show(label_allps_delay);
	gtk_widget_show(scale_allps_delay);
	gtk_widget_show(spin_allps_delay);

	label_allps_feedback = gtk_label_new("Feedback [%]");
	adj_allps_feedback = gtk_adjustment_new(50, 1, 100, 0.1f, 10, 0);
	scale_allps_feedback = gtk_hscale_new(GTK_ADJUSTMENT(adj_allps_feedback));
	gtk_scale_set_digits(GTK_SCALE(scale_allps_feedback), 1);
	gtk_scale_set_draw_value(GTK_SCALE(scale_allps_feedback), FALSE);
	spin_allps_feedback = gtk_spin_button_new(GTK_ADJUSTMENT(adj_allps_feedback), 0.25f, 1);
	gtk_spin_button_set_wrap(GTK_SPIN_BUTTON(spin_allps_feedback), FALSE);
	gtk_table_attach(GTK_TABLE(table2_allps), label_allps_feedback, 0, 1, 1, 2,
			 GTK_FILL, GTK_FILL | GTK_EXPAND, 5, 5);
	gtk_table_attach(GTK_TABLE(table2_allps), scale_allps_feedback, 1, 2, 1, 2,
			 GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, 5, 5);
	gtk_table_attach(GTK_TABLE(table2_allps), spin_allps_feedback, 2, 3, 1, 2,
			 GTK_FILL, GTK_FILL | GTK_EXPAND, 5, 5);
	gtk_widget_show(label_allps_feedback);
	gtk_widget_show(scale_allps_feedback);
	gtk_widget_show(spin_allps_feedback);

	g_signal_connect(G_OBJECT(adj_allps_delay), "value_changed",
			 G_CALLBACK(changed_allps_delay), NULL);
	g_signal_connect(G_OBJECT(adj_allps_feedback), "value_changed",
			 G_CALLBACK(changed_allps_feedback), NULL);



	/* setup the bandpass filter page */
        vbox_bandps = gtk_vbox_new(FALSE, 0);
	gtk_widget_show(vbox_bandps);
	label_bandps = gtk_label_new("Bandpass Filter");
	gtk_widget_show(label_bandps);
	table_bandps = gtk_table_new(2, 3, FALSE);
	gtk_box_pack_start(GTK_BOX(vbox_bandps), table_bandps, FALSE, FALSE, 2);
	gtk_widget_show(table_bandps);

	label_bandps_lo = gtk_label_new("Low cut freq. [Hz]");
	adj_bandps_lo = gtk_adjustment_new(10000, 20, 18000, 1, 100, 0);
	scale_bandps_lo = gtk_hscale_new(GTK_ADJUSTMENT(adj_bandps_lo));
	gtk_scale_set_digits(GTK_SCALE(scale_bandps_lo), 0);
	gtk_scale_set_draw_value(GTK_SCALE(scale_bandps_lo), FALSE);
	spin_bandps_lo = gtk_spin_button_new(GTK_ADJUSTMENT(adj_bandps_lo), 0.5f, 0);
	gtk_spin_button_set_wrap(GTK_SPIN_BUTTON(spin_bandps_lo), FALSE);
	gtk_table_attach(GTK_TABLE(table_bandps), label_bandps_lo, 0, 1, 0, 1,
			 GTK_FILL, GTK_FILL | GTK_EXPAND, 5, 5);
	gtk_table_attach(GTK_TABLE(table_bandps), scale_bandps_lo, 1, 2, 0, 1,
			 GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, 5, 5);
	gtk_table_attach(GTK_TABLE(table_bandps), spin_bandps_lo, 2, 3, 0, 1,
			 GTK_FILL, GTK_FILL | GTK_EXPAND, 5, 5);
	gtk_widget_show(label_bandps_lo);
	gtk_widget_show(scale_bandps_lo);
	gtk_widget_show(spin_bandps_lo);

	label_bandps_hi = gtk_label_new("High cut freq. [Hz]");
	adj_bandps_hi = gtk_adjustment_new(100, 20, 18000, 1, 100, 0);
	scale_bandps_hi = gtk_hscale_new(GTK_ADJUSTMENT(adj_bandps_hi));
	gtk_scale_set_digits(GTK_SCALE(scale_bandps_hi), 0);
	gtk_scale_set_draw_value(GTK_SCALE(scale_bandps_hi), FALSE);
	spin_bandps_hi = gtk_spin_button_new(GTK_ADJUSTMENT(adj_bandps_hi), 0.5f, 0);
	gtk_spin_button_set_wrap(GTK_SPIN_BUTTON(spin_bandps_hi), FALSE);
	gtk_widget_set_size_request(GTK_WIDGET(spin_bandps_hi), 70, -1);
	gtk_table_attach(GTK_TABLE(table_bandps), label_bandps_hi, 0, 1, 1, 2,
			 GTK_FILL, GTK_FILL | GTK_EXPAND, 5, 5);
	gtk_table_attach(GTK_TABLE(table_bandps), scale_bandps_hi, 1, 2, 1, 2,
			 GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, 5, 5);
	gtk_table_attach(GTK_TABLE(table_bandps), spin_bandps_hi, 2, 3, 1, 2,
			 GTK_FILL, GTK_FILL | GTK_EXPAND, 5, 5);
	gtk_widget_show(label_bandps_hi);
	gtk_widget_show(scale_bandps_hi);
	gtk_widget_show(spin_bandps_hi);

	g_signal_connect(G_OBJECT(adj_bandps_lo), "value_changed", G_CALLBACK(changed_bandps_lo), NULL);
	g_signal_connect(G_OBJECT(adj_bandps_hi), "value_changed", G_CALLBACK(changed_bandps_hi), NULL);


	/* setup the impulse response page */
	label_impresp = gtk_label_new("Impulse Response");
	gtk_widget_show(label_impresp);
	table_impresp = gtk_table_new(2, 4, FALSE);
	gtk_widget_show(table_impresp);

	viewp_impresp = gtk_viewport_new(NULL, NULL);
	gtk_viewport_set_shadow_type(GTK_VIEWPORT(viewp_impresp), GTK_SHADOW_IN);
	gtk_table_attach(GTK_TABLE(table_impresp), viewp_impresp, 0, 4, 0, 1,
			 GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, 3, 3);
        gtk_widget_show(viewp_impresp);

	draw_impresp = gtk_drawing_area_new();
	gtk_drawing_area_size(GTK_DRAWING_AREA(draw_impresp), -1, -1);
	g_signal_connect(G_OBJECT(draw_impresp), "configure_event", G_CALLBACK(configure_impresp), NULL);
	g_signal_connect(G_OBJECT(draw_impresp), "expose_event", G_CALLBACK(expose_impresp), NULL);
        gtk_container_add(GTK_CONTAINER(viewp_impresp), draw_impresp);
	gtk_widget_show(draw_impresp);

	label_impresp_time = gtk_label_new("Length [ms]");
	adj_impresp_time = gtk_adjustment_new(1000, 1, 10000, 1, 100, 0);
	scale_impresp_time = gtk_hscale_new(GTK_ADJUSTMENT(adj_impresp_time));
	gtk_scale_set_digits(GTK_SCALE(scale_impresp_time), 0);
	gtk_scale_set_draw_value(GTK_SCALE(scale_impresp_time), FALSE);
	spin_impresp_time = gtk_spin_button_new(GTK_ADJUSTMENT(adj_impresp_time), 0.5f, 0);
	gtk_spin_button_set_wrap(GTK_SPIN_BUTTON(spin_impresp_time), FALSE);
	gtk_widget_set_size_request(GTK_WIDGET(spin_impresp_time), 70, -1);
	gtk_table_attach(GTK_TABLE(table_impresp), label_impresp_time, 0, 1, 1, 2,
			 GTK_FILL, GTK_FILL, 5, 5);
	gtk_table_attach(GTK_TABLE(table_impresp), scale_impresp_time, 1, 2, 1, 2,
			 GTK_FILL | GTK_EXPAND, GTK_FILL, 5, 5);
	gtk_table_attach(GTK_TABLE(table_impresp), spin_impresp_time, 2, 3, 1, 2,
			 GTK_FILL, GTK_FILL, 5, 5);
	gtk_widget_show(label_impresp_time);
	gtk_widget_show(scale_impresp_time);
	gtk_widget_show(spin_impresp_time);
	g_signal_connect(G_OBJECT(adj_impresp_time), "value_changed",
			 G_CALLBACK(changed_impresp_time), NULL);

	button_impresp_recalc = gtk_button_new_with_label("Recalculate");
	g_signal_connect(G_OBJECT(button_impresp_recalc), "clicked",
			 G_CALLBACK(impresp_recalc_clicked), NULL);
	gtk_table_attach(GTK_TABLE(table_impresp), button_impresp_recalc, 3, 4, 1, 2,
			 GTK_FILL, GTK_FILL, 3, 3);
	gtk_widget_show(button_impresp_recalc);


	gtk_notebook_insert_page(GTK_NOTEBOOK(notebook), GTK_WIDGET(vbox_combs),
				 GTK_WIDGET(label_combs), 0);
	gtk_notebook_insert_page(GTK_NOTEBOOK(notebook), GTK_WIDGET(vbox_allps),
				 GTK_WIDGET(label_allps), 1);
	gtk_notebook_insert_page(GTK_NOTEBOOK(notebook), GTK_WIDGET(vbox_bandps),
				 GTK_WIDGET(label_bandps), 2);
	gtk_notebook_insert_page(GTK_NOTEBOOK(notebook), GTK_WIDGET(table_impresp),
				 GTK_WIDGET(label_impresp), 3);

	gtk_table_attach(GTK_TABLE(main_table), notebook, 0, 1, 2, 3,
			 GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, 2, 2);
	gtk_widget_show(notebook);

	/* set colors for just about any widget */
        color.red = NORMAL_R;
        color.green = NORMAL_G;
        color.blue = NORMAL_B;
        gtk_widget_modify_bg(GTK_WIDGET(ports_button), GTK_STATE_NORMAL, &color);
        gtk_widget_modify_bg(GTK_WIDGET(ports_button), GTK_STATE_INSENSITIVE, &color);
        gtk_widget_modify_bg(GTK_WIDGET(export_button), GTK_STATE_NORMAL, &color);
        gtk_widget_modify_bg(GTK_WIDGET(export_button), GTK_STATE_INSENSITIVE, &color);
        gtk_widget_modify_bg(GTK_WIDGET(about_button), GTK_STATE_NORMAL, &color);
        gtk_widget_modify_bg(GTK_WIDGET(about_button), GTK_STATE_INSENSITIVE, &color);
        gtk_widget_modify_bg(GTK_WIDGET(toggle_combs), GTK_STATE_NORMAL, &color);
        gtk_widget_modify_bg(GTK_WIDGET(toggle_allps), GTK_STATE_NORMAL, &color);
        gtk_widget_modify_bg(GTK_WIDGET(toggle_bandps), GTK_STATE_NORMAL, &color);
        gtk_widget_modify_bg(GTK_WIDGET(toggle_enhst), GTK_STATE_NORMAL, &color);
        gtk_widget_modify_bg(GTK_WIDGET(toggle_bypass), GTK_STATE_NORMAL, &color);
        gtk_widget_modify_bg(GTK_WIDGET(scale_decay), GTK_STATE_NORMAL, &color);
        gtk_widget_modify_bg(GTK_WIDGET(spin_decay), GTK_STATE_NORMAL, &color);
        gtk_widget_modify_bg(GTK_WIDGET(spin_decay), GTK_STATE_INSENSITIVE, &color);
        gtk_widget_modify_bg(GTK_WIDGET(scale_drylevel), GTK_STATE_NORMAL, &color);
        gtk_widget_modify_bg(GTK_WIDGET(spin_drylevel), GTK_STATE_NORMAL, &color);
        gtk_widget_modify_bg(GTK_WIDGET(spin_drylevel), GTK_STATE_INSENSITIVE, &color);
        gtk_widget_modify_bg(GTK_WIDGET(scale_wetlevel), GTK_STATE_NORMAL, &color);
        gtk_widget_modify_bg(GTK_WIDGET(spin_wetlevel), GTK_STATE_NORMAL, &color);
        gtk_widget_modify_bg(GTK_WIDGET(spin_wetlevel), GTK_STATE_INSENSITIVE, &color);
        gtk_widget_modify_bg(GTK_WIDGET(GTK_COMBO(combo_revtype)->button), GTK_STATE_NORMAL, &color);
        gtk_widget_modify_bg(GTK_WIDGET(GTK_COMBO(combo_revtype)->entry), GTK_STATE_NORMAL, &color);
        gtk_widget_modify_bg(GTK_WIDGET(GTK_COMBO(combo_revtype)->popup), GTK_STATE_NORMAL, &color);
        gtk_widget_modify_bg(GTK_WIDGET(GTK_COMBO(combo_revtype)->popwin), GTK_STATE_NORMAL, &color);
        gtk_widget_modify_bg(GTK_WIDGET(GTK_COMBO(combo_revtype)->list), GTK_STATE_NORMAL, &color);
        gtk_widget_modify_bg(GTK_WIDGET(button_new), GTK_STATE_NORMAL, &color);
        gtk_widget_modify_bg(GTK_WIDGET(button_copy), GTK_STATE_NORMAL, &color);
        gtk_widget_modify_bg(GTK_WIDGET(button_rename), GTK_STATE_NORMAL, &color);
        gtk_widget_modify_bg(GTK_WIDGET(button_del), GTK_STATE_NORMAL, &color);


        color.red = ACTIVE_R;
        color.green = ACTIVE_G;
        color.blue = ACTIVE_B;
        gtk_widget_modify_bg(GTK_WIDGET(scale_wetlevel), GTK_STATE_ACTIVE, &color);
        gtk_widget_modify_bg(GTK_WIDGET(spin_wetlevel), GTK_STATE_ACTIVE, &color);
        gtk_widget_modify_bg(GTK_WIDGET(ports_button), GTK_STATE_ACTIVE, &color);
        gtk_widget_modify_bg(GTK_WIDGET(export_button), GTK_STATE_ACTIVE, &color);
        gtk_widget_modify_bg(GTK_WIDGET(about_button), GTK_STATE_ACTIVE, &color);
        gtk_widget_modify_bg(GTK_WIDGET(scale_decay), GTK_STATE_ACTIVE, &color);
        gtk_widget_modify_bg(GTK_WIDGET(spin_decay), GTK_STATE_ACTIVE, &color);
        gtk_widget_modify_bg(GTK_WIDGET(scale_drylevel), GTK_STATE_ACTIVE, &color);
        gtk_widget_modify_bg(GTK_WIDGET(spin_drylevel), GTK_STATE_ACTIVE, &color);
        gtk_widget_modify_bg(GTK_WIDGET(scale_wetlevel), GTK_STATE_ACTIVE, &color);
        gtk_widget_modify_bg(GTK_WIDGET(spin_wetlevel), GTK_STATE_ACTIVE, &color);
        gtk_widget_modify_bg(GTK_WIDGET(GTK_COMBO(combo_revtype)->button), GTK_STATE_ACTIVE, &color);
        gtk_widget_modify_bg(GTK_WIDGET(GTK_COMBO(combo_revtype)->entry), GTK_STATE_ACTIVE, &color);
        gtk_widget_modify_bg(GTK_WIDGET(GTK_COMBO(combo_revtype)->popup), GTK_STATE_ACTIVE, &color);
        gtk_widget_modify_bg(GTK_WIDGET(GTK_COMBO(combo_revtype)->popwin), GTK_STATE_ACTIVE, &color);
        gtk_widget_modify_bg(GTK_WIDGET(GTK_COMBO(combo_revtype)->list), GTK_STATE_ACTIVE, &color);
        gtk_widget_modify_bg(GTK_WIDGET(button_new), GTK_STATE_ACTIVE, &color);
        gtk_widget_modify_bg(GTK_WIDGET(button_copy), GTK_STATE_ACTIVE, &color);
        gtk_widget_modify_bg(GTK_WIDGET(button_rename), GTK_STATE_ACTIVE, &color);
        gtk_widget_modify_bg(GTK_WIDGET(button_del), GTK_STATE_ACTIVE, &color);


        color.red = PRELIGHT_R;
        color.green = PRELIGHT_G;
        color.blue = PRELIGHT_B;
        gtk_widget_modify_bg(GTK_WIDGET(scale_wetlevel), GTK_STATE_PRELIGHT, &color);
        gtk_widget_modify_bg(GTK_WIDGET(spin_wetlevel), GTK_STATE_PRELIGHT, &color);
        gtk_widget_modify_bg(GTK_WIDGET(ports_button), GTK_STATE_PRELIGHT, &color);
        gtk_widget_modify_bg(GTK_WIDGET(export_button), GTK_STATE_PRELIGHT, &color);
        gtk_widget_modify_bg(GTK_WIDGET(about_button), GTK_STATE_PRELIGHT, &color);
        gtk_widget_modify_bg(GTK_WIDGET(toggle_bypass), GTK_STATE_PRELIGHT, &color);
        gtk_widget_modify_bg(GTK_WIDGET(scale_drylevel), GTK_STATE_PRELIGHT, &color);
        gtk_widget_modify_bg(GTK_WIDGET(spin_drylevel), GTK_STATE_PRELIGHT, &color);
        gtk_widget_modify_bg(GTK_WIDGET(scale_wetlevel), GTK_STATE_PRELIGHT, &color);
        gtk_widget_modify_bg(GTK_WIDGET(spin_wetlevel), GTK_STATE_PRELIGHT, &color);
        gtk_widget_modify_bg(GTK_WIDGET(GTK_COMBO(combo_revtype)->button), GTK_STATE_PRELIGHT, &color);
        gtk_widget_modify_bg(GTK_WIDGET(GTK_COMBO(combo_revtype)->entry), GTK_STATE_PRELIGHT, &color);
        gtk_widget_modify_bg(GTK_WIDGET(GTK_COMBO(combo_revtype)->popup), GTK_STATE_PRELIGHT, &color);
        gtk_widget_modify_bg(GTK_WIDGET(GTK_COMBO(combo_revtype)->popwin), GTK_STATE_PRELIGHT, &color);
        gtk_widget_modify_bg(GTK_WIDGET(GTK_COMBO(combo_revtype)->list), GTK_STATE_PRELIGHT, &color);
        gtk_widget_modify_bg(GTK_WIDGET(button_new), GTK_STATE_PRELIGHT, &color);
        gtk_widget_modify_bg(GTK_WIDGET(button_copy), GTK_STATE_PRELIGHT, &color);
        gtk_widget_modify_bg(GTK_WIDGET(button_rename), GTK_STATE_PRELIGHT, &color);
        gtk_widget_modify_bg(GTK_WIDGET(button_del), GTK_STATE_PRELIGHT, &color);


        color.red = R_ACTIVE_R;
        color.green = R_ACTIVE_G;
        color.blue = R_ACTIVE_B;
        gtk_widget_modify_bg(GTK_WIDGET(toggle_bypass), GTK_STATE_ACTIVE, &color);


        color.red = NB_NORMAL_R;
        color.green = NB_NORMAL_G;
        color.blue = NB_NORMAL_B;
        gtk_widget_modify_bg(GTK_WIDGET(notebook), GTK_STATE_NORMAL, &color);


        color.red = NB_ACTIVE_R;
        color.green = NB_ACTIVE_G;
        color.blue = NB_ACTIVE_B;
        gtk_widget_modify_bg(GTK_WIDGET(notebook), GTK_STATE_ACTIVE, &color);


        color.red = NB_PRELIGHT_R;
        color.green = NB_PRELIGHT_G;
        color.blue = NB_PRELIGHT_B;
        gtk_widget_modify_bg(GTK_WIDGET(notebook), GTK_STATE_PRELIGHT, &color);


        color.red = N_NORMAL_R;
        color.green = N_NORMAL_G;
        color.blue = N_NORMAL_B;
        gtk_widget_modify_bg(GTK_WIDGET(button_combs_new), GTK_STATE_NORMAL, &color);
        gtk_widget_modify_bg(GTK_WIDGET(button_combs_del), GTK_STATE_NORMAL, &color);
        gtk_widget_modify_bg(GTK_WIDGET(button_combs_next), GTK_STATE_NORMAL, &color);
        gtk_widget_modify_bg(GTK_WIDGET(button_combs_prev), GTK_STATE_NORMAL, &color);
        gtk_widget_modify_bg(GTK_WIDGET(scale_combs_delay), GTK_STATE_NORMAL, &color);
        gtk_widget_modify_bg(GTK_WIDGET(spin_combs_delay), GTK_STATE_NORMAL, &color);
        gtk_widget_modify_bg(GTK_WIDGET(spin_combs_delay), GTK_STATE_INSENSITIVE, &color);
        gtk_widget_modify_bg(GTK_WIDGET(scale_combs_feedback), GTK_STATE_NORMAL, &color);
        gtk_widget_modify_bg(GTK_WIDGET(spin_combs_feedback), GTK_STATE_NORMAL, &color);
        gtk_widget_modify_bg(GTK_WIDGET(spin_combs_feedback), GTK_STATE_INSENSITIVE, &color);
        gtk_widget_modify_bg(GTK_WIDGET(scale_combs_freqresp), GTK_STATE_NORMAL, &color);
        gtk_widget_modify_bg(GTK_WIDGET(spin_combs_freqresp), GTK_STATE_NORMAL, &color);
        gtk_widget_modify_bg(GTK_WIDGET(spin_combs_freqresp), GTK_STATE_INSENSITIVE, &color);
        gtk_widget_modify_bg(GTK_WIDGET(button_allps_new), GTK_STATE_NORMAL, &color);
        gtk_widget_modify_bg(GTK_WIDGET(button_allps_del), GTK_STATE_NORMAL, &color);
        gtk_widget_modify_bg(GTK_WIDGET(button_allps_next), GTK_STATE_NORMAL, &color);
        gtk_widget_modify_bg(GTK_WIDGET(button_allps_prev), GTK_STATE_NORMAL, &color);
        gtk_widget_modify_bg(GTK_WIDGET(scale_allps_delay), GTK_STATE_NORMAL, &color);
        gtk_widget_modify_bg(GTK_WIDGET(spin_allps_delay), GTK_STATE_NORMAL, &color);
        gtk_widget_modify_bg(GTK_WIDGET(spin_allps_delay), GTK_STATE_INSENSITIVE, &color);
        gtk_widget_modify_bg(GTK_WIDGET(scale_allps_feedback), GTK_STATE_NORMAL, &color);
        gtk_widget_modify_bg(GTK_WIDGET(spin_allps_feedback), GTK_STATE_NORMAL, &color);
        gtk_widget_modify_bg(GTK_WIDGET(spin_allps_feedback), GTK_STATE_INSENSITIVE, &color);
        gtk_widget_modify_bg(GTK_WIDGET(scale_bandps_lo), GTK_STATE_NORMAL, &color);
        gtk_widget_modify_bg(GTK_WIDGET(spin_bandps_lo), GTK_STATE_NORMAL, &color);
        gtk_widget_modify_bg(GTK_WIDGET(spin_bandps_lo), GTK_STATE_INSENSITIVE, &color);
        gtk_widget_modify_bg(GTK_WIDGET(scale_bandps_hi), GTK_STATE_NORMAL, &color);
        gtk_widget_modify_bg(GTK_WIDGET(spin_bandps_hi), GTK_STATE_NORMAL, &color);
        gtk_widget_modify_bg(GTK_WIDGET(spin_bandps_hi), GTK_STATE_INSENSITIVE, &color);
        gtk_widget_modify_bg(GTK_WIDGET(scale_impresp_time), GTK_STATE_NORMAL, &color);
        gtk_widget_modify_bg(GTK_WIDGET(spin_impresp_time), GTK_STATE_NORMAL, &color);
        gtk_widget_modify_bg(GTK_WIDGET(spin_impresp_time), GTK_STATE_INSENSITIVE, &color);
        gtk_widget_modify_bg(GTK_WIDGET(button_impresp_recalc), GTK_STATE_NORMAL, &color);
        gtk_widget_modify_bg(GTK_WIDGET(viewp_combs), GTK_STATE_NORMAL, &color);
        gtk_widget_modify_bg(GTK_WIDGET(viewp_allps), GTK_STATE_NORMAL, &color);
        gtk_widget_modify_bg(GTK_WIDGET(viewp_impresp), GTK_STATE_NORMAL, &color);


        color.red = N_ACTIVE_R;
        color.green = N_ACTIVE_G;
        color.blue = N_ACTIVE_B;
        gtk_widget_modify_bg(GTK_WIDGET(button_combs_new), GTK_STATE_ACTIVE, &color);
        gtk_widget_modify_bg(GTK_WIDGET(button_combs_del), GTK_STATE_ACTIVE, &color);
        gtk_widget_modify_bg(GTK_WIDGET(button_combs_next), GTK_STATE_ACTIVE, &color);
        gtk_widget_modify_bg(GTK_WIDGET(button_combs_prev), GTK_STATE_ACTIVE, &color);
        gtk_widget_modify_bg(GTK_WIDGET(scale_combs_delay), GTK_STATE_ACTIVE, &color);
        gtk_widget_modify_bg(GTK_WIDGET(spin_combs_delay), GTK_STATE_ACTIVE, &color);
        gtk_widget_modify_bg(GTK_WIDGET(scale_combs_feedback), GTK_STATE_ACTIVE, &color);
        gtk_widget_modify_bg(GTK_WIDGET(spin_combs_feedback), GTK_STATE_ACTIVE, &color);
        gtk_widget_modify_bg(GTK_WIDGET(scale_combs_freqresp), GTK_STATE_ACTIVE, &color);
        gtk_widget_modify_bg(GTK_WIDGET(spin_combs_freqresp), GTK_STATE_ACTIVE, &color);
        gtk_widget_modify_bg(GTK_WIDGET(button_allps_new), GTK_STATE_ACTIVE, &color);
        gtk_widget_modify_bg(GTK_WIDGET(button_allps_del), GTK_STATE_ACTIVE, &color);
        gtk_widget_modify_bg(GTK_WIDGET(button_allps_next), GTK_STATE_ACTIVE, &color);
        gtk_widget_modify_bg(GTK_WIDGET(button_allps_prev), GTK_STATE_ACTIVE, &color);
        gtk_widget_modify_bg(GTK_WIDGET(scale_allps_delay), GTK_STATE_ACTIVE, &color);
        gtk_widget_modify_bg(GTK_WIDGET(spin_allps_delay), GTK_STATE_ACTIVE, &color);
        gtk_widget_modify_bg(GTK_WIDGET(scale_allps_feedback), GTK_STATE_ACTIVE, &color);
        gtk_widget_modify_bg(GTK_WIDGET(spin_allps_feedback), GTK_STATE_ACTIVE, &color);
        gtk_widget_modify_bg(GTK_WIDGET(scale_bandps_lo), GTK_STATE_ACTIVE, &color);
        gtk_widget_modify_bg(GTK_WIDGET(spin_bandps_lo), GTK_STATE_ACTIVE, &color);
        gtk_widget_modify_bg(GTK_WIDGET(scale_bandps_hi), GTK_STATE_ACTIVE, &color);
        gtk_widget_modify_bg(GTK_WIDGET(spin_bandps_hi), GTK_STATE_ACTIVE, &color);
        gtk_widget_modify_bg(GTK_WIDGET(scale_impresp_time), GTK_STATE_ACTIVE, &color);
        gtk_widget_modify_bg(GTK_WIDGET(spin_impresp_time), GTK_STATE_ACTIVE, &color);
        gtk_widget_modify_bg(GTK_WIDGET(button_impresp_recalc), GTK_STATE_ACTIVE, &color);
        gtk_widget_modify_bg(GTK_WIDGET(viewp_combs), GTK_STATE_ACTIVE, &color);
        gtk_widget_modify_bg(GTK_WIDGET(viewp_allps), GTK_STATE_ACTIVE, &color);
        gtk_widget_modify_bg(GTK_WIDGET(viewp_impresp), GTK_STATE_ACTIVE, &color);


        color.red = N_PRELIGHT_R;
        color.green = N_PRELIGHT_G;
        color.blue = N_PRELIGHT_B;
        gtk_widget_modify_bg(GTK_WIDGET(button_combs_new), GTK_STATE_PRELIGHT, &color);
        gtk_widget_modify_bg(GTK_WIDGET(button_combs_del), GTK_STATE_PRELIGHT, &color);
        gtk_widget_modify_bg(GTK_WIDGET(button_combs_next), GTK_STATE_PRELIGHT, &color);
        gtk_widget_modify_bg(GTK_WIDGET(button_combs_prev), GTK_STATE_PRELIGHT, &color);
        gtk_widget_modify_bg(GTK_WIDGET(scale_combs_delay), GTK_STATE_PRELIGHT, &color);
        gtk_widget_modify_bg(GTK_WIDGET(spin_combs_delay), GTK_STATE_PRELIGHT, &color);
        gtk_widget_modify_bg(GTK_WIDGET(scale_combs_feedback), GTK_STATE_PRELIGHT, &color);
        gtk_widget_modify_bg(GTK_WIDGET(spin_combs_feedback), GTK_STATE_PRELIGHT, &color);
        gtk_widget_modify_bg(GTK_WIDGET(scale_combs_freqresp), GTK_STATE_PRELIGHT, &color);
        gtk_widget_modify_bg(GTK_WIDGET(spin_combs_freqresp), GTK_STATE_PRELIGHT, &color);
        gtk_widget_modify_bg(GTK_WIDGET(button_allps_new), GTK_STATE_PRELIGHT, &color);
        gtk_widget_modify_bg(GTK_WIDGET(button_allps_del), GTK_STATE_PRELIGHT, &color);
        gtk_widget_modify_bg(GTK_WIDGET(button_allps_next), GTK_STATE_PRELIGHT, &color);
        gtk_widget_modify_bg(GTK_WIDGET(button_allps_prev), GTK_STATE_PRELIGHT, &color);
        gtk_widget_modify_bg(GTK_WIDGET(scale_allps_delay), GTK_STATE_PRELIGHT, &color);
        gtk_widget_modify_bg(GTK_WIDGET(spin_allps_delay), GTK_STATE_PRELIGHT, &color);
        gtk_widget_modify_bg(GTK_WIDGET(scale_allps_feedback), GTK_STATE_PRELIGHT, &color);
        gtk_widget_modify_bg(GTK_WIDGET(spin_allps_feedback), GTK_STATE_PRELIGHT, &color);
        gtk_widget_modify_bg(GTK_WIDGET(scale_bandps_lo), GTK_STATE_PRELIGHT, &color);
        gtk_widget_modify_bg(GTK_WIDGET(spin_bandps_lo), GTK_STATE_PRELIGHT, &color);
        gtk_widget_modify_bg(GTK_WIDGET(scale_bandps_hi), GTK_STATE_PRELIGHT, &color);
        gtk_widget_modify_bg(GTK_WIDGET(spin_bandps_hi), GTK_STATE_PRELIGHT, &color);
        gtk_widget_modify_bg(GTK_WIDGET(spin_impresp_time), GTK_STATE_PRELIGHT, &color);
        gtk_widget_modify_bg(GTK_WIDGET(scale_impresp_time), GTK_STATE_PRELIGHT, &color);
        gtk_widget_modify_bg(GTK_WIDGET(button_impresp_recalc), GTK_STATE_PRELIGHT, &color);
        gtk_widget_modify_bg(GTK_WIDGET(viewp_combs), GTK_STATE_PRELIGHT, &color);
        gtk_widget_modify_bg(GTK_WIDGET(viewp_allps), GTK_STATE_PRELIGHT, &color);
        gtk_widget_modify_bg(GTK_WIDGET(viewp_impresp), GTK_STATE_PRELIGHT, &color);


	/* disallow resizing of the main window */
	gtk_window_set_policy(GTK_WINDOW(main_window), FALSE, FALSE, FALSE);

	gtk_widget_show(main_window);
}


/* Re-generate combo items from dynamic list */
void update_combo(REVTYPE * root) {

	REVTYPE * item = root;

	if (item->next != NULL)
		item = item->next;
	else
		return;
	
	revtype_list = NULL;

	while (item != NULL) {
		revtype_list = g_list_append(revtype_list, item->name);
		item = item->next;
	}

	gtk_combo_set_popdown_strings(GTK_COMBO(combo_revtype), revtype_list);
	g_list_free(revtype_list);
}


/* set gui controls/displays to reflect the current reverb type */
void display_revtype(void) {

	if (curr == NULL)
		return;

	current_comb = 0;
	current_allp = 0;

	gtk_adjustment_set_value(GTK_ADJUSTMENT(adj_combs_delay),
				 curr->combs_data[3 * current_comb]);
	gtk_adjustment_set_value(GTK_ADJUSTMENT(adj_combs_feedback),
				 curr->combs_data[3 * current_comb + 1]);
	gtk_adjustment_set_value(GTK_ADJUSTMENT(adj_combs_freqresp),
				 curr->combs_data[3 * current_comb + 2]);

	gtk_adjustment_set_value(GTK_ADJUSTMENT(adj_allps_delay),
				 curr->allps_data[2 * current_allp]);
	gtk_adjustment_set_value(GTK_ADJUSTMENT(adj_allps_feedback),
				 curr->allps_data[2 * current_allp + 1]);

	gtk_adjustment_set_value(GTK_ADJUSTMENT(adj_bandps_lo),
				 curr->bandps_lo);
	gtk_adjustment_set_value(GTK_ADJUSTMENT(adj_bandps_hi),
				 curr->bandps_hi);

	redraw_combs();
	redraw_allps();
}


void redraw_combs(void) {

	GdkColor color;
	GdkGC * gc;
	int width = DRAW_WIDTH;
	int height = DRAW_HEIGHT;
	int i;

	float xscale = (float)(width - 1) / (MAX_COMB_DELAY - 1);

	if (pixmap_combs == NULL)
		return;

	gc = gdk_gc_new(pixmap_combs);

	/* paint background */
	color.red = 0;
	color.green = 0;
	color.blue = 0;
	gc = gdk_gc_new(pixmap_combs);
	gdk_gc_set_rgb_fg_color(gc, &color);
	gdk_draw_rectangle(pixmap_combs,
			   gc,
			   TRUE,
			   0, 0,
			   (GTK_WIDGET(draw_combs))->allocation.width,
			   (GTK_WIDGET(draw_combs))->allocation.height);

	/* lines */
	for (i = 0; i < curr->num_combs; i++) {
		if (i == current_comb) {
			color.red = 65535 * (1 - curr->combs_data[3 * i + 2]);
			color.green = 65535;
			color.blue = 50000;
		} else {
			color.red = 65535 * (1 - curr->combs_data[3 * i + 2]);
			color.green = 0;
			color.blue = 65535;
		}
		gdk_gc_set_rgb_fg_color(gc, &color);
		gdk_draw_line(pixmap_combs, gc,
			      (curr->combs_data[3 * i] - 1) * xscale,
			      (100 - curr->combs_data[3 * i + 1]) * DRAW_HEIGHT / 100,
			      (curr->combs_data[3 * i] - 1) * xscale, height - 1);
	}

	/* refresh */
        gdk_draw_pixmap((GTK_WIDGET(draw_combs))->window,
                        (GTK_WIDGET(draw_combs))->style->fg_gc[GTK_WIDGET_STATE(GTK_WIDGET(draw_combs))],
                        pixmap_combs,
                        0, 0,
                        0, 0,
                        DRAW_WIDTH, DRAW_HEIGHT);

}


void redraw_allps(void) {

	GdkColor color;
	GdkGC * gc;
	int width = DRAW_WIDTH;
	int height = DRAW_HEIGHT;
	int i;

	float xscale = (float)(width - 1) / (MAX_ALLP_DELAY - 1);

	if (pixmap_allps == NULL)
		return;
	
	gc = gdk_gc_new(pixmap_allps);

	/* paint background */
	color.red = 0;
	color.green = 0;
	color.blue = 0;
	gc = gdk_gc_new(pixmap_allps);
	gdk_gc_set_rgb_fg_color(gc, &color);
	gdk_draw_rectangle(pixmap_allps,
			   gc,
			   TRUE,
			   0, 0,
			   (GTK_WIDGET(draw_allps))->allocation.width,
			   (GTK_WIDGET(draw_allps))->allocation.height);

	/* lines */
	for (i = 0; i < curr->num_allps; i++) {
		if (i == current_allp) {
			color.red = 45000;
			color.green = 65535;
			color.blue = 50000;
		} else {
			color.red = 45000;
			color.green = 0;
			color.blue = 65535;
		}
		gdk_gc_set_rgb_fg_color(gc, &color);
		gdk_draw_line(pixmap_allps, gc,
			      (curr->allps_data[2 * i] - 1) * xscale,
			      (100 - curr->allps_data[2 * i + 1]) * DRAW_HEIGHT / 100,
			      (curr->allps_data[2 * i] - 1) * xscale, height - 1);
	}

	/* refresh */
        gdk_draw_pixmap((GTK_WIDGET(draw_allps))->window,
                        (GTK_WIDGET(draw_allps))->style->fg_gc[GTK_WIDGET_STATE(GTK_WIDGET(draw_allps))],
                        pixmap_allps,
                        0, 0,
                        0, 0,
                        DRAW_WIDTH, DRAW_HEIGHT);

}


void redraw_impresp(long int nframes) {

	GdkColor color;
	GdkGC * gc;
	GdkFont * font;
	int width = (GTK_WIDGET(draw_impresp))->allocation.width;
	int height = (GTK_WIDGET(draw_impresp))->allocation.height;
	int i;
	float max = 0.0f;
	float yscale;
	char str[MAXLEN];

	/* number of samples per screen pixel */
	float xscale = (float)nframes / (float)(width - 1);

	for (i = 0; i < nframes; i++)
		if (fabs(impresp_data[i]) > max)
			max = fabs(impresp_data[i]);

	/* number of pixels per unit signal level */
	yscale = (float)(height/2 - 1) / max;

	if (pixmap_impresp == NULL)
		return;

	/* paint background */
	color.red = 0;
	color.green = 0;
	color.blue = 0;
	gc = gdk_gc_new(pixmap_impresp);
	gdk_gc_set_rgb_fg_color(gc, &color);
	gdk_draw_rectangle(pixmap_impresp,
			   gc,
			   TRUE,
			   0, 0,
			   (GTK_WIDGET(draw_impresp))->allocation.width,
			   (GTK_WIDGET(draw_impresp))->allocation.height);

	/* horizontal axis */
	color.red = 65535;
	color.green = 65535;
	color.blue = 65535;
	gdk_gc_set_rgb_fg_color(gc, &color);
	gdk_draw_line(pixmap_impresp, gc, 0, height/2, width-1, height/2);

	/* drawing color */
	color.red = 0;
	color.green = 65535;
	color.blue = 0;
	gdk_gc_set_rgb_fg_color(gc, &color);

	/* plot data */
	for (i = 0; i < nframes; i++) {
		gdk_draw_line(pixmap_impresp, gc,
			      i/xscale, height/2,
			      i/xscale, height/2 - impresp_data[i] * yscale);
	}		

	/* draw string */
	sprintf(str, "max [ y ] = %.3f unit level", max);
	color.red = 58000;
	color.green = 58000;
	color.blue = 63535;
	gdk_gc_set_rgb_fg_color(gc, &color);
	font = gdk_font_load("-*-helvetica-medium-r-normal-*-*-80-*-*-p-*-*");
	gdk_draw_string(pixmap_impresp, font, gc, 10, height - 10, str);

	/* refresh */
        gdk_draw_pixmap((GTK_WIDGET(draw_impresp))->window,
                        (GTK_WIDGET(draw_impresp))->style->fg_gc[GTK_WIDGET_STATE(GTK_WIDGET(draw_impresp))],
                        pixmap_impresp,
                        0, 0,
                        0, 0,
                        width, height);

}



gint
running_cb(gpointer data) {

	if (!client_running) {
		gtk_timeout_remove(tag_running);
		if (ports_window) {
			gtk_widget_destroy(ports_window);
			ports_window = NULL;
		}
		gtk_widget_set_sensitive(GTK_WIDGET(ports_button), FALSE);
		message_box("JACK connection lost",
			    "   JACK has either been shutdown or it   \n"
			    "   disconnected TAP Reverb Editor because   \n"
			    "   it was not fast enough. If you have any   \n"
			    "   valuable work done, please save it now,   \n"
			    "   then restart the program.   ",
			    main_window,
			    GTK_MESSAGE_ERROR);
		return 0;
	} else {
		return 1;
	}
}


int main(int argc, char *argv[]) {

	int c;
	int auto_all = 0;
	int auto_ins = 0;
	int auto_outs = 0;
	char * client_name = NULL;

	printf("TAP Reverb Editor, release 0\n"
	       "(C) 2004 Tom Szilagyi <tszilagyi@users.sourceforge.net>\n"
	       "This is free software, and you are welcome to redistribute it\n"
	       "under certain conditions; see the file COPYING for details.\n");

	while((c = getopt(argc, argv, "aioc:")) != -1) {
                switch (c) {
                case 'a': /* auto-connect all ports to physical JACK ports */
                        auto_all = 1;
                        break;
                case 'i': /* auto-connect input ports to physical JACK ports */
                        auto_ins = 1;
                        break;
                case 'o': /* auto-connect output ports to physical JACK ports */
                        auto_outs = 1;
                        break;
                case 'c': /* client name */
                        if(!(client_name = strdup(optarg))) {
                                fprintf(stderr, "FATAL: out of memory while parsing command-line\n");
                                return 1;
                        }
                        break;
                }
        }

	if (!client_name)
		client_name = strdup("reverbED");

        if ((client = jack_client_new(client_name)) == 0) {
                fprintf(stderr, "\nTAP Reverb Editor couldn't connect to JACK.\n");
                fprintf(stderr, "There are several possible reasons:\n"
                        "\t1) JACK is not running.\n"
                        "\t2) JACK is running as another user, perhaps root.\n"
                        "\t3) There is already another client called 'reverbED'.\n");
                fprintf(stderr, "Please consider the possibilities, and perhaps (re)start JACK.\n");
                return 1;
        }

        printf("\nRegistered to JACK as client '%s'.\n", client_name);

        jack_set_process_callback(client, process, 0);
        jack_on_shutdown(client, jack_shutdown, 0);

        if ((sample_rate = jack_get_sample_rate(client)) > MAX_SAMPLERATE) {
		printf("\nThe JACK sample rate (%ld Hz) is higher than MAX_SAMPLERATE.\n", sample_rate);
		printf("This is an arbitrary limit, which you may safely enlarge\n"
		       "in common.h if you really need to.\n");
		printf("Currently MAX_SAMPLERATE = %d Hz.\n", MAX_SAMPLERATE);
		jack_client_close(client);
		return 1;
	}

        in_L_port = jack_port_register(client, "in_L", JACK_DEFAULT_AUDIO_TYPE, JackPortIsInput, 0);
        in_R_port = jack_port_register(client, "in_R", JACK_DEFAULT_AUDIO_TYPE, JackPortIsInput, 0);
        out_L_port = jack_port_register(client, "out_L", JACK_DEFAULT_AUDIO_TYPE, JackPortIsOutput, 0);
        out_R_port = jack_port_register(client, "out_R", JACK_DEFAULT_AUDIO_TYPE, JackPortIsOutput, 0);


	if ((root = parse_input_file()) == NULL) {
		printf("Something bad happened while parsing ~/.reverbed, exiting.\n");
		exit(1);
	}

	sound_init();
	gtk_init (&argc, &argv);
	display_editor();
	gtk_notebook_set_current_page(GTK_NOTEBOOK(notebook), 3);
	gtk_notebook_set_current_page(GTK_NOTEBOOK(notebook), 2);
	gtk_notebook_set_current_page(GTK_NOTEBOOK(notebook), 1);
	gtk_notebook_set_current_page(GTK_NOTEBOOK(notebook), 0);
	update_combo(root);


        if (jack_activate(client)) {
                fprintf (stderr, "ERROR: cannot activate JACK client.\n");
                return 1;
        }
	client_running = 1;


	if ((auto_ins) || (auto_all)) {
		/* connect input ports */
		if ((ports_in = jack_get_ports(client, NULL, NULL,
					       JackPortIsPhysical|JackPortIsOutput)) == NULL) {
			fprintf(stderr, "Cannot find any physical capture ports.\n");
		} else {
			if (jack_connect(client, ports_in[0], jack_port_name(in_L_port))) {
				fprintf (stderr, "Cannot connect in_L port.\n");
			} else {
				printf("Connected in_L to %s\n", ports_in[0]);
			}
			if (jack_connect(client, ports_in[1], jack_port_name(in_R_port))) {
				fprintf (stderr, "Cannot connect in_R port.\n");
			} else {
				printf("Connected in_R to %s\n", ports_in[1]);
			}
			free(ports_in);
		}
	}

	if ((auto_outs) || (auto_all)) {
		/* connect output ports */
		if ((ports_out = jack_get_ports(client, NULL, NULL,
						JackPortIsPhysical|JackPortIsInput)) == NULL) {
			fprintf(stderr, "Cannot find any physical playback ports.\n");
		} else {
			if (jack_connect(client, jack_port_name(out_L_port), ports_out[0])) {
				fprintf (stderr, "Cannot connect out_L port.\n");
			} else {
				printf("Connected out_L to %s\n", ports_out[0]);
			}
			if (jack_connect(client, jack_port_name(out_R_port), ports_out[1])) {
				fprintf (stderr, "Cannot connect out_R port.\n");
			} else {
				printf("Connected out_R to %s\n", ports_out[1]);
			}
			free(ports_out);
		}
	}

	/* set timeout callback to see if client is still running */
	tag_running = gtk_timeout_add(100, running_cb, 0);

	gtk_main();

	if (client_running) {
		gtk_timeout_remove(tag_running);
		jack_client_close(client);
	}

	return 0;
}
