/*  $Id: sdlmain.cpp,v 1.23 2024/05/05 17:25:58 sarrazip Exp $
    sdlmain.cpp - main() function for Afternoon Stalker

    afternoonstalker - A robot-killing video game.
    Copyright (C) 2001-2012 Pierre Sarrazin <http://sarrazip.com/>

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public
    License along with this program; if not, write to the Free
    Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA  02110-1301, USA.
*/

#include "AfternoonStalkerEngine.h"

#include <stdlib.h>
#include <errno.h>
#include <time.h>
#include <stdio.h>

#ifdef HAVE_GETOPT_LONG
    #include <getopt.h>
#endif

#include <iostream>
#include <string>

using namespace std;


#ifdef HAVE_GETOPT_LONG
static struct option knownOptions[] =
{
    { "help",          no_argument,       NULL, 'h' },
    { "version",       no_argument,       NULL, 'v' },
    { "ms-per-frame",  required_argument, NULL, 'm' },
    { "no-sound",      no_argument,       NULL, 'n' },
    { "full-screen",   no_argument,       NULL, 'f' },
    { "maze",          required_argument, NULL, 's' },
    { "rules",         required_argument, NULL, 'R' },
    { "no-auto-pause", no_argument,       NULL, 'p' },
    { "rendering",     required_argument, NULL, 'r' },

    { NULL, 0, NULL, 0 }  // marks the end
};


static void
displayVersionNo(ostream &out)
{
    out << PACKAGE << " " << VERSION << "\n";
}


static void
displayHelp(ostream &out)
{
    out << "\n";

    displayVersionNo(out);

    out <<
"\n"
"Copyright (C) 2001, 2002, 2003 Pierre Sarrazin <http://sarrazip.com/>\n"
"This program is free software; you may redistribute it under the terms of\n"
"the GNU General Public License.  This program has absolutely no warranty.\n"
    ;

    out <<
"\n"
"Known options:\n"
"--help             Display this help page and exit\n"
"--version          Display this program's version number and exit\n"
"--ms-per-frame=N   N milliseconds per animation frame [default: 55]\n"
"                   Min. 1, max. 1000.  50 ms means 20 frames per second\n"
"--full-screen      Attempt to use full screen mode [window mode]\n"
"--maze=M           Select maze M: 0 = original, 1 or 2 = alternatives\n"
"--rules=R          Rules to use (R must be 1 or 2, default is 1)\n"
"                   1 = original Afternoon Stalker rules\n"
"                   2 = 2008 updated Afternoon Stalker rules\n"
"--no-auto-pause    Do not pause automatically when window loses focus\n"
"--rendering=X      Specify the type of rendering.\n"
"                   Replace X with 'a' for accelerated, 's' for software.\n"
"\n"
    ;
}
#endif  /* HAVE_GETOPT_LONG */


int main(int argc, char *argv[])
{
    /*  Default game parameters:
    */
    int millisecondsPerFrame = 55;
    bool useSound = true;
    bool fullScreen = false;
    size_t mazeIndex = 0;
    AfternoonStalkerEngine::Rules rules = AfternoonStalkerEngine::AS_RULES_1;
    bool processActiveEvent = true;
    bool useAcceleratedRendering = true;

    #ifdef HAVE_GETOPT_LONG

    /*  Interpret the command-line options:
    */
    int c;
    do
    {
        c = getopt_long(argc, argv, "", knownOptions, NULL);

        switch (c)
        {
            case EOF:
                break;  // nothing to do

            case 'm':
                {
                    errno = 0;
                    long n = strtol(optarg, NULL, 10);
                    if (errno == ERANGE || n < 1 || n > 1000)
                    {
                        cerr << "Invalid number of ms per frame.\n";
                        displayHelp(cerr);
                        return EXIT_FAILURE;
                    }

                    millisecondsPerFrame = int(n);
                }
                break;

            case 'n':
                useSound = false;
                break;

            case 'f':
                fullScreen = true;
                break;

            case 's':
                {
                    errno = 0;
                    long n = strtol(optarg, NULL, 10);
                    if (errno == ERANGE || n < 0 || n > 2)
                    {
                        cerr << "Invalid '--maze' argument \"" << optarg << "\".\n";
                        displayHelp(cerr);
                        return EXIT_FAILURE;
                    }
                    mazeIndex = size_t(n);
                }
                break;

            case 'R':
                {
                    string r = optarg;
                    if (r == "1")
                        rules = AfternoonStalkerEngine::AS_RULES_1;
                    else if (r == "2")
                        rules = AfternoonStalkerEngine::AS_RULES_2;
                    else
                    {
                        cerr << "Invalid '--rules' argument \"" << optarg << "\".\n";
                        displayHelp(cerr);
                        return EXIT_FAILURE;
                    }
                }
                break;

            case 'p':
                processActiveEvent = false;
                break;

            case 'r':
                if (!strcmp(optarg, "a"))
                    useAcceleratedRendering = true;
                else if (!strcmp(optarg, "s"))
                    useAcceleratedRendering = false;
                else
                {
                    cout << "Invalid argument for --rendering.\n";
                    displayHelp(cerr);
                    return EXIT_FAILURE;
                }
                break;

            case 'v':
                displayVersionNo(cout);
                return EXIT_SUCCESS;

            case 'h':
                displayHelp(cout);
                return EXIT_SUCCESS;

            default:
                displayHelp(cerr);
                return EXIT_FAILURE;
        }
    } while (c != EOF && c != '?');

    #endif  /* HAVE_GETOPT_LONG */


    /*  Initialize the random number generator:
    */
    const char *s = getenv("SEED");
    unsigned int seed = (s != NULL ? atol(s) : time(NULL));
    #ifndef NDEBUG
    cerr << "seed = " << seed << "\n";
    cerr << "ms-per-frame = " << millisecondsPerFrame << "\n";
    #endif
    srand(seed);


    try
    {
        AfternoonStalkerEngine theAfternoonStalkerEngine(
                                            "Afternoon Stalker",
                                            useSound,
                                            fullScreen,
                                            mazeIndex,
                                            rules,
                                            processActiveEvent,
                                            useAcceleratedRendering);
        theAfternoonStalkerEngine.run(millisecondsPerFrame);
    }
    catch (const string &s)
    {
        cerr << PACKAGE << ": engine init error: " << s << "\n";
        return EXIT_FAILURE;
    }
    catch (int e)
    {
        cerr << PACKAGE << ": init error # " << e << "\n";
        return EXIT_FAILURE;
    }

    return EXIT_SUCCESS;
}
