// Copyright 2019 TiKV Project Authors. Licensed under MIT or Apache-2.0.

//! Table-based CRC-64/NVME computer.
//!
//! The update function computes the CRC value 16 bytes at a time.
//! The processing speed is roughly 5× of the one-byte-at-a-time method.

#![allow(clippy::unreadable_literal)]

pub(crate) fn update(mut state: u64, bytes: &[u8]) -> u64 {
    // Safe.
    let (left, middle, right) = unsafe { bytes.align_to::<[u8; 16]>() };
    for b in left {
        state = update_1(state, *b);
    }
    for b in middle {
        state = update_16(state, b);
    }
    for b in right {
        state = update_1(state, *b);
    }
    state
}

/// Performs the CRC-64/NVME update, one byte at a time.
fn update_1(state: u64, b: u8) -> u64 {
    (state >> 8) ^ TABLE_0[usize::from(b ^ (state as u8))]
}

/// Performs the CRC-64/NVME update, 16 bytes at a time.
fn update_16(state: u64, b: &[u8; 16]) -> u64 {
    let s = state.to_le_bytes();
    TABLE_0[usize::from(b[15])]
        ^ TABLE_1[usize::from(b[14])]
        ^ TABLE_2[usize::from(b[13])]
        ^ TABLE_3[usize::from(b[12])]
        ^ TABLE_4[usize::from(b[11])]
        ^ TABLE_5[usize::from(b[10])]
        ^ TABLE_6[usize::from(b[9])]
        ^ TABLE_7[usize::from(b[8])]
        ^ TABLE_8[usize::from(b[7] ^ s[7])]
        ^ TABLE_9[usize::from(b[6] ^ s[6])]
        ^ TABLE_10[usize::from(b[5] ^ s[5])]
        ^ TABLE_11[usize::from(b[4] ^ s[4])]
        ^ TABLE_12[usize::from(b[3] ^ s[3])]
        ^ TABLE_13[usize::from(b[2] ^ s[2])]
        ^ TABLE_14[usize::from(b[1] ^ s[1])]
        ^ TABLE_15[usize::from(b[0] ^ s[0])]
}

// Constants used in SIMD-based computations.
// All K_nnn constants are computed by bit_reverse(x^nnn mod POLY).
// Used the NVME polynomial (0xAD93D23594C93659) from the
pub(crate) const K_127: u64 = 0x21e9_761e_2526_21ac;
pub(crate) const K_191: u64 = 0xeadc_41fd_2ba3_d420;
pub(crate) const K_255: u64 = 0xe1e0_bb9d_45d7_a44c;
pub(crate) const K_319: u64 = 0xb0bc_2e58_9204_f500;
pub(crate) const K_383: u64 = 0xa3ff_dc1f_e8e8_2a8b;
pub(crate) const K_447: u64 = 0xbdd7_ac0e_e1a4_a0f0;
pub(crate) const K_511: u64 = 0x6224_2240_ace5_045a;
pub(crate) const K_575: u64 = 0x0c32_cdb3_1e18_a84a;
pub(crate) const K_639: u64 = 0x0336_3823_e6e7_91e5;
pub(crate) const K_703: u64 = 0x7b0a_b10d_d0f8_09fe;
pub(crate) const K_767: u64 = 0x34f5_a24e_22d6_6e90;
pub(crate) const K_831: u64 = 0x3c25_5f5e_bc41_4423;
pub(crate) const K_895: u64 = 0x9465_8840_3d4a_dcbc;
pub(crate) const K_959: u64 = 0xd083_dd59_4d96_319d;
pub(crate) const K_1023: u64 = 0x5f85_2fb6_1e8d_92dc;
pub(crate) const K_1087: u64 = 0xa1ca_681e_733f_9c40;

/// The reciprocal of the NVME polynomial (0xAD93D23594C93659)
pub(crate) const POLY: u64 = 0x34d9_2653_5897_936b;

/// The polynomial quotient x¹²⁸/POLY.
pub(crate) const MU: u64 = 0x27ec_fa32_9aef_9f77;

// CRC table for the NVME polynomial (`0xAD93D23594C93659`).
// Generated by running `./build_table 0`.
//
// Essentially, `TABLE_0[m]` is the polynomial remainder in GF(2) when `m`
// (`1 = x⁷¹, 2 = x⁷⁰, 4 = x⁶⁹, …, 128 = x⁶⁴`) is divided by the NVME poly.
static TABLE_0: [u64; 256] = [
    0x0000000000000000,
    0x7f6ef0c830358979,
    0xfedde190606b12f2,
    0x81b31158505e9b8b,
    0xc962e5739841b68f,
    0xb60c15bba8743ff6,
    0x37bf04e3f82aa47d,
    0x48d1f42bc81f2d04,
    0xa61cecb46814fe75,
    0xd9721c7c5821770c,
    0x58c10d24087fec87,
    0x27affdec384a65fe,
    0x6f7e09c7f05548fa,
    0x1010f90fc060c183,
    0x91a3e857903e5a08,
    0xeecd189fa00bd371,
    0x78e0ff3b88be6f81,
    0x078e0ff3b88be6f8,
    0x863d1eabe8d57d73,
    0xf953ee63d8e0f40a,
    0xb1821a4810ffd90e,
    0xceecea8020ca5077,
    0x4f5ffbd87094cbfc,
    0x30310b1040a14285,
    0xdefc138fe0aa91f4,
    0xa192e347d09f188d,
    0x2021f21f80c18306,
    0x5f4f02d7b0f40a7f,
    0x179ef6fc78eb277b,
    0x68f0063448deae02,
    0xe943176c18803589,
    0x962de7a428b5bcf0,
    0xf1c1fe77117cdf02,
    0x8eaf0ebf2149567b,
    0x0f1c1fe77117cdf0,
    0x7072ef2f41224489,
    0x38a31b04893d698d,
    0x47cdebccb908e0f4,
    0xc67efa94e9567b7f,
    0xb9100a5cd963f206,
    0x57dd12c379682177,
    0x28b3e20b495da80e,
    0xa900f35319033385,
    0xd66e039b2936bafc,
    0x9ebff7b0e12997f8,
    0xe1d10778d11c1e81,
    0x606216208142850a,
    0x1f0ce6e8b1770c73,
    0x8921014c99c2b083,
    0xf64ff184a9f739fa,
    0x77fce0dcf9a9a271,
    0x08921014c99c2b08,
    0x4043e43f0183060c,
    0x3f2d14f731b68f75,
    0xbe9e05af61e814fe,
    0xc1f0f56751dd9d87,
    0x2f3dedf8f1d64ef6,
    0x50531d30c1e3c78f,
    0xd1e00c6891bd5c04,
    0xae8efca0a188d57d,
    0xe65f088b6997f879,
    0x9931f84359a27100,
    0x1882e91b09fcea8b,
    0x67ec19d339c963f2,
    0xd75adabd7a6e2d6f,
    0xa8342a754a5ba416,
    0x29873b2d1a053f9d,
    0x56e9cbe52a30b6e4,
    0x1e383fcee22f9be0,
    0x6156cf06d21a1299,
    0xe0e5de5e82448912,
    0x9f8b2e96b271006b,
    0x71463609127ad31a,
    0x0e28c6c1224f5a63,
    0x8f9bd7997211c1e8,
    0xf0f5275142244891,
    0xb824d37a8a3b6595,
    0xc74a23b2ba0eecec,
    0x46f932eaea507767,
    0x3997c222da65fe1e,
    0xafba2586f2d042ee,
    0xd0d4d54ec2e5cb97,
    0x5167c41692bb501c,
    0x2e0934dea28ed965,
    0x66d8c0f56a91f461,
    0x19b6303d5aa47d18,
    0x980521650afae693,
    0xe76bd1ad3acf6fea,
    0x09a6c9329ac4bc9b,
    0x76c839faaaf135e2,
    0xf77b28a2faafae69,
    0x8815d86aca9a2710,
    0xc0c42c4102850a14,
    0xbfaadc8932b0836d,
    0x3e19cdd162ee18e6,
    0x41773d1952db919f,
    0x269b24ca6b12f26d,
    0x59f5d4025b277b14,
    0xd846c55a0b79e09f,
    0xa72835923b4c69e6,
    0xeff9c1b9f35344e2,
    0x90973171c366cd9b,
    0x1124202993385610,
    0x6e4ad0e1a30ddf69,
    0x8087c87e03060c18,
    0xffe938b633338561,
    0x7e5a29ee636d1eea,
    0x0134d92653589793,
    0x49e52d0d9b47ba97,
    0x368bddc5ab7233ee,
    0xb738cc9dfb2ca865,
    0xc8563c55cb19211c,
    0x5e7bdbf1e3ac9dec,
    0x21152b39d3991495,
    0xa0a63a6183c78f1e,
    0xdfc8caa9b3f20667,
    0x97193e827bed2b63,
    0xe877ce4a4bd8a21a,
    0x69c4df121b863991,
    0x16aa2fda2bb3b0e8,
    0xf86737458bb86399,
    0x8709c78dbb8deae0,
    0x06bad6d5ebd3716b,
    0x79d4261ddbe6f812,
    0x3105d23613f9d516,
    0x4e6b22fe23cc5c6f,
    0xcfd833a67392c7e4,
    0xb0b6c36e43a74e9d,
    0x9a6c9329ac4bc9b5,
    0xe50263e19c7e40cc,
    0x64b172b9cc20db47,
    0x1bdf8271fc15523e,
    0x530e765a340a7f3a,
    0x2c608692043ff643,
    0xadd397ca54616dc8,
    0xd2bd67026454e4b1,
    0x3c707f9dc45f37c0,
    0x431e8f55f46abeb9,
    0xc2ad9e0da4342532,
    0xbdc36ec59401ac4b,
    0xf5129aee5c1e814f,
    0x8a7c6a266c2b0836,
    0x0bcf7b7e3c7593bd,
    0x74a18bb60c401ac4,
    0xe28c6c1224f5a634,
    0x9de29cda14c02f4d,
    0x1c518d82449eb4c6,
    0x633f7d4a74ab3dbf,
    0x2bee8961bcb410bb,
    0x548079a98c8199c2,
    0xd53368f1dcdf0249,
    0xaa5d9839ecea8b30,
    0x449080a64ce15841,
    0x3bfe706e7cd4d138,
    0xba4d61362c8a4ab3,
    0xc52391fe1cbfc3ca,
    0x8df265d5d4a0eece,
    0xf29c951de49567b7,
    0x732f8445b4cbfc3c,
    0x0c41748d84fe7545,
    0x6bad6d5ebd3716b7,
    0x14c39d968d029fce,
    0x95708ccedd5c0445,
    0xea1e7c06ed698d3c,
    0xa2cf882d2576a038,
    0xdda178e515432941,
    0x5c1269bd451db2ca,
    0x237c997575283bb3,
    0xcdb181ead523e8c2,
    0xb2df7122e51661bb,
    0x336c607ab548fa30,
    0x4c0290b2857d7349,
    0x04d364994d625e4d,
    0x7bbd94517d57d734,
    0xfa0e85092d094cbf,
    0x856075c11d3cc5c6,
    0x134d926535897936,
    0x6c2362ad05bcf04f,
    0xed9073f555e26bc4,
    0x92fe833d65d7e2bd,
    0xda2f7716adc8cfb9,
    0xa54187de9dfd46c0,
    0x24f29686cda3dd4b,
    0x5b9c664efd965432,
    0xb5517ed15d9d8743,
    0xca3f8e196da80e3a,
    0x4b8c9f413df695b1,
    0x34e26f890dc31cc8,
    0x7c339ba2c5dc31cc,
    0x035d6b6af5e9b8b5,
    0x82ee7a32a5b7233e,
    0xfd808afa9582aa47,
    0x4d364994d625e4da,
    0x3258b95ce6106da3,
    0xb3eba804b64ef628,
    0xcc8558cc867b7f51,
    0x8454ace74e645255,
    0xfb3a5c2f7e51db2c,
    0x7a894d772e0f40a7,
    0x05e7bdbf1e3ac9de,
    0xeb2aa520be311aaf,
    0x944455e88e0493d6,
    0x15f744b0de5a085d,
    0x6a99b478ee6f8124,
    0x224840532670ac20,
    0x5d26b09b16452559,
    0xdc95a1c3461bbed2,
    0xa3fb510b762e37ab,
    0x35d6b6af5e9b8b5b,
    0x4ab846676eae0222,
    0xcb0b573f3ef099a9,
    0xb465a7f70ec510d0,
    0xfcb453dcc6da3dd4,
    0x83daa314f6efb4ad,
    0x0269b24ca6b12f26,
    0x7d0742849684a65f,
    0x93ca5a1b368f752e,
    0xeca4aad306bafc57,
    0x6d17bb8b56e467dc,
    0x12794b4366d1eea5,
    0x5aa8bf68aecec3a1,
    0x25c64fa09efb4ad8,
    0xa4755ef8cea5d153,
    0xdb1bae30fe90582a,
    0xbcf7b7e3c7593bd8,
    0xc399472bf76cb2a1,
    0x422a5673a732292a,
    0x3d44a6bb9707a053,
    0x759552905f188d57,
    0x0afba2586f2d042e,
    0x8b48b3003f739fa5,
    0xf42643c80f4616dc,
    0x1aeb5b57af4dc5ad,
    0x6585ab9f9f784cd4,
    0xe436bac7cf26d75f,
    0x9b584a0fff135e26,
    0xd389be24370c7322,
    0xace74eec0739fa5b,
    0x2d545fb4576761d0,
    0x523aaf7c6752e8a9,
    0xc41748d84fe75459,
    0xbb79b8107fd2dd20,
    0x3acaa9482f8c46ab,
    0x45a459801fb9cfd2,
    0x0d75adabd7a6e2d6,
    0x721b5d63e7936baf,
    0xf3a84c3bb7cdf024,
    0x8cc6bcf387f8795d,
    0x620ba46c27f3aa2c,
    0x1d6554a417c62355,
    0x9cd645fc4798b8de,
    0xe3b8b53477ad31a7,
    0xab69411fbfb21ca3,
    0xd407b1d78f8795da,
    0x55b4a08fdfd90e51,
    0x2ada5047efec8728,
];

// CRC table for the NVME polynomial in the range (`1 = x⁷⁹, …, 128 = x⁷²`).
// Generated by running `./build_table 1`.
static TABLE_1: [u64; 256] = [
    0x0000000000000000,
    0x8776a97d73bddf69,
    0x3a3474a9bfec2db9,
    0xbd42ddd4cc51f2d0,
    0x7468e9537fd85b72,
    0xf31e402e0c65841b,
    0x4e5c9dfac03476cb,
    0xc92a3487b389a9a2,
    0xe8d1d2a6ffb0b6e4,
    0x6fa77bdb8c0d698d,
    0xd2e5a60f405c9b5d,
    0x55930f7233e14434,
    0x9cb93bf58068ed96,
    0x1bcf9288f3d532ff,
    0xa68d4f5c3f84c02f,
    0x21fbe6214c391f46,
    0xe57a831ea7f6fea3,
    0x620c2a63d44b21ca,
    0xdf4ef7b7181ad31a,
    0x58385eca6ba70c73,
    0x91126a4dd82ea5d1,
    0x1664c330ab937ab8,
    0xab261ee467c28868,
    0x2c50b799147f5701,
    0x0dab51b858464847,
    0x8addf8c52bfb972e,
    0x379f2511e7aa65fe,
    0xb0e98c6c9417ba97,
    0x79c3b8eb279e1335,
    0xfeb511965423cc5c,
    0x43f7cc4298723e8c,
    0xc481653febcfe1e5,
    0xfe2c206e177a6e2d,
    0x795a891364c7b144,
    0xc41854c7a8964394,
    0x436efdbadb2b9cfd,
    0x8a44c93d68a2355f,
    0x0d3260401b1fea36,
    0xb070bd94d74e18e6,
    0x370614e9a4f3c78f,
    0x16fdf2c8e8cad8c9,
    0x918b5bb59b7707a0,
    0x2cc986615726f570,
    0xabbf2f1c249b2a19,
    0x62951b9b971283bb,
    0xe5e3b2e6e4af5cd2,
    0x58a16f3228feae02,
    0xdfd7c64f5b43716b,
    0x1b56a370b08c908e,
    0x9c200a0dc3314fe7,
    0x2162d7d90f60bd37,
    0xa6147ea47cdd625e,
    0x6f3e4a23cf54cbfc,
    0xe848e35ebce91495,
    0x550a3e8a70b8e645,
    0xd27c97f70305392c,
    0xf38771d64f3c266a,
    0x74f1d8ab3c81f903,
    0xc9b3057ff0d00bd3,
    0x4ec5ac02836dd4ba,
    0x87ef988530e47d18,
    0x009931f84359a271,
    0xbddbec2c8f0850a1,
    0x3aad4551fcb58fc8,
    0xc881668f76634f31,
    0x4ff7cff205de9058,
    0xf2b51226c98f6288,
    0x75c3bb5bba32bde1,
    0xbce98fdc09bb1443,
    0x3b9f26a17a06cb2a,
    0x86ddfb75b65739fa,
    0x01ab5208c5eae693,
    0x2050b42989d3f9d5,
    0xa7261d54fa6e26bc,
    0x1a64c080363fd46c,
    0x9d1269fd45820b05,
    0x54385d7af60ba2a7,
    0xd34ef40785b67dce,
    0x6e0c29d349e78f1e,
    0xe97a80ae3a5a5077,
    0x2dfbe591d195b192,
    0xaa8d4ceca2286efb,
    0x17cf91386e799c2b,
    0x90b938451dc44342,
    0x59930cc2ae4deae0,
    0xdee5a5bfddf03589,
    0x63a7786b11a1c759,
    0xe4d1d116621c1830,
    0xc52a37372e250776,
    0x425c9e4a5d98d81f,
    0xff1e439e91c92acf,
    0x7868eae3e274f5a6,
    0xb142de6451fd5c04,
    0x363477192240836d,
    0x8b76aacdee1171bd,
    0x0c0003b09dacaed4,
    0x36ad46e16119211c,
    0xb1dbef9c12a4fe75,
    0x0c993248def50ca5,
    0x8bef9b35ad48d3cc,
    0x42c5afb21ec17a6e,
    0xc5b306cf6d7ca507,
    0x78f1db1ba12d57d7,
    0xff877266d29088be,
    0xde7c94479ea997f8,
    0x590a3d3aed144891,
    0xe448e0ee2145ba41,
    0x633e499352f86528,
    0xaa147d14e171cc8a,
    0x2d62d46992cc13e3,
    0x902009bd5e9de133,
    0x1756a0c02d203e5a,
    0xd3d7c5ffc6efdfbf,
    0x54a16c82b55200d6,
    0xe9e3b1567903f206,
    0x6e95182b0abe2d6f,
    0xa7bf2cacb93784cd,
    0x20c985d1ca8a5ba4,
    0x9d8b580506dba974,
    0x1afdf1787566761d,
    0x3b061759395f695b,
    0xbc70be244ae2b632,
    0x013263f086b344e2,
    0x8644ca8df50e9b8b,
    0x4f6efe0a46873229,
    0xc8185777353aed40,
    0x755a8aa3f96b1f90,
    0xf22c23de8ad6c0f9,
    0xa5dbeb4db4510d09,
    0x22ad4230c7ecd260,
    0x9fef9fe40bbd20b0,
    0x189936997800ffd9,
    0xd1b3021ecb89567b,
    0x56c5ab63b8348912,
    0xeb8776b774657bc2,
    0x6cf1dfca07d8a4ab,
    0x4d0a39eb4be1bbed,
    0xca7c9096385c6484,
    0x773e4d42f40d9654,
    0xf048e43f87b0493d,
    0x3962d0b83439e09f,
    0xbe1479c547843ff6,
    0x0356a4118bd5cd26,
    0x84200d6cf868124f,
    0x40a1685313a7f3aa,
    0xc7d7c12e601a2cc3,
    0x7a951cfaac4bde13,
    0xfde3b587dff6017a,
    0x34c981006c7fa8d8,
    0xb3bf287d1fc277b1,
    0x0efdf5a9d3938561,
    0x898b5cd4a02e5a08,
    0xa870baf5ec17454e,
    0x2f0613889faa9a27,
    0x9244ce5c53fb68f7,
    0x153267212046b79e,
    0xdc1853a693cf1e3c,
    0x5b6efadbe072c155,
    0xe62c270f2c233385,
    0x615a8e725f9eecec,
    0x5bf7cb23a32b6324,
    0xdc81625ed096bc4d,
    0x61c3bf8a1cc74e9d,
    0xe6b516f76f7a91f4,
    0x2f9f2270dcf33856,
    0xa8e98b0daf4ee73f,
    0x15ab56d9631f15ef,
    0x92ddffa410a2ca86,
    0xb32619855c9bd5c0,
    0x3450b0f82f260aa9,
    0x89126d2ce377f879,
    0x0e64c45190ca2710,
    0xc74ef0d623438eb2,
    0x403859ab50fe51db,
    0xfd7a847f9cafa30b,
    0x7a0c2d02ef127c62,
    0xbe8d483d04dd9d87,
    0x39fbe140776042ee,
    0x84b93c94bb31b03e,
    0x03cf95e9c88c6f57,
    0xcae5a16e7b05c6f5,
    0x4d93081308b8199c,
    0xf0d1d5c7c4e9eb4c,
    0x77a77cbab7543425,
    0x565c9a9bfb6d2b63,
    0xd12a33e688d0f40a,
    0x6c68ee32448106da,
    0xeb1e474f373cd9b3,
    0x223473c884b57011,
    0xa542dab5f708af78,
    0x180007613b595da8,
    0x9f76ae1c48e482c1,
    0x6d5a8dc2c2324238,
    0xea2c24bfb18f9d51,
    0x576ef96b7dde6f81,
    0xd01850160e63b0e8,
    0x19326491bdea194a,
    0x9e44cdecce57c623,
    0x23061038020634f3,
    0xa470b94571bbeb9a,
    0x858b5f643d82f4dc,
    0x02fdf6194e3f2bb5,
    0xbfbf2bcd826ed965,
    0x38c982b0f1d3060c,
    0xf1e3b637425aafae,
    0x76951f4a31e770c7,
    0xcbd7c29efdb68217,
    0x4ca16be38e0b5d7e,
    0x88200edc65c4bc9b,
    0x0f56a7a1167963f2,
    0xb2147a75da289122,
    0x3562d308a9954e4b,
    0xfc48e78f1a1ce7e9,
    0x7b3e4ef269a13880,
    0xc67c9326a5f0ca50,
    0x410a3a5bd64d1539,
    0x60f1dc7a9a740a7f,
    0xe7877507e9c9d516,
    0x5ac5a8d3259827c6,
    0xddb301ae5625f8af,
    0x14993529e5ac510d,
    0x93ef9c5496118e64,
    0x2ead41805a407cb4,
    0xa9dbe8fd29fda3dd,
    0x9376adacd5482c15,
    0x140004d1a6f5f37c,
    0xa942d9056aa401ac,
    0x2e3470781919dec5,
    0xe71e44ffaa907767,
    0x6068ed82d92da80e,
    0xdd2a3056157c5ade,
    0x5a5c992b66c185b7,
    0x7ba77f0a2af89af1,
    0xfcd1d67759454598,
    0x41930ba39514b748,
    0xc6e5a2dee6a96821,
    0x0fcf96595520c183,
    0x88b93f24269d1eea,
    0x35fbe2f0eaccec3a,
    0xb28d4b8d99713353,
    0x760c2eb272bed2b6,
    0xf17a87cf01030ddf,
    0x4c385a1bcd52ff0f,
    0xcb4ef366beef2066,
    0x0264c7e10d6689c4,
    0x85126e9c7edb56ad,
    0x3850b348b28aa47d,
    0xbf261a35c1377b14,
    0x9eddfc148d0e6452,
    0x19ab5569feb3bb3b,
    0xa4e988bd32e249eb,
    0x239f21c0415f9682,
    0xeab51547f2d63f20,
    0x6dc3bc3a816be049,
    0xd08161ee4d3a1299,
    0x57f7c8933e87cdf0,
];

// CRC table for the NVME polynomial in the range (`1 = x⁸⁷, …, 128 = x⁸⁰`).
// Generated by running `./build_table 2`.
static TABLE_2: [u64; 256] = [
    0x0000000000000000,
    0xff6e4e1f4e4038be,
    0xca05ba6dc417e217,
    0x356bf4728a57daa9,
    0xa0d25288d0b85745,
    0x5fbc1c979ef86ffb,
    0x6ad7e8e514afb552,
    0x95b9a6fa5aef8dec,
    0x757d8342f9e73de1,
    0x8a13cd5db7a7055f,
    0xbf78392f3df0dff6,
    0x4016773073b0e748,
    0xd5afd1ca295f6aa4,
    0x2ac19fd5671f521a,
    0x1faa6ba7ed4888b3,
    0xe0c425b8a308b00d,
    0xeafb0685f3ce7bc2,
    0x1595489abd8e437c,
    0x20febce837d999d5,
    0xdf90f2f77999a16b,
    0x4a29540d23762c87,
    0xb5471a126d361439,
    0x802cee60e761ce90,
    0x7f42a07fa921f62e,
    0x9f8685c70a294623,
    0x60e8cbd844697e9d,
    0x55833faace3ea434,
    0xaaed71b5807e9c8a,
    0x3f54d74fda911166,
    0xc03a995094d129d8,
    0xf5516d221e86f371,
    0x0a3f233d50c6cbcf,
    0xe12f2b58bf0b64ef,
    0x1e416547f14b5c51,
    0x2b2a91357b1c86f8,
    0xd444df2a355cbe46,
    0x41fd79d06fb333aa,
    0xbe9337cf21f30b14,
    0x8bf8c3bdaba4d1bd,
    0x74968da2e5e4e903,
    0x9452a81a46ec590e,
    0x6b3ce60508ac61b0,
    0x5e57127782fbbb19,
    0xa1395c68ccbb83a7,
    0x3480fa9296540e4b,
    0xcbeeb48dd81436f5,
    0xfe8540ff5243ec5c,
    0x01eb0ee01c03d4e2,
    0x0bd42ddd4cc51f2d,
    0xf4ba63c202852793,
    0xc1d197b088d2fd3a,
    0x3ebfd9afc692c584,
    0xab067f559c7d4868,
    0x5468314ad23d70d6,
    0x6103c538586aaa7f,
    0x9e6d8b27162a92c1,
    0x7ea9ae9fb52222cc,
    0x81c7e080fb621a72,
    0xb4ac14f27135c0db,
    0x4bc25aed3f75f865,
    0xde7bfc17659a7589,
    0x2115b2082bda4d37,
    0x147e467aa18d979e,
    0xeb100865efcdaf20,
    0xf68770e226815ab5,
    0x09e93efd68c1620b,
    0x3c82ca8fe296b8a2,
    0xc3ec8490acd6801c,
    0x5655226af6390df0,
    0xa93b6c75b879354e,
    0x9c509807322eefe7,
    0x633ed6187c6ed759,
    0x83faf3a0df666754,
    0x7c94bdbf91265fea,
    0x49ff49cd1b718543,
    0xb69107d25531bdfd,
    0x2328a1280fde3011,
    0xdc46ef37419e08af,
    0xe92d1b45cbc9d206,
    0x1643555a8589eab8,
    0x1c7c7667d54f2177,
    0xe31238789b0f19c9,
    0xd679cc0a1158c360,
    0x291782155f18fbde,
    0xbcae24ef05f77632,
    0x43c06af04bb74e8c,
    0x76ab9e82c1e09425,
    0x89c5d09d8fa0ac9b,
    0x6901f5252ca81c96,
    0x966fbb3a62e82428,
    0xa3044f48e8bffe81,
    0x5c6a0157a6ffc63f,
    0xc9d3a7adfc104bd3,
    0x36bde9b2b250736d,
    0x03d61dc03807a9c4,
    0xfcb853df7647917a,
    0x17a85bba998a3e5a,
    0xe8c615a5d7ca06e4,
    0xddade1d75d9ddc4d,
    0x22c3afc813dde4f3,
    0xb77a09324932691f,
    0x4814472d077251a1,
    0x7d7fb35f8d258b08,
    0x8211fd40c365b3b6,
    0x62d5d8f8606d03bb,
    0x9dbb96e72e2d3b05,
    0xa8d06295a47ae1ac,
    0x57be2c8aea3ad912,
    0xc2078a70b0d554fe,
    0x3d69c46ffe956c40,
    0x0802301d74c2b6e9,
    0xf76c7e023a828e57,
    0xfd535d3f6a444598,
    0x023d132024047d26,
    0x3756e752ae53a78f,
    0xc838a94de0139f31,
    0x5d810fb7bafc12dd,
    0xa2ef41a8f4bc2a63,
    0x9784b5da7eebf0ca,
    0x68eafbc530abc874,
    0x882ede7d93a37879,
    0x77409062dde340c7,
    0x422b641057b49a6e,
    0xbd452a0f19f4a2d0,
    0x28fc8cf5431b2f3c,
    0xd792c2ea0d5b1782,
    0xe2f93698870ccd2b,
    0x1d977887c94cf595,
    0xd9d7c79715952601,
    0x26b989885bd51ebf,
    0x13d27dfad182c416,
    0xecbc33e59fc2fca8,
    0x7905951fc52d7144,
    0x866bdb008b6d49fa,
    0xb3002f72013a9353,
    0x4c6e616d4f7aabed,
    0xacaa44d5ec721be0,
    0x53c40acaa232235e,
    0x66affeb82865f9f7,
    0x99c1b0a76625c149,
    0x0c78165d3cca4ca5,
    0xf3165842728a741b,
    0xc67dac30f8ddaeb2,
    0x3913e22fb69d960c,
    0x332cc112e65b5dc3,
    0xcc428f0da81b657d,
    0xf9297b7f224cbfd4,
    0x064735606c0c876a,
    0x93fe939a36e30a86,
    0x6c90dd8578a33238,
    0x59fb29f7f2f4e891,
    0xa69567e8bcb4d02f,
    0x465142501fbc6022,
    0xb93f0c4f51fc589c,
    0x8c54f83ddbab8235,
    0x733ab62295ebba8b,
    0xe68310d8cf043767,
    0x19ed5ec781440fd9,
    0x2c86aab50b13d570,
    0xd3e8e4aa4553edce,
    0x38f8eccfaa9e42ee,
    0xc796a2d0e4de7a50,
    0xf2fd56a26e89a0f9,
    0x0d9318bd20c99847,
    0x982abe477a2615ab,
    0x6744f05834662d15,
    0x522f042abe31f7bc,
    0xad414a35f071cf02,
    0x4d856f8d53797f0f,
    0xb2eb21921d3947b1,
    0x8780d5e0976e9d18,
    0x78ee9bffd92ea5a6,
    0xed573d0583c1284a,
    0x1239731acd8110f4,
    0x2752876847d6ca5d,
    0xd83cc9770996f2e3,
    0xd203ea4a5950392c,
    0x2d6da45517100192,
    0x180650279d47db3b,
    0xe7681e38d307e385,
    0x72d1b8c289e86e69,
    0x8dbff6ddc7a856d7,
    0xb8d402af4dff8c7e,
    0x47ba4cb003bfb4c0,
    0xa77e6908a0b704cd,
    0x58102717eef73c73,
    0x6d7bd36564a0e6da,
    0x92159d7a2ae0de64,
    0x07ac3b80700f5388,
    0xf8c2759f3e4f6b36,
    0xcda981edb418b19f,
    0x32c7cff2fa588921,
    0x2f50b77533147cb4,
    0xd03ef96a7d54440a,
    0xe5550d18f7039ea3,
    0x1a3b4307b943a61d,
    0x8f82e5fde3ac2bf1,
    0x70ecabe2adec134f,
    0x45875f9027bbc9e6,
    0xbae9118f69fbf158,
    0x5a2d3437caf34155,
    0xa5437a2884b379eb,
    0x90288e5a0ee4a342,
    0x6f46c04540a49bfc,
    0xfaff66bf1a4b1610,
    0x059128a0540b2eae,
    0x30fadcd2de5cf407,
    0xcf9492cd901cccb9,
    0xc5abb1f0c0da0776,
    0x3ac5ffef8e9a3fc8,
    0x0fae0b9d04cde561,
    0xf0c045824a8ddddf,
    0x6579e37810625033,
    0x9a17ad675e22688d,
    0xaf7c5915d475b224,
    0x5012170a9a358a9a,
    0xb0d632b2393d3a97,
    0x4fb87cad777d0229,
    0x7ad388dffd2ad880,
    0x85bdc6c0b36ae03e,
    0x1004603ae9856dd2,
    0xef6a2e25a7c5556c,
    0xda01da572d928fc5,
    0x256f944863d2b77b,
    0xce7f9c2d8c1f185b,
    0x3111d232c25f20e5,
    0x047a26404808fa4c,
    0xfb14685f0648c2f2,
    0x6eadcea55ca74f1e,
    0x91c380ba12e777a0,
    0xa4a874c898b0ad09,
    0x5bc63ad7d6f095b7,
    0xbb021f6f75f825ba,
    0x446c51703bb81d04,
    0x7107a502b1efc7ad,
    0x8e69eb1dffafff13,
    0x1bd04de7a54072ff,
    0xe4be03f8eb004a41,
    0xd1d5f78a615790e8,
    0x2ebbb9952f17a856,
    0x24849aa87fd16399,
    0xdbead4b731915b27,
    0xee8120c5bbc6818e,
    0x11ef6edaf586b930,
    0x8456c820af6934dc,
    0x7b38863fe1290c62,
    0x4e53724d6b7ed6cb,
    0xb13d3c52253eee75,
    0x51f919ea86365e78,
    0xae9757f5c87666c6,
    0x9bfca3874221bc6f,
    0x6492ed980c6184d1,
    0xf12b4b62568e093d,
    0x0e45057d18ce3183,
    0x3b2ef10f9299eb2a,
    0xc440bf10dcd9d394,
];

// CRC table for the NVME polynomial in the range (`1 = x⁹⁵, …, 128 = x⁸⁸`).
// Generated by running `./build_table 3`.
static TABLE_3: [u64; 256] = [
    0x0000000000000000,
    0x8211147cbaf96306,
    0x30fb0eaa2d655567,
    0xb2ea1ad6979c3661,
    0x61f61d545acaaace,
    0xe3e70928e033c9c8,
    0x510d13fe77afffa9,
    0xd31c0782cd569caf,
    0xc3ec3aa8b595559c,
    0x41fd2ed40f6c369a,
    0xf317340298f000fb,
    0x7106207e220963fd,
    0xa21a27fcef5fff52,
    0x200b338055a69c54,
    0x92e12956c23aaa35,
    0x10f03d2a78c3c933,
    0xb301530233bd3853,
    0x3110477e89445b55,
    0x83fa5da81ed86d34,
    0x01eb49d4a4210e32,
    0xd2f74e566977929d,
    0x50e65a2ad38ef19b,
    0xe20c40fc4412c7fa,
    0x601d5480feeba4fc,
    0x70ed69aa86286dcf,
    0xf2fc7dd63cd10ec9,
    0x40166700ab4d38a8,
    0xc207737c11b45bae,
    0x111b74fedce2c701,
    0x930a6082661ba407,
    0x21e07a54f1879266,
    0xa3f16e284b7ef160,
    0x52db80573fede3cd,
    0xd0ca942b851480cb,
    0x62208efd1288b6aa,
    0xe0319a81a871d5ac,
    0x332d9d0365274903,
    0xb13c897fdfde2a05,
    0x03d693a948421c64,
    0x81c787d5f2bb7f62,
    0x9137baff8a78b651,
    0x1326ae833081d557,
    0xa1ccb455a71de336,
    0x23dda0291de48030,
    0xf0c1a7abd0b21c9f,
    0x72d0b3d76a4b7f99,
    0xc03aa901fdd749f8,
    0x422bbd7d472e2afe,
    0xe1dad3550c50db9e,
    0x63cbc729b6a9b898,
    0xd121ddff21358ef9,
    0x5330c9839bccedff,
    0x802cce01569a7150,
    0x023dda7dec631256,
    0xb0d7c0ab7bff2437,
    0x32c6d4d7c1064731,
    0x2236e9fdb9c58e02,
    0xa027fd81033ced04,
    0x12cde75794a0db65,
    0x90dcf32b2e59b863,
    0x43c0f4a9e30f24cc,
    0xc1d1e0d559f647ca,
    0x733bfa03ce6a71ab,
    0xf12aee7f749312ad,
    0xa5b700ae7fdbc79a,
    0x27a614d2c522a49c,
    0x954c0e0452be92fd,
    0x175d1a78e847f1fb,
    0xc4411dfa25116d54,
    0x465009869fe80e52,
    0xf4ba135008743833,
    0x76ab072cb28d5b35,
    0x665b3a06ca4e9206,
    0xe44a2e7a70b7f100,
    0x56a034ace72bc761,
    0xd4b120d05dd2a467,
    0x07ad2752908438c8,
    0x85bc332e2a7d5bce,
    0x375629f8bde16daf,
    0xb5473d8407180ea9,
    0x16b653ac4c66ffc9,
    0x94a747d0f69f9ccf,
    0x264d5d066103aaae,
    0xa45c497adbfac9a8,
    0x77404ef816ac5507,
    0xf5515a84ac553601,
    0x47bb40523bc90060,
    0xc5aa542e81306366,
    0xd55a6904f9f3aa55,
    0x574b7d78430ac953,
    0xe5a167aed496ff32,
    0x67b073d26e6f9c34,
    0xb4ac7450a339009b,
    0x36bd602c19c0639d,
    0x84577afa8e5c55fc,
    0x06466e8634a536fa,
    0xf76c80f940362457,
    0x757d9485facf4751,
    0xc7978e536d537130,
    0x45869a2fd7aa1236,
    0x969a9dad1afc8e99,
    0x148b89d1a005ed9f,
    0xa66193073799dbfe,
    0x2470877b8d60b8f8,
    0x3480ba51f5a371cb,
    0xb691ae2d4f5a12cd,
    0x047bb4fbd8c624ac,
    0x866aa087623f47aa,
    0x5576a705af69db05,
    0xd767b3791590b803,
    0x658da9af820c8e62,
    0xe79cbdd338f5ed64,
    0x446dd3fb738b1c04,
    0xc67cc787c9727f02,
    0x7496dd515eee4963,
    0xf687c92de4172a65,
    0x259bceaf2941b6ca,
    0xa78adad393b8d5cc,
    0x1560c0050424e3ad,
    0x9771d479bedd80ab,
    0x8781e953c61e4998,
    0x0590fd2f7ce72a9e,
    0xb77ae7f9eb7b1cff,
    0x356bf38551827ff9,
    0xe677f4079cd4e356,
    0x6466e07b262d8050,
    0xd68cfaadb1b1b631,
    0x549deed10b48d537,
    0x7fb7270fa7201c5f,
    0xfda633731dd97f59,
    0x4f4c29a58a454938,
    0xcd5d3dd930bc2a3e,
    0x1e413a5bfdeab691,
    0x9c502e274713d597,
    0x2eba34f1d08fe3f6,
    0xacab208d6a7680f0,
    0xbc5b1da712b549c3,
    0x3e4a09dba84c2ac5,
    0x8ca0130d3fd01ca4,
    0x0eb1077185297fa2,
    0xddad00f3487fe30d,
    0x5fbc148ff286800b,
    0xed560e59651ab66a,
    0x6f471a25dfe3d56c,
    0xccb6740d949d240c,
    0x4ea760712e64470a,
    0xfc4d7aa7b9f8716b,
    0x7e5c6edb0301126d,
    0xad406959ce578ec2,
    0x2f517d2574aeedc4,
    0x9dbb67f3e332dba5,
    0x1faa738f59cbb8a3,
    0x0f5a4ea521087190,
    0x8d4b5ad99bf11296,
    0x3fa1400f0c6d24f7,
    0xbdb05473b69447f1,
    0x6eac53f17bc2db5e,
    0xecbd478dc13bb858,
    0x5e575d5b56a78e39,
    0xdc464927ec5eed3f,
    0x2d6ca75898cdff92,
    0xaf7db32422349c94,
    0x1d97a9f2b5a8aaf5,
    0x9f86bd8e0f51c9f3,
    0x4c9aba0cc207555c,
    0xce8bae7078fe365a,
    0x7c61b4a6ef62003b,
    0xfe70a0da559b633d,
    0xee809df02d58aa0e,
    0x6c91898c97a1c908,
    0xde7b935a003dff69,
    0x5c6a8726bac49c6f,
    0x8f7680a4779200c0,
    0x0d6794d8cd6b63c6,
    0xbf8d8e0e5af755a7,
    0x3d9c9a72e00e36a1,
    0x9e6df45aab70c7c1,
    0x1c7ce0261189a4c7,
    0xae96faf0861592a6,
    0x2c87ee8c3cecf1a0,
    0xff9be90ef1ba6d0f,
    0x7d8afd724b430e09,
    0xcf60e7a4dcdf3868,
    0x4d71f3d866265b6e,
    0x5d81cef21ee5925d,
    0xdf90da8ea41cf15b,
    0x6d7ac0583380c73a,
    0xef6bd4248979a43c,
    0x3c77d3a6442f3893,
    0xbe66c7dafed65b95,
    0x0c8cdd0c694a6df4,
    0x8e9dc970d3b30ef2,
    0xda0027a1d8fbdbc5,
    0x581133dd6202b8c3,
    0xeafb290bf59e8ea2,
    0x68ea3d774f67eda4,
    0xbbf63af58231710b,
    0x39e72e8938c8120d,
    0x8b0d345faf54246c,
    0x091c202315ad476a,
    0x19ec1d096d6e8e59,
    0x9bfd0975d797ed5f,
    0x291713a3400bdb3e,
    0xab0607dffaf2b838,
    0x781a005d37a42497,
    0xfa0b14218d5d4791,
    0x48e10ef71ac171f0,
    0xcaf01a8ba03812f6,
    0x690174a3eb46e396,
    0xeb1060df51bf8090,
    0x59fa7a09c623b6f1,
    0xdbeb6e757cdad5f7,
    0x08f769f7b18c4958,
    0x8ae67d8b0b752a5e,
    0x380c675d9ce91c3f,
    0xba1d732126107f39,
    0xaaed4e0b5ed3b60a,
    0x28fc5a77e42ad50c,
    0x9a1640a173b6e36d,
    0x180754ddc94f806b,
    0xcb1b535f04191cc4,
    0x490a4723bee07fc2,
    0xfbe05df5297c49a3,
    0x79f1498993852aa5,
    0x88dba7f6e7163808,
    0x0acab38a5def5b0e,
    0xb820a95cca736d6f,
    0x3a31bd20708a0e69,
    0xe92dbaa2bddc92c6,
    0x6b3caede0725f1c0,
    0xd9d6b40890b9c7a1,
    0x5bc7a0742a40a4a7,
    0x4b379d5e52836d94,
    0xc9268922e87a0e92,
    0x7bcc93f47fe638f3,
    0xf9dd8788c51f5bf5,
    0x2ac1800a0849c75a,
    0xa8d09476b2b0a45c,
    0x1a3a8ea0252c923d,
    0x982b9adc9fd5f13b,
    0x3bdaf4f4d4ab005b,
    0xb9cbe0886e52635d,
    0x0b21fa5ef9ce553c,
    0x8930ee224337363a,
    0x5a2ce9a08e61aa95,
    0xd83dfddc3498c993,
    0x6ad7e70aa304fff2,
    0xe8c6f37619fd9cf4,
    0xf836ce5c613e55c7,
    0x7a27da20dbc736c1,
    0xc8cdc0f64c5b00a0,
    0x4adcd48af6a263a6,
    0x99c0d3083bf4ff09,
    0x1bd1c774810d9c0f,
    0xa93bdda21691aa6e,
    0x2b2ac9deac68c968,
];

// CRC table for the NVME polynomial in the range (`1 = x¹⁰³, …, 128 = x⁹⁶`).
// Generated by running `./build_table 4`.
static TABLE_4: [u64; 256] = [
    0x0000000000000000,
    0x373d15f784905d1e,
    0x6e7a2bef0920ba3c,
    0x59473e188db0e722,
    0xdcf457de12417478,
    0xebc9422996d12966,
    0xb28e7c311b61ce44,
    0x85b369c69ff1935a,
    0x8d3189ef7c157b9b,
    0xba0c9c18f8852685,
    0xe34ba2007535c1a7,
    0xd476b7f7f1a59cb9,
    0x51c5de316e540fe3,
    0x66f8cbc6eac452fd,
    0x3fbff5de6774b5df,
    0x0882e029e3e4e8c1,
    0x2eba358da0bd645d,
    0x1987207a242d3943,
    0x40c01e62a99dde61,
    0x77fd0b952d0d837f,
    0xf24e6253b2fc1025,
    0xc57377a4366c4d3b,
    0x9c3449bcbbdcaa19,
    0xab095c4b3f4cf707,
    0xa38bbc62dca81fc6,
    0x94b6a995583842d8,
    0xcdf1978dd588a5fa,
    0xfacc827a5118f8e4,
    0x7f7febbccee96bbe,
    0x4842fe4b4a7936a0,
    0x1105c053c7c9d182,
    0x2638d5a443598c9c,
    0x5d746b1b417ac8ba,
    0x6a497eecc5ea95a4,
    0x330e40f4485a7286,
    0x04335503ccca2f98,
    0x81803cc5533bbcc2,
    0xb6bd2932d7abe1dc,
    0xeffa172a5a1b06fe,
    0xd8c702ddde8b5be0,
    0xd045e2f43d6fb321,
    0xe778f703b9ffee3f,
    0xbe3fc91b344f091d,
    0x8902dcecb0df5403,
    0x0cb1b52a2f2ec759,
    0x3b8ca0ddabbe9a47,
    0x62cb9ec5260e7d65,
    0x55f68b32a29e207b,
    0x73ce5e96e1c7ace7,
    0x44f34b616557f1f9,
    0x1db47579e8e716db,
    0x2a89608e6c774bc5,
    0xaf3a0948f386d89f,
    0x98071cbf77168581,
    0xc14022a7faa662a3,
    0xf67d37507e363fbd,
    0xfeffd7799dd2d77c,
    0xc9c2c28e19428a62,
    0x9085fc9694f26d40,
    0xa7b8e9611062305e,
    0x220b80a78f93a304,
    0x153695500b03fe1a,
    0x4c71ab4886b31938,
    0x7b4cbebf02234426,
    0xbae8d63682f59174,
    0x8dd5c3c10665cc6a,
    0xd492fdd98bd52b48,
    0xe3afe82e0f457656,
    0x661c81e890b4e50c,
    0x5121941f1424b812,
    0x0866aa0799945f30,
    0x3f5bbff01d04022e,
    0x37d95fd9fee0eaef,
    0x00e44a2e7a70b7f1,
    0x59a37436f7c050d3,
    0x6e9e61c173500dcd,
    0xeb2d0807eca19e97,
    0xdc101df06831c389,
    0x855723e8e58124ab,
    0xb26a361f611179b5,
    0x9452e3bb2248f529,
    0xa36ff64ca6d8a837,
    0xfa28c8542b684f15,
    0xcd15dda3aff8120b,
    0x48a6b46530098151,
    0x7f9ba192b499dc4f,
    0x26dc9f8a39293b6d,
    0x11e18a7dbdb96673,
    0x19636a545e5d8eb2,
    0x2e5e7fa3dacdd3ac,
    0x771941bb577d348e,
    0x4024544cd3ed6990,
    0xc5973d8a4c1cfaca,
    0xf2aa287dc88ca7d4,
    0xabed1665453c40f6,
    0x9cd00392c1ac1de8,
    0xe79cbd2dc38f59ce,
    0xd0a1a8da471f04d0,
    0x89e696c2caafe3f2,
    0xbedb83354e3fbeec,
    0x3b68eaf3d1ce2db6,
    0x0c55ff04555e70a8,
    0x5512c11cd8ee978a,
    0x622fd4eb5c7eca94,
    0x6aad34c2bf9a2255,
    0x5d9021353b0a7f4b,
    0x04d71f2db6ba9869,
    0x33ea0ada322ac577,
    0xb659631caddb562d,
    0x816476eb294b0b33,
    0xd82348f3a4fbec11,
    0xef1e5d04206bb10f,
    0xc92688a063323d93,
    0xfe1b9d57e7a2608d,
    0xa75ca34f6a1287af,
    0x9061b6b8ee82dab1,
    0x15d2df7e717349eb,
    0x22efca89f5e314f5,
    0x7ba8f4917853f3d7,
    0x4c95e166fcc3aec9,
    0x4417014f1f274608,
    0x732a14b89bb71b16,
    0x2a6d2aa01607fc34,
    0x1d503f579297a12a,
    0x98e356910d663270,
    0xafde436689f66f6e,
    0xf6997d7e0446884c,
    0xc1a4688980d6d552,
    0x41088a3e5d7cb183,
    0x76359fc9d9ecec9d,
    0x2f72a1d1545c0bbf,
    0x184fb426d0cc56a1,
    0x9dfcdde04f3dc5fb,
    0xaac1c817cbad98e5,
    0xf386f60f461d7fc7,
    0xc4bbe3f8c28d22d9,
    0xcc3903d12169ca18,
    0xfb041626a5f99706,
    0xa243283e28497024,
    0x957e3dc9acd92d3a,
    0x10cd540f3328be60,
    0x27f041f8b7b8e37e,
    0x7eb77fe03a08045c,
    0x498a6a17be985942,
    0x6fb2bfb3fdc1d5de,
    0x588faa44795188c0,
    0x01c8945cf4e16fe2,
    0x36f581ab707132fc,
    0xb346e86def80a1a6,
    0x847bfd9a6b10fcb8,
    0xdd3cc382e6a01b9a,
    0xea01d67562304684,
    0xe283365c81d4ae45,
    0xd5be23ab0544f35b,
    0x8cf91db388f41479,
    0xbbc408440c644967,
    0x3e7761829395da3d,
    0x094a747517058723,
    0x500d4a6d9ab56001,
    0x67305f9a1e253d1f,
    0x1c7ce1251c067939,
    0x2b41f4d298962427,
    0x7206caca1526c305,
    0x453bdf3d91b69e1b,
    0xc088b6fb0e470d41,
    0xf7b5a30c8ad7505f,
    0xaef29d140767b77d,
    0x99cf88e383f7ea63,
    0x914d68ca601302a2,
    0xa6707d3de4835fbc,
    0xff3743256933b89e,
    0xc80a56d2eda3e580,
    0x4db93f14725276da,
    0x7a842ae3f6c22bc4,
    0x23c314fb7b72cce6,
    0x14fe010cffe291f8,
    0x32c6d4a8bcbb1d64,
    0x05fbc15f382b407a,
    0x5cbcff47b59ba758,
    0x6b81eab0310bfa46,
    0xee328376aefa691c,
    0xd90f96812a6a3402,
    0x8048a899a7dad320,
    0xb775bd6e234a8e3e,
    0xbff75d47c0ae66ff,
    0x88ca48b0443e3be1,
    0xd18d76a8c98edcc3,
    0xe6b0635f4d1e81dd,
    0x63030a99d2ef1287,
    0x543e1f6e567f4f99,
    0x0d792176dbcfa8bb,
    0x3a4434815f5ff5a5,
    0xfbe05c08df8920f7,
    0xccdd49ff5b197de9,
    0x959a77e7d6a99acb,
    0xa2a762105239c7d5,
    0x27140bd6cdc8548f,
    0x10291e2149580991,
    0x496e2039c4e8eeb3,
    0x7e5335ce4078b3ad,
    0x76d1d5e7a39c5b6c,
    0x41ecc010270c0672,
    0x18abfe08aabce150,
    0x2f96ebff2e2cbc4e,
    0xaa258239b1dd2f14,
    0x9d1897ce354d720a,
    0xc45fa9d6b8fd9528,
    0xf362bc213c6dc836,
    0xd55a69857f3444aa,
    0xe2677c72fba419b4,
    0xbb20426a7614fe96,
    0x8c1d579df284a388,
    0x09ae3e5b6d7530d2,
    0x3e932bace9e56dcc,
    0x67d415b464558aee,
    0x50e90043e0c5d7f0,
    0x586be06a03213f31,
    0x6f56f59d87b1622f,
    0x3611cb850a01850d,
    0x012cde728e91d813,
    0x849fb7b411604b49,
    0xb3a2a24395f01657,
    0xeae59c5b1840f175,
    0xddd889ac9cd0ac6b,
    0xa69437139ef3e84d,
    0x91a922e41a63b553,
    0xc8ee1cfc97d35271,
    0xffd3090b13430f6f,
    0x7a6060cd8cb29c35,
    0x4d5d753a0822c12b,
    0x141a4b2285922609,
    0x23275ed501027b17,
    0x2ba5befce2e693d6,
    0x1c98ab0b6676cec8,
    0x45df9513ebc629ea,
    0x72e280e46f5674f4,
    0xf751e922f0a7e7ae,
    0xc06cfcd57437bab0,
    0x992bc2cdf9875d92,
    0xae16d73a7d17008c,
    0x882e029e3e4e8c10,
    0xbf131769baded10e,
    0xe6542971376e362c,
    0xd1693c86b3fe6b32,
    0x54da55402c0ff868,
    0x63e740b7a89fa576,
    0x3aa07eaf252f4254,
    0x0d9d6b58a1bf1f4a,
    0x051f8b71425bf78b,
    0x32229e86c6cbaa95,
    0x6b65a09e4b7b4db7,
    0x5c58b569cfeb10a9,
    0xd9ebdcaf501a83f3,
    0xeed6c958d48adeed,
    0xb791f740593a39cf,
    0x80ace2b7ddaa64d1,
];

// CRC table for the NVME polynomial in the range (`1 = x¹¹¹, …, 128 = x¹⁰⁴`).
// Generated by running `./build_table 5`.
static TABLE_5: [u64; 256] = [
    0x0000000000000000,
    0xe9742a79ef04a5d4,
    0xe63172a0869ed8c3,
    0x0f4558d9699a7d17,
    0xf8bbc31255aa22ed,
    0x11cfe96bbaae8739,
    0x1e8ab1b2d334fa2e,
    0xf7fe9bcb3c305ffa,
    0xc5aea077f3c3d6b1,
    0x2cda8a0e1cc77365,
    0x239fd2d7755d0e72,
    0xcaebf8ae9a59aba6,
    0x3d156365a669f45c,
    0xd461491c496d5188,
    0xdb2411c520f72c9f,
    0x32503bbccff3894b,
    0xbf8466bcbf103e09,
    0x56f04cc550149bdd,
    0x59b5141c398ee6ca,
    0xb0c13e65d68a431e,
    0x473fa5aeeaba1ce4,
    0xae4b8fd705beb930,
    0xa10ed70e6c24c427,
    0x487afd77832061f3,
    0x7a2ac6cb4cd3e8b8,
    0x935eecb2a3d74d6c,
    0x9c1bb46bca4d307b,
    0x756f9e12254995af,
    0x829105d91979ca55,
    0x6be52fa0f67d6f81,
    0x64a077799fe71296,
    0x8dd45d0070e3b742,
    0x4bd1eb2a26b7ef79,
    0xa2a5c153c9b34aad,
    0xade0998aa02937ba,
    0x4494b3f34f2d926e,
    0xb36a2838731dcd94,
    0x5a1e02419c196840,
    0x555b5a98f5831557,
    0xbc2f70e11a87b083,
    0x8e7f4b5dd57439c8,
    0x670b61243a709c1c,
    0x684e39fd53eae10b,
    0x813a1384bcee44df,
    0x76c4884f80de1b25,
    0x9fb0a2366fdabef1,
    0x90f5faef0640c3e6,
    0x7981d096e9446632,
    0xf4558d9699a7d170,
    0x1d21a7ef76a374a4,
    0x1264ff361f3909b3,
    0xfb10d54ff03dac67,
    0x0cee4e84cc0df39d,
    0xe59a64fd23095649,
    0xeadf3c244a932b5e,
    0x03ab165da5978e8a,
    0x31fb2de16a6407c1,
    0xd88f07988560a215,
    0xd7ca5f41ecfadf02,
    0x3ebe753803fe7ad6,
    0xc940eef33fce252c,
    0x2034c48ad0ca80f8,
    0x2f719c53b950fdef,
    0xc605b62a5654583b,
    0x97a3d6544d6fdef2,
    0x7ed7fc2da26b7b26,
    0x7192a4f4cbf10631,
    0x98e68e8d24f5a3e5,
    0x6f18154618c5fc1f,
    0x866c3f3ff7c159cb,
    0x892967e69e5b24dc,
    0x605d4d9f715f8108,
    0x520d7623beac0843,
    0xbb795c5a51a8ad97,
    0xb43c04833832d080,
    0x5d482efad7367554,
    0xaab6b531eb062aae,
    0x43c29f4804028f7a,
    0x4c87c7916d98f26d,
    0xa5f3ede8829c57b9,
    0x2827b0e8f27fe0fb,
    0xc1539a911d7b452f,
    0xce16c24874e13838,
    0x2762e8319be59dec,
    0xd09c73faa7d5c216,
    0x39e8598348d167c2,
    0x36ad015a214b1ad5,
    0xdfd92b23ce4fbf01,
    0xed89109f01bc364a,
    0x04fd3ae6eeb8939e,
    0x0bb8623f8722ee89,
    0xe2cc484668264b5d,
    0x1532d38d541614a7,
    0xfc46f9f4bb12b173,
    0xf303a12dd288cc64,
    0x1a778b543d8c69b0,
    0xdc723d7e6bd8318b,
    0x3506170784dc945f,
    0x3a434fdeed46e948,
    0xd33765a702424c9c,
    0x24c9fe6c3e721366,
    0xcdbdd415d176b6b2,
    0xc2f88cccb8eccba5,
    0x2b8ca6b557e86e71,
    0x19dc9d09981be73a,
    0xf0a8b770771f42ee,
    0xffedefa91e853ff9,
    0x1699c5d0f1819a2d,
    0xe1675e1bcdb1c5d7,
    0x0813746222b56003,
    0x07562cbb4b2f1d14,
    0xee2206c2a42bb8c0,
    0x63f65bc2d4c80f82,
    0x8a8271bb3bccaa56,
    0x85c729625256d741,
    0x6cb3031bbd527295,
    0x9b4d98d081622d6f,
    0x7239b2a96e6688bb,
    0x7d7cea7007fcf5ac,
    0x9408c009e8f85078,
    0xa658fbb5270bd933,
    0x4f2cd1ccc80f7ce7,
    0x40698915a19501f0,
    0xa91da36c4e91a424,
    0x5ee338a772a1fbde,
    0xb79712de9da55e0a,
    0xb8d24a07f43f231d,
    0x51a6607e1b3b86c9,
    0x1b9e8afbc2482e8f,
    0xf2eaa0822d4c8b5b,
    0xfdaff85b44d6f64c,
    0x14dbd222abd25398,
    0xe32549e997e20c62,
    0x0a51639078e6a9b6,
    0x05143b49117cd4a1,
    0xec601130fe787175,
    0xde302a8c318bf83e,
    0x374400f5de8f5dea,
    0x3801582cb71520fd,
    0xd175725558118529,
    0x268be99e6421dad3,
    0xcfffc3e78b257f07,
    0xc0ba9b3ee2bf0210,
    0x29ceb1470dbba7c4,
    0xa41aec477d581086,
    0x4d6ec63e925cb552,
    0x422b9ee7fbc6c845,
    0xab5fb49e14c26d91,
    0x5ca12f5528f2326b,
    0xb5d5052cc7f697bf,
    0xba905df5ae6ceaa8,
    0x53e4778c41684f7c,
    0x61b44c308e9bc637,
    0x88c06649619f63e3,
    0x87853e9008051ef4,
    0x6ef114e9e701bb20,
    0x990f8f22db31e4da,
    0x707ba55b3435410e,
    0x7f3efd825daf3c19,
    0x964ad7fbb2ab99cd,
    0x504f61d1e4ffc1f6,
    0xb93b4ba80bfb6422,
    0xb67e137162611935,
    0x5f0a39088d65bce1,
    0xa8f4a2c3b155e31b,
    0x418088ba5e5146cf,
    0x4ec5d06337cb3bd8,
    0xa7b1fa1ad8cf9e0c,
    0x95e1c1a6173c1747,
    0x7c95ebdff838b293,
    0x73d0b30691a2cf84,
    0x9aa4997f7ea66a50,
    0x6d5a02b4429635aa,
    0x842e28cdad92907e,
    0x8b6b7014c408ed69,
    0x621f5a6d2b0c48bd,
    0xefcb076d5befffff,
    0x06bf2d14b4eb5a2b,
    0x09fa75cddd71273c,
    0xe08e5fb4327582e8,
    0x1770c47f0e45dd12,
    0xfe04ee06e14178c6,
    0xf141b6df88db05d1,
    0x18359ca667dfa005,
    0x2a65a71aa82c294e,
    0xc3118d6347288c9a,
    0xcc54d5ba2eb2f18d,
    0x2520ffc3c1b65459,
    0xd2de6408fd860ba3,
    0x3baa4e711282ae77,
    0x34ef16a87b18d360,
    0xdd9b3cd1941c76b4,
    0x8c3d5caf8f27f07d,
    0x654976d6602355a9,
    0x6a0c2e0f09b928be,
    0x83780476e6bd8d6a,
    0x74869fbdda8dd290,
    0x9df2b5c435897744,
    0x92b7ed1d5c130a53,
    0x7bc3c764b317af87,
    0x4993fcd87ce426cc,
    0xa0e7d6a193e08318,
    0xafa28e78fa7afe0f,
    0x46d6a401157e5bdb,
    0xb1283fca294e0421,
    0x585c15b3c64aa1f5,
    0x57194d6aafd0dce2,
    0xbe6d671340d47936,
    0x33b93a133037ce74,
    0xdacd106adf336ba0,
    0xd58848b3b6a916b7,
    0x3cfc62ca59adb363,
    0xcb02f901659dec99,
    0x2276d3788a99494d,
    0x2d338ba1e303345a,
    0xc447a1d80c07918e,
    0xf6179a64c3f418c5,
    0x1f63b01d2cf0bd11,
    0x1026e8c4456ac006,
    0xf952c2bdaa6e65d2,
    0x0eac5976965e3a28,
    0xe7d8730f795a9ffc,
    0xe89d2bd610c0e2eb,
    0x01e901afffc4473f,
    0xc7ecb785a9901f04,
    0x2e989dfc4694bad0,
    0x21ddc5252f0ec7c7,
    0xc8a9ef5cc00a6213,
    0x3f577497fc3a3de9,
    0xd6235eee133e983d,
    0xd96606377aa4e52a,
    0x30122c4e95a040fe,
    0x024217f25a53c9b5,
    0xeb363d8bb5576c61,
    0xe4736552dccd1176,
    0x0d074f2b33c9b4a2,
    0xfaf9d4e00ff9eb58,
    0x138dfe99e0fd4e8c,
    0x1cc8a6408967339b,
    0xf5bc8c396663964f,
    0x7868d1391680210d,
    0x911cfb40f98484d9,
    0x9e59a399901ef9ce,
    0x772d89e07f1a5c1a,
    0x80d3122b432a03e0,
    0x69a73852ac2ea634,
    0x66e2608bc5b4db23,
    0x8f964af22ab07ef7,
    0xbdc6714ee543f7bc,
    0x54b25b370a475268,
    0x5bf703ee63dd2f7f,
    0xb28329978cd98aab,
    0x457db25cb0e9d551,
    0xac0998255fed7085,
    0xa34cc0fc36770d92,
    0x4a38ea85d973a846,
];

// CRC table for the NVME polynomial in the range (`1 = x¹¹⁹, …, 128 = x¹¹²`).
// Generated by running `./build_table 6`.
static TABLE_6: [u64; 256] = [
    0x0000000000000000,
    0xfc5d27f6bf353971,
    0xcc6369be26fde189,
    0x303e4e4899c8d8f8,
    0xac1ff52f156c5079,
    0x5042d2d9aa596908,
    0x607c9c913391b1f0,
    0x9c21bb678ca48881,
    0x6ce6cc0d724f3399,
    0x90bbebfbcd7a0ae8,
    0xa085a5b354b2d210,
    0x5cd88245eb87eb61,
    0xc0f93922672363e0,
    0x3ca41ed4d8165a91,
    0x0c9a509c41de8269,
    0xf0c7776afeebbb18,
    0xd9cd981ae49e6732,
    0x2590bfec5bab5e43,
    0x15aef1a4c26386bb,
    0xe9f3d6527d56bfca,
    0x75d26d35f1f2374b,
    0x898f4ac34ec70e3a,
    0xb9b1048bd70fd6c2,
    0x45ec237d683aefb3,
    0xb52b541796d154ab,
    0x497673e129e46dda,
    0x79483da9b02cb522,
    0x85151a5f0f198c53,
    0x1934a13883bd04d2,
    0xe56986ce3c883da3,
    0xd557c886a540e55b,
    0x290aef701a75dc2a,
    0x8742166691ab5d0f,
    0x7b1f31902e9e647e,
    0x4b217fd8b756bc86,
    0xb77c582e086385f7,
    0x2b5de34984c70d76,
    0xd700c4bf3bf23407,
    0xe73e8af7a23aecff,
    0x1b63ad011d0fd58e,
    0xeba4da6be3e46e96,
    0x17f9fd9d5cd157e7,
    0x27c7b3d5c5198f1f,
    0xdb9a94237a2cb66e,
    0x47bb2f44f6883eef,
    0xbbe608b249bd079e,
    0x8bd846fad075df66,
    0x7785610c6f40e617,
    0x5e8f8e7c75353a3d,
    0xa2d2a98aca00034c,
    0x92ece7c253c8dbb4,
    0x6eb1c034ecfde2c5,
    0xf2907b5360596a44,
    0x0ecd5ca5df6c5335,
    0x3ef312ed46a48bcd,
    0xc2ae351bf991b2bc,
    0x32694271077a09a4,
    0xce346587b84f30d5,
    0xfe0a2bcf2187e82d,
    0x02570c399eb2d15c,
    0x9e76b75e121659dd,
    0x622b90a8ad2360ac,
    0x5215dee034ebb854,
    0xae48f9168bde8125,
    0x3a5d0a9e7bc12975,
    0xc6002d68c4f41004,
    0xf63e63205d3cc8fc,
    0x0a6344d6e209f18d,
    0x9642ffb16ead790c,
    0x6a1fd847d198407d,
    0x5a21960f48509885,
    0xa67cb1f9f765a1f4,
    0x56bbc693098e1aec,
    0xaae6e165b6bb239d,
    0x9ad8af2d2f73fb65,
    0x668588db9046c214,
    0xfaa433bc1ce24a95,
    0x06f9144aa3d773e4,
    0x36c75a023a1fab1c,
    0xca9a7df4852a926d,
    0xe39092849f5f4e47,
    0x1fcdb572206a7736,
    0x2ff3fb3ab9a2afce,
    0xd3aedccc069796bf,
    0x4f8f67ab8a331e3e,
    0xb3d2405d3506274f,
    0x83ec0e15acceffb7,
    0x7fb129e313fbc6c6,
    0x8f765e89ed107dde,
    0x732b797f522544af,
    0x43153737cbed9c57,
    0xbf4810c174d8a526,
    0x2369aba6f87c2da7,
    0xdf348c50474914d6,
    0xef0ac218de81cc2e,
    0x1357e5ee61b4f55f,
    0xbd1f1cf8ea6a747a,
    0x41423b0e555f4d0b,
    0x717c7546cc9795f3,
    0x8d2152b073a2ac82,
    0x1100e9d7ff062403,
    0xed5dce2140331d72,
    0xdd638069d9fbc58a,
    0x213ea79f66cefcfb,
    0xd1f9d0f5982547e3,
    0x2da4f70327107e92,
    0x1d9ab94bbed8a66a,
    0xe1c79ebd01ed9f1b,
    0x7de625da8d49179a,
    0x81bb022c327c2eeb,
    0xb1854c64abb4f613,
    0x4dd86b921481cf62,
    0x64d284e20ef41348,
    0x988fa314b1c12a39,
    0xa8b1ed5c2809f2c1,
    0x54eccaaa973ccbb0,
    0xc8cd71cd1b984331,
    0x3490563ba4ad7a40,
    0x04ae18733d65a2b8,
    0xf8f33f8582509bc9,
    0x083448ef7cbb20d1,
    0xf4696f19c38e19a0,
    0xc45721515a46c158,
    0x380a06a7e573f829,
    0xa42bbdc069d770a8,
    0x58769a36d6e249d9,
    0x6848d47e4f2a9121,
    0x9415f388f01fa850,
    0x74ba153cf78252ea,
    0x88e732ca48b76b9b,
    0xb8d97c82d17fb363,
    0x44845b746e4a8a12,
    0xd8a5e013e2ee0293,
    0x24f8c7e55ddb3be2,
    0x14c689adc413e31a,
    0xe89bae5b7b26da6b,
    0x185cd93185cd6173,
    0xe401fec73af85802,
    0xd43fb08fa33080fa,
    0x286297791c05b98b,
    0xb4432c1e90a1310a,
    0x481e0be82f94087b,
    0x782045a0b65cd083,
    0x847d62560969e9f2,
    0xad778d26131c35d8,
    0x512aaad0ac290ca9,
    0x6114e49835e1d451,
    0x9d49c36e8ad4ed20,
    0x01687809067065a1,
    0xfd355fffb9455cd0,
    0xcd0b11b7208d8428,
    0x315636419fb8bd59,
    0xc191412b61530641,
    0x3dcc66ddde663f30,
    0x0df2289547aee7c8,
    0xf1af0f63f89bdeb9,
    0x6d8eb404743f5638,
    0x91d393f2cb0a6f49,
    0xa1edddba52c2b7b1,
    0x5db0fa4cedf78ec0,
    0xf3f8035a66290fe5,
    0x0fa524acd91c3694,
    0x3f9b6ae440d4ee6c,
    0xc3c64d12ffe1d71d,
    0x5fe7f67573455f9c,
    0xa3bad183cc7066ed,
    0x93849fcb55b8be15,
    0x6fd9b83dea8d8764,
    0x9f1ecf5714663c7c,
    0x6343e8a1ab53050d,
    0x537da6e9329bddf5,
    0xaf20811f8daee484,
    0x33013a78010a6c05,
    0xcf5c1d8ebe3f5574,
    0xff6253c627f78d8c,
    0x033f743098c2b4fd,
    0x2a359b4082b768d7,
    0xd668bcb63d8251a6,
    0xe656f2fea44a895e,
    0x1a0bd5081b7fb02f,
    0x862a6e6f97db38ae,
    0x7a77499928ee01df,
    0x4a4907d1b126d927,
    0xb61420270e13e056,
    0x46d3574df0f85b4e,
    0xba8e70bb4fcd623f,
    0x8ab03ef3d605bac7,
    0x76ed1905693083b6,
    0xeacca262e5940b37,
    0x169185945aa13246,
    0x26afcbdcc369eabe,
    0xdaf2ec2a7c5cd3cf,
    0x4ee71fa28c437b9f,
    0xb2ba3854337642ee,
    0x8284761caabe9a16,
    0x7ed951ea158ba367,
    0xe2f8ea8d992f2be6,
    0x1ea5cd7b261a1297,
    0x2e9b8333bfd2ca6f,
    0xd2c6a4c500e7f31e,
    0x2201d3affe0c4806,
    0xde5cf45941397177,
    0xee62ba11d8f1a98f,
    0x123f9de767c490fe,
    0x8e1e2680eb60187f,
    0x724301765455210e,
    0x427d4f3ecd9df9f6,
    0xbe2068c872a8c087,
    0x972a87b868dd1cad,
    0x6b77a04ed7e825dc,
    0x5b49ee064e20fd24,
    0xa714c9f0f115c455,
    0x3b3572977db14cd4,
    0xc7685561c28475a5,
    0xf7561b295b4cad5d,
    0x0b0b3cdfe479942c,
    0xfbcc4bb51a922f34,
    0x07916c43a5a71645,
    0x37af220b3c6fcebd,
    0xcbf205fd835af7cc,
    0x57d3be9a0ffe7f4d,
    0xab8e996cb0cb463c,
    0x9bb0d72429039ec4,
    0x67edf0d29636a7b5,
    0xc9a509c41de82690,
    0x35f82e32a2dd1fe1,
    0x05c6607a3b15c719,
    0xf99b478c8420fe68,
    0x65bafceb088476e9,
    0x99e7db1db7b14f98,
    0xa9d995552e799760,
    0x5584b2a3914cae11,
    0xa543c5c96fa71509,
    0x591ee23fd0922c78,
    0x6920ac77495af480,
    0x957d8b81f66fcdf1,
    0x095c30e67acb4570,
    0xf5011710c5fe7c01,
    0xc53f59585c36a4f9,
    0x39627eaee3039d88,
    0x106891def97641a2,
    0xec35b628464378d3,
    0xdc0bf860df8ba02b,
    0x2056df9660be995a,
    0xbc7764f1ec1a11db,
    0x402a4307532f28aa,
    0x70140d4fcae7f052,
    0x8c492ab975d2c923,
    0x7c8e5dd38b39723b,
    0x80d37a25340c4b4a,
    0xb0ed346dadc493b2,
    0x4cb0139b12f1aac3,
    0xd091a8fc9e552242,
    0x2ccc8f0a21601b33,
    0x1cf2c142b8a8c3cb,
    0xe0afe6b4079dfaba,
];

// CRC table for the NVME polynomial in the range (`1 = x¹²⁷, …, 128 = x¹²⁰`).
// Generated by running `./build_table 7`.
static TABLE_7: [u64; 256] = [
    0x0000000000000000,
    0x21e9761e252621ac,
    0x43d2ec3c4a4c4358,
    0x623b9a226f6a62f4,
    0x87a5d878949886b0,
    0xa64cae66b1bea71c,
    0xc4773444ded4c5e8,
    0xe59e425afbf2e444,
    0x3b9296a271a69e0b,
    0x1a7be0bc5480bfa7,
    0x78407a9e3beadd53,
    0x59a90c801eccfcff,
    0xbc374edae53e18bb,
    0x9dde38c4c0183917,
    0xffe5a2e6af725be3,
    0xde0cd4f88a547a4f,
    0x77252d44e34d3c16,
    0x56cc5b5ac66b1dba,
    0x34f7c178a9017f4e,
    0x151eb7668c275ee2,
    0xf080f53c77d5baa6,
    0xd169832252f39b0a,
    0xb35219003d99f9fe,
    0x92bb6f1e18bfd852,
    0x4cb7bbe692eba21d,
    0x6d5ecdf8b7cd83b1,
    0x0f6557dad8a7e145,
    0x2e8c21c4fd81c0e9,
    0xcb12639e067324ad,
    0xeafb158023550501,
    0x88c08fa24c3f67f5,
    0xa929f9bc69194659,
    0xee4a5a89c69a782c,
    0xcfa32c97e3bc5980,
    0xad98b6b58cd63b74,
    0x8c71c0aba9f01ad8,
    0x69ef82f15202fe9c,
    0x4806f4ef7724df30,
    0x2a3d6ecd184ebdc4,
    0x0bd418d33d689c68,
    0xd5d8cc2bb73ce627,
    0xf431ba35921ac78b,
    0x960a2017fd70a57f,
    0xb7e35609d85684d3,
    0x527d145323a46097,
    0x7394624d0682413b,
    0x11aff86f69e823cf,
    0x30468e714cce0263,
    0x996f77cd25d7443a,
    0xb88601d300f16596,
    0xdabd9bf16f9b0762,
    0xfb54edef4abd26ce,
    0x1ecaafb5b14fc28a,
    0x3f23d9ab9469e326,
    0x5d184389fb0381d2,
    0x7cf13597de25a07e,
    0xa2fde16f5471da31,
    0x831497717157fb9d,
    0xe12f0d531e3d9969,
    0xc0c67b4d3b1bb8c5,
    0x25583917c0e95c81,
    0x04b14f09e5cf7d2d,
    0x668ad52b8aa51fd9,
    0x4763a335af833e75,
    0xe84d9340d5a36333,
    0xc9a4e55ef085429f,
    0xab9f7f7c9fef206b,
    0x8a760962bac901c7,
    0x6fe84b38413be583,
    0x4e013d26641dc42f,
    0x2c3aa7040b77a6db,
    0x0dd3d11a2e518777,
    0xd3df05e2a405fd38,
    0xf23673fc8123dc94,
    0x900de9deee49be60,
    0xb1e49fc0cb6f9fcc,
    0x547add9a309d7b88,
    0x7593ab8415bb5a24,
    0x17a831a67ad138d0,
    0x364147b85ff7197c,
    0x9f68be0436ee5f25,
    0xbe81c81a13c87e89,
    0xdcba52387ca21c7d,
    0xfd53242659843dd1,
    0x18cd667ca276d995,
    0x392410628750f839,
    0x5b1f8a40e83a9acd,
    0x7af6fc5ecd1cbb61,
    0xa4fa28a64748c12e,
    0x85135eb8626ee082,
    0xe728c49a0d048276,
    0xc6c1b2842822a3da,
    0x235ff0ded3d0479e,
    0x02b686c0f6f66632,
    0x608d1ce2999c04c6,
    0x41646afcbcba256a,
    0x0607c9c913391b1f,
    0x27eebfd7361f3ab3,
    0x45d525f559755847,
    0x643c53eb7c5379eb,
    0x81a211b187a19daf,
    0xa04b67afa287bc03,
    0xc270fd8dcdeddef7,
    0xe3998b93e8cbff5b,
    0x3d955f6b629f8514,
    0x1c7c297547b9a4b8,
    0x7e47b35728d3c64c,
    0x5faec5490df5e7e0,
    0xba308713f60703a4,
    0x9bd9f10dd3212208,
    0xf9e26b2fbc4b40fc,
    0xd80b1d31996d6150,
    0x7122e48df0742709,
    0x50cb9293d55206a5,
    0x32f008b1ba386451,
    0x13197eaf9f1e45fd,
    0xf6873cf564eca1b9,
    0xd76e4aeb41ca8015,
    0xb555d0c92ea0e2e1,
    0x94bca6d70b86c34d,
    0x4ab0722f81d2b902,
    0x6b590431a4f498ae,
    0x09629e13cb9efa5a,
    0x288be80deeb8dbf6,
    0xcd15aa57154a3fb2,
    0xecfcdc49306c1e1e,
    0x8ec7466b5f067cea,
    0xaf2e30757a205d46,
    0xe44200d2f3d1550d,
    0xc5ab76ccd6f774a1,
    0xa790eceeb99d1655,
    0x86799af09cbb37f9,
    0x63e7d8aa6749d3bd,
    0x420eaeb4426ff211,
    0x203534962d0590e5,
    0x01dc42880823b149,
    0xdfd096708277cb06,
    0xfe39e06ea751eaaa,
    0x9c027a4cc83b885e,
    0xbdeb0c52ed1da9f2,
    0x58754e0816ef4db6,
    0x799c381633c96c1a,
    0x1ba7a2345ca30eee,
    0x3a4ed42a79852f42,
    0x93672d96109c691b,
    0xb28e5b8835ba48b7,
    0xd0b5c1aa5ad02a43,
    0xf15cb7b47ff60bef,
    0x14c2f5ee8404efab,
    0x352b83f0a122ce07,
    0x571019d2ce48acf3,
    0x76f96fcceb6e8d5f,
    0xa8f5bb34613af710,
    0x891ccd2a441cd6bc,
    0xeb2757082b76b448,
    0xcace21160e5095e4,
    0x2f50634cf5a271a0,
    0x0eb91552d084500c,
    0x6c828f70bfee32f8,
    0x4d6bf96e9ac81354,
    0x0a085a5b354b2d21,
    0x2be12c45106d0c8d,
    0x49dab6677f076e79,
    0x6833c0795a214fd5,
    0x8dad8223a1d3ab91,
    0xac44f43d84f58a3d,
    0xce7f6e1feb9fe8c9,
    0xef961801ceb9c965,
    0x319accf944edb32a,
    0x1073bae761cb9286,
    0x724820c50ea1f072,
    0x53a156db2b87d1de,
    0xb63f1481d075359a,
    0x97d6629ff5531436,
    0xf5edf8bd9a3976c2,
    0xd4048ea3bf1f576e,
    0x7d2d771fd6061137,
    0x5cc40101f320309b,
    0x3eff9b239c4a526f,
    0x1f16ed3db96c73c3,
    0xfa88af67429e9787,
    0xdb61d97967b8b62b,
    0xb95a435b08d2d4df,
    0x98b335452df4f573,
    0x46bfe1bda7a08f3c,
    0x675697a38286ae90,
    0x056d0d81edeccc64,
    0x24847b9fc8caedc8,
    0xc11a39c53338098c,
    0xe0f34fdb161e2820,
    0x82c8d5f979744ad4,
    0xa321a3e75c526b78,
    0x0c0f93922672363e,
    0x2de6e58c03541792,
    0x4fdd7fae6c3e7566,
    0x6e3409b0491854ca,
    0x8baa4beab2eab08e,
    0xaa433df497cc9122,
    0xc878a7d6f8a6f3d6,
    0xe991d1c8dd80d27a,
    0x379d053057d4a835,
    0x1674732e72f28999,
    0x744fe90c1d98eb6d,
    0x55a69f1238becac1,
    0xb038dd48c34c2e85,
    0x91d1ab56e66a0f29,
    0xf3ea317489006ddd,
    0xd203476aac264c71,
    0x7b2abed6c53f0a28,
    0x5ac3c8c8e0192b84,
    0x38f852ea8f734970,
    0x191124f4aa5568dc,
    0xfc8f66ae51a78c98,
    0xdd6610b07481ad34,
    0xbf5d8a921bebcfc0,
    0x9eb4fc8c3ecdee6c,
    0x40b82874b4999423,
    0x61515e6a91bfb58f,
    0x036ac448fed5d77b,
    0x2283b256dbf3f6d7,
    0xc71df00c20011293,
    0xe6f486120527333f,
    0x84cf1c306a4d51cb,
    0xa5266a2e4f6b7067,
    0xe245c91be0e84e12,
    0xc3acbf05c5ce6fbe,
    0xa1972527aaa40d4a,
    0x807e53398f822ce6,
    0x65e011637470c8a2,
    0x4409677d5156e90e,
    0x2632fd5f3e3c8bfa,
    0x07db8b411b1aaa56,
    0xd9d75fb9914ed019,
    0xf83e29a7b468f1b5,
    0x9a05b385db029341,
    0xbbecc59bfe24b2ed,
    0x5e7287c105d656a9,
    0x7f9bf1df20f07705,
    0x1da06bfd4f9a15f1,
    0x3c491de36abc345d,
    0x9560e45f03a57204,
    0xb4899241268353a8,
    0xd6b2086349e9315c,
    0xf75b7e7d6ccf10f0,
    0x12c53c27973df4b4,
    0x332c4a39b21bd518,
    0x5117d01bdd71b7ec,
    0x70fea605f8579640,
    0xaef272fd7203ec0f,
    0x8f1b04e35725cda3,
    0xed209ec1384faf57,
    0xccc9e8df1d698efb,
    0x2957aa85e69b6abf,
    0x08bedc9bc3bd4b13,
    0x6a8546b9acd729e7,
    0x4b6c30a789f1084b,
];

// CRC table for the NVME polynomial in the range (`1 = x¹³⁵, …, 128 = x¹²⁸`).
// Generated by running `./build_table 8`.
static TABLE_8: [u64; 256] = [
    0x0000000000000000,
    0x04f28def5347786c,
    0x09e51bdea68ef0d8,
    0x0d179631f5c988b4,
    0x13ca37bd4d1de1b0,
    0x1738ba521e5a99dc,
    0x1a2f2c63eb931168,
    0x1edda18cb8d46904,
    0x27946f7a9a3bc360,
    0x2366e295c97cbb0c,
    0x2e7174a43cb533b8,
    0x2a83f94b6ff24bd4,
    0x345e58c7d72622d0,
    0x30acd52884615abc,
    0x3dbb431971a8d208,
    0x3949cef622efaa64,
    0x4f28def5347786c0,
    0x4bda531a6730feac,
    0x46cdc52b92f97618,
    0x423f48c4c1be0e74,
    0x5ce2e948796a6770,
    0x581064a72a2d1f1c,
    0x5507f296dfe497a8,
    0x51f57f798ca3efc4,
    0x68bcb18fae4c45a0,
    0x6c4e3c60fd0b3dcc,
    0x6159aa5108c2b578,
    0x65ab27be5b85cd14,
    0x7b768632e351a410,
    0x7f840bddb016dc7c,
    0x72939dec45df54c8,
    0x7661100316982ca4,
    0x9e51bdea68ef0d80,
    0x9aa330053ba875ec,
    0x97b4a634ce61fd58,
    0x93462bdb9d268534,
    0x8d9b8a5725f2ec30,
    0x896907b876b5945c,
    0x847e9189837c1ce8,
    0x808c1c66d03b6484,
    0xb9c5d290f2d4cee0,
    0xbd375f7fa193b68c,
    0xb020c94e545a3e38,
    0xb4d244a1071d4654,
    0xaa0fe52dbfc92f50,
    0xaefd68c2ec8e573c,
    0xa3eafef31947df88,
    0xa718731c4a00a7e4,
    0xd179631f5c988b40,
    0xd58beef00fdff32c,
    0xd89c78c1fa167b98,
    0xdc6ef52ea95103f4,
    0xc2b354a211856af0,
    0xc641d94d42c2129c,
    0xcb564f7cb70b9a28,
    0xcfa4c293e44ce244,
    0xf6ed0c65c6a34820,
    0xf21f818a95e4304c,
    0xff0817bb602db8f8,
    0xfbfa9a54336ac094,
    0xe5273bd88bbea990,
    0xe1d5b637d8f9d1fc,
    0xecc220062d305948,
    0xe830ade97e772124,
    0x087a5d878949886b,
    0x0c88d068da0ef007,
    0x019f46592fc778b3,
    0x056dcbb67c8000df,
    0x1bb06a3ac45469db,
    0x1f42e7d5971311b7,
    0x125571e462da9903,
    0x16a7fc0b319de16f,
    0x2fee32fd13724b0b,
    0x2b1cbf1240353367,
    0x260b2923b5fcbbd3,
    0x22f9a4cce6bbc3bf,
    0x3c2405405e6faabb,
    0x38d688af0d28d2d7,
    0x35c11e9ef8e15a63,
    0x31339371aba6220f,
    0x47528372bd3e0eab,
    0x43a00e9dee7976c7,
    0x4eb798ac1bb0fe73,
    0x4a45154348f7861f,
    0x5498b4cff023ef1b,
    0x506a3920a3649777,
    0x5d7daf1156ad1fc3,
    0x598f22fe05ea67af,
    0x60c6ec082705cdcb,
    0x643461e77442b5a7,
    0x6923f7d6818b3d13,
    0x6dd17a39d2cc457f,
    0x730cdbb56a182c7b,
    0x77fe565a395f5417,
    0x7ae9c06bcc96dca3,
    0x7e1b4d849fd1a4cf,
    0x962be06de1a685eb,
    0x92d96d82b2e1fd87,
    0x9fcefbb347287533,
    0x9b3c765c146f0d5f,
    0x85e1d7d0acbb645b,
    0x81135a3ffffc1c37,
    0x8c04cc0e0a359483,
    0x88f641e15972ecef,
    0xb1bf8f177b9d468b,
    0xb54d02f828da3ee7,
    0xb85a94c9dd13b653,
    0xbca819268e54ce3f,
    0xa275b8aa3680a73b,
    0xa687354565c7df57,
    0xab90a374900e57e3,
    0xaf622e9bc3492f8f,
    0xd9033e98d5d1032b,
    0xddf1b37786967b47,
    0xd0e62546735ff3f3,
    0xd414a8a920188b9f,
    0xcac9092598cce29b,
    0xce3b84cacb8b9af7,
    0xc32c12fb3e421243,
    0xc7de9f146d056a2f,
    0xfe9751e24feac04b,
    0xfa65dc0d1cadb827,
    0xf7724a3ce9643093,
    0xf380c7d3ba2348ff,
    0xed5d665f02f721fb,
    0xe9afebb051b05997,
    0xe4b87d81a479d123,
    0xe04af06ef73ea94f,
    0x10f4bb0f129310d6,
    0x140636e041d468ba,
    0x1911a0d1b41de00e,
    0x1de32d3ee75a9862,
    0x033e8cb25f8ef166,
    0x07cc015d0cc9890a,
    0x0adb976cf90001be,
    0x0e291a83aa4779d2,
    0x3760d47588a8d3b6,
    0x3392599adbefabda,
    0x3e85cfab2e26236e,
    0x3a7742447d615b02,
    0x24aae3c8c5b53206,
    0x20586e2796f24a6a,
    0x2d4ff816633bc2de,
    0x29bd75f9307cbab2,
    0x5fdc65fa26e49616,
    0x5b2ee81575a3ee7a,
    0x56397e24806a66ce,
    0x52cbf3cbd32d1ea2,
    0x4c1652476bf977a6,
    0x48e4dfa838be0fca,
    0x45f34999cd77877e,
    0x4101c4769e30ff12,
    0x78480a80bcdf5576,
    0x7cba876fef982d1a,
    0x71ad115e1a51a5ae,
    0x755f9cb14916ddc2,
    0x6b823d3df1c2b4c6,
    0x6f70b0d2a285ccaa,
    0x626726e3574c441e,
    0x6695ab0c040b3c72,
    0x8ea506e57a7c1d56,
    0x8a578b0a293b653a,
    0x87401d3bdcf2ed8e,
    0x83b290d48fb595e2,
    0x9d6f31583761fce6,
    0x999dbcb76426848a,
    0x948a2a8691ef0c3e,
    0x9078a769c2a87452,
    0xa931699fe047de36,
    0xadc3e470b300a65a,
    0xa0d4724146c92eee,
    0xa426ffae158e5682,
    0xbafb5e22ad5a3f86,
    0xbe09d3cdfe1d47ea,
    0xb31e45fc0bd4cf5e,
    0xb7ecc8135893b732,
    0xc18dd8104e0b9b96,
    0xc57f55ff1d4ce3fa,
    0xc868c3cee8856b4e,
    0xcc9a4e21bbc21322,
    0xd247efad03167a26,
    0xd6b562425051024a,
    0xdba2f473a5988afe,
    0xdf50799cf6dff292,
    0xe619b76ad43058f6,
    0xe2eb3a858777209a,
    0xeffcacb472bea82e,
    0xeb0e215b21f9d042,
    0xf5d380d7992db946,
    0xf1210d38ca6ac12a,
    0xfc369b093fa3499e,
    0xf8c416e66ce431f2,
    0x188ee6889bda98bd,
    0x1c7c6b67c89de0d1,
    0x116bfd563d546865,
    0x159970b96e131009,
    0x0b44d135d6c7790d,
    0x0fb65cda85800161,
    0x02a1caeb704989d5,
    0x06534704230ef1b9,
    0x3f1a89f201e15bdd,
    0x3be8041d52a623b1,
    0x36ff922ca76fab05,
    0x320d1fc3f428d369,
    0x2cd0be4f4cfcba6d,
    0x282233a01fbbc201,
    0x2535a591ea724ab5,
    0x21c7287eb93532d9,
    0x57a6387dafad1e7d,
    0x5354b592fcea6611,
    0x5e4323a30923eea5,
    0x5ab1ae4c5a6496c9,
    0x446c0fc0e2b0ffcd,
    0x409e822fb1f787a1,
    0x4d89141e443e0f15,
    0x497b99f117797779,
    0x703257073596dd1d,
    0x74c0dae866d1a571,
    0x79d74cd993182dc5,
    0x7d25c136c05f55a9,
    0x63f860ba788b3cad,
    0x670aed552bcc44c1,
    0x6a1d7b64de05cc75,
    0x6eeff68b8d42b419,
    0x86df5b62f335953d,
    0x822dd68da072ed51,
    0x8f3a40bc55bb65e5,
    0x8bc8cd5306fc1d89,
    0x95156cdfbe28748d,
    0x91e7e130ed6f0ce1,
    0x9cf0770118a68455,
    0x9802faee4be1fc39,
    0xa14b3418690e565d,
    0xa5b9b9f73a492e31,
    0xa8ae2fc6cf80a685,
    0xac5ca2299cc7dee9,
    0xb28103a52413b7ed,
    0xb6738e4a7754cf81,
    0xbb64187b829d4735,
    0xbf969594d1da3f59,
    0xc9f78597c74213fd,
    0xcd05087894056b91,
    0xc0129e4961cce325,
    0xc4e013a6328b9b49,
    0xda3db22a8a5ff24d,
    0xdecf3fc5d9188a21,
    0xd3d8a9f42cd10295,
    0xd72a241b7f967af9,
    0xee63eaed5d79d09d,
    0xea9167020e3ea8f1,
    0xe786f133fbf72045,
    0xe3747cdca8b05829,
    0xfda9dd501064312d,
    0xf95b50bf43234941,
    0xf44cc68eb6eac1f5,
    0xf0be4b61e5adb999,
];

// CRC table for the NVME polynomial in the range (`1 = x¹⁴³, …, 128 = x¹³⁶`).
// Generated by running `./build_table 9`.
static TABLE_9: [u64; 256] = [
    0x0000000000000000,
    0x49e1df807414fdef,
    0x93c3bf00e829fbde,
    0xda2260809c3d0631,
    0x135e585288c464d7,
    0x5abf87d2fcd09938,
    0x809de75260ed9f09,
    0xc97c38d214f962e6,
    0x26bcb0a51188c9ae,
    0x6f5d6f25659c3441,
    0xb57f0fa5f9a13270,
    0xfc9ed0258db5cf9f,
    0x35e2e8f7994cad79,
    0x7c033777ed585096,
    0xa62157f7716556a7,
    0xefc088770571ab48,
    0x4d79614a2311935c,
    0x0498beca57056eb3,
    0xdebade4acb386882,
    0x975b01cabf2c956d,
    0x5e273918abd5f78b,
    0x17c6e698dfc10a64,
    0xcde4861843fc0c55,
    0x8405599837e8f1ba,
    0x6bc5d1ef32995af2,
    0x22240e6f468da71d,
    0xf8066eefdab0a12c,
    0xb1e7b16faea45cc3,
    0x789b89bdba5d3e25,
    0x317a563dce49c3ca,
    0xeb5836bd5274c5fb,
    0xa2b9e93d26603814,
    0x9af2c294462326b8,
    0xd3131d143237db57,
    0x09317d94ae0add66,
    0x40d0a214da1e2089,
    0x89ac9ac6cee7426f,
    0xc04d4546baf3bf80,
    0x1a6f25c626ceb9b1,
    0x538efa4652da445e,
    0xbc4e723157abef16,
    0xf5afadb123bf12f9,
    0x2f8dcd31bf8214c8,
    0x666c12b1cb96e927,
    0xaf102a63df6f8bc1,
    0xe6f1f5e3ab7b762e,
    0x3cd395633746701f,
    0x75324ae343528df0,
    0xd78ba3de6532b5e4,
    0x9e6a7c5e1126480b,
    0x44481cde8d1b4e3a,
    0x0da9c35ef90fb3d5,
    0xc4d5fb8cedf6d133,
    0x8d34240c99e22cdc,
    0x5716448c05df2aed,
    0x1ef79b0c71cbd702,
    0xf137137b74ba7c4a,
    0xb8d6ccfb00ae81a5,
    0x62f4ac7b9c938794,
    0x2b1573fbe8877a7b,
    0xe2694b29fc7e189d,
    0xab8894a9886ae572,
    0x71aaf4291457e343,
    0x384b2ba960431eac,
    0x013ca37bd4d1de1b,
    0x48dd7cfba0c523f4,
    0x92ff1c7b3cf825c5,
    0xdb1ec3fb48ecd82a,
    0x1262fb295c15bacc,
    0x5b8324a928014723,
    0x81a14429b43c4112,
    0xc8409ba9c028bcfd,
    0x278013dec55917b5,
    0x6e61cc5eb14dea5a,
    0xb443acde2d70ec6b,
    0xfda2735e59641184,
    0x34de4b8c4d9d7362,
    0x7d3f940c39898e8d,
    0xa71df48ca5b488bc,
    0xeefc2b0cd1a07553,
    0x4c45c231f7c04d47,
    0x05a41db183d4b0a8,
    0xdf867d311fe9b699,
    0x9667a2b16bfd4b76,
    0x5f1b9a637f042990,
    0x16fa45e30b10d47f,
    0xccd82563972dd24e,
    0x8539fae3e3392fa1,
    0x6af97294e64884e9,
    0x2318ad14925c7906,
    0xf93acd940e617f37,
    0xb0db12147a7582d8,
    0x79a72ac66e8ce03e,
    0x3046f5461a981dd1,
    0xea6495c686a51be0,
    0xa3854a46f2b1e60f,
    0x9bce61ef92f2f8a3,
    0xd22fbe6fe6e6054c,
    0x080ddeef7adb037d,
    0x41ec016f0ecffe92,
    0x889039bd1a369c74,
    0xc171e63d6e22619b,
    0x1b5386bdf21f67aa,
    0x52b2593d860b9a45,
    0xbd72d14a837a310d,
    0xf4930ecaf76ecce2,
    0x2eb16e4a6b53cad3,
    0x6750b1ca1f47373c,
    0xae2c89180bbe55da,
    0xe7cd56987faaa835,
    0x3def3618e397ae04,
    0x740ee998978353eb,
    0xd6b700a5b1e36bff,
    0x9f56df25c5f79610,
    0x4574bfa559ca9021,
    0x0c9560252dde6dce,
    0xc5e958f739270f28,
    0x8c0887774d33f2c7,
    0x562ae7f7d10ef4f6,
    0x1fcb3877a51a0919,
    0xf00bb000a06ba251,
    0xb9ea6f80d47f5fbe,
    0x63c80f004842598f,
    0x2a29d0803c56a460,
    0xe355e85228afc686,
    0xaab437d25cbb3b69,
    0x70965752c0863d58,
    0x397788d2b492c0b7,
    0x027946f7a9a3bc36,
    0x4b989977ddb741d9,
    0x91baf9f7418a47e8,
    0xd85b2677359eba07,
    0x11271ea52167d8e1,
    0x58c6c1255573250e,
    0x82e4a1a5c94e233f,
    0xcb057e25bd5aded0,
    0x24c5f652b82b7598,
    0x6d2429d2cc3f8877,
    0xb706495250028e46,
    0xfee796d2241673a9,
    0x379bae0030ef114f,
    0x7e7a718044fbeca0,
    0xa4581100d8c6ea91,
    0xedb9ce80acd2177e,
    0x4f0027bd8ab22f6a,
    0x06e1f83dfea6d285,
    0xdcc398bd629bd4b4,
    0x9522473d168f295b,
    0x5c5e7fef02764bbd,
    0x15bfa06f7662b652,
    0xcf9dc0efea5fb063,
    0x867c1f6f9e4b4d8c,
    0x69bc97189b3ae6c4,
    0x205d4898ef2e1b2b,
    0xfa7f281873131d1a,
    0xb39ef7980707e0f5,
    0x7ae2cf4a13fe8213,
    0x330310ca67ea7ffc,
    0xe921704afbd779cd,
    0xa0c0afca8fc38422,
    0x988b8463ef809a8e,
    0xd16a5be39b946761,
    0x0b483b6307a96150,
    0x42a9e4e373bd9cbf,
    0x8bd5dc316744fe59,
    0xc23403b1135003b6,
    0x181663318f6d0587,
    0x51f7bcb1fb79f868,
    0xbe3734c6fe085320,
    0xf7d6eb468a1caecf,
    0x2df48bc61621a8fe,
    0x6415544662355511,
    0xad696c9476cc37f7,
    0xe488b31402d8ca18,
    0x3eaad3949ee5cc29,
    0x774b0c14eaf131c6,
    0xd5f2e529cc9109d2,
    0x9c133aa9b885f43d,
    0x46315a2924b8f20c,
    0x0fd085a950ac0fe3,
    0xc6acbd7b44556d05,
    0x8f4d62fb304190ea,
    0x556f027bac7c96db,
    0x1c8eddfbd8686b34,
    0xf34e558cdd19c07c,
    0xbaaf8a0ca90d3d93,
    0x608dea8c35303ba2,
    0x296c350c4124c64d,
    0xe0100dde55dda4ab,
    0xa9f1d25e21c95944,
    0x73d3b2debdf45f75,
    0x3a326d5ec9e0a29a,
    0x0345e58c7d72622d,
    0x4aa43a0c09669fc2,
    0x90865a8c955b99f3,
    0xd967850ce14f641c,
    0x101bbddef5b606fa,
    0x59fa625e81a2fb15,
    0x83d802de1d9ffd24,
    0xca39dd5e698b00cb,
    0x25f955296cfaab83,
    0x6c188aa918ee566c,
    0xb63aea2984d3505d,
    0xffdb35a9f0c7adb2,
    0x36a70d7be43ecf54,
    0x7f46d2fb902a32bb,
    0xa564b27b0c17348a,
    0xec856dfb7803c965,
    0x4e3c84c65e63f171,
    0x07dd5b462a770c9e,
    0xddff3bc6b64a0aaf,
    0x941ee446c25ef740,
    0x5d62dc94d6a795a6,
    0x14830314a2b36849,
    0xcea163943e8e6e78,
    0x8740bc144a9a9397,
    0x688034634feb38df,
    0x2161ebe33bffc530,
    0xfb438b63a7c2c301,
    0xb2a254e3d3d63eee,
    0x7bde6c31c72f5c08,
    0x323fb3b1b33ba1e7,
    0xe81dd3312f06a7d6,
    0xa1fc0cb15b125a39,
    0x99b727183b514495,
    0xd056f8984f45b97a,
    0x0a749818d378bf4b,
    0x43954798a76c42a4,
    0x8ae97f4ab3952042,
    0xc308a0cac781ddad,
    0x192ac04a5bbcdb9c,
    0x50cb1fca2fa82673,
    0xbf0b97bd2ad98d3b,
    0xf6ea483d5ecd70d4,
    0x2cc828bdc2f076e5,
    0x6529f73db6e48b0a,
    0xac55cfefa21de9ec,
    0xe5b4106fd6091403,
    0x3f9670ef4a341232,
    0x7677af6f3e20efdd,
    0xd4ce46521840d7c9,
    0x9d2f99d26c542a26,
    0x470df952f0692c17,
    0x0eec26d2847dd1f8,
    0xc7901e009084b31e,
    0x8e71c180e4904ef1,
    0x5453a10078ad48c0,
    0x1db27e800cb9b52f,
    0xf272f6f709c81e67,
    0xbb9329777ddce388,
    0x61b149f7e1e1e5b9,
    0x2850967795f51856,
    0xe12caea5810c7ab0,
    0xa8cd7125f518875f,
    0x72ef11a56925816e,
    0x3b0ece251d317c81,
];

// CRC table for the NVME polynomial in the range (`1 = x¹⁵¹, …, 128 = x¹⁴⁴`).
// Generated by running `./build_table 10`.
static TABLE_10: [u64; 256] = [
    0x0000000000000000,
    0x52734ea3e726fc54,
    0xa4e69d47ce4df8a8,
    0xf695d3e4296b04fc,
    0x7d141cdcc40c623b,
    0x2f67527f232a9e6f,
    0xd9f2819b0a419a93,
    0x8b81cf38ed6766c7,
    0xfa2839b98818c476,
    0xa85b771a6f3e3822,
    0x5ecea4fe46553cde,
    0x0cbdea5da173c08a,
    0x873c25654c14a64d,
    0xd54f6bc6ab325a19,
    0x23dab82282595ee5,
    0x71a9f681657fa2b1,
    0xc089552048a61b87,
    0x92fa1b83af80e7d3,
    0x646fc86786ebe32f,
    0x361c86c461cd1f7b,
    0xbd9d49fc8caa79bc,
    0xefee075f6b8c85e8,
    0x197bd4bb42e78114,
    0x4b089a18a5c17d40,
    0x3aa16c99c0bedff1,
    0x68d2223a279823a5,
    0x9e47f1de0ef32759,
    0xcc34bf7de9d5db0d,
    0x47b5704504b2bdca,
    0x15c63ee6e394419e,
    0xe353ed02caff4562,
    0xb120a3a12dd9b936,
    0xb5cb8c13c9dba465,
    0xe7b8c2b02efd5831,
    0x112d115407965ccd,
    0x435e5ff7e0b0a099,
    0xc8df90cf0dd7c65e,
    0x9aacde6ceaf13a0a,
    0x6c390d88c39a3ef6,
    0x3e4a432b24bcc2a2,
    0x4fe3b5aa41c36013,
    0x1d90fb09a6e59c47,
    0xeb0528ed8f8e98bb,
    0xb976664e68a864ef,
    0x32f7a97685cf0228,
    0x6084e7d562e9fe7c,
    0x961134314b82fa80,
    0xc4627a92aca406d4,
    0x7542d933817dbfe2,
    0x27319790665b43b6,
    0xd1a444744f30474a,
    0x83d70ad7a816bb1e,
    0x0856c5ef4571ddd9,
    0x5a258b4ca257218d,
    0xacb058a88b3c2571,
    0xfec3160b6c1ad925,
    0x8f6ae08a09657b94,
    0xdd19ae29ee4387c0,
    0x2b8c7dcdc728833c,
    0x79ff336e200e7f68,
    0xf27efc56cd6919af,
    0xa00db2f52a4fe5fb,
    0x569861110324e107,
    0x04eb2fb2e4021d53,
    0x5f4e3e74cb20dba1,
    0x0d3d70d72c0627f5,
    0xfba8a333056d2309,
    0xa9dbed90e24bdf5d,
    0x225a22a80f2cb99a,
    0x70296c0be80a45ce,
    0x86bcbfefc1614132,
    0xd4cff14c2647bd66,
    0xa56607cd43381fd7,
    0xf715496ea41ee383,
    0x01809a8a8d75e77f,
    0x53f3d4296a531b2b,
    0xd8721b1187347dec,
    0x8a0155b2601281b8,
    0x7c94865649798544,
    0x2ee7c8f5ae5f7910,
    0x9fc76b548386c026,
    0xcdb425f764a03c72,
    0x3b21f6134dcb388e,
    0x6952b8b0aaedc4da,
    0xe2d37788478aa21d,
    0xb0a0392ba0ac5e49,
    0x4635eacf89c75ab5,
    0x1446a46c6ee1a6e1,
    0x65ef52ed0b9e0450,
    0x379c1c4eecb8f804,
    0xc109cfaac5d3fcf8,
    0x937a810922f500ac,
    0x18fb4e31cf92666b,
    0x4a88009228b49a3f,
    0xbc1dd37601df9ec3,
    0xee6e9dd5e6f96297,
    0xea85b26702fb7fc4,
    0xb8f6fcc4e5dd8390,
    0x4e632f20ccb6876c,
    0x1c1061832b907b38,
    0x9791aebbc6f71dff,
    0xc5e2e01821d1e1ab,
    0x337733fc08bae557,
    0x61047d5fef9c1903,
    0x10ad8bde8ae3bbb2,
    0x42dec57d6dc547e6,
    0xb44b169944ae431a,
    0xe638583aa388bf4e,
    0x6db997024eefd989,
    0x3fcad9a1a9c925dd,
    0xc95f0a4580a22121,
    0x9b2c44e66784dd75,
    0x2a0ce7474a5d6443,
    0x787fa9e4ad7b9817,
    0x8eea7a0084109ceb,
    0xdc9934a3633660bf,
    0x5718fb9b8e510678,
    0x056bb5386977fa2c,
    0xf3fe66dc401cfed0,
    0xa18d287fa73a0284,
    0xd024defec245a035,
    0x8257905d25635c61,
    0x74c243b90c08589d,
    0x26b10d1aeb2ea4c9,
    0xad30c2220649c20e,
    0xff438c81e16f3e5a,
    0x09d65f65c8043aa6,
    0x5ba511c62f22c6f2,
    0xbe9c7ce99641b742,
    0xecef324a71674b16,
    0x1a7ae1ae580c4fea,
    0x4809af0dbf2ab3be,
    0xc3886035524dd579,
    0x91fb2e96b56b292d,
    0x676efd729c002dd1,
    0x351db3d17b26d185,
    0x44b445501e597334,
    0x16c70bf3f97f8f60,
    0xe052d817d0148b9c,
    0xb22196b4373277c8,
    0x39a0598cda55110f,
    0x6bd3172f3d73ed5b,
    0x9d46c4cb1418e9a7,
    0xcf358a68f33e15f3,
    0x7e1529c9dee7acc5,
    0x2c66676a39c15091,
    0xdaf3b48e10aa546d,
    0x8880fa2df78ca839,
    0x030135151aebcefe,
    0x51727bb6fdcd32aa,
    0xa7e7a852d4a63656,
    0xf594e6f13380ca02,
    0x843d107056ff68b3,
    0xd64e5ed3b1d994e7,
    0x20db8d3798b2901b,
    0x72a8c3947f946c4f,
    0xf9290cac92f30a88,
    0xab5a420f75d5f6dc,
    0x5dcf91eb5cbef220,
    0x0fbcdf48bb980e74,
    0x0b57f0fa5f9a1327,
    0x5924be59b8bcef73,
    0xafb16dbd91d7eb8f,
    0xfdc2231e76f117db,
    0x7643ec269b96711c,
    0x2430a2857cb08d48,
    0xd2a5716155db89b4,
    0x80d63fc2b2fd75e0,
    0xf17fc943d782d751,
    0xa30c87e030a42b05,
    0x5599540419cf2ff9,
    0x07ea1aa7fee9d3ad,
    0x8c6bd59f138eb56a,
    0xde189b3cf4a8493e,
    0x288d48d8ddc34dc2,
    0x7afe067b3ae5b196,
    0xcbdea5da173c08a0,
    0x99adeb79f01af4f4,
    0x6f38389dd971f008,
    0x3d4b763e3e570c5c,
    0xb6cab906d3306a9b,
    0xe4b9f7a5341696cf,
    0x122c24411d7d9233,
    0x405f6ae2fa5b6e67,
    0x31f69c639f24ccd6,
    0x6385d2c078023082,
    0x951001245169347e,
    0xc7634f87b64fc82a,
    0x4ce280bf5b28aeed,
    0x1e91ce1cbc0e52b9,
    0xe8041df895655645,
    0xba77535b7243aa11,
    0xe1d2429d5d616ce3,
    0xb3a10c3eba4790b7,
    0x4534dfda932c944b,
    0x17479179740a681f,
    0x9cc65e41996d0ed8,
    0xceb510e27e4bf28c,
    0x3820c3065720f670,
    0x6a538da5b0060a24,
    0x1bfa7b24d579a895,
    0x49893587325f54c1,
    0xbf1ce6631b34503d,
    0xed6fa8c0fc12ac69,
    0x66ee67f81175caae,
    0x349d295bf65336fa,
    0xc208fabfdf383206,
    0x907bb41c381ece52,
    0x215b17bd15c77764,
    0x7328591ef2e18b30,
    0x85bd8afadb8a8fcc,
    0xd7cec4593cac7398,
    0x5c4f0b61d1cb155f,
    0x0e3c45c236ede90b,
    0xf8a996261f86edf7,
    0xaadad885f8a011a3,
    0xdb732e049ddfb312,
    0x890060a77af94f46,
    0x7f95b34353924bba,
    0x2de6fde0b4b4b7ee,
    0xa66732d859d3d129,
    0xf4147c7bbef52d7d,
    0x0281af9f979e2981,
    0x50f2e13c70b8d5d5,
    0x5419ce8e94bac886,
    0x066a802d739c34d2,
    0xf0ff53c95af7302e,
    0xa28c1d6abdd1cc7a,
    0x290dd25250b6aabd,
    0x7b7e9cf1b79056e9,
    0x8deb4f159efb5215,
    0xdf9801b679ddae41,
    0xae31f7371ca20cf0,
    0xfc42b994fb84f0a4,
    0x0ad76a70d2eff458,
    0x58a424d335c9080c,
    0xd325ebebd8ae6ecb,
    0x8156a5483f88929f,
    0x77c376ac16e39663,
    0x25b0380ff1c56a37,
    0x94909baedc1cd301,
    0xc6e3d50d3b3a2f55,
    0x307606e912512ba9,
    0x6205484af577d7fd,
    0xe98487721810b13a,
    0xbbf7c9d1ff364d6e,
    0x4d621a35d65d4992,
    0x1f115496317bb5c6,
    0x6eb8a21754041777,
    0x3ccbecb4b322eb23,
    0xca5e3f509a49efdf,
    0x982d71f37d6f138b,
    0x13acbecb9008754c,
    0x41dff068772e8918,
    0xb74a238c5e458de4,
    0xe5396d2fb96371b0,
];

// CRC table for the NVME polynomial in the range (`1 = x¹⁵⁹, …, 128 = x¹⁵²`).
// Generated by running `./build_table 11`.
static TABLE_11: [u64; 256] = [
    0x0000000000000000,
    0x668ab3bbc976d29d,
    0xcd15677792eda53a,
    0xab9fd4cc5b9b77a7,
    0xaef3e8bc7d4cd91f,
    0xc8795b07b43a0b82,
    0x63e68fcbefa17c25,
    0x056c3c7026d7aeb8,
    0x693ef72ba20e2155,
    0x0fb444906b78f3c8,
    0xa42b905c30e3846f,
    0xc2a123e7f99556f2,
    0xc7cd1f97df42f84a,
    0xa147ac2c16342ad7,
    0x0ad878e04daf5d70,
    0x6c52cb5b84d98fed,
    0xd27dee57441c42aa,
    0xb4f75dec8d6a9037,
    0x1f688920d6f1e790,
    0x79e23a9b1f87350d,
    0x7c8e06eb39509bb5,
    0x1a04b550f0264928,
    0xb19b619cabbd3e8f,
    0xd711d22762cbec12,
    0xbb43197ce61263ff,
    0xddc9aac72f64b162,
    0x76567e0b74ffc6c5,
    0x10dccdb0bd891458,
    0x15b0f1c09b5ebae0,
    0x733a427b5228687d,
    0xd8a596b709b31fda,
    0xbe2f250cc0c5cd47,
    0x9022fafdd0af163f,
    0xf6a8494619d9c4a2,
    0x5d379d8a4242b305,
    0x3bbd2e318b346198,
    0x3ed11241ade3cf20,
    0x585ba1fa64951dbd,
    0xf3c475363f0e6a1a,
    0x954ec68df678b887,
    0xf91c0dd672a1376a,
    0x9f96be6dbbd7e5f7,
    0x34096aa1e04c9250,
    0x5283d91a293a40cd,
    0x57efe56a0fedee75,
    0x316556d1c69b3ce8,
    0x9afa821d9d004b4f,
    0xfc7031a6547699d2,
    0x425f14aa94b35495,
    0x24d5a7115dc58608,
    0x8f4a73dd065ef1af,
    0xe9c0c066cf282332,
    0xecacfc16e9ff8d8a,
    0x8a264fad20895f17,
    0x21b99b617b1228b0,
    0x473328dab264fa2d,
    0x2b61e38136bd75c0,
    0x4deb503affcba75d,
    0xe67484f6a450d0fa,
    0x80fe374d6d260267,
    0x85920b3d4bf1acdf,
    0xe318b88682877e42,
    0x48876c4ad91c09e5,
    0x2e0ddff1106adb78,
    0x149cd3a8f9c9bf15,
    0x7216601330bf6d88,
    0xd989b4df6b241a2f,
    0xbf030764a252c8b2,
    0xba6f3b148485660a,
    0xdce588af4df3b497,
    0x777a5c631668c330,
    0x11f0efd8df1e11ad,
    0x7da224835bc79e40,
    0x1b28973892b14cdd,
    0xb0b743f4c92a3b7a,
    0xd63df04f005ce9e7,
    0xd351cc3f268b475f,
    0xb5db7f84effd95c2,
    0x1e44ab48b466e265,
    0x78ce18f37d1030f8,
    0xc6e13dffbdd5fdbf,
    0xa06b8e4474a32f22,
    0x0bf45a882f385885,
    0x6d7ee933e64e8a18,
    0x6812d543c09924a0,
    0x0e9866f809eff63d,
    0xa507b2345274819a,
    0xc38d018f9b025307,
    0xafdfcad41fdbdcea,
    0xc955796fd6ad0e77,
    0x62caada38d3679d0,
    0x04401e184440ab4d,
    0x012c2268629705f5,
    0x67a691d3abe1d768,
    0xcc39451ff07aa0cf,
    0xaab3f6a4390c7252,
    0x84be29552966a92a,
    0xe2349aeee0107bb7,
    0x49ab4e22bb8b0c10,
    0x2f21fd9972fdde8d,
    0x2a4dc1e9542a7035,
    0x4cc772529d5ca2a8,
    0xe758a69ec6c7d50f,
    0x81d215250fb10792,
    0xed80de7e8b68887f,
    0x8b0a6dc5421e5ae2,
    0x2095b90919852d45,
    0x461f0ab2d0f3ffd8,
    0x437336c2f6245160,
    0x25f985793f5283fd,
    0x8e6651b564c9f45a,
    0xe8ece20eadbf26c7,
    0x56c3c7026d7aeb80,
    0x304974b9a40c391d,
    0x9bd6a075ff974eba,
    0xfd5c13ce36e19c27,
    0xf8302fbe1036329f,
    0x9eba9c05d940e002,
    0x352548c982db97a5,
    0x53affb724bad4538,
    0x3ffd3029cf74cad5,
    0x5977839206021848,
    0xf2e8575e5d996fef,
    0x9462e4e594efbd72,
    0x910ed895b23813ca,
    0xf7846b2e7b4ec157,
    0x5c1bbfe220d5b6f0,
    0x3a910c59e9a3646d,
    0x2939a751f3937e2a,
    0x4fb314ea3ae5acb7,
    0xe42cc026617edb10,
    0x82a6739da808098d,
    0x87ca4fed8edfa735,
    0xe140fc5647a975a8,
    0x4adf289a1c32020f,
    0x2c559b21d544d092,
    0x4007507a519d5f7f,
    0x268de3c198eb8de2,
    0x8d12370dc370fa45,
    0xeb9884b60a0628d8,
    0xeef4b8c62cd18660,
    0x887e0b7de5a754fd,
    0x23e1dfb1be3c235a,
    0x456b6c0a774af1c7,
    0xfb444906b78f3c80,
    0x9dcefabd7ef9ee1d,
    0x36512e71256299ba,
    0x50db9dcaec144b27,
    0x55b7a1bacac3e59f,
    0x333d120103b53702,
    0x98a2c6cd582e40a5,
    0xfe28757691589238,
    0x927abe2d15811dd5,
    0xf4f00d96dcf7cf48,
    0x5f6fd95a876cb8ef,
    0x39e56ae14e1a6a72,
    0x3c89569168cdc4ca,
    0x5a03e52aa1bb1657,
    0xf19c31e6fa2061f0,
    0x9716825d3356b36d,
    0xb91b5dac233c6815,
    0xdf91ee17ea4aba88,
    0x740e3adbb1d1cd2f,
    0x1284896078a71fb2,
    0x17e8b5105e70b10a,
    0x716206ab97066397,
    0xdafdd267cc9d1430,
    0xbc7761dc05ebc6ad,
    0xd025aa8781324940,
    0xb6af193c48449bdd,
    0x1d30cdf013dfec7a,
    0x7bba7e4bdaa93ee7,
    0x7ed6423bfc7e905f,
    0x185cf180350842c2,
    0xb3c3254c6e933565,
    0xd54996f7a7e5e7f8,
    0x6b66b3fb67202abf,
    0x0dec0040ae56f822,
    0xa673d48cf5cd8f85,
    0xc0f967373cbb5d18,
    0xc5955b471a6cf3a0,
    0xa31fe8fcd31a213d,
    0x08803c308881569a,
    0x6e0a8f8b41f78407,
    0x025844d0c52e0bea,
    0x64d2f76b0c58d977,
    0xcf4d23a757c3aed0,
    0xa9c7901c9eb57c4d,
    0xacabac6cb862d2f5,
    0xca211fd771140068,
    0x61becb1b2a8f77cf,
    0x073478a0e3f9a552,
    0x3da574f90a5ac13f,
    0x5b2fc742c32c13a2,
    0xf0b0138e98b76405,
    0x963aa03551c1b698,
    0x93569c4577161820,
    0xf5dc2ffebe60cabd,
    0x5e43fb32e5fbbd1a,
    0x38c948892c8d6f87,
    0x549b83d2a854e06a,
    0x32113069612232f7,
    0x998ee4a53ab94550,
    0xff04571ef3cf97cd,
    0xfa686b6ed5183975,
    0x9ce2d8d51c6eebe8,
    0x377d0c1947f59c4f,
    0x51f7bfa28e834ed2,
    0xefd89aae4e468395,
    0x8952291587305108,
    0x22cdfdd9dcab26af,
    0x44474e6215ddf432,
    0x412b7212330a5a8a,
    0x27a1c1a9fa7c8817,
    0x8c3e1565a1e7ffb0,
    0xeab4a6de68912d2d,
    0x86e66d85ec48a2c0,
    0xe06cde3e253e705d,
    0x4bf30af27ea507fa,
    0x2d79b949b7d3d567,
    0x2815853991047bdf,
    0x4e9f36825872a942,
    0xe500e24e03e9dee5,
    0x838a51f5ca9f0c78,
    0xad878e04daf5d700,
    0xcb0d3dbf1383059d,
    0x6092e9734818723a,
    0x06185ac8816ea0a7,
    0x037466b8a7b90e1f,
    0x65fed5036ecfdc82,
    0xce6101cf3554ab25,
    0xa8ebb274fc2279b8,
    0xc4b9792f78fbf655,
    0xa233ca94b18d24c8,
    0x09ac1e58ea16536f,
    0x6f26ade3236081f2,
    0x6a4a919305b72f4a,
    0x0cc02228ccc1fdd7,
    0xa75ff6e4975a8a70,
    0xc1d5455f5e2c58ed,
    0x7ffa60539ee995aa,
    0x1970d3e8579f4737,
    0xb2ef07240c043090,
    0xd465b49fc572e20d,
    0xd10988efe3a54cb5,
    0xb7833b542ad39e28,
    0x1c1cef987148e98f,
    0x7a965c23b83e3b12,
    0x16c497783ce7b4ff,
    0x704e24c3f5916662,
    0xdbd1f00fae0a11c5,
    0xbd5b43b4677cc358,
    0xb8377fc441ab6de0,
    0xdebdcc7f88ddbf7d,
    0x752218b3d346c8da,
    0x13a8ab081a301a47,
];

// CRC table for the NVME polynomial in the range (`1 = x¹⁶⁷, …, 128 = x¹⁶⁰`).
// Generated by running `./build_table 12`.
static TABLE_12: [u64; 256] = [
    0x0000000000000000,
    0xf2fa1fae5f5c1165,
    0xd12d190fe62fb1a1,
    0x23d706a1b973a0c4,
    0x9683144c94c8f029,
    0x64790be2cb94e14c,
    0x47ae0d4372e74188,
    0xb55412ed2dbb50ed,
    0x19df0eca71067339,
    0xeb2511642e5a625c,
    0xc8f217c59729c298,
    0x3a08086bc875d3fd,
    0x8f5c1a86e5ce8310,
    0x7da60528ba929275,
    0x5e71038903e132b1,
    0xac8b1c275cbd23d4,
    0x33be1d94e20ce672,
    0xc144023abd50f717,
    0xe293049b042357d3,
    0x10691b355b7f46b6,
    0xa53d09d876c4165b,
    0x57c716762998073e,
    0x741010d790eba7fa,
    0x86ea0f79cfb7b69f,
    0x2a61135e930a954b,
    0xd89b0cf0cc56842e,
    0xfb4c0a51752524ea,
    0x09b615ff2a79358f,
    0xbce2071207c26562,
    0x4e1818bc589e7407,
    0x6dcf1e1de1edd4c3,
    0x9f3501b3beb1c5a6,
    0x677c3b29c419cce4,
    0x958624879b45dd81,
    0xb651222622367d45,
    0x44ab3d887d6a6c20,
    0xf1ff2f6550d13ccd,
    0x030530cb0f8d2da8,
    0x20d2366ab6fe8d6c,
    0xd22829c4e9a29c09,
    0x7ea335e3b51fbfdd,
    0x8c592a4dea43aeb8,
    0xaf8e2cec53300e7c,
    0x5d7433420c6c1f19,
    0xe82021af21d74ff4,
    0x1ada3e017e8b5e91,
    0x390d38a0c7f8fe55,
    0xcbf7270e98a4ef30,
    0x54c226bd26152a96,
    0xa638391379493bf3,
    0x85ef3fb2c03a9b37,
    0x7715201c9f668a52,
    0xc24132f1b2dddabf,
    0x30bb2d5fed81cbda,
    0x136c2bfe54f26b1e,
    0xe19634500bae7a7b,
    0x4d1d2877571359af,
    0xbfe737d9084f48ca,
    0x9c303178b13ce80e,
    0x6eca2ed6ee60f96b,
    0xdb9e3c3bc3dba986,
    0x296423959c87b8e3,
    0x0ab3253425f41827,
    0xf8493a9a7aa80942,
    0xcef87653883399c8,
    0x3c0269fdd76f88ad,
    0x1fd56f5c6e1c2869,
    0xed2f70f23140390c,
    0x587b621f1cfb69e1,
    0xaa817db143a77884,
    0x89567b10fad4d840,
    0x7bac64bea588c925,
    0xd7277899f935eaf1,
    0x25dd6737a669fb94,
    0x060a61961f1a5b50,
    0xf4f07e3840464a35,
    0x41a46cd56dfd1ad8,
    0xb35e737b32a10bbd,
    0x908975da8bd2ab79,
    0x62736a74d48eba1c,
    0xfd466bc76a3f7fba,
    0x0fbc746935636edf,
    0x2c6b72c88c10ce1b,
    0xde916d66d34cdf7e,
    0x6bc57f8bfef78f93,
    0x993f6025a1ab9ef6,
    0xbae8668418d83e32,
    0x4812792a47842f57,
    0xe499650d1b390c83,
    0x16637aa344651de6,
    0x35b47c02fd16bd22,
    0xc74e63aca24aac47,
    0x721a71418ff1fcaa,
    0x80e06eefd0adedcf,
    0xa337684e69de4d0b,
    0x51cd77e036825c6e,
    0xa9844d7a4c2a552c,
    0x5b7e52d413764449,
    0x78a95475aa05e48d,
    0x8a534bdbf559f5e8,
    0x3f075936d8e2a505,
    0xcdfd469887beb460,
    0xee2a40393ecd14a4,
    0x1cd05f97619105c1,
    0xb05b43b03d2c2615,
    0x42a15c1e62703770,
    0x61765abfdb0397b4,
    0x938c4511845f86d1,
    0x26d857fca9e4d63c,
    0xd4224852f6b8c759,
    0xf7f54ef34fcb679d,
    0x050f515d109776f8,
    0x9a3a50eeae26b35e,
    0x68c04f40f17aa23b,
    0x4b1749e1480902ff,
    0xb9ed564f1755139a,
    0x0cb944a23aee4377,
    0xfe435b0c65b25212,
    0xdd945daddcc1f2d6,
    0x2f6e4203839de3b3,
    0x83e55e24df20c067,
    0x711f418a807cd102,
    0x52c8472b390f71c6,
    0xa0325885665360a3,
    0x15664a684be8304e,
    0xe79c55c614b4212b,
    0xc44b5367adc781ef,
    0x36b14cc9f29b908a,
    0xa929caf448f0a0fb,
    0x5bd3d55a17acb19e,
    0x7804d3fbaedf115a,
    0x8afecc55f183003f,
    0x3faadeb8dc3850d2,
    0xcd50c116836441b7,
    0xee87c7b73a17e173,
    0x1c7dd819654bf016,
    0xb0f6c43e39f6d3c2,
    0x420cdb9066aac2a7,
    0x61dbdd31dfd96263,
    0x9321c29f80857306,
    0x2675d072ad3e23eb,
    0xd48fcfdcf262328e,
    0xf758c97d4b11924a,
    0x05a2d6d3144d832f,
    0x9a97d760aafc4689,
    0x686dc8cef5a057ec,
    0x4bbace6f4cd3f728,
    0xb940d1c1138fe64d,
    0x0c14c32c3e34b6a0,
    0xfeeedc826168a7c5,
    0xdd39da23d81b0701,
    0x2fc3c58d87471664,
    0x8348d9aadbfa35b0,
    0x71b2c60484a624d5,
    0x5265c0a53dd58411,
    0xa09fdf0b62899574,
    0x15cbcde64f32c599,
    0xe731d248106ed4fc,
    0xc4e6d4e9a91d7438,
    0x361ccb47f641655d,
    0xce55f1dd8ce96c1f,
    0x3cafee73d3b57d7a,
    0x1f78e8d26ac6ddbe,
    0xed82f77c359accdb,
    0x58d6e59118219c36,
    0xaa2cfa3f477d8d53,
    0x89fbfc9efe0e2d97,
    0x7b01e330a1523cf2,
    0xd78aff17fdef1f26,
    0x2570e0b9a2b30e43,
    0x06a7e6181bc0ae87,
    0xf45df9b6449cbfe2,
    0x4109eb5b6927ef0f,
    0xb3f3f4f5367bfe6a,
    0x9024f2548f085eae,
    0x62deedfad0544fcb,
    0xfdebec496ee58a6d,
    0x0f11f3e731b99b08,
    0x2cc6f54688ca3bcc,
    0xde3ceae8d7962aa9,
    0x6b68f805fa2d7a44,
    0x9992e7aba5716b21,
    0xba45e10a1c02cbe5,
    0x48bffea4435eda80,
    0xe434e2831fe3f954,
    0x16cefd2d40bfe831,
    0x3519fb8cf9cc48f5,
    0xc7e3e422a6905990,
    0x72b7f6cf8b2b097d,
    0x804de961d4771818,
    0xa39aefc06d04b8dc,
    0x5160f06e3258a9b9,
    0x67d1bca7c0c33933,
    0x952ba3099f9f2856,
    0xb6fca5a826ec8892,
    0x4406ba0679b099f7,
    0xf152a8eb540bc91a,
    0x03a8b7450b57d87f,
    0x207fb1e4b22478bb,
    0xd285ae4aed7869de,
    0x7e0eb26db1c54a0a,
    0x8cf4adc3ee995b6f,
    0xaf23ab6257eafbab,
    0x5dd9b4cc08b6eace,
    0xe88da621250dba23,
    0x1a77b98f7a51ab46,
    0x39a0bf2ec3220b82,
    0xcb5aa0809c7e1ae7,
    0x546fa13322cfdf41,
    0xa695be9d7d93ce24,
    0x8542b83cc4e06ee0,
    0x77b8a7929bbc7f85,
    0xc2ecb57fb6072f68,
    0x3016aad1e95b3e0d,
    0x13c1ac7050289ec9,
    0xe13bb3de0f748fac,
    0x4db0aff953c9ac78,
    0xbf4ab0570c95bd1d,
    0x9c9db6f6b5e61dd9,
    0x6e67a958eaba0cbc,
    0xdb33bbb5c7015c51,
    0x29c9a41b985d4d34,
    0x0a1ea2ba212eedf0,
    0xf8e4bd147e72fc95,
    0x00ad878e04daf5d7,
    0xf25798205b86e4b2,
    0xd1809e81e2f54476,
    0x237a812fbda95513,
    0x962e93c2901205fe,
    0x64d48c6ccf4e149b,
    0x47038acd763db45f,
    0xb5f995632961a53a,
    0x1972894475dc86ee,
    0xeb8896ea2a80978b,
    0xc85f904b93f3374f,
    0x3aa58fe5ccaf262a,
    0x8ff19d08e11476c7,
    0x7d0b82a6be4867a2,
    0x5edc8407073bc766,
    0xac269ba95867d603,
    0x33139a1ae6d613a5,
    0xc1e985b4b98a02c0,
    0xe23e831500f9a204,
    0x10c49cbb5fa5b361,
    0xa5908e56721ee38c,
    0x576a91f82d42f2e9,
    0x74bd97599431522d,
    0x864788f7cb6d4348,
    0x2acc94d097d0609c,
    0xd8368b7ec88c71f9,
    0xfbe18ddf71ffd13d,
    0x091b92712ea3c058,
    0xbc4f809c031890b5,
    0x4eb59f325c4481d0,
    0x6d629993e5372114,
    0x9f98863dba6b3071,
];

// CRC table for the NVME polynomial in the range (`1 = x¹⁷⁵, …, 128 = x¹⁶⁸`).
// Generated by running `./build_table 13`.
static TABLE_13: [u64; 256] = [
    0x0000000000000000,
    0x9065cb6e6d39918a,
    0x1412b08f82e4b07f,
    0x84777be1efdd21f5,
    0x2825611f05c960fe,
    0xb840aa7168f0f174,
    0x3c37d190872dd081,
    0xac521afeea14410b,
    0x504ac23e0b92c1fc,
    0xc02f095066ab5076,
    0x445872b189767183,
    0xd43db9dfe44fe009,
    0x786fa3210e5ba102,
    0xe80a684f63623088,
    0x6c7d13ae8cbf117d,
    0xfc18d8c0e18680f7,
    0xa095847c172583f8,
    0x30f04f127a1c1272,
    0xb48734f395c13387,
    0x24e2ff9df8f8a20d,
    0x88b0e56312ece306,
    0x18d52e0d7fd5728c,
    0x9ca255ec90085379,
    0x0cc79e82fd31c2f3,
    0xf0df46421cb74204,
    0x60ba8d2c718ed38e,
    0xe4cdf6cd9e53f27b,
    0x74a83da3f36a63f1,
    0xd8fa275d197e22fa,
    0x489fec337447b370,
    0xcce897d29b9a9285,
    0x5c8d5cbcf6a3030f,
    0x75f22eab76dc949b,
    0xe597e5c51be50511,
    0x61e09e24f43824e4,
    0xf185554a9901b56e,
    0x5dd74fb47315f465,
    0xcdb284da1e2c65ef,
    0x49c5ff3bf1f1441a,
    0xd9a034559cc8d590,
    0x25b8ec957d4e5567,
    0xb5dd27fb1077c4ed,
    0x31aa5c1affaae518,
    0xa1cf977492937492,
    0x0d9d8d8a78873599,
    0x9df846e415bea413,
    0x198f3d05fa6385e6,
    0x89eaf66b975a146c,
    0xd567aad761f91763,
    0x450261b90cc086e9,
    0xc1751a58e31da71c,
    0x5110d1368e243696,
    0xfd42cbc86430779d,
    0x6d2700a60909e617,
    0xe9507b47e6d4c7e2,
    0x7935b0298bed5668,
    0x852d68e96a6bd69f,
    0x1548a38707524715,
    0x913fd866e88f66e0,
    0x015a130885b6f76a,
    0xad0809f66fa2b661,
    0x3d6dc298029b27eb,
    0xb91ab979ed46061e,
    0x297f7217807f9794,
    0xebe45d56edb92936,
    0x7b8196388080b8bc,
    0xfff6edd96f5d9949,
    0x6f9326b7026408c3,
    0xc3c13c49e87049c8,
    0x53a4f7278549d842,
    0xd7d38cc66a94f9b7,
    0x47b647a807ad683d,
    0xbbae9f68e62be8ca,
    0x2bcb54068b127940,
    0xafbc2fe764cf58b5,
    0x3fd9e48909f6c93f,
    0x938bfe77e3e28834,
    0x03ee35198edb19be,
    0x87994ef86106384b,
    0x17fc85960c3fa9c1,
    0x4b71d92afa9caace,
    0xdb14124497a53b44,
    0x5f6369a578781ab1,
    0xcf06a2cb15418b3b,
    0x6354b835ff55ca30,
    0xf331735b926c5bba,
    0x774608ba7db17a4f,
    0xe723c3d41088ebc5,
    0x1b3b1b14f10e6b32,
    0x8b5ed07a9c37fab8,
    0x0f29ab9b73eadb4d,
    0x9f4c60f51ed34ac7,
    0x331e7a0bf4c70bcc,
    0xa37bb16599fe9a46,
    0x270cca847623bbb3,
    0xb76901ea1b1a2a39,
    0x9e1673fd9b65bdad,
    0x0e73b893f65c2c27,
    0x8a04c37219810dd2,
    0x1a61081c74b89c58,
    0xb63312e29eacdd53,
    0x2656d98cf3954cd9,
    0xa221a26d1c486d2c,
    0x324469037171fca6,
    0xce5cb1c390f77c51,
    0x5e397aadfdceeddb,
    0xda4e014c1213cc2e,
    0x4a2bca227f2a5da4,
    0xe679d0dc953e1caf,
    0x761c1bb2f8078d25,
    0xf26b605317daacd0,
    0x620eab3d7ae33d5a,
    0x3e83f7818c403e55,
    0xaee63cefe179afdf,
    0x2a91470e0ea48e2a,
    0xbaf48c60639d1fa0,
    0x16a6969e89895eab,
    0x86c35df0e4b0cf21,
    0x02b426110b6deed4,
    0x92d1ed7f66547f5e,
    0x6ec935bf87d2ffa9,
    0xfeacfed1eaeb6e23,
    0x7adb853005364fd6,
    0xeabe4e5e680fde5c,
    0x46ec54a0821b9f57,
    0xd6899fceef220edd,
    0x52fee42f00ff2f28,
    0xc29b2f416dc6bea2,
    0xe3119cfe83e5c107,
    0x73745790eedc508d,
    0xf7032c7101017178,
    0x6766e71f6c38e0f2,
    0xcb34fde1862ca1f9,
    0x5b51368feb153073,
    0xdf264d6e04c81186,
    0x4f43860069f1800c,
    0xb35b5ec0887700fb,
    0x233e95aee54e9171,
    0xa749ee4f0a93b084,
    0x372c252167aa210e,
    0x9b7e3fdf8dbe6005,
    0x0b1bf4b1e087f18f,
    0x8f6c8f500f5ad07a,
    0x1f09443e626341f0,
    0x4384188294c042ff,
    0xd3e1d3ecf9f9d375,
    0x5796a80d1624f280,
    0xc7f363637b1d630a,
    0x6ba1799d91092201,
    0xfbc4b2f3fc30b38b,
    0x7fb3c91213ed927e,
    0xefd6027c7ed403f4,
    0x13cedabc9f528303,
    0x83ab11d2f26b1289,
    0x07dc6a331db6337c,
    0x97b9a15d708fa2f6,
    0x3bebbba39a9be3fd,
    0xab8e70cdf7a27277,
    0x2ff90b2c187f5382,
    0xbf9cc0427546c208,
    0x96e3b255f539559c,
    0x0686793b9800c416,
    0x82f102da77dde5e3,
    0x1294c9b41ae47469,
    0xbec6d34af0f03562,
    0x2ea318249dc9a4e8,
    0xaad463c57214851d,
    0x3ab1a8ab1f2d1497,
    0xc6a9706bfeab9460,
    0x56ccbb05939205ea,
    0xd2bbc0e47c4f241f,
    0x42de0b8a1176b595,
    0xee8c1174fb62f49e,
    0x7ee9da1a965b6514,
    0xfa9ea1fb798644e1,
    0x6afb6a9514bfd56b,
    0x36763629e21cd664,
    0xa613fd478f2547ee,
    0x226486a660f8661b,
    0xb2014dc80dc1f791,
    0x1e535736e7d5b69a,
    0x8e369c588aec2710,
    0x0a41e7b9653106e5,
    0x9a242cd70808976f,
    0x663cf417e98e1798,
    0xf6593f7984b78612,
    0x722e44986b6aa7e7,
    0xe24b8ff60653366d,
    0x4e199508ec477766,
    0xde7c5e66817ee6ec,
    0x5a0b25876ea3c719,
    0xca6eeee9039a5693,
    0x08f5c1a86e5ce831,
    0x98900ac6036579bb,
    0x1ce77127ecb8584e,
    0x8c82ba498181c9c4,
    0x20d0a0b76b9588cf,
    0xb0b56bd906ac1945,
    0x34c21038e97138b0,
    0xa4a7db568448a93a,
    0x58bf039665ce29cd,
    0xc8dac8f808f7b847,
    0x4cadb319e72a99b2,
    0xdcc878778a130838,
    0x709a628960074933,
    0xe0ffa9e70d3ed8b9,
    0x6488d206e2e3f94c,
    0xf4ed19688fda68c6,
    0xa86045d479796bc9,
    0x38058eba1440fa43,
    0xbc72f55bfb9ddbb6,
    0x2c173e3596a44a3c,
    0x804524cb7cb00b37,
    0x1020efa511899abd,
    0x94579444fe54bb48,
    0x04325f2a936d2ac2,
    0xf82a87ea72ebaa35,
    0x684f4c841fd23bbf,
    0xec383765f00f1a4a,
    0x7c5dfc0b9d368bc0,
    0xd00fe6f57722cacb,
    0x406a2d9b1a1b5b41,
    0xc41d567af5c67ab4,
    0x54789d1498ffeb3e,
    0x7d07ef0318807caa,
    0xed62246d75b9ed20,
    0x69155f8c9a64ccd5,
    0xf97094e2f75d5d5f,
    0x55228e1c1d491c54,
    0xc547457270708dde,
    0x41303e939fadac2b,
    0xd155f5fdf2943da1,
    0x2d4d2d3d1312bd56,
    0xbd28e6537e2b2cdc,
    0x395f9db291f60d29,
    0xa93a56dcfccf9ca3,
    0x05684c2216dbdda8,
    0x950d874c7be24c22,
    0x117afcad943f6dd7,
    0x811f37c3f906fc5d,
    0xdd926b7f0fa5ff52,
    0x4df7a011629c6ed8,
    0xc980dbf08d414f2d,
    0x59e5109ee078dea7,
    0xf5b70a600a6c9fac,
    0x65d2c10e67550e26,
    0xe1a5baef88882fd3,
    0x71c07181e5b1be59,
    0x8dd8a94104373eae,
    0x1dbd622f690eaf24,
    0x99ca19ce86d38ed1,
    0x09afd2a0ebea1f5b,
    0xa5fdc85e01fe5e50,
    0x359803306cc7cfda,
    0xb1ef78d1831aee2f,
    0x218ab3bfee237fa5,
];

// CRC table for the NVME polynomial in the range (`1 = x¹⁸³, …, 128 = x¹⁷⁶`).
// Generated by running `./build_table 14`.
static TABLE_14: [u64; 256] = [
    0x0000000000000000,
    0xc23dfbc6ca591ca3,
    0xb0a2d1decc25aa2d,
    0x729f2a18067cb68e,
    0x559c85eec0dcc731,
    0x97a17e280a85db92,
    0xe53e54300cf96d1c,
    0x2703aff6c6a071bf,
    0xab390bdd81b98e62,
    0x6904f01b4be092c1,
    0x1b9bda034d9c244f,
    0xd9a621c587c538ec,
    0xfea58e3341654953,
    0x3c9875f58b3c55f0,
    0x4e075fed8d40e37e,
    0x8c3aa42b4719ffdd,
    0x62ab31e85be48faf,
    0xa096ca2e91bd930c,
    0xd209e03697c12582,
    0x10341bf05d983921,
    0x3737b4069b38489e,
    0xf50a4fc05161543d,
    0x879565d8571de2b3,
    0x45a89e1e9d44fe10,
    0xc9923a35da5d01cd,
    0x0bafc1f310041d6e,
    0x7930ebeb1678abe0,
    0xbb0d102ddc21b743,
    0x9c0ebfdb1a81c6fc,
    0x5e33441dd0d8da5f,
    0x2cac6e05d6a46cd1,
    0xee9195c31cfd7072,
    0xc55663d0b7c91f5e,
    0x076b98167d9003fd,
    0x75f4b20e7becb573,
    0xb7c949c8b1b5a9d0,
    0x90cae63e7715d86f,
    0x52f71df8bd4cc4cc,
    0x206837e0bb307242,
    0xe255cc2671696ee1,
    0x6e6f680d3670913c,
    0xac5293cbfc298d9f,
    0xdecdb9d3fa553b11,
    0x1cf04215300c27b2,
    0x3bf3ede3f6ac560d,
    0xf9ce16253cf54aae,
    0x8b513c3d3a89fc20,
    0x496cc7fbf0d0e083,
    0xa7fd5238ec2d90f1,
    0x65c0a9fe26748c52,
    0x175f83e620083adc,
    0xd5627820ea51267f,
    0xf261d7d62cf157c0,
    0x305c2c10e6a84b63,
    0x42c30608e0d4fded,
    0x80fefdce2a8de14e,
    0x0cc459e56d941e93,
    0xcef9a223a7cd0230,
    0xbc66883ba1b1b4be,
    0x7e5b73fd6be8a81d,
    0x5958dc0bad48d9a2,
    0x9b6527cd6711c501,
    0xe9fa0dd5616d738f,
    0x2bc7f613ab346f2c,
    0xbe75e1f23705add7,
    0x7c481a34fd5cb174,
    0x0ed7302cfb2007fa,
    0xcceacbea31791b59,
    0xebe9641cf7d96ae6,
    0x29d49fda3d807645,
    0x5b4bb5c23bfcc0cb,
    0x99764e04f1a5dc68,
    0x154cea2fb6bc23b5,
    0xd77111e97ce53f16,
    0xa5ee3bf17a998998,
    0x67d3c037b0c0953b,
    0x40d06fc17660e484,
    0x82ed9407bc39f827,
    0xf072be1fba454ea9,
    0x324f45d9701c520a,
    0xdcded01a6ce12278,
    0x1ee32bdca6b83edb,
    0x6c7c01c4a0c48855,
    0xae41fa026a9d94f6,
    0x894255f4ac3de549,
    0x4b7fae326664f9ea,
    0x39e0842a60184f64,
    0xfbdd7fecaa4153c7,
    0x77e7dbc7ed58ac1a,
    0xb5da20012701b0b9,
    0xc7450a19217d0637,
    0x0578f1dfeb241a94,
    0x227b5e292d846b2b,
    0xe046a5efe7dd7788,
    0x92d98ff7e1a1c106,
    0x50e474312bf8dda5,
    0x7b23822280ccb289,
    0xb91e79e44a95ae2a,
    0xcb8153fc4ce918a4,
    0x09bca83a86b00407,
    0x2ebf07cc401075b8,
    0xec82fc0a8a49691b,
    0x9e1dd6128c35df95,
    0x5c202dd4466cc336,
    0xd01a89ff01753ceb,
    0x12277239cb2c2048,
    0x60b85821cd5096c6,
    0xa285a3e707098a65,
    0x85860c11c1a9fbda,
    0x47bbf7d70bf0e779,
    0x3524ddcf0d8c51f7,
    0xf7192609c7d54d54,
    0x1988b3cadb283d26,
    0xdbb5480c11712185,
    0xa92a6214170d970b,
    0x6b1799d2dd548ba8,
    0x4c1436241bf4fa17,
    0x8e29cde2d1ade6b4,
    0xfcb6e7fad7d1503a,
    0x3e8b1c3c1d884c99,
    0xb2b1b8175a91b344,
    0x708c43d190c8afe7,
    0x021369c996b41969,
    0xc02e920f5ced05ca,
    0xe72d3df99a4d7475,
    0x2510c63f501468d6,
    0x578fec275668de58,
    0x95b217e19c31c2fb,
    0x4832e5b7369cc8c5,
    0x8a0f1e71fcc5d466,
    0xf8903469fab962e8,
    0x3aadcfaf30e07e4b,
    0x1dae6059f6400ff4,
    0xdf939b9f3c191357,
    0xad0cb1873a65a5d9,
    0x6f314a41f03cb97a,
    0xe30bee6ab72546a7,
    0x213615ac7d7c5a04,
    0x53a93fb47b00ec8a,
    0x9194c472b159f029,
    0xb6976b8477f98196,
    0x74aa9042bda09d35,
    0x0635ba5abbdc2bbb,
    0xc408419c71853718,
    0x2a99d45f6d78476a,
    0xe8a42f99a7215bc9,
    0x9a3b0581a15ded47,
    0x5806fe476b04f1e4,
    0x7f0551b1ada4805b,
    0xbd38aa7767fd9cf8,
    0xcfa7806f61812a76,
    0x0d9a7ba9abd836d5,
    0x81a0df82ecc1c908,
    0x439d24442698d5ab,
    0x31020e5c20e46325,
    0xf33ff59aeabd7f86,
    0xd43c5a6c2c1d0e39,
    0x1601a1aae644129a,
    0x649e8bb2e038a414,
    0xa6a370742a61b8b7,
    0x8d6486678155d79b,
    0x4f597da14b0ccb38,
    0x3dc657b94d707db6,
    0xfffbac7f87296115,
    0xd8f80389418910aa,
    0x1ac5f84f8bd00c09,
    0x685ad2578dacba87,
    0xaa67299147f5a624,
    0x265d8dba00ec59f9,
    0xe460767ccab5455a,
    0x96ff5c64ccc9f3d4,
    0x54c2a7a20690ef77,
    0x73c10854c0309ec8,
    0xb1fcf3920a69826b,
    0xc363d98a0c1534e5,
    0x015e224cc64c2846,
    0xefcfb78fdab15834,
    0x2df24c4910e84497,
    0x5f6d66511694f219,
    0x9d509d97dccdeeba,
    0xba5332611a6d9f05,
    0x786ec9a7d03483a6,
    0x0af1e3bfd6483528,
    0xc8cc18791c11298b,
    0x44f6bc525b08d656,
    0x86cb47949151caf5,
    0xf4546d8c972d7c7b,
    0x3669964a5d7460d8,
    0x116a39bc9bd41167,
    0xd357c27a518d0dc4,
    0xa1c8e86257f1bb4a,
    0x63f513a49da8a7e9,
    0xf647044501996512,
    0x347aff83cbc079b1,
    0x46e5d59bcdbccf3f,
    0x84d82e5d07e5d39c,
    0xa3db81abc145a223,
    0x61e67a6d0b1cbe80,
    0x137950750d60080e,
    0xd144abb3c73914ad,
    0x5d7e0f988020eb70,
    0x9f43f45e4a79f7d3,
    0xeddcde464c05415d,
    0x2fe12580865c5dfe,
    0x08e28a7640fc2c41,
    0xcadf71b08aa530e2,
    0xb8405ba88cd9866c,
    0x7a7da06e46809acf,
    0x94ec35ad5a7deabd,
    0x56d1ce6b9024f61e,
    0x244ee47396584090,
    0xe6731fb55c015c33,
    0xc170b0439aa12d8c,
    0x034d4b8550f8312f,
    0x71d2619d568487a1,
    0xb3ef9a5b9cdd9b02,
    0x3fd53e70dbc464df,
    0xfde8c5b6119d787c,
    0x8f77efae17e1cef2,
    0x4d4a1468ddb8d251,
    0x6a49bb9e1b18a3ee,
    0xa8744058d141bf4d,
    0xdaeb6a40d73d09c3,
    0x18d691861d641560,
    0x33116795b6507a4c,
    0xf12c9c537c0966ef,
    0x83b3b64b7a75d061,
    0x418e4d8db02cccc2,
    0x668de27b768cbd7d,
    0xa4b019bdbcd5a1de,
    0xd62f33a5baa91750,
    0x1412c86370f00bf3,
    0x98286c4837e9f42e,
    0x5a15978efdb0e88d,
    0x288abd96fbcc5e03,
    0xeab74650319542a0,
    0xcdb4e9a6f735331f,
    0x0f8912603d6c2fbc,
    0x7d1638783b109932,
    0xbf2bc3bef1498591,
    0x51ba567dedb4f5e3,
    0x9387adbb27ede940,
    0xe11887a321915fce,
    0x23257c65ebc8436d,
    0x0426d3932d6832d2,
    0xc61b2855e7312e71,
    0xb484024de14d98ff,
    0x76b9f98b2b14845c,
    0xfa835da06c0d7b81,
    0x38bea666a6546722,
    0x4a218c7ea028d1ac,
    0x881c77b86a71cd0f,
    0xaf1fd84eacd1bcb0,
    0x6d2223886688a013,
    0x1fbd099060f4169d,
    0xdd80f256aaad0a3e,
];

// CRC table for the NVME polynomial in the range (`1 = x¹⁹¹, …, 128 = x¹⁸⁴`).
// Generated by running `./build_table 15`.
static TABLE_15: [u64; 256] = [
    0x0000000000000000,
    0xeadc41fd2ba3d420,
    0xe161a5a90fd03b2b,
    0x0bbde4542473ef0b,
    0xf61a6d014737e53d,
    0x1cc62cfc6c94311d,
    0x177bc8a848e7de16,
    0xfda7895563440a36,
    0xd8edfc51d6f85911,
    0x3231bdacfd5b8d31,
    0x398c59f8d928623a,
    0xd3501805f28bb61a,
    0x2ef7915091cfbc2c,
    0xc42bd0adba6c680c,
    0xcf9634f99e1f8707,
    0x254a7504b5bc5327,
    0x8502def0f5672149,
    0x6fde9f0ddec4f569,
    0x64637b59fab71a62,
    0x8ebf3aa4d114ce42,
    0x7318b3f1b250c474,
    0x99c4f20c99f31054,
    0x92791658bd80ff5f,
    0x78a557a596232b7f,
    0x5def22a1239f7858,
    0xb733635c083cac78,
    0xbc8e87082c4f4373,
    0x5652c6f507ec9753,
    0xabf54fa064a89d65,
    0x41290e5d4f0b4945,
    0x4a94ea096b78a64e,
    0xa048abf440db726e,
    0x3edc9bb2b259d1f9,
    0xd400da4f99fa05d9,
    0xdfbd3e1bbd89ead2,
    0x35617fe6962a3ef2,
    0xc8c6f6b3f56e34c4,
    0x221ab74edecde0e4,
    0x29a7531afabe0fef,
    0xc37b12e7d11ddbcf,
    0xe63167e364a188e8,
    0x0ced261e4f025cc8,
    0x0750c24a6b71b3c3,
    0xed8c83b740d267e3,
    0x102b0ae223966dd5,
    0xfaf74b1f0835b9f5,
    0xf14aaf4b2c4656fe,
    0x1b96eeb607e582de,
    0xbbde4542473ef0b0,
    0x510204bf6c9d2490,
    0x5abfe0eb48eecb9b,
    0xb063a116634d1fbb,
    0x4dc428430009158d,
    0xa71869be2baac1ad,
    0xaca58dea0fd92ea6,
    0x4679cc17247afa86,
    0x6333b91391c6a9a1,
    0x89eff8eeba657d81,
    0x82521cba9e16928a,
    0x688e5d47b5b546aa,
    0x9529d412d6f14c9c,
    0x7ff595effd5298bc,
    0x744871bbd92177b7,
    0x9e943046f282a397,
    0x7db9376564b3a3f2,
    0x976576984f1077d2,
    0x9cd892cc6b6398d9,
    0x7604d33140c04cf9,
    0x8ba35a64238446cf,
    0x617f1b99082792ef,
    0x6ac2ffcd2c547de4,
    0x801ebe3007f7a9c4,
    0xa554cb34b24bfae3,
    0x4f888ac999e82ec3,
    0x44356e9dbd9bc1c8,
    0xaee92f60963815e8,
    0x534ea635f57c1fde,
    0xb992e7c8dedfcbfe,
    0xb22f039cfaac24f5,
    0x58f34261d10ff0d5,
    0xf8bbe99591d482bb,
    0x1267a868ba77569b,
    0x19da4c3c9e04b990,
    0xf3060dc1b5a76db0,
    0x0ea18494d6e36786,
    0xe47dc569fd40b3a6,
    0xefc0213dd9335cad,
    0x051c60c0f290888d,
    0x205615c4472cdbaa,
    0xca8a54396c8f0f8a,
    0xc137b06d48fce081,
    0x2bebf190635f34a1,
    0xd64c78c5001b3e97,
    0x3c9039382bb8eab7,
    0x372ddd6c0fcb05bc,
    0xddf19c912468d19c,
    0x4365acd7d6ea720b,
    0xa9b9ed2afd49a62b,
    0xa204097ed93a4920,
    0x48d84883f2999d00,
    0xb57fc1d691dd9736,
    0x5fa3802bba7e4316,
    0x541e647f9e0dac1d,
    0xbec22582b5ae783d,
    0x9b88508600122b1a,
    0x7154117b2bb1ff3a,
    0x7ae9f52f0fc21031,
    0x9035b4d22461c411,
    0x6d923d874725ce27,
    0x874e7c7a6c861a07,
    0x8cf3982e48f5f50c,
    0x662fd9d36356212c,
    0xc6677227238d5342,
    0x2cbb33da082e8762,
    0x2706d78e2c5d6869,
    0xcdda967307febc49,
    0x307d1f2664bab67f,
    0xdaa15edb4f19625f,
    0xd11cba8f6b6a8d54,
    0x3bc0fb7240c95974,
    0x1e8a8e76f5750a53,
    0xf456cf8bded6de73,
    0xffeb2bdffaa53178,
    0x15376a22d106e558,
    0xe890e377b242ef6e,
    0x024ca28a99e13b4e,
    0x09f146debd92d445,
    0xe32d072396310065,
    0xfb726ecac96747e4,
    0x11ae2f37e2c493c4,
    0x1a13cb63c6b77ccf,
    0xf0cf8a9eed14a8ef,
    0x0d6803cb8e50a2d9,
    0xe7b44236a5f376f9,
    0xec09a662818099f2,
    0x06d5e79faa234dd2,
    0x239f929b1f9f1ef5,
    0xc943d366343ccad5,
    0xc2fe3732104f25de,
    0x282276cf3becf1fe,
    0xd585ff9a58a8fbc8,
    0x3f59be67730b2fe8,
    0x34e45a335778c0e3,
    0xde381bce7cdb14c3,
    0x7e70b03a3c0066ad,
    0x94acf1c717a3b28d,
    0x9f11159333d05d86,
    0x75cd546e187389a6,
    0x886add3b7b378390,
    0x62b69cc6509457b0,
    0x690b789274e7b8bb,
    0x83d7396f5f446c9b,
    0xa69d4c6beaf83fbc,
    0x4c410d96c15beb9c,
    0x47fce9c2e5280497,
    0xad20a83fce8bd0b7,
    0x5087216aadcfda81,
    0xba5b6097866c0ea1,
    0xb1e684c3a21fe1aa,
    0x5b3ac53e89bc358a,
    0xc5aef5787b3e961d,
    0x2f72b485509d423d,
    0x24cf50d174eead36,
    0xce13112c5f4d7916,
    0x33b498793c097320,
    0xd968d98417aaa700,
    0xd2d53dd033d9480b,
    0x38097c2d187a9c2b,
    0x1d430929adc6cf0c,
    0xf79f48d486651b2c,
    0xfc22ac80a216f427,
    0x16feed7d89b52007,
    0xeb596428eaf12a31,
    0x018525d5c152fe11,
    0x0a38c181e521111a,
    0xe0e4807cce82c53a,
    0x40ac2b888e59b754,
    0xaa706a75a5fa6374,
    0xa1cd8e2181898c7f,
    0x4b11cfdcaa2a585f,
    0xb6b64689c96e5269,
    0x5c6a0774e2cd8649,
    0x57d7e320c6be6942,
    0xbd0ba2dded1dbd62,
    0x9841d7d958a1ee45,
    0x729d962473023a65,
    0x792072705771d56e,
    0x93fc338d7cd2014e,
    0x6e5bbad81f960b78,
    0x8487fb253435df58,
    0x8f3a1f7110463053,
    0x65e65e8c3be5e473,
    0x86cb59afadd4e416,
    0x6c17185286773036,
    0x67aafc06a204df3d,
    0x8d76bdfb89a70b1d,
    0x70d134aeeae3012b,
    0x9a0d7553c140d50b,
    0x91b09107e5333a00,
    0x7b6cd0face90ee20,
    0x5e26a5fe7b2cbd07,
    0xb4fae403508f6927,
    0xbf47005774fc862c,
    0x559b41aa5f5f520c,
    0xa83cc8ff3c1b583a,
    0x42e0890217b88c1a,
    0x495d6d5633cb6311,
    0xa3812cab1868b731,
    0x03c9875f58b3c55f,
    0xe915c6a27310117f,
    0xe2a822f65763fe74,
    0x0874630b7cc02a54,
    0xf5d3ea5e1f842062,
    0x1f0faba33427f442,
    0x14b24ff710541b49,
    0xfe6e0e0a3bf7cf69,
    0xdb247b0e8e4b9c4e,
    0x31f83af3a5e8486e,
    0x3a45dea7819ba765,
    0xd0999f5aaa387345,
    0x2d3e160fc97c7973,
    0xc7e257f2e2dfad53,
    0xcc5fb3a6c6ac4258,
    0x2683f25bed0f9678,
    0xb817c21d1f8d35ef,
    0x52cb83e0342ee1cf,
    0x597667b4105d0ec4,
    0xb3aa26493bfedae4,
    0x4e0daf1c58bad0d2,
    0xa4d1eee1731904f2,
    0xaf6c0ab5576aebf9,
    0x45b04b487cc93fd9,
    0x60fa3e4cc9756cfe,
    0x8a267fb1e2d6b8de,
    0x819b9be5c6a557d5,
    0x6b47da18ed0683f5,
    0x96e0534d8e4289c3,
    0x7c3c12b0a5e15de3,
    0x7781f6e48192b2e8,
    0x9d5db719aa3166c8,
    0x3d151cedeaea14a6,
    0xd7c95d10c149c086,
    0xdc74b944e53a2f8d,
    0x36a8f8b9ce99fbad,
    0xcb0f71ecadddf19b,
    0x21d33011867e25bb,
    0x2a6ed445a20dcab0,
    0xc0b295b889ae1e90,
    0xe5f8e0bc3c124db7,
    0x0f24a14117b19997,
    0x0499451533c2769c,
    0xee4504e81861a2bc,
    0x13e28dbd7b25a88a,
    0xf93ecc4050867caa,
    0xf283281474f593a1,
    0x185f69e95f564781,
];
