/*  triconnected.cpp
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#include "triconnected.h"
#include "dfspalmtree.h"
#include "functions.h"
#include "combinatorics.h"

using namespace std;

namespace Reduze {

// TriConnectedComponents

TriConnectedComponents::TriConnectedComponents(const Topology& t,
		bool suppress_edge_permutations) {
	LOGXX("Finding triconnected components of '" << t.name() << "'");
#ifdef DEBUG
	t.print_dot("topo.dot");
#endif
	LOGXX(t);
	list<Topology> other;
	list<Topology> bcc = t.biconnected_vacuum_components(other);
	for (list<Topology>::const_iterator c = other.begin(); c != other.end(); ++c) {
		map<int, Edge>::const_iterator e;
		for (e = c->edges().begin(); e != c->edges().end(); ++e)
			nonbiconnected_.insert_edge(e->second);
	}
	LOGXX("found " << bcc.size() << " biconnected vacuum components");
	list<SplitComponents> tcc;
#ifdef DEBUG
	int sccount = 0;
#endif
	for (list<Topology>::const_iterator b = bcc.begin(); b != bcc.end(); ++b) {
		if (b->num_edges() == 1) {
			nonbiconnected_.insert_edge(b->edges().begin()->second);
		} else {
			SplitComponents s(*b);
#ifdef DEBUG
			s.print_dot("scomps" + to_string(++sccount) + ".dot");
#endif
			s.merge_to_SPQR();
			s.init_twist(suppress_edge_permutations);

			tcc.push_back(s);
#ifdef DEBUG
			s.print_dot("spqr" + to_string(sccount) + ".dot");
#endif
		}
	}
	LOGXX("found " << tcc.size() << " triconnected components");
	comps_.assign(tcc.begin(), tcc.end());
}

bool TriConnectedComponents::set_next_twist() {
	for (size_t c = 0; c < comps_.size(); ++c) {
		LOGXX("  next twist in component number " << c);
		if (comps_[c].set_next_twist())
			return true;
	}
	return false;
}

Topology TriConnectedComponents::merge_to_graph() const {
	Topology res(nonbiconnected_);
	for (size_t c = 0; c < comps_.size(); ++c) {
		Topology t = comps_[c].merge_to_graph();
		map<int, Edge>::const_iterator e;
		for (e = t.edges().begin(); e != t.edges().end(); ++e)
			res.insert_edge(e->second);
	}
	return res;
}

set<int> TriConnectedComponents::flipped_edges() const {
	set<int> res;
	for (size_t c = 0; c < comps_.size(); ++c) {
		set<int> ids = comps_[c].flipped_edges();
		res.insert(ids.begin(), ids.end());
	}
	return res;
}

/* note: there might be room for improvements,
 * a) restrict to non-equivalent twists for bonds
 * b) more ambitious: use colored GI, see e.g. GMI algorithm of
 *    Raghavendra Rao B.V., Jayalal Sarma, M.N. (2009) */
list<pair<Topology, set<int> > > //
TriConnectedComponents::find_topologies_of_matroid() {
	// iterate over all possible twists of underlying graph
	list<pair<Topology, set<int> > > res;
	int num_graphs = 0;
	do {
		++num_graphs;
		LOGXX("\n*** twisting option number " << num_graphs << ":");
		Topology topo(merge_to_graph());
		topo.set_name("twisted" + to_string(num_graphs));
#ifdef DEBUG
		topo.print_dot("twisted" + to_string(num_graphs) + ".dot");
		for (size_t c = 0; c < comps_.size(); ++c)
		comps_[c].print_dot("twisted" + to_string(num_graphs) + "comps" + to_string(c) + ".dot");
#endif
		res.push_back(make_pair(topo, flipped_edges()));
	} while (set_next_twist());
	LOGXX("found " << res.size() << " twisted graphs");
	return res;
}

// SplitComponents


/// sorts edges such that multiple edges come after each other
/* note: throws away separated nodes (without edges)
 * This could be linear time if implemented directly in Topology
 * (swapping list<Edge>) provided we have "almost continuous" indices.
 */
Topology sort_edges_for_nodes(const Topology& t) {
	// sort edges first for their minimal node id, then for maximal node id
	VERIFY(t.num_nodes() > 0);
	int max_node_id = *t.nodes().rbegin();
	vector<list<Edge> > es(max_node_id + 1); // edge bucket for sorting
	map<int, list<Edge> >::const_iterator p;
	for (p = t.edges_of_nodes().begin(); p != t.edges_of_nodes().end(); ++p) {
		list<Edge>::const_iterator e;
		for (e = p->second.begin(); e != p->second.end(); ++e)
			if (e->opposite(p->first) >= p->first) // each edge only once
				es[e->opposite(p->first)].push_back(*e);
	}
	Topology tsorted;
	for (size_t i = 0; i < es.size(); ++i)
		for (list<Edge>::const_iterator e = es[i].begin(); e != es[i].end(); ++e)
			tsorted.insert_edge(*e);
	return tsorted;
}

Topology split_off_multiple_edges(const Topology& t, list<list<Edge> >& comps,
		int& max_edge_id) {
	Topology ts = sort_edges_for_nodes(t); // multi-edges become neigbors in list
	Topology ret(t); // keep original order of edge (no sort)
	map<int, list<Edge> >::const_iterator p;
	for (p = ts.edges_of_nodes().begin(); p != ts.edges_of_nodes().end(); ++p) {
		list<Edge>::const_iterator e;
		for (e = p->second.begin(); e != p->second.end();) {
			// collect chunk of edges connecting same nodes
			list<Edge> es;
			while (e != p->second.end() && (es.empty() || //
					(e->compare_undirected(es.back()) == 0))) {
				es.push_back(*e++);
			}
			if (es.size() > 1) { // multiple edge
				LOGNXX("splitting off multiple edges: ");
				list<Edge>::const_iterator me;
				for (me = es.begin(); me != es.end(); ++me) {
					LOGNXX(*me << " ");
					ret.remove_edge(me->id);
					ts.remove_edge(me->id);
				}
				LOGXX("");
				Edge ve(es.front().from, es.front().to, ++max_edge_id);
				ret.insert_edge(ve);
				ts.insert_edge(ve);
				es.push_back(ve);
				comps.push_back(es);
			}
		}
	}
	return ret;
}

SplitComponents::SplitComponents(const Topology& t) :
	are_original_edges_indistinguishable_(false) {
	VERIFY(t.num_nodes() > 0 && t.num_edges() > 1);
	VERIFY(t.num_nodes() > 0 && *t.nodes().begin() >= 0 /* min node id */);
	int max_node_id = *t.nodes().rbegin();
	int max_edge_id = t.edges().rbegin()->first;
	int max_orig_edge_id = t.edges().rbegin()->first;
	list<list<Edge> > cl;
	Topology tp = split_off_multiple_edges(t, cl, max_edge_id);
#ifdef DEBUG
	LOGXX("Finding split components of topology:\n" << tp);
	tp.print_dot("dfsinput.dot");
#endif
	size_t num_split_off = cl.size(); // num of comps not generated by DFSPalmTree
	int root = *tp.nodes().begin();
	if (tp.num_edges() >= 2 && tp.num_nodes() >= 2) {
		// true biconnected component left at this point
		DFSPalmTree dfstree(tp, root, max_node_id, max_edge_id,
				max_orig_edge_id);
		list<list<Edge> > dfsres = dfstree.find_split_components();
		cl.splice(cl.end(), dfsres);
		max_edge_id = dfstree.max_edge_id();
	} else {
		++num_split_off;
		list<Edge> el;
		const map<int, Edge>& em = tp.edges();
		for (map<int, Edge>::const_iterator e = em.begin(); e != em.end(); ++e)
			el.push_back(e->second);
		cl.push_back(el);
	}
	comps_ = vector<list<Edge> > (cl.begin(), cl.end());
	first_virtual_edge_id_ = max_orig_edge_id + 1;
	comp_type_.resize(comps_.size());
	last_virtual_edge_id_ = max_edge_id;
	vector<list<int> > comps_of_edge(last_virtual_edge_id_ + 1);
	for (size_t c = 0; c < comps_.size(); ++c) {
		ComponentType type;
		if (c < num_split_off) {
			// multiple edges which were split off or
			// single virtual edge (if graph is a pure multi-edge)
			type = Bond;
		} else if (comps_[c].size() >= 4) {
			type = TriComponent;
		} else if (comps_[c].size() == 3) {
			list<Edge>::const_iterator e1 = comps_[c].begin();
			list<Edge>::const_iterator e2 = ++comps_[c].begin();
			type = e1->compare_undirected(*e2) == 0 ? Bond : Polygon;
		} else {
			ABORT("unknown component type (is graph really biconnected ?)");
		}
		comp_type_[c] = type;
		list<Edge>::const_iterator e;
		for (e = comps_[c].begin(); e != comps_[c].end(); ++e)
			comps_of_edge[e->id].push_back(c);
	}
	for (size_t eid = 0; eid < comps_of_edge.size(); ++eid) {
		ASSERT(comps_of_edge[eid].size() <= 2);
		if (comps_of_edge[eid].size() != 2)
			continue;
		int a = *comps_of_edge[eid].begin();
		int b = *++comps_of_edge[eid].begin();
		tree_.insert_edge(Edge(a, b, eid));
	}
	LOGXX("found " << comps_.size() << " split components");
	name_ = "splitc";
}

void SplitComponents::merge(int type_mask) {
	list<Edge> edges_to_keep;
	set<int> nodes_to_keep; // needed for single node (component) tree
	// merge components
	vector<int> cmap(comps_.size());
	for (size_t c = 0; c < comps_.size(); ++c)
		cmap[c] = c;
	const map<int, Edge>& es = tree_.edges();
	for (map<int, Edge>::const_iterator ee = es.begin(); ee != es.end(); ++ee) {
		const Edge& e = ee->second;
		if ((comp_type_[e.from] & type_mask) && (comp_type_[e.to] & type_mask)) {
			int keep = cmap[min(e.from, e.to)];
			int del = cmap[max(e.from, e.to)];
			LOGXX("-> merging component " << del << " into " << keep
					<< ", dropping virtual edge " << e.id);
			cmap[del] = keep;
			comps_[keep].splice(comps_[keep].begin(), comps_[del]);
			nodes_to_keep.insert(keep);
		} else {
			edges_to_keep.push_back(e);
		}
	}
	// setup new tree
	tree_ = Topology(); // clear
	list<Edge>::const_iterator e;
	for (e = edges_to_keep.begin(); e != edges_to_keep.end(); ++e)
		if (cmap[e->from] != cmap[e->to])
			tree_.insert_edge(Edge(cmap[e->from], cmap[e->to], e->id));
	for (set<int>::const_iterator n = nodes_to_keep.begin(); n
			!= nodes_to_keep.end(); ++n)
		tree_.insert_node(*n);
	// remove duplicate virtual edges within each (merged) component
	vector<list<pair<size_t, list<Edge>::iterator> > > last(
			last_virtual_edge_id_ + 1);
	for (size_t c = 0; c < comps_.size(); ++c) {
		list<Edge>::iterator e;
		for (e = comps_[c].begin(); e != comps_[c].end();) {
			int eid = e->id;
			if (!last[eid].empty() && last[eid].back().first == c) {
				comps_[c].erase(last[eid].back().second);
				comps_[c].erase(e++);
				last[eid].clear();
			} else {
				last[eid].push_back(make_pair(c, e++));
			}
		}
	}
}

void SplitComponents::merge_to_SPQR() {
	merge(Bond);
	merge(Polygon);
	name_ = "tricc";
}

bool SplitComponents::is_virtual_edge(int edge_id) const {
	return edge_id >= first_virtual_edge_id_;
}

Topology SplitComponents::merge_to_graph() const {
	Topology t;
	for (size_t c = 0; c < comps_.size(); ++c)
		for (list<Edge>::const_iterator e = comps_[c].begin(); e
				!= comps_[c].end(); ++e)
			if (!is_virtual_edge(e->id))
				t.insert_edge(Edge(e->from, e->to, e->id));
	return t;
}

set<int> SplitComponents::flipped_edges() const {
	set<int> ids;
	map<int, bool>::const_iterator i;
	for (i = edge_orientation_.begin(); i != edge_orientation_.end(); ++i)
		if (i->second)
			ids.insert(i->first);
	return ids;
}

Topology SplitComponents::convert_to_topology(const list<Edge>& edges) const {
	Topology t;
	for (list<Edge>::const_iterator e = edges.begin(); e != edges.end(); ++e)
		t.insert_edge(*e);
	return t;
}

void SplitComponents::perform_twist(int vid, const Edge& te) {
	// perform twist at virtual edge with id eid on rhs subtree
	Topology rhs_t = tree_.connected_component(te.to, vid);
	Edge ve;
	list<Edge>::iterator e;
	for (e = comps_[te.to].begin(); e != comps_[te.to].end(); ++e)
		if (e->id == vid)
			ve = *e;
	VERIFY(ve.id == vid);
	LOGXX("  Tutte twisting at (" << ve.from << "," << ve.to << ";" << vid
			<< ") between components [" << te.from << "," << te.to << "]");
	set<int>::const_iterator c;
	for (c = rhs_t.nodes().begin(); c != rhs_t.nodes().end(); ++c) {
		LOGXX("    swapping nodes (" << ve.from << "," << ve.to << ") in comp "
				<< *c);
		for (e = comps_[*c].begin(); e != comps_[*c].end(); ++e) {
			if (e->id == vid)
				continue; // keep both same orientation for both edges with vid
			if (e->from == ve.from)
				e->from = ve.to;
			else if (e->from == ve.to)
				e->from = ve.from;
			if (e->to == ve.from)
				e->to = ve.to;
			else if (e->to == ve.to)
				e->to = ve.from;
			edge_orientation_[e->id] = !edge_orientation_[e->id];
			//e->reverse();
		}
	} // loop over components to perform twisting at one virtual edge
}

void SplitComponents::init_twist(bool are_edges_equiv) {
	are_original_edges_indistinguishable_ = are_edges_equiv;
	// first: init polygon edge combination
	edge_pos_.resize(comps_.size());
	nodes_.resize(comps_.size());
	num_vedges_.resize(comps_.size());
	edge_orientation_.clear();
	for (size_t c = 0; c < comps_.size(); ++c) {
		// find out old edge positions
		num_vedges_[c] = 0;
		if (comp_type_[c] != Polygon || comps_[c].size() < 3)
			continue;
		map<int, list<Edge> > adj; // edges of node
		list<Edge> edges;
		list<Edge>::const_iterator e;
		for (e = comps_[c].begin(); e != comps_[c].end(); ++e) {
			adj[e->from].push_back(*e);
			adj[e->to].push_back(*e);
			edges.push_back(*e);
			if (is_virtual_edge(e->id))
				++num_vedges_[c];
		}
		if (are_original_edges_indistinguishable_ && num_vedges_[c] < 2)
			continue;
		edges.sort();
		edges.reverse();
		// follow edges once around the cycle
		nodes_[c].reserve(comps_[c].size()); // store old node order
		map<int, int> pos;
		map<int, bool> initial_flip;
		Edge ee = edges.front(); // keep original orientation for this one
		for (size_t i = 0; i < comps_[c].size(); ++i) {
			pos[ee.id] = i;
			nodes_[c].push_back(ee.from);
			int n = ee.to;
			ASSERT(adj[n].size() == 2);
			ee = (adj[n].front().id == ee.id ? adj[n].back() : adj[n].front()); //next
			// we use a canonical orientation of edges inside polygon;
			// orientation of virtual edge _pair_ is arbitrary, but for
			// non-virtual edges inside polygon we must remember potential flip
			if (ee.from != n) {
				ee.reverse();
				initial_flip[ee.id] = true;
			}
		}
		ASSERT(ee == edges.front()); // with original orientation
		ASSERT(nodes_[c].size() == comps_[c].size());
		edge_pos_[c].reserve(comps_[c].size()); // store old edge order
		for (e = edges.begin(); e != edges.end(); ++e)
			edge_pos_[c].push_back(pos[e->id]);
		ASSERT(edge_pos_[c].size() == comps_[c].size());
		// set new edge_pos
		vector<int> new_edge_pos(comps_[c].size());
		for (size_t i = 0; i < new_edge_pos.size(); ++i)
			new_edge_pos[i] = i;
		int* off = &new_edge_pos[0];
		size_t num_pick = new_edge_pos.size(); // iterate over all edge permutations
		if (are_original_edges_indistinguishable_)
			num_pick = num_vedges_[c]; // iterate only over different virtual edge positions
		ASSERT(num_pick >= 1 && new_edge_pos.size() >= num_pick);
		int* b = off + 1; // begin of (virtual) edge positions to permute (1st edge fixed)
		int* m = off + num_pick; // begin of indistinguishable edge positions
		int* l = off + new_edge_pos.size(); // end of all edge positions
		benbear::adjust_combination(b, m, l);
		update_polygon_edge_combination(c, new_edge_pos, initial_flip);
	}

	// second: init Tutte twist counters
	// (don't put into separate method since edge_orientation_ setup needed)
	twist_.clear();
	const map<int, Edge>& tes = tree_.edges(); // tree edges
	for (map<int, Edge>::const_iterator e = tes.begin(); e != tes.end(); ++e)
		twist_[e->first] = false;
}

bool SplitComponents::set_next_polygon_edge_combination() {
	for (size_t c = 0; c < comps_.size(); ++c) {
		if (comp_type_[c] != Polygon || //
				(are_original_edges_indistinguishable_ && num_vedges_[c] < 2) || //
				comps_[c].size() < 3)
			continue;
		LOGXX("  next polygon edge combination");
		vector<int> new_edge_pos(edge_pos_[c]);
		// we keep first virtual edge fixed at position 0 by convention
		int* off = &new_edge_pos[0];
		size_t num_pick = new_edge_pos.size(); // iterate over all edge permutations
		if (are_original_edges_indistinguishable_)
			num_pick = num_vedges_[c]; // iterate only over different virtual edge positions
		ASSERT(num_pick >= 1 && new_edge_pos.size() >= num_pick);
		int* b = off + 1; // begin of (virtual) edge positions to permute (1st edge fixed)
		int* m = off + num_pick; // begin of indistinguishable edge positions
		int* l = off + new_edge_pos.size(); // end of all edge positions
		// permute virtual edges among themselves
		bool done = next_permutation(b, m);
		if (!done) {
			// permute virtual with real edges
			//benbear::adjust_combination(b, m, l);
			done = benbear::next_combination(b, m, l);
		}
		update_polygon_edge_combination(c, new_edge_pos, map<int, bool> ());
		if (done)
			return true;
	}
	return false;
}

void SplitComponents::update_polygon_edge_combination(int c /* component */,
		const vector<int>& new_edge_pos, const map<int, bool>& initial_flip) {
	if ((are_original_edges_indistinguishable_ && num_vedges_[c] < 2)
			|| comps_[c].size() < 3)
		return;
	ASSERT(edge_pos_[c][0] == new_edge_pos[0]); // highest virtual edge fixed
	map<int, Edge> edges;
	list<Edge>::iterator e;
	for (e = comps_[c].begin(); e != comps_[c].end(); ++e)
		edges[e->id] = *e;

	// since all inside-polygon twists can be generated by neighbor edge twists,
	// it is possible to keep the original node order

	// modify edges to reflect new_edge_pos_

	map<int, map<int, int> > relabel; // node relabeling for each virt. edge
	map<int, bool> flip(initial_flip); // whether edge in polygon has been flipped
	size_t count = 0;
	// assign edges starting from highest id, i.e. with virtual edges
	map<int, Edge>::reverse_iterator ep;
	for (ep = edges.rbegin(); ep != edges.rend(); ++ep, ++count) {
		size_t i1 = new_edge_pos[count];
		size_t i2 = (i1 == nodes_[c].size() - 1 ? 0 : i1 + 1);
		Edge oe(ep->second); // old edge
		Edge ne(nodes_[c][i1], nodes_[c][i2], ep->first); // new edge
		if (is_virtual_edge(ep->first)) {
			relabel[ep->first][oe.from] = ne.from;
			relabel[ep->first][oe.to] = ne.to;
		}
		// if position of edge changed by an odd number, edge was reversed
		// ((new_edge_pos[count] - edge_pos_[c][count]) % 2 != 0)
		if (flip[ep->first]) {
			edge_orientation_[ep->first] = !edge_orientation_[ep->first];
			//ne.reverse();
		}
		ep->second = ne;
	}

	// update current polygon
	comps_[c].clear();
	for (ep = edges.rbegin(); ep != edges.rend(); ++ep)
		comps_[c].push_back(ep->second);

	// update node ids in other components:
	list<Edge> tedges = tree_.edges_of_node(c);
	list<Edge>::const_iterator te;
	for (te = tedges.begin(); te != tedges.end(); ++te) { // neighbor comps
		int neighbor = te->opposite(c);
		// relabel nodes in components according to the virtual edge
		// which connects them to current component
		Topology subtree = tree_.connected_component(neighbor, te->id);
		set<int>::const_iterator n;
		for (n = subtree.nodes().begin(); n != subtree.nodes().end(); ++n) {
			ASSERT(relabel.find(te->id) != relabel.end());
			const map<int, int>& rel = relabel[te->id];
			for (e = comps_[*n].begin(); e != comps_[*n].end(); ++e) {
				map<int, int>::const_iterator r;
				if ((r = rel.find(e->from)) != rel.end())
					e->from = r->second;
				if ((r = rel.find(e->to)) != rel.end())
					e->to = r->second;
				if (flip[te->id] /*&& !is_virtual_edge(te->id)*/)
					edge_orientation_[e->id] = !edge_orientation_[e->id];
				//e->reverse();
			}
			for (size_t i = 0; i < nodes_[*n].size(); ++i)
				if (rel.find(nodes_[*n][i]) != rel.end())
					nodes_[*n][i] = rel.find(nodes_[*n][i])->second;
		}
	}
	edge_pos_[c] = new_edge_pos;
}

bool SplitComponents::set_next_Tutte_twist() {
	const map<int, Edge>& tes = tree_.edges();
	for (map<int, Edge>::const_iterator e = tes.begin(); e != tes.end(); ++e) {
		perform_twist(e->first, e->second);
		if ((twist_[e->first] = !twist_[e->first]))
			return true;
	}
	return false;
}

bool SplitComponents::set_next_twist() {
	return set_next_Tutte_twist() || set_next_polygon_edge_combination();
}

void SplitComponents::print_dot(const string& filename) const {
	ofstream of(filename.c_str());
	of << "graph " << name_ << " {\n";
	int off = 0;
	for (size_t c = 0; c < comps_.size(); ++c) {
		if (comps_[c].empty())
			continue;
		map<int, int> nodes;
		of << "  subgraph cluster" << c << " {\n";
		int max_node_id = 0;
		list<Edge>::const_iterator e;
		for (e = comps_[c].begin(); e != comps_[c].end(); ++e) {
			of << "    " << e->from + off << " -- " << e->to + off//
					<< " [label=\"" << e->id;
			if (e == comps_[c].begin())
				of << "in" << c;
			of << "\",fontsize=10"//
					<< (is_virtual_edge(e->id) ? ",style=dotted" : "")//
					<< "];\n";
			//<< ";\n";
			nodes[e->from + off] = e->from;
			nodes[e->to + off] = e->to;
			max_node_id = max(max_node_id, e->from + off);
			max_node_id = max(max_node_id, e->to + off);
		}
		// nodes
		for (map<int, int>::const_iterator n = nodes.begin(); n != nodes.end(); ++n)
			of << "    " << n->first << " [label=\"" << n->second << "\"];\n";
		of << "  }\n";
		off = max_node_id + 1;
	}
	map<int, Edge> es = tree_.edges();
	for (map<int, Edge>::const_iterator ee = es.begin(); ee != es.end(); ++ee) {
		const Edge& e = ee->second;
		of << "  cluster" << e.from << " -- " << "cluster" << e.to//
				<< " [label=\"" << e.id << "\"];\n";
		//<< "\n";
	}
	of << "}\n";
	of.close();
}

void SplitComponents::print_postscript(const string& filename) const {
	string dotfilename = filename + ".dot";
	print_dot(dotfilename);
	stringstream ss;
	// intended for fdp, but this crashes too often in version 2.20
	ss << "dot -Tps " << dotfilename << " -o " << filename;
	//ss << "neato -Tps " << dotfilename << " -o " << filename;
	//ss << "fdp -Tps " << dotfilename << " -o " << filename;
	int ret = system(ss.str().c_str());
	VERIFY(ret == 0);
}

} // namespace Reduze
