/*  job_selectreductions.h
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#ifndef JOB_SELECTREDUCTIONS_H_
#define JOB_SELECTREDUCTIONS_H_

#include "job.h"
#include <ginac/ginac.h>

namespace Reduze {

class IntegralFamily;

/// Job to select reduction equations
class SelectReductions: public Job {
public:
	static YAMLSpec yaml_spec() {
		YAMLSpec s;
		s.set_keyword("select_reductions");
		s.set_short_description("Selects reductions for integrals.");
		s.set_long_description(""//
					"Job to select reductions for given list of integrals."
					" Existing reduction results are searched in order to"
					" reduce the chosen integrals. "
					" The chosen integrals may be in non-minimal form"
					" or related by a crossing to existing reductions --"
					" the algorithm employs known permutation symmetries,"
					" shift symmetries and crossing relations to obtain"
					" the solutions."
					" If shift symmetries are applied, the output"
					" may be postprocessed by a job reduce_files in order"
					" to obtain fully reduced solutions.");
		s.add_option("input_file", true, "string", ""//
					"Input file name. The file must contain a list of integrals"
					" in Reduze format.");
		s.add_option("output_file", true, "string", "Output file name.");
		s.add_option("notfound_file", false, "string", ""//
					"Output file name. Selected integrals for which no solution"
					" was found will be written to this file. Empty string disables"
					" this feature.");
		s.add_option("preferred_masters_file", false, "string",
					"Input file with preferred master integrals.");
		s.add_option("generate_crossed", false, "boolean", ""//
					"Whether to generate solutions for all possible crossings"
					" of the input integrals. Useful for basis changes.");
		s.add_options(Job::yaml_spec());
		return s;
	}
	virtual YAMLSpec yaml_spec_link() const {
		return yaml_spec();
	}
	SelectReductions() :
		Job(), generate_crossed_(false) {
		add_auto_options();
	}
	virtual ~SelectReductions() {
	}
	virtual void run_serial();
	virtual bool find_dependencies(const std::set<std::string>&,//
			std::list<std::string>&, std::list<std::string>&, std::list<Job*>&);
	virtual std::string get_description() const;

protected:
	virtual void add_auto_options() {
		add_auto_io("input_file", input_filename_);
		add_auto_io("output_file", output_filename_);
		add_auto_io("notfound_file", notfound_file_);
		add_auto_io("preferred_masters_file", preferred_masters_file_);
		add_auto_io("generate_crossed", generate_crossed_);
	}
	virtual void init();

private:
	/// input file containing requested integrals
	std::string input_filename_;
	/// output file (Reduze format) containing the reductions
	std::string output_filename_;
	/// selected integrals without solutions will be written here
	std::string notfound_file_;
	/// preferred master integrals
	std::string preferred_masters_file_;
	/// whether input integrals should be supplemented by all crossed versions
	bool generate_crossed_;
	// integral families which differ only in the external momenta of the propagators
	// such that an integral i1 of int.cat. A is equal to an integral i2 of int.cat B
	// after applying a map m and vice versa: i2 = i1/.m and i1 = i2/.m
	// map<A,(B,m)>: results for int.cat A are taken from the results of int.cat. B
	//	typedef std::map<const IntegralFamily*, std::pair<
	//			const IntegralFamily*, GiNaC::exmap> > IC_Map_pair_by_IC;
	//	IC_Map_pair_by_IC identify_;
	// the same as identify_ but with the exmaps where a symbol (user defined) is replaced by one
	//	IC_Map_pair_by_IC identify_with_symbol_replaced_by_one_;
};

}

#endif /* JOB_SELECTREDUCTIONS_H_ */
