/*  job_runreduction.h
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#ifndef JOB_RUNREDUCTION_H_
#define JOB_RUNREDUCTION_H_

#include "job.h"
#include "reduzer.h"
#include "rspoint.h"

namespace Reduze {

/// Job to run a reduction (low level version)
class RunReduction: public DistributedJob {
public:
	static YAMLSpec yaml_spec() {
		YAMLSpec s;
		s.set_keyword("run_reduction");
		s.set_short_description("Low-level job to run a reduction.");
		s.set_long_description(""//
					"Low-level job to run a reduction. It should typically"
					" not be necessary to use this job directly. Instead,"
					" please see the jobs reduce_files and reduce_sectors.");

		// if description or defaults change, please check them also in ReduceSectors
		s.add_option("sector", false, "sequence", "A sector");
		s.add_option("tmp_directory", true, "string",
				"Temporary directory where database environment is created");
		s.add_option("identity_filenames", true, "sequence of strings",
				"Input files with identities");
		s.add_option("subst_filenames", true, "sequence of strings",
				"Input files with substitution identities");
		s.add_option("output_file", true, "string", "Output file");
		s.add_option("output_file_external", true, "string", "Output file for subsector identities.");
		s.add_option("solutions", false, "reduction_generic_options", //
				"Options and generic selections of integrals");
		s.add_option("reduzer_options", false, "reduzer_options", //
				"Options for the reduction algorithm.");
		s.add_option("preferred_masters_file", false, "string", //
				"A file with integrals which should be used as"
					" master integrals");
		s.add_option("delete_files", false, "sequence of strings", //
				"List of names of files which will be deleted after a"
					" successful reduction run or after a valid"
					" and reusable database has been setup");
		s.add_option("delete_directories", false, "sequence of strings", //
				"List of names of directories which will be deleted"
					" after a successful reduction run or after a valid"
					" and reusable database has been setup"
					" If a directory is not empty it will not be deleted.");
		s.add_option("subst_subsectors", false, "boolean", //
				"Whether to load subsector reductions for substitutions."
				" For this option to work a sector must be specified.");
		s.add_options(DistributedJob::yaml_spec());
		return s;
	}
	virtual YAMLSpec yaml_spec_link() const {
		return yaml_spec();
	}

public:
	// standard constructor
	RunReduction();
	// constructor for reductions of sectors
	RunReduction(
			const Sector& sector, //
			const std::string& tmp_directory,
			const std::list<std::string>& identity_filenames,//
			const std::list<std::string>& subst_filenames,//
			const std::string& output_file, //
			const std::string& output_file_external, //
			const ReductionGenericOptions& reduction_generic_options, //
			const ReduzerOptions& reduzer_options, //
			const std::string preferred_masters_file, //
			const std::list<std::string>& delete_files, //
			const std::list<std::string>& delete_directories);
	// constructor for reduction of files
	RunReduction(const std::string& tmp_directory, //
			const std::list<std::string>& identity_filenames, //
			const std::list<std::string>& subst_filenames,//
			const std::string& output_file, //
			const ReduzerOptions& reduzer_options, //
			const std::string preferred_masters_file, //
			const std::list<std::string>& delete_files,//
			const std::list<std::string>& delete_directories);
	virtual ~RunReduction();

	virtual void run_serial();
#ifdef HAVE_MPI
	virtual void run_manager(MPI::Intracomm* comm, int jobcenter_rank);
	virtual void run_worker(MPI::Intracomm* comm, int manager_rank);
#endif // HAVE_MPI
	virtual bool find_dependencies(const std::set<std::string>& outothers, //
			std::list<std::string>& in, //
			std::list<std::string>& out, //
			std::list<Job*>& auxjobs);
	virtual std::string get_description() const;

	/// returns the database file name without directory "database.db"
	static std::string db_file_name();
	/// returns the directory name of the database "dbhome/" or "dbhome_tr_off/"
	static std::string db_dir_name(bool use_transactions);
	/// whether the reduction can be resumed from the database
	static bool resume_reduction(const std::string& tmp_directory,
			bool use_transactions, bool conditional);

protected:
	virtual void add_auto_options() {
		add_auto_io("tmp_directory", tmp_directory_);
		add_auto_io("identity_filenames", identity_filenames_);
		add_auto_io("subst_filenames", subst_filenames_);
		add_auto_io("output_file", output_file_);
		add_auto_io("output_file_external", output_file_external_);
		add_auto_io("solutions", reduction_generic_options_);
		add_auto_io("reduzer_options", reduzer_options_);
		add_auto_io("preferred_masters_file", preferred_masters_file_);
		add_auto_io("delete_files", delete_files_);
		add_auto_io("delete_directories", delete_directories_);
		add_auto_io("subst_subsectors", subst_subsectors_);
	}
	virtual void read_manual_options(const YAML::Node&);
	virtual void print_manual_options(YAML::Emitter& os) const;

private:
	/// returns the full path of the database directory
	std::string db_dir() const;
	/// removes invalid databases in the directory tmp_directory assuming the current run has resume_reduction yes/no
	void remove_invalid_database(const std::string& tmp_directory,
			bool resume_reduction);
	/// some preparation before setting up the database
	/** determine the integrals in the files used for setting up
	 * the database.
	 * add the output_file to the identities (previous reductions) if needed and available
	 * remove substitutions which are not needed
	 * extends subst_filenames_ to include subsector substitutions as needed
	 */
	void find_integrals_and_substitutions(std::set<INT>& all_ints, unord_set_INT& identity_ints,//
			std::set<INT>& allowed_ints, std::set<INT>& helper_ints, //
			std::set<INT>& critical_ints, std::set<INT>& discard_ints);
	virtual void compute(ReduzerMem* r);
#ifdef HAVE_DATABASE
	virtual void compute(ReduzerDb* r);
#endif // HAVE_DATABASE
private:
	/// one or zero element list of sectors
	std::list<Sector> sector_;
	/// a unique temporary directory, where the database directory and the database file will be created
	std::string tmp_directory_;

	/// file names of the identities to be inserted
	std::list<std::string> identity_filenames_;
	/// substitutions for which this job waits before it gets executed
	/** if a sector is specified then sub-sector results are used anyway, if available and needed **/
	std::list<std::string> subst_filenames_;

	/// output file for the reductions in Reduze format
	std::string output_file_;
	/// output file for subsector reductions in Reduze format
	std::string output_file_external_;

	ReductionGenericOptions reduction_generic_options_;
	/// options for the Reduzer
	ReduzerOptions reduzer_options_;

	/// file with preferred master integrals
	std::string preferred_masters_file_;
	/// list of files which can be deleted after the setup of the database
	std::list<std::string> delete_files_;
	/// list of directories which can be deleted after the setup of the database
	std::list<std::string> delete_directories_;

public:
	/// whether to load subsector identities for substitutions
	bool subst_subsectors_;
};

}

#endif /* JOB_RUNREDUCTION_H_ */
