/*  job_reducesectors.cpp
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#include "job_reducesectors.h"
#include "job_setupsectormappings.h"
#include "job_generateseeds.h"
#include "job_generateidentities.h"
#include "job_runreduction.h"
#include "job_printreductioninfo.h"
#include "job_applycrossing.h"
#include "job_distributeexternal.h"
#include "functions.h"
#include "yamlutils.h"
#include "files.h"
#include "kinematics.h"
#include "integralfamily.h"
#include "sectormappings.h"
#include "equationlist.h"

using namespace std;

namespace Reduze {

// register job types at JobFactory
namespace {
JobProxy<ReduceSectors> dummy;
}

void ReduceSectors::init() {
#ifdef HAVE_MPI
	if (min_workers_ < 1) {
		ERROR("min_workers must be 1 or higher");
	}
	if (max_workers_ < min_workers_) {
		ERROR("max_workers must be equal to min_workers or higher");
	}
#endif
	if (num_seeds_for_identities_auxjobs_ < 1)
		ERROR("Value of option \"num_seeds_for_identities_auxjobs\" must"
				<< " be 1 or higher.");
}

void ReduceSectors::run_serial() {
}

bool ReduceSectors::find_dependencies(const std::set<std::string>& outothers,
		std::list<std::string>& in, std::list<std::string>& out,
		std::list<Job*>& auxjobs) {
	set<string> outall = outothers;

	if (!preferred_masters_file_.empty())
		in.push_back(preferred_masters_file_);

	//find_dependencies_all_sectormappings(outothers, in, auxjobs);
	set<Sector> secs;
	try {
		secs = sector_selection_.find_sectors();
	} catch (exception& e) {
		LOGX("can't determine ReduceSectors::find_dependencies() yet:\n" << e.what());
		return false;
	}

	Files* files = Files::instance();

	// perform reductions only for uncrossed sectors and calculate crossed
	// results from the uncrossed

	map<Sector, set<Sector> > sectors; // sectors with crossed sectors
	for (set<Sector>::const_iterator se = secs.begin(); se != secs.end(); ++se) {
		const Sector snox = Crossing::uncross(INT(*se)).first.get_sector();
		if (*se != snox)
			sectors[snox].insert(*se);
		else
			sectors[snox];
	}

	SeedGeneratorOptions seed_options;
	IdentityGeneratorOptions id_options;
	id_options.set_subsectors_to_zero(
			reduction_generic_options_.set_subsectors_to_zero_);
	id_options.set_map_on_coeff(reduction_generic_options_.map_on_coeff_);

	for (map<Sector, set<Sector> >::const_iterator se = sectors.begin(); se
			!= sectors.end(); ++se) {
		const Sector& sector = se->first;
		const set<Sector>& x_sectors = se->second;
		const string sec_string = sector.get_safe_string_for_filename();
		const IntegralFamily* intfam = sector.integralfamily();
		const Kinematics* kin = intfam->kinematics();
		const int num_props = intfam->num_propagators();
		const int num_t = sector.t();
		list<string> identity_filenames;
		string altres = files->get_filename_sectorreduction(sector,
				alternative_input_directory_);
		if (alternative_input_directory_ != "" && is_readable_file(altres)) {
			LOG("using file " << altres << " from alternative input directory");
			identity_filenames.push_back(altres);
		}
		string output_file = files->get_filename_sectorreduction(sector);
		string output_file_external = files->get_filename_sectorexternal(sector);

		const string tmp_sectordir = this->get_tmp_dir(sector);
		const string tmp_identities_dir = this->get_tmp_dir_identities(sector);
		const string tmp_seeds_dir = this->get_tmp_dir_seeds(sector);

#ifdef HAVE_DATABASE
		bool resume_reduction = RunReduction::resume_reduction(tmp_sectordir,
				reduzer_options_.use_transactions, is_conditional());
#else // HAVE_DATABASE
		bool resume_reduction = false;
#endif // HAVE_DATABASE
		LOGX("Resume reduction:       " << (resume_reduction ? "true" : "false") << " for sector " << sector);
		bool output_readable = is_readable_file(output_file);
		LOGX("Output file readable:   " << (output_readable ? "true" : "false") << " for sector " << sector);
		bool create_identities = !is_conditional() || (!output_readable
				&& !resume_reduction);
		LOGX("Create identities jobs: " << (create_identities ? "true" : "false") << " for sector " << sector);

		const map<string, list<RSFiniteGenericSelection> >& identities =
				identities_.identities();
		list<string> delete_files, delete_directories;
		map<string, list<RSFiniteGenericSelection> >::const_iterator i;
		for (i = identities.begin(); i != identities.end(); ++i) {
			string type = i->first;
			const list<RSFiniteGenericSelection>& ranges = i->second;
			long int num_seeds = 0;
			list<RSFiniteGenericSelection>::const_iterator it;
			for (it = ranges.begin(); it != ranges.end(); ++it) {
				long int add_seeds = it->num_integrals(num_props, num_t);
				if (add_seeds == 0) {
					YAML::Emitter ye;
					it->print(ye);
					WARNING("No seed integrals for identities of sector "
							<< sector << " in range " << ye.c_str());
				}
				num_seeds += add_seeds;
			}
			if (type == "ibp_lee" && num_props <= 1) {
				LOGX("Skipping IBP-Lee identities since number of loop <= 1");
				continue;
			} else if (type == "lorentz" && kin->independent_external_momenta()
					<= 1) {
				LOGX("Skipping Lorentz Invariant Identities since number of independent external momenta <= 1");
				continue;
			} else if (num_seeds == 0) {
				continue;
			}
			string seed_output = tmp_seeds_dir + "seeds_" + sec_string + "_"
					+ type;
			ASSERT(num_seeds > 0);
			ASSERT(num_seeds_for_identities_auxjobs_ > 0);
			int num_id_jobs = num_seeds / num_seeds_for_identities_auxjobs_;
			if (num_seeds % num_seeds_for_identities_auxjobs_)
				num_id_jobs += 1;

			if (create_identities) {
				ASSERT(num_id_jobs >= 1);
				Job* seed_job = new GenerateSeeds(sector, ranges, seed_options,
						seed_output);
				seed_job->set_conditional(is_conditional());
				auxjobs.push_back(seed_job);
				outall.insert(get_canonical_filename(seed_output));
				delete_files.push_back(seed_output);
				int to = 1, from = 1;
				for (int lauf = 1; lauf <= num_id_jobs; ++lauf) {
					to = from + num_seeds_for_identities_auxjobs_ - 1;
					string tag = to_string(lauf) + "_of_" + to_string(
							num_id_jobs);
					string id_output = tmp_identities_dir + "identities_"
							+ sec_string + "_" + type + "_" + tag;
					Job* job = new GenerateIdentities(seed_output, id_output,
							type, id_options, from, to);
					job->set_conditional(is_conditional());
					auxjobs.push_back(job);
					outall.insert(get_canonical_filename(id_output));
					identity_filenames.push_back(id_output);
					delete_files.push_back(id_output);
					from = to + 1;
				}
			}
		}
		identity_filenames.insert(identity_filenames.end(),
				shared_equation_files_.begin(), shared_equation_files_.end());
		if (create_identities) {
			delete_directories.push_back(tmp_seeds_dir);
			delete_directories.push_back(tmp_identities_dir);
		}
		if (!perform_reductions_)
			continue;
		if (!delete_temporary_files_) {
			delete_files.clear();
			delete_directories.clear();
		}
		RunReduction* job = new RunReduction(sector, tmp_sectordir,
				identity_filenames, shared_substitution_files_, output_file,
				output_file_external, reduction_generic_options_,
				reduzer_options_, preferred_masters_file_, delete_files,
				delete_directories);
		job->set_conditional(is_conditional());
#ifdef HAVE_MPI
		if (!use_full_parallelization_) {
			job->set_num_workers_range(0, 0);
		}
		else {
			job->set_num_workers_range(min_workers_, max_workers_);
		}
#else
		job->set_num_workers_range(0, 0);
#endif // HAVE_MPI
		auxjobs.push_back(job); // reduction job
		in.push_back(output_file); // this is a meta job, runs once output file present
		outall.insert(get_canonical_filename(output_file));

		// construct crossed reductions from uncrossed result
		for (set<Sector>::const_iterator c = x_sectors.begin(); c
				!= x_sectors.end(); ++c) {
			const Sector& xsector = *c;
			string xfile = files->get_filename_sectorreduction(xsector);
			Crossing x = Crossing::uncross(INT(xsector)).second;
			ApplyCrossing *xj = new ApplyCrossing(output_file, xfile, x);
			xj->set_conditional(is_conditional());
			auxjobs.push_back(xj);
			in.push_back(xfile);
			outall.insert(get_canonical_filename(xfile));
		}
	}

	set_conditional(false); // perform this job anyway
	if (run_distribute_external_) {
		DistributeExternal* dj = new DistributeExternal();
		dj->sector_selection_.select_all_ = true;
		auxjobs.push_back(dj);
	}
	return true;
}

std::string ReduceSectors::get_description() const {
	std::stringstream s;
	s << "reduce sectors";
	return s.str();
}

string ReduceSectors::get_tmp_dir(const Sector& sector) const {
	Files* files = Files::instance();
	string sec_string = sector.get_safe_string_for_filename();
	string tmp_dir = cat_directory_and_filename(files->get_tmp_directory(),
			sec_string) + '/';
	return tmp_dir;
}
string ReduceSectors::get_tmp_dir_identities(const Sector& sector) const {
	return cat_directory_and_filename(this->get_tmp_dir(sector), "identities/");
}
string ReduceSectors::get_tmp_dir_seeds(const Sector& sector) const {
	return cat_directory_and_filename(this->get_tmp_dir(sector), "seeds/");
}

}
// namespace Reduze
