/*  combinatorics.h
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#ifndef COMBINATORICS_H
#define COMBINATORICS_H

#include <vector>
#include <map>
#include <set>
#include <list>
#include <ostream>
#include "functions.h"
#include "yamlconfigurable.h"

namespace YAML {
class Emitter;
class Node;
}

namespace Reduze {

/// A permutation
class Permutation: public YAMLConfigurable {
public:
	static YAMLSpec yaml_spec() {
		YAMLSpec s;
		s.set_keyword("permutation");
		s.set_short_description("A permutation in cycle notation.");
		s.set_long_description(""//
					"A permutation of integers in cycle notation."
					" A permutation is defined as a sequence of sequences"
					" e.g. [[1, 2], [3, 4, 5]]. The identity permutation"
					" is the empty sequence [].");
		return s;
	}
	virtual YAMLSpec yaml_spec_link() const {
		return yaml_spec();
	}

	// constructors

	/// constructs identity permutation
	Permutation() {
	}
	/// constructs a permutation from a vector of cycles
	explicit Permutation(const std::vector<std::vector<int> >& cprod) :
		perm_(cprod) {
		init();
	}
	/// constructs a permutation which transforms 'first' to 'second'
	explicit Permutation(const std::map<int, int>& m);
	/// construct a permutation which transforms 'from' to 'to'
	Permutation(const std::vector<int>& from, const std::vector<int>& to);

	// getter
	const std::vector<std::vector<int> >& vector_rep() const {
		return perm_;
	}

	// invert and compose

	/// returns the inverse prmutation
	Permutation inverse() const;
	/// computes the product: x -> (p1 * p2)(x)  =  x -> p1(p2(x))
	/** gives the permutation that first maps x with p2 and then with p1 */
	static Permutation compose(const Permutation& p1, const Permutation& p2);

	// others

	/// adds an offset to all numbers
	/** useful to translate between 0 and 1 based numberings **/
	void relabel(int offset_to_add);
	/// swap the content
	void swap(Permutation& other);
	/// erase the numbers from the the cycles
	void erase(std::set<int> i);
	/// apply the permutation on the integer 'n'
	int apply(int n) const;

	// comparison

	bool is_identity() const {
		return perm_.empty();
	}

	inline bool operator ==(const Permutation& other) const {
		return perm_ == other.perm_;
	}
	inline bool operator !=(const Permutation& other) const {
		return perm_ != other.perm_;
	}
	inline bool operator <(const Permutation& other) const {
		return perm_ < other.perm_;
	}

	// output

	/// prints to YAML
	virtual void print(YAML::Emitter&) const;
	/// prints to YAML, add offset 1 to all indices in output
	virtual void print_to_onebased(YAML::Emitter&) const;
	/// prints to mma, add offset 1 to all indices in output
	virtual void print_mma_to_onebased(std::ostream&) const;
	/// reads from YAML
	virtual void read(const YAML::Node&);
	/// reads from YAML, subtract offset 1 from all indices in output
	virtual void read_from_onebased(const YAML::Node&);

	friend std::ostream& operator<<(std::ostream& os, const Permutation& p);

private:
	/// some checks, calls unique()
	void init();
	/// transformation to disjoint cycles
	void unique();

private:
	std::vector<std::vector<int> > perm_;
};

inline void operator>>(const YAML::Node& n, Permutation& p) {
	p.read(n);
}

inline YAML::Emitter& operator<<(YAML::Emitter& ye, const Permutation& p) {
	p.print(ye);
	return ye;
}

//
//
//


class PermutationSet {
public:
	PermutationSet() {
	}
	explicit PermutationSet(const std::set<Permutation>& p) :
		pset_(p) {
	}

	void insert(const Permutation& p) {
		pset_.insert(p);
	}

	const std::set<Permutation>& pset() const {
		return pset_;
	}

	/// completes the set of permutations to a group (unity erased)
	void complete_permutations();

private:
	/// makes all pairs from this set and the set 's2'
	/** adds then the pairs to this set and defines 's2' as the set with the
	 * 	new pairs (the ones not already in this set) */
	void make_pairs(std::set<Permutation>& s2);

private:
	std::set<Permutation> pset_;
};

/// helper class to get all 'vector<T> A' from 'vector<set<T> > B' where the i-th T in A is from the i-th set of B
/// throws if any set/list of the input vector is empty
/// if the input vector is empty the member get_next() will produce one empty result
/** usage:
 **   vector<set<T> > A; // some vector filled with N non-empty sets
 **   combinations<T> combs(A);
 **   vector<T> result;
 **   while (combs.get_next(result)) {
 **     // do something with the result, (result.size() == 0 or N)
 **   }
 **/

template<class T>
class combinations {
public:

	combinations(const std::vector<std::set<T> >& s) :
			good_(true), s_(s.size()), p_(s.size()) {
		for (unsigned i = 0; i < s_.size(); ++i) {
			std::list<T> l(s[i].begin(), s[i].end());
			s_[i].swap(l);
			if (s_[i].empty())
				throw std::runtime_error(
						"Error in combinations: encountered empty set.");
			p_[i] = s_[i].begin();
		}
	}

	combinations(const std::vector<std::list<T> >& s) :
			good_(true), s_(s), p_(s.size()) {
		for (unsigned i = 0; i < s_.size(); ++i) {
			if (s_[i].empty())
				throw std::runtime_error(
						"Error in combinations: encountered empty list.");
			p_[i] = s_[i].begin();
		}
	}

	bool get_next(std::vector<T>& res) {
		if (good_) {
			std::vector<T> r;
			r.reserve(s_.size());
			typename std::vector<typename std::list<T>::const_iterator>::const_iterator
					it;
			for (it = p_.begin(); it != p_.end(); ++it)
				r.push_back(*(*it));
			r.swap(res);
			good_ = update(s_.size() - 1);
			return true;;
		}
		return false;
	}

private:
	bool update(int i) {
		if (i < 0)
			return false;
		ASSERT(p_[i] != s_[i].end());
		if (++p_[i] != s_[i].end())
			return true;
		p_[i] = s_[i].begin();
		return update(i - 1);
	}

private:
	bool good_;
	std::vector<std::list<T> > s_;
	// a vector of iterators pointing initially to the first element of the lists in s_
	std::vector<typename std::list<T>::const_iterator> p_;
};

// restricted growth string for set partitions

/// helper class to generate set partitions
/** vector of n numbers k_0, ..., k_{n-1} which fulfill
 ** k_0 <= k_{i+1} <= 1 + max{k_0,...,k_i} (k0 = const = 0).
 ** A p-partition is a partition with p subsets (blocks)
 ** The trivial partition has one subset (block) containing all n elements (1-partition).
 ** The singular partition has n blocks, each containing one element (n-partition).
 ** ref: Michael Orlov, Efficient Generation of Set Partitions. 2002.
 **      https://www.cs.bgu.ac.il/~orlovm/papers/partitions.pdf
 **      http://www.informatik.uni-ulm.de/ni/Lehre/WS03/DMM/Software/partitions.pdf **/
class restricted_growth_string {
public:
	/// restricted growth string of size n, for partitions of sets with n elements
	// if 'restrict_num_blocks' is set to a number > 0 then only partitions
	// with exactly 'restrict_num_blocks' subsets (blocks) are allowed to be generated
	restricted_growth_string(int n, int restrict_num_blocks, int min_block_size);

	// transform to the next rgs_, returns false if the rgs_ after last rsg_ has been reached
	bool next();

	/// number of blocks (subsets)
	int num_blocks() const;

	/// number of partitions that can be created
	/* calculated recursively with the associated Stirling numbers of the second kind */
	unsigned long num_partitions() const;

private:
	restricted_growth_string();
	bool min_block_size_good() const;
	bool next_no_filter();

	/// the restricted growth string and the corresponding maximum values
	std::vector<int> rgs_, max_;
	/// restrict to partitions into 'restrict_num_blocks_' blocks
	int restrict_num_blocks_;
	/// restricted to blocks with given minimal block size
	int min_block_size_;
	/// count generated rgs to compare at the end with num_partitions()
	unsigned long count_;
	template<class T> friend class SetPartition;
};

// set partition

/** usage:
 **   set<T> A; // some set
 **   for(SetPartition<T> s(A); s.good(); ++s) {
 **     const vector<set<T> >& partition = *s;
 **     // do something with the partition
 **   }
 **/

template<class T>
class SetPartition {
public:

	/// initial partition for the set of all partitions of the 'init_set'
	SetPartition(const std::set<T>& init_set) {
		*this = SetPartition(init_set, 0, init_set.empty() ? 0 : 1);
	}

	/// construct the initial partition for a subset of the partitions of 'init_set'
	/** The set 'init_set' is the initial set from which the partitions are build.
	 ** The parameter 'restrict_num_blocks' controls the number of blocks
	 ** (subsets) of the partition: if set to 0 then all possible numbers of
	 ** blocks are allowed. If set to a value bigger than zero then only the
	 ** partitions into this number of blocks are generated.
	 ** The parameter 'min_block_size' controls the minimum size that each block
	 ** in a partition must have. For a non-empty initial set the value 1 will
	 ** choose all sizes. **/
	SetPartition(const std::set<T>& init_set, int restrict_num_blocks,
			int min_block_size)
	try :
			init_set_(init_set.begin(), init_set.end()), rgs_(
					restricted_growth_string(init_set.size(),
							restrict_num_blocks, min_block_size)), good_(true) {
		set_partition();
	}
	catch (std::exception& e) {
		throw std::runtime_error(
				"Invalid SetPartition: " + to_string(e.what()));
	}

	/// whether it is safe to dereference or increment the object
	bool good() const {
		return good_;
	}

	/// the dereference gives a constant reference to the partition vector<set<T> >
	const std::vector<std::set<T> >& operator *() const {
		if (!good_)
			throw std::out_of_range("error evaluating partition");
		return partition_;
	}

	/// increment to get access to the next partition
	const SetPartition<T>& operator ++() {
		if (!good_)
			throw std::out_of_range("error evaluating partition");
		if ((good_ = rgs_.next()))
			set_partition();
		return *this;
	}

	/// number of partitions that can be created
	unsigned long num_partitions() const {
		return rgs_.num_partitions();
	}

	const std::vector<std::set<T> >* partition_ptr() const {
		return &partition_;
	}

	const std::vector<T>& init_set() const {
		return init_set_;
	}

private:
	void set_partition() {
		ASSERT(good_);
		partition_ = std::vector<std::set<T> >(rgs_.num_blocks(),
				std::set<T>());
		const std::vector<int>& v = rgs_.rgs_;
		for (int i = 0; i < v.size(); ++i)
			partition_[v[i]].insert(init_set_[i]);
	}

	/// the set to be partitioned in disjoint blocks
	std::vector<T> init_set_;
	/// the current partition
	std::vector<std::set<T> > partition_;
	/// current partition as a restricted growth string
	restricted_growth_string rgs_;
	/// whether it is safe to dereference or increment the object
	bool good_;
};

// subset

/** usage:
 **   set<T> A; // some set
 **   // loop over all sub sets, including the empty and full set
 **   for (SubSet<T> s(A); s.good(); ++s) {
 **     const set<T>& subset = *s;
 **     // do something with the sub set
 **   }
 **/

template<class T>
class SubSet {
public:
	SubSet(const std::set<T>& init_set)
	try :
			p_(init_set, ((init_set.size() < 2) ? (init_set.size()) : 2),
					((init_set.empty()) ? 0 : 1)), first_(true), second_(true), full_(
					std::set<T>(p_.init_set().begin(), p_.init_set().end())) {
		if (p_.good())
			current_ = p_.partition_ptr()->begin();
		if (init_set.size() < 2)
			second_ = false;
	}
	catch (std::exception& e) {
		throw std::runtime_error("Error in SubSet: " + to_string(e.what()));
	}

	/// whether it is safe to dereference or increment the object
	bool good() const {
		return (first_ || second_
				|| (p_.good() && current_ != p_.partition_ptr()->end()));
	}

	/// the dereference gives a constant reference to the subset set<T>
	const std::set<T>& operator *() const {
		if (!good())
			throw std::out_of_range("error evaluating subset");
		if (first_)
			return empty_;
		if (second_)
			return full_;
		return *current_;
	}

	/// increment to get access to the next subset
	const SubSet<T>& operator ++() {
		if (!good())
			throw std::out_of_range("error evaluating subset");
		if (first_)
			first_ = false;
		else if (second_)
			second_ = false;
		else if (++current_ == p_.partition_ptr()->end() && p_.good())
			if ((++p_).good())
				current_ = p_.partition_ptr()->begin();
		return *this;
	}

	unsigned long num_subsets() const {
		return power_ulong(2, full_.size());
	}

	void print(std::ostream& of) const {
		if (!good())
			throw std::out_of_range("error evaluating subset");
		const std::set<T>* sub;
		if (first_)
			sub = &empty_;
		else if (second_)
			sub = &full_;
		else
			sub = &(*current_);
		typename std::set<T>::const_iterator s;
		of << "{";
		for (s = sub->begin(); s != sub->end(); ++s)
			of << ((s != sub->begin()) ? ", " : "") << *s;
		of << "}";
	}

private:
	SetPartition<T> p_;
	typename std::vector<std::set<T> >::const_iterator current_;
	bool first_, second_;
	std::set<T> empty_, full_;
};

} // namespace Reduze


/*
 This is the mail archive of the libstdc++@gcc.gnu.org mailing list for the libstdc++ project.
 Index Nav: 	[Date Index] [Subject Index] [Author Index] [Thread Index]
 Message Nav: 	[Date Prev] [Date Next] 	[Thread Prev] [Thread Next]
 Other format: 	[Raw text]
 Algorithm code: Combination opposite to std::permutation

 From: BenBear <cxj26424 at 163 dot com>
 To: libstdc++ at gcc dot gnu dot org
 Date: Thu, 21 Apr 2005 21:41:08 +0800
 Subject: Algorithm code: Combination opposite to std::permutation

 Hi, libstdc++-v3

 adjust_combination ()
 next_combination ()
 prev_combination ()

 I wrote a algorithm of combination, which is opposite to
 std::permutation.  It provides three operations, adjust_combination(),
 next_combination() and prev_combination().  The next_combination() and
 prev_combination() is just like the std::next_permutation() and
 std::perv_permutation().

 Each operation's arguments are three iterators, first, middle and last.
 I treat the range [first, middle) as the elements selected, while the
 range [middle, last) are the elements left.  The range [first, middle)
 is treated in dictionary ordering, like the std::permutation.  [middle,
 last) is maintained in a special ordering.

 A simple example:

 select 2 elements from {1, 2, 3, 4}
 f   m   l              // f: first,   m: middle,    l: last
 1 2, 3 4
 1 3, 4 2
 1 4, 3 2
 2 3, 4 1
 2 4, 3 1
 3 4, 2 1

 The first two number are the elements selected, in dictionary ordering.
 The next combination of {3 4}/{2 1} will be {1 2}/{3 4}.

 Ben Bear
 */

// Combination algorithm implementation

// Copyright (C) 2004, BenBear
//
// This file is an algorithm of the combination. This library is free
// software; you can redistribute it and/or modify it under the terms
// of the GNU General Public License as published by the Free Software
// Foundation; either version 2, or (at your option) any later
// version.

// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this library; see the file COPYING.  If not, write to
// the Free Software Foundation, 59 Temple Place - Suite 330, Boston,
// MA 02111-1307, USA.

#include <algorithm>

namespace benbear {

using namespace std;

template<typename BiIterator>
void sort_combination(BiIterator first, BiIterator last) {
	if (first == last) // no element
		return;

	BiIterator i = first;
	++i;
	if (i == last) // one element
		return;

	int half = distance(first, last) / 2; // half of the length
	BiIterator middle = first;
	advance(middle, half); // middle += half

	sort_combination(first, middle); // sort first part
	sort_combination(middle, last); // sort second part

	inplace_merge(first, middle, last); // merge two parts
}

/// sets up a combination according to the rules
template<typename BiIterator>
void adjust_combination(BiIterator first, BiIterator middle, BiIterator last) {
	// the front part or the back part have no elements
	if ((first == middle) || (middle == last))
		return;

	sort_combination(first, middle);
	sort_combination(middle, last);

	BiIterator b = middle;
	--b;
	BiIterator j = lower_bound(middle, last, *b);
	reverse(j, last);
	reverse(middle, last);
}

/// sets up a combination to its smallest (or largest) element
template<typename BiIterator>
void init_combination(BiIterator first, BiIterator middle, BiIterator last,
		bool min) {
	sort_combination(first, last);

	if (min == false) {
		// the max combination
		reverse(first, last);
		reverse(first, middle);
	}
}

template<typename BiIterator>
bool next_combination(BiIterator first, BiIterator middle, BiIterator last) {
	if ((first == middle) || (middle == last))
		return false;

	// last element of [first, middle)
	BiIterator b = middle;
	--b;

	if (*b < *middle) {
		BiIterator j = b;
		while ((++b != last) && (*j < *b)) {
			iter_swap(j, b);
			j = b;
		}
		return true;
	}

	BiIterator e = last;
	--e;
	while (e != middle) {
		BiIterator k = e;
		--k;
		if (!(*k < *e))
			e = k;
		else
			break;
	}

	BiIterator f = e;
	++f;
	while ((f != last) && !(*f < *e))
		++f;

	if (!(*first < *e)) {
		reverse(first, middle);
		reverse(first, last);
		return false;
	}

	if (*b < *e) {
		BiIterator bb = b;
		while ((++bb != e) && !(*b < *bb))
			;
		reverse(bb, f);
		reverse(b, f);
	} else {
		BiIterator i = b;
		while (!(*--i < *e))
			;

		BiIterator j = last;
		while (!(*i < *--j))
			;

		iter_swap(i, j);
		reverse(++i, middle);
		reverse(i, j);
	}
	return true;
}

template<typename BiIterator>
bool prev_combination(BiIterator first, BiIterator middle, BiIterator last) {
	if ((first == middle) || (middle == last))
		return false;

	BiIterator b = middle;
	--b;

	if (*middle < *b) {
		BiIterator i = upper_bound(first, middle, *middle);
		if (i != b)
			iter_swap(i, middle);
		else {
			BiIterator s = middle;
			while ((++s != last) && !(*s < *middle))
				;
			reverse(b, s);
		}

		return true;
	}

	BiIterator e = last;
	--e;
	while (e != middle) {
		BiIterator k = e;
		--k;
		if (!(*k < *e))
			e = k;
		else
			break;
	}

	BiIterator f = e;
	++f;
	while ((f != last) && !(*f < *e))
		++f;

	if (f == last) {
		reverse(first, last);
		reverse(first, middle);
		return false;
	}

	BiIterator i = upper_bound(first, middle, *f);
	if (i == b) {
		BiIterator s = f;
		while ((++s != last) && !(*s < *f))
			;

		reverse(b, f);
		reverse(b, s);
	} else {
		iter_swap(i, f);
		reverse(++i, f);
		reverse(i, middle);
	}
	return true;
}

} // end of namespace benbear

/*
 // for test:
 #include <iostream>
 #include <cstdlib>
 #include <vector>
 using namespace std;

 template<typename BiIterator>
 void show_array(BiIterator f, BiIterator m, BiIterator l) {
 while (f != m)
 cout << *f++;
 cout << " ";
 while (f != l)
 cout << *f++;
 cout << endl;
 }

 template<typename BiIterator>
 void combi(BiIterator f, BiIterator m, BiIterator l, bool next) {
 benbear::init_combination(f, m, l, next);
 int c = 0;
 cout << endl;
 do {
 show_array(f, m, l);
 c++;
 } while (next ? benbear::next_combination(f, m, l)
 : benbear::prev_combination(f, m, l));
 cout << "The end: ";
 show_array(f, m, l);
 cout << "There's: " << c << endl;
 }

 int main() {
 int a[9] = { 1, 2, 3, 4, 5, 6, 7, 8, 9 };
 bool dir = false;

 for (int i = 1; i < 10; i++) {
 random_shuffle(a, a + 7);
 sort(a, a + 3);
 benbear::adjust_combination(a, a + 3, a + 7);
 show_array(a, a + 3, a + 7);
 }
 system("Pause");

 for (int i = 1, j = 9; i < j; i++) {
 combi(a, a + i, a + j, dir);
 system("Pause");
 }

 sort(a, a + 9);
 benbear::sort_combination(a, a + 7);
 a[1] = 1;
 a[4] = 6;
 //a[6] = 6;
 for (int i = 1, j = 7; i < j; i++) {
 combi(a, a + i, a + j, dir);
 system("Pause");
 }

 int n;
 cout << "The n: ";
 cin >> n;
 vector<int> vec(n);
 cout << "please. input numbers: " << endl;
 for (int i = 0; i < n; i++)
 cin >> vec[i];

 for (int i = 1; i < n; i++) {
 combi(vec.begin(), vec.begin() + i, vec.end(), dir);
 system("Pause");
 }

 system("Pause");
 return 0;
 }
 */

#endif // COMBINATORICS_H
