//
// File: vk_layer.h
//
/*
 * Copyright (c) 2015-2017 The Khronos Group Inc.
 * Copyright (c) 2015-2017 Valve Corporation
 * Copyright (c) 2015-2017 LunarG, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

/* Need to define dispatch table
 * Core struct can then have ptr to dispatch table at the top
 * Along with object ptrs for current and next OBJ
 */
#pragma once

#include "vulkan.h"
#if defined(__GNUC__) && __GNUC__ >= 4
#define VK_LAYER_EXPORT __attribute__((visibility("default")))
#elif defined(__SUNPRO_C) && (__SUNPRO_C >= 0x590)
#define VK_LAYER_EXPORT __attribute__((visibility("default")))
#else
#define VK_LAYER_EXPORT
#endif

#define MAX_NUM_UNKNOWN_EXTS 250

// Loader-Layer version negotiation API.  Versions add the following features:
//   Versions 0/1 - Initial.  Doesn't support vk_layerGetPhysicalDeviceProcAddr
//                  or vk_icdNegotiateLoaderLayerInterfaceVersion.
//   Version 2    - Add support for vk_layerGetPhysicalDeviceProcAddr and
//                  vk_icdNegotiateLoaderLayerInterfaceVersion.
#define CURRENT_LOADER_LAYER_INTERFACE_VERSION 2
#define MIN_SUPPORTED_LOADER_LAYER_INTERFACE_VERSION 1

#define VK_CURRENT_CHAIN_VERSION 1

// Typedef for use in the interfaces below
typedef PFN_vkVoidFunction(VKAPI_PTR *PFN_GetPhysicalDeviceProcAddr)(VkInstance instance, const char *pName);

// Version negotiation values
typedef enum VkNegotiateLayerStructType {
	LAYER_NEGOTIATE_UNINTIALIZED = 0,
	LAYER_NEGOTIATE_INTERFACE_STRUCT = 1,
} VkNegotiateLayerStructType;

// Version negotiation structures
typedef struct VkNegotiateLayerInterface {
	VkNegotiateLayerStructType sType;
	void *pNext;
	uint32_t loaderLayerInterfaceVersion;
	PFN_vkGetInstanceProcAddr pfnGetInstanceProcAddr;
	PFN_vkGetDeviceProcAddr pfnGetDeviceProcAddr;
	PFN_GetPhysicalDeviceProcAddr pfnGetPhysicalDeviceProcAddr;
} VkNegotiateLayerInterface;

// Version negotiation functions
typedef VkResult(VKAPI_PTR *PFN_vkNegotiateLoaderLayerInterfaceVersion)(VkNegotiateLayerInterface *pVersionStruct);

// Function prototype for unknown physical device extension command
typedef VkResult(VKAPI_PTR *PFN_PhysDevExt)(VkPhysicalDevice phys_device);

// ------------------------------------------------------------------------------------------------
// CreateInstance and CreateDevice support structures

/* Sub type of structure for instance and device loader ext of CreateInfo.
 * When sType == VK_STRUCTURE_TYPE_LOADER_INSTANCE_CREATE_INFO
 * or sType == VK_STRUCTURE_TYPE_LOADER_DEVICE_CREATE_INFO
 * then VkLayerFunction indicates struct type pointed to by pNext
 */
typedef enum VkLayerFunction_ { VK_LAYER_LINK_INFO = 0, VK_LOADER_DATA_CALLBACK = 1 } VkLayerFunction;

typedef struct VkLayerInstanceLink_ {
	struct VkLayerInstanceLink_ *pNext;
	PFN_vkGetInstanceProcAddr pfnNextGetInstanceProcAddr;
	PFN_GetPhysicalDeviceProcAddr pfnNextGetPhysicalDeviceProcAddr;
} VkLayerInstanceLink;

/*
 * When creating the device chain the loader needs to pass
 * down information about it's device structure needed at
 * the end of the chain. Passing the data via the
 * VkLayerDeviceInfo avoids issues with finding the
 * exact instance being used.
 */
typedef struct VkLayerDeviceInfo_ {
	void *device_info;
	PFN_vkGetInstanceProcAddr pfnNextGetInstanceProcAddr;
} VkLayerDeviceInfo;

typedef VkResult(VKAPI_PTR *PFN_vkSetInstanceLoaderData)(VkInstance instance, void *object);
typedef VkResult(VKAPI_PTR *PFN_vkSetDeviceLoaderData)(VkDevice device, void *object);

typedef struct {
	VkStructureType sType; // VK_STRUCTURE_TYPE_LOADER_INSTANCE_CREATE_INFO
	const void *pNext;
	VkLayerFunction function;
	union {
		VkLayerInstanceLink *pLayerInfo;
		PFN_vkSetInstanceLoaderData pfnSetInstanceLoaderData;
	} u;
} VkLayerInstanceCreateInfo;

typedef struct VkLayerDeviceLink_ {
	struct VkLayerDeviceLink_ *pNext;
	PFN_vkGetInstanceProcAddr pfnNextGetInstanceProcAddr;
	PFN_vkGetDeviceProcAddr pfnNextGetDeviceProcAddr;
} VkLayerDeviceLink;

typedef struct {
	VkStructureType sType; // VK_STRUCTURE_TYPE_LOADER_DEVICE_CREATE_INFO
	const void *pNext;
	VkLayerFunction function;
	union {
		VkLayerDeviceLink *pLayerInfo;
		PFN_vkSetDeviceLoaderData pfnSetDeviceLoaderData;
	} u;
} VkLayerDeviceCreateInfo;

#ifdef __cplusplus
extern "C" {
#endif

VKAPI_ATTR VkResult VKAPI_CALL vkNegotiateLoaderLayerInterfaceVersion(VkNegotiateLayerInterface *pVersionStruct);

typedef enum VkChainType {
	VK_CHAIN_TYPE_UNKNOWN = 0,
	VK_CHAIN_TYPE_ENUMERATE_INSTANCE_EXTENSION_PROPERTIES = 1,
	VK_CHAIN_TYPE_ENUMERATE_INSTANCE_LAYER_PROPERTIES = 2,
	VK_CHAIN_TYPE_ENUMERATE_INSTANCE_VERSION = 3,
} VkChainType;

typedef struct VkChainHeader {
	VkChainType type;
	uint32_t version;
	uint32_t size;
} VkChainHeader;

typedef struct VkEnumerateInstanceExtensionPropertiesChain {
	VkChainHeader header;
	VkResult(VKAPI_PTR *pfnNextLayer)(const struct VkEnumerateInstanceExtensionPropertiesChain *, const char *,
									  uint32_t *, VkExtensionProperties *);
	const struct VkEnumerateInstanceExtensionPropertiesChain *pNextLink;

#if defined(__cplusplus)
	inline VkResult CallDown(const char *pLayerName, uint32_t *pPropertyCount,
							 VkExtensionProperties *pProperties) const {
		return pfnNextLayer(pNextLink, pLayerName, pPropertyCount, pProperties);
	}
#endif
} VkEnumerateInstanceExtensionPropertiesChain;

typedef struct VkEnumerateInstanceLayerPropertiesChain {
	VkChainHeader header;
	VkResult(VKAPI_PTR *pfnNextLayer)(const struct VkEnumerateInstanceLayerPropertiesChain *, uint32_t *,
									  VkLayerProperties *);
	const struct VkEnumerateInstanceLayerPropertiesChain *pNextLink;

#if defined(__cplusplus)
	inline VkResult CallDown(uint32_t *pPropertyCount, VkLayerProperties *pProperties) const {
		return pfnNextLayer(pNextLink, pPropertyCount, pProperties);
	}
#endif
} VkEnumerateInstanceLayerPropertiesChain;

typedef struct VkEnumerateInstanceVersionChain {
	VkChainHeader header;
	VkResult(VKAPI_PTR *pfnNextLayer)(const struct VkEnumerateInstanceVersionChain *, uint32_t *);
	const struct VkEnumerateInstanceVersionChain *pNextLink;

#if defined(__cplusplus)
	inline VkResult CallDown(uint32_t *pApiVersion) const {
		return pfnNextLayer(pNextLink, pApiVersion);
	}
#endif
} VkEnumerateInstanceVersionChain;

#ifdef __cplusplus
}
#endif
