/*  ---------------------------------------------------------------
    xhkeys - xhkconf Jun 2004
    Copyright (C) 2002, 2004  Michael Glickman  <wmalms@yahoo.com>
    License: GPL
    --------------------------------------------------------------- */
#include <X11/Xlib.h>
#include <X11/Xutil.h>

#include "xhkeys_conf.h"

#ifdef PLUGIN_SUPPORT

#include <stdio.h>		// FILENAME_MAX
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <dlfcn.h>
#include "xhkeys_plugin.h"

#ifdef HAVE_DIRENT_H
#define USE_PLUGIN_LOOKUP
#include <dirent.h>
#endif


#ifdef USE_DMALLOC
#include <dmalloc.h>
#endif

#include "xhkeys.h"

#ifndef FILENAME_MAX
# define FILENAME_MAX 4096
#endif

// xhconf.c
extern const char *resFileName;

// resources.c
extern PluginData *Plugins;
extern int PluginCount;
extern int maxPlugins;

// common.c
extern const char *PluginDirs[];
extern const char *DefaultPluginNames[];

#define NEW_PLUGIN 1000
#define DONE 1001

#define IS_ACTIVE_FLAG  1

#ifdef USE_PLUGIN_LOOKUP
static char **newPluginNames;
static char **newPluginDescr;
static char **newPluginPaths;
static int *newPluginFlags;	// IS_ACTIVE_FLAG
static int newPluginCount, maxNewPluginCount;
static int maxNewPluginCount;
#endif


void AppendPluginVersion(int version, const char *prefix, char *text, int textLen)
{
    int len = strlen(text);
    
    snprintf(text+len, textLen-len, "%s %d.%d.%d", prefix,
    	version >> 12, (version >> 8) & 15, version & 255);
}    

#ifdef USE_PLUGIN_LOOKUP
void LookupDirPlugins(const char *dirName)
{
    DIR *dir;
    struct dirent *dirEntry;
    char fileName[sizeof(dirEntry->d_name)];
    char fullPath[FILENAME_MAX];
    PLUGIN_HANDLE *handle;
    int i, len, flag;
    int newPluginCountOriginal = newPluginCount;
    Bool validated;
    char msg[121];	// Sufficient for description
    const char *tmpName;
    char *descrPtr;
    
//    dirName = ".";
    dir = opendir(dirName);
    if (dir == NULL) return;

    
    while (newPluginCount < maxNewPluginCount && 
           (dirEntry = readdir(dir)) != NULL)
    {
	strcpy(fileName, dirEntry->d_name);
	len = strlen(fileName);	
	if (len <= 3) continue;
	
	if (strcmp(fileName+len-3, ".so")) continue;
	
	snprintf(fullPath, sizeof(fullPath), "%s/%s", dirName, fileName);

	fileName[len-3] = '\0';	// Get rid of ".so"

	// Is plugin already included from a previous library ?
	for (i=0; i<newPluginCountOriginal; i++) {
	    if (strcmp(fileName, newPluginNames[i]) == 0) break;	
	}
	if (i <	newPluginCountOriginal) continue;	// Already included

	flag = 0;
	
	// Is plugin already active
	for (i=0; i<PluginCount; i++) {
	    if ((tmpName = Plugins[i].name) != NULL &&
	         strcmp(fileName, tmpName) == 0) break;	
	}

	if (i<PluginCount) {
	    flag |= IS_ACTIVE_FLAG;
	    handle = Plugins[i].handle;
	    strncpy(fullPath, Plugins[i].fullPath, FILENAME_MAX);
	    validated = True;	    
	} else {    
	    // For a non-active plugin, let's check it
	    handle = dlopen(fullPath, RTLD_LAZY);
	    
	    if (handle == NULL) {
		snprintf(msg, sizeof(msg), "Plugin '%s' is invalid", fullPath); 
		validated = False;
	    }	
	    else 
		validated =  validatePlugin(fullPath, handle, msg, sizeof(msg));
	}    

	if (validated) {
	    // We've got it!    
	    XHKEYS_PLUGIN_PROPERTYEXCHANGER  *getPropPtr;
	    
	    descrPtr = NULL;	// No description
	    getPropPtr = dlsym(handle, "property");
	    
	    
	    if (getPropPtr != NULL) {
		void *buffer = Plugins[i].buffer;
			
		if ((*getPropPtr)(buffer, XHK_PLUGIN_DESCRIPTION, msg, sizeof(msg))) {
	    	    int version;
		    if ((*getPropPtr)(buffer, XHK_PLUGIN_VERSION, &version)) 
			AppendPluginVersion(version, "  v.",  msg, sizeof(msg));

		    descrPtr = strdup(msg);
		}

	    }	    

	    newPluginNames[newPluginCount] = strdup(fileName);
	    newPluginPaths[newPluginCount] = strdup(fullPath);
	    newPluginDescr[newPluginCount] = descrPtr;
	    newPluginFlags[newPluginCount++] = flag;
	} else
	    fprintf (stderr, "%s\n", msg);
		    
	if ((flag & IS_ACTIVE_FLAG) == 0 && handle != NULL)
	    dlclose(handle);       
    }	
    
}

void FreePluginLookupData(void)
{
    int i;
    
    if (newPluginNames != NULL) {
	for (i=0; i<newPluginCount; i++)
	    free(newPluginNames[i]);

	free(newPluginNames);
    }	            

    if (newPluginDescr != NULL) {
	char *tmp;
	
	for (i=0; i<newPluginCount; i++) {
	    if ((tmp = newPluginDescr[i]) != NULL)
		free(tmp);
	}
	free(newPluginDescr);
    }	            

    if (newPluginPaths != NULL) {
	char *tmp;
	
	for (i=0; i<newPluginCount; i++) {
	    if ((tmp = newPluginPaths[i]) != NULL)
		free(tmp);
	}
	free(newPluginPaths);
    }	            

    if (newPluginFlags != NULL) free(newPluginFlags);
    
    newPluginCount = 0;
}


void LookupPlugins()
{
    int i;
    const char *dirName;
    
    maxNewPluginCount = maxPlugins - PluginCount;

    newPluginNames = malloc(maxNewPluginCount * sizeof(char *));
    newPluginDescr = malloc(maxNewPluginCount * sizeof(char *));
    newPluginPaths = malloc(maxNewPluginCount * sizeof(char *));
    newPluginFlags = malloc(maxNewPluginCount * sizeof(int));
    newPluginCount = 0;
    
    if (newPluginNames == NULL || newPluginDescr == NULL ||
	newPluginFlags == NULL) {
	FreePluginLookupData();
	return;
    }	

    for (i=0; (dirName=PluginDirs[i]) != NULL; i++) {
//	if (*dirName != '.')	// Bypass current directory
	    LookupDirPlugins(dirName);
    }    
}

Bool SelectPluginToAdd(char *pluginName, int pluginNameLen, char fullPath[FILENAME_MAX], int *flag, int *index)
{
    int i, len;
    char text[121];
    const char *descr;

    if (newPluginNames == NULL)
	LookupPlugins();
	
    if (newPluginCount == 0) {
	fprintf (stderr, "No xhkeys plugins found\n");
	return False;
    }	

    for (;;)
    {
	printf ("\n\nPlugins to add:\n");
	printf (    "---------------\n");
	printf ("FYI. (*) stands for already active plugins that\n");
        printf ("can be re-included with a different init string\n");
    
	for (i=0; i<newPluginCount; i++)
	{
    	    snprintf(text, sizeof(text), "%2d. %s", i+1, newPluginNames[i]);

            if (newPluginFlags[i] & IS_ACTIVE_FLAG) {
	        len = strlen(text);
	        snprintf(text+len, sizeof(text)-len, "(*)");
	    }
	
	    descr = newPluginDescr[i];
	    if (descr != NULL) {	    
		len = strlen(text);
	        snprintf(text+len, sizeof(text)-len, ": %s", descr);
	    }    

	    text[sizeof(text)-1] = '\0';
	    printf ("%s\n", text);
        }

	printf (" D. Done\n");

	enterText("\nSelect option: ", text, 4);
	if (*text == '\0' || strchr("DdQq", *text) != NULL) break;	// Cancel

	i = strtol(text, (char **)&descr, 10);
	if (*descr != '\0' || i<=0 || i>newPluginCount) continue;
	
	strncpy(pluginName, newPluginNames[--i], pluginNameLen);
	strncpy(fullPath, newPluginPaths[i], FILENAME_MAX);
	*flag = newPluginFlags[i];
	
	*index = i;
	return True;        
    }	
    
    return False;  
    
}


#else
Bool LookupPlugin(const char *pluginName, char fullPath[FILENAME_MAX])
{
    PLUGIN_HANDLE handle;
    Bool success = False;
    Bool printed = False;
    char *fullPluginName;
    const char *prefName;
    char msg[81];
    int i;
        
    {	int len, maxLen = 0;
	
	for (i=0; (prefName = PluginDirs[i]) != NULL; i++) {
	    if ((len = strlen(prefName)) > maxLen) maxLen = len;	
	}    

        fullPluginName = malloc(strlen(pluginName) + maxLen + 7);
	if (fullPluginName == NULL) return True;  // That's not a mistake!
    }

    handle = NULL;

    for (i=0; (prefName = PluginDirs[i]) != NULL; i++) {
	if (*prefName == '.') continue;
	
        strcpy(fullPluginName, prefName);
        strcat(fullPluginName, "/");
        strcat(fullPluginName, pluginName);
        strcat(fullPluginName, ".so");
        handle = (PLUGIN_HANDLE) dlopen (fullPluginName, RTLD_LAZY);
        if (handle == NULL) continue;

	success = validatePlugin(fullPluginName, handle, msg, sizeof(msg));
	dlclose(handle);

	if (success) break;	
	
	printf("%s\n", msg);
	printed = True;
    }	

    strcpy(fullPath, fullPluginName);
    free(fullPluginName);
    
    if (success == False && printed == False) 
	printf ("\n\nPlugin '%s' is not found\n", pluginName);
	
    return success;
}    


Bool SelectPluginToAdd(char *pluginName, int pluginNameLen, char fullPath[FILENAME_MAX], int *flag)
{
    int i;
    const char *tmp;
    
    *flag = 0;	
    
    enterText("\nEnter plugin name\n", pluginName, pluginNameLen);
    if (*pluginName == False) return False;
    
    for (i=0; i<PluginCount; i++) {
	if ((tmp = Plugins[i].name) != NULL &&
            strcmp(pluginName, tmp) == 0) break;	
    }

    if (i<PluginCount) {
	*flag |= IS_ACTIVE_FLAG;
	strncpy(fullPath, Plugins[i].fullPath, FILENAME_MAX);
	return  YesNo("Plugin is already active.\nInclude with a different init string", False);
    }
    
    return  LookupPlugin(pluginName, fullPath);
}
#endif

static int GetFreePluginPosition()
{
    int i;
    for (i=0; i<PluginCount && Plugins[i].lineno <= (i+1); i++);

    return i;
}

static Bool AddPluginData()
{
    Bool result = False;    
    char pluginName[256];
    char fullPath[FILENAME_MAX];
    char initString[1024];
    PLUGIN_HANDLE handle;
    XHKEYS_PLUGIN_INIT  *initFunc;
    char *errMsg;
    int flag;
    int lineNo;
    void *buffer = NULL;
#ifdef USE_PLUGIN_LOOKUP
    int newPluginIndex;
#endif    
#if defined(CDROM_SUPPORT) || defined(MIXER_SUPPORT)
    char *shortName;
#endif
    
    while (PluginCount < maxPlugins &&
#ifdef USE_PLUGIN_LOOKUP
           SelectPluginToAdd(pluginName, sizeof(pluginName), fullPath, &flag, &newPluginIndex))
#else
           SelectPluginToAdd(pluginName, sizeof(pluginName), fullPath, &flag))
#endif	   
    {
	
	lineNo = GetFreePluginPosition();
        if (lineNo>= maxPlugins) break;	// Should never happen

	Plugins[PluginCount].lineno = lineNo+1;
	Plugins[PluginCount].name = strdup(pluginName);

#if defined(CDROM_SUPPORT) || defined(MIXER_SUPPORT)
	shortName = NULL;
	
	if ((flag & IS_ACTIVE_FLAG) == 0) {
	    const char *defName;
	    for (lineNo=0; (defName = DefaultPluginNames[lineNo*2]) != NULL; lineNo++) {
		if (strcmp(pluginName, defName) == 0) {
		    shortName = strdup(DefaultPluginNames[lineNo*2+1]);
		    break;
		}  				 	
	    }	    
        }

	Plugins[PluginCount].shortName = shortName;
	if (shortName != NULL)
	    printf("\nAdding %s(%s)\n",  pluginName,  shortName);
	else    
#else
	Plugins[PluginCount].shortName = NULL;
#endif	    
	    printf("\nAdding %s\n",  pluginName);


	Plugins[PluginCount].fullPath = strdup(fullPath);
	Plugins[PluginCount].handle = handle = dlopen(fullPath, RTLD_LAZY);
	
	*initString = '\0';
	initFunc = dlsym(handle, "init");
	
	if (handle != NULL) {
	    char msg[81];
	    
	    enterText("Init string: ", initString, sizeof(initString));

	    if ((*initFunc)(&buffer, initString, msg, sizeof(msg)) == False) {
		fprintf (stderr, "Init error: %s\n", msg);
		continue;
	    }	    	 

	}	

        Plugins[PluginCount].buffer = buffer;

        if (*initString == '\0')
	    Plugins[PluginCount].initString = NULL;
	else
	    Plugins[PluginCount].initString = strdup(initString);

        if (UpdatePlugin(resFileName, PluginCount, &errMsg)) {
	    PluginCount++;
#ifdef USE_PLUGIN_LOOKUP
	    newPluginFlags[newPluginIndex] |= IS_ACTIVE_FLAG;
#endif	    
	    printf ("Plugin %s has been added\n", pluginName);
	    result = True;
	}
	else
	    fprintf(stderr, "%s\n", errMsg);
    }    
    
    return result;
}

static int EditPluginData(int index, char **errMsg)
{
    Bool result = False;
    PLUGIN_HANDLE handle;
    XHKEYS_PLUGIN_PROPERTYEXCHANGER *descriptPtr = NULL;
    PluginData *plData;
    const char *tmp;
    char *buffer;
    char text[81];
    int reply;

    if (index<0 || index>=PluginCount) return False;
    
    plData = Plugins + index;    
    
    handle = plData->handle;

    if (handle != NULL)
	descriptPtr = dlsym(handle, "property");

    strncpy(text,  plData->name, sizeof(text));

    buffer = plData->buffer;

    if (descriptPtr != NULL) {
        int version;
        if((*descriptPtr)(buffer, XHK_PLUGIN_VERSION, &version)) 
	    AppendPluginVersion(version, "  version",  text, sizeof(text));
    }	    

    printf ("\n%s\n", text);

    if (descriptPtr != NULL) {
        char descr[121];
        if ((*descriptPtr)(buffer, XHK_PLUGIN_DESCRIPTION, descr, sizeof(descr))) {
	    printf("Description: %s\n", descr);
	}    
    }

    printf ("Location:    %s\n", plData->fullPath);
    printf ("--------------------------------\n");

    for (;;) {    
        tmp = plData->shortName;  if (tmp == NULL) tmp = "";
	printf ("1. Alias:       %s\n", tmp);
	tmp = plData->initString;   if (tmp == NULL) tmp = "";
	printf ("2. Init.string: %s\n", tmp);
	printf ("D. Delete\n");
	printf ("Q. Done\n");
    
	enterText("\nSelect option: ", text, 4);
    
        reply = toupper(text[0]);
	if (reply == '\0' || strchr("Qq", reply) != NULL) break;	

	if (reply == '1') {
	    tmp = plData->shortName;  
	    if (tmp != NULL) { free((char *)tmp); plData->shortName = NULL;}
	    
	    enterText("\nNew alias: ", text, sizeof(text));
	    if (strlen(text) > 0) plData->shortName = strdup(text);

	    result = True;
	}		    
	else	
	if (reply == '2') {
	    tmp = plData->initString;  
	    if (tmp != NULL) { free((char *)tmp); plData->initString = NULL;}
	    
	    enterText("\nNew init strng: ", text, sizeof(text));
	    if (strlen(text) > 0) plData->initString = strdup(text);
	    
	    result = True;
	}		    
	else
	if (strchr("Dd", reply) != NULL){
	    // Delete
	    tmp = plData->name; 
	    snprintf(text, sizeof(text), "Delete '%s'", tmp);
	    if (YesNo(text, True) == False) continue;
	    
	    if (tmp != NULL) {
		free((char *)tmp); plData->name = NULL;
	    }	
	    tmp = plData->shortName; 
	    if (tmp != NULL) {
		free((char *)tmp); plData->shortName = NULL;
	    }	
	    tmp = plData->initString; 
	    if (tmp != NULL) {
		free((char *)tmp); plData->initString = NULL;
	    }	
	    tmp = plData->fullPath; 
	    if (tmp != NULL) {
		free((char *)tmp); plData->fullPath = NULL;
	    }	
	    plData->handle = NULL;
	    
	    result = True;	    
	    break;
	}    
    }
    
    if (result)
        result = UpdatePlugin(resFileName, index, errMsg);

    return result;

}

static int SelectPluginToEdit()
{
    int i, len;
    char text[81];
    char *tmp1;
    int realCount;

    int *indices = malloc(sizeof(int) * PluginCount);
    
    if (indices == NULL) return DONE;


    do {
        printf ("\n\nDeclared plugins:\n");
	printf (    "-----------------\n");
	
	realCount = 0;
        
	for (i=0; i<PluginCount; i++) {
	    snprintf(text, sizeof(text), "%2d. ", realCount+1);
	    len = strlen(text);
	    	    
	    if (GetPluginInfoText(Plugins+i, text+len, sizeof(text)-len)) {
	        printf("%s\n", text);    
		indices[realCount++] = i;
	    }	
        }

        printf(" N. New plugin(-s)\n");    
	printf(" Q. Quit\n");    

	enterText("\nSelect option: ", text, 4);

	len = strlen(text);

	if (len == 0) {
	    i = DONE; break;
	}    

	i = *text;

	if (strchr("Qq", i) != NULL) 
	    i = DONE;
	else     	
	if (strchr("Nn", i) != NULL) 
	    i = NEW_PLUGIN; 
	else {
	    i = strtol(text, &tmp1, 10);

	    if (*tmp1 == '\0' && i>0 && i<=realCount)
		i = indices[i-1];
	    else
		i = -1;	// next iteration
	}
    } while (i < 0);

    free(indices);
    return i;
}


//====================================================================
Bool configurePlugins(char **errMsg)
{
    int sel;
    
    for (;;)
    {
	sel = SelectPluginToEdit();
	if (sel == DONE) break;

	if (sel == NEW_PLUGIN)	
	    AddPluginData();
	else
	    EditPluginData(sel, errMsg);    
	
    }

#ifdef USE_PLUGIN_LOOKUP
    FreePluginLookupData();
#endif
    return True;
}

Bool editPluginByName(const char *pluginName, char **errMsg)
{
    const char *tmp;
    int i;

    // Looking for alias
    for (i=0; i<PluginCount; i++) {
	tmp = Plugins[i].shortName;
	if (tmp != NULL &&
	   strcasecmp(pluginName, tmp) == 0) break;
    
    }

    if (i>= PluginCount) {
	for (i=0; i<PluginCount; i++) {
	    tmp = Plugins[i].name;
    	    if (tmp != NULL &&
	      strcasecmp(pluginName, tmp) == 0) break;
    	}
    }	

    if (i < PluginCount)
	return EditPluginData(i, errMsg);

    printf("Plugin '%s' is not found\n", pluginName);    
    return False;
}
#endif

