// -*- C++ -*-
//
// ACDCSampler.cc is a part of ThePEG - Toolkit for HEP Event Generation
// Copyright (C) 1999-2019 Leif Lonnblad
//
// ThePEG is licenced under version 3 of the GPL, see COPYING for details.
// Please respect the MCnet academic guidelines, see GUIDELINES for details.
//
//
// This is the implementation of the non-inlined, non-templated member
// functions of the ACDCSampler class.
//

#include "ThePEG/Handlers/ACDCSampler.h"
#include "ThePEG/Interface/ClassDocumentation.h"
#include "ThePEG/Interface/Parameter.h"
#include "ThePEG/Repository/EventGenerator.h"
#include "ThePEG/Persistency/PersistentOStream.h"
#include "ThePEG/Persistency/PersistentIStream.h"
#include "ThePEG/Utilities/Throw.h"

using namespace ThePEG;

ACDCSampler::~ACDCSampler() {}

IBPtr ACDCSampler::clone() const {
  return new_ptr(*this);
}

IBPtr ACDCSampler::fullclone() const {
  return new_ptr(*this);
}

void ACDCSampler::initialize() {
  theSampler.clear();
  theSampler.setRnd(0);
  theSampler.eps(theEps);
  theSampler.margin(theMargin);
  theSampler.nTry(2);
  theSampler.maxTry(eventHandler()->maxLoop());
  bool nozero = false;


  for ( int i = 0, N = eventHandler()->nBins(); i < N; ++i ) {
    nozero |= theSampler.addFunction(eventHandler()->nDim(i), eventHandler());
  }
  if( eventHandler()->nBins() ==0 ) Throw<EventInitNoXSec>()
    << "The event handler '" << eventHandler()->name()
    << "' cannot be initialized because there are no selected subprocesses"
    << Exception::maybeabort;
  if ( !nozero ) Throw<EventInitNoXSec>()
    << "The event handler '" << eventHandler()->name()
    << "' cannot be initialized because the cross-section for the selected "
    << "sub-processes was zero." << Exception::maybeabort;
  theSampler.clear();
}

double ACDCSampler::generate() {
  if ( !theSampler.generate() ) throw EventLoopException()
    << "The maximum number of attempts (" << eventHandler()->maxLoop()
    << ") to generate the kinematics in the ACDCSampler was exceeded. For "
    << "the event handler '" << eventHandler()->name() << "'."
    << Exception::eventerror;
  lastPoint() = theSampler.lastPoint();
  return 1.0;
}

void ACDCSampler::rejectLast() {
  theSampler.reject();
}

CrossSection ACDCSampler::integratedXSec() const {
  return theSampler.integral()*nanobarn;
}

CrossSection ACDCSampler::integratedXSecErr() const {
  return theSampler.integralErr()*nanobarn;
}

int ACDCSampler::lastBin() const {
  return theSampler.last() - 1;
}

double ACDCSampler::sumWeights() const {
  return theSampler.n();
}

double ACDCSampler::sumWeights2() const {
  return theSampler.n();
}

void ACDCSampler::dofinish() {
  if ( theSampler.n() <= 0 &&
       eventHandler() && eventHandler()->statLevel() > 1 ) {
    generator()->log()
      << "No events generated by the ACDC sampler '" << name() << "'" << endl;
  }
  else if ( eventHandler() && eventHandler()->statLevel() > 1 )
    generator()->log()
      << "Statistics for the ACDC sampler '" << name() << "':" << endl
      << "Number of samplers:    " << setw(14) << theSampler.size() << endl
      << "Number of bins:        " << setw(14) << theSampler.nBins() << endl
      << "Depth of bins:         " << setw(14) << theSampler.depth() << endl
      << "efficiency:            " << setw(14) << theSampler.efficiency() << endl
      << "Total integrated xsec: " << setw(14) << theSampler.integral() << endl
      << "        error in xsec: " << setw(14) << theSampler.integralErr() << endl;
  if ( theSampler.compensating() )
    generator()->logWarning(
      ACDCStillCompensating()
      << "The run was ended while the ACDCSampler '" << name()
      << "' was still trying to compensate for weights larger than 1. "
      << "The cross section estimates may therefore be statistically "
      << "inaccurate. At least " << theSampler.compleft() << " additional "
      << "samplings are needed to get out of compensation mode. "
      << "This may be avoided if you increase the value of the "
      << "Ntry parameter determining how many points are presampled before "
      << "the run." << Exception::warning);
  
    
  SamplerBase::dofinish();
}

void ACDCSampler::doinitrun() {
  SamplerBase::doinitrun();
  eventHandler()->initrun();
  theSampler.setRnd(0);
  theSampler.eps(theEps);
  theSampler.margin(theMargin);
  theSampler.nTry(theNTry);
  theSampler.maxTry(eventHandler()->maxLoop());
  bool nozero = false;
  for ( int i = 0, N = eventHandler()->nBins(); i < N; ++i )
    if ( theSampler.addFunction(eventHandler()->nDim(i), eventHandler()) )
      nozero = true;
  if ( !nozero ) throw EventInitNoXSec()
    << "The event handler '" << eventHandler()->name()
    << "' cannot be initialized because the cross-section for the selected "
    << "sub-processes was zero." << Exception::maybeabort;
}

void ACDCSampler::persistentOutput(PersistentOStream & os) const {
  os << theEps << theMargin << theNTry;
  theSampler.output(os);
}

void ACDCSampler::persistentInput(PersistentIStream & is, int) {
  is >> theEps >> theMargin >> theNTry;
  theSampler.input(is);
  if ( generator() ) theSampler.setRnd(0);
}

ClassDescription<ACDCSampler> ACDCSampler::initACDCSampler;
// Definition of the static class description member.

void ACDCSampler::Init() {

  static ClassDocumentation<ACDCSampler> documentation
    ("This class inherits from ThePEG::SampleBase and implements "
     "the Auto Compensating Divide-and-Conquer phase space generator, "
     "ACDCGenerator::ACDCGen.");

  static Parameter<ACDCSampler,double> interfaceMargin
    ("Margin",
     "The factor controlling the loss of efficiency when compensating "
     "for a previously underestimated phase space point. If close to one, "
     "the efficiency is increased at the expence of increased number "
     "of cells.",
     &ACDCSampler::theMargin, 1.1, 1.0, 2.0, true, false, true);

  static Parameter<ACDCSampler,double> interfaceEps
    ("Epsilon",
     "The smallest possible cell division allowed.",
     &ACDCSampler::theEps, 100.0*Constants::epsilon, Constants::epsilon,
     1.0e-6, true, false, true);

  static Parameter<ACDCSampler,int> interfaceNTry
    ("Ntry",
     "The number of phase space points tried in the initialization.",
     &ACDCSampler::theNTry, 1000, 2, 1000000, true, false, true);

  interfaceNTry.rank(10);
  interfaceEps.rank(9);

}

