(module __crypto-aes
   (static
    (final-class AES-Param
       (state::bstring read-only)
       (Nr::long read-only)
       (w::bstring read-only)))
   (import __crypto-block-ciphers
	   __crypto-util))

(define (key->param key)
   (when (not (or (=fx (string-length key) 16)
		  (=fx (string-length key) 24)
		  (=fx (string-length key) 32)))
      (error 'key-expansion
	     "AES key must be either 16, 24 or 32 characters long."
	     (string-length key)))
   (let* ((state (make-state))
	  (Nr (case (string-length key)
		 ((16) 10) ((24) 12) ((32) 14) (else 0)))
	  (w (key-expansion key Nr)))
      (instantiate::AES-Param
	 (state state)
	 (Nr Nr)
	 (w w))))

(register-cipher! 'aes (instantiate::Block-Cipher
			  (name "AES (Advanced Encryption Standard)")
			  (block-size 16)
			  (preferred-key-length 24)
			  (encrypt! aes-encrypt!)
			  (decrypt! aes-decrypt!)
			  (key->encrypt-param key->param)
			  (key->decrypt-param key->param)))

(define-macro (hex-values->string-macro str)
   (define (str16->b32 str) (hex-bytes8->bits32 str 0))
   (define gram (regular-grammar ()
		   ("0x" (ignore))
		   ("," (ignore))
		   ((+ xdigit) (string-ref (string-hex-intern (the-string)) 0))
		   ((+ blank) (ignore))
		   (else (let ((char (the-failure)))
			    (if (eof-object? char)
				char
				(error #f "can't happen" char))))))
   (let ((p (open-input-string str))
	 (str (make-string 256)))
      (unwind-protect
	 (let loop ((i 0))
	    (let ((c (read/rp gram p)))
	       (if (eof-object? c)
		   str
		   (begin
		      (string-set! str i c)
		      (loop (+fx i 1))))))
	 (close-input-port p))))

(define *Nb* 4)

(define *state-size* 16)
;; state is 16 bytes long.
;; hardcoded.
(define (make-state::bstring)
   (make-string *state-size*))
;; IMPORTANT: states must be column-major.
(define (state-ref::char s x y)
   (string-ref s (+fx (*fx x 4) y)))
(define (state-set! s x y c::char)
   (string-set! s (+fx (*fx x 4) y) c))
(define (fill-state! state::bstring block::bstring block-pos::long)
   ;; blit-string! is valid as states are column-major.
   (blit-string! block block-pos state 0 *state-size*))
(define (fill-block! block::bstring block-pos state::bstring)
   (blit-string! state 0 block block-pos *state-size*))

(define (aes-encrypt! block::bstring block-pos::long
		      to::bstring to-pos::long
		      param::AES-Param)
   (with-access::AES-Param param (state Nr w)
      (fill-state! state block block-pos)
      (encode-state! state w Nr)
      (fill-block! to to-pos state)))

(define (aes-decrypt! block::bstring block-pos::long
		      to::bstring to-pos::long
		      param::AES-Param)
   (with-access::AES-Param param (state Nr w)
      (fill-state! state block block-pos)
      (decode-state! state w Nr)
      (fill-block! to to-pos state)))

(define *rcon* (hex-values->string-macro
		"0x00 0x01 0x02 0x04 0x08 0x10 0x20 0x40 0x80 0x1b 0x36"))
(define (rcon i) (string-ref *rcon* i))

;; words are 4 bytes long.
(define (word-ref  w i j) (string-ref w (+fx (*fx i 4) j)))
(define (word-set! w i j v) (string-set! w (+fx (*fx i 4) j) v))
   
;; Nr == Number of rounds
(define (key-expansion::bstring key::bstring Nr::long)
   (let* ((Nk (/fx (string-length key) 4))
	  (w (make-string (*fx 4 (*fx *Nb* (+fx Nr 1))))))
      ;; key is first copied verbatim.
      (blit-string! key 0 w 0 (string-length key))

      (let loop ((i Nk))
	 (when (<fx i (*fx *Nb* (+fx Nr 1)))
	    (let ((t0 (word-ref w (-fx i 1) 0))
		  (t1 (word-ref w (-fx i 1) 1))
		  (t2 (word-ref w (-fx i 1) 2))
		  (t3 (word-ref w (-fx i 1) 3)))
	       (when (zerofx? (remainderfx i Nk))
		  (let ((tmp t0))
		     ;; RotWord: [a0,a1,a2,a3] -> [a1,a2,a3,a0]
		     (set! t0 (char-xor (s-box t1) (rcon (/fx i Nk))))
		     (set! t1 (s-box t2))
		     (set! t2 (s-box t3))
		     (set! t3 (s-box tmp))))
	       (when (and (>fx Nk 6)
			  (=fx 4 (remainderfx i Nk)))
		  (set! t0 (s-box t0))
		  (set! t1 (s-box t1))
		  (set! t2 (s-box t2))
		  (set! t3 (s-box t3)))
	       (word-set! w i 0 (char-xor t0 (word-ref w (-fx i Nk) 0)))
	       (word-set! w i 1 (char-xor t1 (word-ref w (-fx i Nk) 1)))
	       (word-set! w i 2 (char-xor t2 (word-ref w (-fx i Nk) 2)))
	       (word-set! w i 3 (char-xor t3 (word-ref w (-fx i Nk) 3)))
	       (loop (+fx i 1)))))
      w))

(define (s-box c)
   (string-ref *s-box* (char->integer c)))

(define (inv-s-box c)
   (string-ref *s-inv-box* (char->integer c)))

(define (sub-bytes! state)
   (let loop ((i 0))
      (when (<fx i *state-size*)
	 (string-set! state i (s-box (string-ref state i)))
	 (loop (+fx i 1)))))

(define (inv-sub-bytes! state)
   (let loop ((i 0))
      (when (<fx i *state-size*)
	 (string-set! state i (inv-s-box (string-ref state i)))
	 (loop (+fx i 1)))))

;; shift to the right.
(define (shift-row! s row by)
   ;; yeah. unrolled by hand...
   (let ((c0 (state-ref s 0 row))
	 (c1 (state-ref s 1 row))
	 (c2 (state-ref s 2 row))
	 (c3 (state-ref s 3 row)))
      (state-set! s by row c0)
      (state-set! s (remainderfx (+fx 1 by) 4) row c1)
      (state-set! s (remainderfx (+fx 2 by) 4) row c2)
      (state-set! s (remainderfx (+fx 3 by) 4) row c3)))

(define (shift-rows! state)
   ;; note: we shift to the right. -> row 1 is shifted by 3 instead of 1
   (shift-row! state 1 3)
   (shift-row! state 2 2)
   (shift-row! state 3 1))

(define (inv-shift-rows! state)
   (shift-row! state 1 1)
   (shift-row! state 2 2)
   (shift-row! state 3 3))

(define-macro (^char^ x . L)
   (cond
      ((null? L) x)
      (else `(char-xor ,x (^char^ ,@L)))))

;; Multiplications is gaulois field.
(define (g*2 c) (string-ref *times-2* (char->integer c)))
(define (g*3 c) (char-xor c (g*2 c)))
(define (g*9 c) (string-ref *times-9* (char->integer c)))
(define (g*b c) (string-ref *times-b* (char->integer c)))
(define (g*d c) (string-ref *times-d* (char->integer c)))
(define (g*e c) (string-ref *times-e* (char->integer c)))
   

(define (mix-columns! state)
   (define (mix-column! s col)
      (let ((c0 (state-ref state col 0))
	    (c1 (state-ref state col 1))
	    (c2 (state-ref state col 2))
	    (c3 (state-ref state col 3)))
	 (state-set! s col 0 (^char^ (g*2 c0) (g*3 c1) c2       c3      ))
	 (state-set! s col 1 (^char^ c0       (g*2 c1) (g*3 c2) c3      ))
	 (state-set! s col 2 (^char^ c0       c1       (g*2 c2) (g*3 c3)))
	 (state-set! s col 3 (^char^ (g*3 c0) c1       c2       (g*2 c3)))))
   (mix-column! state 0)
   (mix-column! state 1)
   (mix-column! state 2)
   (mix-column! state 3))

(define (inv-mix-columns! state)
   (define (inv-mix-column! s col)
      (let ((c0 (state-ref state col 0))
	    (c1 (state-ref state col 1))
	    (c2 (state-ref state col 2))
	    (c3 (state-ref state col 3)))
	 (state-set! s col 0 (^char^ (g*e c0) (g*b c1) (g*d c2) (g*9 c3)))
	 (state-set! s col 1 (^char^ (g*9 c0) (g*e c1) (g*b c2) (g*d c3)))
	 (state-set! s col 2 (^char^ (g*d c0) (g*9 c1) (g*e c2) (g*b c3)))
	 (state-set! s col 3 (^char^ (g*b c0) (g*d c1) (g*9 c2) (g*e c3)))))
   (inv-mix-column! state 0)
   (inv-mix-column! state 1)
   (inv-mix-column! state 2)
   (inv-mix-column! state 3))

(define (add-round-key! state w i)
   (define (add-column-key! s w col i)
      (state-set! s col 0 (char-xor (word-ref w i 0) (state-ref s col 0)))
      (state-set! s col 1 (char-xor (word-ref w i 1) (state-ref s col 1)))
      (state-set! s col 2 (char-xor (word-ref w i 2) (state-ref s col 2)))
      (state-set! s col 3 (char-xor (word-ref w i 3) (state-ref s col 3))))
   (add-column-key! state w 0 i)
   (add-column-key! state w 1 (+fx i 1))
   (add-column-key! state w 2 (+fx i 2))
   (add-column-key! state w 3 (+fx i 3)))
      
;; Nr = Number of rounds.
(define (encode-state! state::bstring w::bstring Nr::long)
   (add-round-key! state w 0)

   (let loop ((round 1))
      (unless (=fx round Nr)
	 (sub-bytes! state)
	 (shift-rows! state)
	 (mix-columns! state)
	 (add-round-key! state w (*fx round *Nb*))
	 (loop (+fx round 1))))

   (sub-bytes! state)
   (shift-rows! state)
   (add-round-key! state w (*fx Nr *Nb*)))

(define (decode-state! state::bstring w::bstring Nr::long)
   (add-round-key! state w (*fx Nr *Nb*))

   (let loop ((round (-fx Nr 1)))
      (unless (=fx round 0)
	 (inv-shift-rows! state)
	 (inv-sub-bytes! state)
	 (add-round-key! state w (*fx round *Nb*))
	 (inv-mix-columns! state)
	 (loop (-fx round 1))))

   (inv-shift-rows! state)
   (inv-sub-bytes! state)
   (add-round-key! state w 0))

;; tables are from: http://code.google.com/p/byte-oriented-aes/downloads/list
;; If the link should go down:
;;    - s-box and s-inv-box are in the AES-spec.
;;    - times-2,3,etc can be precomputed. They are the the result of the
;;      multiplication in the gaulois field. (There is a reason I just copied
;;      the tables)

(define *s-box*
   (hex-values->string-macro
"0x63, 0x7c, 0x77, 0x7b, 0xf2, 0x6b, 0x6f, 0xc5, 0x30, 0x01, 0x67, 0x2b, 0xfe, 0xd7, 0xab, 0x76,
0xca, 0x82, 0xc9, 0x7d, 0xfa, 0x59, 0x47, 0xf0, 0xad, 0xd4, 0xa2, 0xaf, 0x9c, 0xa4, 0x72, 0xc0,
0xb7, 0xfd, 0x93, 0x26, 0x36, 0x3f, 0xf7, 0xcc, 0x34, 0xa5, 0xe5, 0xf1, 0x71, 0xd8, 0x31, 0x15,
0x04, 0xc7, 0x23, 0xc3, 0x18, 0x96, 0x05, 0x9a, 0x07, 0x12, 0x80, 0xe2, 0xeb, 0x27, 0xb2, 0x75,
0x09, 0x83, 0x2c, 0x1a, 0x1b, 0x6e, 0x5a, 0xa0, 0x52, 0x3b, 0xd6, 0xb3, 0x29, 0xe3, 0x2f, 0x84,
0x53, 0xd1, 0x00, 0xed, 0x20, 0xfc, 0xb1, 0x5b, 0x6a, 0xcb, 0xbe, 0x39, 0x4a, 0x4c, 0x58, 0xcf,
0xd0, 0xef, 0xaa, 0xfb, 0x43, 0x4d, 0x33, 0x85, 0x45, 0xf9, 0x02, 0x7f, 0x50, 0x3c, 0x9f, 0xa8,
0x51, 0xa3, 0x40, 0x8f, 0x92, 0x9d, 0x38, 0xf5, 0xbc, 0xb6, 0xda, 0x21, 0x10, 0xff, 0xf3, 0xd2,
0xcd, 0x0c, 0x13, 0xec, 0x5f, 0x97, 0x44, 0x17, 0xc4, 0xa7, 0x7e, 0x3d, 0x64, 0x5d, 0x19, 0x73,
0x60, 0x81, 0x4f, 0xdc, 0x22, 0x2a, 0x90, 0x88, 0x46, 0xee, 0xb8, 0x14, 0xde, 0x5e, 0x0b, 0xdb,
0xe0, 0x32, 0x3a, 0x0a, 0x49, 0x06, 0x24, 0x5c, 0xc2, 0xd3, 0xac, 0x62, 0x91, 0x95, 0xe4, 0x79,
0xe7, 0xc8, 0x37, 0x6d, 0x8d, 0xd5, 0x4e, 0xa9, 0x6c, 0x56, 0xf4, 0xea, 0x65, 0x7a, 0xae, 0x08,
0xba, 0x78, 0x25, 0x2e, 0x1c, 0xa6, 0xb4, 0xc6, 0xe8, 0xdd, 0x74, 0x1f, 0x4b, 0xbd, 0x8b, 0x8a,
0x70, 0x3e, 0xb5, 0x66, 0x48, 0x03, 0xf6, 0x0e, 0x61, 0x35, 0x57, 0xb9, 0x86, 0xc1, 0x1d, 0x9e,
0xe1, 0xf8, 0x98, 0x11, 0x69, 0xd9, 0x8e, 0x94, 0x9b, 0x1e, 0x87, 0xe9, 0xce, 0x55, 0x28, 0xdf,
0x8c, 0xa1, 0x89, 0x0d, 0xbf, 0xe6, 0x42, 0x68, 0x41, 0x99, 0x2d, 0x0f, 0xb0, 0x54, 0xbb, 0x16"))

(define *s-inv-box*
   (hex-values->string-macro
"0x52, 0x09, 0x6a, 0xd5, 0x30, 0x36, 0xa5, 0x38, 0xbf, 0x40, 0xa3, 0x9e, 0x81, 0xf3, 0xd7, 0xfb,
0x7c, 0xe3, 0x39, 0x82, 0x9b, 0x2f, 0xff, 0x87, 0x34, 0x8e, 0x43, 0x44, 0xc4, 0xde, 0xe9, 0xcb,
0x54, 0x7b, 0x94, 0x32, 0xa6, 0xc2, 0x23, 0x3d, 0xee, 0x4c, 0x95, 0x0b, 0x42, 0xfa, 0xc3, 0x4e,
0x08, 0x2e, 0xa1, 0x66, 0x28, 0xd9, 0x24, 0xb2, 0x76, 0x5b, 0xa2, 0x49, 0x6d, 0x8b, 0xd1, 0x25,
0x72, 0xf8, 0xf6, 0x64, 0x86, 0x68, 0x98, 0x16, 0xd4, 0xa4, 0x5c, 0xcc, 0x5d, 0x65, 0xb6, 0x92,
0x6c, 0x70, 0x48, 0x50, 0xfd, 0xed, 0xb9, 0xda, 0x5e, 0x15, 0x46, 0x57, 0xa7, 0x8d, 0x9d, 0x84,
0x90, 0xd8, 0xab, 0x00, 0x8c, 0xbc, 0xd3, 0x0a, 0xf7, 0xe4, 0x58, 0x05, 0xb8, 0xb3, 0x45, 0x06,
0xd0, 0x2c, 0x1e, 0x8f, 0xca, 0x3f, 0x0f, 0x02, 0xc1, 0xaf, 0xbd, 0x03, 0x01, 0x13, 0x8a, 0x6b,
0x3a, 0x91, 0x11, 0x41, 0x4f, 0x67, 0xdc, 0xea, 0x97, 0xf2, 0xcf, 0xce, 0xf0, 0xb4, 0xe6, 0x73,
0x96, 0xac, 0x74, 0x22, 0xe7, 0xad, 0x35, 0x85, 0xe2, 0xf9, 0x37, 0xe8, 0x1c, 0x75, 0xdf, 0x6e,
0x47, 0xf1, 0x1a, 0x71, 0x1d, 0x29, 0xc5, 0x89, 0x6f, 0xb7, 0x62, 0x0e, 0xaa, 0x18, 0xbe, 0x1b,
0xfc, 0x56, 0x3e, 0x4b, 0xc6, 0xd2, 0x79, 0x20, 0x9a, 0xdb, 0xc0, 0xfe, 0x78, 0xcd, 0x5a, 0xf4,
0x1f, 0xdd, 0xa8, 0x33, 0x88, 0x07, 0xc7, 0x31, 0xb1, 0x12, 0x10, 0x59, 0x27, 0x80, 0xec, 0x5f,
0x60, 0x51, 0x7f, 0xa9, 0x19, 0xb5, 0x4a, 0x0d, 0x2d, 0xe5, 0x7a, 0x9f, 0x93, 0xc9, 0x9c, 0xef,
0xa0, 0xe0, 0x3b, 0x4d, 0xae, 0x2a, 0xf5, 0xb0, 0xc8, 0xeb, 0xbb, 0x3c, 0x83, 0x53, 0x99, 0x61,
0x17, 0x2b, 0x04, 0x7e, 0xba, 0x77, 0xd6, 0x26, 0xe1, 0x69, 0x14, 0x63, 0x55, 0x21, 0x0c, 0x7d"))    

(define *times-2* ;; (x & 0x80? 0x1B: 0) ^ (x + x)
   (hex-values->string-macro
"0x00, 0x02, 0x04, 0x06, 0x08, 0x0a, 0x0c, 0x0e, 0x10, 0x12, 0x14, 0x16, 0x18, 0x1a, 0x1c, 0x1e, 
0x20, 0x22, 0x24, 0x26, 0x28, 0x2a, 0x2c, 0x2e, 0x30, 0x32, 0x34, 0x36, 0x38, 0x3a, 0x3c, 0x3e, 
0x40, 0x42, 0x44, 0x46, 0x48, 0x4a, 0x4c, 0x4e, 0x50, 0x52, 0x54, 0x56, 0x58, 0x5a, 0x5c, 0x5e, 
0x60, 0x62, 0x64, 0x66, 0x68, 0x6a, 0x6c, 0x6e, 0x70, 0x72, 0x74, 0x76, 0x78, 0x7a, 0x7c, 0x7e, 
0x80, 0x82, 0x84, 0x86, 0x88, 0x8a, 0x8c, 0x8e, 0x90, 0x92, 0x94, 0x96, 0x98, 0x9a, 0x9c, 0x9e, 
0xa0, 0xa2, 0xa4, 0xa6, 0xa8, 0xaa, 0xac, 0xae, 0xb0, 0xb2, 0xb4, 0xb6, 0xb8, 0xba, 0xbc, 0xbe, 
0xc0, 0xc2, 0xc4, 0xc6, 0xc8, 0xca, 0xcc, 0xce, 0xd0, 0xd2, 0xd4, 0xd6, 0xd8, 0xda, 0xdc, 0xde, 
0xe0, 0xe2, 0xe4, 0xe6, 0xe8, 0xea, 0xec, 0xee, 0xf0, 0xf2, 0xf4, 0xf6, 0xf8, 0xfa, 0xfc, 0xfe, 
0x1b, 0x19, 0x1f, 0x1d, 0x13, 0x11, 0x17, 0x15, 0x0b, 0x09, 0x0f, 0x0d, 0x03, 0x01, 0x07, 0x05, 
0x3b, 0x39, 0x3f, 0x3d, 0x33, 0x31, 0x37, 0x35, 0x2b, 0x29, 0x2f, 0x2d, 0x23, 0x21, 0x27, 0x25, 
0x5b, 0x59, 0x5f, 0x5d, 0x53, 0x51, 0x57, 0x55, 0x4b, 0x49, 0x4f, 0x4d, 0x43, 0x41, 0x47, 0x45, 
0x7b, 0x79, 0x7f, 0x7d, 0x73, 0x71, 0x77, 0x75, 0x6b, 0x69, 0x6f, 0x6d, 0x63, 0x61, 0x67, 0x65, 
0x9b, 0x99, 0x9f, 0x9d, 0x93, 0x91, 0x97, 0x95, 0x8b, 0x89, 0x8f, 0x8d, 0x83, 0x81, 0x87, 0x85, 
0xbb, 0xb9, 0xbf, 0xbd, 0xb3, 0xb1, 0xb7, 0xb5, 0xab, 0xa9, 0xaf, 0xad, 0xa3, 0xa1, 0xa7, 0xa5, 
0xdb, 0xd9, 0xdf, 0xdd, 0xd3, 0xd1, 0xd7, 0xd5, 0xcb, 0xc9, 0xcf, 0xcd, 0xc3, 0xc1, 0xc7, 0xc5, 
0xfb, 0xf9, 0xff, 0xfd, 0xf3, 0xf1, 0xf7, 0xf5, 0xeb, 0xe9, 0xef, 0xed, 0xe3, 0xe1, 0xe7, 0xe5"))

;; no need for times-3: x ^ *times-2*[x]


(define *times-9*
   (hex-values->string-macro
"0x00, 0x09, 0x12, 0x1b, 0x24, 0x2d, 0x36, 0x3f, 0x48, 0x41, 0x5a, 0x53, 0x6c, 0x65, 0x7e, 0x77, 
0x90, 0x99, 0x82, 0x8b, 0xb4, 0xbd, 0xa6, 0xaf, 0xd8, 0xd1, 0xca, 0xc3, 0xfc, 0xf5, 0xee, 0xe7, 
0x3b, 0x32, 0x29, 0x20, 0x1f, 0x16, 0x0d, 0x04, 0x73, 0x7a, 0x61, 0x68, 0x57, 0x5e, 0x45, 0x4c, 
0xab, 0xa2, 0xb9, 0xb0, 0x8f, 0x86, 0x9d, 0x94, 0xe3, 0xea, 0xf1, 0xf8, 0xc7, 0xce, 0xd5, 0xdc, 
0x76, 0x7f, 0x64, 0x6d, 0x52, 0x5b, 0x40, 0x49, 0x3e, 0x37, 0x2c, 0x25, 0x1a, 0x13, 0x08, 0x01, 
0xe6, 0xef, 0xf4, 0xfd, 0xc2, 0xcb, 0xd0, 0xd9, 0xae, 0xa7, 0xbc, 0xb5, 0x8a, 0x83, 0x98, 0x91, 
0x4d, 0x44, 0x5f, 0x56, 0x69, 0x60, 0x7b, 0x72, 0x05, 0x0c, 0x17, 0x1e, 0x21, 0x28, 0x33, 0x3a, 
0xdd, 0xd4, 0xcf, 0xc6, 0xf9, 0xf0, 0xeb, 0xe2, 0x95, 0x9c, 0x87, 0x8e, 0xb1, 0xb8, 0xa3, 0xaa, 
0xec, 0xe5, 0xfe, 0xf7, 0xc8, 0xc1, 0xda, 0xd3, 0xa4, 0xad, 0xb6, 0xbf, 0x80, 0x89, 0x92, 0x9b, 
0x7c, 0x75, 0x6e, 0x67, 0x58, 0x51, 0x4a, 0x43, 0x34, 0x3d, 0x26, 0x2f, 0x10, 0x19, 0x02, 0x0b, 
0xd7, 0xde, 0xc5, 0xcc, 0xf3, 0xfa, 0xe1, 0xe8, 0x9f, 0x96, 0x8d, 0x84, 0xbb, 0xb2, 0xa9, 0xa0, 
0x47, 0x4e, 0x55, 0x5c, 0x63, 0x6a, 0x71, 0x78, 0x0f, 0x06, 0x1d, 0x14, 0x2b, 0x22, 0x39, 0x30, 
0x9a, 0x93, 0x88, 0x81, 0xbe, 0xb7, 0xac, 0xa5, 0xd2, 0xdb, 0xc0, 0xc9, 0xf6, 0xff, 0xe4, 0xed, 
0x0a, 0x03, 0x18, 0x11, 0x2e, 0x27, 0x3c, 0x35, 0x42, 0x4b, 0x50, 0x59, 0x66, 0x6f, 0x74, 0x7d, 
0xa1, 0xa8, 0xb3, 0xba, 0x85, 0x8c, 0x97, 0x9e, 0xe9, 0xe0, 0xfb, 0xf2, 0xcd, 0xc4, 0xdf, 0xd6, 
0x31, 0x38, 0x23, 0x2a, 0x15, 0x1c, 0x07, 0x0e, 0x79, 0x70, 0x6b, 0x62, 0x5d, 0x54, 0x4f, 0x46"))

(define *times-b*
   (hex-values->string-macro
"0x00, 0x0b, 0x16, 0x1d, 0x2c, 0x27, 0x3a, 0x31, 0x58, 0x53, 0x4e, 0x45, 0x74, 0x7f, 0x62, 0x69, 
0xb0, 0xbb, 0xa6, 0xad, 0x9c, 0x97, 0x8a, 0x81, 0xe8, 0xe3, 0xfe, 0xf5, 0xc4, 0xcf, 0xd2, 0xd9, 
0x7b, 0x70, 0x6d, 0x66, 0x57, 0x5c, 0x41, 0x4a, 0x23, 0x28, 0x35, 0x3e, 0x0f, 0x04, 0x19, 0x12, 
0xcb, 0xc0, 0xdd, 0xd6, 0xe7, 0xec, 0xf1, 0xfa, 0x93, 0x98, 0x85, 0x8e, 0xbf, 0xb4, 0xa9, 0xa2, 
0xf6, 0xfd, 0xe0, 0xeb, 0xda, 0xd1, 0xcc, 0xc7, 0xae, 0xa5, 0xb8, 0xb3, 0x82, 0x89, 0x94, 0x9f, 
0x46, 0x4d, 0x50, 0x5b, 0x6a, 0x61, 0x7c, 0x77, 0x1e, 0x15, 0x08, 0x03, 0x32, 0x39, 0x24, 0x2f, 
0x8d, 0x86, 0x9b, 0x90, 0xa1, 0xaa, 0xb7, 0xbc, 0xd5, 0xde, 0xc3, 0xc8, 0xf9, 0xf2, 0xef, 0xe4, 
0x3d, 0x36, 0x2b, 0x20, 0x11, 0x1a, 0x07, 0x0c, 0x65, 0x6e, 0x73, 0x78, 0x49, 0x42, 0x5f, 0x54, 
0xf7, 0xfc, 0xe1, 0xea, 0xdb, 0xd0, 0xcd, 0xc6, 0xaf, 0xa4, 0xb9, 0xb2, 0x83, 0x88, 0x95, 0x9e, 
0x47, 0x4c, 0x51, 0x5a, 0x6b, 0x60, 0x7d, 0x76, 0x1f, 0x14, 0x09, 0x02, 0x33, 0x38, 0x25, 0x2e, 
0x8c, 0x87, 0x9a, 0x91, 0xa0, 0xab, 0xb6, 0xbd, 0xd4, 0xdf, 0xc2, 0xc9, 0xf8, 0xf3, 0xee, 0xe5, 
0x3c, 0x37, 0x2a, 0x21, 0x10, 0x1b, 0x06, 0x0d, 0x64, 0x6f, 0x72, 0x79, 0x48, 0x43, 0x5e, 0x55, 
0x01, 0x0a, 0x17, 0x1c, 0x2d, 0x26, 0x3b, 0x30, 0x59, 0x52, 0x4f, 0x44, 0x75, 0x7e, 0x63, 0x68, 
0xb1, 0xba, 0xa7, 0xac, 0x9d, 0x96, 0x8b, 0x80, 0xe9, 0xe2, 0xff, 0xf4, 0xc5, 0xce, 0xd3, 0xd8, 
0x7a, 0x71, 0x6c, 0x67, 0x56, 0x5d, 0x40, 0x4b, 0x22, 0x29, 0x34, 0x3f, 0x0e, 0x05, 0x18, 0x13, 
0xca, 0xc1, 0xdc, 0xd7, 0xe6, 0xed, 0xf0, 0xfb, 0x92, 0x99, 0x84, 0x8f, 0xbe, 0xb5, 0xa8, 0xa3"))

(define *times-d*
   (hex-values->string-macro
"0x00, 0x0d, 0x1a, 0x17, 0x34, 0x39, 0x2e, 0x23, 0x68, 0x65, 0x72, 0x7f, 0x5c, 0x51, 0x46, 0x4b, 
0xd0, 0xdd, 0xca, 0xc7, 0xe4, 0xe9, 0xfe, 0xf3, 0xb8, 0xb5, 0xa2, 0xaf, 0x8c, 0x81, 0x96, 0x9b, 
0xbb, 0xb6, 0xa1, 0xac, 0x8f, 0x82, 0x95, 0x98, 0xd3, 0xde, 0xc9, 0xc4, 0xe7, 0xea, 0xfd, 0xf0, 
0x6b, 0x66, 0x71, 0x7c, 0x5f, 0x52, 0x45, 0x48, 0x03, 0x0e, 0x19, 0x14, 0x37, 0x3a, 0x2d, 0x20, 
0x6d, 0x60, 0x77, 0x7a, 0x59, 0x54, 0x43, 0x4e, 0x05, 0x08, 0x1f, 0x12, 0x31, 0x3c, 0x2b, 0x26, 
0xbd, 0xb0, 0xa7, 0xaa, 0x89, 0x84, 0x93, 0x9e, 0xd5, 0xd8, 0xcf, 0xc2, 0xe1, 0xec, 0xfb, 0xf6, 
0xd6, 0xdb, 0xcc, 0xc1, 0xe2, 0xef, 0xf8, 0xf5, 0xbe, 0xb3, 0xa4, 0xa9, 0x8a, 0x87, 0x90, 0x9d, 
0x06, 0x0b, 0x1c, 0x11, 0x32, 0x3f, 0x28, 0x25, 0x6e, 0x63, 0x74, 0x79, 0x5a, 0x57, 0x40, 0x4d, 
0xda, 0xd7, 0xc0, 0xcd, 0xee, 0xe3, 0xf4, 0xf9, 0xb2, 0xbf, 0xa8, 0xa5, 0x86, 0x8b, 0x9c, 0x91, 
0x0a, 0x07, 0x10, 0x1d, 0x3e, 0x33, 0x24, 0x29, 0x62, 0x6f, 0x78, 0x75, 0x56, 0x5b, 0x4c, 0x41, 
0x61, 0x6c, 0x7b, 0x76, 0x55, 0x58, 0x4f, 0x42, 0x09, 0x04, 0x13, 0x1e, 0x3d, 0x30, 0x27, 0x2a, 
0xb1, 0xbc, 0xab, 0xa6, 0x85, 0x88, 0x9f, 0x92, 0xd9, 0xd4, 0xc3, 0xce, 0xed, 0xe0, 0xf7, 0xfa, 
0xb7, 0xba, 0xad, 0xa0, 0x83, 0x8e, 0x99, 0x94, 0xdf, 0xd2, 0xc5, 0xc8, 0xeb, 0xe6, 0xf1, 0xfc, 
0x67, 0x6a, 0x7d, 0x70, 0x53, 0x5e, 0x49, 0x44, 0x0f, 0x02, 0x15, 0x18, 0x3b, 0x36, 0x21, 0x2c, 
0x0c, 0x01, 0x16, 0x1b, 0x38, 0x35, 0x22, 0x2f, 0x64, 0x69, 0x7e, 0x73, 0x50, 0x5d, 0x4a, 0x47, 
0xdc, 0xd1, 0xc6, 0xcb, 0xe8, 0xe5, 0xf2, 0xff, 0xb4, 0xb9, 0xae, 0xa3, 0x80, 0x8d, 0x9a, 0x97"))


(define *times-e*
   (hex-values->string-macro
"0x00, 0x0e, 0x1c, 0x12, 0x38, 0x36, 0x24, 0x2a, 0x70, 0x7e, 0x6c, 0x62, 0x48, 0x46, 0x54, 0x5a, 
0xe0, 0xee, 0xfc, 0xf2, 0xd8, 0xd6, 0xc4, 0xca, 0x90, 0x9e, 0x8c, 0x82, 0xa8, 0xa6, 0xb4, 0xba, 
0xdb, 0xd5, 0xc7, 0xc9, 0xe3, 0xed, 0xff, 0xf1, 0xab, 0xa5, 0xb7, 0xb9, 0x93, 0x9d, 0x8f, 0x81, 
0x3b, 0x35, 0x27, 0x29, 0x03, 0x0d, 0x1f, 0x11, 0x4b, 0x45, 0x57, 0x59, 0x73, 0x7d, 0x6f, 0x61, 
0xad, 0xa3, 0xb1, 0xbf, 0x95, 0x9b, 0x89, 0x87, 0xdd, 0xd3, 0xc1, 0xcf, 0xe5, 0xeb, 0xf9, 0xf7, 
0x4d, 0x43, 0x51, 0x5f, 0x75, 0x7b, 0x69, 0x67, 0x3d, 0x33, 0x21, 0x2f, 0x05, 0x0b, 0x19, 0x17, 
0x76, 0x78, 0x6a, 0x64, 0x4e, 0x40, 0x52, 0x5c, 0x06, 0x08, 0x1a, 0x14, 0x3e, 0x30, 0x22, 0x2c, 
0x96, 0x98, 0x8a, 0x84, 0xae, 0xa0, 0xb2, 0xbc, 0xe6, 0xe8, 0xfa, 0xf4, 0xde, 0xd0, 0xc2, 0xcc, 
0x41, 0x4f, 0x5d, 0x53, 0x79, 0x77, 0x65, 0x6b, 0x31, 0x3f, 0x2d, 0x23, 0x09, 0x07, 0x15, 0x1b, 
0xa1, 0xaf, 0xbd, 0xb3, 0x99, 0x97, 0x85, 0x8b, 0xd1, 0xdf, 0xcd, 0xc3, 0xe9, 0xe7, 0xf5, 0xfb, 
0x9a, 0x94, 0x86, 0x88, 0xa2, 0xac, 0xbe, 0xb0, 0xea, 0xe4, 0xf6, 0xf8, 0xd2, 0xdc, 0xce, 0xc0, 
0x7a, 0x74, 0x66, 0x68, 0x42, 0x4c, 0x5e, 0x50, 0x0a, 0x04, 0x16, 0x18, 0x32, 0x3c, 0x2e, 0x20, 
0xec, 0xe2, 0xf0, 0xfe, 0xd4, 0xda, 0xc8, 0xc6, 0x9c, 0x92, 0x80, 0x8e, 0xa4, 0xaa, 0xb8, 0xb6, 
0x0c, 0x02, 0x10, 0x1e, 0x34, 0x3a, 0x28, 0x26, 0x7c, 0x72, 0x60, 0x6e, 0x44, 0x4a, 0x58, 0x56, 
0x37, 0x39, 0x2b, 0x25, 0x0f, 0x01, 0x13, 0x1d, 0x47, 0x49, 0x5b, 0x55, 0x7f, 0x71, 0x63, 0x6d, 
0xd7, 0xd9, 0xcb, 0xc5, 0xef, 0xe1, 0xf3, 0xfd, 0xa7, 0xa9, 0xbb, 0xb5, 0x9f, 0x91, 0x83, 0x8d"))
