/**
 * This is part of an XML patch/diff library.
 *
 * Copyright (C) 2005 Nokia Corporation.
 *
 * Contact: Jari Urpalainen <jari.urpalainen@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * @date Created: Tue May 24 13:28:44 EEST 2005
 */

#include "config.h"

#include <stddef.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <assert.h>
#include <signal.h>

#include <ctype.h>
#include <unistd.h>

#include <glib.h>

#include <libxml/tree.h>

#include "xml_diff.h"
#include "xml_patch.h"


static
void usage()
{
    printf ("xml_diff [-s] [-v] [-b] -f from-doc -t to-doc [-o output-doc] \n"
	    "       v == verbose\n"
	    "       s == shorter requests,\n"
	    "       b == bug testing with patch\n" \
	    "       o == dumb output to document\n"
	    "returns 0 if documents don't differ, -1 for an error\n"
	    "version: %s\n", PACKAGE_VERSION);
}

/** patching, don't care about request namespaces */
static int
patch (xmlDocPtr doc, xmlNodePtr node)
{
    int rc = 0;

    if (!strcmp ((char*)node->name, "add"))
	rc = xml_patch_add (doc, node);
    else if (!strcmp ((char*)node->name, "replace"))
	rc = xml_patch_replace (doc, node);
    else if (!strcmp ((char*)node->name, "remove"))
	rc = xml_patch_remove (doc, node);
    else
	rc = -1;
    return rc;
}


int
main (int argc, char *argv[])
{
    int opt, rc = 0, fPrint = 0, fShort = 0, fBugs = 0;
    const char *pcszDocFrom = NULL, *pcszDocTo = NULL, *pcszOutput = NULL;
    xmlDocPtr docFrom, docTo, docDiff;

    while ((opt = getopt (argc, argv, "vsbf:t:o:")) != -1) {
	switch (opt) {
	case ':':
	case 'h':
	case '?':
	    usage ();
	    return -1;

	case 'v':
	    fPrint = 1;
	    break;

	case 's':
	    fShort = 1;
	    break;

	case 'b':
	    fBugs = 1;
	    break;

	case 'f':
	    pcszDocFrom = optarg;
	    break;

	case 't':
	    pcszDocTo = optarg;
	    break;

	case 'o':
	    pcszOutput = optarg;
	    break;
	}
    }
    if (!pcszDocTo || !pcszDocFrom) {
	usage ();
	return -1;
    }
    docDiff = xmlNewDoc ((const xmlChar*)"1.0");

    docDiff->children = xmlNewDocNode (docDiff, NULL, (const xmlChar*)"changes", NULL);
    xmlSetNs (docDiff->children,
	      xmlNewNs (docDiff->children,
              (const xmlChar*)"urn:xml-changes",
              (const xmlChar*)"x"));

    docFrom = xmlParseFile (pcszDocFrom);
    docTo = xmlParseFile (pcszDocTo);

    rc = xml_exec_diff (docFrom, docTo, fPrint, fShort, docDiff->children);

    if (fPrint && rc > 0)
	xmlDocDump (stdout, docDiff);

    if (pcszOutput && rc > 0)
	xmlSaveFile (pcszOutput, docDiff);

    if (rc > 0 && fBugs) {
	xmlNodePtr node = xmlDocGetRootElement (docDiff);

	for (node = node ? node->children : NULL;
	     node; node = node->next) {

	    if (node->type == XML_ELEMENT_NODE) {
		rc = patch (docFrom, node);

		if (rc) {
		    fprintf (stderr, "Patch could not be applied with the produced diff file !\n");
		    break;
		}
	    }
	}
	if (!rc) {
	    rc = xml_exec_diff (docFrom, docTo, FALSE, fShort, docDiff ? docDiff->children : NULL);

	    if (rc) {
		fprintf (stderr, "Additional patching test FAILED !\n");
	    }
	    else {
		fprintf (stdout, "Additional patching test succeeded\n");
	    }
	}
    }
    xmlFreeDoc (docFrom);
    xmlFreeDoc (docTo);
    xmlFreeDoc (docDiff);

    xmlCleanupParser ();

    return rc;
}

