// $Id$

// Fish Supper
// Copyright 2006, 2007, 2009, 2010 Matthew Clarke <mafferyew@googlemail.com>
//
// This file is part of Fish Supper.
//
// Fish Supper is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Fish Supper is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Fish Supper.  If not, see <http://www.gnu.org/licenses/>.




#include "Settings.h"




// *******************
// *** CONSTRUCTOR ***
// *******************
FS::Settings::Settings()
{
    // default values
    fullscreen = false;
    joystick_selected = false;
    sound = true;
    furthest_level = 1;
    
    read_config_file();
    
} // FS::Settings::Settings

// ******************
// *** DESTRUCTOR ***
// ******************
FS::Settings::~Settings()
{
    // By time this gets called, fs_dir has become corrupted.
    //write_config_file();
    
} // FS::Settings::~Settings




// ************************
// *** MEMBER FUNCTIONS ***
// ************************

// **************************************************

void FS::Settings::read_config_file()
{
    std::string my_file = fs_dir;
    my_file += "/config.txt";
    
    std::ifstream fin( my_file.c_str() );
    if ( !fin.is_open() )
    {
        std::cerr << "Can't open " << my_file << " for reading. Using defaults.\n";
        return;
    } // if
    
    std::string my_line;
    while ( getline(fin, my_line) )
    {
        // split this into two, each side of '='
        int split_index = my_line.find('=');
        std::string attr = my_line.substr(0, split_index);
        int val = atoi( (my_line.substr(split_index + 1)).c_str() );
        
        if ( attr == "fullscreen" )
        {
            fullscreen = (bool) val;
        }
        else if ( (attr == "joystick_selected") && joystick_available )
        {
            // If no joysticks are available, this setting remains at 
            // default value (i.e. false).
            joystick_selected = (bool) val;
        }
        else if ( attr == "sound" )
        {
            sound = (bool) val;
        } 
        else if ( attr == "furthest_level" )
        {
            furthest_level = val;
        } // if ... else
    } // while
    
    fin.close();
    
} // FS::Settings::read_config_file

// **************************************************
        
void FS::Settings::write_config_file() 
{
    std::string my_file = fs_dir;
    my_file += "/config.txt";
    
    std::ofstream fout( my_file.c_str() );
    if ( !fout.is_open() )
    {
        std::cerr << "Can't open " << my_file << " for writing. " <<
                "Unable to save settings.\n";
        return;
    } // if
    
    fout << "joystick_selected=" << ((joystick_selected) ? "1" : "0") << std::endl;
    fout << "sound=" << ((sound) ? "1" : "0") << std::endl;
    fout << "fullscreen=" << ((fullscreen) ? "1" : "0") << std::endl;
    fout << "furthest_level=" << furthest_level << std::endl;
    
    fout.close();
    
} // FS::Settings::write_config_file

// **************************************************

void FS::Settings::print_settings() const
{
    std::cout << "fullscreen=" << fullscreen << std::endl;
    std::cout << "joystick_selected=" << joystick_selected << std::endl;
    std::cout << "sound=" << sound << std::endl;
    std::cout << "furthest_level=" << furthest_level << std::endl;
    
} // FS::Settings::print_settings

// **************************************************

void FS::Settings::set_furthest_level(int lev)
{
    if ( lev > furthest_level )
    {
        furthest_level = lev;
        if (furthest_level > NUM_LEVELS)    // this happens when you finish the last level
        {
            furthest_level = NUM_LEVELS;
        } // if
    } // if
    
} // FS::Settings::set_furthest_level
 
// **************************************************

void FS::Settings::set_joystick_selected(bool b)
{
    if ( joystick_available )
    {
        joystick_selected = b;
    } 
    else
    {
        joystick_selected = false;
    } // if ... else
    
} // FS::Settings::set_joystick_selected
 
        
