// $Id: SFont_gl.cpp,v 1.1 2009-01-31 11:28:39 matthew Exp $

// Fish Supper
// Copyright 2006, 2007, 2009, 2010 Matthew Clarke <mafferyew@googlemail.com>
//
// This file is part of Fish Supper.
//
// Fish Supper is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Fish Supper is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Fish Supper.  If not, see <http://www.gnu.org/licenses/>.



#include "SFont_gl.h"




// *******************
// *** CONSTRUCTOR ***
// *******************

Mafferyew::SFont_gl::SFont_gl(const std::string& image_file) :
        spacing(DEFAULT_SPACING),
        font_height(0)  // can't set real value till file opened
{
    init(image_file);

} // Mafferyew::SFont_gl::SFont_gl


// ******************
// *** DESTRUCTOR ***
// ******************

Mafferyew::SFont_gl::~SFont_gl()
{
    // cleanup
    std::vector<Font_char>::iterator iter = my_font_chars.begin();
    while ( iter != my_font_chars.end() )
    {
        glDeleteTextures( 1, &(*iter).texture );
        ++iter;
    } // while
    
} // Mafferyew::SFont_gl::~SFont_gl




// ************************
// *** MEMBER FUNCTIONS ***
// ************************

// **************************************************

void Mafferyew::SFont_gl::write(int x, 
                             int y, 
                             const std::string& text) const
{
    glEnable(GL_BLEND); 
    glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
    
    for ( unsigned int i = 0; i < text.size(); ++i )
    {
        char c = text.at(i);
        int index = (c - BEGIN_ASCII);
        
        if ( index < 0 || index >= NUM_FONT_CHARS )
        {
            x += spacing;
            continue;
        } // if
        
        glBindTexture(GL_TEXTURE_2D, my_font_chars[index].texture);
        glBegin(GL_QUADS);
            glTexCoord2i(0, 0);
            glVertex3f(x, y, 0.0);

            glTexCoord2i(1, 0);
            glVertex3f(x + my_font_chars[index].texture_w, 
                       y, 
                       0.0);

            glTexCoord2i(1, 1);
            glVertex3f(x + my_font_chars[index].texture_w, 
                       y + my_font_chars[index].texture_h, 
                       0.0);

            glTexCoord2i(0, 1);
            glVertex3f(x, 
                       y + my_font_chars[index].texture_h, 
                       0.0);
        glEnd();
        
        x += my_font_chars[index].visible_w;
    } // for
    
} // Mafferyew::SFont_gl::write

// **************************************************

void Mafferyew::SFont_gl::write_center(int screen_width, 
                                    int y, 
                                    const std::string& text) const
{
    int text_w = text_width(text);
    int x = screen_width - text_w;
    x /= 2;
    write(x, y, text);
    
} // Mafferyew::SFont_gl::write_center

// **************************************************

int Mafferyew::SFont_gl::text_width(const std::string& text) const
{
    int width = 0;
    
    for ( unsigned int i = 0; i < text.size(); ++i )
    {
        char c = text.at(i);
        int index = (c - BEGIN_ASCII);
        
        if ( index < 0 || index >= NUM_FONT_CHARS )
        {
            width += spacing;
        } 
        else
        {
            width += my_font_chars[index].visible_w;
        } // if ... else
    } // for
    
    return width;
    
} // Mafferyew::SFont_gl::text_width

// **************************************************

void Mafferyew::SFont_gl::set_spacing(int spc)
{
    if (spc < MIN_SPACING)
    {
        spacing = MIN_SPACING;
    }
    else if (spc > MAX_SPACING)
    {
        spacing = MAX_SPACING;
    }
    else
    {
        spacing = spc;
    } // if ... else
    
} // Mafferyew::SFont_gl::set_spacing

// **************************************************

void Mafferyew::SFont_gl::init(const std::string& image_file)
{
    SDL_Surface* image = IMG_Load( image_file.c_str() );
    SDL_SetAlpha(image, 0, 0);
    font_height = image->h - 1;
    
    int x = 0;
    int begin = 0;
    int end = 0;
    Uint32 pink = SDL_MapRGB(image->format, 255, 0, 255);
    
    while ( x < image->w )
    {
        if ( Useful_gl::get_pixel(image, x, 0) != pink )
        {
            // first clear pixel found - this is start of a character
            begin = x;
            
            do
            {
                ++x;
            } while ( Useful_gl::get_pixel(image, x, 0) != pink );
            end = x;    // index of first pink pixel after character
            
            // create texture for character - [begin,end)
            create_font_char(image, begin, end);
        } // if            
        ++x;
    } // while
    
    SDL_FreeSurface(image);
    
} // Mafferyew::SFont_gl::init

// **************************************************

void Mafferyew::SFont_gl::create_font_char(SDL_Surface* image, 
                                        int begin_x, 
                                        int end_x)
{
    int font_char_w = end_x - begin_x;
    int conv_w = Useful_gl::power_of_two(font_char_w);
    int conv_h = Useful_gl::power_of_two(font_height);    
    SDL_Surface* converted;
    
#if SDL_BYTEORDER == SDL_BIG_ENDIAN
    converted = SDL_CreateRGBSurface(SDL_SWSURFACE,
                                     conv_w, conv_h,
                                     image->format->BitsPerPixel,
                                     0xff000000, 0x00ff0000, 0x0000ff00, 0x000000ff);
#else
    converted = SDL_CreateRGBSurface(SDL_SWSURFACE,
                                     conv_w, conv_h,
                                     image->format->BitsPerPixel,
                                     0x000000ff, 0x0000ff00, 0x00ff0000, 0xff000000);
#endif
    
    SDL_Rect src_rect = { begin_x, 1, font_char_w, font_height };
    SDL_BlitSurface(image, &src_rect, converted, 0);
    
    GLuint texture;
    glGenTextures(1, &texture);
    glBindTexture(GL_TEXTURE_2D, texture);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
    glTexImage2D(GL_TEXTURE_2D, 
                 0, // level - 0 unless multiple resolutions 
                 converted->format->BytesPerPixel, 
                 converted->w, converted->h, 
                 0, // border 
                 GL_RGBA, 
                 GL_UNSIGNED_BYTE, 
                 converted->pixels);
    
    SDL_FreeSurface(converted);

    Font_char fc = { texture, font_char_w, conv_w, conv_h };
    my_font_chars.push_back(fc);
   
} // Mafferyew::SFont_gl::create_font_char

// **************************************************




