#include <unistd.h>
#include <stdlib.h>
#include <string.h>

#include "gnet_lib.h"
#include "gnet_channel.h"
#include "gnet_msg.h"
#include "gnet_proto.h"

static inline int
compare(struct bnode *n1, char *guid){
    int i;

    for(i = 0; i < 16; i++)
	if(n1->guid[i] < guid[i])
	    return -1;
	else if(n1->guid[i] > guid[i])
	    return 1;
	    
    return 0;
}

static inline int
find(struct bnode *root, struct bnode **node, char *guid){
    int res;

    *node = NULL;

    while(root){
	*node = root;
	
	if(!(res = compare(root, guid))){
	    TRACE("node found");
	    return 0;
	}

	if(res < 0)
	    root = root->left;
	else
	    root = root->right;
    }

    TRACE("node not found");
    
    return -1;
}

static inline int
add(struct bnode *node, struct bnode *root){
    struct bnode *n;

    if(find(root, &n, node->guid) == 0){
	WARN("guid already in tree!");
	return -1;
    }

    if(n == NULL)
	return -1;

    if(compare(n, node->guid) < 0){
	n->left = node;
	node->plink = &n->left;
    }else{
	n->right = node;
	node->plink = &n->right;
    }

    return 0;
}

static inline void
delete(struct bnode *bnode){
    struct bnode *n;

    if(bnode->left == NULL){
	*(bnode->plink) = bnode->right;
	if(bnode->right)
	    bnode->right->plink = bnode->plink;

	free(bnode);
    }else

	if(bnode->right == NULL){
	    *(bnode->plink) = bnode->left;
	    bnode->left->plink = bnode->plink;
	    
	    free(bnode);
	}else{

	    for(n = bnode->left; n->right; n = n->right);
    
	    bnode->chan = n->chan;
	    bnode->list = n->list;
	    memcpy(bnode->guid, n->guid, 16);
	    
	    *(n->plink) = n->left;
	    if(n->left)
		n->left->plink = n->plink;

	    free(n);
	}
}

int
gnet_delete_guid(struct gnet *gnet, char *guid){
    struct bnode *node;

    if(find(&gnet->g_guids_root, &node, guid) < 0)
	return -1;

    list_del(&node->list);
    delete(node);
    
    gnet->g_guids--;

    return 0;
}

int
gnet_save_guid(struct gnet *gnet, struct channel *chan, char *guid){
    struct bnode *new;


    if(!(new = malloc(sizeof(struct bnode))))
	return -1;

    memset(new, 0, sizeof(struct bnode));
	
    new->chan = chan;
    memcpy(new->guid, guid, 16);

    if(add(new, &gnet->g_guids_root) < 0){
	free(new);
	return -1;
    }

    list_add(&new->list, &chan->c_guids);
    gnet->g_guids++;

    return 0;
}

void
gnet_make_guid(char *guid){
    int i;

    for(i = 0; i < 16; i++)
	guid[i] = (char)((((float)random())/RAND_MAX)*256);

    guid[8] = 0xff;
    guid[15] = 0x00;
}

struct message*
gnet_create_message(char *guid, unsigned char type, unsigned char ttl, unsigned char hops, unsigned short data_len){
    struct message *msg;

    if(!(msg = malloc(sizeof(struct message))))
	return NULL;

    memset(msg, 0, sizeof(struct message));

    if(!(msg->m_data = malloc(GNET_HDR_SIZE + data_len))){
	free(msg);
	return NULL;
    } 

    if(guid)
	memcpy(msg->m_data + GNET_FIELD_ID, guid, 16);
    else
	gnet_make_guid(msg->m_data + GNET_FIELD_ID);

    *(msg->m_data + GNET_FIELD_TYPE) = type;
    *(msg->m_data + GNET_FIELD_TTL) = ttl;
    *(msg->m_data + GNET_FIELD_HOPS) = hops;
    *(uint32_t*)(msg->m_data + GNET_FIELD_SIZE) = HTOGS(data_len);

    return msg;
} 

void
gnet_delete_message(struct message *msg){
    if(--msg->m_count <= 0){
//	TRACE("deleting message");
	free(msg->m_data);
	free(msg);
    }
}

int
gnet_add_message(struct gnet *gnet, struct channel *chan, struct message *msg){
    struct msgref *ref;

    if(!(ref = malloc(sizeof(struct msgref))))
	return -1;

    ref->msg = msg;
    msg->m_count++;

    list_add_tail(&ref->list, &chan->c_out);
    
    return 0;
}

int
gnet_deliver_message_all(struct gnet *gnet, struct channel *skip, struct message *msg){
    struct list_head *p;
    struct channel *chan;
    int delivered = 0;

    list_for_each(p, &gnet->g_channels){
	chan = list_entry(p, struct channel, c_list);
	if((chan->c_state == CHANNEL_STATE_CONNECTED) && (chan != skip)){
//	    TRACE("adding message to channel queue...");
	    delivered += gnet_add_message(gnet, chan, msg);
	    gnet_test_wr(gnet, chan);
	}	    
    }

    return delivered;
}

int
gnet_deliver_message_one(struct gnet *gnet, struct channel *chan, struct message *msg){
    if(chan->c_state == CHANNEL_STATE_CONNECTED){
//	    TRACE("adding message to channel queue...");
	    gnet_test_wr(gnet, chan);
	    return gnet_add_message(gnet, chan, msg);
    }else
	return -1;    
}

int
gnet_deliver_message_guid(struct gnet *gnet, struct message *msg, char *guid){
    struct bnode *node;
    
    if(find(&gnet->g_guids_root, &node, guid) < 0){
	TRACE("message guid not found");
	return -1;
    }

    return gnet_deliver_message_one(gnet, node->chan, msg);
}

void
gnet_trace_message(char *msg){
    TRACE("*MSG_ID: %hhx%hhx%hhx%hhx%hhx%hhx%hhx%hhx%hhx%hhx%hhx%hhx%hhx%hhx%hhx%hhx",
	  msg[0], msg[1], msg[2], msg[3],
	  msg[4], msg[5], msg[6], msg[7],
	  msg[8], msg[9], msg[10], msg[11],
	  msg[12], msg[13], msg[14], msg[15]);
    TRACE("*MSG_TYPE: %hhx", GNET_TYPE(msg));
    TRACE("*MSG_TTL: %hhd", GNET_TTL(msg));
    TRACE("*MSG_HOPS: %hhd", GNET_HOPS(msg));
    TRACE("*MSG_SIZE: %d", GTOHS(GNET_SIZE(msg)));
}

