/* $Id: sid.c,v 1.6 2003/05/03 03:20:54 andrewbaker Exp $ */
/* 
** Copyright (C) 2001-2002 Andrew R. Baker <andrewb@snort.org>
** Copyright (C) 2001 Martin Roesch <roesch@sourcefire.com>
**
** This program is distributed under the terms of version 1.0 of the 
** Q Public License.  See LICENSE.QPL for further details.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
**
*/

#include "config.h"

#include <stdio.h>
#include <errno.h>
#include <sys/types.h>
#ifdef SOLARIS
    #include <strings.h>
#endif
#include "sid.h"
#include "classification.h"
#include "util.h"
#include "mstring.h"
#include "strlcatu.h"

#include <string.h>
#include <stdlib.h>

Sid  *SidMapList;
Sid fakeSid;

void ReadSidFile(FILE *);
void ParseLine(char *);
Sid *NewSidNode();
ReferenceData *NewRefNode();
void ReadGenFile(FILE *);
void ParseGenLine(char *);

static void SidDestroy(Sid *);
static void ReferenceDataDestroy(ReferenceData *);

void InitSidMap(char *file)
{
    FILE *sidfile;

    if((sidfile = fopen(file, "r")) == NULL)
    {
        LogMessage("ERROR => Unable to open SID file \"%s\": %s\n", file, 
                strerror(errno));
        
        return;
    }

    ReadSidFile(sidfile);
}


void ReadSidFile(FILE *fp)
{
    char buf[SID_BUFFER_SIZE];
    char *index;
    
    bzero(buf, SID_BUFFER_SIZE);
    
    while(fgets(buf, SID_BUFFER_SIZE, fp) != NULL)
    {
        index = buf;

        /* advance through any whitespace at the beginning of the line */
        while(*index == ' ' || *index == '\t')
            index++;

        /* if it's not a comment or a <CR>, send it to the parser */
        if((*index != '#') && (*index != 0x0a) && (index != NULL))
        {
            ParseLine(index);
        }
    }
}

void FreeSidMap()
{
    Sid *next = NULL;
    while(SidMapList)
    {
        next = SidMapList->next;
        SidDestroy(SidMapList);
        SidMapList = next;
    }

    SidMapList = NULL;
}

static void SidDestroy(Sid *sid)
{
    if(!sid)
        return;

    if(sid->msg)
        free(sid->msg);
    
    if(sid->CSVmsg)
        free(sid->CSVmsg);
    
    if(sid->ref)
        ReferenceDataDestroy(sid->ref);

    free(sid);
}

static void ReferenceDataDestroy(ReferenceData *refData)
{
    ReferenceData *next = NULL;
    while(refData)
    {
        next = refData->next;
        
        if(refData->system)
            free(refData->system);
        
        if(refData->id)
            free(refData->id);
        
        if(refData->url)
            free(refData->url);

        refData = next;
    }
}

void ParseLine(char *data)
{
    char **toks;
    int num_toks;
    char **refdata;
    int ref_toks;
    int i;
    char *idx;
    int size = 0;
    Sid *tmp; 
    int id_len;
    ReferenceData *rn;

    toks = mSplit(data, "|", 32, &num_toks, '\0');

    if(num_toks < 2)
    {
        LogMessage("WARNING => Bad line in SID file: \"%s\"\n", data);
    }

    tmp = NewSidNode();
    
    for(i = 0; i < num_toks; i++)
    {
        strip(toks[i]);
        idx = toks[i];
        while(*idx == ' ') idx++;
            
        switch(i)
        {
            case 0: /* sid */
                tmp->gen = 1;
                tmp->sid = strtoul(idx, NULL, 10);
                break;

            case 1: /* msg */
                tmp->msg = strdup(idx);
                break;

            default: /* reference data */
                if(tmp->ref == NULL)
                {
                    tmp->ref = NewRefNode();
                    rn = tmp->ref;
                }
                else
                {
                    rn = tmp->ref;

                    while(rn->next != NULL) rn = rn->next;

                    rn->next = NewRefNode();

                    rn = rn->next;
                }

                refdata = mSplit(idx, " ,", 32, &ref_toks, '\0');

                if(ref_toks >= 2)
                {
                    rn->id = strdup(refdata[1]);
                    id_len = strlen(rn->id);
                }
                else
                {
                    rn->id = NULL;
                    id_len = 0;
                }
                if(ref_toks >= 1)
                {
                    rn->system = strdup(refdata[0]);

                    if(!strcasecmp(rn->system, "bugtraq"))
                    {
                        size = strlen(BUGTRAQ_URL_HEAD) + id_len + 1;
                        rn->url = (char *) calloc(size, sizeof(char)); 
                        strlcat(rn->url, BUGTRAQ_URL_HEAD, size); 
                    }
                    else if(!strcasecmp(rn->system, "cve"))
                    {
                        size = strlen(CVE_URL_HEAD) + id_len + 1;
                        rn->url = (char *) calloc(size, sizeof(char)); 
                        strlcat(rn->url, CVE_URL_HEAD, size); 
                    }
                    else if(!strcasecmp(rn->system, "arachnids"))
                    {
                        size = strlen(ARACHNIDS_URL_HEAD) + id_len + 1;
                        rn->url = (char *) calloc(size, sizeof(char)); 
                        strlcat(rn->url, ARACHNIDS_URL_HEAD, size); 
                    }
                    else if(!strcasecmp(rn->system, "mcafee"))
                    {
                        size = strlen(MCAFEE_URL_HEAD) + id_len + 5;
                        rn->url = (char *) calloc(size, sizeof(char)); 
                        strlcat(rn->url, MCAFEE_URL_HEAD, size - 4); 
                    }
                    else if(!strcasecmp(rn->system, "nessus"))
                    {
                        size = strlen(NESSUS_URL_HEAD) + id_len + 1;
                        rn->url = (char *) calloc(size, sizeof(char)); 
                        strlcat(rn->url, NESSUS_URL_HEAD, size); 
                    }
                    else
                    {
                        size = strlen(URL_HEAD) + id_len + 1;
                        rn->url = (char *) calloc(size, sizeof(char)); 
                        strlcat(rn->url, URL_HEAD, size); 
                    }
                    if(rn->id)
                    {
                        strlcat(rn->url, rn->id, size); 
                        if(!strcasecmp(rn->system, "mcafee"))
                            strlcat(rn->url, ".htm",size);
                    }
                }

                break;
        }
    }

    FreeToks(toks, num_toks);
}


Sid *GetSid(u_int32_t gen, u_int32_t sid)
{
    Sid *tmp;

    tmp = SidMapList;
    
    while(tmp != NULL)
    {
        if(tmp->gen == gen && tmp->sid == sid)
        {
            return tmp;
        }

        tmp = tmp->next;
    }

    tmp = NewSidNode();
    tmp->gen = gen;
    tmp->sid = sid;
    tmp->rev = 0;
    tmp->msg = (char *)malloc(42);
    snprintf(tmp->msg, 42, "Snort Alert [%u:%u:0]", gen, sid);
    
    return tmp;
}



void PrintXref(int gen, int sid, FILE *fp)
{
    Sid *tmp;
    ReferenceData *ref;

    tmp = SidMapList;

    while(tmp != NULL)
    {
        if(tmp->gen == gen && tmp->sid == sid)
        {
            if(tmp->ref != NULL)
            {
                ref = tmp->ref;
                
                while(ref != NULL)
                {
                    fprintf(fp, "[Xref => %s]\n", ref->url);
                    ref = ref->next;
                }
            }
            
            return;
        }

        tmp = tmp->next;
    }
}



Sid *NewSidNode()
{
    Sid *tmp;

    if(SidMapList == NULL)
    {
        SidMapList = (Sid *) SafeAlloc(sizeof(Sid));

        return SidMapList;
    }
    else
    {
        tmp = SidMapList;

        while(tmp->next != NULL) tmp = tmp->next;

        tmp->next = (Sid *) SafeAlloc(sizeof(Sid));

        return tmp->next;
    }
}

ReferenceData *NewRefNode()
{
    ReferenceData *tmp;

    tmp = (ReferenceData *) SafeAlloc(sizeof(ReferenceData));

    return tmp;
}


void InitGenData(char *file)
{
    FILE *genfile;

    if((genfile = fopen(file, "r")) == NULL)
    {
        LogMessage("ERROR => Unable to open Generator file \"%s\": %s\n", file, 
                strerror(errno));
        
        return;
    }

    ReadGenFile(genfile);
}


void ReadGenFile(FILE *fp)
{
    char buf[SID_BUFFER_SIZE];
    char *index;
    
    bzero(buf, SID_BUFFER_SIZE);
    
    while(fgets(buf, SID_BUFFER_SIZE, fp) != NULL)
    {
        index = buf;

        /* advance through any whitespace at the beginning of the line */
        while(*index == ' ' || *index == '\t')
            index++;

        /* if it's not a comment or a <CR>, send it to the parser */
        if((*index != '#') && (*index != 0x0a) && (index != NULL))
        {
            ParseGenLine(index);
        }
    }
}


void ParseGenLine(char *data)
{
    char **toks;
    int num_toks;
    char **refdata;
    int ref_toks;
    int i;
    char *idx;
    Sid *tmp; 
    ReferenceData *rn;

    toks = mSplit(data, "|", 32, &num_toks, '\0');

    if(num_toks < 2)
    {
        LogMessage("WARNING => Bad line in SID file: \"%s\"\n", data);
    }

    tmp = NewSidNode();
    
    for(i=0;i<num_toks;i++)
    {
        strip(toks[i]);
        idx = toks[i];
        while(*idx == ' ') idx++;
            
        switch(i)
        {
            case 0: /* gen */
                tmp->gen = strtoul(idx, NULL, 10);
                break;

            case 1: /* sid */
                tmp->sid = strtoul(idx, NULL, 10);
                break;

            case 2: /* msg */
                tmp->msg = strdup(idx);
                break;

            default: /* reference data */
                if(tmp->ref == NULL)
                {
                    tmp->ref = NewRefNode();
                    rn = tmp->ref;
                }
                else
                {
                    rn = tmp->ref;

                    while(rn->next != NULL) rn = rn->next;

                    rn->next = NewRefNode();

                    rn = rn->next;
                }

                refdata = mSplit(idx, " ,", 32, &ref_toks, '\0');
                
                if(ref_toks >= 1)
                {
                    rn->system = strdup(refdata[0]);

                    if(!strncasecmp(rn->system, "bugtraq", 7))
                    {
                        rn->url = strdup(BUGTRAQ_URL_HEAD); 
                    }
                    else if(!strncasecmp(rn->system, "cve", 3))
                    {
                        rn->url = strdup(CVE_URL_HEAD); 
                    }
                    else if(!strncasecmp(rn->system, "arachnids", 9))
                    {
                        rn->url = strdup(ARACHNIDS_URL_HEAD); 
                    }
                    else if(!strncasecmp(rn->system, "mcaffee", 7))
                    {
                        rn->url = strdup(MCAFEE_URL_HEAD); 
                    }
                    else
                    {
                        rn->url = strdup(URL_HEAD); 
                    }
                }

                if(ref_toks >= 2)
                {
                    rn->id = strdup(refdata[1]);
                }

                break;
        }
    }
}

Sid *FakeSid(u_int32_t gen, u_int32_t sid)
{
    if(fakeSid.msg != NULL)
        free(fakeSid.msg);

    fakeSid.gen = gen;
    fakeSid.sid = sid;
    fakeSid.rev = 0;
    fakeSid.msg = (char *)malloc(256);
    snprintf(fakeSid.msg, 256, "Snort Signature ID: %u,%u", gen, sid);
    return &fakeSid;
}
