/* $Id: ConfigFile.c,v 1.2 2004/03/25 05:04:40 andrewbaker Exp $ */
/*
** Copyright (C) 2001 Martin Roesch <roesch@sourcefire.com>
** Copyright (C) 2001-2004 Andrew R. Baker <andrewb@sourcefire.com>          
**
** This program is distributed under the terms of version 1.0 of the 
** Q Public License.  See LICENSE.QPL for further details.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
**
*/


#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <string.h>
#ifdef SOLARIS
    #include <strings.h>
#endif
#include <errno.h>
#include <stdlib.h>
#include <ctype.h>

//#include "barnyard.h"
#include "ConfigFile.h"
#include "util.h"
#include "mstring.h"
#include "output-plugins/op_plugbase.h"
#include "input-plugins/dp_plugbase.h"



/*---------------------  Global Variables --------------------*/
int file_line;
char *file_name;

/*----------------------- Local Functions --------------------*/
int ContinuationCheck(char *);
int ParseConfigLine(char *, ConfigFileVars *cfvars);
void OutputDirective(char *);
int ConfigDirective(char *, ConfigFileVars *cfvars);

int ConfigFile_Parse(char *filename, ConfigFileVars **cfvars)
{
    FILE *fp = NULL;
    char readbuf[STD_BUF];
    char *index;
    ConfigFileVars *tmp = NULL;
    int continuation = 0;
    char *new_line = NULL;
    char *saved_line = NULL;
    
    if(!filename || !cfvars)
        return -1;

    if(!(tmp = (ConfigFileVars *)calloc(1, sizeof(ConfigFileVars))))
    {
        FatalError("Out of memory (wanted %u bytes)\n", sizeof(ConfigFileVars));
        return -1;
    }

    if((fp = fopen(filename, "r")) == NULL)
    {
        FatalError("Failed to open config file '%s': %s\n", filename,
                strerror(errno));
        goto error;
    }

    file_name = filename;
    file_line = 0;

    if((fp = fopen(filename, "r")) == NULL)
    {
        FatalError("Failed to open config file \"%s\", check the filename "
                   "and try again\n", filename);
    }

    while((fgets(readbuf, STD_BUF, fp)) != NULL)
    {
        file_line++;
        index = readbuf;

        /* advance through any whitespace at the beginning of the line */
        while(*index == ' ' || *index == '\t')
            index++;

        /* if it's not a comment or a <CR>, send it to the parser */
        if((*index != '#') && (*index != 0x0a) && (*index != ';') && 
           (index != NULL))
        {
            if(continuation == 1)
            {
                new_line = (char *) calloc((strlen(saved_line) + strlen(index)
                            +1), sizeof(char)); 
                strncat(new_line, saved_line, strlen(saved_line));
                strncat(new_line, index, strlen(index));
                free(saved_line);
                saved_line = NULL;
                index = new_line;

                if(strlen(index) > CONFIG_PARSE_MAX_SIZE)
                {
                    FatalError("Please don't try to overflow the parser, "
                            "that's not very nice of you... (%d-byte limit "
                            "on config lines, pardner)\n", 
                            CONFIG_PARSE_MAX_SIZE);
                }
            }
            
            /* check for a '\' continuation character at the end of the line
             * if it's there we need to get the next line in the file
             */
            if(ContinuationCheck(index) == 0) 
            {
                ParseConfigLine(index, tmp);

                if(new_line != NULL)
                {
                    free(new_line);
                    new_line = NULL;
                    continuation = 0;
                }
            }
            else
            {
                /* save the current line */
                saved_line = strdup(index);

                /* set the flag to let us know the next line is 
                 * a continuation line
                 */ 
                continuation = 1;
            }   
        }

        memset(readbuf, 0, STD_BUF);
    }
    fclose(fp);
    *cfvars = tmp;
    return 0;
    
error:
    if(tmp)
        ConfigFileVars_Free(tmp);
    return -1;
}

int ContinuationCheck(char *line)
{
    char *idx;  /* indexing var for moving around on the string */

    idx = line + strlen(line) - 1;

    while(isspace((int)*idx))
    {
        idx--;
    }

    if(*idx == '\\')
    {
        /* clear the '\' so there isn't a problem on the appended string */
        *idx = '\x0';
        return 1;
    }

    return 0;
}


int ParseConfigLine(char *line, ConfigFileVars *cfvars)
{
    char **toks;
    int num_toks;
    
    if(!line || !cfvars)
        return -1;
    
    toks = mSplit(line, " ", 2, &num_toks, '\0');

    if(num_toks <= 1)
    {
        FatalError("ERROR %s(%d) => Got bad config line: %s\n", file_name, 
                file_line, line);
    }
    else
    {
        if(!strcasecmp(toks[0], "processor"))
        {
            LogMessage("WARNING: processor configuration lines are no longer "
                    "needed\n");
        }
        else if(!strcasecmp(toks[0], "output"))
        {
            OutputDirective(toks[1]);
        }
        else if(!strcasecmp(toks[0], "config"))
        {
            ConfigDirective(toks[1], cfvars);
        }
        else
        {
            FatalError("FATAL ERROR %s(%d) => Unknown directive: %s\n", 
                    file_name, file_line, toks[0]);
        }
    }
    FreeToks(toks, num_toks);
    return 0;
}

void OutputDirective(char *directive)
{
    char **toks;            /* tokens for the line */
    int num_toks;           /* number of tokens */
    char *op_args = NULL;   /* output plugin arguments */
    char *output_plugin_name; /* output plugin function name being called */
    DpFunctionalNode *dpfn;
    OutputPlugin *outputPlugin;

    /* split the output directive head from its arguments */
    toks = mSplit(directive, ":", 2, &num_toks, '\0');

    /* lookup the output plugin to see if it's linked/loaded */
    if(num_toks > 1)
    {
        op_args = toks[1];
    }

    /* get the output function name */
    output_plugin_name = toks[0];

    if(output_plugin_name == NULL)
    {
        LogMessage("Got bad config line:\n");
        LogMessage("\"%s\"\n", directive);
        return;
    }

    /* lookup the output plugin */
    outputPlugin = LookupOutputPlugin(output_plugin_name);

    if(outputPlugin != NULL)
    {
        /* look up data processors of the same type */
        dpfn = GetDpType(outputPlugin->type);
	
        if(dpfn != NULL)
        {
            AddOutputPlugin(&dpfn->oList, outputPlugin);
            OutputPluginSetup(outputPlugin, op_args);
        }
        else
        {
            LogMessage("WARNING %s(%d) => No data processor registered "
                        "for type \"%s\", ignoring \"%s\" output directive\n", 
                        file_name, file_line, outputPlugin->type, 
                        outputPlugin->name);
        }
    }
    else
    {
        FatalError("WARNING %s(%d) => Unknown output plugin \"%s\" "
                "referenced, ignoring!", file_name, file_line, 
                output_plugin_name); 
    }

    FreeToks(toks, num_toks);
    return;
}

int ConfigDirective(char *directive, ConfigFileVars *cfvars)
{
    char **toks;
    char *args = NULL;
    char *config_name;
    int num_toks;
    
    if(!directive || !cfvars)
        return -1;

    toks = mSplit(directive, ":", 2, &num_toks, '\0');

    if(num_toks > 1)
    {
        args = toks[1];
    }

    config_name = toks[0];
    
    if(!strcasecmp(config_name, "daemon"))
    {
        cfvars->daemon_flag = 1;
    }
    else if(!strcasecmp(config_name, "localtime"))
    {
        cfvars->localtime_flag = 1;
    }
    else if(args == NULL)
    {
        LogMessage("Invalid config directive: '%s'\n", directive);
    }
    else if(!strcasecmp(config_name, "hostname"))
    {
        if(cfvars->hostname)
            FatalError("Multiple %s config directives\n", config_name);

        if(!(cfvars->hostname = strdup(args)))
            FatalError("Out of memory (wanted %u bytes)\n", strlen(args) + 1);
    }
    else if(!strcasecmp(config_name, "interface"))
    {
        if(cfvars->interface)
            FatalError("Multiple %s config directives\n", config_name);

        if(!(cfvars->interface = strdup(args)))
            FatalError("Out of memory (wanted %u bytes)\n", strlen(args) + 1);
    }
    else if(!strcasecmp(config_name, "filter"))
    {
        if(cfvars->bpf_filter)
            FatalError("Multiple %s config directives\n", config_name);

        if(!(cfvars->bpf_filter = strdup(args)))
            FatalError("Out of memory (wanted %u bytes)\n", strlen(args) + 1);
    }
    else if(strcasecmp(config_name, "sid-msg-map") == 0)
    {
        if(cfvars->sid_msg_file)
            FatalError("Multiple %s config directives\n", config_name);

        if(!(cfvars->sid_msg_file = strdup(args)))
            FatalError("Out of memory (wanted %u bytes)\n", strlen(args) + 1);
    }
    else if(strcasecmp(config_name, "gen-msg-map") == 0)
    {
        if(cfvars->gen_msg_file)
            FatalError("Multiple %s config directives\n", config_name);

        if(!(cfvars->gen_msg_file = strdup(args)))
            FatalError("Out of memory (wanted %u bytes)\n", strlen(args) + 1);
    }
    else if(strcasecmp(config_name, "class-file") == 0)
    {
        if(cfvars->class_file)
            FatalError("Multiple %s config directives\n", config_name);

        if(!(cfvars->class_file = strdup(args)))
            FatalError("Out of memory (wanted %u bytes)\n", strlen(args) + 1);
    }
    else
    {
        LogMessage("Unrecognized config directive: '%s'\n", directive);
    }
    FreeToks(toks, num_toks);

    return 0;
}

int ConfigFileVars_Fprintf(ConfigFileVars *cfvars, FILE *stream)
{
    if(!cfvars)
        return -1;

    if(!stream)
        stream = stdout;

    fprintf(stream, "Config file variables:\n");
    fprintf(stream, "  Hostname:        %s\n",
            cfvars->hostname ? cfvars->hostname : "Not specified");
    fprintf(stream, "  Interface:       %s\n",
            cfvars->interface ? cfvars->interface : "Not specified");
    fprintf(stream, "  BPF Filter:      %s\n",
            cfvars->bpf_filter ? cfvars->bpf_filter : "Not specified");
    fprintf(stream, "  Class file:      %s\n",
            cfvars->class_file ? cfvars->class_file : "Not specified");
    fprintf(stream, "  Sid-msg file:    %s\n",
            cfvars->sid_msg_file ? cfvars->sid_msg_file : "Not specified");
    fprintf(stream, "  Gen-msg file:    %s\n",
            cfvars->gen_msg_file ? cfvars->gen_msg_file : "Not specified");
    fprintf(stream, "  Daemon flag:     %s\n",
            cfvars->daemon_flag ? "Set" : "Not Set");
    fprintf(stream, "  Localtime flag:  %s\n",
            cfvars->localtime_flag ? "Set" : "Not Set");
    
    return 0;
}

int ConfigFileVars_Free(ConfigFileVars *cfvars)
{
    if(!cfvars)
        return -1;
    
    if(cfvars->hostname)
        free(cfvars->hostname);
    if(cfvars->interface)
        free(cfvars->interface);
    if(cfvars->bpf_filter)
        free(cfvars->bpf_filter);
    if(cfvars->class_file)
        free(cfvars->class_file);
    if(cfvars->sid_msg_file)
        free(cfvars->sid_msg_file);
    if(cfvars->gen_msg_file)
        free(cfvars->gen_msg_file);
    memset(cfvars, 0, sizeof(ConfigFileVars));
    free(cfvars);

    return 0;
}
