/* ------------------------------------------------------------------------
 *
 * client_tipc.c
 *
 * Short description: TIPC subscription demo (client side)
 *
 * ------------------------------------------------------------------------
 *
 * Copyright (c) 2003, Ericsson Research Canada
 * Copyright (c) 2010 Wind River Systems
 * All rights reserved.
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice, this
 * list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 * Neither the name of Ericsson Research Canada nor the names of its
 * contributors may be used to endorse or promote products derived from this
 * software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * ------------------------------------------------------------------------
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <stdlib.h>
#include <stdio.h>
#include <assert.h>
#include <string.h>
#include <sys/param.h>
#include <sys/poll.h>
#include <netdb.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <linux/tipc.h>


#ifndef TIPC_SERVICE_ADDR
#define TIPC_SERVICE_ADDR 2
#endif

#define SERVER_TYPE 18888
#define TIMEOUT 500000

static void print_sub(const char *str, struct tipc_subscr *s)
{
	printf("%s {%u,%u,%u}, timeout %u, user ref: %u\n",
	       str, ntohl(s->seq.type), ntohl(s->seq.lower),
	       ntohl(s->seq.upper), ntohl(s->timeout),
	       (unsigned int)s->usr_handle[0]);
}

static void print_evt(const char *str, struct tipc_event *evt)
{
	printf("%s ", str);

	if (evt->event == htonl(TIPC_PUBLISHED))
		printf("published ");
	else if (evt->event == htonl(TIPC_WITHDRAWN))
		printf("withdrawn ");
	else if (evt->event == htonl(TIPC_SUBSCR_TIMEOUT))
		printf("timeout ");
	else {
		printf("unknown event type (%u)\n", ntohl(evt->event));
		return;
	}

	printf ("{%u,%u,%u} port id <%x:%u>\n", ntohl(evt->s.seq.type),
	        ntohl(evt->found_lower), ntohl(evt->found_upper),
	        ntohl(evt->port.node), ntohl(evt->port.ref));
	if (evt->s.seq.type != htonl(0))
		printf("associated with network node %x\n",
		       ntohl(evt->port.node));

	print_sub("generated by subscription to", &evt->s);
}

int main(int argc, char *argv[])
{
	struct sockaddr_tipc topsrv;
	struct tipc_subscr subscr;
	struct tipc_event event;
	int sd;

	printf("TIPC Topology subscriber started\n");

	/* Connect to topology server */

	memset(&topsrv, 0, sizeof(topsrv));
	topsrv.family = AF_TIPC;
	topsrv.addrtype = TIPC_SERVICE_ADDR;
	topsrv.addr.name.name.type = TIPC_TOP_SRV;
	topsrv.addr.name.name.instance = TIPC_TOP_SRV;

	sd = socket (AF_TIPC, SOCK_SEQPACKET, 0);

	if (0 > connect(sd, (struct sockaddr *)&topsrv, sizeof(topsrv))) {
		perror("Client: failed to connect to topology server");
		exit(1);
	}
	printf("Client: connected to topology server\n");

	/* Subscribe to subscription server name events */

	subscr.seq.type  = (argc > 1) ? htonl(atoi(argv[1])) : htonl(SERVER_TYPE);
	subscr.seq.lower = htonl(0);
	subscr.seq.upper = htonl(100);
	subscr.timeout   = (argc > 2) ? htonl(atoi(argv[2])) : htonl(TIMEOUT);
	subscr.filter    = htonl(TIPC_SUB_SERVICE);
	subscr.usr_handle[0] = (argc > 3) ? (char)atoi(argv[3]) : (char)2;

	if (send(sd, &subscr, sizeof(subscr), 0) != sizeof(subscr)) {
		perror("Client: failed to subscribe to subscription server name");
		exit(1);
	}
	print_sub("Client: issued subscription to", &subscr);

	/* Subscribe to network node name events */

	subscr.seq.type  = htonl(0);
	subscr.seq.lower = htonl(0);
	subscr.seq.upper = htonl(~0);
	subscr.timeout   = htonl(TIPC_WAIT_FOREVER);
	subscr.filter    = htonl(TIPC_SUB_PORTS);
	subscr.usr_handle[0] = (char)3;

	if (send(sd, &subscr, sizeof(subscr), 0) != sizeof(subscr)) {
		perror("Client: failed to subscribe to network node names");
		exit(1);
	}
	print_sub("Client: issued subscription to", &subscr);

	printf("Client: subscriptions remain active until client is killed\n");

	/* Now wait for the subscriptions to fire */

	while (recv(sd, &event, sizeof(event), 0) == sizeof(event)) {
		print_evt("Client: received event for", &event);
	}

	perror("Client: failed to receive event");
	exit(1);
}
