# SPDX-License-Identifier: MIT
# SPDX-FileCopyrightText: 2021 Taneli Hukkinen
# Licensed to PSF under a Contributor Agreement.

import copy
import datetime
from decimal import Decimal as D
import importlib
from pathlib import Path
import sys
import tempfile
import unittest

from . import tomllib


class TestMiscellaneous(unittest.TestCase):
    def test_load(self):
        content = "one=1 \n two='two' \n arr=[]"
        expected = {"one": 1, "two": "two", "arr": []}
        with tempfile.TemporaryDirectory() as tmp_dir_path:
            file_path = Path(tmp_dir_path) / "test.toml"
            file_path.write_text(content)

            with open(file_path, "rb") as bin_f:
                actual = tomllib.load(bin_f)
        self.assertEqual(actual, expected)

    def test_incorrect_load(self):
        content = "one=1"
        with tempfile.TemporaryDirectory() as tmp_dir_path:
            file_path = Path(tmp_dir_path) / "test.toml"
            file_path.write_text(content)

            with open(file_path, "r") as txt_f:
                with self.assertRaises(TypeError) as exc_info:
                    tomllib.load(txt_f)  # type: ignore[arg-type]
            # Mypyc extension leads to different message than pure Python
            self.assertIn(
                str(exc_info.exception),
                (
                    "File must be opened in binary mode, e.g. use `open('foo.toml', 'rb')`",  # noqa: E501
                    "bytes object expected; got str",
                ),
            )

    def test_parse_float(self):
        doc = """
              val=0.1
              biggest1=inf
              biggest2=+inf
              smallest=-inf
              notnum1=nan
              notnum2=-nan
              notnum3=+nan
              """
        obj = tomllib.loads(doc, parse_float=D)
        expected = {
            "val": D("0.1"),
            "biggest1": D("inf"),
            "biggest2": D("inf"),
            "smallest": D("-inf"),
            "notnum1": D("nan"),
            "notnum2": D("-nan"),
            "notnum3": D("nan"),
        }
        for k, expected_val in expected.items():
            actual_val = obj[k]
            self.assertIsInstance(actual_val, D)
            if actual_val.is_nan():
                self.assertTrue(expected_val.is_nan())
            else:
                self.assertEqual(actual_val, expected_val)

    def test_deepcopy(self):
        doc = """
              [bliibaa.diibaa]
              offsettime=[1979-05-27T00:32:00.999999-07:00]
              """
        obj = tomllib.loads(doc)
        obj_copy = copy.deepcopy(obj)
        self.assertEqual(obj_copy, obj)
        expected_obj = {
            "bliibaa": {
                "diibaa": {
                    "offsettime": [
                        datetime.datetime(
                            1979,
                            5,
                            27,
                            0,
                            32,
                            0,
                            999999,
                            tzinfo=datetime.timezone(datetime.timedelta(hours=-7)),
                        )
                    ]
                }
            }
        }
        self.assertEqual(obj_copy, expected_obj)

    def test_inline_array_recursion_limit(self):
        nest_count = 470
        recursive_array_toml = "arr = " + nest_count * "[" + nest_count * "]"
        tomllib.loads(recursive_array_toml)

        nest_count = sys.getrecursionlimit() + 2
        recursive_array_toml = "arr = " + nest_count * "[" + nest_count * "]"
        with self.assertRaisesRegex(
            RecursionError,
            r"maximum recursion depth exceeded"
            r"|"
            r"TOML inline arrays/tables are nested more than the allowed [0-9]+ levels",
        ):
            tomllib.loads(recursive_array_toml)

    def test_inline_table_recursion_limit(self):
        nest_count = 310
        recursive_table_toml = nest_count * "key = {" + nest_count * "}"
        tomllib.loads(recursive_table_toml)

        nest_count = sys.getrecursionlimit() + 2
        recursive_table_toml = nest_count * "key = {" + nest_count * "}"
        with self.assertRaisesRegex(
            RecursionError,
            r"maximum recursion depth exceeded"
            r"|"
            r"TOML inline arrays/tables are nested more than the allowed [0-9]+ levels",
        ):
            tomllib.loads(recursive_table_toml)

    def test_types_import(self):
        """Test that `_types` module runs.

        The module is for type annotations only, so it is otherwise
        never imported by tests.
        """
        importlib.import_module(f"{tomllib.__name__}._types")
