import { type Config } from 'svgo';
import type { Plugin } from 'vite';
interface Options {
    /**
     * Output type
     *
     * `dataurl` can also take the following options, which are verbatim SVGO
     * `datauri` options:
     *
     * - `?dataurl=base64` (default, same as `?dataurl`)
     * - `?dataurl=enc` URL encoded string
     * - `?dataurl=unenc` Plain SVG
     *
     * @default "component"
     */
    type?: 'src' | 'url' | 'component' | 'dataurl';
    /**
     * Verbatim [SVGO](https://github.com/svg/svgo) options
     *
     * If no options are given, the SVG will be optimized with the default SVGO
     * options.
     * If `false` SVGO will be bypassed altogether
     */
    svgoOptions?: Config | false;
    /**
     * Paths to apply the SVG plugin on. This can be useful if you want to apply
     * different SVGO options/plugins on different SVGs.
     *
     * The paths are path prefixes and should be relative to your
     * `svelte.config.js` file.
     *
     * @example
     * ```
     * {
     *   includePaths: ['src/assets/icons/', 'src/images/icons/']
     * }
     * ```
     */
    includePaths?: string[];
    /**
     * Hook that lets you transform the svg to a raw Svelte component yourself,
     * before being passed to the Svelte compiler.
     *
     * @param rawSvg The raw SVG data as read from disk
     * @param splitSvg The SVG split into parts, e.g its attributes and
     *  its content
     * @returns This should return a complete Svelte component that can be passed
     *  to the Svelte compiler
     */
    preCompileHook?(rawSvg: string, splitSvg: SplitSvg): string;
}
type SplitSvg = {
    /**
     * The attributes of an SVG as a string
     *
     * Given `<svg width="200" height="100">` this will be
     * `width="200" height="100"`
     */
    attributes: string | undefined;
    /**
     * The inner content of an SVG
     *
     * Given `<svg><g><path/></g></svg>` this will be `<g><path/></g>`.
     */
    content: string | undefined;
    /**
     * The default generated, by this plugin, Svelte component as a string
     *
     * Given `<svg width="100"><path/></svg>` this will be something like
     * `<svg width="100" {...$$props}>{@html "<path/>"}</svg>`
     */
    component: string;
};
declare function readSvg(options?: Options): Plugin;
export = readSvg;
