/*
 * Copyright (c) 1992, 1993, 1994, 1995, 1996, 1998, 2000, 2009, 2013, 2016, 2019, 2023
 * The Regents of the University of California. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of the University nor the names of its contributors
 *       may be used to endorse or promote products derived from this software
 *       without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
#ifndef lint
static const char rcsid[] __attribute__((unused)) =
    "@(#) $Id: dns.c 1537 2023-09-05 17:50:56Z leres $ (LBL)";
#endif

/*
 * dns - domain name system routines
 */

#include <sys/types.h>
#include <sys/socket.h>

#include <netinet/in.h>

#include <arpa/nameser.h>
#include <arpa/inet.h>

#include <ctype.h>
#ifdef HAVE_MEMORY_H
#include <memory.h>
#endif
#include <netdb.h>
#include <resolv.h>
#include <string.h>
#include <stdio.h>

#include "gnuc.h"
#ifdef HAVE_OS_PROTO_H
#include "os-proto.h"
#endif

#include "arpwatch.h"
#include "dns.h"

#ifdef HAVE_DN_SKIPNAME
#ifndef BUFSIZ
#define BUFSIZ 1024
#endif

static char hostbuf[BUFSIZ+1];

#if PACKETSZ > 1024
#define	MAXPACKET	PACKETSZ
#else
#define	MAXPACKET	1024
#endif

typedef union {
	HEADER hdr;
	u_char buf[MAXPACKET];
} querybuf;
#endif

int
gethinfo(char *hostname, char *cpu, int cpulen, char *os, int oslen)
{
#ifdef HAVE_DN_SKIPNAME
	querybuf *qb;
	u_char *cp, *eom;
	char *bp;
	int n;
	HEADER *hp;
	int type, buflen, ancount, qdcount;
	querybuf qbuf;

	qb = &qbuf;
	n = res_query(hostname, C_IN, T_HINFO, qb->buf, sizeof(qb->buf));
	if (n < 0)
		return (0);

	eom = qb->buf + n;
	/*
	 * find first satisfactory answer
	 */
	hp = &qb->hdr;
	ancount = ntohs(hp->ancount);
	qdcount = ntohs(hp->qdcount);
	bp = hostbuf;
	buflen = sizeof(hostbuf);
	cp = qb->buf + sizeof(HEADER);
	if (qdcount) {
		cp += dn_skipname(cp, eom) + QFIXEDSZ;
		while (--qdcount > 0)
			cp += dn_skipname(cp, eom) + QFIXEDSZ;
	}
	while (--ancount >= 0 && cp < eom) {
		if ((n = dn_expand((u_char *)qb->buf, (u_char *)eom,
		    (u_char *)cp, (char *)bp, buflen)) < 0)
			break;
		cp += n;
		type = _getshort(cp);
		cp += sizeof(u_short);			/* class */
		cp += sizeof(u_short) + sizeof(u_int32_t);
		n = _getshort(cp);
		cp += sizeof(u_short);
		if (type == T_HINFO) {
			/* Unpack */
			n = *cp++;
			if (n > cpulen - 1)
				return (0);
			memmove(cpu, cp, n);
			cp += n;
			cpu[n] = '\0';
			n = *cp++;
			if (n > oslen - 1)
				return (0);
			memmove(os, cp, n);
			os[n] = '\0';
			return (1);
		}
		/* Skip unexpected junk */
		cp += n;
	}
#endif
	return (0);
}

/* Return the canonical name of the host (NULL if not found) */
char *
gethname(u_int32_t a)
{
	int32_t options;
	struct hostent *hp;

	options = _res.options;
	_res.options |= RES_AAONLY;
	_res.options &= ~(RES_DEFNAMES | RES_DNSRCH);
	hp = gethostbyaddr((char *)&a, sizeof(a), AF_INET);
	_res.options = options;
	if (hp == NULL)
		return (NULL);
	return (hp->h_name);
}

/* Return the simple name of the host (NULL if not found) */
char *
getsname(u_int32_t a)
{
	char *s, *cp;

	s = gethname(a);
	if (s != NULL) {
		cp = strchr(s, '.');
		if (cp != NULL)
			*cp = '\0';
	}
	return (s);
}
