#include "instantiation/inst-del.h"

#include "float.h"
#include "instantiation/free-vars.h"
#include "instantiation/inst-index.h"
#include "instantiation/inst-pre.h"
#include "instantiation/inst-trigger-selection.h"
#include "pre/pre.h"
#include "proof/proof.h"
#include "symbolic/DAG-print.h"
#include "symbolic/DAG-ptr.h"
#include "symbolic/DAG.h"
#include "utils/bitset.h"
#include "utils/options.h"
#include "utils/stack.h"
#include "utils/statistics.h"

/*
  --------------------------------------------------------------
  Options
  --------------------------------------------------------------
*/

/**
   \addtogroup arguments_user

   - --inst-deletion-loops

   Prevent matching loops by forcing sort inst [unstable]

   For now I'll only capture loops in which a new term is generated by a
   quantified formula instantiated and that term would lead to new
   instantiations of that same quantified formula.

   TODO: Limitations:

   - only used with --inst-deletion-track-vars because it's easier for now (when
   marking clauses I don't know from which quantified formulas they came from)

   - terms that appear in more than one instance will be only considered by one
   of the qnt_forms (because of how I'm using of DAG_tmp in inst_mark_var_rec)

   - Two or more quantified formulas could be producing a loop: Q1 generates
   terms triggering insts in Q2, which produces terms generating insts in Q1
   and so on. This is not captured (neither do I know now how I would do that)

   - It introduces a huge overhead... retrieving terms from the index becomes
   very expensive. Think how to optimize.
*/
bool inst_deletion_loops;

/**
   \addtogroup arguments_user

   - --inst-deletion-track-vars

   Track vars activity rather than clauses */
bool inst_deletion_track_vars;

/*
  --------------------------------------------------------------
  Setting
  --------------------------------------------------------------
*/

typedef struct Tclause_vars
{
	unsigned id;
	Tstrategy strategy;
	Tstack_unsigned vars;
} Tclause_vars;

TSstack(_clause_vars, Tclause_vars); /* typedefs Tstack_clause_vars */

/**
    \brief stores id and literals of clauses from trigger and sort instantiations
*/
static Tstack_clause_vars inst_clause_vars;

/**
    \brief stores the variables from all instances (not the lemmas) produced
*/
static Tstack_var inst_vars;

Tstack_DAG* lit_qforms;
Tstack_var* index_lits;

Tstrategy inst_successful;
unsigned inst_done_round;
bool inst_check_loop;
bool inst_marking;

static void
lits_inst_hook_resize(unsigned old_alloc, unsigned new_alloc)
{
	MY_REALLOC(lit_qforms, new_alloc * sizeof(Tstack_DAG));
	MY_REALLOC(index_lits, new_alloc * sizeof(Tstack_var));
	for (unsigned i = old_alloc; i < new_alloc; ++i) {
		lit_qforms[i] = NULL;
		index_lits[i] = NULL;
	}
}

void
inst_promote_vars(void)
{
	for (unsigned i = 0; i < stack_size(inst_vars); ++i)
		if (SAT_var_get_activity(stack_get(inst_vars, i)))
			promote_var_lvl(stack_get(inst_vars, i));
	inst_marking = true;
}

void
inst_promote_clauses(void)
{
	for (unsigned i = 0; i < stack_size(inst_clause_vars); ++i)
		if (SAT_clause_get_activity(stack_get(inst_clause_vars, i).id))
			stack_apply(stack_get(inst_clause_vars, i).vars, promote_var_lvl);
	inst_marking = true;
}

/** \brief workaround for quantified formula being marked */
static TDAG current_qform;

static void
inst_mark_var_rec(TDAG DAG)
{
	Tvar var;
	if (DAG_tmp_bool[DAG]) return;
	DAG_tmp_bool[DAG] = 1;
	if (quantifier(DAG_symb(DAG))) return;
	if (DAG_literal(DAG)) {
		var = lit_var(DAG_to_lit(DAG));
		set_var_lvl_arg(var, inst_done_round);
		/* TODO: should add only if it was undefined before */
		stack_push(inst_vars, var);
		/* Associate new vars to respective quantified formula */
		if (
			inst_deletion_loops && inst_successful == inst_TRIGGERS &&
			get_var_lvl(var) == get_deepest_lvl()) {
			if (!lit_qforms[var]) stack_INIT(lit_qforms[var]);
			stack_push(lit_qforms[var], current_qform);
			stack_sort(lit_qforms[var], DAG_cmp_q);
			/* TODO: is this necessary? */
			stack_uniq(lit_qforms[var]);
		}
		return;
	}
	for (unsigned i = 0; i < DAG_arity(DAG); ++i)
		inst_mark_var_rec(DAG_arg(DAG, i));
}

void
inst_mark_instances(void)
{
	DAG_tmp_reserve();
	for (unsigned i = 0; i < stack_size(lemmas); ++i) {
		current_qform = DAG_arg0(DAG_arg(stack_get(lemmas, i), 0));
		inst_mark_var_rec(DAG_arg(stack_get(lemmas, i), 1));
	}
	for (unsigned i = 0; i < stack_size(lemmas); ++i)
		DAG_tmp_reset_bool(DAG_arg(stack_get(lemmas, i), 1));
	DAG_tmp_release();
	inst_marking = false;
}

static void
inst_mark_clause_rec(TDAG DAG)
{
	Tvar var;
	if (DAG_tmp_bool[DAG]) return;
	DAG_tmp_bool[DAG] = 1;
	if (quantifier(DAG_symb(DAG))) return;
	if (DAG_literal(DAG)) {
		var = lit_var(DAG_to_lit(DAG));
		set_var_lvl_arg(var, inst_done_round);
		stack_push(stack_top(inst_clause_vars).vars, var);
		return;
	}
	for (unsigned i = 0; i < DAG_arity(DAG); ++i)
		inst_mark_var_rec(DAG_arg(DAG, i));
}

/* TODO: Instead of storing literals just have them retrieved from the SAT
   solver when checking the active clauses? */
void
inst_mark_clause(Tclause clause, unsigned clause_id)
{
	/* Does not need to consider clauses in root level */
	if (!inst_done_round || !clause_id) return;
#if DEBUG_INST > 2
	my_DAG_message("Adding clause %d:\n", clause_id);
	for (i = 0; i < clause->nb_lits; ++i)
		my_DAG_message(
			"\tlit %d; var %d; %D\n", clause->lits[i], lit_var(clause->lits[i]),
			lit_to_DAG(clause->lits[i]));
	my_message_return();
#endif
	stack_inc(inst_clause_vars);
	stack_top(inst_clause_vars).id = clause_id;
	stack_top(inst_clause_vars).strategy = inst_successful;
	stack_INIT(stack_top(inst_clause_vars).vars);
	for (unsigned i = 0; i < clause->nb_lits; ++i) {
		/* Literals may stand for complex formulas */
		DAG_tmp_reserve();
		inst_mark_clause_rec(lit_to_DAG(clause->lits[i]));
		DAG_tmp_reset_bool(lit_to_DAG(clause->lits[i]));
		DAG_tmp_release();
		stack_push(stack_top(inst_clause_vars).vars, lit_var(clause->lits[i]));
	}
}

void
inst_del_init(void)
{
	stack_INIT(inst_clause_vars);
	lit_qforms = NULL;
	index_lits = NULL;

	stack_INIT(inst_vars);

	DAG_set_hook_resize(lits_inst_hook_resize);

	inst_check_loop = false;
	inst_marking = false;
	inst_done_round = 0;

	/* Options */
	inst_deletion_loops = false;
	options_new(
		0, "inst-deletion-loops", "Prevent matching loops by forcing sort inst",
		&inst_deletion_loops);

	inst_deletion_track_vars = false;
	options_new(
		0, "inst-deletion-track-vars", "Track vars activity rather than clauses",
		&inst_deletion_track_vars);
}

void
inst_del_done(void)
{
	unsigned i;
	float total_ccfv = 0, promoted_ccfv = 0, total_triggers = 0,
				promoted_triggers = 0, total_sorts = 0, promoted_sorts = 0;
	/* Computing stats */
#ifdef STATS_INST
	for (i = 0; i < stack_size(inst_clause_vars); ++i) {
		if (stack_get(inst_clause_vars, i).strategy == inst_CIs)
			++total_ccfv;
		else if (stack_get(inst_clause_vars, i).strategy == inst_TRIGGERS)
			++total_triggers;
		else if (stack_get(inst_clause_vars, i).strategy == inst_ENUM)
			++total_sorts;
		if (!SAT_clause_get_activity(stack_get(inst_clause_vars, i).id)) continue;
		if (stack_get(inst_clause_vars, i).strategy == inst_CIs)
			++promoted_ccfv;
		else if (stack_get(inst_clause_vars, i).strategy == inst_TRIGGERS)
			++promoted_triggers;
		else if (stack_get(inst_clause_vars, i).strategy == inst_ENUM)
			++promoted_sorts;
	}
	assert(
		total_ccfv >= promoted_ccfv && total_triggers >= promoted_triggers &&
		total_sorts >= promoted_sorts);
	stats_float(
		"del/promoted_ccfv", "how many instantiation from ccfv were promoted",
		"%7.2f", total_ccfv && promoted_ccfv ? promoted_ccfv / total_ccfv : 0);
	stats_float(
		"del/promoted_triggers",
		"how many instantiation from triggers were promoted", "%7.2f",
		total_triggers && promoted_triggers ? promoted_triggers / total_triggers
																				: 0);
	stats_float(
		"del/promoted_sorts", "how many instantiation from sorts were promoted",
		"%7.2f", total_sorts && promoted_sorts ? promoted_sorts / total_sorts : 0);
#endif
	while (!stack_is_empty(inst_clause_vars)) {
		stack_free(stack_top(inst_clause_vars).vars);
		stack_dec(inst_clause_vars);
	}
	stack_free(inst_clause_vars);
	if (inst_deletion_track_vars && inst_deletion_loops)
		for (i = 0; i < stack_size(inst_vars); ++i)
			if (lit_qforms[stack_get(inst_vars, i)])
				stack_free(lit_qforms[stack_get(inst_vars, i)]);
	stack_free(inst_vars);
}
