// Copyright 2019 Drone.IO Inc. All rights reserved.
// Use of this source code is governed by the Drone Non-Commercial License
// that can be found in the LICENSE file.

package acl

import (
	"errors"
	"net/http"
	"net/http/httptest"
	"testing"

	"github.com/drone/drone/handler/api/request"
	"github.com/drone/drone/mock"

	"github.com/go-chi/chi"
	"github.com/golang/mock/gomock"
)

func TestCheckMembership_Admin(t *testing.T) {
	controller := gomock.NewController(t)
	defer controller.Finish()

	w := httptest.NewRecorder()
	r := httptest.NewRequest("GET", "/api/secrets/github", nil)
	r = r.WithContext(
		request.WithUser(noContext, mockUserAdmin),
	)

	router := chi.NewRouter()
	router.Route("/api/secrets/{namespace}", func(router chi.Router) {
		router.Use(CheckMembership(nil, true))
		router.Get("/", func(w http.ResponseWriter, r *http.Request) {
			w.WriteHeader(http.StatusTeapot)
		})
	})

	router.ServeHTTP(w, r)

	if got, want := w.Code, http.StatusTeapot; got != want {
		t.Errorf("Want status code %d, got %d", want, got)
	}
}

func TestCheckMembership_NilUser_Unauthorized(t *testing.T) {
	controller := gomock.NewController(t)
	defer controller.Finish()

	w := httptest.NewRecorder()
	r := httptest.NewRequest("GET", "/api/secrets/github", nil)

	router := chi.NewRouter()
	router.Route("/api/secrets/{namespace}", func(router chi.Router) {
		router.Use(CheckMembership(nil, true))
		router.Get("/", func(w http.ResponseWriter, r *http.Request) {
			t.Errorf("Must not invoke next handler in middleware chain")
		})
	})

	router.ServeHTTP(w, r)

	if got, want := w.Code, http.StatusUnauthorized; got != want {
		t.Errorf("Want status code %d, got %d", want, got)
	}
}

func TestCheckMembership_AuthorizeRead(t *testing.T) {
	controller := gomock.NewController(t)
	defer controller.Finish()

	w := httptest.NewRecorder()
	r := httptest.NewRequest("GET", "/api/secrets/github", nil)
	r = r.WithContext(
		request.WithUser(noContext, mockUser),
	)

	mockOrgService := mock.NewMockOrganizationService(controller)
	mockOrgService.EXPECT().Membership(gomock.Any(), gomock.Any(), "github").Return(true, false, nil).Times(1)

	router := chi.NewRouter()
	router.Route("/api/secrets/{namespace}", func(router chi.Router) {
		router.Use(CheckMembership(mockOrgService, false))
		router.Get("/", func(w http.ResponseWriter, r *http.Request) {
			w.WriteHeader(http.StatusTeapot)
		})
	})

	router.ServeHTTP(w, r)

	if got, want := w.Code, http.StatusTeapot; got != want {
		t.Errorf("Want status code %d, got %d", want, got)
	}
}

func TestCheckMembership_AuthorizeAdmin(t *testing.T) {
	controller := gomock.NewController(t)
	defer controller.Finish()

	w := httptest.NewRecorder()
	r := httptest.NewRequest("GET", "/api/secrets/github", nil)
	r = r.WithContext(
		request.WithUser(noContext, mockUser),
	)

	mockOrgService := mock.NewMockOrganizationService(controller)
	mockOrgService.EXPECT().Membership(gomock.Any(), gomock.Any(), "github").Return(true, true, nil).Times(1)

	router := chi.NewRouter()
	router.Route("/api/secrets/{namespace}", func(router chi.Router) {
		router.Use(CheckMembership(mockOrgService, true))
		router.Get("/", func(w http.ResponseWriter, r *http.Request) {
			w.WriteHeader(http.StatusTeapot)
		})
	})

	router.ServeHTTP(w, r)

	if got, want := w.Code, http.StatusTeapot; got != want {
		t.Errorf("Want status code %d, got %d", want, got)
	}
}

func TestCheckMembership_Unauthorized_Admin(t *testing.T) {
	controller := gomock.NewController(t)
	defer controller.Finish()

	w := httptest.NewRecorder()
	r := httptest.NewRequest("GET", "/api/secrets/github", nil)
	r = r.WithContext(
		request.WithUser(noContext, mockUser),
	)

	mockOrgService := mock.NewMockOrganizationService(controller)
	mockOrgService.EXPECT().Membership(gomock.Any(), gomock.Any(), "github").Return(true, false, nil).Times(1)

	router := chi.NewRouter()
	router.Route("/api/secrets/{namespace}", func(router chi.Router) {
		router.Use(CheckMembership(mockOrgService, true))
		router.Get("/", func(w http.ResponseWriter, r *http.Request) {
			t.Errorf("Must not invoke next handler in middleware chain")
		})
	})

	router.ServeHTTP(w, r)

	if got, want := w.Code, http.StatusUnauthorized; got != want {
		t.Errorf("Want status code %d, got %d", want, got)
	}
}

func TestCheckMembership_Unauthorized_Read(t *testing.T) {
	controller := gomock.NewController(t)
	defer controller.Finish()

	w := httptest.NewRecorder()
	r := httptest.NewRequest("GET", "/api/secrets/github", nil)
	r = r.WithContext(
		request.WithUser(noContext, mockUser),
	)

	mockOrgService := mock.NewMockOrganizationService(controller)
	mockOrgService.EXPECT().Membership(gomock.Any(), gomock.Any(), "github").Return(false, false, nil).Times(1)

	router := chi.NewRouter()
	router.Route("/api/secrets/{namespace}", func(router chi.Router) {
		router.Use(CheckMembership(mockOrgService, false))
		router.Get("/", func(w http.ResponseWriter, r *http.Request) {
			t.Errorf("Must not invoke next handler in middleware chain")
		})
	})

	router.ServeHTTP(w, r)

	if got, want := w.Code, http.StatusUnauthorized; got != want {
		t.Errorf("Want status code %d, got %d", want, got)
	}
}

func TestCheckMembership_Unauthorized_Error(t *testing.T) {
	controller := gomock.NewController(t)
	defer controller.Finish()

	w := httptest.NewRecorder()
	r := httptest.NewRequest("GET", "/api/secrets/github", nil)
	r = r.WithContext(
		request.WithUser(noContext, mockUser),
	)

	mockOrgService := mock.NewMockOrganizationService(controller)
	mockOrgService.EXPECT().Membership(gomock.Any(), gomock.Any(), "github").Return(true, true, errors.New("")).Times(1)

	router := chi.NewRouter()
	router.Route("/api/secrets/{namespace}", func(router chi.Router) {
		router.Use(CheckMembership(mockOrgService, false))
		router.Get("/", func(w http.ResponseWriter, r *http.Request) {
			t.Errorf("Must not invoke next handler in middleware chain")
		})
	})

	router.ServeHTTP(w, r)

	if got, want := w.Code, http.StatusUnauthorized; got != want {
		t.Errorf("Want status code %d, got %d", want, got)
	}
}
