      PROGRAM ILSDRV 
C****************************************************************************
C
C PUPOSE FOR THE DRIVER:
C
C    THIS PROGRAM USES AN ITERATIVE METHOD THAT ADHERES TO THE PROPOSED
C    ITERATIVE LINEAR SOLVER STANDARD VERSION 1.0 (S.F. ASHBY, M.K. SEAGER)
C    TO SOLVE THE LINEAR SYSTEM AX=B, WHERE THE SYSTEM ARISES FROM THE 
C    DISCRETIZATION OF A PARTIAL DIFFERENTIAL EQUATION.
C
C    THE PARTICULAR ITERATIVE SOLVER USED HERE IS CGCODE, A PACKAGE OF
C    FORTRAN 77 SUBROUTINES FOR THE SOLUTION OF LINEAR SYSTEMS USING
C    CONJUGATE GRADIENT METHODS.
C
C    THE INPUT PARAMETERS ARE READ FROM THE USER-PROVIDED FILE "IN".
C    THE OUTPUT FROM THIS DRIVER PROGRAM APPEARS IN THE FILE "OUT".
C    THE OUTPUT FROM THE SOLVER APPEARS IN THE FILE "OUTN".
C
C NOTD1:
C
C    THIS PROGRAM CALLS A SUBROUTINE TO SOLVE THE LINEAR SYSTEM ARISING FROM 
C    THE DISCRETIZATION OF THE FOLLOWING ELLIPTIC BOUNDARY VALUE PROBLEM:
C    (POISSON'S EQUATION WITH DIRICHLET BOUNDARY CONDITIONS ON A RECTANGLE)
C
C           LU = F, U IN OMEGA
C            U = G, U ON BOUNDARY OF OMEGA
C
C    WHERE 
C            OMEGA = [AX,BX]X[AY,BY]
C    AND
C            L = THE LAPLACEAN OPERATOR
C
C NOTE2:
C
C    WE DISCRETIZE THE ABOVE PROBLEM ON AN [NX BY NY] GRID, WHICH LEADS 
C    TO A SPARSE SYSTEM OF LINEAR EQUATIONS OF THE FORM:
C           
C            A*X = B
C
C    HERE, A IS OF ORDER N = NX*NY.
C
C PDE PROBLEM SPECIFIC INPUT FOR PDE DRIVER VIA USER SUPPLIED SUBPROGRAM
C   (1)  THE RIGHT HAND SIDE OF THE PROBLEM (CU0F(X,Y))
C   (2)  THE BOUDARY CONDITION OF THE PDE   (CU0G(X,Y))
C   (3)  ANALYTIC SOLN OF THE PDE FOR TESTS (CU0U(X,Y)) 
C
C INPUT FROM USER VIA UNIT IREAD:
C   (1)  LEVEL OF IO FROM SOLVER (IOLEVL)              
C   (2)  PRECONDITIONING KEY (IPCOND)                
C   (3)  STOPPING CRITERION KEY (ISTOP)                
C   (4)  MAXIMUM NUMBER OF ITERATIONS ALLOWED (ITMAX) 
C   (5)  TOLERANCE FOR THE SOLUTION (ERRTOL)           
C   (6)  METHOD DEPENDENT INPUT PARAMETERS (ETC...)  
C        INCLUDING:  ICYCLE,NCE,ICG,NDEG,COND,AA,BB
C        (SEE CGCODE REPORT AND DOCUMENTATION FOR DETAILED EXPLANATION)
C   (7)  WHETHER TO PRINT NUMERICAL RESULTS (KY)    
C   (8)  APPROX NUMBER OF OPS (UNITS OF N) TO FIGURE MFLOPS (NUMOPS)
C   (9)  NUMBER OF INTERIOR POINTS X DIRECTION (NX) 
C   (10) NUMBER OF INTERIOR POINTS Y DIRECTION (NY)  
C   (11) LEFT ENDPOINT FOR X IN SPACE (AX)            
C   (12) LEFT ENDPOINT FOR Y IN SPACE (AY)             
C   (13) RIGHT ENDPOINT FOR X IN SPACE (BX)             
C   (14) RIGHT ENDPOINT FOR Y IN SPACE (BY)              
C
C OUTPUT FROM DRIVER:
C   (1)  THE STATISTICS CONCERNING THE RUN VIA OUTPUT TO UNIT IRITE
C   (2)  THE METHOD DEPENDENT OUTPUT MESSAGES TO UNIT IOUNIT
C   (3)  THE APPROXIMATED SOLUTION AT EACH GRID POINT VIA OUTPUT TO UNIT IOUNIT
C
C PARAMETERS:
C   IREAD           = UNIT TO READ ALL INPUT FROM
C   IRITE           = UNIT TO WRITE STATISTICS TO
C   IOUNIT          = UNIT TO WRITE NUMBERS TO
C   NGRID           = MAX DIMENSIONS IN EACH DIRECTION OF THE GRID
C   NXX             = MAX LENGTH OF ALL VECTORS
C   NONZK           = MAX NUMBER OF NONZEROS IN THE SYSTEM MATRIX
C   ETC...          = METHOD DEPENDENT PARAMETERS
C    INCLUDING: NIPAR,NRPAR,MAXICY,MAXNCE,NIWK,N1,N2,NRWK
C    (SEE CGCODE REPORT AND DOCUMENTATION FOR DETAILED EXPLANATION)
C
C VARIABLES: 
C   A,IA            = DISCRETIZED OPERATOR MATRIX A
C   B               = RHS VECTOR
C   X               = SOLUTION VECTOR
C   TRUE,RHS        = TEMPORARY VECTORS USED BY DRIVER FOR STATISTICS
C   IPARAM,RPARAM   = INTEGER AND DBLE PARAMETERS FOR THE SOLVER
C   IWORK,RWORK     = INTEGER AND DBLE WORK ARRAYS FOR THE SOLVER 
C   TARRAY,T0,AFTER,
C   BEFORE,OVERHD,
C   GARBGE          = TIMING TEMPORARY VARIABLES
C   TITLE           = TITLE OF METHOD FOR OUTPUT
C   KY              = 0=NO OUTPUT,1=PRINT NUMERICAL SOLUTION
C   NUMOPS          = APPROX OPERATION COUNT FOR MEGAFLOP FIGURES
C   NX,NY           = THE NUMBER OF POINTS IN X AND Y DIRECTIONS IN GRID
C   N               = THE DIMENSION OF THE SYSTEM MATRIX, = NX*NY
C   AX,AY,BX,BY     = ENDPOINTS IN SPACE FOR X AND Y DIRECTIONS.
C   IRITE           = I/O UNIT FOR STATISTICS
C   IOUNIT          = I/O UNIT FOR NUMERICAL SOLUTION AT GRID POINTS
C   ETC...          = METHOD DEPENDENT VARIABLES
C    INCLUDING: IOLEVL,IPCOND,ISTOP,ITMAX,ERRTOL,ICYCLE,NCE,ICG,NDEG,COND,AA,BB
C    (SEE CGCODE REPORT AND DOCUMENTATION FOR DETAILED EXPLANATION)
C   
C REQUIRED EXTERNAL ROUTINES:
C   MATVEC          = MATRIX-VECTOR PRODUCT ROUTINE
C   PCONDL          = PRECONDIONING ROUTINE
C    (SEE CGCODE REPORT AND DOCUMENTATION FOR DETAILED EXPLANATION)
C
C AUTHOR -->  MICHAEL JAY HOLST
C DATE   -->  18 MARCH 1990
C****************************************************************************
      IMPLICIT  DOUBLE PRECISION(A-H,O-Z)
C
C     *** PARAMETERS ***
      PARAMETER    (IREAD=7,IRITE=8,IOUNIT=9)
      PARAMETER    (NGRID=63,NXX=NGRID*NGRID,NONZK=5*NXX)
      PARAMETER    (NIPAR=40,NRPAR=40)
      PARAMETER    (MAXICY=5,MAXNCE=5)
      PARAMETER    (NIWK=MAXICY*MAXNCE)
      PARAMETER    (N1=5*NXX,N2=4*MAXICY*MAXNCE+2,NRWK=N1+N2)
C
C     *** STORAGE AND EXTERNALS ***
      DIMENSION    A(NONZK+10),IA(10),Q(NXX),IQ(10)
      DIMENSION    X(NXX),TRUE(NXX),B(NXX),RHS(NXX)
      DIMENSION    IPARAM(NIPAR),RPARAM(NRPAR)
      DIMENSION    RWORK(NRWK),IWORK(NIWK)
      REAL         TARRAY(2),T0,AFTER,BEFORE,OVERHD,GARBGE
      CHARACTER*50 TITLE
      EXTERNAL     MATVEC,PCONDL
C
C     *** OPEN I/O FILES ***
      OPEN(UNIT=IREAD, FILE='in',  STATUS='UNKNOWN')
      OPEN(UNIT=IRITE, FILE='out', STATUS='UNKNOWN')
      OPEN(UNIT=IOUNIT,FILE='outn',STATUS='UNKNOWN') 
      REWIND(IREAD)
      REWIND(IRITE)
      REWIND(IOUNIT)
C
C     *** SETUP SOME CGCODE PARAMETERS ***
      IPARAM(1) = NIPAR
      IPARAM(2) = NRPAR
      IPARAM(3) = NIWK
      IPARAM(4) = NRWK
      IPARAM(5) = IOUNIT
C
C     *** READ IN PARAMETERS FROM USER AND DISCRETIZE THE PDE ***
      CALL SETUP (IPARAM,RPARAM,NX,NY,AX,AY,BX,BY,KY,NUMOPS,IREAD,IRITE)
      IF ((NX .GT. NGRID) .OR. (NY .GT. NGRID)) GOTO 91
      CALL BUILD (N,AX,AY,BX,BY,NX,NY,A,IA,B,TRUE,IRITE) 
      CALL BLDPC (A,IA,Q,IQ,N) 
C
C     *** SAVE RHS FOR LATER SINCE IT IS CHANGED BY CGCODE ***
      CALL DCOPY(N,B,1,RHS,1) 
C
C     *** INITIAL GUESS ***
      CALL GETX0(X,N)
C
C     *** FOR CONVEX: START TIMER ***
CC    CALL ETIME(TARRAY)
CC    T0 = TARRAY(1)
CC    CALL ETIME(TARRAY)
CC    OVERHD = TARRAY(1) - T0
CC    CALL ETIME(TARRAY)
CC    BEFORE = TARRAY(1)
C
C     *** FOR CRAY XMP: START TIMER ***
C     GARBGE = SECOND( )
C     T0 = SECOND( )
C     OVERHD = SECOND( ) - T0
C     BEFORE = SECOND( )
C
C     *** CALL THE ITERATIVE SOLVER ***
      CALL DCGDRV (MATVEC,PCONDL,PCONDR,A,IA,X,B,N,Q,IQ,P,IP,  
     2             IPARAM,RPARAM,IWORK,RWORK,IERROR)
C
C     *** FOR CRAY XMP: STOP TIMER ***
C     AFTER = SECOND( )
C
C     *** FOR CONVEX: STOP TIMER ***
CC    CALL ETIME(TARRAY)
CC    AFTER = TARRAY(1)
C
C     *** CALCULATE EXECUTION TIME (AND TRY TO APPROXIMATE MFLOP RATE) ***
      CPUTME = (AFTER - BEFORE) - OVERHD
      ZMFLPS = DBLE(N*NUMOPS) / CPUTME / 1.0E6
C
C     *** CALCULATE THE TRUE ERROR IN APPROXIMATION ***
      CALL DAXPY(N,-1.0D0,X,1,TRUE,1)
      ERROR = DNRM2(N,TRUE,1)
C
C     *** CALCULATE THE RESIDUAL ERROR IN APPROXIMATION ***
      CALL MATVEC(0,A,IA,BDUMM,X,TRUE,N)
      CALL DAXPY(N,-1.0D0,RHS,1,TRUE,1)
      RESID = DNRM2(N,TRUE,1)
C
C     *** OUTPUT THE PARAMETER INFORMATION AND RESULT ***
      TITLE = ' SOLUTION OF POISSON EQUATION USING CGCODE '
      CALL SOUT(TITLE,KY,N,X,IPARAM,RPARAM,IERROR,CPUTME,ZMFLPS,
     2          NUMOPS,ERROR,RESID,AX,AY,BX,BY,NX,NY,IRITE)
      GOTO 99
C
C     *** PROBLEMS ***
 91   CONTINUE
      WRITE (IRITE,*) ' NOT ENOUGH STORAGE DECLARED FOR THIS GRID '
C
C     *** END IT ***
 99   CONTINUE
      CLOSE(IREAD)
      CLOSE(IRITE)
      CLOSE(IOUNIT)
      STOP 'ILSDRVOK'
      END
      SUBROUTINE SETUP (IPARAM,RPARAM,NX,NY,AX,AY,BX,BY,KY,NUMOPS,
     2                  IREAD,IRITE)
C****************************************************************************
C THIS ROUTINE READS IN SOME INITIAL VALUES ABOUT THE PDE AND FOR THE SOLVER.
C****************************************************************************
      IMPLICIT  DOUBLE PRECISION(A-H,O-Z)
      DIMENSION IPARAM(*),RPARAM(*)
C
C     *** INPUT THE CONTROLING PARAMETERS ***
      READ (IREAD,*) IOLEVL
      READ (IREAD,*) IPCOND
      READ (IREAD,*) ISTOP
      READ (IREAD,*) ITMAX
      READ (IREAD,*) ERRTOL
      IPARAM(6) = IOLEVL
      IPARAM(7) = IPCOND
      IPARAM(8) = ISTOP
      IPARAM(9) = ITMAX
      RPARAM(1) = ERRTOL
C
C     *** READ METHOD PARAMETERS FROM USER ***
      READ (IREAD,*)
      READ (IREAD,*) ICYCLE
      READ (IREAD,*) NCE
      READ (IREAD,*) ICG
      READ (IREAD,*)
      READ (IREAD,*) NDEG
      READ (IREAD,*) COND
      READ (IREAD,*) AA
      READ (IREAD,*) BB
      IPARAM(31) = ICYCLE
      IPARAM(32) = NCE
      IPARAM(33) = ICG
      IPARAM(34) = NDEG
      RPARAM(31) = COND
      RPARAM(32) = AA
      RPARAM(33) = BB
C
C     *** READ IN SOLUTION KEY ***
      READ (IREAD,*)
      READ (IREAD,*) KY
      READ (IREAD,*) NUMOPS
C
C     *** READ IN PDE PARAMETERS ***
      READ (IREAD,*)
      READ (IREAD,*) NX
      READ (IREAD,*) NY
      READ (IREAD,*) AX
      READ (IREAD,*) AY
      READ (IREAD,*) BX
      READ (IREAD,*) BY
C
C     *** RETURN AND END ***
      RETURN
      END
      SUBROUTINE GETX0(X,N)
C****************************************************************************
C MAKE THE INITIAL GUESS AT THE SOLUTION.
C****************************************************************************
      IMPLICIT  DOUBLE PRECISION(A-H,O-Z)
      DIMENSION X(*)
      DO 10 I = 1, N
         X(I) = 0.0D0
 10   CONTINUE
      RETURN
      END
      SUBROUTINE SOUT(TITLE,KY,N,X,IPARAM,RPARAM,IERROR,CPUTME,ZMFLPS,
     2                NUMOPS,ERROR,RESID,AX,AY,BX,BY,NX,NY,IRITE)
C****************************************************************************
C THIS ROUTINE PRINTS OUT THE CONTROLLING PARAMETERS, ITERATION INFORMATION,
C AND IF SPECIFIED, ALSO PRINTS OUT THE COMPUTED SOLUTION.
C****************************************************************************
      IMPLICIT  DOUBLE PRECISION(A-H,O-Z)
      DIMENSION    X(*),IPARAM(*),RPARAM(*)
      CHARACTER*50 TITLE
C
C     *** DECODE IPARAM ARRAY ***
      IOUNIT = IPARAM(5)
      IOLEVL = IPARAM(6)
      IPCOND = IPARAM(7)
      ISTOP  = IPARAM(8)
      ITMAX  = IPARAM(9) 
      ITERS  = IPARAM(10)
      ICYCLE = IPARAM(31)
      NCE    = IPARAM(32) 
      ICG    = IPARAM(33)
      NDEG   = IPARAM(34)
C
C     *** DECODE RPARAM ARRAY ***
      ERRTOL = RPARAM(1)  
      STPTST = RPARAM(2)
      CONDES = RPARAM(31)
      AA     = RPARAM(32) 
      BB     = RPARAM(33)
      SCRLRS = RPARAM(34)
C
C     *** MAKE HEADER LISTING PARAMETERS FOR THE PROBLEM SOLVED ***
      ZHX = (BX-AX) / DBLE(NX + 1)
      ZHY = (BY-AY) / DBLE(NY + 1)
      WRITE(IRITE,600)
      WRITE(IRITE,610)
      WRITE(IRITE,600)
      WRITE(IRITE,801)
      WRITE(IRITE,530)' INTERIOR PTS IN X          (NX)=======> ',NX
      WRITE(IRITE,530)' INTERIOR PTS IN Y          (NY)=======> ',NY
      WRITE(IRITE,520)' MINIMUM X IN GRID          (AX)=======> ',AX
      WRITE(IRITE,520)' MINIMUM Y IN GRID          (AY)=======> ',AY
      WRITE(IRITE,520)' MAXIMUM X IN GRID          (BX)=======> ',BX
      WRITE(IRITE,520)' MAXIMUM Y IN GRID          (BY)=======> ',BY
      WRITE(IRITE,520)' STEPSIZE IN X              (ZHX)======> ',ZHX
      WRITE(IRITE,520)' STEPSIZE IN Y              (ZHY)======> ',ZHY
      WRITE(IRITE,801)
      WRITE(IRITE,600)
      WRITE(IRITE,801)
      WRITE(IRITE,500)' THE PROBLEM TITLE IS:   ',TITLE 
      WRITE(IRITE,801)
      WRITE(IRITE,530)' DIMENSION OF LINEAR SYSTEM (N=NX*NY)==> ',N
      WRITE(IRITE,530)' INFORMATION LEVEL          (IOLEVL)===> ',IOLEVL
      WRITE(IRITE,530)' PRECONDITIONING KEY        (IPCOND)===> ',IPCOND
      WRITE(IRITE,530)' STOPPING CRITERION KEY     (ISTOP)====> ',ISTOP 
      WRITE(IRITE,530)' MAXIMUM ALLOWED ITERATION  (ITMAX)====> ',ITMAX
      WRITE(IRITE,520)' ERROR TOLERANCE            (ERRTOL)===> ',ERRTOL
      WRITE(IRITE,530)' CONDITION ESTIMATE RATE    (ICYCLE)===> ',ICYCLE
      WRITE(IRITE,530)' CONDITION ESTIMATES        (NCE)======> ',NCE
      WRITE(IRITE,530)' CG METHOD USED             (ICG)======> ',ICG
      WRITE(IRITE,530)' DEGREE OF PREC POLY        (NDEG)=====> ',NDEG
      WRITE(IRITE,520)' INITIAL MIN EIG ESTIMATE   (AA)=======> ',AA
      WRITE(IRITE,520)' INITIAL MAX EIG ESTIMATE   (BB)=======> ',BB
      WRITE(IRITE,801)
      WRITE(IRITE,530)' OUTPUT KEY                 (KY)=======> ',KY
      WRITE(IRITE,530)' OP COUNT (IN UNITS OF N)   (NUMOPS)===> ',NUMOPS
      WRITE(IRITE,801)
      WRITE(IRITE,530)' COMPLETION CODE            (IERROR)===> ',IERROR
      WRITE(IRITE,530)' ITERATIONS TAKEN           (ITERS)====> ',ITERS
      WRITE(IRITE,520)' FINAL STOPPING TEST        (STPTST)===> ',STPTST
      WRITE(IRITE,520)' FINAL CONDITION ESTIMATE   (CONDES)===> ',CONDES
      WRITE(IRITE,520)' SCALED RELATIVE RESIDUAL   (SCRLRS)===> ',SCRLRS
      WRITE(IRITE,520)' EXECUTION TIME             (CPUTME)===> ',CPUTME
      WRITE(IRITE,520)' APPROXIMATE MEGAFLOP RATE  (ZMFLPS)===> ',ZMFLPS
      WRITE(IRITE,520)' RESIDUAL ERROR (B-A*XCG)   (RESID)====> ',RESID
      WRITE(IRITE,520)' PDE ANAL ERROR (XTRUE-XCG) (ERROR)====> ',ERROR
      WRITE(IRITE,801)
      WRITE(IRITE,600)
      WRITE(IRITE,801)
C
C     *** PRINT OUT THE SOLUTION VALUES ***
      IF (KY.EQ.1) THEN
         WRITE(IOUNIT,540) TITLE 
         WRITE(IOUNIT,510) (NX+2)*(NY+2)
         WRITE(IOUNIT,801)
         X0   = AX
         Y0   = AY
         XNP1 = BX
         YNP1 = BY
         DO 10 I = 0, NX+1
            XI = AX + DBLE(I) * ZHX
            WRITE (IOUNIT,550) XI,Y0,CU0G(XI,Y0)
 10      CONTINUE
         DO 30 J = 1, NY
            YJ = AY + DBLE(J) * ZHY
            WRITE (IOUNIT,550) X0,YJ,CU0G(X0,YJ)
            DO 20 I = 1, NX
               XI = AX + DBLE(I) * ZHX
               IDX = (J-1)*NX + I
               WRITE (IOUNIT,550) XI,YJ,X(IDX)
 20         CONTINUE
            WRITE (IOUNIT,550) XNP1,YJ,CU0G(XNP1,YJ)
 30      CONTINUE
         DO 40 I = 0, NX+1
            XI = AX + DBLE(I) * ZHX
            WRITE (IOUNIT,550) XI,YNP1,CU0G(XI,YNP1)
 40      CONTINUE
         WRITE (IOUNIT,801)
      ENDIF
C
C     *** FORMAT STATEMENTS ***
 500  FORMAT (1X,A,A)
 510  FORMAT (I10)
 520  FORMAT (1X,A,1PD15.7) 
 530  FORMAT (1X,A,I15)
 540  FORMAT (1X,A)
 550  FORMAT (1X,3(1PD15.7,10X))
 600  FORMAT (1X,'=======',
     2'===============================================================',
     3           '=======')
 610  FORMAT (1X,'=======',
     2'============ PARTIAL DIFFERENTIAL EQUATION SOLVER =============',
     3           '=======')
 801  FORMAT (1X)
C
C     *** RETURN AND END ***
      RETURN
      END
      SUBROUTINE BUILD (N,AX,AY,BX,BY,NX,NY,A,IA,B,TRUE,IRITE) 
C****************************************************************************
C THIS ROUTINE BUILDS THE DISCRETE SYSTEM FROM THE USERS PDE SUBROUTINES.
C****************************************************************************
      IMPLICIT  DOUBLE PRECISION(A-H,O-Z)
      DIMENSION A(*),IA(*),B(*),TRUE(*)
C
C     *** DEFINE N AND SETUP VARIOUS CONSTANTS FOR THE BUILD ***
      N     = NX * NY
      ZHX   = (BX-AX) / DBLE(NX + 1)
      ZHY   = (BY-AY) / DBLE(NY + 1)
      ZHXOY = ZHX / ZHY
      ZHYOX = ZHY / ZHX
      ZHPT  = ZHXOY / ZHYOX
      X0    = AX
      Y0    = AY
      XNP1  = BX
      YNP1  = BY
      IA(1) = 1
      IA(2) = N+1
      IA(3) = 2*N+1
      IA(4) = 3*N+1
      IA(5) = 4*N+1
      IA(6) = NX
      IA(7) = NY
C
C     *** BUILD THE MESH POINTS, THE OPERATOR, THE RHS, AND TRUE ANAL SOLN ***
      IROW  = 0
      DO 10 J = 1, NY
         DO 10 I = 1, NX
            XI   = AX + DBLE(I) * ZHX
            YJ   = AY + DBLE(J) * ZHY
            IROW = IROW+1
            TRUE(IROW) = CU0U(XI,YJ)
            B(IROW) = - ZHX*ZHY*CU0F(XI,YJ)
C
C           *** SOUTH NEIGHBOR ***
            COEF = - ZHXOY
            IF (J .NE. 1) THEN
               A(IROW+IA(5)-1) = COEF
            ELSE
               A(IROW+IA(5)-1) = 0.0D0
               B(IROW) = B(IROW) - COEF*CU0G(XI,Y0)
            ENDIF
C
C           *** WEST NEIGHBOR ***
            COEF = - ZHYOX
            IF (I .NE. 1) THEN
               A(IROW+IA(3)-1) = COEF
            ELSE
               A(IROW+IA(3)-1) = 0.0D0
               B(IROW) = B(IROW) - COEF*CU0G(X0,YJ)
            ENDIF
C
C           *** POINT ITSELF ***
            COEF = 4.0D0*ZHPT
            A(IROW+IA(1)-1) = COEF
C
C           *** EAST NEIGHBOR ***
            COEF = - ZHYOX
            IF (I .NE. NX) THEN
               A(IROW+IA(2)-1) = COEF
            ELSE
               A(IROW+IA(2)-1) = 0.0D0
               B(IROW) = B(IROW) - COEF*CU0G(XNP1,YJ)
            ENDIF
C
C           *** NORTH NEIGHBOR ***
            COEF = - ZHXOY
            IF (J .NE. NY) THEN
               A(IROW+IA(4)-1) = COEF
            ELSE
               A(IROW+IA(4)-1) = 0.0D0
               B(IROW) = B(IROW) - COEF*CU0G(XI,YNP1)
            ENDIF
 10   CONTINUE
C
C     *** RETURN AND END ***
      RETURN
      END
      SUBROUTINE MATVEC(JOB,A,IA,W,X,Y,N)
C****************************************************************************
C SPARSE MATRIX MULTIPLICATION.
C INTERFACE ROUTINE TO THE USERMV 5 DIAGONAL MATRIX-VECTOR PRODUCT ROUTINE.
C****************************************************************************
      IMPLICIT  DOUBLE PRECISION(A-H,O-Z)
      DIMENSION A(*),IA(*),W(*),X(*),Y(*)
C
C     *** DO THE MULTIPLICATION ***
      CALL USERMV(A(IA(5)),A(IA(3)),A(IA(1)),A(IA(2)),A(IA(4)),
     2            IA(6),IA(7),X,Y)
C
C     *** GO HOME ***
      RETURN
      END
      SUBROUTINE USERMV(CL,BL,A,BU,CU,NX,NY,X,Y)
C****************************************************************************
C A FIVE DIAGONAL MATRIX-VECTOR PRODUCT ROUTINE.
C THE DIAGONALS ARE AS FOLLOWS:
C    THE COEFFICIENTS OF THE ITH EQUATION ARE STORED IN THE ITH COMPONENTS 
C    OF THE ARRAYS CONTAINING THE DIAGONALS.  
C    THE DIAGONALS OF THE OPERATOR ARE FROM LEFT TO RIGHT:   CL,BL,A,BU,CU
C    NX => THE NUMBER OF MESH POINTS IN THE X-DIRECTION.  
C          WITH THE NATURAL ORDERING, NX IS THE ORDER OF EACH TRIDIAGONAL 
C          BLOCK IN THE BLOCK-TRIDIAGONAL OPERATOR.
C    NY => THE NUMBER OF MESH POINTS IN THE Y-DIRECTION. 
C          WITH THE NATURAL ORDERING, NY IS THE NUMBER OF TRIDIAGONAL 
C          BLOCKS IN THE BLOCK-TRIDIAGONAL OPERATOR.
C****************************************************************************
      IMPLICIT  DOUBLE PRECISION(A-H,O-Z)
      DIMENSION CL(*),BL(*),A(*),BU(*),CU(*),X(*),Y(*)
C
C     *** RECOVER MATRIX DIMENSION ***
      N = NX*NY
C
C     *** HANDLE FIRST BLOCK ***
      I = 1
      Y(I) = A(I)*X(I)+BU(I)*X(I+1)+CU(I)*X(I+NX)
      DO 10 I=2,NX
         Y(I) = BL(I)*X(I-1)+A(I)*X(I)+BU(I)*X(I+1)+CU(I)*X(I+NX)
 10   CONTINUE
C
C     *** HANDLE MIDDLE BLOCKS ***
      DO 20 I=NX+1,N-NX
         Y(I) = CL(I)*X(I-NX)+BL(I)*X(I-1)+A(I)*X(I)+BU(I)*X(I+1)
     2         +CU(I)*X(I+NX)
 20   CONTINUE
C
C     *** HANDLE LAST BLOCK ***
      DO 30 I=N-(NX-1),N-1
         Y(I) = CL(I)*X(I-NX)+BL(I)*X(I-1)+A(I)*X(I)+BU(I)*X(I+1)
 30   CONTINUE
      I = N
      Y(I) = CL(I)*X(I-NX)+BL(I)*X(I-1)+A(I)*X(I)
C
C     *** RETURN AND END ***
      RETURN
      END
      SUBROUTINE BLDPC(A,IA,Q,IQ,N)
C****************************************************************************
C BUILD A SIMPLE DIAGONAL SCALING PRECONDITIONER FOR A FIVE-DIAGONAL MATRIX.
C****************************************************************************
      IMPLICIT  DOUBLE PRECISION(A-H,O-Z)
      DIMENSION A(*),IA(*),Q(*),IQ(*)
      DO 10 I = 1, N
         Q(I) = 1.0D0 / A(IA(1)+(I-1))
 10   CONTINUE 
      RETURN
      END
      SUBROUTINE PCONDL(JOB,Q,IQ,W,X,Y,N)
C****************************************************************************
C A SIMPLE DIAGONAL SCALING PRECONDITIONER FOR A FIVE-DIAGONAL OPERATOR.
C****************************************************************************
      IMPLICIT  DOUBLE PRECISION(A-H,O-Z)
      DIMENSION Q(*),IQ(*),W(*),X(*),Y(*)
      DO 10 I = 1, N
         Y(I) = Q(I)*X(I)
 10   CONTINUE
      RETURN
      END
      FUNCTION CU0F (X,Y) 
C****************************************************************************
C THIS ROUTINE DEFINES THE RIGHT HAND SIDE OF THE PDE.
C****************************************************************************
      IMPLICIT  DOUBLE PRECISION(A-H,O-Z)
      PARAMETER (PI = 3.1415926535,NNX=3,NNY=1)
      CU0F = - PI*PI*DBLE(NNX*NNX + NNY*NNY)
     2       * (DSIN(DBLE(NNX)*PI*X) * DSIN(DBLE(NNY)*PI*Y))
      RETURN
      END
      FUNCTION CU0G (X,Y) 
C****************************************************************************
C THIS ROUTINE IS THE BOUDARY CONDITION FOR THE PDE.
C****************************************************************************
      IMPLICIT  DOUBLE PRECISION(A-H,O-Z)
      CU0G = 0.0D0
      RETURN
      END
      FUNCTION CU0U (X,Y) 
C****************************************************************************
C THIS ROUTINE IS THE ANALYTIC SOLUTION TO THE PDE (FOR TESTING PURPOSES).
C****************************************************************************
      IMPLICIT  DOUBLE PRECISION(A-H,O-Z)
      PARAMETER (PI = 3.1415926535,NNX=3,NNY=1)
      CU0U = DSIN(DBLE(NNX)*PI*X) * DSIN(DBLE(NNY)*PI*Y)
      RETURN
      END
