/*##########################################################################*/
/*                                                                          */
/* Copyright (c) 2004 CASPUR Consortium                                     */
/*                  C.A.S.P.U.R.                                            */ 
/*		 Via Dei Tizii, 6B -00185-Roma (Italy)                      */
/*		 E_MAIL:mirdir@caspur.it                                    */
/* All rights reserved.                                                     */
/*                                                                          */
/* Permission is hereby granted, without written agreement and without      */
/* license or royalty fees, to use, copy, modify, and distribute this       */
/* software and its documentation for any purpose, provided that the        */
/* above copyright notice and the following two paragraphs and the author   */
/* reference appear in all copies of this software.                         */
/*                                                                          */
/* IN NO EVENT SHALL THE CASPUR CONSORTIUM BE LIABLE TO ANY PARTY FOR       */
/* DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING  */
/* OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE       */
/* CASPUR CONSORTIUM HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.    */
/*                                                                          */
/* THE CASPUR CONSORTIUM SPECIFICALLY DISCLAIMS ANY WARRANTIES,             */
/* INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY */
/* AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER   */
/* IS ON AN "AS IS" BASIS, AND THE CASPUR CONSORTIUM HAS NO OBLIGATION TO   */
/* PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.   */
/*                                                                          */
/*##########################################################################*/

/* 

   Revision 2.1 2005/08/05 12:38:00 carol
   Bug fix: in case of i/o errors while reading the source tree, 
            print the right return code (-9)
*/ 

/*$Id: mirdir.c, v 2 2004/11/19 15:00:00
 
   Version tested and working under Linux, Solaris, 
   Tru64, AIX, IRIX,MACOS/X, HP/UX
   Timestamp for symbolic links is not supported.*/

/* $Id: mirdir.c,v 1.48 1998/10/02 10:51:17 ruten Exp $ */
/*
   $Log: mirdir.c,v $
   Revision 1.48  1998/10/02 10:51:17  ruten
   Adjustments for SunOS 5.6

   Revision 1.47  1998/02/09 11:59:48  ruten
   Option -a added

   Revision 1.46  1997/09/24 10:54:31  ruten
   Final file counter reported in statistics

   Revision 1.45  1997/09/15 09:36:01  ruten
   Gather/print statistics added.

   Revision 1.44  1997/06/12 13:50:47  ruten
   Set timestamp disabled in all places in presence of -d option

   Revision 1.43  1996/05/17 16:09:55  ruten
   For regular files two messages (Removing + Creating) replaced
   with one - Replacing ...
   Message "Removing file or link" split to "Removing file" and "Removing link"

   Revision 1.42  1996/03/06 18:03:07  ruten
   Small change in printout.

   Revision 1.41  1996/02/22 18:57:37  ruten
   Named pipes support included.

   Revision 1.40  1996/02/15 15:02:25  ruten
   The last change for Sun leaded to blow-up on Linux, as
   d_reclen has different meanings. So my_scandir is rewritten to avoid
   compatibility problems.
   Another change - core is ignored in source only if it's not a directory.

*/

/* #define tv_sec _tv_sec */  /* to defeat SunOS bug */

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <sys/stat.h>
#include <utime.h>
#include <fcntl.h>
#include <dirent.h>
#include <time.h>

/*
#include <sys/types.h>
#include <limits.h>
*/

#ifndef PATH_MAX
#define PATH_MAX 1023
#endif


#define MAXEXCEPT 1000

#define IS_DIR(s)  S_ISDIR(s.st_mode)    /* Macro to check if the file is a directory. */
#ifdef S_ISLNK
#define IS_LINK(s) S_ISLNK(s.st_mode)    /* Checks if the file is a symbolic link. */
#else
#define IS_LINK(s) ((s.st_mode&0170000)==0120000) 
#endif
#define IS_REG(s)  S_ISREG(s.st_mode)    /* Checks if the file is a regular file. */
#define IS_PIPE(s) S_ISFIFO(s.st_mode)   /* Checks if the file is a pipe or a FIFO special file. */



/*extern char *strdup(const char *p);*/


/* Flags */
static int print_diff;
static int print_action;
static int initial_print_action;
static int no_chown;
static int global_keepacc;
static int no_timecheck;
static int do_action;
static int ACCESS;





/* Number of bytes to compare in files. < 0 - compare all */
static long check_head;

static int changed;




/* Struct to keep the characteristic of each Exception in the exceptions file */
typedef struct {
  char *name;
  int  nlen;
  char fullstop;
  char keepdir;
  char keepowner;
  char keepacc;
} except_t;




static except_t zero_except;

static except_t e[MAXEXCEPT];    /* Buffer that keeps track of every exception. */
static int nexcept = 0;          /* Number of exceptions. */
static int currexcept;

/* Total files processed */
static int counter = 0;

/* statistics */
static struct {
  long lread_kb,lread_rest;
  long rread_kb,rread_rest;
  long write_kb,write_rest;
  long chown,lchown,chmod,settime;
  long unlink,rmdir,mkdir,mkfifo,symlink;
  long replfile,crefile;
} si;



static char Pfrom[PATH_MAX+1];   /* Source Directory */
static char Pto[PATH_MAX+1];     /* Target Directory */



typedef struct {
  int Lfrom;
  int Lto;
  int except_off;
  int nodelete;
  except_t *exc;
  int print_action;
} layer_t;




static char Remove[]    = "Removing file ----------->";
static char Remlink[]   = "Removing link ----------->";
static char Remdir[]    = "Removing subtree -------->";
static char Skipcheck[] = "Skipping check for ------>";
static char Skipchange[]= "Skipping changes for ---->";
static char Skiprepl[]  = "Skipping replacement of ->";
static char Skipdel[]   = "Skipping deletion of ---->";
static char Creadir[]   = "Creating directory ------>";
static char Creafile[]  = "Creating file ----------->";
static char Replfile[]  = "Replacing file ---------->";
static char Creapipe[]  = "Creating named pipe ----->";
static char Setlink[]   = "Setting link ------------>";
static char Setown[]    = "Setting owner for ------->";
static char Setlnkown[] = "Setting owner for link -->";
static char Setmode[]   = "Setting access for ------>";
static char Settime[]   = "Setting times for ------->";


/* Enumeration */
typedef enum { err_v, none_v, dir_v, reg_v, link_v, pipe_v } kind_t;
 
/*static char *ckind[] = {"error","absent","directory","file","link","named pipe" };*/

static int cmpr(layer_t *t, int nosource, int notarget);
static int try_remove(layer_t *t, int nosource, kind_t kind);



/**************************************** FUNCTION: addlong ****************************************/
static void addlong(long* high, long * low, long add)
{
  *low += add;            /* Adds the number of bytes read */
  *high += (*low >> 10);
  *low &= 0x3ff;
}


/**************************************** FUNCTION: no_actions ****************************************/
static void no_actions(void)
{
  fprintf(stderr,"No actions for %s\n",Pto);
}



/**************************************** FUNCTION: set_except ****************************************/
static void set_except(layer_t *new, int off)
{
  int Lt;
  new->nodelete = 0;
  new->exc = &zero_except;
  new->except_off = off;

  if( off )
    return;
  Lt = new->Lto;
  for( ;currexcept < nexcept; currexcept++ )
    {
      int Le = e[currexcept].nlen;     /* Keep the length of the exception name */
      int L = (Le > Lt)? Lt:Le; /* min */
      int rc = memcmp(e[currexcept].name,Pto,L);
      
      /* Memory area for the name of the exception is smaller than path where the copy is destinated. */
      if( rc < 0 )
	continue;
      /* Memory area for the name of the exception is bigger than path where the copy is destinated. */
      if( rc > 0 )
	break;
      /* Size of the name of the exception is smaller than the size of the name of the path where we are copying */
      if( Le < Lt )
	continue;
      /* Size of the exception name is bigger. */
      if( Le > Lt )
	if( e[currexcept].name[Lt] == '/' )
	  {
	    new->nodelete = 1;
	    return;
	  }
	else
	  break;
      assert( Le == Lt );
      /* The exception is stored */
      new->exc = &(e[currexcept]);
      return;
    }
  new->except_off = 1;
} 





/**************************************** FUNCTION: new_level ****************************************/
static int new_level(layer_t *new, const layer_t *old, const char *d_name)
{
  int L = strlen(d_name);
  int Lf = old->Lfrom;
  int Lt = old->Lto;
  
  new->Lfrom = Lf + 1 + L;
  if( new->Lfrom >= sizeof(Pfrom) )
    {
      fprintf(stderr,"Name too long: %s/%s\n",Pfrom,d_name);
      return -1;
    }
  new->Lto = Lt + 1 + L;
  if( new->Lto >= sizeof(Pto) )   
    {
      fprintf(stderr,"Name too long: %s/%s\n",Pto  ,d_name);
      return -1;
    }
  
  Pto[Lt] = '/';
  strcpy(Pto+Lt+1,d_name);
  
  Pfrom[Lf] = '/';
  strcpy(Pfrom+Lf+1,d_name);
  
  set_except(new,old->except_off);
  
  new->print_action = print_action;
  return 0;
}




/**************************************** FUNCTION: reset_level ****************************************/
void reset_level(const layer_t *old)
{
  Pfrom[old->Lfrom] = 0;
  Pto[old->Lto] = 0;
  print_action = old->print_action;
}





/**************************************** FUNCTION: fatal ****************************************/
void fatal(const char *mes)
  {
    fprintf(stderr,"%s\n",mes);
    exit(16);
  }




/**************************************** FUNCTION: fatal2 ****************************************/
void fatal2(const char *mes, const char *add)
  {
    fprintf(stderr,"%s%s\n",mes,add);     /* Prints an error message */
    exit(16);
  }



/**************************************** FUNCTION: show_names ****************************************/
static void show_names(int *flag)
  {
  if( !(*flag) && print_diff )
    printf("Files %s and %s differ:\n",Pfrom,Pto);
  *flag = 1;
  }



/**************************************** FUNCTION: exceptsort ****************************************/
static int exceptsort(const void *p1, const void *p2)             /* Compare name of exceptions */
  {
    const except_t *q1 = (const except_t *)p1;
    const except_t *q2 = (const except_t *)p2;
    return strcmp(q1->name,q2->name);
  }




/**************************************** FUNCTION: do_lchown ****************************************/
static int do_lchown(struct stat *s1, struct stat *s2, except_t *e)
{
  struct stat *s;

#ifdef HAVE_LCHOWN
  changed = 1;
  
  if( e->keepowner )
    s = s2;
  else
    s = s1;
  
  if( print_action )
    printf("%s %s : %d.%d\n",Setlnkown,Pto,s->st_uid,s->st_gid);

  si.lchown++;
  if( !do_action )
    return 0;
  
  if( lchown(Pto,s->st_uid,s->st_gid) == 0 )
    return 0;
  
  fprintf(stderr,"lchown failed\n");
  perror(Pto);
  return -1;

#else
  return 0;  
#endif
}



/**************************************** FUNCTION: do_chown ****************************************/
static int do_chown(struct stat *s1, struct stat *s2, except_t *e)
{
  struct stat *s;

  changed = 1;

  /* If the field "keepowner" is activated. */
  if( e->keepowner )
    s = s2;
  else
    s = s1;

  if( print_action )
     printf("%s %s : %d.%d\n",Setown,Pto,s->st_uid,s->st_gid);

  si.chown++;
  if( !do_action )
    return 0;


  /* Change owner and group of the file */
  if( chown(Pto,s->st_uid,s->st_gid) == 0 )
    return 0;


  fprintf(stderr,"chown failed\n");
  perror(Pto);
  return -1;
}






/**************************************** FUNCTION: do_chmod ****************************************/
static int do_chmod(struct stat *s1, struct stat *s2, except_t *e)
{
  struct stat *s;
  int mode=0;
  /*int ot;*/

  changed = 1;

  /* If old file access bits or "keepacc" of the exception is activated. */
  if( global_keepacc || e->keepacc )      
    s = s2;
  else
    s = s1;
     
  /* Kept only permission bits */
  mode = s->st_mode & ACCESS;


  if( print_action && !global_keepacc)
     printf("%s %s : %o\n",Setmode,Pto,mode);

  si.chmod++;
  if( !do_action )
    return 0;
 
  if( chmod(Pto,mode) == 0 )
    return 0;

  fprintf(stderr,"chmod failed\n");
  perror(Pto);
  return -1;
}








/**************************************** FUNCTION: do_utime ****************************************/
static int do_utime(struct stat *s1)
{
  struct utimbuf tm;

  changed = 1;

  if( print_action )
     printf("%s %s\n",Settime,Pto);

  si.settime++;
  if( !do_action )
    return 0;

  /* Copying the time for the las access time. */
  tm.actime = s1->st_atime;
  /* Time of the last modification in the contents of the file. */
  tm.modtime = s1->st_mtime;

  /* Modifying file time associated with tm. */
  if( utime(Pto,&tm) == 0 )
    return 0;
   
  fprintf(stderr,"utime failed\n");
  perror(Pto);
  return -1;
}



/**************************************** FUNCTION: do_lutime ****************************************/
static int do_lutime(struct stat *s1)
{
  /*struct utimbuf tm;*/

#ifdef HAVE_LUTIME
  
  changed = 1;
  
  if( print_action )
    printf("%s %s\n",Settime,Pto);
  
  si.settime++;
  if( !do_action )
    return 0;

  tm.actime = s1->st_atime;
  tm.modtime = s1->st_mtime;
  
  if( lutime(Pto,&tm) == 0 )
    return 0;
  
  fprintf(stderr,"lutime failed\n");
  perror(Pto);
  return -1;
#else
  return 0;  
#endif
}






/**************************************** FUNCTION: do_unlink ****************************************/
static int do_unlink(kind_t kind)
{
  changed = 1;

  if( print_action )
     printf("%s %s\n",(kind == link_v)?Remlink:Remove,Pto);

  si.unlink++;
  if( !do_action )
    return 0;

  if( unlink(Pto) == 0 )
    return 0;

  fprintf(stderr,"unlink failed\n");
  perror(Pto);
  return -1;
}






/**************************************** FUNCTION: do_rmdir ****************************************/
static int do_rmdir(void)
{
  changed = 1;

  si.rmdir++;
  if( !do_action )
    return 0;

  if( rmdir(Pto) == 0 )
    return 0;

  fprintf(stderr,"rmdir failed\n");
  perror(Pto);
  return -1;
}





/**************************************** FUNCTION: do_mkdir ****************************************/
static int do_mkdir(void)
{
  changed = 1;

  if( print_action )
     printf("%s %s\n",Creadir,Pto);

  si.mkdir++;
  if( !do_action )
    return 0;

  if( mkdir(Pto,0700) == 0 )
    return 0;

  fprintf(stderr,"mkdir failed\n");
  perror(Pto);
  return -1;
}






/**************************************** FUNCTION: do_pipe ****************************************/
static int do_pipe(void)
{
  changed = 1;

  if( print_action )
    printf("%s %s\n",Creapipe,Pto);

  si.mkfifo++;
  if( !do_action )
    return 0;

  if( mkfifo(Pto,0600) == 0 )   /* FIFO special file. */
    return 0;

  fprintf(stderr,"mkfifo failed\n");
  perror(Pto);
  return -1;
}






/**************************************** FUNCTION: do_symlink ****************************************/
static int do_symlink(struct stat *s1, struct stat *s2, except_t *e, const char *link)
{
  struct stat *s;
  int rc;
  int oldmask;

  changed = 1;

  if( global_keepacc || e->keepacc )
    s = s2;
  else
    s = s1;

  if( print_action )
    printf("%s %s as %s\n",Setlink,Pto,link);

  si.symlink++;
  if( !do_action )
    return 0;

  oldmask = umask(~(s->st_mode & ACCESS));

  rc =  symlink(link,Pto);
  umask(oldmask);

  if( rc == 0 )
    return 0;

  fprintf(stderr,"symlink failed\n");
  perror(Pto);
  return -1;
}





static char iobuf[16384];




/**************************************** FUNCTION: do_copy ****************************************/
static int do_copy(layer_t *t, kind_t kind)
{
  int id = -1 , od = -1;
  int rc;
  int removed = 0;
  int ret;
  
  changed = 1;
  
  if( kind == reg_v )
    si.replfile++;

  else
    si.crefile++;

  
  if( !do_action )
    {
      if( print_action )
	{
          int rc1;
          if( kind == reg_v )
            print_action = 0;

          rc1 = try_remove(t,0,kind); /* just to print the message */
          if( kind == reg_v )
            print_action = 1;
          if( rc1 <= 0 )
            return rc1;

          printf("%s %s\n",(kind == reg_v)?Replfile:Creafile,Pto);
	} 
      return 0;
    }
  
  id = open(Pfrom,O_RDONLY);
  if( id < 0 )
    {
      perror(Pfrom);
      ret=-9;
      goto esc;
    }
  for(;;)
    {
      int w;
      rc = read(id,iobuf,sizeof(iobuf));
      if( rc < 0 )
	{
          perror(Pfrom);
	  ret=-9;
          goto esc;
	}
      addlong(&si.lread_kb,&si.lread_rest,rc);

      if( od < 0 )
	{
          int rc1;
          int pra = print_action;
          if( kind == reg_v )
            print_action = 0;

          rc1 = try_remove(t,0,kind);
          if( kind == reg_v )
            print_action = pra;
          if( rc1 <= 0 )
            {
	      close(id);
	      return rc1;
            }
          removed = 1;
          if( print_action )
            printf("%s %s\n",(kind == reg_v)?Replfile:Creafile,Pto);


	  od = open(Pto,O_WRONLY|O_CREAT|O_EXCL,0600);
          if( od < 0 )
            {
	      perror(Pto); 
	      ret=-10;
	      goto esc;
            }
	  
	}
      if( rc == 0 )
	break;

      w = write(od,iobuf,rc);
      if( w >= 0 )
	addlong(&si.write_kb,&si.write_rest,w);

      if( w != rc )
	{
          perror(Pto);
	  ret=-10;
          goto esc;
	}
    }
  close(id);
  close(od);
  return 0;

 esc:
  if( id >= 0 ) 
    close(id);
  if( od >= 0 ) 
    {
      close(od);
      unlink(Pto);
    }
  else if( !removed )
    no_actions();

  /*return -1;*/
  return ret;
}





/**************************************** FUNCTION: myalphasort ****************************************/
static int myalphasort(const void *a, const void *b)
{
  return strcmp(*(const char **)a,*(const char **)b);
}



/**************************************** FUNCTION: freedir ****************************************/
static void freedir(char *p[], int n)
{
  if( p )
    {
      int i;
      for(i=0; i<n; i++)
	free(p[i]);
      free(p);
    }
}



/**************************************** FUNCTION: my_scandir ****************************************/
static int my_scandir(const char *path, char **resp[])
{
  static int STEP = 50;
  int nalloc = 0;
  int n = 0;
  struct dirent *entp;
  char **res = NULL;
  DIR *dirp = opendir(path);

  *resp = NULL;
  if( dirp == NULL )
    return -1;

  while( (entp = readdir(dirp)) != NULL )         /* Every entry from the directory is read until there are no more. */
    {
      int L = strlen(entp->d_name);
      /*
	printf("====L %d, path >>%s<<, d_name %s\n",L,path,entp->d_name);
	continue;
      */
      /* skip . and .. */
      if( L <= 2 && entp->d_name[0] == '.' )
	if( entp->d_name[1] == '\0' ||
	    entp->d_name[1] == '.' )
	  continue;
      if( n >= nalloc )
	{
	  nalloc += STEP;
	  if( n == 0 ) 
	    res = (char **)malloc(nalloc*sizeof(char *));
	  else
	    res = (char **)realloc(res,nalloc*sizeof(char *));
	  if( res == NULL )
	    goto esc;   
	}
      res[n] = (char *)malloc(L+1);
      if( res[n] == NULL )
	goto esc;
      memcpy(res[n],entp->d_name,L+1);
      n++;
    }
  closedir(dirp);
  if( n != nalloc )
    {
      res = realloc(res,n*sizeof(char *));
      if( res == NULL )
	goto esc;
    }
  if( n )
    qsort(res,n,sizeof(char *),myalphasort); /* The buffer of files is sorted followinf "myalphasort". */
  *resp = res;
  return n;
 esc:
  fatal2("Out of memory scanning ",path);
  return -1;
}




/**************************************** FUNCTION: getexceptions ****************************************/
static void getexceptions(char *fn)
{
  FILE *f;
  char buffer[PATH_MAX+1];
  int i;

  f = fopen(fn,"r");
  if( !f )
    fatal2("Cannot open exceptions file ",fn);

  while( fgets(buffer,sizeof(buffer),f) )       /* New line of the file. */
    {
      char *p = strtok(buffer,"\n\t ");
      if( p )
	{
	  if( *p == '#' )                       /* Comentary. */
	    continue;
	  if( nexcept == MAXEXCEPT )                         /* Control of errors */
	    fatal("Too many exceptions in exception file");
	  e[nexcept].nlen = strlen(p);
	  if( e[nexcept].nlen >= sizeof(Pto) )
	    fatal2("Exception name too long:",p);
	  e[nexcept].name = strdup(p);
	  e[nexcept].fullstop = 1;
	  if( !e[nexcept].name )
	    fatal("Out of memory");

	  /* Next option is anylized. The structure is correctly filled. */
	  p = strtok(NULL,"\n\t, ");      
	  if( p )
	    {
	      e[nexcept].fullstop = 0;
	      if( strcmp(p,"dir") == 0 )
		e[nexcept].keepdir = 1;
	      else if( strcmp(p,"owner") == 0 )
		e[nexcept].keepowner = 1;
	      else if( strcmp(p,"access") == 0 )
		e[nexcept].keepacc = 1;
	      else
		fatal2("Wrong option in exception file: ",p);
	    }
	  nexcept++;
	}
    }
  fclose(f);
  /* The buffer os exceptions is sorted. */
  qsort(e,nexcept,sizeof(e[0]),exceptsort);
  for(i=1;i<nexcept;i++)
    if( strcmp(e[i-1].name,e[i].name) == 0 ) /* Check of duplicated. */
      fatal2("Duplicated name in exception file: ",e[i].name);
  printf("\nExceptions:\n");
  for(i=0;i<nexcept;i++)
    printf(" %s %s %s %s\n",e[i].name,
	   e[i].keepacc?"access":"",
	   e[i].keepdir?"dir":"",
	   e[i].keepowner?"owner":"");
  printf("Exceptions end\n");
}




/**************************************** FUNCTION: try_remove ****************************************/
static int try_remove(layer_t *t, int nosource, kind_t kind)
    {
      int retorno;

      if( kind == none_v )
	;
      else if( kind == dir_v )
	{
	  char **d;
	  int n;
	  int i;
	  int fail;
	  
	  /* Field "keepdir" of the exception is activated. */
	  if( t->exc->keepdir )
	    {
	      printf("%s %s\n",Skipchange,Pto);
	      return 0;
	    } 
	  /* Indicates that it will remove. */
	  if( !t->nodelete && print_action )
	    printf("%s %s\n",Remdir,Pto); 

	  if( !do_action )
	    return 1;
	  fail = 0;
	  d = NULL;
	  n = my_scandir(Pto,&d);

	  for(i=0; i<n; i++)                    /* For every entry. */
	    {
	      layer_t new;
	      if( new_level(&new,t,d[i]) != 0 ) /* We create a new level. */
		continue;

	      if( !t->nodelete )                
		new.print_action = print_action = 0;

	      retorno=cmpr(&new,1,0);
	      if(retorno  != 0 )
		fail = 1;

	      reset_level(t);
	      if( fail )
		break;
	    }
	  freedir(d,n); 

	  if( fail )	    
	      /*return -1;*/
	      return retorno;
	    

	  if( t->nodelete )
	    {
	      printf("%s %s\n",nosource?Skipdel:Skiprepl,Pto);
	      return 0;
	    }
	  if( do_rmdir() )         /* The directory is erased. */
	    /*	    return -1;*/
	    return -3;
	}
      else if( do_unlink(kind) )
	/*	return -1;*/
	return -4;
      
      return 1;
    }





/**************************************** FUNCTION: cmpr ****************************************/
static int cmpr(layer_t *t, int nosource, int notarget)
{
  int rc, valRet,Vret;
  struct stat s1;
  struct stat s2;
  kind_t kind_from,kind_to;



  counter++;  

  if( initial_print_action && counter%100 == 0 )
    {
      printf("Current %7d ---------> %s\n",counter,Pto);
      fflush(stdout);
    }
  
  if( t->exc->fullstop )
    {
      printf("%s %s\n",Skipcheck,Pto);
      return 0;
    }
  
  if( nosource )
    ;

  else if( lstat(Pfrom,&s1) )   /* The attributes of the source path are kept. */
    {
      perror(Pfrom);
      no_actions();
      return -1;
    }

  if( nosource )               
    kind_from = none_v;

  /* Source File is a Regular File. */
  else if( IS_REG(s1) )
    {
      int Lc = 5; /* length of "/core" */
      if( t->Lfrom >= Lc && strcmp(Pfrom+t->Lfrom-Lc,"/core") == 0 )
	{
	  printf("File %s ignored\n",Pfrom);
	  if( notarget )
	    return 0;
	  nosource = 1;
	  kind_from = none_v;
	}
      else
	kind_from = reg_v;
    }
  /* Source File is a DIRECTORY. */
  else if( IS_DIR(s1) )
    kind_from = dir_v; 
  /* Source File is a SYMBOLIC LINK. */
  else if( IS_LINK(s1) )
    kind_from = link_v;
  /* Source File is a PIFE FILE or FIFO FILE. */
  else if( IS_PIPE(s1) )
    kind_from = pipe_v;
  /* Source File is a SPECIAL FILE */
  else
    {
      fprintf(stderr,"Special file %s\n",Pfrom);
      no_actions();
      /*return -1;*/
      return -2;
    }
  
  if( notarget )         /* If there is no target, se copia s1 en s2. */
    memcpy(&s2,&s1,sizeof(s1));
  
  else if( lstat(Pto,&s2) ) /* If there is, the attributes are kept. */
    {
      perror(Pto);
      no_actions();
      return -1;
    }

  if( notarget )
    kind_to = none_v;

  /* Target File is a REGULAR FILE.*/
  else if( IS_REG(s2) )
    kind_to = reg_v;
  /* Target File is a DIRECTORY. */
  else if( IS_DIR(s2) )
    kind_to = dir_v; 
  /* Target File is a SYMBOLIC FILE. */
  else if( IS_LINK(s2) )
    kind_to = link_v;
  /* Target File is a PILE FILE. */
  else if( IS_PIPE(s2) )
    kind_to = pipe_v;
  /* Target File is a SPECIAL FILE. */
  else
    {
      fprintf(stderr,"Special file %s\n",Pto);
      no_actions();
      /*      return -1;*/
      return -2;
    }
  /*
    printf("cmpr %s %s %s %s nodel %d fullst %d kdir %d kown %d kacc %d\n",
    Pfrom,ckind[kind_from],Pto,ckind[kind_to],t->nodelete,t->exc->fullstop,t->exc->keepdir,t->exc->keepowner,t->exc->keepacc);
    printf("%s %s\n",ckind[kind_from],ckind[kind_to]);
  */


  switch( kind_from )
    {

    case none_v:
      rc = try_remove(t,nosource,kind_to);
      if( rc > 0 )
        rc = 0;
      return rc;


    case pipe_v:
      if( kind_to != kind_from )   /* Not the same kind */
        {
	  rc = try_remove(t,nosource,kind_to);
	  if( rc <= 0 )
	    return rc;

	  kind_to = none_v;
        }

      if( kind_to == none_v )
        {
	  if( do_pipe() )
	    /*return -1;*/
	    return -5;
	  print_action = 0;
        }

      rc = 0;
      if( kind_to == none_v && !no_timecheck )
        rc |= do_utime(&s1);                    /* Sets the time. */
      if( kind_to == none_v || (s1.st_mode & ACCESS) != (s2.st_mode & ACCESS) )
        rc |= do_chmod(&s1,&s2,t->exc);         /* Sets the permission bits. */
      if( kind_to == none_v || s1.st_uid != s2.st_uid || s1.st_gid != s2.st_gid )
        if( !no_chown && do_chown(&s1,&s2,t->exc) )    /* Sets the user id and the group id.*/
	  rc = -1;

      if( rc )
        /*return -1;*/
	return -6;
      break;



    case link_v:
      {
	char buf1[PATH_MAX+1];
	int rc1=readlink(Pfrom,buf1,sizeof(buf1));    /* Keep the value of the symbolic link. */

	if( rc1 < 0 )
	  {
	    fprintf(stderr,"Failed to read the link %s\n",Pfrom);
	    no_actions();
	    /*return -1;*/
	    return -7;
	  }

	buf1[rc1] = 0;
	if( kind_to == kind_from )   /* Same kind of file. */
	  {
	    char buf2[PATH_MAX+1];
	    int rc2=readlink(Pto,buf2,sizeof(buf2)); 
	    if( rc1 != rc2 || memcmp(buf1,buf2,rc1) != 0 )
	      {
		if( do_unlink(kind_to) )
		  /*return -1;*/
		  return -4;
		kind_to = none_v;
	      }
	  }
	else
	  {
	    rc = try_remove(t,nosource,kind_to);
	    if( rc <= 0 )
	      return rc;
	    kind_to = none_v;
	  }


	if( kind_to == none_v )
	  {
	    if( do_symlink(&s1,&s2,t->exc,buf1) )
	      /*return -1;*/
	      return -8;
	    print_action = 0;
	  }

	if( kind_to == none_v || s1.st_uid != s2.st_uid || s1.st_gid != s2.st_gid )
	  if( !no_chown && do_lchown(&s1,&s2,t->exc) )      /* Sets the user id and the group id.*/
	    /*return -1;*/
	    return -6;

	if( kind_to == none_v && !no_timecheck )
	  do_lutime(&s1);       /* Sets the time. */

      }
      break;



    case reg_v:
      {
	int rc = 0;
	int replace;

	if( kind_to == kind_from )     /* Same kind of file. */
	  { 
	    if( s1.st_size != s2.st_size )
	      {
		show_names(&rc); 
		if( print_diff )
		  printf("   size %ld %ld\n",(long int)s1.st_size,(long int)s2.st_size); 
	      }

	    if( !no_timecheck && s1.st_mtime != s2.st_mtime )   /* If there is time checking. */
	      {
		char *q;
		show_names(&rc);
		if( print_diff )
		  {
		    printf("   time ");
		    for(q=ctime(&s1.st_mtime); *q && *q != '\n'; q++)
		      putchar(*q);
		    printf("   %c   ",(s1.st_mtime<s2.st_mtime)?'<':'>'); 
		    printf("%s",ctime(&s2.st_mtime));
		  }
	      }

	    if( rc == 0 && check_head && s1.st_size > 0 )    /* If a number of bytes must be compared. */
	      {
		char buf2[sizeof(iobuf)];
		int f1,f2;
		long len = s1.st_size;
		if( check_head > 0 && len > check_head ) len = check_head;

		f1 = open(Pfrom,O_RDONLY);
		if( f1 < 0 )
		  {
		    perror(Pfrom);
		    no_actions();
		    /*return -1; */
		    return -9;
		  }
		f2 = open(Pto,O_RDONLY);
		if( f2 < 0 )
		  {
		    perror(Pto);
		    len = 0;
		  }

		while( len > 0 )   /* There are bytes to read... */
		  {
		    int L = sizeof(iobuf);
		    int w;
		    if( L > len ) L = len;

		    w = read(f1,iobuf,L);
		    if( w >= 0 )
		      addlong(&si.lread_kb,&si.lread_rest,w);

		    if( w != L )
		      {
			perror(Pfrom);
			no_actions();
			close(f1);
			close(f2);
			/*return -1;*/
			return -9;
		      }

		    w = read(f2,buf2,L);
		    if( w >= 0 )
		      addlong(&si.rread_kb,&si.rread_rest,w);

		    if( w != L )
		      {
			perror(Pto);
			len = 0;
			break;
		      }
		    else if( memcmp(iobuf,buf2,L) != 0 )
		      {
			len = 0;
			break;
		      }
		    else
		      len -= L;
		  }
		close(f1);
		close(f2);
		if( len )
		  {
		    show_names(&rc);
		    if( print_diff )
		      printf("   different contents\n"); 
		  }
	      }
	    replace = rc;
	    if( (s1.st_mode & ACCESS) != (s2.st_mode & ACCESS) )    /* Different permission bits. */
	      {
		show_names(&rc);
		if( print_diff )
		  printf("   access %o %o\n",s1.st_mode & ACCESS ,s2.st_mode & ACCESS); 
	      }
	    if( !no_chown && s1.st_uid != s2.st_uid )               /* Owner Id is not the same. */
	      {
		show_names(&rc);
		if( print_diff )
		  printf("   uid %d %d\n",s1.st_uid,s2.st_uid); 
	      }
	    if( !no_chown && s1.st_gid != s2.st_gid )              /* Group Id is not the same. */
	      {
		show_names(&rc);
		if( print_diff )
		  printf("   gid %d %d\n",s1.st_gid,s2.st_gid); 
	      }
	    if( rc == 0 )
	      return 0;
	  }

	else
	  replace = 1;

	if( replace )
	  {
	    valRet=do_copy(t,kind_to);
	    if(valRet)	      
		/*return -1;*/
		return valRet;	      
	    print_action = 0;
	    kind_to = none_v;
	  }

	rc = 0;
	if( kind_to == none_v || s1.st_uid != s2.st_uid || s1.st_gid != s2.st_gid )
	  if( !no_chown && do_chown(&s1,&s2,t->exc) )/* Sets the user id and the group id.*/
	    rc = -1;

	if( kind_to == none_v || (s1.st_mode & ACCESS) != (s2.st_mode & ACCESS) )
	  rc |= do_chmod(&s1,&s2,t->exc);       /* Sets the permission bits. */
	
	if( kind_to == none_v && !no_timecheck )
	  rc |= do_utime(&s1);                  /* Sets the time. */

	if( rc )
	  /*return -1;*/
	  return -6;
      }
      break;



    case dir_v:
      {
	int rc = 0;
	char **d1 = NULL, **d2 = NULL;
	int n1 = 0 , n2 = 0;
	int i1,i2;
	int ret2=0;

	
	n1 = my_scandir(Pfrom,&d1);      /* All the directories are sorted. */
	if( n1 < 0 )
	  {
	    perror(Pfrom);
	    no_actions();
	    /*return -1;*/
	    return -11;
	  }

	if( kind_to != kind_from && kind_to != none_v )   /* Different kinds. */
	  {
	    if( do_unlink(kind_to) )
	      {
		freedir(d1,n1); 
		/*return -1;*/
		return -4;
	      }
	    kind_to = none_v;
	  }

	if( kind_to == none_v )
	  { 
	    if( do_mkdir() )
	      {
		freedir(d1,n1); 
		/*return -1;*/
		return -12;
	      }
	    t->print_action = print_action = 0;
	    n2 = 0;
	  }

	else
	  {
	    n2 = my_scandir(Pto,&d2);
	    if( n2 < 0 )
	      {
		perror(Pto);
		freedir(d1,n1);
		/*return -1;*/
		return -11;
	      }
	  }
	for(i1=0,i2=0;i1<n1 || i2<n2; )
	  {
	    int order;
	    const char *q;
	    int nosrc,notrgt;
	    layer_t new;
	   

	    if( i1 >= n1 )
	      order = 1;
	    else if( i2 >= n2 )
          order = -1;
	    else
	      {
		const char *p1 = d1[i1], *p2 = d2[i2];
		order = myalphasort(&p1,&p2);
	      }
	    if( order < 0 )
	      {
		nosrc = 0;
		notrgt = 1;
		q = d1[i1];
		i1++;
	      }
	    else if( order > 0 )         
	      {
		nosrc = 1;
		notrgt = 0;
		q = d2[i2];
		i2++;
	      }
	    else
	      {
		nosrc = 0;
		notrgt = 0;
		q = d2[i2];
		i1++;
		i2++;
	      }
	    if( order && t->exc->keepdir )
	      {
		printf("%s %s/%s\n",Skipcheck,Pto,q);
		continue;
	      }

	    if( new_level(&new,t,q) != 0 )
	      continue;
	    if( (Vret=cmpr(&new,nosrc,notrgt)) != 0 )
	      {		
		ret2=Vret;
		rc = -1;
	      }
	    reset_level(t);	    	   	    	
	  }

	freedir(d1,n1); 
	freedir(d2,n2);



	/*if(rc)
	  {	  
	     return Vret;
	  }*/

	/*
	if( kind_to == none_v && !no_timecheck )
	  {
	    printf("*********************** tiempo para directorio. ******************\n");
	    rc |= do_utime(&s1);
	  }
	*/
	/*	rc=0;*/

	if( notarget )
	  memcpy(&s2,&s1,sizeof(s1));
	else if( lstat(Pto,&s2) )
	  {
	    perror(Pto);
	    no_actions();
	    return -1;
	  }

	if( kind_to == none_v || (s1.st_mode & ACCESS) != (s2.st_mode & ACCESS) )
	  rc |= do_chmod(&s1,&s2,t->exc);

	if( kind_to == none_v || s1.st_uid != s2.st_uid || s1.st_gid != s2.st_gid )
	  if( !no_chown && do_chown(&s1,&s2,t->exc) )
	    rc = -1;

	if(!no_timecheck)
	  if(kind_to==none_v || s1.st_mtime != s2.st_mtime) 
	    rc |= do_utime(&s1);
	
	if( rc )
	  {
	    /* 2/8/2005 - c.r.g - Change to control the code error.*/
	    if(ret2)
	      return ret2;
	    /*return -1;*/	    
	    return -6;
	  }
      }
      break;
    }
  
  return 0;
}



/**************************************** FUNCTION: help ****************************************/
void help(char *name)
{
  fprintf(stderr,"  HELP \n\
Format: %s -hVvqisod -t count -e exceptfile from_dir1 to_dir1 from_dir2 to_dir2 ... \n\
   -h - produce this help\n\
   -V - print the program version\n\
   -v - verbose mode, print all the differences\n\
   -q - quiet mode, don't print the actions\n\
        by default actions are printed and differences - not\n\
   -i - inspect mode, suppress actions execution (but not their printing)\n\
   -s - compare the complete files (option -t ignored in this case)\n\
   -o - no attempts to check/change/keep file ownership (no chown's)\n\
   -a - always keep old file access bits for target files\n\
   -l - limit access bit check/setting to rwx bits (no 's','S','t' etc.)\n\
   -d - no file change date/time check\n\
   -t - number of bytes to compare in files\n\
   -e - exception file name,\n\
         use -e none  to run without an exception file\n\
         use -e<exceptionFile> to run with exception file\n\
        The exception file contains the path in the target indicating the files/directories \n\
         that won't be copied.\n\
Example:\n\
   mirdir -e none ../blip /afs/cern.ch/blop\n\
\n\
   Bugs/comments to mirdir@caspur.it \n\
",name);
  exit(8);
} 



/**************************************** MAIN ****************************************/
int main(int argc, char *argv[])
  {
    int rctot = 0;
    char init_wd[PATH_MAX+1];
    char *exceptfile;
    mode_t oldmask;
    extern int optind;
    extern char *optarg;
    int c;
    layer_t zero_level;
    int save_print_action;
    int show_version;

    /*int contador=0;*/
    
    do_action = 1;
    print_action = 1;
    print_diff = 0;
    check_head = 0;
    no_chown = 0;
    global_keepacc = 0;
    ACCESS = 07777;                    /* r,w,x,s for everybody: user, group, other */
    no_timecheck = 0;
    show_version = 0;
    exceptfile = NULL;


    while( (c=getopt(argc,argv,"hvqit:soaldVe:")) != EOF )  /* Decode the argument list */
      switch(c)
	{
	case 'V':
	  show_version = 1;
	  break;
	  
	case 'v':
	  print_diff = 1;
	  break;
	  
	case 'q':
	  print_action = 0;
	  break;

	case 'i':
	  do_action = 0;
	  break;

	case 't':
	  if( strspn(optarg,"0123456789") != strlen(optarg) )   /* We check that the argument, that is kept in the variable "optarg" */
	    help(argv[0]);                                      /* is composed only by numbers, that indicate the bytes to compare in */
								/* files                       */
	  if( check_head >= 0 )
	    check_head = atol(optarg);                          /* We convert to long the number of bytes to compare. */
	  break;

	case 's':
	  check_head = -1;
	  break;

	case 'o':
	  no_chown = 1;
	  break;

	case 'a':
	  global_keepacc = 1;
	  break;

	case 'l':
	  ACCESS = 0777;    /* r,w,x for everybody. We take out the set_uid bit that it was set as default. */
	  break;

	case 'd':
	  no_timecheck = 1;
	  break; 

	case 'e':
	  exceptfile = optarg;
	  break;

	case 'h':

	default:
	  help(argv[0]);
	  
	} 
    
    
    initial_print_action = print_action;
    
    if( show_version )
      {
	printf("$Id: %s 2004/11/12 15:00:00 ruten Exp $\n", PACKAGE_STRING);
	if( optind == argc )
	  exit(0);
      }
    
    save_print_action = print_action;
    
    if( optind+2 > argc || !exceptfile )    /* The number entry arguments are ok. */
      help(argv[0]);
    
    if( (argc - optind) % 2 != 0 )
      {
	fprintf(stderr,"No target directory for source directory %s\n",argv[argc-1]);
	help(argv[0]);
      }
    
    if( !getcwd(init_wd,sizeof(init_wd)) )     /* Current directory is kept. */
      {
	perror("getcwd");
	exit(1);
      }
    
    if( strcmp(exceptfile,"none") != 0 )
      getexceptions(exceptfile);

    oldmask = umask(0);   

    for(; optind < argc; optind += 2 )
      {
	int rc;
	time_t t1,t2;
	
	time(&t1); 

	if( chdir(argv[optind]) != 0 || !getcwd(Pfrom,sizeof(Pfrom)) )
	  {
	    perror(argv[optind]);
	    exit(1);
	  }
	if( chdir(init_wd) != 0 )
	  {
	    perror(init_wd);
	    exit(1);
	  }
	if( chdir(argv[optind+1]) != 0 || !getcwd(Pto,sizeof(Pto)) )
	  {
	    perror(argv[optind+1]);
	    exit(1);
	  }
	if( chdir(init_wd) != 0 )
	  {
	    perror(init_wd);
	    exit(1);
	  }
	
	zero_level.Lfrom = strlen(Pfrom);
	zero_level.Lto  = strlen(Pto);
	zero_level.print_action = save_print_action;
	currexcept = 0;
	set_except(&zero_level,0);
	
	printf("\n\nMirroring %s to %s%s\n",Pfrom,Pto,do_action?"":", inspect mode"); 
	
	fflush(stdout);
	
	changed = 0;
	rc = cmpr(&zero_level,0,0);

	if( rc );
	/*rc = 16;*/
	if( rc == 0 && changed )
	  rc = 1; 
	if( rctot < rc )
	  rctot = rc;
	time(&t2);
	t2 -= t1;

	printf("Elapsed time %lu:%02ld:%02ld, rc %d\n",t2/3600,(t2%3600)/60,t2%60,rc);
	{
	  long w;
	  w = counter;
	  if( w )
	    printf("Statistics: files/directories compared: %ld\n",w);
	  /*
	    w = si.chown + si.lchown + si.chmod + si.settime;
	    if( w )
	    printf("Statistics: change file attributes (chown etc.): %ld\n",w);
	  */
	  w = si.unlink + si.rmdir;
	  if( w )
	    printf("Statistics: files/directories removed: %ld\n",w);
	  
	  w = si.mkdir + si.mkfifo + si.symlink + si.crefile;
	  if( w )
	    printf("Statistics: files/directories created: %ld\n",w);

	  w = si.replfile;
	  if( w )
	    printf("Statistics: files/directories replaced: %ld\n",w);
	  if( si.lread_kb || si.lread_rest )
	    printf("Statistics: source read: %ld Kbytes\n",si.lread_kb);
	  if( si.rread_kb || si.rread_rest )
	    printf("Statistics: target read: %ld Kbytes\n",si.rread_kb);
	  if( si.write_kb || si.write_rest )
	    printf("Statistics: written: %ld Kbytes\n",si.write_kb);
	}
	memset(&si,0,sizeof(si));
      }
    umask(oldmask);
    exit(rctot);
  } 
