/*
 * Copyright 2021-2024 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.opentest4j.reporting.tooling.spi.htmlreport;

import org.apiguardian.api.API;
import org.w3c.dom.Element;

import java.nio.file.Path;
import java.util.List;

import static java.util.Collections.emptyList;
import static org.apiguardian.api.API.Status.EXPERIMENTAL;

/**
 * Implementations of this interface can contribute additional sections to the
 * HTML report when registered via the Java service loader mechanism.
 *
 * @since 0.2.0
 */
@API(status = EXPERIMENTAL, since = "0.2.0")
public interface Contributor {

	/**
	 * Contribute additional sections for the given execution element.
	 *
	 * @param context the DOM element representing the execution
	 * @return additional sections to be rendered
	 */
	default List<Section> contributeSectionsForExecution(Context context) {
		return emptyList();
	}

	/**
	 * Contribute additional sections for the given test node element.
	 *
	 * @param context the DOM element representing the test node
	 * @return additional sections to be rendered
	 */
	default List<Section> contributeSectionsForTestNode(Context context) {
		return emptyList();
	}

	/**
	 * The context to contribute sections for.
	 */
	interface Context {

		/**
		 * {@return the DOM element representing the execution or test node}
		 */
		Element element();

		/**
		 * {@return the path to the source XML file}
		 */
		Path sourceXmlFile();

		/**
		 * {@return the path to the target HTML file}
		 */
		Path targetHtmlFile();

	}

}
