// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/UnstableParticles.hh"
#include "Rivet/Projections/Beam.hh"

namespace Rivet {


  /// @brief pi0 spectrum at 14 and 34 GeV
  class TASSO_1982_I168232 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(TASSO_1982_I168232);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {


      declare(Beam(), "Beams");
      declare(UnstableParticles(), "UFS");

      // Book histograms
      size_t ih = 2;
      for (double eVal : allowedEnergies()) {
        const string en = toString(round(eVal));
        if (isCompatibleWithSqrtS(eVal))  _sqs = en;
        book(_h[en+"E"], ih, 1, 1);
        book(_h[en+"p"], ih, 2, 2);
        book(_h[en+"x"], ih, 3, 3);
        ++ih;
      }
      raiseBeamErrorIf(_sqs.empty());
    }


    /// Perform the per-event analysis
    void analyze(const Event& event) {
      // Get beams and average beam momentum
      const ParticlePair& beams = apply<Beam>(event, "Beams").beams();
      const double meanBeamMom = 0.5*(beams.first.p3().mod() + beams.second.p3().mod());
      MSG_DEBUG("Avg beam momentum = " << meanBeamMom);

      for (const Particle& p : apply<UnstableParticles>(event, "UFS").particles(Cuts::pid==PID::PI0)) {
        if (!p.parents().empty() && p.parents()[0].pid()==PID::K0S) continue;
        double xE = p.E()/meanBeamMom;
        double beta = p.p3().mod()/p.E();
        _h[_sqs+"E"]->fill(p.E()/GeV);
        _h[_sqs+"p"]->fill(p.p3().mod()/GeV);
        _h[_sqs+"x"]->fill(xE, 1.0/beta);
      }
    }


    /// Normalise histograms etc., after the run
    void finalize() {

      for (auto& item : _h) {
        if (item.first.find("x") != string::npos) {
          const double en = stod(item.first.substr(0,2));
          scale(item.second, sqr(en)*crossSection()/microbarn/sumOfWeights());
        }
        else {
          scale(item.second, crossSection()/nanobarn/sumOfWeights());
        }
      }
    }

    /// @}


    /// @name Histograms
    /// @{
    map<string,Histo1DPtr> _h;
    string _sqs = "";
    /// @}

  };

  RIVET_DECLARE_PLUGIN(TASSO_1982_I168232);
}
