/*********************************************************************
 *
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * ===================================================================
 * Revision History ::
 * YYYY.MM.DD   Change ID       Developer
 *              Description
 * -------------------------------------------------------------------
 * 2002.04.25                   Vlad Skarzhevskyy
 *              Initial implementation.
 *
 * ===================================================================
 *
 ********************************************************************/

#include "pure-sfv.h"

int sfvTestFile(pure_sfv_params* params, char* sfv_file, char* sfv_dir)
{
    int                 rc = 0;
    crc_info_type       sfv_info;
    crc_info_type       dir_info;
    crc_info_item_type *dir_item;
    int                 i, file_idx;
    unsigned long       crc_val;
    char*               file_name;
    char*               sfv_file_name;
    string              sfv_file_tmp;
    int                 list_show = 0;

    sfvInfoInit(&sfv_info);
    sfvInfoInit(&dir_info);

    if (sfv_dir[0] != '\0') {
        if (chdir(sfv_dir) != 0) {
            fprintf(stderr, "sfv: %s: %s\n", sfv_dir, strerror(errno));
            return ERROR_DIR_ACCESS;
        }
    }
    rc = dirRead("", &dir_info);
    if (rc) return rc;

    sfv_file_name = sfv_file;
    file_idx = dirFindFile(sfv_file_name, &dir_info);
    if ((!file_idx) && (params->ignore_case)) {
        makeCleanName(sfv_file_name, sfv_file_tmp);
        file_idx = dirFindFileLike(sfv_file_tmp, &dir_info);
        if (file_idx) {
           sfv_file_name = dir_info.items[file_idx - 1].file_name;
        }
    }
    if (!file_idx) {
        fprintf(stderr, "Error: sfv file [%s] not found!\n", sfv_file);
        rc = ERROR_FILE_ACCESS;
    } else {
        if (!params->quiet) {
            printf("----( Verifying: %s )--------\n", sfv_file_name);
        }
        rc = sfvReadFile(params, sfv_file_name, &sfv_info);
    }

    if (!rc)
    {
        for (i = 0; i < sfv_info.len; i++)
        {
            file_name = sfv_info.items[i].file_name;

            if (params->ignore_dir_name) {
                file_name = getBaseNameWinUnix(file_name);
            }

            file_idx = dirFindFile(file_name, &dir_info);
            if ((!file_idx) && (params->ignore_case)) {
                file_idx = dirFindFileLike(sfv_info.items[i].file_name_clean, &dir_info);
                if (file_idx) {
                    file_name = dir_info.items[file_idx - 1].file_name;
                }
            }

            if (!params->quiet) {
                printf("%-49s ", file_name);
            }
            if (!includeFile(file_name, params)) {
                /* Exclude files not in Wildcard */
                if (!params->quiet) { printf("Not tested\n"); }
                sfv_info.cnt.files_ignored ++;
                sfv_info.items[i].ignored = 1;
                continue;
            }
            sfv_info.cnt.files_tested++;

            if (!file_idx) {
                sfv_info.cnt.files_not_found ++;
                if (!params->quiet) { printf("No such file\n"); }
                continue;
            }
            dir_item = &(dir_info.items[file_idx - 1]);
            rc = fileReadAttributes(dir_item);
            if (rc) {
                sfv_info.cnt.files_error ++;
                return rc;
            }
            if (dir_item->is_dir) {
                sfv_info.cnt.files_broken ++;
                if (!params->quiet) { printf("Is a directory\n"); }
                continue;
            }

            sfv_info.items[i].found = 1;
            sfv_info.cnt.files_found ++;

            if (params->mode == MODE_COUNT) {
                if (!params->quiet) { printf("Found\n"); }
                continue;
            }

            rc = getFileCRC(file_name, params, &crc_val);
            if (!rc) {
                sfv_info.items[i].new_crc = crc_val;
                sfv_info.items[i].tested = 1;
                if (crc_val != sfv_info.items[i].crc) {
                    sfv_info.cnt.files_broken ++;
                    if (!params->quiet) { printf("different CRC\n"); }
                } else {
                    sfv_info.items[i].tested = 2;
                    sfv_info.cnt.files_ok ++;
                    if (!params->quiet) { printf("OK\n"); }
                }
            } else {
                if (!params->quiet) { printf("Error reading\n"); }
                sfv_info.cnt.files_error ++;
                sfv_info.cnt.files_broken ++;
            }
        }
        if (!params->quiet) {
            printf("--------------------------------------------------------------------------------\n");
        }
    }

    if ((!params->ignore_nofiles_sfv) || (sfv_info.cnt.files_found != 0)) {

        if (sfv_info.cnt.files_broken || sfv_info.cnt.files_not_found || rc ) {
            if (!params->quiet) { printf("Errors Occured.\n"); }
        } else {
            if (!params->quiet) { printf("Everything OK.\n"); }
        }

        if (!params->quiet) {
        if (sfv_info.cnt.files_tested != sfv_info.len) {
            printf("Tested %i of %i files, ", sfv_info.cnt.files_tested, sfv_info.len);
        } else {
            printf("Tested %i files, ", sfv_info.len);
        }
        printf("Successful %i, ", sfv_info.cnt.files_ok);
        printf("Different %i, ", sfv_info.cnt.files_broken);
        printf("Missing %i\n", sfv_info.cnt.files_not_found);
        }

        sfv_info.cnt.sfv_tested ++;
        addCnt(&(params->cnt), &(sfv_info.cnt));

        if (params->list_files != 0)
        {
            for (i = 0; i < sfv_info.len; i++)
            {
                file_name = sfv_info.items[i].file_name;
                list_show = 0;
                switch (params->list_files) {
                    case 'g':
                        if (sfv_info.items[i].tested == 2) {
                            list_show = 1;
                        }
                        break;
                    case 'b':
                        if (sfv_info.items[i].tested == 1) {
                            list_show = 1;
                        }
                        break;
                    case 'm':
                        if ((sfv_info.items[i].ignored == 0) &&
                            (!sfv_info.items[i].found)){
                            list_show = 1;
                        }
                        break;
                    case 'i':
                        if (sfv_info.items[i].ignored) {
                            list_show = 1;
                        }
                        break;
                }

                if (list_show) {
                    printf("%s\n", file_name);
                }
            }
        }

    }

    sfvInfoFree(&sfv_info);
    sfvInfoFree(&dir_info);
    return rc;
}

int isSfvFile(char* extension)
{
    DBUG_PRINT("isSfvFile", ("extension [%s]", extension));
    if (strcasecmp(extension, sfv_extension) == 0) {
        return 1;
    }
    if (strcasecmp(extension, "crc") == 0) {
        return 1;
    }
    return 0;
}

int sfvTestInDir(pure_sfv_params* params, char* sfv_dir)
{
    int             rc;
    int             i;
    crc_info_type   dir_info;
    char            current_dir_name[PATH_MAX];

    sfvInfoInit(&dir_info);

    DBUG_PRINT("FindSFVinDir", ("enter dir [%s]", sfv_dir));

    getcwd(current_dir_name, PATH_MAX);

    if (sfv_dir[0] != '\0') {
        if (chdir(sfv_dir) != 0) {
            fprintf(stderr, "sfv: directory [%s], cd [%s]: %s\n", current_dir_name, sfv_dir, strerror(errno));
            return ERROR_DIR_ACCESS;
        }
        getcwd(current_dir_name, PATH_MAX);
    }
    rc = dirRead("", &dir_info);
    if (rc) return rc;
    rc = dirReadAttributes("", &dir_info);
    if (rc) return rc;

    for(i = 0; i < dir_info.len; i++)
    {
        if (dir_info.items[i].is_dir) {
            DBUG_PRINT("sfvTestInDir", ("dir[%s]", dir_info.items[i].file_name));
            continue;
        }
        if (( (params->sfv_files.len == 0) &&
              (isSfvFile(dir_info.items[i].ext)) &&
              (isFile("", dir_info.items[i].file_name))) ||
            ( (params->sfv_files.len != 0) &&
              (includeFileSFV(dir_info.items[i].file_name, params))))  {

            if (!params->quiet) {
                printf("Curent directory: %s \n", current_dir_name);
            }
            rc = sfvTestFile(params, dir_info.items[i].file_name, "");
        }
    }

    sfvInfoFree(&dir_info);
    return 0;
}

int sfvTestInDirRecursive(pure_sfv_params* params, char* sfv_dir)
{
    int             rc;
    int             i;
    crc_info_type   dir_info;
    char            current_dir_name[PATH_MAX];

    sfvInfoInit(&dir_info);

    getcwd(current_dir_name, PATH_MAX);

    if (sfv_dir[0] != '\0') {
        if (chdir(sfv_dir) != 0) {
            fprintf(stderr, "sfv: directory [%s], cd [%s]: %s\n", current_dir_name, sfv_dir, strerror(errno));
            return ERROR_DIR_ACCESS;
        }
        getcwd(current_dir_name, PATH_MAX);
    }
    DBUG_PRINT("DirRecursive", ("enter dir [%s]", current_dir_name));

    rc = sfvTestInDir(params, "");
    if (rc) return rc;

    rc = dirRead("", &dir_info);
    if (rc) return rc;
    rc = dirReadAttributes("", &dir_info);
    if (rc) return rc;

    params->cnt.directories_scaned ++;

    for(i = 0; i < dir_info.len; i++)
    {
        if (dir_info.items[i].is_dir) {
            if (chdir(dir_info.items[i].file_name) != 0) {
                fprintf(stderr, "sfv: directory [%s], cd [%s]: %s\n", current_dir_name, dir_info.items[i].file_name, strerror(errno));
                /* return ERROR_DIR_ACCESS; */
                continue;
            }
            rc = sfvTestInDirRecursive(params, "");
            if (rc) return rc;

            if ((chdir(current_dir_name) != 0)) {
                fprintf(stderr, "sfv: cd [%s] %s\n", current_dir_name,  strerror(errno));
                return ERROR_DIR_ACCESS;
            }
        }
    }

    sfvInfoFree(&dir_info);
    return 0;
}

/* EOF */
