package PAR::StrippedPARL::Dynamic;
use 5.008001;
use strict;
use warnings;
our $VERSION = '0.958';

use base 'PAR::StrippedPARL::Base';

our $Data_Pos = tell DATA;

=head1 NAME

PAR::StrippedPARL::Dynamic - Data package containing a dynamic PARL

=head1 SYNOPSIS

  # For details, see PAR::StrippedPARL::Base.
  PAR::StrippedPARL::Dynamic->write_parl($file) or die "Some error...";

=head1 DESCRIPTION

This class is internal to PAR. Do not use it outside of PAR.

This class is basically just a container for a dynamic binary PAR loader
which doesn't include the PAR code like the F<parl> or F<parl.exe>
you are used to. If you're really curious, I'll tell you it is
just a copy of the F<myldr/par> (or F<myldr/par.exe>) file.

The data is appended during the C<make> phase of the PAR build process,
but only if applicable: If you perl is static, you won't get the dynamic
loader.

If the binary data isn't appended during the build process, the class
methods will return the empty list.

=head1 CLASS METHODS

Inherits the methods from L<PAR::StrippedPARL::Base>.

=cut

sub _data_pos {
    my $class = shift;
    return $Data_Pos;
}

=head1 AUTHORS

Steffen Mueller E<lt>smueller@cpan.orgE<gt>,
Audrey Tang E<lt>cpan@audreyt.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2006-2009 by Steffen Mueller E<lt>smueller@cpan.orgE<gt>.

This program is free software; you can redistribute it and/or 
modify it under the same terms as Perl itself.

See F<LICENSE>.

=cut

$^V eq 5.36.1 
    or die sprintf("Perl (%s) version (%vd) doesn't match the version (5.36.1) ".
                   "that PAR::Packer was built with; please rebuild PAR::Packer", 
                   $^X, $^V);

1;

__DATA__
M35J0``,````$````__\``+@`````````0```````````````````````````
M````````````````````@`````X?N@X`M`G-(;@!3,TA5&AI<R!P<F]G<F%M
M(&-A;FYO="!B92!R=6X@:6X@1$]3(&UO9&4N#0T*)`````````!010``9(8+
M`%V^C60``````````/``+@(+`@(H`$(```#*`````@```!`````0`````$``
M`0`````0`````@``!``````````%``(```````!@`0``!```/=8```,``(``
M`"`````````0```````````0````````$``````````````0```````!`+T!
M````$`$`]`H````@`0!0*P```-```-0!``````````````!0`0`P`````,``
M`!P`````````````````````````````````````````````````````````
MV!(!`'`"```````````````````````````````````N=&5X=````/A!````
M$````$(````$``````````````````!@``!@+F1A=&$```"``````&`````"
M````1@``````````````````0```P"YR9&%T80``#$,```!P````1````$@`
M`````````````````$```$`N8G5I;&1I9#4`````P`````(```",````````
M``````````!```!`+G!D871A``#4`0```-`````"````C@``````````````
M````0```0"YX9&%T80``T`$```#@`````@```)```````````````````$``
M`$`N8G-S`````.`!````\`````````````````````````````"```#`+F5D
M871A``"]`0`````!```"````D@``````````````````0```0"YI9&%T80``
M]`H````0`0``#````)0``````````````````$```,`N<G-R8P```%`K````
M(`$`4"L```"@``````````````````!```#`+G)E;&]C```P`````%`!```"
M````S```````````````````0```0@``````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M`````````````````````````````````````````````$B#["A(C0V%/P``
MZ.`Y``!%,<`QTC')Z/0Y``!%,<`QTC')Z/@Y``!%,<`QTC')Z/PY``!%,<`Q
MTC')2(/$*.G\.0``D)"0D)"0D)"0D)"03(T%J4\``#'22(T-$````.D;-P``
M9F8N#Q^$``````##D)"0D)"0D)"0D)"0D)"0055!5%575E-(@^PH28G-2(G7
M2(7)#X2U````2(L%'J```$B+*$B%[0^$H@````^V`83`=`0\/757,?9,BV4`
M387D=#)(8_9(B>MF#Q]$``!)B?!,B>I,B>'HPC@``(7`=0=!@#PT/71'3(MC
M"$B#PPA-A>1UVDR)X$B#Q"A;7E]=05Q!7<-F+@\?A```````2(G.#[9&`4B#
MQ@&$P'0$/#UU\$0I[NN3#Q^$``````!(*>M-C60T`4C!^P-,B>")'TB#Q"A;
M7E]=05Q!7<-FD$4QY$R)X$B#Q"A;7E]=05Q!7<-F9BX/'X0``````&:0059!
M54%455=64TB![+````!(BQT8GP``2(L#2(F$)*@````QP$F)S$B%R70%@#D`
M=29(BX0DJ````$@K`P^%'`$``$B!Q+````!;7E]=05Q!74%>PP\?`.@3-P``
M28G%2(7`=,U(C2TD7@``2(U\)"!,B>GH%S<``$B%P`^$G@```(!X&"Y(C7`8
M=0R`?@$`=-]F#Q]$``"`>!@N#X26````3(GAZ'8W``!(B?%)B<;H:S<``$&-
M3`8"2&/)Z$XV``!)B?%-B>!(B>I)B<9(B<'H^C8``$R)\4B)^NC_-@``3(GQ
M@_C_=!"+1"0P)0#P```]`$```'1?Z'(W``!,B?'HJC4``$R)Z>B"-@``2(7`
M#X5K____9@\?A```````3(GIZ%@U``!,B>'HD#8``.G]_O__#Q\`@'X!+@^%
M8/___X!^`@`/A"7____I4?___P\?@`````#HH_[__^N?Z/PT``"09F8N#Q^$
M``````!!5%-(@>RX````2(L=KYT``$B+`TB)A"2H````,<!(C50D($F)S.AE
M-0``A<!U,8M4)#"!X@#P``"!^@!```!T"('Z`*```'49N@(```!,B>'HJC0`
M`(7`#Y3`#[;`ZP(QP$B+E"2H````2"L3=0M(@<2X````6T%<P^AQ-```D%6X
M.``!`%=64^C2,P``2"G$2(LU*)T``#'20;@"````2(L&2(F$)"@``0`QP(G-
M2(U<)"#HQS0``$B->/](B?I(P?H_2,'J,$B-!!</M\!(*=!(*<</B(4````/
M'T0``$4QP$B)^HGIZ),T``!!N`@``0!(B=J)Z>@3-0``2)A(@_@'=DM(C40#
M^$@YV')!2(T5&DP``.L1#Q^$``````!(@^@!2#G8<B=(BPI(.0AU[T@IV$@!
M^$B+E"0H``$`2"L6=25(@<0X``$`6UY?7<-(@>\```$`2('_``#__W6`2,?`
M_____^O.Z((S``"0D$%4N#`0``!55U93Z.`R``!(*<1(BS4VG```0;C_#P``
M2(L&2(F$)"@0```QP$B-7"0@28G,2(G52(T-=UL``$B)VNAY-```A<!^18U0
M_TAC^H!\/"``=7"%TG0T28GXNB\```!(B=GHU#,``$B%P'0?2(L-R)L``.AC
M,@``28GY28G83(GB2(L(Z-HQ``#K&DB+#:F;``#H1#(``$F)Z$R)XDB+".C&
M,0``2(N$)"@0``!(*P9U%DB!Q#`0``!;7E]=05S##Q\`2&/XZX_HKC(``)!F
M9BX/'X0``````&:005=!5D%505155U932(/L.$B+'5F;``!(BP-(B40D*#'`
M,<"`.CU(B=8/E,!(B<](`<9(B?'H)C0``$B-5"0D2(GY2(G%Z`;[__])B<1(
MA<`/A/H```!(B<'H`C0``$@YQ0^&R0```$QC?"0D3(LM'9L``$G!YP,/M@>$
MP`^$C@$``#P]#X2&`0``2(GX#Q\`#[90`4B#P`&$TG0%@/H]=>](*?A(F$V+
M90!(C4PH`NB<,@``30'\28D$)$B%P'1028M%`$B-3P%*BQ0X#[8'B`*$P'4<
MZQYF#Q^$```````/M@%(@\(!2(/!`8@"A,!T!#P]=>O&`CTQP&8/'T0```^V
M#`:(3`(!2(/``83)=?!(BT0D*$@K`P^%!@$``$B#Q#A;7E]=05Q!74%>05_#
MD#'`9@\?1```#[84!D&(%`1(@\`!A-)U\.O&9@\?1```3(LM.9H``$V+=0!)
M@SX`#X2R````3(GP13'D#Q]$``!(@\`(1(GB08/$`4B#.`!U[X/"`TACTDC!
MX@.+!3_9``"%P'53QP4QV0```0```$B)T>BI,0``2(G!2(7`#X1=____36/\
M3(GR2<'G`TV)^.B[,0``28E%`$B)P4&-1"0!1(ED)"1(F$C'!,$`````Z9+^
M__\/'P!,B?'H`#(``$F)10!(B<%(A<`/A!#___]-8_Q)P><#Z\,/'X``````
M,<#IC_[__[H0````13'DZ6?____H=S```)!F#Q]$``!32(/L,$B+#229``#H
MOR\``$R-#5A;``!,C04A+P``2(L(2(T5C%@``$B)P^A'+P``3(T-F5@``,=$
M)"@`````3(T%*1$``$R)3"0@2(L+2(T5?%@``.@5+P``D$B#Q#!;PV9F+@\?
MA```````#Q\`2(/L*$B+#;68``#H4"\``$R-#>E:``!,C06R+@``2(L(2(T5
M'5@``$B#Q"CIURX``&9F+@\?A```````#Q]``#'`PV9F+@\?A```````9I!!
M5[C8$```059!54%455=64^@*+P``2"G$3(LM8)@``$F+10!(B80DR!```#'`
M2(UL)$!(B<Y)B=1(C0WI5P``2(GJZ!CX__^Z+P```$B)\4B)P^CH,```2(7`
M=#M(B?)(C0W+5P``Z*3\__](B?!(BY0DR!```$DK50`/A7P!``!(@<38$```
M6UY?74%<05U!7D%?PP\?`$R)X4B-/9U7``!,C26?5P``Z+(P``!(B?I(B<'H
M]S```$F)QTB%P'2<08`_``^$UP````\?1```2(7;=!-,B?I(B=GH<#```(7`
M#X2:````3(GYZ(`P``!)C50'_TF)QDDYUW(/ZQV0Q@(`2(/J`4DYUW0%@#HO
M=.],B?GH5S```$F)QDB)\>A,,```28U$!@%(/?\/```/ARC___](B70D,#'2
M3(VT),````!,C0WV5@``3(ED)"A,B?%!N``0``!,B7PD(.@]+@``2(GJ3(GQ
MZ,(O``"%P'40BT0D4"4`\```/0"```!T3DB)^C')Z"0P``!)B<=(A<`/A,7^
M__]!@#\`#X4N____2(7;=`6`.RYT%$&^`0```$R-/6Y6``#I7O___V:0@'L!
M`'2\Z^0/'X0``````+H!````3(GQZ.,M``"%P'6A3(GR2(T-1U8``.@@^___
M3(GQZ&@O``#I<O[__^BN+0``D&9F+@\?A```````9I!!5$B#[""Z+P```$F)
MS.AM+P``2(U0`4B%P$P/1>),B>%(@\0@05SI)"\```\?0`!!5;@X$```0516
M4^C0+```2"G$2(L=)I8``$B+`TB)A"0H$```,<!)B<U(A<D/A)T```"`.0`/
MA)0```#H[RX``$R-0`%(B<9)@?@`$```#X>S````3(UD)"!!N0`0``!,B>I,
MB>'HY2P``$F-1#3_3#G@=Q?K&F8/'X0``````,8``$B#Z`%,.>!T!8`X+W3O
MNB\```!,B>'HLBX``$B%P'0M3#G@=UA,B>'H<"X``$B+E"0H$```2"L3=4](
M@<0X$```6UY!7$%=PP\?1```2(N$)"@0``!(*P-U+TB-#0-5``!(@<0X$```
M6UY!7$%=Z2HN``!F+@\?A```````Q@``ZZ,/'P`QP.NDZ%\L``"09F8N#Q^$
M```````/'P!!5T%6055!5%575E-(@^PX2(LM"94``$B+-2*5``!(C1W%5```
M2(M%`$B)1"0H,<!,C60D)$R)XDB)V>C`]/__2(7`=#M(8U0D)$B+!DB-!-!(
MBU`(2(D02(72=-@/'P!(BU`02(/`"$B)$$B%TG7P3(GB2(G9Z(7T__](A<!U
MQ4B+-;F4``!(C1UE5```3(GB2(G9Z&?T__](A<!T0DAC5"0D2(L&2(T$T$B+
M4`A(B1!(A=)TV&8N#Q^$``````!(BU`02(/`"$B)$$B%TG7P3(GB2(G9Z"7T
M__](A<!UODB+-5F4``!(C1T-5```3(GB2(G9Z`?T__](A<!T0DAC5"0D2(L&
M2(T$T$B+4`A(B1!(A=)TV&8N#Q^$``````!(BU`02(/`"$B)$$B%TG7P3(GB
M2(G9Z,7S__](A<!UODB+-?F3``!(C1VV4P``3(GB2(G9Z*?S__](A<!T0DAC
M5"0D2(L&2(T$T$B+4`A(B1!(A=)TV&8N#Q^$``````!(BU`02(/`"$B)$$B%
MTG7P3(GB2(G9Z&7S__](A<!UODB+'9F3``!,C2U=4P``3(GB3(GIZ$?S__](
MA<!T0DAC5"0D2(L#2(T$T$B+4`A(B1!(A=)TV&8N#Q^$``````!(BU`02(/`
M"$B)$$B%TG7P3(GB3(GIZ`7S__](A<!UODB+-3F3``!(C1T-4P``3(GB2(G9
MZ.?R__](A<!T0DAC5"0D2(L&2(T$T$B+4`A(B1!(A=)TV&8N#Q^$``````!(
MBU`02(/`"$B)$$B%TG7P3(GB2(G9Z*7R__](A<!UODB+'=F2``!,C35;4@``
M3(GB3(GQZ(?R__](A<!T0DAC5"0D2(L#2(T$T$B+4`A(B1!(A=)TV&8N#Q^$
M``````!(BU`02(/`"$B)$$B%TG7P3(GB3(GQZ$7R__](A<!UODB+-7F2``!(
MC1U94@``3(GB2(G9Z"?R__](A<!T0DAC5"0D2(L&2(T$T$B+4`A(B1!(A=)T
MV&8N#Q^$``````!(BU`02(/`"$B)$$B%TG7P3(GB2(G9Z.7Q__](A<!UODB+
M-1F2``!,C3T#4@``3(GB3(GYZ,?Q__](A<!T0DAC5"0D2(L&2(T$T$B+4`A(
MB1!(A=)TV&8N#Q^$``````!(BU`02(/`"$B)$$B%TG7P3(GB3(GYZ(7Q__](
MA<!UODB+/;F1``!(C36M40``3(GB2(GQZ&?Q__](A<!T0DAC5"0D2(L'2(T$
MT$B+4`A(B1!(A=)TV&8N#Q^$``````!(BU`02(/`"$B)$$B%TG7P3(GB2(GQ
MZ"7Q__](A<!UODB+/5F1``!(C37D4```3(GB2(GQZ`?Q__](A<!T0DAC5"0D
M2(L'2(T$T$B+4`A(B1!(A=)TV&8N#Q^$``````!(BU`02(/`"$B)$$B%TG7P
M3(GB2(GQZ,7P__](A<!UODR)XDB-#?M0``#HL?#__TB)PDB%P'0(3(GYZ%'U
M__],B>)(C0WM4```Z)+P__](B<)(A<!T#$B-#>M0``#H+O7__TR)XDB-#>=0
M``#H;_#__TB)PDB%P'0R3(GQZ`_U__](BT0D*$@K10!U0$B-%>%0``!,B>E(
M@\0X6UY?74%<05U!7D%?Z>7T__],B>)(C0VN4```Z";P__](B<)(A<!TOTB)
MV>C&]/__Z[7H7R<``)!F9BX/'X0```````\?`%-(@^PP2(L=%)```$B-#110
M``!(BP-(B40D*#'`2(U4)"3HV>___TB%P'0L#[80@/HP#Y3`A-(/E,()T(/P
M`0^VP$B+5"0H2"L3=1!(@\0P6\-F#Q]$```QP.OFZ.\F``"09F8N#Q^$````
M```/'P!(@^PHZ%<G``")P>A`)P``2(7`=`-(BP!(@\0HPP\?`$%6055!5%93
M2(/L0$B+-7V/``!,C2WL3P``2(L&2(E$)#@QP$F)S$B-5"0T3(GIZ#SO__](
MB<-(A<!T!8`X`'4O2(M$)#A(*P8/A:$```!,B>),B>E(@\1`6UY!7$%=05[I
MNO/__V8N#Q^$``````!,B>)(B<'H-2@``$B%P'0@2(M$)#A(*P9U9DB#Q$!;
M7D%<05U!7L-F#Q^$``````!,B>'HV"<``$B)V4F)QNC-)P``28U,!@+HLR8`
M`$B)7"0@38G@3(T-@TX``$B-%7Y.``!(B<%)B<;H4B<``$B+1"0X2"L&=0A,
MB?+I8O___^C+)0``D&8N#Q^$``````!!5T%6055!5%575E-(@>R8`0``2(L]
M=HX``$R--1A.``!(BP=(B80DB`$``$B-!01.``!(B<Y,C6PD4$R)\4C'A"3@
M`````````$R)ZDC'A"0X`0```````$C'A"0``0```````$B)A"38````2(T%
MJ$X``$B)A"08`0``2(T%H$X``$B)A"0@`0``2(T%F4X``$B)A"0H`0``2(T%
MCTX``$B)A"0P`0``2(T%A$X``$B)A"3X````Z+/M__])B<1(A<!T"8`X``^%
M@@0``.AM)0``B<'H5B4``$B%P`^$[0,``$B+*$B%[0^$X0,``$B)Z>B))@``
M2(U,``'H;R4``$0/MD4`2(G#183`=#-)B=],C24E3@``#Q^$``````!%#[;`
M3(GY3(GB2(/%`>C])0``1`^V10!)@\<"183`==](C:PD$`$``$B-#8A-``#K
M%V8/'T0``$B+30A(@\4(2(7)#X0'`0``3(GJZ/_L__])B<1(A<!TWX`X`'3:
M2(G!Z$KO__^%P'3.3(GAZ-XE``!(B<=(A<`/A-(```!FD$B)^>C8)0``2(G9
M2(G%Z,TE``!$C:0%!`0``$UCY$R)X>BJ)```2(E<)"A(C15P30``28GX28G'
M2(T%;$T``$R-#7),``!(B40D($R)^>@])0``NL`!``!,B?GHL"0``(/X_P^$
M?P(``$R)ZDR)^>A,)```B<.%P`^$P@```$B+'D4QY.C7(@``38GY2(T574T`
M`$B+2!A)B=CHP2,``$B+/4J,``!(BX0DB`$``$@K!P^%5`0``$R)X$B!Q)@!
M``!;7E]=05Q!74%>05_#9I`Q[4R-)9Y,``#K'`\?1```2&/%3(NDQ-````!-
MA>1T0$&`/"0`=#E,B>&#Q0'H+N[__X7`=-I,B>'HPB0``$B)QTB%P`^%YO[_
M_TACQ4R+I,30````387D=<9F#Q]$```Q_^G)_O__9@\?A```````BT0D8"4`
M\```/0!````/A2K___^+;"1HZ$,C```YQ0^%&?___XM$)&`E_P$``#W``0``
M#X4%____3(GAZ$\C``!,B>I(C0W;2P``28G$Z#WK__](B<)(A<`/A%$"``!(
MBP[HR?+__TB)Q4B%P`^$/0(``$B)Z>@E)```28G`@_@'?CU(F$B-%7=,``!,
MB40D2$B-3`7XZ.4C``"%P'4A3(M$)$A!@_@(#X3B`@``08U`]TB8@'P%`"\/
MA-$"``"02(TU\$H``$R)ZDB)\>B^ZO__2(7`#X3E`0``#[8`A,`/A-H!```\
M,`^$T@$``$B)\4B-%2Q+``!(C35"2@``Z#SO__],C2UM2@``2(TM(TP``.@I
M(@``2(ET)"A-B?A,B>&)1"0@3(T-2TH``$B-%?-+``#H%R,``.LU#Q]$``#H
M6R$``(,X$74XZ/$A``"#PP%(B70D,$V)Z8E<)"A-B?A(B>I,B>&)1"0@Z.`B
M``"ZP`$``$R)X>A3(@``@_C_=+Y,B?GHIB$``$R)\4R)XNBK[O__3(GAZ(/Z
M___IO?W__V8/'T0``.CS(```@S@1#X1S_?__Z.4@``!(BQY%,>2+*.A8(```
M38GY28G82(T5FTH``$B+2!B);"0@Z#XA``#I>/W__V8/'X0``````#';2(T-
M,DH``.L5#Q]$``!(8\-(BXS$\````$B%R70Y3(GJ@\,!Z'7I__](A<!TX(`X
M`'3;2(G!Z&,B``!(B<5(A<`/A=;[__](8\-(BXS$\````$B%R77'N0T```!(
MC2W;20``Z#8A``!!N%,```!(B</IQOO__P\?A```````2(G!Z*CY__](BSTQ
MB0``2(N$)(@!``!(*P</A3L!``!,B>%(@<28`0``6UY?74%<05U!7D%?Z>4A
M```/'T0``$B++NF[_?__#Q^$``````"Z```!`$B)Z>@3(0``08G%A<`/A!;^
M__^)P>B1Z___2(G%2(7`#X@#_O__13'`2(U0]D2)Z>AV(```2(V4)$H!``!!
MN`8```!$B>GH\"```(&\)$H!````0T%##X7-_?__9H&\)$X!``!(10^%O?W_
M_TB-5<Y%,<!$B>GH,"```$B-G"10`0``1(GI0;@H````2(G:Z*<@``!(B5PD
M($V)^$R)X4B-!=9'``!,C0T&2```QH0D>`$```!(B40D*$B-%9))``#HQ2``
M`.GR_?__2(MV"$B%]@^$(_W__TB)\>C[(```2(/X`P^&$?W__TB-3`;\2(T5
M44D``.C`(```A<!(#T3NZ?7\___H$!\``)!F9BX/'X0```````\?0`!!5%=6
M4TB#[#A(BPVPAP``2(L]N8<``$B+!TB)1"0H,<!,C60D).@U'@``2(T-B4<`
M`$B)PTB+``^VD+````"(D+$```!,B>+H8^?__TB%P'0)@#@Q#X15`0``Z%#R
M__](BPM!N0,```!!N`$```!(C17P2```Z)T=``!(B<9(A<!T)TB+0!!(BPM,
MC0772```2(L0Z%<=``!(BT802(L0]D(.0`^%-@$``$B+"T&Y`P```$&X`0``
M`$B-%:U(``#H4QT``$B)QDB%P'0J3(GB2(T-JT8``.C4YO__28G`2(M&$$V%
MP`^$U````$B+$$B+"^CY'```2(L+0;@!````2(T5:$@``.@4'0``2(L+3(T%
M;D@``$B)PNC2'```2(L#2(T-448``$R)XL:`L0````#H>>;__TB)!?K%``!(
MB<%(A<!T.KK``0``Z*`>``"-4`&#^@%V"8/X$0^%C````$B+1"0H2"L'#X6>
M````2(/$.%M>7T%<PP\?@`````!(BPVYQ0``Z+3W__](B06EQ0``2(G!2(7`
M=:M(C0WN1P``Z($<``!F#Q^$``````"`>`$`#X2F_O__Z9S^__^02(L5><4`
M`$B+"$B+$NBNZ?__Z2#___]F#Q^$``````!(BPOH*!P``.F]_O__08G!2(L%
M2<4``$R+!3K%``!(C0VS1P``2(L0Z!L<``#H[AP``)!F9BX/'X0``````&:0
M055!5%-(@^PP2(L=H(4``$B+`TB)1"0H,<!)B<WH3N___[HO````2(G!28G$
MZ(X>``!(C5`!2(7`3`]%XDR)X>A+'@``2(U4)"1(C0UF10``28G$Z#?E__](
MA<!T(@^V$(32#Y3`@/HP#Y3""-!U$$V%[70+08!]``!U'`\?0`!(BT0D*$@K
M`W5"2(/$,%M!7$%=PP\?0`!!N`0```!(C17&10``3(GAZ`,>``"%P'7/2(M$
M)"A(*P-U$4R)Z4B#Q#!;05Q!7>FTY?__Z`\<``"0D)"0D)"0D)"0D)"0D)!!
M5T%6055!5%575E-(@>RX`@``2(L]MH0``$B+!TB)A"2H`@``,<!(B<I,C5DX
M2(U,)"!(C9PDH````$F)RF:008L#38L+28/"$$F#PP@/R(G`38G(28E"\$R)
MR$G!Z"!(P>@H2<'I."4`_P``3`G(38G!2<'@"$G!X1A!@>```/\`18G)3`G(
M3`G`28E"^$DYVG6L3(V)``(```\?1```2(M!:$@S04!(@\$(2#-!"$@S0?A,
MC00`2,'H'TP)P(/@_TB)07A,.<EUU4R+&DB+<A!(BUH@2(MZ&$R)V$R)V4B+
M:@A,BU0D*$C!Z1M(P>`%2`G(`T0D($Z-E!>9>8):C8P8F7F"6DB)\$@Q^$@A
MZ$@Q^$2-#`%(B>E(B>A(P>@"2,'A'DR)RT@)P4B)\$C!ZQM!B<A,,<!,(=A(
M,?!,`=!-B<I)P>(%20G:1HTD$$R)V$V)VDG!Z@)(P>`>38GE3`G03(M4)#!)
MP>T;2#'!B<-,(<E.C906F7F"6DPQP4P!T4V)XDG!X@5-">I,BVPD.$$!RDR)
MR4G!Z0)(P>$>18G23XV$*)EY@EI,"<E-B=5(,<A!B<E,(>!)P>T;2#'83`'`
M38G02<'@!4T)Z$R+;"1`00'`3(G@2<'L`DC!X!Y%B<!*C9PKF7F"6DP)X$V)
MQ4@QP4G![1M!B<1,(=%,,<E(`=E,B<-(P>,%3`GK3(ML)$@!RTR)T4G!Z@)(
MP>$>B=M/C8PIF7F"6DP)T4F)W4@QR$G![1M!B<I,(<!,,>!,`<A)B=E)P>$%
M30GI3(ML)%!!`<%,B<!)P>@"2,'@'D6)R4^-I"R9>8):3`G`38G-2#'!2<'M
M&T&)P$@AV4PQT4P!X4V)S$G!Y`5-">Q,BVPD6$$!S$B)V4C!ZP)(P>$>18GD
M3XV4*IEY@EI("=E-B>5(,<A)P>T;B<M,(<A,,<!,`=!-B>))P>(%30GJ3(ML
M)&!!`<),B<A)P>D"2,'@'D6)TD^-A"B9>8):3`G(38G52#'!2<'M&T&)P4PA
MX4@QV4P!P4V)T$G!X`5-">A,BVPD:$$!R$R)X4G![`)(P>$>18G`2HV<*YEY
M@EI,">%-B<5(,<A)P>T;08G,3"'03#'(2`'83(G#2,'C!4P)ZTR+;"1P`<-,
MB=!)P>H"2,'@'HG;3XV,*9EY@EI,"=!)B=U(,<%)P>T;08G"3"'!3#'A3`')
M28G92<'A!4T)Z4R+;"1X00')3(G!2<'H`DC!X1Y%B<E/C:0LF7F"6DP)P4V)
MS4@QR$G![1M!B<A((=A,,=!,`>!-B<Q)P>0%30GL3(NL)(````!!`<1(B=A(
MP>`>2,'K`D6)Y$^-E"J9>8):2`G838GE2#'!2<'M&XG#3"')3#'!3`'138GB
M2<'B!4T)ZDR+K"2(````00'*3(G)2<'I`DC!X1Y%B=)/C80HF7F"6DP)R4V)
MU4@QR$G![1M!B<E,(>!(,=A,`<!-B=!)P>`%30GH3(NL))````!!`<!,B>!)
MP>P"2,'@'D6)P$J-G"N9>8):3`G@38G%2#'!2<'M&T&)Q$PAT4PQR4@!V4R)
MPTC!XP5,">M,BZPDF`````'+3(G12,'A'DG!Z@*)VT^-C"F9>8):3`G128G=
M2#'(2<'M&T&)RDPAP$PQX$P!R$F)V4G!X05-">E,BZPDH````$$!P4R)P$G!
MZ`)(P>`>18G)3XVD+)EY@EI,"<!-B<U(,<%)P>T;08G`2"'93#'13`'A38G,
M2<'D!4T)[$:--"%(B=E(P>L"2,'A'D@)V4B+G"2H````2#'(08G,3"'(38V4
M&IEY@EI,B?-,,<!(P>L;3`'038GR2<'B!4D)VDB+G"2P````00'"3(G(2,'@
M'DG!Z0)%B=)-C808F7F"6DP)R$R)TT@QP4C!ZQM!B<%,(?%,,>%)`<A,B=%(
MP>$%2`G93(GS2<'N`DC!XQZ)R4P)\TD!R$B+C"2X````2#'818G%3"'028V,
M#)EY@EI-B>Q,,<A)P>P;2`'(3(GI2,'A!4P)X8G)2`'(3(G12<'J`DC!X1Y!
MB<1,"=%)B=J)VTDQRD4QT$R+E"3`````3XV,$:'KV6Y-B>)-`<%-B>!)P>H;
M2<'@!4T)T$6)P$T!P4V)Z$G!X!Y)P>T"18G*2`.<),@```!-">A)B<V)R4@#
MC"30````33'%1#'H38G52(V<`Z'KV6Y,B=!)P>T;2,'@!4P)Z(G`2`'#3(G@
M2<'L`DC!X!Y!B=U,">!-B<1%B<!,`X0DV````$DQQ$4QX4V)[$Z-C`FAZ]EN
M3(GI2<'L&TC!X05,">&)R4D!R4R)T4G!Z@)(P>$>18G,3`G128G"B<!),<I!
M,=I,B>-/C900H>O9;DV)X$C!ZQM)P>`%20G818G`30'"38GH2<'M`DG!X!Y$
MB=--">A)B<U(`X0DX````(G)33'%2`.,).@```!%,>E)B=U.C8P(H>O9;DB)
MV$G![1M(P>`%3`GHB<!)`<%,B>!)P>P"2,'@'D6)S4P)X$V)Q$6)P$P#A"3P
M````23'$13'B38GL3HV4$:'KV6Y,B>E)P>P;2,'A!4P)X8G)20'*2(G92,'K
M`DC!X1Y%B=1("=E(B<.)P$@#A"3X````2#'+03'93(GC3XV,"*'KV6Y-B>!(
MP>L;2<'@!4D)V$6)P$T!P4V)Z$G![0))P>`>1(G+30GH28G-B<E-,<5(`XPD
M``$``$4QZDF)W4Z-E!"AZ]EN2(G82<'M&TC!X`5,">B)P$D!PDR)X$G![`)(
MP>`>18G53`G@38G$18G`3`.$)`@!``!),<1%,>%-B>Q.C8P)H>O9;DR)Z4G!
M[!M(P>$%3`GAB<E)`<E(B=E(P>L"2,'A'D6)S$@)V4B)PXG`2`.$)!`!``!(
M,<M!,=I,B>-/C900H>O9;DV)X$C!ZQM)P>`%20G818G`30'"38GH2<'M`DG!
MX!Y$B=--">A)B<V)R4TQQ44QZ4F)W4Z-C`BAZ]EN2(G82,'@!4G![1M(`XPD
M&`$``$P)Z(G`20'!3(G@2<'L`DC!X!Y%B<U,">!-B<1%B<!,`X0D(`$``$DQ
MQ$4QXDV)[$Z-E!&AZ]EN3(GI2<'L&TC!X05,">&)R4D!RDB)V4C!ZP)(P>$>
M18G42`G92(G#B<!(`X0D*`$``$@QRT$QV4R)XT^-C`BAZ]EN38G@2,'K&TG!
MX`5)"=A%B<!-`<%-B>A)P>T"2<'@'D2)RTT)Z$F)S8G)33'%13'J28G=3HV4
M$*'KV6Y(B=A)P>T;2,'@!4P)Z$@#C"0P`0``B<!)`<),B>!)P>P"2,'@'D6)
MU4P)X$V)Q$6)P$P#A"0X`0``23'$13'A38GL3HV,":'KV6Y,B>E)P>P;2,'A
M!4P)X8G)20')2(G92,'K`DC!X1Y%B<Q("=E(B<.)P$@#A"1``0``2#'+03':
M3(GC3XV4$*'KV6Y-B>!(P>L;2<'@!4D)V$6)P$T!PDV)Z$G![0))P>`>1(G3
M30GH28G-B<E-,<5%,>E)B=U.C8P(H>O9;DB)V$G![1M(P>`%3`GHB<!)`<%,
MB>!(P>`>2<'L`D6)S4@#C"1(`0``3`G@38G$18G`3`.$)%`!``!),<1%,>).
MC:01H>O9;DR)Z4V)ZDG!ZAM(P>$%3`G1B<E)`<Q(B=E(P>L"2,'A'D6)XD@)
MV4B)PXG`2`.$)%@!``!(,<M!,=E,B=-/C8P(H>O9;DV)T$C!ZQM)P>`%20G8
M3(GK2<'M`DC!XQY%`<%)B<B)R4P)ZT6)R4DQV$F)WD&)W44QX$V)S$Z-A`"A
MZ]EN3(G(2<'L&TC!X`5,">!%C20`38G02<'J`KC<O!N/2<'@'DT)T$T)QD6)
MPDPATTTASDD)WDB+G"1@`0``2`'#2`'93(GC3`'Q38GF2,'C!4G![AM,"?,!
MRTR)R4G!Z0)(P>$>B=M,"<E)"<A)B<Y!B<E-(>!-(=9-"<9,BX0D:`$``$D!
MP$T!Z$F)W4T!\$F)WDG!Y05)P>X;30GU28G.10'%38G@2<'L`DG!X!Y%B>U-
M">!%B<1-"<9)(=Y,(>%,"?%,B[0D<`$``$D!QDT!UDV)ZDP!\4V)[DG!X@5)
MP>X;30GR00'*2(G92,'K`DC!X1Y%B=)("=E)"<A)B<Z)RTTAZ$TAYDT)QDR+
MA"1X`0``20'`30'(38G130'P38G62<'A!4G![AM-"?%)B<Y%`<%-B>A)P>T"
M2<'@'D6)R4T)Z$6)Q4T)QDTAUDPAZ4P)\4R+M"2``0``20'&30'F38G,3`'Q
M38G.2<'D!4G![AM-"?1!`<Q,B=%)P>H"2,'A'D6)Y$P)T4D)R$F)SD&)RDTA
MR$TA[DT)QDR+A"2(`0``20'`20'83(GC30'P38GF2,'C!4G![AM,"?-)B<Y$
M`<--B<A)P>D"2<'@'HG;30G(18G!30G&32'F3"')3`GQ3(NT))`!``!)`<9-
M`>Y)B=U,`?%)B=Y)P>4%2<'N&TT)]4$!S4R)X4C!X1Y)P>P"18GM3`GA20G(
M28G.08G,22'832'.30G&3(N$))@!``!)`<!-`=!-B>I-`?!-B>Y)P>(%2<'N
M&TT)\DF)SD4!PDF)V$C!ZP))P>`>18G220G81(G#30G&32'N2"'93`GQ3(NT
M)*`!``!)`<9-`<Y-B=%,`?%-B=9)P>$%2<'N&TT)\4$!R4R)Z4G![0)(P>$>
M18G)3`GI20G(28G.08G-32'022'>30G&3(N$)*@!``!)`<!-`>!-B<Q-`?!-
MB<Y)P>0%2<'N&TT)]$F)SD4!Q$V)T$G!Z@))P>`>18GD30G018G"30G&32'.
M3"'13`GQ3(NT)+`!``!)`<9)`=Y,B>-,`?%-B>9(P>,%2<'N&TP)\P'+3(G)
M2<'I`DC!X1Z)VTP)R4D)R$F)SD&)R4TAX$TAUDT)QDR+A"2X`0``20'`30'H
M28G=30'P28G>2<'E!4G![AM-"?5)B<Y%`<5-B>!)P>P"2<'@'D6)[4T)X$6)
MQ$T)QDDAWDPAX4P)\4R+M"3``0``20'&30'638GJ3`'Q38GN2<'B!4G![AM-
M"?)!`<I(B=E(P>L"2,'A'D6)TD@)V4D)R$F)SHG+32'H32'F30G&3(N$),@!
M``!)`<!-`<A-B=%-`?!-B=9)P>$%2<'N&TT)\4F)SD4!P4V)Z$G![0))P>`>
M18G)30GH18G%30G&32'63"'I3`GQ3(NT)-`!``!)`<9-`>9-B<Q,`?%-B<Y)
MP>0%2<'N&TT)]$$!S$R)T4G!Z@)(P>$>18GD3`G120G(28G.08G*32'(32'N
M30G&3(N$)-@!``!)`<!)`=A,B>--`?!-B>9(P>,%2<'N&TP)\TF)SD0!PTV)
MR$G!Z0))P>`>B=M-"<A-"<9%B<%-(>9,(<E,"?%,B[0DX`$``$D!QDT![DF)
MW4P!\4F)WDG!Y05)P>X;30GU00'-3(GA2<'L`DC!X1Y%B>U,">%)"<A)B<Q!
MB<])(=A-(<Q-"<1,BX0DZ`$``$D!P$T!T$V)ZDT!X$V)[$G!X@5)P>P;30GB
M10'"28G82,'K`DG!X!Y%B=))"=A(B<M%B<1,"<-,(>M,(>%("=E(BYPD\`$`
M`$@!PT@#A"3X`0``20'93(G33`'X3(N\)!@"``!)`<E,B=%(P>L;2,'A!4@)
MV8G)20')3(GI2<'M`DC!X1Y$B<M,">E)"<A)B<U!B<Y-(=!-(>5-"<5-C40%
M`$B)V$F)W4G![1M(P>`%3`GHB<!)`<!,B=!)P>H"2,'@'D6)Q4D)PKC6P6+*
M3#'11#')3(N,)``"``!)`<%-`>%-B>Q,`<E-B>E)P>P;2<'A!4T)X4F)W$6)
MR4P!R4G!Y!Y)`<=(P>L"08G)3`GC38G418G223'<13'$3(N$)`@"``!)`<!-
M`?!-B<Y-`>!-B<Q)P>X;2<'D!4T)]$V)[DG![0))P>8>18GD30GU3(NT)!`"
M``!-`>!%B<1)`<9-`?))B=Y-,>Y!,<Y,B>%-`?)-B>9(P>$%2<'N&TP)\8G)
M20'*3(G)2<'I`DC!X1Y%B=9)"<F)V4R)ZTPQRTP!^4R+O"0@`@``1#'#3(T$
M&4R)\4R)\TC!X05(P>L;20''2`G9B<E)`<A,B>%)P>P"2,'A'D2)PTP)X46)
M[$V)S46)R4DQS4T!_$R+O"0X`@``13'53XT4+$F)W$F)W4G![1M)P>0%30GL
M38GU2<'N`DG!Y1Y%B>1-">Y,BZPD*`(``$T!XD6)U$D!Q4T!Z4F)S8G)33'U
M13'%38G@30'I38GE2<'@!4G![1M-">A%B<!-`<%)B=A(P>L"2<'@'D6)S4D)
MV$B+G"0P`@``2`'#2`'93(GS3#'#1#'320''3(T4&4R)Z4R)ZTC!ZQM(P>$%
M2`G9B<E)`<I,B>%)P>P"2,'A'D2)TTD)S$2)\4V)QD6)P$TQYDP!^4R+O"1(
M`@``13'.3HT,,4B)V4F)WDD!QTG![AM(P>$%3`GQB<E)`<E,B>E)P>T"2,'A
M'D6)SDP)Z4R+K"1``@``20'%30'H38GE23'-13'53XT4*$V)\$V)]4G![1M)
MP>`%30GH18G`30'"28G82,'K`DG!X!Y%B=5,"<-%B>!)B<R)R4T!^$DQW$R+
MO"18`@``13',3XT,($V)Z$V)[$D!QTG![!M)P>`%30G@18G`30'!38GP2<'N
M`DG!X!Y%B<Q-"?!,B[0D4`(``$D!QDP!\4F)WDTQQD4QUDZ-%#%,B>%-B>9)
MP>X;2,'A!4P)\8G)20'*3(GI2<'M`DC!X1Y%B=9)"<V)V4R)PT6)P$PQZTP!
M^4R+O"1H`@``1#'+3(T,&4R)\4R)\TC!ZQM(P>$%2`G9B<E)`<E,B>%)P>P"
M2,'A'D2)RTP)X4R+I"1@`@``20'$30'@38GL20''23',13'43XT4($F)V$F)
MW$G![!M)P>`%30G@18G`30'"38GP2<'N`DG!X!Y%B=1-"<9%B>A)B<V)R4TQ
M]4T!^$R+O"1X`@``13'-3XT,*$V)X$V)Y4G![1M)P>`%30GH18G`30'!28G8
M2,'K`DG!X!Y%B<U)"=A(BYPD<`(``$@!PT@!V4R)\TPQPT0QTTR-%!E,B>E,
MB>M(P>L;2,'A!4@)V8G)20'*3(GA2<'L`DC!X1Y$B=-)"<Q$B?%-B<9)`<=-
M,>9,`?E%B<!,B[PDB`(``$4QSDF)V4D!SDB)V4G!Z1M)`<=(P>$%3`G)B<E)
M`<Y,B>E)P>T"2,'A'D6)\4P)Z4R+K"2``@``20'%30'H38GE23'-13'53XT4
M*$V)R$V)S4G![1M)P>`%30GH18G`30'"28G82,'K`DG!X!Y%B=5,"<-%B>!)
MB<R)R4DQW$T!^$4Q]$V)[DT!X$V)[$G![AM)P>0%30GT38G.18GD30'@2<'F
M'DG!Z0)%B<1-"?%,B[0DD`(``$D!QD@#A"28`@``3`'Q28G>3`'8B=M-,<Y(
M`=A%,=9-B>),`?%-B>9)P>(%2<'N&TT)\D0!T4V)ZDG![0))P>(>B<E-"=5-
MB<I$`TH@33'J00']3(E*($4QT$F)RDR):AA,`<!)B<A)P>H;`>E)P>`%2(E*
M"$T)T$0!P$B)`DR)X$G![`)(P>`>3`G@`?!(BS6(;@``2(E"$$B+A"2H`@``
M2"L&=11(@<2X`@``6UY?74%<05U!7D%?P^B.!0``D&9F+@\?A```````9I!(
M@^PHN8````#H(@8``+J)J\WON?[<NIA(B5`(2('JF<GZ*TC'``$C16=(B4@0
M2,=`&'94,A!(B5`@2,=`*`````!(QT`P`````,=`>`````!(@\0HPV8N#Q^$
M``````!!54%455=64TB#["A)8\!,BT$H2(G/2(G32(M7,$F)Q$B-=SA!C0S`
M3#G!2(E/*$AC3WA(@](`2,'H'4@!T$B)1S"%R71(O4````!(B=HIS40YY4$/
M3^Q(`?%,8^U-B>CHBP4``(M'>`'HB4=X@_A`=`Y(@\0H6UY?74%<05W#D$B)
M^4$I[$P!Z^B2Z/__08/\/P^.V````$&#[$!%B>5!P>T&08UM`4C!Y09(`=UF
MD/,/;P-(B?E(@\-`#Q$&\P]O2]`/$4X0\P]O4^`/$58@\P]O6_`/$5XPZ$+H
M__](.>MUS4'!Y09%*>Q)8\1(@_@(<EU(BU4`2(U."$B#X?A(B5<X2(M4!?A(
MB50&^$@ISD@!\$@I]4B#X/A(@_@(<AA(@^#X,=),BT05`$R)!!%(@\((2#G"
M<NY$B6=X2(/$*%M>7UU!7$%=PP\?@`````"H!'4D2(7`=-\/ME4`B%<XJ`)T
MU`^W5`7^9HE4!O[KR$B)W>ER____BU4`B5<XBU0%_(E4!OSKL`\?A```````
M05155U932(/L($B+0BB+<BB+>C!(P>@##\Z#X#\/STB)RXU(`4B828G4QD0"
M.(!(C6HX2&/!2`'H@_DX#XY.`0``0;A`````02G(=!`QTHG1@\(!Q@0(`$0Y
MPG+R3(GAZ"OG__])QT0D.`````!)QT0D0`````!(QT40`````$C'11@`````
M2,=%(`````!(QT4H`````$C'13``````08E\)'!,B>%!B70D=.C?YO__28L$
M)$R)X4C!Z!B(`TF+!"1(P>@0B$,!28L$)(AC`DF+!"2(0P-)BT0D"$C!Z!B(
M0P1)BT0D"$C!Z!"(0P5)BT0D"(AC!DF+1"0(B$,'28M$)!!(P>@8B$,(28M$
M)!!(P>@0B$,)28M$)!"(8PI)BT0D$(A#"TF+1"082,'H&(A##$F+1"082,'H
M$(A##4F+1"08B&,.28M$)!B(0P])BT0D($C!Z!B(0Q!)BT0D($C!Z!"(0Q%)
MBT0D((AC$DF+1"0@B$,32(/$(%M>7UU!7.E:`@``9BX/'X0``````+HX````
M13')*<J#^@AR28G12,<``````$C'1`CX`````$B-2`A(@^'X2"G(`<*#XOB#
M^@@/@M7^__^#XO@QP$&)P(/`"$Z)#`$YT'+RZ;W^__]F#Q]$``#VP@1U(X72
M#X2J_O__Q@``]L("#X2>_O__,<EFB4P0_NF2_O__#Q\`QP``````QT00_```
M``#I?/[__Y"0D)"0D)"0D)"0D)#_);+.``"0D/\EHLX``)"0_R62S@``D)#_
M)8+.``"0D/\E<LX``)"0_R5BS@``D)#_)5+.``"0D/\E.LX``)"0_R4JS@``
MD)#_)1K.``"0D/\E"LX``)"0_R7ZS0``D)#_)>K-``"0D/\EVLT``)"0_R7*
MS0``D)#_);K-``"0D/\EJLT``)"0_R6:S0``D)#_)8K-``"0D/\E>LT``)"0
M_R72RP``D)"0D)"0D)"0D/\ENLP``)"0D)"0D)"0D)#_)3K-``"0D`\?A```
M````45!(/0`0``!(C4PD&'(92('I`!```$B#"0!(+0`0``!(/0`0``!WYT@I
MP4B#"0!86<.0D)"0D)"0D)"0D)"0D/\E4LL``)"0D)"0D)"0D)#_)4K+``"0
MD)"0D)"0D)"0_R5*RP``D)"0D)"0D)"0D/\E0LL``)"0D)"0D)"0D)#_)3K+
M``"0D)"0D)"0D)"0_R4RRP``D)"0D)"0D)"0D/\E0LL``)"0D)"0D)"0D)#_
M)4++``"0D)"0D)"0D)"0_R5:RP``D)"0D)"0D)"0D/\E4LL``)"0D)"0D)"0
MD)#_)4K+``"0D)"0D)"0D)"0_R5"RP``D)"0D)"0D)"0D/\E.LL``)"0D)"0
MD)"0D)#_)3++``"0D)"0D)"0D)"0_R4JRP``D)"0D)"0D)"0D/\E(LL``)"0
MD)"0D)"0D)#_)1K+``"0D)"0D)"0D)"0_R42RP``D)"0D)"0D)"0D/\E"LL`
M`)"0D)"0D)"0D)#_)0++``"0D)"0D)"0D)"0_R7ZR@``D)"0D)"0D)"0D/\E
M\LH``)"0D)"0D)"0D)#_)>K*``"0D)"0D)"0D)"0_R7RR@``D)"0D)"0D)"0
MD/\EZLH``)"0D)"0D)"0D)#_)>+*``"0D)"0D)"0D)"0_R7:R@``D)"0D)"0
MD)"0D/\ETLH``)"0D)"0D)"0D)#_)<K*``"0D)"0D)"0D)"0_R7"R@``D)"0
MD)"0D)"0D/\ENLH``)"0D)"0D)"0D)#_);+*``"0D)"0D)"0D)"0_R6JR@``
MD)"0D)"0D)"0D/\EHLH``)"0D)"0D)"0D)#_)9K*``"0D)"0D)"0D)"0_R62
MR@``D)"0D)"0D)"0D/\EBLH``)"0D)"0D)"0D)#_)8+*``"0D)"0D)"0D)"0
M_R5ZR@``D)"0D)"0D)"0D$B#["@QTNA5````_Q47R0``D)"0D)"0D)"0D)"0
MD)"0PY"0D)"0D)"0D)"0D)"0D,.0D)"0D)"0D)"0D)"0D)##D)"0D)"0D)"0
MD)"0D)"0PY"0D)"0D)"0D)"0D)"0D%932(/L*$B)SDB)T[D(````Z+H$``!(
MA=L/A!$"``!(N@````!9`0``2(ES0$B--8S___](C0U%!P``2(F3\````$B-
M%4\'``!(B5-02(L5?,@``$B)LX````!(C35N____2(E+2$B-#7/___](B5,8
M2(T56*4``$B)LX@```!(C35J____2(F+D````$BY2`$``+P+``!(B;.8````
M2(TUB_W__TB)4SA(C14@_?__2(E+"$B-#17^__](B7,@2(TUZ@,``$B)4RA(
MC14/!```2,<#`````,=#$`<```!(B4LP2(ES>$B)DQ@!``!(A<`/A+T```!(
MBX#@````2(7`#X2M````BQ73I```A=(/A9\```!(@SW3%`````^$#0(``$B#
M/<T4````#X1O`@``2(,]QQ0````/A%$"``!(@SW!%`````^$,P(``$B#/;L4
M````#X05`@``2(,]M10````/A/<!``!(@SVO%`````^$V0$``$B#/:D4````
M#X2[`0``9@]O!6,4```/$0!F#V\-:!0```\12!!F#V\5;!0```\14"!F#V\=
M<!0```\16#`QR>AU`P``,<E(B8/H````2(T%Y1,``$B)0UA(C05:%```2(E#
M8$B-!<^C``!(B4-H2(T%I*4``$B)0W!(C06)9@``2(F#(`$``$B-!;=F``!(
MB8,H`0``2(T%G;/__TB)@S`!``#H@0(``+@!````2(/$*%M>PP\?1```2(EP
M0$B--87]__](N@````!9`0``2(T-3`4``$B)L(````!(C35V_?__2(F0\```
M`$B-%1@%``!(B;"(````2(TU:OW__TB)4$A(BQ5/Q@``2(E(4$B+B$`!``!(
MB;"0````2(TU5OW__TB)L)@```!(C34HHP``2(D*2(T5?OO__TB-#1?[__](
MB7`X2+Y(`0``O`L``$B)<`A(C37^^___2(E0($B-%=,!``!(B4@H2(T-^`$`
M`$C'``````#'0!`'````2(EP,$B)4'A(B8@8`0``2(7`#X2F_O__2(G#2(N`
MX````$B%P`^%YOW__^F._O__9BX/'X0``````$B+$$B)%;82``#IY/W__Y!(
MBU`X2(D5W1(``.DU_O__2(M0,$B)%<42``#I%_[__TB+4"A(B16M$@``Z?G]
M__](BU`@2(D5E1(``.G;_?__2(M0&$B)%7T2``#IO?W__TB+4!!(B15E$@``
MZ9_]__](BU`(2(D531(``.F!_?__3(L%.:(``(L5.Z(``$B+#3RB``#IQP``
M``\?@`````!(@^PX@_H"='=W5872#X1]````,<!-A<!(C17!____2(D-"J(`
M``^4P$R)!?"A``#'!>ZA```!````B070H0``Z-L```!(@_C_2(D%R*$```^5
MP`^VP$B#Q#C##Q]$``"X`0```(/Z`W7LN@,```!(@\0XZ4@````/'X0`````
M`+H"````2(/$..DR````9I`QTN@I````A<!TNTB+#7:A``")1"0LZ$4```"+
M1"0L2,<%7J$``/_____KFI"0D)"X`0```,.0D)"0D)"0D)"0PY"0D)"0D)"0
MD)"0D)"0D/\E*L0``)"0D)"0D)"0D)#_)2K$``"0D)"0D)"0D)"0_R4BQ```
MD)"0D)"0D)"0D/\EHL0``)"0D)"0D)"0D)!!54%42(/L*$R-+1&A``!)B<Q(
MB=%,B>KH\_K__TR)ZDR)X4B#Q"A!7$%=Z0````#_)=K#``"0D)"0D)"0D)"0
M_R6RQ0``D)`/'X0``````$%505164TB#[%B)C"2`````2(E4)#A,B40D,.CP
M]___2(U4)#A(BQWD8```3(U$)#!(C8PD@````$B+`TB)1"1(,<#HV/;__TB+
M%:%@``!(BPV*8```3(T%$_?__^@N]___2(L%9V```$R+)9!@``"`.`!U-NB.
M]O__2(D%!Z```$B)P4B%P`^$A@$``.AN]O__3(LE9V```$R)X>C_]O__2(L`
MQH`/"@```$R)X>CM]O__2(L`@(AB!````XN$)(````"-2`2-<`-(8\E(P>$#
MZ&+V__],BU0D.$2+A"2`````28G!2(D%DY\``$F+`DF)`4B-!3\B``!)B4$(
M2(L%BP\``$F)01!(C04L(@``28E!&$&#^`$/CNL```!)@\((08/``[H$````
M#Q]$``!)BP)(8\J#P@%)@\((28D$R40YPG7J2&/22(L-.)\``$&)\$G'!-$`
M````2(T5;L;__TC'1"0@`````.AX]?__08G$A<!T3DB+#0J?``!,C6PD1.AP
M]?__3(GJ2(T->1\``.A9O___2(7`=#I(BPWEG@``Z$CU___H:_7__TB+1"1(
M2"L#=69$B>!(@\186UY!7$%=PTB+#;R>``#H#_7__T&)Q.NA2(L-FYX``$B%
MR70'Z'G9___KLTB-#<D>``!,B>KH^+[__TB)!7F>``!(B<%(A<!TE>O9N@0`
M``#I,____[D!````Z%/V___H'O;__Y"0D)"0D)"0D)"0D)"0Z8N^__^0D)"0
MD)"0D)"0D/__________P%%```$``````````````/__________````````
M```````````````````````````````````*4$%2+G!M"@``````````X')`
M``$`````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````)7,O)7,`+W!R;V,O<V5L9B]E
M>&4`1'EN84QO861E<CHZ8F]O=%]$>6YA3&]A9&5R``!);G1E<FYA;',Z.E!!
M4CHZ0D]/5``N`%!!4E]414U0`%!!4E]04D]'3D%-10`Z`"5S)7,E<P`O`%!%
M4DPU3$E"`%!%4DQ,24(`4$523#5/4%0`4$523$E/`%!!4E])3DE424%,25I%
M1`!005)?4U!!5TY%1`!005)?0TQ%04X`4$%27T1%0E5'`%!!4E]#04-(10!0
M05)?1TQ/0D%,7T1%0E5'`%!!4E]'3$]"04Q?5$U01$E2`%!!4E]435!$25(`
M4$%27T=,3T)!3%]414U0`%!!4E]'3$]"04Q?0TQ%04X`,0!0051(`%5315(`
M4UE35$5-`"]T;7``5$U01$E2`%1%35!$25(`5$5-4`!435``55-%4DY!344`
M)3`R>``E<R5S)7,E<P!P87(M`"5S.B!C<F5A=&EO;B!O9B!P<FEV871E('-U
M8F1I<F5C=&]R>2`E<R!F86EL960@*&5R<FYO/25I*0H````````E<SH@<')I
M=F%T92!S=6)D:7)E8W1O<GD@)7,@:7,@=6YS869E("AP;&5A<V4@<F5M;W9E
M(&ET(&%N9"!R971R>2!Y;W5R(&]P97)A=&EO;BD*`'!A<FPN97AE`"YP87(`
M`$-!0TA%`"5S)7-C86-H92TE<R5S`"5S)7-T96UP+25U)7,`)7,E<W1E;7`M
M)74M)74E<P`8`'!E<FP`,`!005(Z.E!A8VME<CHZ5D524TE/3@`Q+C`U.```
M``````!5;F%B;&4@=&\@8W)E871E(&-A8VAE(&1I<F5C=&]R>0``````````
M)7,Z(&-R96%T:6]N(&]F('!R:79A=&4@8V%C:&4@<W5B9&ER96-T;W)Y("5S
M(&9A:6QE9"`H97)R;F\])6DI"@`M90`M+0``<&5R;'AS:2YC`````````&5V
M86P@)V5X96,@+W5S<B]B:6XO<&5R;"`@+5,@)#`@)'LQ*R(D0")])PII9B`P
M.R`C(&YO="!R=6YN:6YG('5N9&5R('-O;64@<VAE;&P*<&%C:V%G92!?7W!A
M<E]P;#L*;7D@*"1005)?34%'24,L("1&24Q%7V]F9G-E=%]S:7IE+"`D8V%C
M:&5?;F%M95]S:7IE*3L*0D5'24X@>PHD4$%27TU!1TE#(#T@(EQN4$%2+G!M
M7&XB.PHD1DE,15]O9F9S971?<VEZ92`](#0[("`@(R!P86-K*").(BD*)&-A
M8VAE7VYA;65?<VEZ92`](#0P.PHD4$M:25!?34%'24,@/2`B4$M<,#`S7#`P
M-"(["GT*<W5B(&9I;F1?<&%R7VUA9VEC"GL*;7D@*"1F:"D@/2!`7SL*;7D@
M)&-H=6YK7W-I>F4@/2`V-"`J(#$P,C0["FUY("1B=68["FUY("1S:7IE(#T@
M+7,@)&9H.PIM>2`D<&]S(#T@*"1S:7IE+3$I("T@*"1S:7IE+3$I("4@)&-H
M=6YK7W-I>F4[("`@("`@(R!.3U1%.B`D<&]S(&ES(&$@;75L=&EP;&4@;V8@
M)&-H=6YK7W-I>F4*=VAI;&4@*"1P;W,@/CT@,"D@>PIS965K("1F:"P@)'!O
M<RP@,#L*<F5A9"`D9F@L("1B=68L("1C:'5N:U]S:7IE("L@;&5N9W1H*"10
M05)?34%'24,I.PII9B`H*&UY("1I(#T@<FEN9&5X*"1B=68L("1005)?34%'
M24,I*2`^/2`P*2!["G)E='5R;B`D<&]S("L@)&D["GT*)'!O<R`M/2`D8VAU
M;FM?<VEZ93L*?0IR971U<FX@+3$["GT*;7D@*"1P87)?=&5M<"P@)'!R;V=N
M86UE+"!`=&UP9FEL92P@)4UO9'5L94-A8VAE*3L*14Y$('L@:68@*"1%3E9[
M4$%27T-,14%.?2D@>PIR97%U:7)E($9I;&4Z.E1E;7`["G)E<75I<F4@1FEL
M93HZ0F%S96YA;64["G)E<75I<F4@1FEL93HZ4W!E8SL*;7D@)'1O<&1I<B`]
M($9I;&4Z.D)A<V5N86UE.CID:7)N86UE*"1P87)?=&5M<"D["F]U=',H<7%;
M4F5M;W9I;F<@9FEL97,@:6X@(B1P87)?=&5M<")=*3L*1FEL93HZ1FEN9#HZ
M9FEN9&1E<'1H*'-U8B![("@@+60@*2`_(')M9&ER(#H@=6YL:6YK('TL("1P
M87)?=&5M<"D["G)M9&ER("1P87)?=&5M<#L*:68@*"UD("1P87)?=&5M<"`F
M)B`D7D\@;F4@)TU35VEN,S(G*2!["FUY("1T;7`@/2!N97<@1FEL93HZ5&5M
M<"@*5$5-4$Q!5$4@/3X@)W1M<%A86%A8)RP*1$E2(#T^($9I;&4Z.D)A<V5N
M86UE.CID:7)N86UE*"1T;W!D:7(I+`I3549&25@@/3X@)RYC;60G+`I53DQ)
M3DL@/3X@,"P**3L*;7D@)&9I;&5N86UE(#T@)'1M<"T^9FEL96YA;64["G!R
M:6YT("1T;7`@/#PB+BXN(CL*>#TQ.R!W:&EL92!;(%PD>"`M;'0@,3`@73L@
M9&\*<FT@+7)F("<D<&%R7W1E;7`G"FEF(%L@7"$@+60@)R1P87)?=&5M<"<@
M73L@=&AE;@IB<F5A:PIF:0IS;&5E<"`Q"G@]8&5X<'(@7"1X("L@,6`*9&]N
M90IR;2`G)&9I;&5N86UE)PHN+BX*8VQO<V4@)'1M<#L*8VAM;V0@,#<P,"P@
M)&9I;&5N86UE.PIM>2`D8VUD(#T@(B1F:6QE;F%M92`^+V1E=B]N=6QL(#(^
M)C$@)B(["G-Y<W1E;2@D8VUD*3L*;W5T<RAQ<5M3<&%W;F5D(&)A8VMG<F]U
M;F0@<')O8V5S<R!T;R!P97)F;W)M(&-L96%N=7`Z("1F:6QE;F%M95TI.PI]
M"GT@?0I"14=)3B!["DEN=&5R;F%L<SHZ4$%2.CI"3T]4*"D@:68@9&5F:6YE
M9"`F26YT97)N86QS.CI005(Z.D)/3U0["F5V86P@>PI?<&%R7VEN:71?96YV
M*"D["FUY("1Q=6EE="`]("$D14Y6>U!!4E]$14)51WT["FUY("5S>7,@/2`H
M"G!A=&A?<V5P("`@(#T^("@D7D\@/7X@+UY-4U=I;B\@/R`G.R<@.B`G.B<I
M+`I?97AE("`@("`@("`]/B`H)%Y/(#U^("]>*#\Z35-7:6Y\3U,R?&-Y9W=I
M;BDO(#\@)RYE>&4G(#H@)R<I+`I?9&5L:6T@("`@("`]/B`H)%Y/(#U^("]>
M35-7:6Y\3U,R+R`_("=<7"<@.B`G+R<I+`HI.PI?<V5T7W!R;V=N86UE*"D[
M"F]U=',H<7%;7"1P<F]G;F%M92`]("(D<')O9VYA;64B72D["E]S971?<&%R
M7W1E;7`H*3L*;W5T<RAQ<5M<)$5.5GM005)?5$5-4'T@/2`B)$5.5GM005)?
M5$5-4'TB72D["FUY("@D<W1A<G1?<&]S+"`D<W1A<G1?;V9?1DE,15]S96-T
M:6]N*3L*34%'24,Z('L*;&]C86P@)%-)1WM?7U=!4DY?7WT@/2!S=6(@>WT[
M"G5N;&5S<R`H;W!E;B!?1D@L("<\.G)A=R<L("1P<F]G;F%M92D@>PIO=71S
M*'%Q6T-A;B=T(')E860@9G)O;2!F:6QE("(D<')O9VYA;64B72D[("`C(&1O
M;B=T('5S92`D(2!H97)E(&%S(&ET(')E<75I<F5S($5R<FYO+G!M"FQA<W0@
M34%'24,["GT*;7D@)&UA9VEC7W!O<R`](&9I;F1?<&%R7VUA9VEC*"I?1D@I
M.PII9B`H)&UA9VEC7W!O<R`\(#`I('L*;W5T<RAQ<5M#86XG="!F:6YD(&UA
M9VEC('-T<FEN9R`B)%!!4E]-04=)0R(@:6X@9FEL92`B)'!R;V=N86UE(ETI
M.PIL87-T($U!1TE#.PI]"F]U=',H(D9O=6YD(%!!4B!M86=I8R!A="!P;W-I
M=&EO;B`D;6%G:6-?<&]S(BD["G-E96L@7T9(+"`D;6%G:6-?<&]S("T@)$9)
M3$5?;V9F<V5T7W-I>F4@+2!L96YG=&@H(EPP0T%#2$4B*2P@,#L*<F5A9"!?
M1D@L("1B=68L(&QE;F=T:"@B7#!#04-(12(I.PII9B`H)&)U9B!N92`B7#!#
M04-(12(@)B8@)&)U9B!N92`B7#!#3$5!3B(I('L*;W5T<R@B3F\@8V%C:&4@
M;6%R:V5R(&9O=6YD(BD["FQA<W0@34%'24,["GT*;W5T<RAQ<5M#86-H92!M
M87)K97(@(B1B=68B(&9O=6YD72D["G-E96L@7T9(+"`D;6%G:6-?<&]S("T@
M)$9)3$5?;V9F<V5T7W-I>F4L(#`["G)E860@7T9(+"`D8G5F+"`D1DE,15]O
M9F9S971?<VEZ93L*;7D@)&]F9G-E="`]('5N<&%C:R@B3B(L("1B=68I.PIO
M=71S*")/9F9S970@9G)O;2!S=&%R="!O9B!&24Q%<R!I<R`D;V9F<V5T(BD[
M"G-E96L@7T9(+"`D;6%G:6-?<&]S("T@)$9)3$5?;V9F<V5T7W-I>F4@+2`D
M;V9F<V5T+"`P.PHD<W1A<G1?;V9?1DE,15]S96-T:6]N(#T@=&5L;"!?1D@[
M"FUY("5R97%U:7)E7VQI<W0["G)E860@7T9(+"`D8G5F+"`T.R`@("`@("`@
M("`@("`@("`@("`@("`@("`@(",@<F5A9"!T:&4@9FER<W0@(D9)3$4B"G=H
M:6QE("@D8G5F(&5Q(")&24Q%(BD@>PIR96%D(%]&2"P@)&)U9BP@-#L*<F5A
M9"!?1D@L("1B=68L('5N<&%C:R@B3B(L("1B=68I.PIM>2`D9G5L;&YA;64@
M/2`D8G5F.PIO=71S*'%Q6U5N<&%C:VEN9R!&24Q%("(D9G5L;&YA;64B+BXN
M72D["FUY("1C<F,@/2`H("1F=6QL;F%M92`]?B!S?%XH6V$M9EQD77LX?2DO
M?'P@*2`_("0Q(#H@=6YD968["FUY("@D8F%S96YA;64L("1E>'0I(#T@*"1B
M=68@/7X@;7PH/SHN*B\I/R@N*BDH7"XN*BE\*3L*<F5A9"!?1D@L("1B=68L
M(#0["G)E860@7T9(+"`D8G5F+"!U;G!A8VLH(DXB+"`D8G5F*3L*:68@*&1E
M9FEN960H)&5X="D@86YD("1E>'0@(7X@+UPN*#\Z<&U\<&Q\:7A\86PI)"]I
M*2!["FUY("1F:6QE;F%M92`](%]S879E7V%S*"(D8W)C)&5X="(L("1B=68L
M(#`W-34I.PHD4$%2.CI(96%V>3HZ1G5L;$-A8VAE>R1F=6QL;F%M97T@/2`D
M9FEL96YA;64["B1005(Z.DAE879Y.CI&=6QL0V%C:&5[)&9I;&5N86UE?2`]
M("1F=6QL;F%M93L*?0IE;'-E('L*)$UO9'5L94-A8VAE>R1F=6QL;F%M97T@
M/2!["F)U9B`]/B`D8G5F+`IC<F,@/3X@)&-R8RP*;F%M92`]/B`D9G5L;&YA
M;64L"GT["B1R97%U:7)E7VQI<W1[)&9U;&QN86UE?2LK.PI]"G)E860@7T9(
M+"`D8G5F+"`T.PI]"FQO8V%L($!)3D,@/2`H<W5B('L*;7D@*"1S96QF+"`D
M;6]D=6QE*2`]($!?.PIR971U<FX@:68@<F5F("1M;V1U;&4@;W(@(21M;V1U
M;&4["FUY("1I;F9O(#T@)$UO9'5L94-A8VAE>R1M;V1U;&5].PIR971U<FX@
M=6YL97-S("1I;F9O.PID96QE=&4@)')E<75I<F5?;&ES='LD;6]D=6QE?3L*
M)$E.0WLD;6]D=6QE?2`]("(O;&]A9&5R+R1I;F9O+R1M;V1U;&4B.PII9B`H
M)$5.5GM005)?0TQ%04Y](&%N9"!D969I;F5D*"9)3SHZ1FEL93HZ;F5W*2D@
M>PIM>2`D9F@@/2!)3SHZ1FEL92T^;F5W7W1M<&9I;&4@;W(@9&EE(")#86XG
M="!C<F5A=&4@=&5M<"!F:6QE.B`D(2(["B1F:"T^8FEN;6]D92@I.PHD9F@M
M/G!R:6YT*"1I;F9O+3Y[8G5F?2D["B1F:"T^<V5E:R@P+"`P*3L*<F5T=7)N
M("1F:#L*?0IE;'-E('L*;7D@)&9I;&5N86UE(#T@7W-A=F5?87,H(B1I;F9O
M+3Y[8W)C?2YP;2(L("1I;F9O+3Y[8G5F?2D["B1I;F9O+3Y[9FEL97T@/2`D
M9FEL96YA;64["F]P96X@;7D@)&9H+"`G/#IR87<G+"`D9FEL96YA;64@;W(@
M9&EE('%Q6T-A;B=T(')E860@(B1F:6QE;F%M92(Z("0A73L*<F5T=7)N("1F
M:#L*?0ID:64@(D)O;W1S=')A<'!I;F<@9F%I;&5D.B!C86XG="!F:6YD(&UO
M9'5L92`D;6]D=6QE(2(["GTL($!)3D,I.PIR97%U:7)E(%A33&]A9&5R.PIR
M97%U:7)E(%!!4CHZ2&5A=GD["G)E<75I<F4@0V%R<#HZ2&5A=GD["G)E<75I
M<F4@17AP;W)T97(Z.DAE879Y.PI005(Z.DAE879Y.CI?:6YI=%]D>6YA;&]A
M9&5R*"D["G)E<75I<F4@24\Z.D9I;&4["G=H:6QE("AM>2`D9FEL96YA;64@
M/2`H<V]R="!K97ES("5R97%U:7)E7VQI<W0I6S!=*2!["G5N;&5S<R`H)$E.
M0WLD9FEL96YA;65](&]R("1F:6QE;F%M92`]?B`O0E-$4$%.+RD@>PII9B`H
M)&9I;&5N86UE(#U^("]<+G!M8S\D+VDI('L*<F5Q=6ER92`D9FEL96YA;64[
M"GT*96QS92!["F1O("1F:6QE;F%M92!U;FQE<W,@)&9I;&5N86UE(#U^("]S
M:71E8W5S=&]M:7IE7"YP;"0O.PI]"GT*9&5L971E("1R97%U:7)E7VQI<W1[
M)&9I;&5N86UE?3L*?0II9B`H)&)U9B!N92`D4$M:25!?34%'24,I('L*;W5T
M<RAQ<5M.;R!Z:7`@9F]U;F0@869T97(@1DE,12!S96-T:6]N(&EN(&9I;&4@
M(B1P<F]G;F%M92)=*3L*;&%S="!-04=)0R`["GT*)'-T87)T7W!O<R`]("AT
M96QL(%]&2"D@+2`T.R`@("`@("`@("`@("`@("`C('-T87)T(&]F('II<`I]
M"FUY($!P87)?87)G<SL*;7D@*"1O=70L("1B=6YD;&4L("1L;V=F:"P@)&-A
M8VAE7VYA;64I.PID96QE=&4@)$5.5GM005)?05!07U)%55-%?3L@(R!S86YI
M=&EZ92`H4D554T4@;6%Y(&)E(&$@<V5C=7)I='D@<')O8FQE;2D*:68@*"$D
M<W1A<G1?<&]S(&]R("@D05)'5ELP72!E<2`G+2UP87(M;W!T:6]N<R<@)B8@
M<VAI9G0I*2!["FUY("5D:7-T7V-M9"`]('%W*`IP("`@8FQI8E]T;U]P87(*
M:2`@(&EN<W1A;&Q?<&%R"G4@("!U;FEN<W1A;&Q?<&%R"G,@("!S:6=N7W!A
M<@IV("`@=F5R:69Y7W!A<@HI.PII9B`H0$%21U8@86YD("1!4D=66S!=(&5Q
M("<M+7)E=7-E)RD@>PIS:&EF="!`05)'5CL*)$5.5GM005)?05!07U)%55-%
M?2`]('-H:69T($!!4D=6.PI]"F5L<V4@>R`C(&YO<FUA;"!P87)L(&)E:&%V
M:6]U<@IM>2!`861D7W1O7VEN8SL*=VAI;&4@*$!!4D=6*2!["B1!4D=66S!=
M(#U^("]>+2A;04E-3T),8G%P:75S5'9=*2@N*BDO(&]R(&QA<W0["FEF("@D
M,2!E<2`G22<I('L*<'5S:"!`861D7W1O7VEN8RP@)#(["GT*96QS:68@*"0Q
M(&5Q("=-)RD@>PIE=F%L(")U<V4@)#(B.PI]"F5L<VEF("@D,2!E<2`G02<I
M('L*=6YS:&EF="!`<&%R7V%R9W,L("0R.PI]"F5L<VEF("@D,2!E<2`G3R<I
M('L*)&]U="`]("0R.PI]"F5L<VEF("@D,2!E<2`G8B<I('L*)&)U;F1L92`]
M("=S:71E)SL*?0IE;'-I9B`H)#$@97$@)T(G*2!["B1B=6YD;&4@/2`G86QL
M)SL*?0IE;'-I9B`H)#$@97$@)W$G*2!["B1Q=6EE="`](#$["GT*96QS:68@
M*"0Q(&5Q("=,)RD@>PIO<&5N("1L;V=F:"P@(CX^(BP@)#(@;W(@9&EE('%Q
M6T-A;B=T(&]P96X@;&]G(&9I;&4@(B0R(CH@)"%=.PI]"F5L<VEF("@D,2!E
M<2`G5"<I('L*)&-A8VAE7VYA;64@/2`D,CL*?0IS:&EF="A`05)'5BD["FEF
M("AM>2`D8VUD(#T@)&1I<W1?8VUD>R0Q?2D@>PID96QE=&4@)$5.5GLG4$%2
M7U1%35`G?3L*:6YI=%]I;F,H*3L*<F5Q=6ER92!005(Z.D1I<W0["B9[(E!!
M4CHZ1&ES=#HZ)&-M9")]*"D@=6YL97-S($!!4D=6.PHF>R)005(Z.D1I<W0Z
M.B1C;60B?2@D7RD@9F]R($!!4D=6.PIE>&ET.PI]"GT*=6YS:&EF="!`24Y#
M+"!`861D7W1O7VEN8SL*?0I]"FEF("@D;W5T*2!["GL*<F5Q=6ER92!)3SHZ
M1FEL93L*<F5Q=6ER92!!<F-H:79E.CI::7`["G)E<75I<F4@1&EG97-T.CI3
M2$$["GT*;7D@)'!A<B`]('-H:69T*$!!4D=6*3L*;7D@)'II<#L*:68@*&1E
M9FEN960@)'!A<BD@>PIO<&5N(&UY("1F:"P@)SPZ<F%W)RP@)'!A<B!O<B!D
M:64@<7%;0V%N)W0@9FEN9"!P87(@9FEL92`B)'!A<B(Z("0A73L*8FQE<W,H
M)&9H+"`G24\Z.D9I;&4G*3L*)'II<"`]($%R8VAI=F4Z.EII<"T^;F5W.PHH
M("1Z:7`M/G)E861&<F]M1FEL94AA;F1L92@D9F@L("1P87(I(#T]($%R8VAI
M=F4Z.EII<#HZ05I?3TLH*2`I"F]R(&1I92!Q<5M%<G)O<B!R96%D:6YG('II
M<"!A<F-H:79E("(D<&%R(ET["GT*;7D@)6UE=&%?<&%R(#T@9&\@>PII9B`H
M)'II<"!A;F0@;7D@)&UE=&$@/2`D>FEP+3YC;VYT96YT<R@G345402YY;6PG
M*2D@>PHD;65T82`]?B!S+RXJ7G!A<CHD+R]M<SL*)&UE=&$@/7X@<R]>7%,N
M*B\O;7,["B1M971A(#U^("]>("`H6UXZ72LI.B`H+BLI)"]M9SL*?0I].PII
M9B`H9&5F:6YE9"`D<&%R*2!["F]P96X@;7D@)'!H+"`G/#IR87<G+"`D<&%R
M(&]R(&1I92!Q<5M#86XG="!R96%D('!A<B!F:6QE("(D<&%R(CH@)"%=.PIM
M>2`D8G5F.PIR96%D("1P:"P@)&)U9BP@-#L*9&EE('%Q6R(D<&%R(B!I<R!N
M;W0@82!P87(@9FEL95T@=6YL97-S("1B=68@97$@)%!+6DE07TU!1TE#.PIC
M;&]S92`D<&@["GT*0W)E871E4&%T:"@D;W5T*2`["FUY("1F:"`]($E/.CI&
M:6QE+3YN97<H"B1O=70L"DE/.CI&:6QE.CI/7T-214%4*"D@?"!)3SHZ1FEL
M93HZ3U]21%=2*"D@?"!)3SHZ1FEL93HZ3U]44E5.0R@I+`HP-S<W+`HI(&]R
M(&1I92!Q<5M#86XG="!C<F5A=&4@9FEL92`B)&]U="(Z("0A73L*)&9H+3YB
M:6YM;V1E*"D["G-E96L@7T9(+"`P+"`P.PIM>2`D;&]A9&5R.PII9B`H9&5F
M:6YE9"`D<W1A<G1?;V9?1DE,15]S96-T:6]N*2!["G)E860@7T9(+"`D;&]A
M9&5R+"`D<W1A<G1?;V9?1DE,15]S96-T:6]N.PI](&5L<V4@>PIL;V-A;"`D
M+R`]('5N9&5F.PHD;&]A9&5R(#T@/%]&2#X["GT*:68@*"$D14Y6>U!!4E]6
M15)"051)37T@86YD("1L;V%D97(@/7X@+UXH/SHC(7Q<0')E;2DO*2!["G)E
M<75I<F4@4$%2.CI&:6QT97(Z.E!O9%-T<FEP.PI005(Z.D9I;'1E<CHZ4&]D
M4W1R:7`M/F%P<&QY*%PD;&]A9&5R+"`D,"D["GT*)&9H+3YP<FEN="@D;&]A
M9&5R*0IO<B!D:64@<7%;17)R;W(@=W)I=&EN9R!L;V%D97(@=&\@(B1O=70B
M.B`D(5T["FEF("@D8G5N9&QE*2!["G)E<75I<F4@4$%2.CI(96%V>3L*4$%2
M.CI(96%V>3HZ7VEN:71?9'EN86QO861E<B@I.PII;FET7VEN8R@I.PIR97%U
M:7)E7VUO9'5L97,H*3L*;7D@0&EN8R`](&=R97`@>R`A+T)31%!!3B\@?0IG
M<F5P('L**"1B=6YD;&4@;F4@)W-I=&4G*2!O<@HH)%\@;F4@)$-O;F9I9SHZ
M0V]N9FEG>V%R8VAL:6)E>'!](&%N9`HD7R!N92`D0V]N9FEG.CI#;VYF:6=[
M<')I=FQI8F5X<'TI.PI]($!)3D,["G-\+RM\+WQG+"!S?"\D?'P@9F]R96%C
M:"!`:6YC.PIM>2`E9FEL97,["B1F:6QE<WLD7WTK*R!F;W(@0$1Y;F%,;V%D
M97(Z.F1L7W-H87)E9%]O8FIE8W1S+"!V86QU97,@)4E.0SL*;7D@)&QI8E]E
M>'0@/2`D0V]N9FEG.CI#;VYF:6=[;&EB7V5X='T[("`@("`@("`@(R!86%@@
M;&EB7V5X="!V<R!D;&5X="`_"FUY("5W<FET=&5N.PIF;W)E86-H(&UY("1K
M97D@*'-O<G0@:V5Y<R`E9FEL97,I('L*;7D@*"1F:6QE+"`D;F%M92D["FEF
M("AD969I;F5D*&UY("1F8R`]("1005(Z.DAE879Y.CI&=6QL0V%C:&5[)&ME
M>7TI*2!["B@D9FEL92P@)&YA;64I(#T@*"1K97DL("1F8RD["GT*96QS92![
M"F9O<F5A8V@@;7D@)&1I<B`H0&EN8RD@>PII9B`H)&ME>2`]?B!M?%Y<421D
M:7)<12\H+BHI)'QI*2!["B@D9FEL92P@)&YA;64I(#T@*"1K97DL("0Q*3L*
M;&%S=#L*?0II9B`H)&ME>2`]?B!M?%XO;&]A9&5R+UM>+UTK+R@N*BDD?"D@
M>PII9B`H;7D@)')E9B`]("1-;V1U;&5#86-H97LD,7TI('L**"1F:6QE+"`D
M;F%M92D@/2`H)')E9BP@)#$I.PIL87-T.PI]"FEF("@M9B`B)&1I<B\D,2(I
M('L**"1F:6QE+"`D;F%M92D@/2`H(B1D:7(O)#$B+"`D,2D["FQA<W0["GT*
M?0I]"GT*;F5X="!U;FQE<W,@9&5F:6YE9"`D;F%M93L*;F5X="!I9B`D=W)I
M='1E;GLD;F%M97TK*SL*;F5X="!I9B`A<F5F*"1F:6QE*2!A;F0@)&9I;&4@
M/7X@+UPN7%$D;&EB7V5X=%Q%)"]I.PIO=71S*'-P<FEN=&8H<7%;4&%C:VEN
M9R!&24Q%("(E<R(N+BY=+"!R968@)&9I;&4@/R`D9FEL92T^>VYA;65](#H@
M)&9I;&4I*3L*;7D@)&-O;G1E;G0["FEF("AR968H)&9I;&4I*2!["B1C;VYT
M96YT(#T@)&9I;&4M/GMB=69].PI]"F5L<V4@>PIL;V-A;"`D+R`]('5N9&5F
M.PIO<&5N(&UY("1T:"P@)SPZ<F%W)RP@)&9I;&4@;W(@9&EE('%Q6T-A;B=T
M(')E860@(B1F:6QE(CH@)"%=.PHD8V]N=&5N="`](#PD=&@^.PIC;&]S92`D
M=&@["E!!4CHZ1FEL=&5R.CI0;V13=')I<"T^87!P;'DH7"1C;VYT96YT+"`B
M/&5M8F5D9&5D/B\D;F%M92(I"FEF("$D14Y6>U!!4E]615)"051)37T@86YD
M("1N86UE(#U^("]<+B@_.G!M?&EX?&%L*20O:3L*4$%2.CI&:6QT97(Z.E!A
M=&-H0V]N=&5N="T^;F5W+3YA<'!L>2A<)&-O;G1E;G0L("1F:6QE+"`D;F%M
M92D["GT*)&9H+3YP<FEN="@B1DE,12(L"G!A8VLH)TXG+"!L96YG=&@H)&YA
M;64I("L@.2DL"G-P<FEN=&8H(B4P.'@O)7,B+"!!<F-H:79E.CI::7`Z.F-O
M;7!U=&5#4D,S,B@D8V]N=&5N="DL("1N86UE*2P*<&%C:R@G3B<L(&QE;F=T
M:"@D8V]N=&5N="DI+`HD8V]N=&5N="D*;W(@9&EE('%Q6T5R<F]R('=R:71I
M;F<@96UB961D960@1DE,12!T;R`B)&]U="(Z("0A73L*;W5T<RAQ<5M7<FET
M=&5N(&%S("(D;F%M92)=*3L*?0I]"FEF("@D>FEP*2!["B1Z:7`M/G=R:71E
M5&]&:6QE2&%N9&QE*"1F:"D@/3T@07)C:&EV93HZ6FEP.CI!6E]/2PIO<B!D
M:64@<7%;17)R;W(@=W)I=&EN9R!Z:7`@<&%R="!O9B`B)&]U=")=.PI]"FEF
M("@D;65T85]P87)[8VQE86Y]*2!["B1F:"T^<')I;G0H(EPP0TQ%04XB*3L*
M?0IE;'-E('L*:68@*"%D969I;F5D("1C86-H95]N86UE*2!["FUY("1C='@@
M/2!$:6=E<W0Z.E-(02T^;F5W*#$I.PIS965K("1F:"P@,"P@,#L*)&-T>"T^
M861D9FEL92@D9F@I.PIS965K("1F:"P@,"P@,CL*)&-A8VAE7VYA;64@/2`D
M8W1X+3YH97AD:6=E<W0["GT*)&-A8VAE7VYA;64@/2!S=6)S='(@)&-A8VAE
M7VYA;64L(#`L("1C86-H95]N86UE7W-I>F4["B1C86-H95]N86UE("X](")<
M,"(@>"`H)&-A8VAE7VYA;65?<VEZ92`M(&QE;F=T:"`D8V%C:&5?;F%M92D[
M"B1F:"T^<')I;G0H)&-A8VAE7VYA;64L"B)<,$-!0TA%(BD["GT*;7D@)&]F
M9G-E="`]("1F:"T^=&5L;"`M(&QE;F=T:"@D;&]A9&5R*3L*)&9H+3YP<FEN
M="AP86-K*"=.)RP@)&]F9G-E="DL"B1005)?34%'24,I.PHD9F@M/F-L;W-E
M(&]R(&1I92!Q<5M%<G)O<B!W<FET:6YG('1R86EL97(@;V8@(B1O=70B.B`D
M(5T["F-H;6]D(#`W-34L("1O=70["F5X:70["GT*>PIL87-T('5N;&5S<R!D
M969I;F5D("1S=&%R=%]P;W,["E]F:7A?<')O9VYA;64H*3L*<F5Q=6ER92!0
M05(["E!!4CHZ2&5A=GDZ.E]I;FET7V1Y;F%L;V%D97(H*3L*>PIR97%U:7)E
M($9I;&4Z.D9I;F0["G)E<75I<F4@07)C:&EV93HZ6FEP.PI]"FUY("1F:"`]
M($E/.CI&:6QE+3YN97<[("`@("`@("`@("`@("`@("`@("`@("`@("`@("`C
M($%R8VAI=F4Z.EII<"!O<&5R871E<R!O;B!A;B!)3SHZ2&%N9&QE"B1F:"T^
M9F1O<&5N*&9I;&5N;RA?1D@I+"`G<B<I(&]R(&1I92!Q<5MF9&]P96XH*2!F
M86EL960Z("0A73L*07)C:&EV93HZ6FEP.CIS971#:'5N:U-I>F4H+7,@7T9(
M*3L*;7D@)'II<"`]($%R8VAI=F4Z.EII<"T^;F5W.PHH)'II<"T^<F5A9$9R
M;VU&:6QE2&%N9&QE*"1F:"P@)'!R;V=N86UE*2`]/2!!<F-H:79E.CI::7`Z
M.D%:7T]+*"DI"F]R(&1I92!Q<5M%<G)O<B!R96%D:6YG('II<"!A<F-H:79E
M("(D<')O9VYA;64B73L*07)C:&EV93HZ6FEP.CIS971#:'5N:U-I>F4H-C0@
M*B`Q,#(T*3L*<'5S:"!`4$%2.CI,:6)#86-H92P@)'II<#L*)%!!4CHZ3&EB
M0V%C:&5[)'!R;V=N86UE?2`]("1Z:7`["F]U=',H(D5X=')A8W1I;F<@>FEP
M+BXN(BD["FEF("AD969I;F5D("1%3E9[4$%27U1%35!]*2![(",@<VAO=6QD
M(&)E('-E="!A="!T:&ES('!O:6YT(0IF;W)E86-H(&UY("1M96UB97(@*"`D
M>FEP+3YM96UB97)S("D@>PIN97AT(&EF("1M96UB97(M/FES1&ER96-T;W)Y
M.PIM>2`D;65M8F5R7VYA;64@/2`D;65M8F5R+3YF:6QE3F%M93L*;F5X="!U
M;FQE<W,@)&UE;6)E<E]N86UE(#U^(&U["EX*+S]S:&QI8B\**#\Z)$-O;F9I
M9SHZ0V]N9FEG>V%R8VAN86UE?2\I/PHH6UXO72LI"B0*?7@["FUY("1E>'1R
M86-T7VYA;64@/2`D,3L*;7D@)&1E<W1?;F%M92`]($9I;&4Z.E-P96,M/F-A
M=&9I;&4H)$5.5GM005)?5$5-4'TL("1E>'1R86-T7VYA;64I.PII9B`H+68@
M)&1E<W1?;F%M92`F)B`M<R!?(#T]("1M96UB97(M/G5N8V]M<')E<W-E9%-I
M>F4H*2D@>PIO=71S*'%Q6U-K:7!P:6YG("(D;65M8F5R7VYA;64B('-I;F-E
M(&ET(&%L<F5A9'D@97AI<W1S(&%T("(D9&5S=%]N86UE(ETI.PI](&5L<V4@
M>PIO=71S*'%Q6T5X=')A8W1I;F<@(B1M96UB97)?;F%M92(@=&\@(B1D97-T
M7VYA;64B+BXN72D["B@D;65M8F5R+3YE>'1R86-T5&]&:6QE3F%M960H)&1E
M<W1?;F%M92D@/3T@07)C:&EV93HZ6FEP.CI!6E]/2R@I*0IO<B!D:64@<7%;
M17)R;W(@97AT<F%C=&EN9R!Z:7`@;65M8F5R('1O("(D9&5S=%]N86UE(ET[
M"F-H;6]D*#`U-34L("1D97-T7VYA;64I(&EF("1>3R!E<2`B:'!U>"(["GT*
M?0I]"GT*=6YL97-S("@D4$%2.CI,:6)#86-H97LD<')O9VYA;65]*2!["F1I
M92`\/"`B+B(@=6YL97-S($!!4D=6.PI5<V%G93H@)#`@6R`M06QI8BYP87(@
M72!;("U)9&ER(%T@6R`M36UO9'5L92!=(%L@<W)C+G!A<B!=(%L@<')O9W)A
M;2YP;"!="B0P(%L@+4)\+6(@72!;+4]O=71F:6QE72!S<F,N<&%R"BX*)$5.
M5GM005)?4%)/1TY!345](#T@)'!R;V=N86UE(#T@)#`@/2!S:&EF="A`05)'
M5BD["GT*<W5B($-R96%T95!A=&@@>PIM>2`H)&YA;64I(#T@0%\["G)E<75I
M<F4@1FEL93HZ0F%S96YA;64["FUY("@D8F%S96YA;64L("1P871H+"`D97AT
M*2`]($9I;&4Z.D)A<V5N86UE.CIF:6QE<&%R<V4H)&YA;64L("@G7"XN*B<I
M*3L*<F5Q=6ER92!&:6QE.CI0871H.PI&:6QE.CI0871H.CIM:W!A=&@H)'!A
M=&@I('5N;&5S<R@M92`D<&%T:"D[(",@;6MP871H(&1I97,@=VET:"!E<G)O
M<@I]"G-U8B!R97%U:7)E7VUO9'5L97,@>PIR97%U:7)E(&QI8CL*<F5Q=6ER
M92!$>6YA3&]A9&5R.PIR97%U:7)E(&EN=&5G97(["G)E<75I<F4@<W1R:6-T
M.PIR97%U:7)E('=A<FYI;F=S.PIR97%U:7)E('9A<G,["G)E<75I<F4@0V%R
M<#L*<F5Q=6ER92!#87)P.CI(96%V>3L*<F5Q=6ER92!%<G)N;SL*<F5Q=6ER
M92!%>'!O<G1E<CHZ2&5A=GD["G)E<75I<F4@17AP;W)T97(["G)E<75I<F4@
M1F-N=&P["G)E<75I<F4@1FEL93HZ5&5M<#L*<F5Q=6ER92!&:6QE.CI3<&5C
M.PIR97%U:7)E(%A33&]A9&5R.PIR97%U:7)E($-O;F9I9SL*<F5Q=6ER92!)
M3SHZ2&%N9&QE.PIR97%U:7)E($E/.CI&:6QE.PIR97%U:7)E($-O;7!R97-S
M.CI:;&EB.PIR97%U:7)E($%R8VAI=F4Z.EII<#L*<F5Q=6ER92!$:6=E<W0Z
M.E-(03L*<F5Q=6ER92!005(["G)E<75I<F4@4$%2.CI(96%V>3L*<F5Q=6ER
M92!005(Z.D1I<W0["G)E<75I<F4@4$%2.CI&:6QT97(Z.E!O9%-T<FEP.PIR
M97%U:7)E(%!!4CHZ1FEL=&5R.CI0871C:$-O;G1E;G0["G)E<75I<F4@871T
M<FEB=71E<SL*979A;"![(')E<75I<F4@0W=D('T["F5V86P@>R!R97%U:7)E
M(%=I;C,R('T["F5V86P@>R!R97%U:7)E(%-C86QA<CHZ571I;"!].PIE=F%L
M('L@<F5Q=6ER92!!<F-H:79E.CI5;GII<#HZ0G5R<W0@?3L*979A;"![(')E
M<75I<F4@5&EE.CI(87-H.CI.86UE9$-A<'1U<F4@?3L*979A;"![(')E<75I
M<F4@4&5R;$E/.R!R97%U:7)E(%!E<FQ)3SHZ<V-A;&%R('T["F5V86P@>R!R
M97%U:7)E('5T9C@@?3L*?0IS=6(@7W-E=%]P87)?=&5M<"!["FEF("AD969I
M;F5D("1%3E9[4$%27U1%35!](&%N9"`D14Y6>U!!4E]414U0?2`]?B`O*"XK
M*2\I('L*)'!A<E]T96UP(#T@)#$["G)E='5R;CL*?0IF;W)E86-H(&UY("1P
M871H("@**&UA<"`D14Y6>R1??2P@<7<H(%!!4E]435!$25(@5$U01$E2(%1%
M35!$25(@5$5-4"!435`@*2DL"G%W*"!#.EQ<5$5-4"`O=&UP("X@*0HI('L*
M;F5X="!U;FQE<W,@9&5F:6YE9"`D<&%T:"!A;F0@+60@)'!A=&@@86YD("UW
M("1P871H.PIM>2`D=7-E<FYA;64["FUY("1P=W5I9#L*979A;"![*"1P=W5I
M9"D@/2!G971P=W5I9"@D/BD@:68@9&5F:6YE9"`D/CM].PII9B`H(&1E9FEN
M960H)E=I;C,R.CI,;V=I;DYA;64I("D@>PHD=7-E<FYA;64@/2`F5VEN,S(Z
M.DQO9VEN3F%M93L*?0IE;'-I9B`H9&5F:6YE9"`D<'=U:60I('L*)'5S97)N
M86UE(#T@)'!W=6ED.PI]"F5L<V4@>PHD=7-E<FYA;64@/2`D14Y6>U5315).
M04U%?2!\?"`D14Y6>U5315)]('Q\("=365-414TG.PI]"FUY("1S=&UP9&ER
M(#T@(B1P871H)'-Y<WM?9&5L:6U]<&%R+2(N=6YP86-K*")(*B(L("1U<V5R
M;F%M92D["FUK9&ER("1S=&UP9&ER+"`P-S4U.PIM>2`D8V%C:&5?9&ER.PII
M9B`H)$5.5GM005)?0TQ%04Y]*2!["B1C86-H95]D:7(@/2`B=&5M<"TD)"([
M"GT*96QS92!["F]P96X@;7D@)&9H+"`B/#IR87<B+"`D<')O9VYA;64@;W(@
M9&EE('%Q6T-A;B=T(')E860@(B1P<F]G;F%M92(Z("0A73L*:68@*"AM>2`D
M;6%G:6-?<&]S(#T@9FEN9%]P87)?;6%G:6,H)&9H*2D@/CT@,"D@>PIS965K
M("1F:"P@)&UA9VEC7W!O<PHM("1&24Q%7V]F9G-E=%]S:7IE"BT@;&5N9W1H
M*")<,$-!0TA%(BDL(#`["FUY("1B=68["G)E860@)&9H+"`D8G5F+"!L96YG
M=&@H(EPP0T%#2$4B*3L*:68@*"1B=68@97$@(EPP0T%#2$4B*2!["G-E96L@
M)&9H+"`D;6%G:6-?<&]S"BT@)$9)3$5?;V9F<V5T7W-I>F4*+2!L96YG=&@H
M(EPP0T%#2$4B*0HM("1C86-H95]N86UE7W-I>F4L(#`["G)E860@)&9H+"`D
M8G5F+"`D8V%C:&5?;F%M95]S:7IE.PHD8G5F(#U^(',O7#`O+V<["B1C86-H
M95]D:7(@/2`B8V%C:&4M)&)U9B(["GT*?0IC;&]S92`D9F@["GT*:68@*"$D
M8V%C:&5?9&ER*2!["B1C86-H95]D:7(@/2`B=&5M<"TD)"(["B1%3E9[4$%2
M7T-,14%.?2`](#$["GT*)'-T;7!D:7(@+CT@(B1S>7-[7V1E;&EM?21C86-H
M95]D:7(B.PIM:V1I<B`D<W1M<&1I<BP@,#<U-3L*)$5.5GM005)?5$5-4'T@
M/2`D<W1M<&1I<CL*;&%S=#L*?0HD<&%R7W1E;7`@/2`D,2!I9B`D14Y6>U!!
M4E]414U0?2!A;F0@)$5.5GM005)?5$5-4'T@/7X@+R@N*RDO.PI]"G-U8B!?
M<V%V95]A<R!["FUY("@D;F%M92P@)&-O;G1E;G1S+"`D;6]D92D@/2!`7SL*
M;7D@)&9U;&QN86UE(#T@(B1P87)?=&5M<"\D;F%M92(["G5N;&5S<R`H+64@
M)&9U;&QN86UE*2!["FUY("1T96UP;F%M92`]("(D9G5L;&YA;64N)"0B.PIM
M>2`D9F@["BAO<&5N("1F:"P@)SXZ<F%W)RP@)'1E;7!N86UE*0HF)B`H<')I
M;G0@)&9H("1C;VYT96YT<RD*)B8@*&-L;W-E("1F:"D*;W(@9&EE('%Q6T5R
M<F]R('=R:71I;F<@(B1T96UP;F%M92(Z("0A73L*8VAM;V0@)&UO9&4L("1T
M96UP;F%M92!I9B!D969I;F5D("1M;V1E.PIR96YA;64H)'1E;7!N86UE+"`D
M9G5L;&YA;64I(&]R('5N;&EN:R@D=&5M<&YA;64I.PI]"G)E='5R;B`D9G5L
M;&YA;64["GT*<W5B(%]S971?<')O9VYA;64@>PII9B`H9&5F:6YE9"`D14Y6
M>U!!4E]04D]'3D%-17T@86YD("1%3E9[4$%27U!23T=.04U%?2`]?B`O*"XK
M*2\I('L*)'!R;V=N86UE(#T@)#$["GT*)'!R;V=N86UE('Q\/2`D,#L*:68@
M*"1%3E9[4$%27U1%35!](&%N9"!I;F1E>"@D<')O9VYA;64L("1%3E9[4$%2
M7U1%35!]*2`^/2`P*2!["B1P<F]G;F%M92`]('-U8G-T<B@D<')O9VYA;64L
M(')I;F1E>"@D<')O9VYA;64L("1S>7-[7V1E;&EM?2D@*R`Q*3L*?0II9B`H
M(21%3E9[4$%27U!23T=.04U%?2!O<B!I;F1E>"@D<')O9VYA;64L("1S>7-[
M7V1E;&EM?2D@/CT@,"D@>PII9B`H;W!E;B!M>2`D9F@L("<\)RP@)'!R;V=N
M86UE*2!["G)E='5R;B!I9B`M<R`D9F@["GT*:68@*"US("(D<')O9VYA;64D
M<WES>U]E>&5](BD@>PHD<')O9VYA;64@+CT@)'-Y<WM?97AE?3L*<F5T=7)N
M.PI]"GT*9F]R96%C:"!M>2`D9&ER("AS<&QI="`O7%$D<WES>W!A=&A?<V5P
M?5Q%+RP@)$5.5GM0051(?2D@>PIN97AT(&EF(&5X:7-T<R`D14Y6>U!!4E]4
M14U0?2!A;F0@)&1I<B!E<2`D14Y6>U!!4E]414U0?3L*)&1I<B`]?B!S+UQ1
M)'-Y<WM?9&5L:6U]7$4D+R\["B@H)'!R;V=N86UE(#T@(B1D:7(D<WES>U]D
M96QI;7TD<')O9VYA;64D<WES>U]E>&5](BDL(&QA<W0I"FEF("US("(D9&ER
M)'-Y<WM?9&5L:6U])'!R;V=N86UE)'-Y<WM?97AE?2(["B@H)'!R;V=N86UE
M(#T@(B1D:7(D<WES>U]D96QI;7TD<')O9VYA;64B*2P@;&%S="D*:68@+7,@
M(B1D:7(D<WES>U]D96QI;7TD<')O9VYA;64B.PI]"GT*<W5B(%]F:7A?<')O
M9VYA;64@>PHD,"`]("1P<F]G;F%M92!\?#T@)$5.5GM005)?4%)/1TY!345]
M.PII9B`H:6YD97@H)'!R;V=N86UE+"`D<WES>U]D96QI;7TI(#P@,"D@>PHD
M<')O9VYA;64@/2`B+B1S>7-[7V1E;&EM?21P<F]G;F%M92(["GT*;7D@)'!W
M9"`]("AD969I;F5D("9#=V0Z.F=E=&-W9"D@/R!#=V0Z.F=E=&-W9"@I"CH@
M*"AD969I;F5D("97:6XS,CHZ1V5T0W=D*2`_(%=I;C,R.CI'971#=V0H*2`Z
M(&!P=V1@*3L*8VAO;7`H)'!W9"D["B1P<F]G;F%M92`]?B!S+UXH/SU<+EPN
M/UQ1)'-Y<WM?9&5L:6U]7$4I+R1P=V0D<WES>U]D96QI;7TO.PHD14Y6>U!!
M4E]04D]'3D%-17T@/2`D<')O9VYA;64["GT*<W5B(%]P87)?:6YI=%]E;G8@
M>PII9B`H("1%3E9[4$%27TE.251)04Q)6D5$?2LK(#T](#$@*2!["G)E='5R
M;CL*?2!E;'-E('L*)$5.5GM005)?24Y)5$E!3$E:141](#T@,CL*?0IF;W(@
M*'%W*"!34$%73D5$(%1%35`@0TQ%04X@1$5"54<@0T%#2$4@4%)/1TY!344@
M*2`I('L*9&5L971E("1%3E9[(E!!4E\D7R)].PI]"F9O<B`H<7<O(%1-4$1)
M4B!414U0($-,14%.($1%0E5'("\I('L*)$5.5GLB4$%27R1?(GT@/2`D14Y6
M>R)005)?1TQ/0D%,7R1?(GT@:68@97AI<W1S("1%3E9[(E!!4E]'3$]"04Q?
M)%\B?3L*?0II9B`H)$5.5GM005)?5$5-4'TI('L*9&5L971E("1%3E9[4$%2
M7T-,14%.?3L*?0I]"G-U8B!O=71S('L*<F5T=7)N(&EF("1Q=6EE=#L*:68@
M*"1L;V=F:"D@>PIP<FEN="`D;&]G9F@@(D!?7&XB.PI]"F5L<V4@>PIP<FEN
M="`B0%]<;B(["GT*?0IS=6(@:6YI=%]I;F,@>PIR97%U:7)E($-O;F9I9SL*
M<'5S:"!`24Y#+"!G<F5P(&1E9FEN960L(&UA<"`D0V]N9FEG.CI#;VYF:6=[
M)%]]+"!Q=R@*87)C:&QI8F5X<"!P<FEV;&EB97AP('-I=&5A<F-H97AP('-I
M=&5L:6)E>'`*=F5N9&]R87)C:&5X<"!V96YD;W)L:6)E>'`**3L*?0IP86-K
M86=E(&UA:6X["G)E<75I<F4@4$%2.PIU;G-H:69T($!)3D,L(%PF4$%2.CIF
M:6YD7W!A<CL*4$%2+3YI;7!O<G0H0'!A<E]A<F=S*3L*9&EE('%Q6W!A<BYP
M;#H@0V%N)W0@;W!E;B!P97)L('-C<FEP="`B)'!R;V=N86UE(CH@3F\@<W5C
M:"!F:6QE(&]R(&1I<F5C=&]R>5QN70IU;FQE<W,@+64@)'!R;V=N86UE.PID
M;R`D<')O9VYA;64["D-/4D4Z.F5X:70H)#$I(&EF("@D0"`]?B]>7U1+7T58
M251?7"@H7&0K*5PI+RD["F1I92`D0"!I9B`D0#L*?3L*)#HZ7U]%4E)/4B`]
M("1`(&EF("1`.PI]"D-/4D4Z.F5X:70H)#$I(&EF("@D.CI?7T524D]2(#U^
M+UY?5$M?15A)5%]<*"A<9"LI7"DO*3L*9&EE("0Z.E]?15)23U(@:68@)#HZ
M7U]%4E)/4CL*,3L*7U]%3D1?7PH`\$X``````````````````(`400`!````
M```````````(1T```0```````````````$=```$``````````````/`400`!
M```````````````0$T$``0``````````````H$9```$``````````````%@3
M00`!``````````````!'0T,Z("A'3E4I(#$Q+C0N,```````````````````
M`$=#0SH@*$=.52D@,3$N-"XP````````````````````1T-#.B`H1TY5*2`Q
M,2XT+C````````````````````!'0T,Z("A'3E4I(#$Q+C0N,```````````
M`````````$=#0SH@*$=.52D@,3$N-"XP````````````````````1T-#.B`H
M1TY5*2`Q,2XT+C````````````````````!'0T,Z("A'3E4I(#$Q+C0N,```
M`````````````````$=#0SH@*$=.52D@,3$N-"XP````````````````````
M1T-#.B`H1TY5*2`Q,2XT+C````````````````````!'0T,Z("A'3E4I(#$Q
M+C0N,````````````````````$=#0SH@*$=.52D@,3$N-"XP````````````
M````````1T-#.B`H1TY5*2`Q,2XT+C````````````````````!'0T,Z("A'
M3E4I(#$Q+C0N,````````````````````$=#0SH@*$=.52D@,3$N-"XP````
M````````````````1T-#.B`H1TY5*2`Q,2XT+C````````````````````!'
M0T,Z("A'3E4I(#$Q+C0N,````````````````````````````````0```(`4
M`0!@L```0````/`4`0"0L```0````!`3`0"@L```0````%@3`0#`L```0```
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M```````````````````````````````````````````"````&0```!S````<
MC```4E-$4\.P'_7N:6;"VY3J??TKD/`!````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M```````````````````````````````````````````````````0``!$$```
M`.```%`0``!E$```".```'`0``!Q$```#.```(`0``!C$0``$.```'`1``#5
M$@``).```.`2``!@$P``/.```&`3``!/%```2.```%`4```C%0``6.```#`5
M``!:%P``;.```&`7``#"%P``A.```-`7```!&```C.```!`8```3&```E.``
M`"`8```C&@``F.```#`:``!<&@``L.```&`:``!R&P``N.```(`;``!R(```
MR.```(`@``#B(```X.```/`@```-(0``Z.```!`A```&(@``\.```!`B``#!
M*````.$``-`H``#C*@``&.$``/`J``#"*P``*.$``-`K``!#0@``1.$``%!"
M``"F0@``7.$``+!"```X1```9.$``$!$``!C1@``>.$``/!)```"2@``B.$`
M`!!*```12@``D.$``"!*```A2@``E.$``#!*```Q2@``F.$``$!*``!!2@``
MG.$``%!*````3@``H.$```!.```93@``K.$``"!.``#<3@``L.$``.!.``#F
M3@``N.$``/!.``#Q3@``O.$``$!/``!P3P``P.$``)!/``"S40``-.$``,!1
M``#%40``S.$`````````````````````````````````````````````````
M`````````````00!``1"```!`````0````$,!P`,0@@P!V`&<`50!,`"T```
M`1$)`!$!%@`*,`E@"'`'4`;`!-`"X````0H$``H!%P`#,`+``1$&`!$!!R`)
M,`A@!W`!4`$3!P`3`08""S`*8`EP"%`"P````1`)`!!B##`+8`IP"5`(P`;0
M!.`"\````04"``52`3`!!`$`!$(```$````!&0H`&0$;`A$P$&`/<`Y0#<`+
MT`G@`O`!!@(`!C("P`$3!@`3`0<""S`*8`G``M`!$`D`$&(,,`M@"G`)4`C`
M!M`$X`+P```!!0(`!5(!,`$$`0`$0@```0P&``QR"#`'8`;`!-`"X`$3"@`3
M`3,`##`+8`IP"5`(P`;0!.`"\`$)!0`)8@4P!&`#<`+````!"00`"5(%,`3`
M`M`!"@4`"J(&,`5@!,`"T````1,*`!,!5P`,,`M@"G`)4`C`!M`$X`+P`00!
M``1"```!#`<`#$((,`=@!G`%4`3``M````$*!@`*,@8P!6`$<`-0`L`!!`$`
M!$(```$````!`````0````$````!!@,`!D(",`%@```!`````00!``1B```!
M`````0````$(`P`(0@3``M````$`````````````````````````````````
M````````````````````````````````````````7;Z-9`````#2``$``0``
M`!$````1````*``!`&P``0"P``$`T"@``/`@``"03P``(&```#`:``#P*@``
M$!@``&`:``"`(```(!@``(`;```0(@``$"$``$!$``!00@``L$(``-`7``#:
M``$`\``!``L!`0`0`0$`&@$!`"<!`0`S`0$`1`$!`%`!`0!>`0$`:P$!`'@!
M`0"%`0$`EP$!`*$!`0"J`0$`M0$!`````0`"``,`!``%``8`!P`(``D`"@`+
M``P`#0`.``\`$`!P87(N97AE`%A37TEN=&5R;F%L<U]005)?0D]/5`!G971?
M=7-E<FYA;65?9G)O;5]G971P=W5I9`!M86EN`&UY7W!A<E]P;`!P87)?8F%S
M96YA;64`<&%R7V-L96%N=7``<&%R7V-U<G)E;G1?97AE8P!P87)?9&ER;F%M
M90!P87)?96YV7V-L96%N`'!A<E]F:6YD<')O9P!P87)?:6YI=%]E;G8`<&%R
M7VUK=&UP9&ER`'!A<E]S971U<%]L:6)P871H`'-H85]F:6YA;`!S:&%?:6YI
M=`!S:&%?=7!D871E`'AS7VEN:70`````````````````````````````````
M`````````````````````````````````````````````````````````&@0
M`0```````````%0:`0#8$@$``!(!````````````9!H!`'`4`0`0$@$`````
M``````#0&@$`@!0!`,@2`0```````````.0:`0`X%0$`````````````````
M````````````````2!4!``````!8%0$``````&05`0``````=!4!``````"`
M%0$``````)`5`0``````H!4!``````"T%0$``````,@5`0``````U!4!````
M``#D%0$``````/`5`0``````_!4!```````(%@$``````!P6`0``````,!8!
M``````!`%@$``````$P6`0``````5!8!``````!@%@$``````&@6`0``````
M=!8!``````"`%@$``````(P6`0``````E!8!``````"<%@$``````*@6`0``
M````M!8!``````#`%@$``````,P6`0``````U!8!``````#<%@$``````.@6
M`0``````_!8!```````0%P$``````!@7`0``````)!<!```````P%P$`````
M`#P7`0``````1!<!``````!0%P$``````%@7`0``````9!<!``````!P%P$`
M`````'P7`0``````B!<!``````"4%P$``````*`7`0``````K!<!``````"X
M%P$`````````````````Q!<!`````````````````-P7`0``````[!<!````
M````&`$``````!@8`0``````,!@!``````!`&`$``````%08`0``````9!@!
M``````!T&`$``````(@8`0``````H!@!``````"P&`$``````,08`0``````
MV!@!``````#H&`$```````@9`0``````'!D!```````L&0$``````$`9`0``
M````4!D!``````!<&0$``````&P9`0````````````````!X&0$`````````
M````````2!4!``````!8%0$``````&05`0``````=!4!``````"`%0$`````
M`)`5`0``````H!4!``````"T%0$``````,@5`0``````U!4!``````#D%0$`
M`````/`5`0``````_!4!```````(%@$``````!P6`0``````,!8!``````!`
M%@$``````$P6`0``````5!8!``````!@%@$``````&@6`0``````=!8!````
M``"`%@$``````(P6`0``````E!8!``````"<%@$``````*@6`0``````M!8!
M``````#`%@$``````,P6`0``````U!8!``````#<%@$``````.@6`0``````
M_!8!```````0%P$``````!@7`0``````)!<!```````P%P$``````#P7`0``
M````1!<!``````!0%P$``````%@7`0``````9!<!``````!P%P$``````'P7
M`0``````B!<!``````"4%P$``````*`7`0``````K!<!``````"X%P$`````
M````````````Q!<!`````````````````-P7`0``````[!<!````````&`$`
M`````!@8`0``````,!@!``````!`&`$``````%08`0``````9!@!``````!T
M&`$``````(@8`0``````H!@!``````"P&`$``````,08`0``````V!@!````
M``#H&`$```````@9`0``````'!D!```````L&0$``````$`9`0``````4!D!
M``````!<&0$``````&P9`0````````````````!X&0$`````````````````
M#P!?7V-X85]A=&5X:70``!8`7U]E<G)N;P```",`7U]G971R965N=``````N
M`%]?;6%I;@`````P`%]?;65M8W!Y7V-H:P``3`!?7W-P<FEN=&9?8VAK`$\`
M7U]S=&%C:U]C:&M?9F%I;```40!?7W-T86-K7V-H:U]G=6%R9`!L`%]D;&Q?
M8W)T,`!S`%]I;7!U<F5?<'1R````@P!A8V-E<W,``````@%C86QL;V,`````
M0P%C;&]S961I<@``>P%C>6=W:6Y?9&5T86-H7V1L;`!]`6-Y9W=I;E]I;G1E
M<FYA;````)D!9&QL7V1L;&-R=#````"V`65N=FER;VX```#3`65X:70``#T"
M9G!R:6YT9@```$D"9G)E90``KP)G971P:60`````N@)G971P=W5I9```R`)G
M971U:60`````C`-L<V5E:P".`VQS=&%T`),#;6%L;&]C`````*(#;65M8VAR
M`````*0#;65M8W!Y`````*8#;65M;6]V90```*H#;6MD:7(`[`-O<&5N``#P
M`V]P96YD:7(````#!'!O<VEX7VUE;6%L:6=N`````"@$<'1H<F5A9%]A=&9O
M<FL`````M@1R96%D``"W!')E861D:7(```"Y!')E861L:6YK``"\!')E86QL
M;V,```#J!')M9&ER`'\%<W!R:6YT9@```(<%<W1A=```D@5S=')C:'(`````
ME`5S=')C;7``````F05S=')D=7``````HP5S=')L96X`````J`5S=')N8VUP
M````L`5S=')R8VAR````M`5S=')S='(`````N@5S=')T;VL`````+`9U;FQI
M;FL`````+0!?7V5M=71L<U]G971?861D<F5S<P``&P!03%]D;U]U;F1U;7``
M`&\!4&5R;%]A=&9O<FM?;&]C:P``<`%097)L7V%T9F]R:U]U;FQO8VL`````
M[`%097)L7V-R;V%K7VYO8V]N=&5X=```@`)097)L7V=E=%]S=@```*\"4&5R
M;%]G=E]F971C:'!V````=@-097)L7VUG7W-E=````.X#4&5R;%]N97=84P``
M``#P`U!E<FQ?;F5W6%-?9FQA9W,``,8%4&5R;%]S869E<WES;6%L;&]C````
M`*<&4&5R;%]S=E]S971P=@"P!E!E<FQ?<W9?<V5T<'9N`````.(&4&5R;%]S
M>7-?:6YI=#,`````XP9097)L7W-Y<U]T97)M`&@'7U]E;75T;'-?=BY03%]C
M=7)R96YT7V-O;G1E>'0`:P=B;V]T7T1Y;F%,;V%D97(```!\!W!E<FQ?86QL
M;V,`````?@=P97)L7V-O;G-T<G5C=`````!_!W!E<FQ?9&5S=')U8W0`@`=P
M97)L7V9R964`@0=P97)L7W!A<G-E`````(('<&5R;%]R=6X``(D"1V5T36]D
M=6QE2&%N9&QE00```!`!```0`0``$`$``!`!```0`0``$`$``!`!```0`0``
M$`$``!`!```0`0``$`$``!`!```0`0``$`$``!`!```0`0``$`$``!`!```0
M`0``$`$``!`!```0`0``$`$``!`!```0`0``$`$``!`!```0`0``$`$``!`!
M```0`0``$`$``!`!```0`0``$`$``!`!```0`0``$`$``!`!```0`0``$`$`
M`!`!```0`0``$`$``!`!```0`0``$`$``!`!```0`0!C>6=W:6XQ+F1L;``4
M$`$`8WEG9V-C7W,M<V5H+3$N9&QL```H$`$`*!`!`"@0`0`H$`$`*!`!`"@0
M`0`H$`$`*!`!`"@0`0`H$`$`*!`!`"@0`0`H$`$`*!`!`"@0`0`H$`$`*!`!
M`"@0`0`H$`$`*!`!`"@0`0`H$`$`8WEG<&5R;#5?,S8N9&QL`#P0`0!+15).
M14PS,BYD;&P`````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````!``#````,```@`X```#```"`$```
M`/```(`8````(`$`@```````````````````!``!````8```@`(```!X``"`
M`P```)```(`$````J```@````````````````````0`)!```4`$`````````
M`````````````0`)!```8`$``````````````````````0`)!```<`$`````
M`````````````````0`)!```@`$```````````````````$```#``0"`V```
M@````````````````````0`)!```D`$``````````````````````0`!````
M"`$`@````````````````````0`)!```H`$``````````````````````0`!
M````.`$`@````````````````````0`)!```L`$``-`A`0!H!0``````````
M```X)P$`J`@`````````````X"\!`&@%`````````````$@U`0"H"```````
M``````#P/0$`/@``````````````,#X!`%@"`````````````(A``0#:!0``
M```````````&`%<`20!.`$4`6`!%````*````!`````@`````0`(````````
M`0```````````````0````$```8&!@`$!`0``@("``````!"0D(`G9V=`&YN
M;@"!@8$`L;&Q`/___P!S<W,`GY^?`%A86`"RLK(`_?W]`&MK:P#\_/P`U]?7
M`!86%@#:VMH`EI:6`(Z.C@!&1D8`R\O+``L+"P"@H*``_O[^`,G)R0#!P<$`
M`P,#`!$1$0"3DY,`L[.S`.[N[@#T]/0`X^/C`.7EY0"BHJ(`NKJZ`*^OKP`'
M!P<`<7%Q`+6UM0#U]?4`B8F)`(J*B@`T-#0`Z.CH`#4U-0!:6EH`M[>W`-C8
MV`#0T-``'1T=``P,#`"/CX\`OKZ^`&UM;0`)"0D`FYN;`"\O+P#-S<T`V=G9
M`!D9&0!-34T`1$1$`%!04``\/#P`9&1D`,+"P@`R,C(`"`@(``T-#0`/#P\`
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M```````````!04%!04%!04%!04%!04$!`0$U0",%03%#04%$12X!`0$!`0$!
M!0$\%Q<!`1,!`0$!`0$!`14!`285`0$/`0$!`0$!`0$3`1$!!34!)@$!`0$!
M`0$!(A(0,0\%`2`!`0$!`0$!`1`*("(5%2XA+@$!`0$!'R`A$"(0(R,E)B<!
M`0$!)1`0$!`0$!`0$!=%`0$!01`0$!`0$!`0$!`0%P$!`0\0$!40$!`0$!`0
M$`$!`0$2$!`3'Q`0$!`0$`<!`0$!(!`0$`$/$!`0$!$!`0$!`0<@$`H!`040
M$!`Q`0$!`0$!`0$!`0$!004/`0$!`0$!`0$!`0$!`0$!`0$!`0$!@`$``,`!
M``#@`P``\`,``/`#``#P`P``X`,``,`!``"``0``@`$``(`#``"``P````<`
M```'``".#P``_Q\``"@````@````0`````$`"`````````0`````````````
M``$````!`````````0$!``("`@`#`P,`!`0$``4%!0`&!@8`!P<'``@("``)
M"0D`"@H*``L+"P`,#`P`#0T-``X.#@`/#P\`$!`0`!$1$0`2$A(`$Q,3`!04
M%``5%14`%A86`!<7%P`8&!@`&1D9`!H:&@`;&QL`'!P<`!T='0`>'AX`'Q\?
M`"`@(``A(2$`(B(B`",C(P`D)"0`)24E`"8F)@`G)R<`*"@H`"DI*0`J*BH`
M*RLK`"PL+``M+2T`+BXN`"\O+P`P,#``,3$Q`#(R,@`S,S,`-#0T`#4U-0`V
M-C8`-S<W`#@X.``Y.3D`.CHZ`#L[.P`\/#P`/3T]`#X^/@`_/S\`0$!``$%!
M00!"0D(`0T-#`$1$1`!%144`1D9&`$='1P!(2$@`24E)`$I*2@!+2TL`3$Q,
M`$U-30!.3DX`3T]/`%!04`!145$`4E)2`%-34P!45%0`5555`%965@!75U<`
M6%A8`%E960!:6EH`6UM;`%Q<7`!=75T`7EY>`%]?7P!@8&``86%A`&)B8@!C
M8V,`9&1D`&5E90!F9F8`9V=G`&AH:`!I:6D`:FIJ`&MK:P!L;&P`;6UM`&YN
M;@!O;V\`<'!P`'%Q<0!R<G(`<W-S`'1T=`!U=74`=G9V`'=W=P!X>'@`>7EY
M`'IZ>@![>WL`?'Q\`'U]?0!^?GX`?W]_`("`@`"!@8$`@H*"`(.#@P"$A(0`
MA86%`(:&A@"'AX<`B(B(`(F)B0"*BHH`BXN+`(R,C`"-C8T`CHZ.`(^/CP"0
MD)``D9&1`)*2D@"3DY,`E)24`)65E0"6EI8`EY>7`)B8F`"9F9D`FIJ:`)N;
MFP"<G)P`G9V=`)Z>G@"?GY\`H*"@`*&AH0"BHJ(`HZ.C`*2DI`"EI:4`IJ:F
M`*>GIP"HJ*@`J:FI`*JJJ@"KJZL`K*RL`*VMK0"NKJX`KZ^O`+"PL`"QL;$`
MLK*R`+.SLP"TM+0`M;6U`+:VM@"WM[<`N+BX`+FYN0"ZNKH`N[N[`+R\O`"]
MO;T`OKZ^`+^_OP#`P,``P<'!`,+"P@##P\,`Q,3$`,7%Q0#&QL8`Q\?'`,C(
MR`#)R<D`RLK*`,O+RP#,S,P`S<W-`,[.S@#/S\\`T-#0`-'1T0#2TM(`T]/3
M`-34U`#5U=4`UM;6`-?7UP#8V-@`V=G9`-K:V@#;V]L`W-S<`-W=W0#>WMX`
MW]_?`.#@X`#AX>$`XN+B`./CXP#DY.0`Y>7E`.;FY@#GY^<`Z.CH`.GIZ0#J
MZNH`Z^OK`.SL[`#M[>T`[N[N`._O[P#P\/``\?'Q`/+R\@#S\_,`]/3T`/7U
M]0#V]O8`]_?W`/CX^`#Y^?D`^OKZ`/O[^P#\_/P`_?W]`/[^_@#___\`-#0T
M/#P\/#P\/#P\/#P\/#P\/#P\/#P\+`P$!`0$!`1<M-34U-34U-34U-3<U-34
MU-34U-34U-2T-%$$!`0$!&3,[/3T]/ST_/3T]/ST_/3\]/S\_/3T[,0T45$$
M!`0$9,3D[/3T]/3T]/3T]/3T]/3T]/3T].SLO#1145$$!`1DQ.3L[.SD[.SL
MY.3DY.SLY.SD[.SL[.2\-%%145$$!&2\Y.SLK(RLU,24E)R4C(R4E)2<G)SL
MW+PT45%145$$9+S<Y.SDQ%&4K&3DI$PT+%S$S%&TU.3<O#1145%145%DO-SD
MY.SDY%&$=.2,;*2<;(R\=*S4Y-R\-`0$!`0$!&2\W.3D[.3DO%&,U-2,K,QT
MS-RD4<3DW+PT!`0$!`0$9+S<Y.3DY.3<C'R<M'R$G&S,Y+QTQ.3<M#0$!`0$
M!`1<O-S<Y.3DY-2<I%R4C,P\E&3,O(Q1W-2T-`0$!`0$!&2TU-S<W.3<C(RL
M++RTS%K$9)QLE*3<U+0T!`0$!`0$7+34W-SDW-QTE-1,I/2,/-1TE'2<3-S,
MM#0$!`0$!`1DM-3<W-SDW'11]'2$[,QDY'QLI+1LI-2T-`0$!`0$!&2TS-S<
MS'R,A+3\S,3\]+3\O&3,K(11O*PT!`0$!`0$7+3,U,R,I-S<[/S\_/S\]/SL
MM.RL?)11K#0$!`0$!`1<K,S,;)ST_/S\_/S\_/S\_/S\_,24C%&L-`0$!`0$
M!%RLQ+Q\Y/S\_/S\_/S\_/S\_/S\[,RL4:PT!`0$!`0$7*RT3,3\_/SL]/S\
M_/S\_/S\_/STO%'$K#0$!`0$!`1<K+1,U/S\]+3<_/S\_/S\_/S\_,Q1S,2D
M-`0$!`0$!%RLQ*11_/S\M)ST_/S\_/S\_/STE%',Q*0L!`0$!`0$7*2D483L
M_/S,9+3L_/S\_/S\_-1LK,R\I#0$!`0$!`1<I&1TO/3\_,11A)SD_/S\_/SD
MC%',S,2D-`0$!`0$!%RD;(SD]/S\G%',493L_/S\_+QTS,S$O)PL!`0$!`0$
M7*2L9%QDI*11K,S$=)3D]/SD4;S,S,2\G#0$!`0$!`1<I+2\O(Q14<3$S,RL
M47Q\E&Q\O,S$Q+R<+`0$!`0$!%R<O,3$Q,3$Q,3$Q,3$45%14;S$Q,3$M)PT
M!`0$!`0$5)RTQ,3$Q,3$Q,3$Q,3$Q,3$Q,3$Q+RTG"P$!`0$!`14G+S$Q+R\
MO+R\O+R\O+R\O+R\O+R\O+2<+`0$!`0$!%R<M,2\M+2TM+2TM+2TM+2TM+2T
MM+2TK)PL!`0$!`0$5(24G)R4E)24E)24E)24E)24E)24E)24A"0$!`0$!`1$
M5%Q<7%Q<7%Q<7%Q<7%Q<7%Q<7%Q<7%Q4-`0$!`0$!````#\````?````#P``
M``<````#`````0`````````_````/P```#\````_````/P```#\````_````
M/P```#\````_````/P```#\````_````/P```#\````_````/P```#\````_
M````/P```#\````_````/P```#\````_*````!`````@`````0`(````````
M`0```````````````0````$```8&!@`$!`0``@("``````!"0D(`G9V=`&YN
M;@"!@8$`L;&Q`/___P!S<W,`GY^?`%A86`"RLK(`_?W]`&MK:P#\_/P`U]?7
M`!86%@#:VMH`EI:6`(Z.C@!&1D8`R\O+``L+"P"@H*``_O[^`,G)R0#!P<$`
M`P,#`!$1$0"3DY,`L[.S`.[N[@#T]/0`X^/C`.7EY0"BHJ(`NKJZ`*^OKP`'
M!P<`<7%Q`+6UM0#U]?4`B8F)`(J*B@`T-#0`Z.CH`#4U-0!:6EH`M[>W`-C8
MV`#0T-``'1T=``P,#`"/CX\`OKZ^`&UM;0`)"0D`FYN;`"\O+P#-S<T`V=G9
M`!D9&0!-34T`1$1$`%!04``\/#P`9&1D`,+"P@`R,C(`"`@(``T-#0`/#P\`
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M```````````!04%!04%!04%!04%!04$!`0$U0",%03%#04%$12X!`0$!`0$!
M!0$\%Q<!`1,!`0$!`0$!`14!`285`0$/`0$!`0$!`0$3`1$!!34!)@$!`0$!
M`0$!(A(0,0\%`2`!`0$!`0$!`1`*("(5%2XA+@$!`0$!'R`A$"(0(R,E)B<!
M`0$!)1`0$!`0$!`0$!=%`0$!01`0$!`0$!`0$!`0%P$!`0\0$!40$!`0$!`0
M$`$!`0$2$!`3'Q`0$!`0$`<!`0$!(!`0$`$/$!`0$!$!`0$!`0<@$`H!`040
M$!`Q`0$!`0$!`0$!`0$!004/`0$!`0$!`0$!`0$!`0$!`0$!`0$!@`$``,`!
M``#@`P``\`,``/`#``#P`P``X`,``,`!``"``0``@`$``(`#``"``P````<`
M```'``".#P``_Q\``"@````@````0`````$`"`````````0`````````````
M``$````!`````````0$!``("`@`#`P,`!`0$``4%!0`&!@8`!P<'``@("``)
M"0D`"@H*``L+"P`,#`P`#0T-``X.#@`/#P\`$!`0`!$1$0`2$A(`$Q,3`!04
M%``5%14`%A86`!<7%P`8&!@`&1D9`!H:&@`;&QL`'!P<`!T='0`>'AX`'Q\?
M`"`@(``A(2$`(B(B`",C(P`D)"0`)24E`"8F)@`G)R<`*"@H`"DI*0`J*BH`
M*RLK`"PL+``M+2T`+BXN`"\O+P`P,#``,3$Q`#(R,@`S,S,`-#0T`#4U-0`V
M-C8`-S<W`#@X.``Y.3D`.CHZ`#L[.P`\/#P`/3T]`#X^/@`_/S\`0$!``$%!
M00!"0D(`0T-#`$1$1`!%144`1D9&`$='1P!(2$@`24E)`$I*2@!+2TL`3$Q,
M`$U-30!.3DX`3T]/`%!04`!145$`4E)2`%-34P!45%0`5555`%965@!75U<`
M6%A8`%E960!:6EH`6UM;`%Q<7`!=75T`7EY>`%]?7P!@8&``86%A`&)B8@!C
M8V,`9&1D`&5E90!F9F8`9V=G`&AH:`!I:6D`:FIJ`&MK:P!L;&P`;6UM`&YN
M;@!O;V\`<'!P`'%Q<0!R<G(`<W-S`'1T=`!U=74`=G9V`'=W=P!X>'@`>7EY
M`'IZ>@![>WL`?'Q\`'U]?0!^?GX`?W]_`("`@`"!@8$`@H*"`(.#@P"$A(0`
MA86%`(:&A@"'AX<`B(B(`(F)B0"*BHH`BXN+`(R,C`"-C8T`CHZ.`(^/CP"0
MD)``D9&1`)*2D@"3DY,`E)24`)65E0"6EI8`EY>7`)B8F`"9F9D`FIJ:`)N;
MFP"<G)P`G9V=`)Z>G@"?GY\`H*"@`*&AH0"BHJ(`HZ.C`*2DI`"EI:4`IJ:F
M`*>GIP"HJ*@`J:FI`*JJJ@"KJZL`K*RL`*VMK0"NKJX`KZ^O`+"PL`"QL;$`
MLK*R`+.SLP"TM+0`M;6U`+:VM@"WM[<`N+BX`+FYN0"ZNKH`N[N[`+R\O`"]
MO;T`OKZ^`+^_OP#`P,``P<'!`,+"P@##P\,`Q,3$`,7%Q0#&QL8`Q\?'`,C(
MR`#)R<D`RLK*`,O+RP#,S,P`S<W-`,[.S@#/S\\`T-#0`-'1T0#2TM(`T]/3
M`-34U`#5U=4`UM;6`-?7UP#8V-@`V=G9`-K:V@#;V]L`W-S<`-W=W0#>WMX`
MW]_?`.#@X`#AX>$`XN+B`./CXP#DY.0`Y>7E`.;FY@#GY^<`Z.CH`.GIZ0#J
MZNH`Z^OK`.SL[`#M[>T`[N[N`._O[P#P\/``\?'Q`/+R\@#S\_,`]/3T`/7U
M]0#V]O8`]_?W`/CX^`#Y^?D`^OKZ`/O[^P#\_/P`_?W]`/[^_@#___\`-#0T
M/#P\/#P\/#P\/#P\/#P\/#P\/#P\+`P$!`0$!`1<M-34U-34U-34U-3<U-34
MU-34U-34U-2T-%$$!`0$!&3,[/3T]/ST_/3T]/ST_/3\]/S\_/3T[,0T45$$
M!`0$9,3D[/3T]/3T]/3T]/3T]/3T]/3T].SLO#1145$$!`1DQ.3L[.SD[.SL
MY.3DY.SLY.SD[.SL[.2\-%%145$$!&2\Y.SLK(RLU,24E)R4C(R4E)2<G)SL
MW+PT45%145$$9+S<Y.SDQ%&4K&3DI$PT+%S$S%&TU.3<O#1145%145%DO-SD
MY.SDY%&$=.2,;*2<;(R\=*S4Y-R\-`0$!`0$!&2\W.3D[.3DO%&,U-2,K,QT
MS-RD4<3DW+PT!`0$!`0$9+S<Y.3DY.3<C'R<M'R$G&S,Y+QTQ.3<M#0$!`0$
M!`1<O-S<Y.3DY-2<I%R4C,P\E&3,O(Q1W-2T-`0$!`0$!&2TU-S<W.3<C(RL
M++RTS%K$9)QLE*3<U+0T!`0$!`0$7+34W-SDW-QTE-1,I/2,/-1TE'2<3-S,
MM#0$!`0$!`1DM-3<W-SDW'11]'2$[,QDY'QLI+1LI-2T-`0$!`0$!&2TS-S<
MS'R,A+3\S,3\]+3\O&3,K(11O*PT!`0$!`0$7+3,U,R,I-S<[/S\_/S\]/SL
MM.RL?)11K#0$!`0$!`1<K,S,;)ST_/S\_/S\_/S\_/S\_,24C%&L-`0$!`0$
M!%RLQ+Q\Y/S\_/S\_/S\_/S\_/S\[,RL4:PT!`0$!`0$7*RT3,3\_/SL]/S\
M_/S\_/S\_/STO%'$K#0$!`0$!`1<K+1,U/S\]+3<_/S\_/S\_/S\_,Q1S,2D
M-`0$!`0$!%RLQ*11_/S\M)ST_/S\_/S\_/STE%',Q*0L!`0$!`0$7*2D483L
M_/S,9+3L_/S\_/S\_-1LK,R\I#0$!`0$!`1<I&1TO/3\_,11A)SD_/S\_/SD
MC%',S,2D-`0$!`0$!%RD;(SD]/S\G%',493L_/S\_+QTS,S$O)PL!`0$!`0$
M7*2L9%QDI*11K,S$=)3D]/SD4;S,S,2\G#0$!`0$!`1<I+2\O(Q14<3$S,RL
M47Q\E&Q\O,S$Q+R<+`0$!`0$!%R<O,3$Q,3$Q,3$Q,3$45%14;S$Q,3$M)PT
M!`0$!`0$5)RTQ,3$Q,3$Q,3$Q,3$Q,3$Q,3$Q+RTG"P$!`0$!`14G+S$Q+R\
MO+R\O+R\O+R\O+R\O+R\O+2<+`0$!`0$!%R<M,2\M+2TM+2TM+2TM+2TM+2T
MM+2TK)PL!`0$!`0$5(24G)R4E)24E)24E)24E)24E)24E)24A"0$!`0$!`1$
M5%Q<7%Q<7%Q<7%Q<7%Q<7%Q<7%Q<7%Q4-`0$!`0$!````#\````?````#P``
M``<````#`````0`````````_````/P```#\````_````/P```#\````_````
M/P```#\````_````/P```#\````_````/P```#\````_````/P```#\````_
M````/P```#\````_````/P```#\````_```!``0`$!````$`"`!H!0```0`@
M(````0`(`*@(```"`!`0```!``@`:`4```,`("````$`"`"H"```!````%@"
M-````%8`4P!?`%8`10!2`%,`20!/`$X`7P!)`$X`1@!/``````"]!._^```!
M```````````````````````_``````````0`!``!````````````````````
MN`$```$`4P!T`'(`:0!N`&<`1@!I`&P`90!)`&X`9@!O````E`$```$`,``P
M`#``,``P`#0`0@`P````)``"``$`0P!O`&T`<`!A`&X`>0!.`&$`;0!E````
M```@````+``"``$`1@!I`&P`90!$`&4`<P!C`'(`:0!P`'0`:0!O`&X`````
M`"`````P``@``0!&`&D`;`!E`%8`90!R`',`:0!O`&X``````#``+@`P`"X`
M,``N`#`````D``(``0!)`&X`=`!E`'(`;@!A`&P`3@!A`&T`90```"`````H
M``(``0!,`&4`9P!A`&P`0P!O`'``>0!R`&D`9P!H`'0````@````+``"``$`
M3`!E`&<`80!L`%0`<@!A`&0`90!M`&$`<@!K`',``````"`````L``(``0!/
M`'(`:0!G`&D`;@!A`&P`1@!I`&P`90!N`&$`;0!E````(````"0``@`!`%``
M<@!O`&0`=0!C`'0`3@!A`&T`90``````(````#0`"``!`%``<@!O`&0`=0!C
M`'0`5@!E`'(`<P!I`&\`;@```#``+@`P`"X`,``N`#````!$`````0!6`&$`
M<@!&`&D`;`!E`$D`;@!F`&\``````"0`!````%0`<@!A`&X`<P!L`&$`=`!I
M`&\`;@````````"P!#P_>&UL('9E<G-I;VX](C$N,"(@96YC;V1I;F<](E54
M1BTX(B!S=&%N9&%L;VYE/2)Y97,B/SX-"CQA<W-E;6)L>2!X;6QN<STB=7)N
M.G-C:&5M87,M;6EC<F]S;V9T+6-O;3IA<VTN=C$B(&UA;FEF97-T5F5R<VEO
M;CTB,2XP(CX-"B`@("`\87-S96UB;'E)9&5N=&ET>2!V97)S:6]N/2(P+C`N
M,"XP(B!T>7!E/2)W:6XS,B(@;F%M93TB4$%2+5!A8VME<BY!<'!L:6-A=&EO
M;B(@+SX-"B`@("`\9&5S8W)I<'1I;VX^4$%2+5!A8VME<B!!<'!L:6-A=&EO
M;CPO9&5S8W)I<'1I;VX^#0H@("`@/&1E<&5N9&5N8WD^#0H@("`@("`@(#QD
M97!E;F1E;G1!<W-E;6)L>3X-"B`@("`@("`@("`@(#QA<W-E;6)L>4ED96YT
M:71Y('1Y<&4](G=I;C,R(B!N86UE/2)-:6-R;W-O9G0N5VEN9&]W<RY#;VUM
M;VXM0V]N=')O;',B('9E<G-I;VX](C8N,"XP+C`B('!U8FQI8TME>51O:V5N
M/2(V-3DU8C8T,30T8V-F,61F(B!L86YG=6%G93TB*B(@<')O8V5S<V]R07)C
M:&ET96-T=7)E/2(J(B`O/@T*("`@("`@("`\+V1E<&5N9&5N=$%S<V5M8FQY
M/@T*("`@(#PO9&5P96YD96YC>3X-"B`@("`\=')U<W1);F9O('AM;&YS/2)U
M<FXZ<V-H96UA<RUM:6-R;W-O9G0M8V]M.F%S;2YV,R(^#0H@("`@("`@(#QS
M96-U<FET>3X-"B`@("`@("`@("`@(#QR97%U97-T9610<FEV:6QE9V5S/@T*
M("`@("`@("`@("`@("`@(#QR97%U97-T961%>&5C=71I;VY,979E;"!L979E
M;#TB87-);G9O:V5R(B!U:4%C8V5S<STB9F%L<V4B("\^#0H@("`@("`@("`@
M("`\+W)E<75E<W1E9%!R:79I;&5G97,^#0H@("`@("`@(#PO<V5C=7)I='D^
M#0H@("`@/"]T<G5S=$EN9F\^#0H@("`@/&-O;7!A=&EB:6QI='D@>&UL;G,]
M(G5R;CIS8VAE;6%S+6UI8W)O<V]F="UC;VTZ8V]M<&%T:6)I;&ET>2YV,2(^
M#0H@("`@("`@(#QA<'!L:6-A=&EO;CX-"B`@("`@("`@("`@(#PA+2T@5VEN
M9&]W<R`X+C$@+2T^#0H@("`@("`@("`@("`\<W5P<&]R=&5D3U,@260](GLQ
M9C8W-F,W-BTX,&4Q+30R,SDM.35B8BTX,V0P9C9D,&1A-SA](B\^#0H@("`@
M("`@("`@("`\(2TM(%=I;F1O=W,@5FES=&$@+2T^#0H@("`@("`@("`@("`\
M<W5P<&]R=&5D3U,@260](GME,C`Q,30U-RTQ-30V+30S8S4M835F92TP,#AD
M965E,V0S9C!](B\^#0H@("`@("`@("`@("`\(2TM(%=I;F1O=W,@-R`M+3X-
M"B`@("`@("`@("`@(#QS=7!P;W)T961/4R!)9#TB>S,U,3,X8CEA+35D.38M
M-&9B9"TX93)D+6$R-#0P,C(U9CDS87TB+SX-"B`@("`@("`@("`@(#PA+2T@
M5VEN9&]W<R`X("TM/@T*("`@("`@("`@("`@/'-U<'!O<G1E9$]3($ED/2)[
M-&$R9C(X93,M-3-B.2TT-#0Q+6)A.6,M9#8Y9#1A-&$V93,X?2(O/@T*("`@
M("`@("`@("`@/"$M+2!7:6YD;W=S(#$P("TM/@T*("`@("`@("`@("`@/'-U
M<'!O<G1E9$]3($ED/2)[.&4P9C=A,3(M8F9B,RTT9F4X+6(Y834M-#AF9#4P
M83$U83EA?2(O/@T*("`@("`@("`\+V%P<&QI8V%T:6]N/@T*("`@(#PO8V]M
M<&%T:6)I;&ET>3X-"CPO87-S96UB;'D^#0H`````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M`````````````````````````````````%````P```#8H0```&````P````@
MH````+```!@```!@H'"@@*"0H*"@L*#`H```````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
I````````````````````````````````````````````````````````
