package PAR::StrippedPARL::Dynamic;
use 5.008001;
use strict;
use warnings;
our $VERSION = '0.958';

use base 'PAR::StrippedPARL::Base';

our $Data_Pos = tell DATA;

=head1 NAME

PAR::StrippedPARL::Dynamic - Data package containing a dynamic PARL

=head1 SYNOPSIS

  # For details, see PAR::StrippedPARL::Base.
  PAR::StrippedPARL::Dynamic->write_parl($file) or die "Some error...";

=head1 DESCRIPTION

This class is internal to PAR. Do not use it outside of PAR.

This class is basically just a container for a dynamic binary PAR loader
which doesn't include the PAR code like the F<parl> or F<parl.exe>
you are used to. If you're really curious, I'll tell you it is
just a copy of the F<myldr/par> (or F<myldr/par.exe>) file.

The data is appended during the C<make> phase of the PAR build process,
but only if applicable: If you perl is static, you won't get the dynamic
loader.

If the binary data isn't appended during the build process, the class
methods will return the empty list.

=head1 CLASS METHODS

Inherits the methods from L<PAR::StrippedPARL::Base>.

=cut

sub _data_pos {
    my $class = shift;
    return $Data_Pos;
}

=head1 AUTHORS

Steffen Mueller E<lt>smueller@cpan.orgE<gt>,
Audrey Tang E<lt>cpan@audreyt.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2006-2009 by Steffen Mueller E<lt>smueller@cpan.orgE<gt>.

This program is free software; you can redistribute it and/or 
modify it under the same terms as Perl itself.

See F<LICENSE>.

=cut

$^V eq 5.32.1 
    or die sprintf("Perl (%s) version (%vd) doesn't match the version (5.32.1) ".
                   "that PAR::Packer was built with; please rebuild PAR::Packer", 
                   $^X, $^V);

1;

__DATA__
M35J0``,````$````__\``+@`````````0```````````````````````````
M````````````````````@`````X?N@X`M`G-(;@!3,TA5&AI<R!P<F]G<F%M
M(&-A;FYO="!B92!R=6X@:6X@1$]3(&UO9&4N#0T*)`````````!010``9(8+
M`-1#%V,``````````/``+@(+`@(G`$0```#.`````@```!`````0`````$``
M`0`````0`````@``!``````````%``(```````!@`0``!```H1@!``,``(``
M`"`````````0```````````0````````$``````````````0```````!`+T!
M````$`$`Q`H````@`0!0*P```-```-0!``````````````!0`0`P`````,``
M`!P`````````````````````````````````````````````````````````
MP!(!`'`"```````````````````````````````````N=&5X=````-A"````
M$````$0````$``````````````````!@``!@+F1A=&$```!@`````&`````"
M````2```````````````````0```P"YR9&%T80``[$0```!P````1@```$H`
M`````````````````$```$`N8G5I;&1I9#4`````P`````(```"0````````
M``````````!```!`+G!D871A``#4`0```-`````"````D@``````````````
M````0```0"YX9&%T80``S`$```#@`````@```)0``````````````````$``
M`$`N8G-S`````.`!````\`````````````````````````````"```#`+F5D
M871A``"]`0`````!```"````E@``````````````````0```0"YI9&%T80``
MQ`H````0`0``#````)@``````````````````$```,`N<G-R8P```%`K````
M(`$`4"L```"D``````````````````!```#`+G)E;&]C```P`````%`!```"
M````T```````````````````0```0@``````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M`````````````````````````````````````````````$B#["A(C0UE0```
MZ+`Z``!%,<`QTC')Z,0Z``!%,<`QTC')Z,@Z``!%,<`QTC')Z,PZ``!%,<`Q
MTC')2(/$*.G,.@``D)"0D)"0D)"0D)"03(T%J4\``#'22(T-$````.G;-P``
M9F8N#Q^$``````##D)"0D)"0D)"0D)"0D)"0055!5%575E-(@^PH28G-2(G7
M2(7)#X2U````2(L%GJ```$B+*$B%[0^$H@````^V`83`=`0\/757,?9,BV4`
M387D=#)(8_9(B>MF#Q]$``!)B?!,B>I,B>'HDCD``(7`=0=!@#PT/71'3(MC
M"$B#PPA-A>1UVDR)X$B#Q"A;7E]=05Q!7<-F+@\?A```````2(G.#[9&`4B#
MQ@&$P'0$/#UU\$0I[NN3#Q^$``````!(*>M-C60T`4C!^P-,B>")'TB#Q"A;
M7E]=05Q!7<-FD$4QY$R)X$B#Q"A;7E]=05Q!7<-F9BX/'X0``````&:0059!
M54%455=64TB![+````!(BQV8GP``2(L#2(F$)*@````QP$F)S$B%R70%@#D`
M=29(BX0DJ````$@K`P^%'`$``$B!Q+````!;7E]=05Q!74%>PP\?`.CC-P``
M28G%2(7`=,U(C2TD7@``2(U\)"!,B>GHYS<``$B%P`^$G@```(!X&"Y(C7`8
M=0R`?@$`=-]F#Q]$``"`>!@N#X26````3(GAZ$8X``!(B?%)B<;H.S@``$&-
M3`8"2&/)Z!XW``!)B?%-B>!(B>I)B<9(B<'HRC<``$R)\4B)^NC/-P``3(GQ
M@_C_=!"+1"0P)0#P```]`$```'1?Z$(X``!,B?'H>C8``$R)Z>A2-P``2(7`
M#X5K____9@\?A```````3(GIZ"@V``!,B>'H8#<``.G]_O__#Q\`@'X!+@^%
M8/___X!^`@`/A"7____I4?___P\?@`````#HH_[__^N?Z+PU``"09F8N#Q^$
M``````!!5%-(@>RX````2(L=+YX``$B+`TB)A"2H````,<!(C50D($F)S.@U
M-@``A<!U,8M4)#"!X@#P``"!^@!```!T"('Z`*```'49N@(```!,B>'H:C4`
M`(7`#Y3`#[;`ZP(QP$B+E"2H````2"L3=0M(@<2X````6T%<P^@Q-0``D$%4
MN#`0``!55U93Z)`T``!(*<1(BSVFG0``0;C_#P``2(L'2(F$)"@0```QP$B-
M="0@28G,2(G32(T-9UP``$B)\N@Y-@``A<!^38U0_TACZH!\+"``=7B%TG0\
M28GHNB\```!(B?'HE#4``$B%P'0G2(L%&)T``$B+".@0-```28GI28GP3(GB
M2(G!Z(<S``#K(@\?1```2(L%\9P``$B+".CI,P``28G83(GB2(G!Z&LS``!(
MBX0D*!```$@K!W432('$,!```%M>7UU!7,-(8^CKA^A6-```D`\?1```05=!
M5D%505155U932(/L.$B+'<F<``!(BP-(B40D*#'`,<"`.CU(B=8/E,!(B<](
M`<9(B?'HYC4``$B-5"0D2(GY2(G%Z/;[__])B<1(A<`/A/H```!(B<'HPC4`
M`$@YQ0^&R0```$QC?"0D3(LMC9P``$G!YP,/M@>$P`^$C@$``#P]#X2&`0``
M2(GX#Q\`#[90`4B#P`&$TG0%@/H]=>](*?A(F$V+90!(C4PH`NA<-```30'\
M28D$)$B%P'1028M%`$B-3P%*BQ0X#[8'B`*$P'4<ZQYF#Q^$```````/M@%(
M@\(!2(/!`8@"A,!T!#P]=>O&`CTQP&8/'T0```^V#`:(3`(!2(/``83)=?!(
MBT0D*$@K`P^%!@$``$B#Q#A;7E]=05Q!74%>05_#D#'`9@\?1```#[84!D&(
M%`1(@\`!A-)U\.O&9@\?1```3(LMJ9L``$V+=0!)@SX`#X2R````3(GP13'D
M#Q]$``!(@\`(1(GB08/$`4B#.`!U[X/"`TACTDC!X@.+!2_:``"%P'53QP4A
MV@```0```$B)T>AI,P``2(G!2(7`#X1=____36/\3(GR2<'G`TV)^.A[,P``
M28E%`$B)P4&-1"0!1(ED)"1(F$C'!,$`````Z9+^__\/'P!,B?'HP#,``$F)
M10!(B<%(A<`/A!#___]-8_Q)P><#Z\,/'X``````,<#IC_[__[H0````13'D
MZ6?____H)S(``)!F#Q]$``!32(/L,$B+'72:``!(BPOH;#$``$B+"^AD,0``
M3(T--5P``$R-!<8P``!(C15T60``2(G!Z.\P``!(BPOH/S$``$R-#7E9``#'
M1"0H`````$R-!2D2``!,B4PD($B)P4B-%5Q9``#HM3```)!(@\0P6\-F9BX/
M'X0```````\?`$B#["A(BP7UF0``2(L(Z.TP``!,C0V^6P``3(T%3S```$B)
MP4B-%?I8``!(@\0HZ70P``!F9BX/'X0``````)`QP,-F9BX/'X0``````&:0
M05>XV!```$%6055!5%575E/HJC```$@IQ$R++<"9``!)BT4`2(F$),@0```Q
MP$B-;"1`2(G.28G42(T-R5@``$B)ZNCX^/__NB\```!(B?%(B</HF#(``$B%
MP'0[2(GR2(T-JU@``.B4_/__2(GP2(N4),@0``!)*U4`#X5\`0``2('$V!``
M`%M>7UU!7$%=05Y!7\,/'P!,B>%(C3U]6```3(TE?U@``.AB,@``2(GZ2(G!
MZ*<R``!)B<=(A<!TG$&`/P`/A-<````/'T0``$B%VW033(GZ2(G9Z"`R``"%
MP`^$F@```$R)^>@P,@``28U4!_])B<9).==R#^L=D,8"`$B#Z@%).==T!8`Z
M+W3O3(GYZ`<R``!)B<9(B?'H_#$``$F-1`8!2#W_#P``#X<H____2(ET)#`Q
MTDR-M"3`````3(T-UE<``$R)9"0H3(GQ0;@`$```3(E\)"#HW2\``$B)ZDR)
M\>AR,0``A<!U$(M$)%`E`/```#T`@```=$Y(B?HQR>C4,0``28G'2(7`#X3%
M_O__08`_``^%+O___TB%VW0%@#LN=!1!O@$```!,C3U.5P``Z5[___]FD(![
M`0!TO.OD#Q^$``````"Z`0```$R)\>B#+P``A<!UH4R)\DB-#2=7``#H$/O_
M_TR)\>@8,0``Z7+^___H3B\``)!F9BX/'X0``````&:0051(@^P@NB\```!)
MB<SH'3$``$B-4`%(A<!,#T7B3(GA2(/$($%<Z=0P```/'T``056X.!```$%4
M5E/H<"X``$@IQ$B+'8:7``!(BP-(B80D*!```#'`28G-2(7)#X2=````@#D`
M#X24````Z)\P``!,C4`!2(G&28'X`!````^'LP```$R-9"0@0;D`$```3(GJ
M3(GAZ(4N``!)C40T_TPYX'<7ZQIF#Q^$``````#&``!(@^@!3#G@=`6`."]T
M[[HO````3(GAZ&(P``!(A<!T+4PYX'=83(GAZ"`P``!(BY0D*!```$@K$W5/
M2('$.!```%M>05Q!7<,/'T0``$B+A"0H$```2"L#=2](C0WC50``2('$.!``
M`%M>05Q!7>G:+P``9BX/'X0``````,8``.NC#Q\`,<#KI.C_+0``D&9F+@\?
MA```````#Q\`05=!5D%505155U932(/L.$B++6F6``!(BS6"E@``2(T=I54`
M`$B+10!(B40D*#'`3(UD)"1,B>)(B=GHH/7__TB%P'0[2&-4)"1(BP9(C030
M2(M0"$B)$$B%TG38#Q\`2(M0$$B#P`A(B1!(A=)U\$R)XDB)V>AE]?__2(7`
M=<5(BS49E@``2(T=154``$R)XDB)V>A']?__2(7`=$)(8U0D)$B+!DB-!-!(
MBU`(2(D02(72=-AF+@\?A```````2(M0$$B#P`A(B1!(A=)U\$R)XDB)V>@%
M]?__2(7`=;Y(BS6YE0``2(T=[50``$R)XDB)V>CG]/__2(7`=$)(8U0D)$B+
M!DB-!-!(BU`(2(D02(72=-AF+@\?A```````2(M0$$B#P`A(B1!(A=)U\$R)
MXDB)V>BE]/__2(7`=;Y(BS59E0``2(T=EE0``$R)XDB)V>B']/__2(7`=$)(
M8U0D)$B+!DB-!-!(BU`(2(D02(72=-AF+@\?A```````2(M0$$B#P`A(B1!(
MA=)U\$R)XDB)V>A%]/__2(7`=;Y(BQWYE```3(TM/50``$R)XDR)Z>@G]/__
M2(7`=$)(8U0D)$B+`TB-!-!(BU`(2(D02(72=-AF+@\?A```````2(M0$$B#
MP`A(B1!(A=)U\$R)XDR)Z>CE\___2(7`=;Y(BS69E```2(T=[5,``$R)XDB)
MV>C'\___2(7`=$)(8U0D)$B+!DB-!-!(BU`(2(D02(72=-AF+@\?A```````
M2(M0$$B#P`A(B1!(A=)U\$R)XDB)V>B%\___2(7`=;Y(BQTYE```3(TU.U,`
M`$R)XDR)\>AG\___2(7`=$)(8U0D)$B+`TB-!-!(BU`(2(D02(72=-AF+@\?
MA```````2(M0$$B#P`A(B1!(A=)U\$R)XDR)\>@E\___2(7`=;Y(BS79DP``
M2(T=.5,``$R)XDB)V>@'\___2(7`=$)(8U0D)$B+!DB-!-!(BU`(2(D02(72
M=-AF+@\?A```````2(M0$$B#P`A(B1!(A=)U\$R)XDB)V>C%\O__2(7`=;Y(
MBS5YDP``3(T]XU(``$R)XDR)^>BG\O__2(7`=$)(8U0D)$B+!DB-!-!(BU`(
M2(D02(72=-AF+@\?A```````2(M0$$B#P`A(B1!(A=)U\$R)XDR)^>AE\O__
M2(7`=;Y(BST9DP``2(TUC5(``$R)XDB)\>A'\O__2(7`=$)(8U0D)$B+!TB-
M!-!(BU`(2(D02(72=-AF+@\?A```````2(M0$$B#P`A(B1!(A=)U\$R)XDB)
M\>@%\O__2(7`=;Y(BSVYD@``2(TUQ%$``$R)XDB)\>CG\?__2(7`=$)(8U0D
M)$B+!TB-!-!(BU`(2(D02(72=-AF+@\?A```````2(M0$$B#P`A(B1!(A=)U
M\$R)XDB)\>BE\?__2(7`=;Y,B>)(C0W;40``Z)'Q__](B<)(A<!T"$R)^>A!
M]?__3(GB2(T-S5$``.AR\?__2(G"2(7`=`Q(C0W+40``Z![U__],B>)(C0W'
M40``Z$_Q__](B<)(A<!T,DR)\>C_]/__2(M$)"A(*T4`=4!(C17!40``3(GI
M2(/$.%M>7UU!7$%=05Y!7^G5]/__3(GB2(T-CE$``.@&\?__2(G"2(7`=+](
MB=GHMO3__^NUZ/\H``"09F8N#Q^$```````/'P!32(/L,$B+'721``!(BP-(
MB40D*(L%5D```(/X_W012(M4)"A(*Q-U/$B#Q#!;PY!(C50D)$B-#<M0``#H
MG_#__TB)PC'`2(72=!(/M@(\,`^5PH3`#Y7`#[;`(=")!0U```#KNNB&*```
MD`\?1```2(/L*.@'*0``B<'H\"@``$B%P'0#2(L`2(/$*,,/'P!!5D%50516
M4TB#[$!(BS7=D```3(TMS%```$B+!DB)1"0X,<!)B<Q(C50D-$R)Z>@<\/__
M2(G#2(7`=`6`.`!U+TB+1"0X2"L&#X6A````3(GB3(GI2(/$0%M>05Q!74%>
MZ:KS__]F+@\?A```````3(GB2(G!Z.4I``!(A<!T($B+1"0X2"L&=69(@\1`
M6UY!7$%=05[#9@\?A```````3(GAZ(@I``!(B=E)B<;H?2D``$F-3`8"Z&,H
M``!(B5PD($V)X$R-#6-/``!(C15>3P``2(G!28G&Z`(I``!(BT0D.$@K!G4(
M3(GRZ6+____H:R<``)!F+@\?A```````05>XJ($``$%6055!5%575E/HNB8`
M`$@IQ$B+/="/``!,C37R3@``2(L'2(F$))B!``!(C07>3@``2(G.3(UL)%!,
MB?%(QX0DX`````````!,B>I(QX0D.`$```````!(QX0D``$```````!(B80D
MV````$B-!8)/``!(B80D&`$``$B-!7I/``!(B80D(`$``$B-!7-/``!(B80D
M*`$``$B-!6E/``!(B80D,`$``$B-!5Y/``!(B80D^````.B-[O__28G$2(7`
M=`F`.``/A5P$``#H%R<``(G!Z``G``!(A<`/A,<#``!(BRA(A>T/A+L#``!(
MB>GH,R@``$B-3``!Z!DG``!$#[9%`$B)PT6$P'0M28G?3(TE_TX``&:010^V
MP$R)^4R)XDB#Q0'HK2<``$0/MD4`28/'`D6$P'7?2(VL)!`!``!(C0UH3@``
MZQ=F#Q]$``!(BTT(2(/%"$B%R0^$!P$``$R)ZNC?[?__28G$2(7`=-^`.`!T
MVDB)P>@J\/__A<!TSDR)X>B.)P``2(G'2(7`#X32````9I!(B?GHB"<``$B)
MV4B)Q>A])P``1(VD!00$``!-8^1,B>'H6B8``$B)7"0H2(T54$X``$F)^$F)
MQTB-!4Q.``!,C0U230``2(E$)"!,B?GH[28``+K``0``3(GYZ&`F``"#^/\/
MA%\"``!,B>I,B?GH_"4``(G#A<`/A,(```!(BQXQ[>AX)```38GY2(T5/DX`
M`$B+2!A)B=CH<B4``$B+-:N-``!(BX0DF($``$@K!@^%504``$B)Z$B!Q*B!
M``!;7E]=05Q!74%>05_##Q\`,>U,C25^30``ZQP/'T0``$ACQ4R+I,30````
M387D=$!!@#PD`'0Y3(GA@\4!Z`[O__^%P'3:3(GAZ'(F``!(B<=(A<`/A>;^
M__](8\5,BZ3$T````$V%Y'7&9@\?1```,?_IR?[__V8/'X0``````(M$)&`E
M`/```#T`0```#X4J____BVPD:.CS)```.<4/A1G___^+1"1@)?\!```]P`$`
M``^%!?___TR)X>C_)```3(GJ2(T-NTP``$B)Q>@=[/__2(G"2(7`#X0Q`@``
M2(L.Z,GR__](B<=(A<`/A!T"``!(B?GHU24``$F)Q(/X!WXU2)A(C15730``
M2(U,!_CHFB4``(7`=1Y!@_P(#X1X`P``08U$)/=(F(`\!R\/A&<#```/'P"+
M!4([``"#^/\/A!$#``!(C0W)2P``A<`/A,P!``!(C14G3```2(TU/4L``.A'
M[___3(TM:$L``$R-)1=-``#H]",``$B)="0H38GX2(GIB40D($R-#49+``!(
MC17G3```Z.(D``#K,.@;(P``@S@1=3CHP2,``(/#`4B)="0P38GIB5PD*$V)
M^$R)XDB)Z8E$)"#HL"0``+K``0``2(GIZ",D``"#^/]TODR)^>AV(P``3(GQ
M2(GJZ+ON__](B>GHH_K__^G<_?__9@\?1```Z+,B``"#.!$/A)/]___HI2(`
M`$B+'HLHZ!LB``!-B?E)B=A(C16>2P``2(M(&(EL)"`Q[>@/(P``Z9C]__]F
M+@\?A```````,=M(C0TR2P``ZQ4/'T0``$ACPTB+C,3P````2(7)=#E,B>J#
MPP'H=>K__TB%P'3@@#@`=-M(B<'H,R0``$B)Q4B%P`^%_/O__TACPTB+C,3P
M````2(7)=<>Y#0```$B-+=M*``#H!B,``$&X4P```$B)P^GL^___#Q^$````
M``!(B<'HR/G__TB+/;&*``!(BX0DF($``$@K!P^%6P(``$R)X4B!Q*B!``!;
M7E]=05Q!74%>05_IM2,```\?1```2(L^Z=O]___'!5XY````````2(GYN@``
M`0#HX2(``$B-#>%)``!!B<2%P`^$%?[__T&X`@```$C'PN[___^)P>A+(@``
M2(V<))`!``!!N`8```!$B>%(B=KHPB(``("\))`!````=0Z`O"21`0``0P^$
M40$``$4QP#'21(GAZ`TB``#HZ!L``$F)Q>L1#Q\`08G`2(G:3(GIZ#(<``!!
MN`"```!(B=I$B>'H<2(``(7`?]U$B>%(C9PD0`$``$B-O"1@`0``Z$4A``!,
MB>I(B=E)B=WHAQT``$B)^TB-M"2(`0``3(TEPDD```\?1```10^V10!(B=E,
MB>)(@\,"28/%`>AH(@``2#G>=>-(C05]2```QH0DB`$```!(B40D*$B)?"0@
M3(T-FT@``$V)^$B-%2I*``!(B>GH,2(``.F._?__#Q]``$B-#;A(``!,B>I(
MB4PD2.B$Z/__2(7`#X2C_O__#[8`A,`/A)C^__\\,`^$D/[__\<%[C<```$`
M``!(BTPD2.FZ_/__2(MV"$B%]@^$C_S__TB)\>@?(@``2(/X`P^&??S__TB-
M3`;\2(T5I4D``.CD(0``A<!(#T3^Z6'\__^`O"22`0``00^%H?[__X"\)),!
M``!##X63_O__@+PDE`$``$@/A87^__^`O"25`0``10^%=_[__T&X`@```$C'
MPL;___]$B>'H?"```$&X*0```$B)VD2)X>C[(```2(T%94<``$B)7"0@2(E$
M)"CIZ_[__^BP'P``D&9F+@\?A```````#Q]``$%6055!5%=64TB#[#A(BQWL
MAP``2(L]%8@``$B+"TB+!TB)1"0H,<!,C60D).C.'@``2(L+2(G&Z,,>```/
MMI:X````2(T-4$<``(B0N0```$R)XN@WY___2(7`=`F`.#$/A)D!``#H1/'_
M_TB+"^B,'@``0;D#````0;@!````2(T5N$@``$B)P>@I'@``2(G&2(7`=#-(
MBT`02(L+3(LHZ%H>``!,C0642```2(G!3(GJZ-@=``!(BT803(LH0?9%#D`/
MA78!``!(BPOH+AX``$&Y`P```$&X`0```$B-%6%(``!(B<'HRQT``$B)QDB%
MP'0X2(T-9D8``$R)XNB,YO__2(L+28G%2(7`#X3]````2(M&$$R+,.CA'0``
M38GH2(G!3(GRZ&,=``!(BPOHRQT``$&X`0```$B-%09(``!(B<'H=AT``$B+
M"TF)Q>BK'0``3(T%`4@``$R)ZDB)P>@I'0``2(L+Z)$=``!(C0WC10``3(GB
MQH"Y`````.@+YO__2(D%C,4``$B)P4B%P'0\NL`!``#H`A\``(U0`8/Z`78)
M@_@1#X6I````2(M$)"A(*P</A;L```!(@\0X6UY?05Q!74%>PP\?1```2(L-
M2<4``.AD]O__2(D%-<4``$B)P4B%P'6I2(T-=D<``.C1'```9@\?A```````
M@'@!``^$8O[__^E8_O__D$B+!0G%``!,BS!(BT803(LHZ-H<``!,B?),B>GH
M/^C__^GW_O__9BX/'X0``````$B+"^BX'```3(GJ2(G!Z%T<``#I<O[__T&)
MP4B+!;[$``!,BP6OQ```2(T-($<``$B+$.A0'```Z",=``"09I!!5D%50513
M2(/L.$B+'9Z%``!(BP-(B40D*#'`28G-Z.SM__^Z+P```$B)P4F)Q.C<'@``
M2(7`2(U0`4P/1>),B>'HF1X``$V%[4F)QHL%330``$$/E<2#^/]T;(7`#Y7`
M02'$183D=1](BT0D*$@K`P^%F````$B#Q#A;05Q!74%>PP\?1```08!]``!T
MVD&X!````$B-%4]%``!,B?'H7!X``(7`=<%(BT0D*$@K`W5=3(GI2(/$.%M!
M7$%=05[I.^7__P\?`$B-5"0D2(T-8T0``.@WY/__2(7`=`L/M@"$P'0$/#!U
M%\<%K3,```````#I<?___P\?A```````QP66,P```0```.E5____Z`P<``"0
MD)"0D)"0D)"0D)!!5T%6055!5%575E-(@>RX`@``2(L]=H0``$B+!TB)A"2H
M`@``,<!(B<I,C5DX2(U,)"!(C9PDH````$F)RF:008L#38L+28/"$$F#PP@/
MR(G`38G(28E"\$R)R$G!Z"!(P>@H2<'I."4`_P``3`G(38G!2<'@"$G!X1A!
M@>```/\`18G)3`G(3`G`28E"^$DYVG6L3(V)``(```\?1```2(M!:$@S04!(
M@\$(2#-!"$@S0?A,C00`2,'H'TP)P(/@_TB)07A,.<EUU4R+&DB+<A!(BUH@
M2(MZ&$R)V$R)V4B+:@A,BU0D*$C!Z1M(P>`%2`G(`T0D($Z-E!>9>8):C8P8
MF7F"6DB)\$@Q^$@AZ$@Q^$2-#`%(B>E(B>A(P>@"2,'A'DR)RT@)P4B)\$C!
MZQM!B<A,,<!,(=A(,?!,`=!-B<I)P>(%20G:1HTD$$R)V$V)VDG!Z@)(P>`>
M38GE3`G03(M4)#!)P>T;2#'!B<-,(<E.C906F7F"6DPQP4P!T4V)XDG!X@5-
M">I,BVPD.$$!RDR)R4G!Z0)(P>$>18G23XV$*)EY@EI,"<E-B=5(,<A!B<E,
M(>!)P>T;2#'83`'`38G02<'@!4T)Z$R+;"1`00'`3(G@2<'L`DC!X!Y%B<!*
MC9PKF7F"6DP)X$V)Q4@QP4G![1M!B<1,(=%,,<E(`=E,B<-(P>,%3`GK3(ML
M)$@!RTR)T4G!Z@)(P>$>B=M/C8PIF7F"6DP)T4F)W4@QR$G![1M!B<I,(<!,
M,>!,`<A)B=E)P>$%30GI3(ML)%!!`<%,B<!)P>@"2,'@'D6)R4^-I"R9>8):
M3`G`38G-2#'!2<'M&T&)P$@AV4PQT4P!X4V)S$G!Y`5-">Q,BVPD6$$!S$B)
MV4C!ZP)(P>$>18GD3XV4*IEY@EI("=E-B>5(,<A)P>T;B<M,(<A,,<!,`=!-
MB>))P>(%30GJ3(ML)&!!`<),B<A)P>D"2,'@'D6)TD^-A"B9>8):3`G(38G5
M2#'!2<'M&T&)P4PAX4@QV4P!P4V)T$G!X`5-">A,BVPD:$$!R$R)X4G![`)(
MP>$>18G`2HV<*YEY@EI,">%-B<5(,<A)P>T;08G,3"'03#'(2`'83(G#2,'C
M!4P)ZTR+;"1P`<-,B=!)P>H"2,'@'HG;3XV,*9EY@EI,"=!)B=U(,<%)P>T;
M08G"3"'!3#'A3`')28G92<'A!4T)Z4R+;"1X00')3(G!2<'H`DC!X1Y%B<E/
MC:0LF7F"6DP)P4V)S4@QR$G![1M!B<A((=A,,=!,`>!-B<Q)P>0%30GL3(NL
M)(````!!`<1(B=A(P>`>2,'K`D6)Y$^-E"J9>8):2`G838GE2#'!2<'M&XG#
M3"')3#'!3`'138GB2<'B!4T)ZDR+K"2(````00'*3(G)2<'I`DC!X1Y%B=)/
MC80HF7F"6DP)R4V)U4@QR$G![1M!B<E,(>!(,=A,`<!-B=!)P>`%30GH3(NL
M))````!!`<!,B>!)P>P"2,'@'D6)P$J-G"N9>8):3`G@38G%2#'!2<'M&T&)
MQ$PAT4PQR4@!V4R)PTC!XP5,">M,BZPDF`````'+3(G12,'A'DG!Z@*)VT^-
MC"F9>8):3`G128G=2#'(2<'M&T&)RDPAP$PQX$P!R$F)V4G!X05-">E,BZPD
MH````$$!P4R)P$G!Z`)(P>`>18G)3XVD+)EY@EI,"<!-B<U(,<%)P>T;08G`
M2"'93#'13`'A38G,2<'D!4T)[$:--"%(B=E(P>L"2,'A'D@)V4B+G"2H````
M2#'(08G,3"'(38V4&IEY@EI,B?-,,<!(P>L;3`'038GR2<'B!4D)VDB+G"2P
M````00'"3(G(2,'@'DG!Z0)%B=)-C808F7F"6DP)R$R)TT@QP4C!ZQM!B<%,
M(?%,,>%)`<A,B=%(P>$%2`G93(GS2<'N`DC!XQZ)R4P)\TD!R$B+C"2X````
M2#'818G%3"'028V,#)EY@EI-B>Q,,<A)P>P;2`'(3(GI2,'A!4P)X8G)2`'(
M3(G12<'J`DC!X1Y!B<1,"=%)B=J)VTDQRD4QT$R+E"3`````3XV,$:'KV6Y-
MB>)-`<%-B>!)P>H;2<'@!4T)T$6)P$T!P4V)Z$G!X!Y)P>T"18G*2`.<),@`
M``!-">A)B<V)R4@#C"30````33'%1#'H38G52(V<`Z'KV6Y,B=!)P>T;2,'@
M!4P)Z(G`2`'#3(G@2<'L`DC!X!Y!B=U,">!-B<1%B<!,`X0DV````$DQQ$4Q
MX4V)[$Z-C`FAZ]EN3(GI2<'L&TC!X05,">&)R4D!R4R)T4G!Z@)(P>$>18G,
M3`G128G"B<!),<I!,=I,B>-/C900H>O9;DV)X$C!ZQM)P>`%20G818G`30'"
M38GH2<'M`DG!X!Y$B=--">A)B<U(`X0DX````(G)33'%2`.,).@```!%,>E)
MB=U.C8P(H>O9;DB)V$G![1M(P>`%3`GHB<!)`<%,B>!)P>P"2,'@'D6)S4P)
MX$V)Q$6)P$P#A"3P````23'$13'B38GL3HV4$:'KV6Y,B>E)P>P;2,'A!4P)
MX8G)20'*2(G92,'K`DC!X1Y%B=1("=E(B<.)P$@#A"3X````2#'+03'93(GC
M3XV,"*'KV6Y-B>!(P>L;2<'@!4D)V$6)P$T!P4V)Z$G![0))P>`>1(G+30GH
M28G-B<E-,<5(`XPD``$``$4QZDF)W4Z-E!"AZ]EN2(G82<'M&TC!X`5,">B)
MP$D!PDR)X$G![`)(P>`>18G53`G@38G$18G`3`.$)`@!``!),<1%,>%-B>Q.
MC8P)H>O9;DR)Z4G![!M(P>$%3`GAB<E)`<E(B=E(P>L"2,'A'D6)S$@)V4B)
MPXG`2`.$)!`!``!(,<M!,=I,B>-/C900H>O9;DV)X$C!ZQM)P>`%20G818G`
M30'"38GH2<'M`DG!X!Y$B=--">A)B<V)R4TQQ44QZ4F)W4Z-C`BAZ]EN2(G8
M2,'@!4G![1M(`XPD&`$``$P)Z(G`20'!3(G@2<'L`DC!X!Y%B<U,">!-B<1%
MB<!,`X0D(`$``$DQQ$4QXDV)[$Z-E!&AZ]EN3(GI2<'L&TC!X05,">&)R4D!
MRDB)V4C!ZP)(P>$>18G42`G92(G#B<!(`X0D*`$``$@QRT$QV4R)XT^-C`BA
MZ]EN38G@2,'K&TG!X`5)"=A%B<!-`<%-B>A)P>T"2<'@'D2)RTT)Z$F)S8G)
M33'%13'J28G=3HV4$*'KV6Y(B=A)P>T;2,'@!4P)Z$@#C"0P`0``B<!)`<),
MB>!)P>P"2,'@'D6)U4P)X$V)Q$6)P$P#A"0X`0``23'$13'A38GL3HV,":'K
MV6Y,B>E)P>P;2,'A!4P)X8G)20')2(G92,'K`DC!X1Y%B<Q("=E(B<.)P$@#
MA"1``0``2#'+03':3(GC3XV4$*'KV6Y-B>!(P>L;2<'@!4D)V$6)P$T!PDV)
MZ$G![0))P>`>1(G330GH28G-B<E-,<5%,>E)B=U.C8P(H>O9;DB)V$G![1M(
MP>`%3`GHB<!)`<%,B>!(P>`>2<'L`D6)S4@#C"1(`0``3`G@38G$18G`3`.$
M)%`!``!),<1%,>).C:01H>O9;DR)Z4V)ZDG!ZAM(P>$%3`G1B<E)`<Q(B=E(
MP>L"2,'A'D6)XD@)V4B)PXG`2`.$)%@!``!(,<M!,=E,B=-/C8P(H>O9;DV)
MT$C!ZQM)P>`%20G83(GK2<'M`DC!XQY%`<%)B<B)R4P)ZT6)R4DQV$F)WD&)
MW44QX$V)S$Z-A`"AZ]EN3(G(2<'L&TC!X`5,">!%C20`38G02<'J`KC<O!N/
M2<'@'DT)T$T)QD6)PDPATTTASDD)WDB+G"1@`0``2`'#2`'93(GC3`'Q38GF
M2,'C!4G![AM,"?,!RTR)R4G!Z0)(P>$>B=M,"<E)"<A)B<Y!B<E-(>!-(=9-
M"<9,BX0D:`$``$D!P$T!Z$F)W4T!\$F)WDG!Y05)P>X;30GU28G.10'%38G@
M2<'L`DG!X!Y%B>U-">!%B<1-"<9)(=Y,(>%,"?%,B[0D<`$``$D!QDT!UDV)
MZDP!\4V)[DG!X@5)P>X;30GR00'*2(G92,'K`DC!X1Y%B=)("=E)"<A)B<Z)
MRTTAZ$TAYDT)QDR+A"1X`0``20'`30'(38G130'P38G62<'A!4G![AM-"?%)
MB<Y%`<%-B>A)P>T"2<'@'D6)R4T)Z$6)Q4T)QDTAUDPAZ4P)\4R+M"2``0``
M20'&30'F38G,3`'Q38G.2<'D!4G![AM-"?1!`<Q,B=%)P>H"2,'A'D6)Y$P)
MT4D)R$F)SD&)RDTAR$TA[DT)QDR+A"2(`0``20'`20'83(GC30'P38GF2,'C
M!4G![AM,"?-)B<Y$`<--B<A)P>D"2<'@'HG;30G(18G!30G&32'F3"')3`GQ
M3(NT))`!``!)`<9-`>Y)B=U,`?%)B=Y)P>4%2<'N&TT)]4$!S4R)X4C!X1Y)
MP>P"18GM3`GA20G(28G.08G,22'832'.30G&3(N$))@!``!)`<!-`=!-B>I-
M`?!-B>Y)P>(%2<'N&TT)\DF)SD4!PDF)V$C!ZP))P>`>18G220G81(G#30G&
M32'N2"'93`GQ3(NT)*`!``!)`<9-`<Y-B=%,`?%-B=9)P>$%2<'N&TT)\4$!
MR4R)Z4G![0)(P>$>18G)3`GI20G(28G.08G-32'022'>30G&3(N$)*@!``!)
M`<!-`>!-B<Q-`?!-B<Y)P>0%2<'N&TT)]$F)SD4!Q$V)T$G!Z@))P>`>18GD
M30G018G"30G&32'.3"'13`GQ3(NT)+`!``!)`<9)`=Y,B>-,`?%-B>9(P>,%
M2<'N&TP)\P'+3(G)2<'I`DC!X1Z)VTP)R4D)R$F)SD&)R4TAX$TAUDT)QDR+
MA"2X`0``20'`30'H28G=30'P28G>2<'E!4G![AM-"?5)B<Y%`<5-B>!)P>P"
M2<'@'D6)[4T)X$6)Q$T)QDDAWDPAX4P)\4R+M"3``0``20'&30'638GJ3`'Q
M38GN2<'B!4G![AM-"?)!`<I(B=E(P>L"2,'A'D6)TD@)V4D)R$F)SHG+32'H
M32'F30G&3(N$),@!``!)`<!-`<A-B=%-`?!-B=9)P>$%2<'N&TT)\4F)SD4!
MP4V)Z$G![0))P>`>18G)30GH18G%30G&32'63"'I3`GQ3(NT)-`!``!)`<9-
M`>9-B<Q,`?%-B<Y)P>0%2<'N&TT)]$$!S$R)T4G!Z@)(P>$>18GD3`G120G(
M28G.08G*32'(32'N30G&3(N$)-@!``!)`<!)`=A,B>--`?!-B>9(P>,%2<'N
M&TP)\TF)SD0!PTV)R$G!Z0))P>`>B=M-"<A-"<9%B<%-(>9,(<E,"?%,B[0D
MX`$``$D!QDT![DF)W4P!\4F)WDG!Y05)P>X;30GU00'-3(GA2<'L`DC!X1Y%
MB>U,">%)"<A)B<Q!B<])(=A-(<Q-"<1,BX0DZ`$``$D!P$T!T$V)ZDT!X$V)
M[$G!X@5)P>P;30GB10'"28G82,'K`DG!X!Y%B=))"=A(B<M%B<1,"<-,(>M,
M(>%("=E(BYPD\`$``$@!PT@#A"3X`0``20'93(G33`'X3(N\)!@"``!)`<E,
MB=%(P>L;2,'A!4@)V8G)20')3(GI2<'M`DC!X1Y$B<M,">E)"<A)B<U!B<Y-
M(=!-(>5-"<5-C40%`$B)V$F)W4G![1M(P>`%3`GHB<!)`<!,B=!)P>H"2,'@
M'D6)Q4D)PKC6P6+*3#'11#')3(N,)``"``!)`<%-`>%-B>Q,`<E-B>E)P>P;
M2<'A!4T)X4F)W$6)R4P!R4G!Y!Y)`<=(P>L"08G)3`GC38G418G223'<13'$
M3(N$)`@"``!)`<!-`?!-B<Y-`>!-B<Q)P>X;2<'D!4T)]$V)[DG![0))P>8>
M18GD30GU3(NT)!`"``!-`>!%B<1)`<9-`?))B=Y-,>Y!,<Y,B>%-`?)-B>9(
MP>$%2<'N&TP)\8G)20'*3(G)2<'I`DC!X1Y%B=9)"<F)V4R)ZTPQRTP!^4R+
MO"0@`@``1#'#3(T$&4R)\4R)\TC!X05(P>L;20''2`G9B<E)`<A,B>%)P>P"
M2,'A'D2)PTP)X46)[$V)S46)R4DQS4T!_$R+O"0X`@``13'53XT4+$F)W$F)
MW4G![1M)P>0%30GL38GU2<'N`DG!Y1Y%B>1-">Y,BZPD*`(``$T!XD6)U$D!
MQ4T!Z4F)S8G)33'U13'%38G@30'I38GE2<'@!4G![1M-">A%B<!-`<%)B=A(
MP>L"2<'@'D6)S4D)V$B+G"0P`@``2`'#2`'93(GS3#'#1#'320''3(T4&4R)
MZ4R)ZTC!ZQM(P>$%2`G9B<E)`<I,B>%)P>P"2,'A'D2)TTD)S$2)\4V)QD6)
MP$TQYDP!^4R+O"1(`@``13'.3HT,,4B)V4F)WDD!QTG![AM(P>$%3`GQB<E)
M`<E,B>E)P>T"2,'A'D6)SDP)Z4R+K"1``@``20'%30'H38GE23'-13'53XT4
M*$V)\$V)]4G![1M)P>`%30GH18G`30'"28G82,'K`DG!X!Y%B=5,"<-%B>!)
MB<R)R4T!^$DQW$R+O"18`@``13',3XT,($V)Z$V)[$D!QTG![!M)P>`%30G@
M18G`30'!38GP2<'N`DG!X!Y%B<Q-"?!,B[0D4`(``$D!QDP!\4F)WDTQQD4Q
MUDZ-%#%,B>%-B>9)P>X;2,'A!4P)\8G)20'*3(GI2<'M`DC!X1Y%B=9)"<V)
MV4R)PT6)P$PQZTP!^4R+O"1H`@``1#'+3(T,&4R)\4R)\TC!ZQM(P>$%2`G9
MB<E)`<E,B>%)P>P"2,'A'D2)RTP)X4R+I"1@`@``20'$30'@38GL20''23',
M13'43XT4($F)V$F)W$G![!M)P>`%30G@18G`30'"38GP2<'N`DG!X!Y%B=1-
M"<9%B>A)B<V)R4TQ]4T!^$R+O"1X`@``13'-3XT,*$V)X$V)Y4G![1M)P>`%
M30GH18G`30'!28G82,'K`DG!X!Y%B<U)"=A(BYPD<`(``$@!PT@!V4R)\TPQ
MPT0QTTR-%!E,B>E,B>M(P>L;2,'A!4@)V8G)20'*3(GA2<'L`DC!X1Y$B=-)
M"<Q$B?%-B<9)`<=-,>9,`?E%B<!,B[PDB`(``$4QSDF)V4D!SDB)V4G!Z1M)
M`<=(P>$%3`G)B<E)`<Y,B>E)P>T"2,'A'D6)\4P)Z4R+K"2``@``20'%30'H
M38GE23'-13'53XT4*$V)R$V)S4G![1M)P>`%30GH18G`30'"28G82,'K`DG!
MX!Y%B=5,"<-%B>!)B<R)R4DQW$T!^$4Q]$V)[DT!X$V)[$G![AM)P>0%30GT
M38G.18GD30'@2<'F'DG!Z0)%B<1-"?%,B[0DD`(``$D!QD@#A"28`@``3`'Q
M28G>3`'8B=M-,<Y(`=A%,=9-B>),`?%-B>9)P>(%2<'N&TT)\D0!T4V)ZDG!
M[0))P>(>B<E-"=5-B<I$`TH@33'J00']3(E*($4QT$F)RDR):AA,`<!)B<A)
MP>H;`>E)P>`%2(E*"$T)T$0!P$B)`DR)X$G![`)(P>`>3`G@`?!(BS5(;@``
M2(E"$$B+A"2H`@``2"L&=11(@<2X`@``6UY?74%<05U!7D%?P^B.!0``D&9F
M+@\?A```````9I!(@^PHN8````#H,@8``+J)J\WON?[<NIA(B5`(2('JF<GZ
M*TC'``$C16=(B4@02,=`&'94,A!(B5`@2,=`*`````!(QT`P`````,=`>```
M``!(@\0HPV8N#Q^$``````!!54%455=64TB#["A)8\!,BT$H2(G/2(G32(M7
M,$F)Q$B-=SA!C0S`3#G!2(E/*$AC3WA(@](`2,'H'4@!T$B)1S"%R71(O4``
M``!(B=HIS40YY4$/3^Q(`?%,8^U-B>CHFP4``(M'>`'HB4=X@_A`=`Y(@\0H
M6UY?74%<05W#D$B)^4$I[$P!Z^B2Z/__08/\/P^.V````$&#[$!%B>5!P>T&
M08UM`4C!Y09(`=UFD/,/;P-(B?E(@\-`#Q$&\P]O2]`/$4X0\P]O4^`/$58@
M\P]O6_`/$5XPZ$+H__](.>MUS4'!Y09%*>Q)8\1(@_@(<EU(BU4`2(U."$B#
MX?A(B5<X2(M4!?A(B50&^$@ISD@!\$@I]4B#X/A(@_@(<AA(@^#X,=),BT05
M`$R)!!%(@\((2#G"<NY$B6=X2(/$*%M>7UU!7$%=PP\?@`````"H!'4D2(7`
M=-\/ME4`B%<XJ`)TU`^W5`7^9HE4!O[KR$B)W>ER____BU4`B5<XBU0%_(E4
M!OSKL`\?A```````05155U932(/L($B+0BB+<BB+>C!(P>@##\Z#X#\/STB)
MRXU(`4B828G4QD0".(!(C6HX2&/!2`'H@_DX#XY.`0``0;A`````02G(=!`Q
MTHG1@\(!Q@0(`$0YPG+R3(GAZ"OG__])QT0D.`````!)QT0D0`````!(QT40
M`````$C'11@`````2,=%(`````!(QT4H`````$C'13``````08E\)'!,B>%!
MB70D=.C?YO__28L$)$R)X4C!Z!B(`TF+!"1(P>@0B$,!28L$)(AC`DF+!"2(
M0P-)BT0D"$C!Z!B(0P1)BT0D"$C!Z!"(0P5)BT0D"(AC!DF+1"0(B$,'28M$
M)!!(P>@8B$,(28M$)!!(P>@0B$,)28M$)!"(8PI)BT0D$(A#"TF+1"082,'H
M&(A##$F+1"082,'H$(A##4F+1"08B&,.28M$)!B(0P])BT0D($C!Z!B(0Q!)
MBT0D($C!Z!"(0Q%)BT0D((AC$DF+1"0@B$,32(/$(%M>7UU!7.EJ`@``9BX/
M'X0``````+HX````13')*<J#^@AR28G12,<``````$C'1`CX`````$B-2`A(
M@^'X2"G(`<*#XOB#^@@/@M7^__^#XO@QP$&)P(/`"$Z)#`$YT'+RZ;W^__]F
M#Q]$``#VP@1U(X72#X2J_O__Q@``]L("#X2>_O__,<EFB4P0_NF2_O__#Q\`
MQP``````QT00_`````#I?/[__Y"0D)"0D)"0D)"0D)#_)=K-``"0D/\ERLT`
M`)"0_R6ZS0``D)#_):K-``"0D/\EFLT``)"0_R6*S0``D)#_)7K-``"0D/\E
M:LT``)"0_R5:S0``D)#_)4K-``"0D/\E.LT``)"0_R4JS0``D)#_)1K-``"0
MD/\E"LT``)"0_R7ZS```D)#_)>K,``"0D/\EVLP``)"0_R7*S```D)#_);K,
M``"0D/\EJLP``)"0_R7ZR@``D)"0D)"0D)"0D/\EZLL``)"0D)"0D)"0D)#_
M)>++``"0D)"0D)"0D)"045!(/0`0``!(C4PD&'(92('I`!```$B#"0!(+0`0
M``!(/0`0``!WYT@IP4B#"0!86<.0D)"0D)"0D)"0D)"0D/\E>LH``)"0D)"0
MD)"0D)#_)7+*``"0D)"0D)"0D)"0_R5RR@``D)"0D)"0D)"0D/\E:LH``)"0
MD)"0D)"0D)#_)6+*``"0D)"0D)"0D)"0_R5:R@``D)"0D)"0D)"0D/\E:LH`
M`)"0D)"0D)"0D)#_)6K*``"0D)"0D)"0D)"0_R5BR@``D)"0D)"0D)"0D/\E
M>LH``)"0D)"0D)"0D)#_)7+*``"0D)"0D)"0D)"0_R5JR@``D)"0D)"0D)"0
MD/\E8LH``)"0D)"0D)"0D)#_)5K*``"0D)"0D)"0D)"0_R52R@``D)"0D)"0
MD)"0D/\E2LH``)"0D)"0D)"0D)#_)4+*``"0D)"0D)"0D)"0_R4ZR@``D)"0
MD)"0D)"0D/\E,LH``)"0D)"0D)"0D)#_)2K*``"0D)"0D)"0D)"0_R4BR@``
MD)"0D)"0D)"0D/\E&LH``)"0D)"0D)"0D)#_)1+*``"0D)"0D)"0D)"0_R4*
MR@``D)"0D)"0D)"0D/\E&LH``)"0D)"0D)"0D)#_)1+*``"0D)"0D)"0D)"0
M_R4*R@``D)"0D)"0D)"0D/\E`LH``)"0D)"0D)"0D)#_)?K)``"0D)"0D)"0
MD)"0_R7RR0``D)"0D)"0D)"0D/\EZLD``)"0D)"0D)"0D)#_)>+)``"0D)"0
MD)"0D)"0_R7:R0``D)"0D)"0D)"0D/\ETLD``)"0D)"0D)"0D)#_)<K)``"0
MD)"0D)"0D)"0_R7"R0``D)"0D)"0D)"0D/\ENLD``)"0D)"0D)"0D)#_);+)
M``"0D)"0D)"0D)"0_R6JR0``D)"0D)"0D)"0D/\EHLD``)"0D)"0D)"0D)!(
M@^PH,=+H50```/\5+\@``)"0D)"0D)"0D)"0D)"0D,.0D)"0D)"0D)"0D)"0
MD)##D)"0D)"0D)"0D)"0D)"0PY"0D)"0D)"0D)"0D)"0D,.0D)"0D)"0D)"0
MD)"0D)!64TB#["A(B<Y(B=.Y"````.C*!```2(7;#X01`@``2+H`````50$`
M`$B)<T!(C36,____2(T-50<``$B)D_````!(C15?!P``2(E34$B+%93'``!(
MB;.`````2(TU;O___TB)2TA(C0US____2(E3&$B-%8BD``!(B;.(````2(TU
M:O___TB)BY````!(N4@!``"["P``2(FSF````$B--8O]__](B5,X2(T5(/W_
M_TB)2PA(C0T5_O__2(ES($B--?H#``!(B5,H2(T5'P0``$C'`P````#'0Q`&
M````2(E+,$B)<WA(B9,8`0``2(7`#X2]````2(N`X````$B%P`^$K0```(L5
M`Z0``(72#X6?````2(,]XQ,````/A`T"``!(@SW=$P````^$;P(``$B#/=<3
M````#X11`@``2(,]T1,````/A#,"``!(@SW+$P````^$%0(``$B#/<43````
M#X3W`0``2(,]OQ,````/A-D!``!(@SVY$P````^$NP$``&8/;P5S$P``#Q$`
M9@]O#7@3```/$4@09@]O%7P3```/$5`@9@]O'8`3```/$5@P,<GHA0,``#')
M2(F#Z````$B-!143``!(B4-82(T%:A,``$B)0V!(C07_H@``2(E#:$B-!=2D
M``!(B4-P2(T%F6<``$B)@R`!``!(C07'9P``2(F#*`$``$B-!<VR__](B8,P
M`0``Z)$"``"X`0```$B#Q"A;7L,/'T0``$B)<$!(C36%_?__2+H`````50$`
M`$B-#5P%``!(B;"`````2(TU=OW__TB)D/````!(C14H!0``2(FPB````$B-
M-6K]__](B5!(2(L59\4``$B)2%!(BXA``0``2(FPD````$B--5;]__](B;"8
M````2(TU6*(``$B)"DB-%7[[__](C0T7^___2(EP.$B^2`$``+L+``!(B7`(
M2(TU_OO__TB)4"!(C17C`0``2(E(*$B-#0@"``!(QP``````QT`0!@```$B)
M<#!(B5!X2(F(&`$``$B%P`^$IO[__TB)PTB+@.````!(A<`/A>;]___ICO[_
M_V8N#Q^$``````!(BQ!(B17&$0``Z>3]__^02(M0.$B)%>T1``#I-?[__TB+
M4#!(B175$0``Z1?^__](BU`H2(D5O1$``.GY_?__2(M0($B)%:41``#IV_W_
M_TB+4!A(B16-$0``Z;W]__](BU`02(D5=1$``.F?_?__2(M0"$B)%5T1``#I
M@?W__TR+!6FA``"+%6NA``!(BPULH0``Z=<````/'X``````2(/L.(/Z`G1W
M=U6%T@^$?0```#'`387`2(T5P?___TB)#3JA```/E,!,B04@H0``QP4>H0``
M`0```(D%`*$``.CK````2(/X_TB)!?B@```/E<`/ML!(@\0XPP\?1```N`$`
M``"#^@-U[+H#````2(/$..E8````#Q^$``````"Z`@```$B#Q#CI0@```&:0
M,=+H.0```(7`=+M(BPVFH```B40D+.A5````BT0D+$C'!8Z@``#_____ZYH/
M'T``Z3O___^0D)"0D)"0D)"0D+@!````PY"0D)"0D)"0D)##D)"0D)"0D)"0
MD)"0D)"0_R4RPP``D)"0D)"0D)"0D/\E.L,``)"0D)"0D)"0D)#_)3+#``"0
MD)"0D)"0D)"0_R6RPP``D)"0D)"0D)"0D$%5051(@^PH3(TM,:```$F)S$B)
MT4R)ZNCC^O__3(GJ3(GA2(/$*$%<05WI`````/\EZL(``)"0D)"0D)"0D)#_
M);K$``"0D`\?A```````055!5%932(/L6(F,)(````!(B50D.$R)1"0PZ-#W
M__](C50D.$B+'81@``!,C40D,$B-C"2`````2(L#2(E$)$@QP.BX]O__2(L5
M46```$B+#3I@``!,C07S]O__Z`[W__](BP4'8```2(LU$&```(`X`'4SZ&[V
M__](B04GGP``2(G!2(7`#X2,`0``Z$[V__](BS7G7P``2(L.Z-_V___&@"$*
M````2(L.Z-#V__](BPZ`B'($```"Z,'V__^`B'($```!BX0D@````(U(!(UP
M`TACR4C!X0/H.?;__TR+5"0X28G!2(D%LIX``$F+`DF)`4B-!58A``!)B4$(
M2(L%H@X``$F)01!(C05#(0``28E!&(N$)(````"#^`$/CNH```!)@\((@\`#
MN@0````/'X``````38L"2&/*@\(!28/""$V)!,DYT'7K2)A(BPU2G@``08GP
M2(T5H,3__TG'!,$`````2,=$)"``````Z%+U__]!B<2%P'1.2(L-))X``$R-
M;"1$Z$KU__],B>I(C0V3'@``Z'.^__](A<!T.DB+#?^=``#H(O7__^A%]?__
M2(M$)$A(*P-U9D2)X$B#Q%A;7D%<05W#2(L-UIT``.CI]/__08G$ZZ%(BPVU
MG0``2(7)=`?H$]G__^NS2(T-XQT``$R)ZN@2OO__2(D%DYT``$B)P4B%P'25
MZ]FX!````.DT____N0$```#H/?;__^CX]?__D)"0D)"0D)#IJ[W__Y"0D)"0
MD)"0D)"0__________^@4D```0``````````````__________\`````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M`````````````````````````````````````````````````````````/__
M__\`````V')```$`````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M```````````````````````````````````````````````````````````E
M<R\E<P`O<')O8R]S96QF+V5X90!$>6YA3&]A9&5R.CIB;V]T7T1Y;F%,;V%D
M97(``$EN=&5R;F%L<SHZ4$%2.CI"3T]4`"X`4$%27U1%35``4$%27U!23T=.
M04U%`#H`)7,E<R5S`"\`4$523#5,24(`4$523$Q)0@!015),-4]05`!015),
M24\`4$%27TE.251)04Q)6D5$`%!!4E]34$%73D5$`%!!4E]#3$5!3@!005)?
M1$5"54<`4$%27T-!0TA%`%!!4E]'3$]"04Q?1$5"54<`4$%27T=,3T)!3%]4
M35!$25(`4$%27U1-4$1)4@!005)?1TQ/0D%,7U1%35``4$%27T=,3T)!3%]#
M3$5!3@`Q`%!!5$@`55-%4@!365-414T`+W1M<`!435!$25(`5$5-4$1)4@!4
M14U0`%1-4`!54T523D%-10`E,#)X`"5S)7,E<R5S`'!A<BT`)7,Z(&-R96%T
M:6]N(&]F('!R:79A=&4@<W5B9&ER96-T;W)Y("5S(&9A:6QE9"`H97)R;F\]
M)6DI"@```````"5S.B!P<FEV871E('-U8F1I<F5C=&]R>2`E<R!I<R!U;G-A
M9F4@*'!L96%S92!R96UO=F4@:70@86YD(')E=')Y('EO=7(@;W!E<F%T:6]N
M*0H`<&%R;"YE>&4`+G!A<@`E<R5S8V%C:&4M)7,E<P`E<R5S=&5M<"TE=25S
M`"5S)7-T96UP+25U+25U)7,`&`!P97)L`#``4$%2.CI086-K97(Z.E9%4E-)
M3TX`,2XP-38```````!5;F%B;&4@=&\@8W)E871E(&-A8VAE(&1I<F5C=&]R
M>0``````````)7,Z(&-R96%T:6]N(&]F('!R:79A=&4@8V%C:&4@<W5B9&ER
M96-T;W)Y("5S(&9A:6QE9"`H97)R;F\])6DI"@`M90`M+0``<&5R;'AS:2YC
M`````````&5V86P@)V5X96,@+W5S<B]B:6XO<&5R;"`@+5,@)#`@)'LQ*R(D
M0")])PII9B`P.R`C(&YO="!R=6YN:6YG('5N9&5R('-O;64@<VAE;&P*<&%C
M:V%G92!?7W!A<E]P;#L*;7D@*"1005)?34%'24,L("1&24Q%7V]F9G-E=%]S
M:7IE+"`D0T%#2$5?;6%R:V5R+"`D8V%C:&5?;F%M95]S:7IE*3L*0D5'24X@
M>PHD4$%27TU!1TE#(#T@(EQN4$%2+G!M7&XB.PHD1DE,15]O9F9S971?<VEZ
M92`](#0[("`@(R!P86-K*").(BD*)&-A8VAE7VUA<FME<B`](")<,$-!0TA%
M(CL*)&-A8VAE7VYA;65?<VEZ92`](#0P.PI]"G-U8B!F:6YD7W!A<E]M86=I
M8PI["FUY("@D9F@I(#T@0%\["FUY("1C:'5N:U]S:7IE(#T@-C0@*B`Q,#(T
M.PIM>2`D8G5F.PIM>2`D<VEZ92`]("US("1F:#L*;7D@)'!O<R`]("1S:7IE
M("T@)'-I>F4@)2`D8VAU;FM?<VEZ93L@("`@("`C($Y/5$4Z("1P;W,@:7,@
M82!M=6QT:7!L92!O9B`D8VAU;FM?<VEZ90IW:&EL92`H)'!O<R`^/2`P*2![
M"G-E96L@)&9H+"`D<&]S+"`P.PIR96%D("1F:"P@)&)U9BP@)&-H=6YK7W-I
M>F4@*R!L96YG=&@H)%!!4E]-04=)0RD["FEF("@H;7D@)&D@/2!R:6YD97@H
M)&)U9BP@)%!!4E]-04=)0RDI(#X](#`I('L*)'!O<R`K/2`D:3L*<F5T=7)N
M("1P;W,["GT*)'!O<R`M/2`D8VAU;FM?<VEZ93L*?0IR971U<FX@+3$["GT*
M;7D@*"1P87)?=&5M<"P@)'!R;V=N86UE+"!`=&UP9FEL92P@)4UO9'5L94-A
M8VAE*3L*14Y$('L@:68@*"1%3E9[4$%27T-,14%.?2D@>PIR97%U:7)E($9I
M;&4Z.E1E;7`["G)E<75I<F4@1FEL93HZ0F%S96YA;64["G)E<75I<F4@1FEL
M93HZ4W!E8SL*;7D@)'1O<&1I<B`]($9I;&4Z.D)A<V5N86UE.CID:7)N86UE
M*"1P87)?=&5M<"D["F]U=',H<7%;4F5M;W9I;F<@9FEL97,@:6X@(B1P87)?
M=&5M<")=*3L*1FEL93HZ1FEN9#HZ9FEN9&1E<'1H*'-U8B![("@@+60@*2`_
M(')M9&ER(#H@=6YL:6YK('TL("1P87)?=&5M<"D["G)M9&ER("1P87)?=&5M
M<#L*:68@*"UD("1P87)?=&5M<"`F)B`D7D\@;F4@)TU35VEN,S(G*2!["FUY
M("1T;7`@/2!N97<@1FEL93HZ5&5M<"@*5$5-4$Q!5$4@/3X@)W1M<%A86%A8
M)RP*1$E2(#T^($9I;&4Z.D)A<V5N86UE.CID:7)N86UE*"1T;W!D:7(I+`I3
M549&25@@/3X@)RYC;60G+`I53DQ)3DL@/3X@,"P**3L*;7D@)&9I;&5N86UE
M(#T@)'1M<"T^9FEL96YA;64["G!R:6YT("1T;7`@/#PB+BXN(CL*>#TQ.R!W
M:&EL92!;(%PD>"`M;'0@,3`@73L@9&\*<FT@+7)F("<D<&%R7W1E;7`G"FEF
M(%L@7"$@+60@)R1P87)?=&5M<"<@73L@=&AE;@IB<F5A:PIF:0IS;&5E<"`Q
M"G@]8&5X<'(@7"1X("L@,6`*9&]N90IR;2`G)&9I;&5N86UE)PHN+BX*8VQO
M<V4@)'1M<#L*8VAM;V0@,#<P,"P@)&9I;&5N86UE.PIM>2`D8VUD(#T@(B1F
M:6QE;F%M92`^+V1E=B]N=6QL(#(^)C$@)B(["G-Y<W1E;2@D8VUD*3L*;W5T
M<RAQ<5M3<&%W;F5D(&)A8VMG<F]U;F0@<')O8V5S<R!T;R!P97)F;W)M(&-L
M96%N=7`Z("1F:6QE;F%M95TI.PI]"GT@?0I"14=)3B!["DEN=&5R;F%L<SHZ
M4$%2.CI"3T]4*"D@:68@9&5F:6YE9"`F26YT97)N86QS.CI005(Z.D)/3U0[
M"F5V86P@>PI?<&%R7VEN:71?96YV*"D["FUY("1Q=6EE="`]("$D14Y6>U!!
M4E]$14)51WT["FUY("5S>7,@/2`H"G!A=&A?<V5P("`@(#T^("@D7D\@/7X@
M+UY-4U=I;B\@/R`G.R<@.B`G.B<I+`I?97AE("`@("`@("`]/B`H)%Y/(#U^
M("]>*#\Z35-7:6Y\3U,R?&-Y9W=I;BDO(#\@)RYE>&4G(#H@)R<I+`I?9&5L
M:6T@("`@("`]/B`H)%Y/(#U^("]>35-7:6Y\3U,R+R`_("=<7"<@.B`G+R<I
M+`HI.PI?<V5T7W!R;V=N86UE*"D["F]U=',H<7%;7"1P<F]G;F%M92`]("(D
M<')O9VYA;64B72D["E]S971?<&%R7W1E;7`H*3L*;W5T<RAQ<5M<)$5.5GM0
M05)?5$5-4'T@/2`B)$5.5GM005)?5$5-4'TB72D["FUY("@D<W1A<G1?<&]S
M+"`D9&%T85]P;W,I.PI-04=)0SH@>PIL;V-A;"`D4TE'>U]?5T%23E]??2`]
M('-U8B![?3L*=6YL97-S("AO<&5N(%]&2"P@)SPZ<F%W)RP@)'!R;V=N86UE
M*2!["F]U=',H<7%;0V%N)W0@<F5A9"!F<F]M(&9I;&4@(B1P<F]G;F%M92)=
M*3L@(",@9&]N)W0@=7-E("0A(&AE<F4@87,@:70@<F5Q=6ER97,@17)R;F\N
M<&T*;&%S="!-04=)0SL*?0IM>2`D;6%G:6-?<&]S(#T@9FEN9%]P87)?;6%G
M:6,H*E]&2"D["FEF("@D;6%G:6-?<&]S(#P@,"D@>PIO=71S*'%Q6T-A;B=T
M(&9I;F0@;6%G:6,@<W1R:6YG("(D4$%27TU!1TE#(B!I;B!F:6QE("(D<')O
M9VYA;64B72D["FQA<W0@34%'24,["GT*;W5T<R@B1F]U;F0@4$%2(&UA9VEC
M(&%T('!O<VET:6]N("1M86=I8U]P;W,B*3L*<V5E:R!?1D@L("1M86=I8U]P
M;W,@+2`D1DE,15]O9F9S971?<VEZ92P@,#L*<F5A9"!?1D@L("1B=68L("1&
M24Q%7V]F9G-E=%]S:7IE.PIM>2`D;V9F<V5T(#T@=6YP86-K*").(BP@)&)U
M9BD["F]U=',H(D]F9G-E="!F<F]M('-T87)T(&]F($9)3$5S(&ES("1O9F9S
M970B*3L*<V5E:R!?1D@L("1M86=I8U]P;W,@+2`D1DE,15]O9F9S971?<VEZ
M92`M("1O9F9S970L(#`["B1D871A7W!O<R`]('1E;&P@7T9(.PIM>2`E<F5Q
M=6ER95]L:7-T.PIR96%D(%]&2"P@)&)U9BP@-#L@("`@("`@("`@("`@("`@
M("`@("`@("`@("`C(')E860@=&AE(&9I<G-T(")&24Q%(@IW:&EL92`H)&)U
M9B!E<2`B1DE,12(I('L*<F5A9"!?1D@L("1B=68L(#0["G)E860@7T9(+"`D
M8G5F+"!U;G!A8VLH(DXB+"`D8G5F*3L*;7D@)&9U;&QN86UE(#T@)&)U9CL*
M;W5T<RAQ<5M5;G!A8VMI;F<@1DE,12`B)&9U;&QN86UE(BXN+ETI.PIM>2`D
M8W)C(#T@*"`D9G5L;&YA;64@/7X@<WQ>*%MA+69<9%U[.'TI+WQ\("D@/R`D
M,2`Z('5N9&5F.PIM>2`H)&)A<V5N86UE+"`D97AT*2`]("@D8G5F(#U^(&U\
M*#\Z+BHO*3\H+BHI*%PN+BHI?"D["G)E860@7T9(+"`D8G5F+"`T.PIR96%D
M(%]&2"P@)&)U9BP@=6YP86-K*").(BP@)&)U9BD["FEF("AD969I;F5D*"1E
M>'0I(&%N9"`D97AT("%^("]<+B@_.G!M?'!L?&EX?&%L*20O:2D@>PIM>2`D
M9FEL96YA;64@/2!?<V%V95]A<R@B)&-R8R1E>'0B+"`D8G5F+"`P-S4U*3L*
M)%!!4CHZ2&5A=GDZ.D9U;&Q#86-H97LD9G5L;&YA;65](#T@)&9I;&5N86UE
M.PHD4$%2.CI(96%V>3HZ1G5L;$-A8VAE>R1F:6QE;F%M97T@/2`D9G5L;&YA
M;64["GT*96QS:68@*"`D9G5L;&YA;64@/7X@;7Q>+S]S:&QI8B]\(&%N9"!D
M969I;F5D("1%3E9[4$%27U1%35!]("D@>PIM>2`D9FEL96YA;64@/2!?<V%V
M95]A<R@B)&)A<V5N86UE)&5X="(L("1B=68L(#`W-34I.PIO=71S*")32$Q)
M0CH@)&9I;&5N86UE7&XB*3L*?0IE;'-E('L*)$UO9'5L94-A8VAE>R1F=6QL
M;F%M97T@/2!["F)U9B`]/B`D8G5F+`IC<F,@/3X@)&-R8RP*;F%M92`]/B`D
M9G5L;&YA;64L"GT["B1R97%U:7)E7VQI<W1[)&9U;&QN86UE?2LK.PI]"G)E
M860@7T9(+"`D8G5F+"`T.PI]"FQO8V%L($!)3D,@/2`H<W5B('L*;7D@*"1S
M96QF+"`D;6]D=6QE*2`]($!?.PIR971U<FX@:68@<F5F("1M;V1U;&4@;W(@
M(21M;V1U;&4["FUY("1I;F9O(#T@)$UO9'5L94-A8VAE>R1M;V1U;&5].PIR
M971U<FX@=6YL97-S("1I;F9O.PID96QE=&4@)')E<75I<F5?;&ES='LD;6]D
M=6QE?3L*)$E.0WLD;6]D=6QE?2`]("(O;&]A9&5R+R1I;F9O+R1M;V1U;&4B
M.PII9B`H)$5.5GM005)?0TQ%04Y](&%N9"!D969I;F5D*"9)3SHZ1FEL93HZ
M;F5W*2D@>PIM>2`D9F@@/2!)3SHZ1FEL92T^;F5W7W1M<&9I;&4@;W(@9&EE
M(")#86XG="!C<F5A=&4@=&5M<"!F:6QE.B`D(2(["B1F:"T^8FEN;6]D92@I
M.PHD9F@M/G!R:6YT*"1I;F9O+3Y[8G5F?2D["B1F:"T^<V5E:R@P+"`P*3L*
M<F5T=7)N("1F:#L*?0IE;'-E('L*;7D@)&9I;&5N86UE(#T@7W-A=F5?87,H
M(B1I;F9O+3Y[8W)C?2YP;2(L("1I;F9O+3Y[8G5F?2D["B1I;F9O+3Y[9FEL
M97T@/2`D9FEL96YA;64["F]P96X@;7D@)&9H+"`G/#IR87<G+"`D9FEL96YA
M;64@;W(@9&EE('%Q6T-A;B=T(')E860@(B1F:6QE;F%M92(Z("0A73L*<F5T
M=7)N("1F:#L*?0ID:64@(D)O;W1S=')A<'!I;F<@9F%I;&5D.B!C86XG="!F
M:6YD(&UO9'5L92`D;6]D=6QE(2(["GTL($!)3D,I.PIR97%U:7)E(%A33&]A
M9&5R.PIR97%U:7)E(%!!4CHZ2&5A=GD["G)E<75I<F4@0V%R<#HZ2&5A=GD[
M"G)E<75I<F4@17AP;W)T97(Z.DAE879Y.PI005(Z.DAE879Y.CI?:6YI=%]D
M>6YA;&]A9&5R*"D["G)E<75I<F4@24\Z.D9I;&4["G=H:6QE("AM>2`D9FEL
M96YA;64@/2`H<V]R="!K97ES("5R97%U:7)E7VQI<W0I6S!=*2!["G5N;&5S
M<R`H)$E.0WLD9FEL96YA;65](&]R("1F:6QE;F%M92`]?B`O0E-$4$%.+RD@
M>PII9B`H)&9I;&5N86UE(#U^("]<+G!M8S\D+VDI('L*<F5Q=6ER92`D9FEL
M96YA;64["GT*96QS92!["F1O("1F:6QE;F%M92!U;FQE<W,@)&9I;&5N86UE
M(#U^("]S:71E8W5S=&]M:7IE7"YP;"0O.PI]"GT*9&5L971E("1R97%U:7)E
M7VQI<W1[)&9I;&5N86UE?3L*?0IU;FQE<W,@*"1B=68@97$@(E!+7#`P,UPP
M,#0B*2!["F]U=',H<7%;3F\@>FEP(&9O=6YD(&%F=&5R($9)3$4@<V5C=&EO
M;B!I;B!F:6QE("(D<')O9VYA;64B72D["FQA<W0@34%'24,@.PI]"B1S=&%R
M=%]P;W,@/2`H=&5L;"!?1D@I("T@-#L@("`@("`@("`@("`@("`@(R!S=&%R
M="!O9B!Z:7`*?0IM>2!`<&%R7V%R9W,["FUY("@D;W5T+"`D8G5N9&QE+"`D
M;&]G9F@L("1C86-H95]N86UE*3L*9&5L971E("1%3E9[4$%27T%04%]21553
M17T[(",@<V%N:71I>F4@*%)%55-%(&UA>2!B92!A('-E8W5R:71Y('!R;V)L
M96TI"FEF("@A)'-T87)T7W!O<R!O<B`H)$%21U9;,%T@97$@)RTM<&%R+6]P
M=&EO;G,G("8F('-H:69T*2D@>PIM>2`E9&ES=%]C;60@/2!Q=R@*<"`@(&)L
M:6)?=&]?<&%R"FD@("!I;G-T86QL7W!A<@IU("`@=6YI;G-T86QL7W!A<@IS
M("`@<VEG;E]P87(*=B`@('9E<FEF>5]P87(**3L*:68@*$!!4D=6(&%N9"`D
M05)'5ELP72!E<2`G+2UR975S92<I('L*<VAI9G0@0$%21U8["B1%3E9[4$%2
M7T%04%]2155317T@/2!S:&EF="!`05)'5CL*?0IE;'-E('L@(R!N;W)M86P@
M<&%R;"!B96AA=FEO=7(*;7D@0&%D9%]T;U]I;F,["G=H:6QE("A`05)'5BD@
M>PHD05)'5ELP72`]?B`O7BTH6T%)34]"3&)Q<&EU<U1V72DH+BHI+R!O<B!L
M87-T.PII9B`H)#$@97$@)TDG*2!["G!U<V@@0&%D9%]T;U]I;F,L("0R.PI]
M"F5L<VEF("@D,2!E<2`G32<I('L*979A;"`B=7-E("0R(CL*?0IE;'-I9B`H
M)#$@97$@)T$G*2!["G5N<VAI9G0@0'!A<E]A<F=S+"`D,CL*?0IE;'-I9B`H
M)#$@97$@)T\G*2!["B1O=70@/2`D,CL*?0IE;'-I9B`H)#$@97$@)V(G*2![
M"B1B=6YD;&4@/2`G<VET92<["GT*96QS:68@*"0Q(&5Q("=")RD@>PHD8G5N
M9&QE(#T@)V%L;"<["GT*96QS:68@*"0Q(&5Q("=Q)RD@>PHD<75I970@/2`Q
M.PI]"F5L<VEF("@D,2!E<2`G3"<I('L*;W!E;B`D;&]G9F@L("(^/B(L("0R
M(&]R(&1I92!Q<5M#86XG="!O<&5N(&QO9R!F:6QE("(D,B(Z("0A73L*?0IE
M;'-I9B`H)#$@97$@)U0G*2!["B1C86-H95]N86UE(#T@)#(["GT*<VAI9G0H
M0$%21U8I.PII9B`H;7D@)&-M9"`]("1D:7-T7V-M9'LD,7TI('L*9&5L971E
M("1%3E9[)U!!4E]414U0)WT["FEN:71?:6YC*"D["G)E<75I<F4@4$%2.CI$
M:7-T.PHF>R)005(Z.D1I<W0Z.B1C;60B?2@I('5N;&5S<R!`05)'5CL*)GLB
M4$%2.CI$:7-T.CHD8VUD(GTH)%\I(&9O<B!`05)'5CL*97AI=#L*?0I]"G5N
M<VAI9G0@0$E.0RP@0&%D9%]T;U]I;F,["GT*?0II9B`H)&]U="D@>PI["G)E
M<75I<F4@24\Z.D9I;&4["G)E<75I<F4@07)C:&EV93HZ6FEP.PIR97%U:7)E
M($1I9V5S=#HZ4TA!.PI]"FUY("1P87(@/2!S:&EF="A`05)'5BD["FUY("1Z
M:7`["FEF("AD969I;F5D("1P87(I('L*;W!E;B!M>2`D9F@L("<\.G)A=R<L
M("1P87(@;W(@9&EE('%Q6T-A;B=T(&9I;F0@<&%R(&9I;&4@(B1P87(B.B`D
M(5T["F)L97-S*"1F:"P@)TE/.CI&:6QE)RD["B1Z:7`@/2!!<F-H:79E.CI:
M:7`M/FYE=SL**"`D>FEP+3YR96%D1G)O;49I;&5(86YD;&4H)&9H+"`D<&%R
M*2`]/2!!<F-H:79E.CI::7`Z.D%:7T]+*"D@*0IO<B!D:64@<7%;17)R;W(@
M<F5A9&EN9R!Z:7`@87)C:&EV92`B)'!A<B)=.PI]"FUY("5M971A7W!A<B`]
M(&1O('L*:68@*"1Z:7`@86YD(&UY("1M971A(#T@)'II<"T^8V]N=&5N=',H
M)TU%5$$N>6UL)RDI('L*)&UE=&$@/7X@<R\N*EYP87(Z)"\O;7,["B1M971A
M(#U^(',O7EQ3+BHO+VUS.PHD;65T82`]?B`O7B`@*%M>.ETK*3H@*"XK*20O
M;6<["GT*?3L*:68@*&1E9FEN960@)'!A<BD@>PIO<&5N(&UY("1P:"P@)SPZ
M<F%W)RP@)'!A<B!O<B!D:64@<7%;0V%N)W0@<F5A9"!P87(@9FEL92`B)'!A
M<B(Z("0A73L*;7D@)&)U9CL*<F5A9"`D<&@L("1B=68L(#0["F1I92!Q<5LB
M)'!A<B(@:7,@;F]T(&$@<&%R(&9I;&5=('5N;&5S<R`D8G5F(&5Q(")02UPP
M,#-<,#`T(CL*8VQO<V4@)'!H.PI]"D-R96%T95!A=&@H)&]U="D@.PIM>2`D
M9F@@/2!)3SHZ1FEL92T^;F5W*`HD;W5T+`I)3SHZ1FEL93HZ3U]#4D5!5"@I
M('P@24\Z.D9I;&4Z.D]?5U)/3DQ9*"D@?"!)3SHZ1FEL93HZ3U]44E5.0R@I
M+`HP-S<W+`HI(&]R(&1I92!Q<5M#86XG="!C<F5A=&4@9FEL92`B)&]U="(Z
M("0A73L*)&9H+3YB:6YM;V1E*"D["G-E96L@7T9(+"`P+"`P.PIM>2`D;&]A
M9&5R.PII9B`H9&5F:6YE9"`D9&%T85]P;W,I('L*<F5A9"!?1D@L("1L;V%D
M97(L("1D871A7W!O<SL*?2!E;'-E('L*;&]C86P@)"\@/2!U;F1E9CL*)&QO
M861E<B`](#Q?1D@^.PI]"FEF("@A)$5.5GM005)?5D520D%424U](&%N9"`D
M;&]A9&5R(#U^("]>*#\Z(R%\7$!R96TI+RD@>PIR97%U:7)E(%!!4CHZ1FEL
M=&5R.CI0;V13=')I<#L*4$%2.CI&:6QT97(Z.E!O9%-T<FEP+3YA<'!L>2A<
M)&QO861E<BP@)#`I.PI]"FEF("@D;65T85]P87)[8VQE86Y]*2!["FUY("1P
M87)?8VQE86X@/2`B/3$B.PIM>2`D<&%S<U]P87)?8VQE86X@/2!U8R`B7U]P
M87-S7W!A<E]C;&5A;E]?(CL*)&QO861E<B`]?B!S>R@_/#TD>W!A<W-?<&%R
M7V-L96%N?2DH("LI?0I[)'!A<E]C;&5A;B`N("@B("(@>"`H;&5N9W1H*"0Q
M*2`M(&QE;F=T:"@D<&%R7V-L96%N*2DI?65G.PI]"B1F:"T^<')I;G0H)&QO
M861E<BD*;W(@9&EE('%Q6T5R<F]R('=R:71I;F<@;&]A9&5R('1O("(D;W5T
M(CH@)"%=.PII9B`H)&)U;F1L92D@>PIR97%U:7)E(%!!4CHZ2&5A=GD["E!!
M4CHZ2&5A=GDZ.E]I;FET7V1Y;F%L;V%D97(H*3L*:6YI=%]I;F,H*3L*<F5Q
M=6ER95]M;V1U;&5S*"D["FUY($!I;F,@/2!G<F5P('L@(2]"4T1004XO('T*
M9W)E<"!["B@D8G5N9&QE(&YE("=S:71E)RD@;W(**"1?(&YE("1#;VYF:6<Z
M.D-O;F9I9WMA<F-H;&EB97AP?2!A;F0*)%\@;F4@)$-O;F9I9SHZ0V]N9FEG
M>W!R:79L:6)E>'!]*3L*?2!`24Y#.PIS?"\K?"]\9RP@<WPO)'Q\(&9O<F5A
M8V@@0&EN8SL*;7D@)69I;&5S.PHD9FEL97-[)%]]*RL@9F]R($!$>6YA3&]A
M9&5R.CID;%]S:&%R961?;V)J96-T<RP@=F%L=65S("5)3D,["FUY("1L:6)?
M97AT(#T@)$-O;F9I9SHZ0V]N9FEG>VQI8E]E>'1].R`@("`@("`@(",@6%A8
M(&QI8E]E>'0@=G,@9&QE>'0@/PIM>2`E=W)I='1E;CL*9F]R96%C:"!M>2`D
M:V5Y("AS;W)T(&ME>7,@)69I;&5S*2!["FUY("@D9FEL92P@)&YA;64I.PII
M9B`H9&5F:6YE9"AM>2`D9F,@/2`D4$%2.CI(96%V>3HZ1G5L;$-A8VAE>R1K
M97E]*2D@>PHH)&9I;&4L("1N86UE*2`]("@D:V5Y+"`D9F,I.PI]"F5L<V4@
M>PIF;W)E86-H(&UY("1D:7(@*$!I;F,I('L*:68@*"1K97D@/7X@;7Q>7%$D
M9&ER7$4O*"XJ*21\:2D@>PHH)&9I;&4L("1N86UE*2`]("@D:V5Y+"`D,2D[
M"FQA<W0["GT*:68@*"1K97D@/7X@;7Q>+VQO861E<B];7B]=*R\H+BHI)'PI
M('L*:68@*&UY("1R968@/2`D36]D=6QE0V%C:&5[)#%]*2!["B@D9FEL92P@
M)&YA;64I(#T@*"1R968L("0Q*3L*;&%S=#L*?0II9B`H+68@(B1D:7(O)#$B
M*2!["B@D9FEL92P@)&YA;64I(#T@*"(D9&ER+R0Q(BP@)#$I.PIL87-T.PI]
M"GT*?0I]"FYE>'0@=6YL97-S(&1E9FEN960@)&YA;64["FYE>'0@:68@)'=R
M:71T96Y[)&YA;65]*RL["FYE>'0@:68@(7)E9B@D9FEL92D@86YD("1F:6QE
M(#U^("]<+EQ1)&QI8E]E>'1<120O:3L*;W5T<RAS<')I;G1F*'%Q6U!A8VMI
M;F<@1DE,12`B)7,B+BXN72P@<F5F("1F:6QE(#\@)&9I;&4M/GMN86UE?2`Z
M("1F:6QE*2D["FUY("1C;VYT96YT.PII9B`H<F5F*"1F:6QE*2D@>PHD8V]N
M=&5N="`]("1F:6QE+3Y[8G5F?3L*?0IE;'-E('L*;&]C86P@)"\@/2!U;F1E
M9CL*;W!E;B!M>2`D=&@L("<\.G)A=R<L("1F:6QE(&]R(&1I92!Q<5M#86XG
M="!R96%D("(D9FEL92(Z("0A73L*)&-O;G1E;G0@/2`\)'1H/CL*8VQO<V4@
M)'1H.PI005(Z.D9I;'1E<CHZ4&]D4W1R:7`M/F%P<&QY*%PD8V]N=&5N="P@
M(CQE;6)E9&1E9#XO)&YA;64B*0II9B`A)$5.5GM005)?5D520D%424U](&%N
M9"`D;F%M92`]?B`O7"XH/SIP;7QI>'QA;"DD+VD["E!!4CHZ1FEL=&5R.CI0
M871C:$-O;G1E;G0M/FYE=RT^87!P;'DH7"1C;VYT96YT+"`D9FEL92P@)&YA
M;64I.PI]"B1F:"T^<')I;G0H(D9)3$4B+`IP86-K*"=.)RP@;&5N9W1H*"1N
M86UE*2`K(#DI+`IS<')I;G1F*"(E,#AX+R5S(BP@07)C:&EV93HZ6FEP.CIC
M;VUP=71E0U)#,S(H)&-O;G1E;G0I+"`D;F%M92DL"G!A8VLH)TXG+"!L96YG
M=&@H)&-O;G1E;G0I*2P*)&-O;G1E;G0I"F]R(&1I92!Q<5M%<G)O<B!W<FET
M:6YG(&5M8F5D9&5D($9)3$4@=&\@(B1O=70B.B`D(5T["F]U=',H<7%;5W)I
M='1E;B!A<R`B)&YA;64B72D["GT*?0II9B`H)'II<"D@>PHD>FEP+3YW<FET
M951O1FEL94AA;F1L92@D9F@I(#T]($%R8VAI=F4Z.EII<#HZ05I?3TL*;W(@
M9&EE('%Q6T5R<F]R('=R:71I;F<@>FEP('!A<G0@;V8@(B1O=70B73L*?0HD
M8V%C:&5?;F%M92`]('-U8G-T<B`D8V%C:&5?;F%M92P@,"P@)&-A8VAE7VYA
M;65?<VEZ93L*:68@*"$D8V%C:&5?;F%M92!A;F0@;7D@)&UT:6UE(#T@*'-T
M870H)&]U="DI6SE=*2!["FUY("1C='@@/2!$:6=E<W0Z.E-(02T^;F5W*#$I
M.PIO<&5N(&UY("1T:"P@(CPZ<F%W(BP@)&]U=#L*)&-T>"T^861D9FEL92@D
M=&@I.PIC;&]S92`D=&@["B1C86-H95]N86UE(#T@)&-T>"T^:&5X9&EG97-T
M.PI]"B1C86-H95]N86UE("X](")<,"(@>"`H)&-A8VAE7VYA;65?<VEZ92`M
M(&QE;F=T:"`D8V%C:&5?;F%M92D["B1C86-H95]N86UE("X]("1C86-H95]M
M87)K97(["FUY("1O9F9S970@/2`D9F@M/G1E;&P@*R!L96YG=&@H)&-A8VAE
M7VYA;64I("T@;&5N9W1H*"1L;V%D97(I.PHD9F@M/G!R:6YT*"1C86-H95]N
M86UE+`IP86-K*"=.)RP@)&]F9G-E="DL"B1005)?34%'24,I"B8F("1F:"T^
M8VQO<V4*;W(@9&EE('%Q6T5R<F]R('=R:71I;F<@=')A:6QE<B!O9B`B)&]U
M="(Z("0A73L*8VAM;V0@,#<U-2P@)&]U=#L*97AI=#L*?0I["FQA<W0@=6YL
M97-S(&1E9FEN960@)'-T87)T7W!O<SL*7V9I>%]P<F]G;F%M92@I.PIR97%U
M:7)E(%!!4CL*4$%2.CI(96%V>3HZ7VEN:71?9'EN86QO861E<B@I.PI["G)E
M<75I<F4@1FEL93HZ1FEN9#L*<F5Q=6ER92!!<F-H:79E.CI::7`["GT*;7D@
M)&9H(#T@24\Z.D9I;&4M/FYE=SL@("`@("`@("`@("`@("`@("`@("`@("`@
M("`@(",@07)C:&EV93HZ6FEP(&]P97)A=&5S(&]N(&%N($E/.CI(86YD;&4*
M)&9H+3YF9&]P96XH9FEL96YO*%]&2"DL("=R)RD@;W(@9&EE('%Q6V9D;W!E
M;B@I(&9A:6QE9#H@)"%=.PI!<F-H:79E.CI::7`Z.G-E=$-H=6YK4VEZ92@M
M<R!?1D@I.PIM>2`D>FEP(#T@07)C:&EV93HZ6FEP+3YN97<["B@D>FEP+3YR
M96%D1G)O;49I;&5(86YD;&4H)&9H+"`D<')O9VYA;64I(#T]($%R8VAI=F4Z
M.EII<#HZ05I?3TLH*2D*;W(@9&EE('%Q6T5R<F]R(')E861I;F<@>FEP(&%R
M8VAI=F4@(B1P<F]G;F%M92)=.PI!<F-H:79E.CI::7`Z.G-E=$-H=6YK4VEZ
M92@V-"`J(#$P,C0I.PIP=7-H($!005(Z.DQI8D-A8VAE+"`D>FEP.PHD4$%2
M.CI,:6)#86-H97LD<')O9VYA;65](#T@)'II<#L*;W5T<R@B17AT<F%C=&EN
M9R!Z:7`N+BXB*3L*:68@*&1E9FEN960@)$5.5GM005)?5$5-4'TI('L@(R!S
M:&]U;&0@8F4@<V5T(&%T('1H:7,@<&]I;G0A"F9O<F5A8V@@;7D@)&UE;6)E
M<B`H("1Z:7`M/FUE;6)E<G,@*2!["FYE>'0@:68@)&UE;6)E<BT^:7-$:7)E
M8W1O<GD["FUY("1M96UB97)?;F%M92`]("1M96UB97(M/F9I;&5.86UE.PIN
M97AT('5N;&5S<R`D;65M8F5R7VYA;64@/7X@;7L*7@HO/W-H;&EB+PHH/SHD
M0V]N9FEG.CI#;VYF:6=[=F5R<VEO;GTO*3\**#\Z)$-O;F9I9SHZ0V]N9FEG
M>V%R8VAN86UE?2\I/PHH6UXO72LI"B0*?7@["FUY("1E>'1R86-T7VYA;64@
M/2`D,3L*;7D@)&1E<W1?;F%M92`]($9I;&4Z.E-P96,M/F-A=&9I;&4H)$5.
M5GM005)?5$5-4'TL("1E>'1R86-T7VYA;64I.PII9B`H+68@)&1E<W1?;F%M
M92`F)B`M<R!?(#T]("1M96UB97(M/G5N8V]M<')E<W-E9%-I>F4H*2D@>PIO
M=71S*'%Q6U-K:7!P:6YG("(D;65M8F5R7VYA;64B('-I;F-E(&ET(&%L<F5A
M9'D@97AI<W1S(&%T("(D9&5S=%]N86UE(ETI.PI](&5L<V4@>PIO=71S*'%Q
M6T5X=')A8W1I;F<@(B1M96UB97)?;F%M92(@=&\@(B1D97-T7VYA;64B+BXN
M72D["B@D;65M8F5R+3YE>'1R86-T5&]&:6QE3F%M960H)&1E<W1?;F%M92D@
M/3T@07)C:&EV93HZ6FEP.CI!6E]/2R@I*0IO<B!D:64@<7%;17)R;W(@97AT
M<F%C=&EN9R!Z:7`@;65M8F5R('1O("(D9&5S=%]N86UE(ET["F-H;6]D*#`U
M-34L("1D97-T7VYA;64I(&EF("1>3R!E<2`B:'!U>"(["GT*?0I]"GT*=6YL
M97-S("@D4$%2.CI,:6)#86-H97LD<')O9VYA;65]*2!["F1I92`\/"`B+B(@
M=6YL97-S($!!4D=6.PI5<V%G93H@)#`@6R`M06QI8BYP87(@72!;("U)9&ER
M(%T@6R`M36UO9'5L92!=(%L@<W)C+G!A<B!=(%L@<')O9W)A;2YP;"!="B0P
M(%L@+4)\+6(@72!;+4]O=71F:6QE72!S<F,N<&%R"BX*)$5.5GM005)?4%)/
M1TY!345](#T@)'!R;V=N86UE(#T@)#`@/2!S:&EF="A`05)'5BD["GT*<W5B
M($-R96%T95!A=&@@>PIM>2`H)&YA;64I(#T@0%\["G)E<75I<F4@1FEL93HZ
M0F%S96YA;64["FUY("@D8F%S96YA;64L("1P871H+"`D97AT*2`]($9I;&4Z
M.D)A<V5N86UE.CIF:6QE<&%R<V4H)&YA;64L("@G7"XN*B<I*3L*<F5Q=6ER
M92!&:6QE.CI0871H.PI&:6QE.CI0871H.CIM:W!A=&@H)'!A=&@I('5N;&5S
M<R@M92`D<&%T:"D[(",@;6MP871H(&1I97,@=VET:"!E<G)O<@I]"G-U8B!R
M97%U:7)E7VUO9'5L97,@>PIR97%U:7)E(&QI8CL*<F5Q=6ER92!$>6YA3&]A
M9&5R.PIR97%U:7)E(&EN=&5G97(["G)E<75I<F4@<W1R:6-T.PIR97%U:7)E
M('=A<FYI;F=S.PIR97%U:7)E('9A<G,["G)E<75I<F4@0V%R<#L*<F5Q=6ER
M92!#87)P.CI(96%V>3L*<F5Q=6ER92!%<G)N;SL*<F5Q=6ER92!%>'!O<G1E
M<CHZ2&5A=GD["G)E<75I<F4@17AP;W)T97(["G)E<75I<F4@1F-N=&P["G)E
M<75I<F4@1FEL93HZ5&5M<#L*<F5Q=6ER92!&:6QE.CI3<&5C.PIR97%U:7)E
M(%A33&]A9&5R.PIR97%U:7)E($-O;F9I9SL*<F5Q=6ER92!)3SHZ2&%N9&QE
M.PIR97%U:7)E($E/.CI&:6QE.PIR97%U:7)E($-O;7!R97-S.CI:;&EB.PIR
M97%U:7)E($%R8VAI=F4Z.EII<#L*<F5Q=6ER92!$:6=E<W0Z.E-(03L*<F5Q
M=6ER92!005(["G)E<75I<F4@4$%2.CI(96%V>3L*<F5Q=6ER92!005(Z.D1I
M<W0["G)E<75I<F4@4$%2.CI&:6QT97(Z.E!O9%-T<FEP.PIR97%U:7)E(%!!
M4CHZ1FEL=&5R.CI0871C:$-O;G1E;G0["G)E<75I<F4@871T<FEB=71E<SL*
M979A;"![(')E<75I<F4@0W=D('T["F5V86P@>R!R97%U:7)E(%=I;C,R('T[
M"F5V86P@>R!R97%U:7)E(%-C86QA<CHZ571I;"!].PIE=F%L('L@<F5Q=6ER
M92!!<F-H:79E.CI5;GII<#HZ0G5R<W0@?3L*979A;"![(')E<75I<F4@5&EE
M.CI(87-H.CI.86UE9$-A<'1U<F4@?3L*979A;"![(')E<75I<F4@4&5R;$E/
M.R!R97%U:7)E(%!E<FQ)3SHZ<V-A;&%R('T["F5V86P@>R!R97%U:7)E('5T
M9C@@?3L*?0IS=6(@7W-E=%]P87)?=&5M<"!["FEF("AD969I;F5D("1%3E9[
M4$%27U1%35!](&%N9"`D14Y6>U!!4E]414U0?2`]?B`O*"XK*2\I('L*)'!A
M<E]T96UP(#T@)#$["G)E='5R;CL*?0IF;W)E86-H(&UY("1P871H("@**&UA
M<"`D14Y6>R1??2P@<7<H(%!!4E]435!$25(@5$U01$E2(%1%35!$25(@5$5-
M4"!435`@*2DL"G%W*"!#.EQ<5$5-4"`O=&UP("X@*0HI('L*;F5X="!U;FQE
M<W,@9&5F:6YE9"`D<&%T:"!A;F0@+60@)'!A=&@@86YD("UW("1P871H.PIM
M>2`D=7-E<FYA;64["FUY("1P=W5I9#L*979A;"![*"1P=W5I9"D@/2!G971P
M=W5I9"@D/BD@:68@9&5F:6YE9"`D/CM].PII9B`H(&1E9FEN960H)E=I;C,R
M.CI,;V=I;DYA;64I("D@>PHD=7-E<FYA;64@/2`F5VEN,S(Z.DQO9VEN3F%M
M93L*?0IE;'-I9B`H9&5F:6YE9"`D<'=U:60I('L*)'5S97)N86UE(#T@)'!W
M=6ED.PI]"F5L<V4@>PHD=7-E<FYA;64@/2`D14Y6>U5315).04U%?2!\?"`D
M14Y6>U5315)]('Q\("=365-414TG.PI]"FUY("1S=&UP9&ER(#T@(B1P871H
M)'-Y<WM?9&5L:6U]<&%R+2(N=6YP86-K*")(*B(L("1U<V5R;F%M92D["FUK
M9&ER("1S=&UP9&ER+"`P-S4U.PIM>2`D8V%C:&5?9&ER.PII9B`H)$5.5GM0
M05)?0TQ%04Y]*2!["B1C86-H95]D:7(@/2`B=&5M<"TD)"(["GT*96QS92![
M"F]P96X@;7D@)&9H+"`B/#IR87<B+"`D<')O9VYA;64@;W(@9&EE('%Q6T-A
M;B=T(')E860@(B1P<F]G;F%M92(Z("0A73L*:68@*"AM>2`D;6%G:6-?<&]S
M(#T@9FEN9%]P87)?;6%G:6,H)&9H*2D@/CT@,"D@>PIS965K("1F:"P@)&UA
M9VEC7W!O<PHM("1&24Q%7V]F9G-E=%]S:7IE"BT@;&5N9W1H*"1C86-H95]M
M87)K97(I+"`P.PIM>2`D8G5F.PIR96%D("1F:"P@)&)U9BP@;&5N9W1H*"1C
M86-H95]M87)K97(I.PII9B`H)&)U9B!E<2`D8V%C:&5?;6%R:V5R*2!["G-E
M96L@)&9H+"`D;6%G:6-?<&]S"BT@)$9)3$5?;V9F<V5T7W-I>F4*+2!L96YG
M=&@H)&-A8VAE7VUA<FME<BD*+2`D8V%C:&5?;F%M95]S:7IE+"`P.PIR96%D
M("1F:"P@)&)U9BP@)&-A8VAE7VYA;65?<VEZ93L*)&)U9B`]?B!S+UPP+R]G
M.PHD8V%C:&5?9&ER(#T@(F-A8VAE+21B=68B.PI]"GT*8VQO<V4@)&9H.PI]
M"FEF("@A)&-A8VAE7V1I<BD@>PHD8V%C:&5?9&ER(#T@(G1E;7`M)"0B.PHD
M14Y6>U!!4E]#3$5!3GT@/2`Q.PI]"B1S=&UP9&ER("X]("(D<WES>U]D96QI
M;7TD8V%C:&5?9&ER(CL*;6MD:7(@)'-T;7!D:7(L(#`W-34["B1%3E9[4$%2
M7U1%35!](#T@)'-T;7!D:7(["FQA<W0["GT*)'!A<E]T96UP(#T@)#$@:68@
M)$5.5GM005)?5$5-4'T@86YD("1%3E9[4$%27U1%35!](#U^("\H+BLI+SL*
M?0IS=6(@7W-A=F5?87,@>PIM>2`H)&YA;64L("1C;VYT96YT<RP@)&UO9&4I
M(#T@0%\["FUY("1F=6QL;F%M92`]("(D<&%R7W1E;7`O)&YA;64B.PIU;FQE
M<W,@*"UE("1F=6QL;F%M92D@>PIM>2`D=&5M<&YA;64@/2`B)&9U;&QN86UE
M+B0D(CL*;7D@)&9H.PHH;W!E;B`D9F@L("<^.G)A=R<L("1T96UP;F%M92D*
M)B8@*'!R:6YT("1F:"`D8V]N=&5N=',I"B8F("AC;&]S92`D9F@I"F]R(&1I
M92!Q<5M%<G)O<B!W<FET:6YG("(D=&5M<&YA;64B.B`D(5T["F-H;6]D("1M
M;V1E+"`D=&5M<&YA;64@:68@9&5F:6YE9"`D;6]D93L*<F5N86UE*"1T96UP
M;F%M92P@)&9U;&QN86UE*2!O<B!U;FQI;FLH)'1E;7!N86UE*3L*?0IR971U
M<FX@)&9U;&QN86UE.PI]"G-U8B!?<V5T7W!R;V=N86UE('L*:68@*&1E9FEN
M960@)$5.5GM005)?4%)/1TY!345](&%N9"`D14Y6>U!!4E]04D]'3D%-17T@
M/7X@+R@N*RDO*2!["B1P<F]G;F%M92`]("0Q.PI]"B1P<F]G;F%M92!\?#T@
M)#`["FEF("@D14Y6>U!!4E]414U0?2!A;F0@:6YD97@H)'!R;V=N86UE+"`D
M14Y6>U!!4E]414U0?2D@/CT@,"D@>PHD<')O9VYA;64@/2!S=6)S='(H)'!R
M;V=N86UE+"!R:6YD97@H)'!R;V=N86UE+"`D<WES>U]D96QI;7TI("L@,2D[
M"GT*:68@*"$D14Y6>U!!4E]04D]'3D%-17T@;W(@:6YD97@H)'!R;V=N86UE
M+"`D<WES>U]D96QI;7TI(#X](#`I('L*:68@*&]P96X@;7D@)&9H+"`G/"<L
M("1P<F]G;F%M92D@>PIR971U<FX@:68@+7,@)&9H.PI]"FEF("@M<R`B)'!R
M;V=N86UE)'-Y<WM?97AE?2(I('L*)'!R;V=N86UE("X]("1S>7-[7V5X97T[
M"G)E='5R;CL*?0I]"F9O<F5A8V@@;7D@)&1I<B`H<W!L:70@+UQ1)'-Y<WMP
M871H7W-E<'U<12\L("1%3E9[4$%42'TI('L*;F5X="!I9B!E>&ES=',@)$5.
M5GM005)?5$5-4'T@86YD("1D:7(@97$@)$5.5GM005)?5$5-4'T["B1D:7(@
M/7X@<R]<421S>7-[7V1E;&EM?5Q%)"\O.PHH*"1P<F]G;F%M92`]("(D9&ER
M)'-Y<WM?9&5L:6U])'!R;V=N86UE)'-Y<WM?97AE?2(I+"!L87-T*0II9B`M
M<R`B)&1I<B1S>7-[7V1E;&EM?21P<F]G;F%M921S>7-[7V5X97TB.PHH*"1P
M<F]G;F%M92`]("(D9&ER)'-Y<WM?9&5L:6U])'!R;V=N86UE(BDL(&QA<W0I
M"FEF("US("(D9&ER)'-Y<WM?9&5L:6U])'!R;V=N86UE(CL*?0I]"G-U8B!?
M9FEX7W!R;V=N86UE('L*)#`@/2`D<')O9VYA;64@?'P]("1%3E9[4$%27U!2
M3T=.04U%?3L*:68@*&EN9&5X*"1P<F]G;F%M92P@)'-Y<WM?9&5L:6U]*2`\
M(#`I('L*)'!R;V=N86UE(#T@(BXD<WES>U]D96QI;7TD<')O9VYA;64B.PI]
M"FUY("1P=V0@/2`H9&5F:6YE9"`F0W=D.CIG971C=V0I(#\@0W=D.CIG971C
M=V0H*0HZ("@H9&5F:6YE9"`F5VEN,S(Z.D=E=$-W9"D@/R!7:6XS,CHZ1V5T
M0W=D*"D@.B!@<'=D8"D["F-H;VUP*"1P=V0I.PHD<')O9VYA;64@/7X@<R]>
M*#\]7"Y<+C]<421S>7-[7V1E;&EM?5Q%*2\D<'=D)'-Y<WM?9&5L:6U]+SL*
M)$5.5GM005)?4%)/1TY!345](#T@)'!R;V=N86UE.PI]"G-U8B!?<&%R7VEN
M:71?96YV('L*:68@*"`D14Y6>U!!4E])3DE424%,25I%1'TK*R`]/2`Q("D@
M>PIR971U<FX["GT@96QS92!["B1%3E9[4$%27TE.251)04Q)6D5$?2`](#([
M"GT*9F]R("AQ=R@@4U!!5TY%1"!414U0($-,14%.($1%0E5'($-!0TA%(%!2
M3T=.04U%("D@*2!["F1E;&5T92`D14Y6>R)005)?)%\B?3L*?0IF;W(@*'%W
M+R!435!$25(@5$5-4"!#3$5!3B!$14)51R`O*2!["B1%3E9[(E!!4E\D7R)]
M(#T@)$5.5GLB4$%27T=,3T)!3%\D7R)](&EF(&5X:7-T<R`D14Y6>R)005)?
M1TQ/0D%,7R1?(GT["GT*:68@*"1%3E9[4$%27U1%35!]*2!["F1E;&5T92`D
M14Y6>U!!4E]#3$5!3GT["GT*?0IS=6(@;W5T<R!["G)E='5R;B!I9B`D<75I
M970["FEF("@D;&]G9F@I('L*<')I;G0@)&QO9V9H(")`7UQN(CL*?0IE;'-E
M('L*<')I;G0@(D!?7&XB.PI]"GT*<W5B(&EN:71?:6YC('L*<F5Q=6ER92!#
M;VYF:6<["G!U<V@@0$E.0RP@9W)E<"!D969I;F5D+"!M87`@)$-O;F9I9SHZ
M0V]N9FEG>R1??2P@<7<H"F%R8VAL:6)E>'`@<')I=FQI8F5X<"!S:71E87)C
M:&5X<"!S:71E;&EB97AP"G9E;F1O<F%R8VAE>'`@=F5N9&]R;&EB97AP"BD[
M"GT*<&%C:V%G92!M86EN.PIR97%U:7)E(%!!4CL*=6YS:&EF="!`24Y#+"!<
M)E!!4CHZ9FEN9%]P87(["E!!4BT^:6UP;W)T*$!P87)?87)G<RD["F1I92!Q
M<5MP87(N<&PZ($-A;B=T(&]P96X@<&5R;"!S8W)I<'0@(B1P<F]G;F%M92(Z
M($YO('-U8V@@9FEL92!O<B!D:7)E8W1O<GE<;ET*=6YL97-S("UE("1P<F]G
M;F%M93L*9&\@)'!R;V=N86UE.PI#3U)%.CIE>&ET*"0Q*2!I9B`H)$`@/7XO
M7E]42U]%6$E47UPH*%QD*RE<*2\I.PID:64@)$`@:68@)$`["GT["B0Z.E]?
M15)23U(@/2`D0"!I9B`D0#L*?0I#3U)%.CIE>&ET*"0Q*2!I9B`H)#HZ7U]%
M4E)/4B`]?B]>7U1+7T58251?7"@H7&0K*5PI+RD["F1I92`D.CI?7T524D]2
M(&EF("0Z.E]?15)23U(["C$["E]?14Y$7U\*`````-!/````````````````
M``!H%$$``0``````````````<!1!``$``````````````,A'0``!````````
M``````#`1T```0``````````````^!)!``$``````````````&!'0``!````
M``````````!($T$``0``````````````1T-#.B`H1TY5*2`Q,2XR+C`@,C`R
M,3`W,C@@*$9E9&]R82!#>6=W:6X@,3$N,BXP+3(I`````````````````$=#
M0SH@*$=.52D@,3$N,RXP````````````````````1T-#.B`H1TY5*2`Q,2XS
M+C````````````````````!'0T,Z("A'3E4I(#$Q+C,N,```````````````
M`````$=#0SH@*$=.52D@,3$N,BXP(#(P,C$P-S(X("A&961O<F$@0WEG=VEN
M(#$Q+C(N,"TR*0````````````````!'0T,Z("A'3E4I(#$Q+C(N,"`R,#(Q
M,#<R."`H1F5D;W)A($-Y9W=I;B`Q,2XR+C`M,BD`````````````````1T-#
M.B`H1TY5*2`Q,2XR+C`@,C`R,3`W,C@@*$9E9&]R82!#>6=W:6X@,3$N,BXP
M+3(I`````````````````$=#0SH@*$=.52D@,3$N,BXP(#(P,C$P-S(X("A&
M961O<F$@0WEG=VEN(#$Q+C(N,"TR*0````````````````!'0T,Z("A'3E4I
M(#$Q+C(N,"`R,#(Q,#<R."`H1F5D;W)A($-Y9W=I;B`Q,2XR+C`M,BD`````
M````````````1T-#.B`H1TY5*2`Q,2XR+C`@,C`R,3`W,C@@*$9E9&]R82!#
M>6=W:6X@,3$N,BXP+3(I`````````````````$=#0SH@*$=.52D@,3$N,BXP
M(#(P,C$P-S(X("A&961O<F$@0WEG=VEN(#$Q+C(N,"TR*0``````````````
M``!'0T,Z("A'3E4I(#$Q+C(N,"`R,#(Q,#<R."`H1F5D;W)A($-Y9W=I;B`Q
M,2XR+C`M,BD`````````````````1T-#.B`H1TY5*2`Q,2XR+C`@,C`R,3`W
M,C@@*$9E9&]R82!#>6=W:6X@,3$N,BXP+3(I`````````````````$=#0SH@
M*$=.52D@,3$N,BXP(#(P,C$P-S(X("A&961O<F$@0WEG=VEN(#$Q+C(N,"TR
M*0````````````````!'0T,Z("A'3E4I(#$Q+C,N,```````````````````
M`$=#0SH@*$=.52D@,3$N,RXP```````````````````````````````!````
M:!0!`."P``!`````<!0!`/"P``!`````^!(!`""Q``!`````2!,!`$"Q``!`
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M`````````````````````````````@```!D````<P```')```%)31%.PEOBC
MQ_3%$!]N]GG:7=`Z`0``````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````$```1!````#@``!0$```91``
M``C@``!P$```<1````S@``"`$```8Q$``!#@``!P$0``U1(``"3@``#@$@``
M8!,``#S@``!@$P``.Q0``$C@``!`%```:A8``%S@``!P%@``XA8``'3@``#P
M%@``)!<``'S@```P%P``,Q<``(3@``!`%P``0QD``(C@``!0&0``?!D``*#@
M``"`&0``DAH``*C@``"@&@``DA\``+C@``"@'P``"R```-#@```0(```+2``
M`-C@```P(```)B$``.#@```P(0``X2@``/#@``#P*```;BL```CA``!P*P``
MA2P``!SA``"0+````T,``#SA```00P``9D,``%3A``!P0P``^$0``%SA````
M10``(T<``'#A``#`2@``TDH``(#A``#@2@``X4H``(CA``#P2@``\4H``(SA
M````2P```4L``)#A```02P``$4L``)3A```@2P``T$X``)CA``#03@``Z4X`
M`*3A``#P3@``K$\``*CA``"P3P``M4\``+#A``#`3P``QD\``+3A``#03P``
MT4\``+CA```@4```4%```+SA``!P4```F5(``"SA``"@4@``I5(``,CA````
M``````````````````````````````````````````````````````````$$
M`0`$0@```0````$````!#`<`#$((,`=@!G`%4`3``M````$1"0`1`18`"C`)
M8`AP!U`&P`30`N````$*!``*`1<``S`"P`$3!P`3`08""S`*8`EP"%`"P```
M`1`)`!!B##`+8`IP"5`(P`;0!.`"\````04"``52`3`!!`$`!$(```$````!
M&0H`&0$;`A$P$&`/<`Y0#<`+T`G@`O`!!@(`!C("P`$3!@`3`0<""S`*8`G`
M`M`!$`D`$&(,,`M@"G`)4`C`!M`$X`+P```!!0(`!5(!,`$$`0`$0@```0P&
M``QR"#`'8`;`!-`"X`$9"@`9`340$3`08`]P#E`-P`O0">`"\`$-!P`-8@DP
M"&`'<`;`!-`"X````0L%``MB!S`&P`30`N````$*!0`*H@8P!6`$P`+0```!
M$PH`$P%7``PP"V`*<`E0",`&T`3@`O`!!`$`!$(```$,!P`,0@@P!V`&<`50
M!,`"T````0H&``HR!C`%8`1P`U`"P`$$`0`$0@```0````$````!`````0``
M``$&`P`&0@(P`6````$````!!`$`!&(```$````!`````0````$(`P`(0@3`
M`M````$`````````````````````````````````````````````````````
M`````````````````````````-1#%V,`````T@`!``$````1````$0```"@`
M`0!L``$`L``!`/`H```0(```<%```!A@``!0&0``<"L``#`7``"`&0``H!\`
M`$`7``"@&@``,"$``#`@````10``$$,``'!#``#P%@``V@`!`/```0`+`0$`
M$`$!`!H!`0`G`0$`,P$!`$0!`0!0`0$`7@$!`&L!`0!X`0$`A0$!`)<!`0"A
M`0$`J@$!`+4!`0````$``@`#``0`!0`&``<`"``)``H`"P`,``T`#@`/`!``
M<&%R+F5X90!84U]);G1E<FYA;'-?4$%27T)/3U0`9V5T7W5S97)N86UE7V9R
M;VU?9V5T<'=U:60`;6%I;@!M>5]P87)?<&P`<&%R7V)A<V5N86UE`'!A<E]C
M;&5A;G5P`'!A<E]C=7)R96YT7V5X96,`<&%R7V1I<FYA;64`<&%R7V5N=E]C
M;&5A;@!P87)?9FEN9'!R;V<`<&%R7VEN:71?96YV`'!A<E]M:W1M<&1I<@!P
M87)?<V5T=7!?;&EB<&%T:`!S:&%?9FEN86P`<VAA7VEN:70`<VAA7W5P9&%T
M90!X<U]I;FET````````````````````````````````````````````````
M``````````````````````````````````````````!0$`$````````````\
M&@$`P!(!`/@1`0```````````*`:`0!H%`$`L!(!````````````M!H!`"`5
M`0```````````````````````````#`5`0``````0!4!``````!,%0$`````
M`%P5`0``````:!4!``````!X%0$``````(@5`0``````G!4!``````"P%0$`
M`````+P5`0``````S!4!``````#8%0$``````.05`0``````[!4!``````#X
M%0$```````P6`0``````(!8!```````P%@$``````#P6`0``````1!8!````
M``!0%@$``````%@6`0``````9!8!``````!P%@$``````'P6`0``````A!8!
M``````",%@$``````)@6`0``````I!8!``````"P%@$``````+P6`0``````
MQ!8!``````#,%@$``````-@6`0``````[!8!````````%P$``````!@7`0``
M````(!<!```````L%P$``````#@7`0``````1!<!``````!,%P$``````%@7
M`0``````8!<!``````!L%P$``````'@7`0``````A!<!``````"0%P$`````
M`)P7`0``````J!<!``````"T%P$``````,`7`0````````````````#,%P$`
M`````-P7`0``````[!<!````````&`$``````!@8`0``````,!@!``````!`
M&`$``````%08`0``````9!@!``````!T&`$``````(@8`0``````H!@!````
M``"P&`$``````,08`0``````V!@!``````#H&`$``````/P8`0``````#!D!
M```````@&0$``````#`9`0``````/!D!``````!,&0$`````````````````
M6!D!`````````````````#`5`0``````0!4!``````!,%0$``````%P5`0``
M````:!4!``````!X%0$``````(@5`0``````G!4!``````"P%0$``````+P5
M`0``````S!4!``````#8%0$``````.05`0``````[!4!``````#X%0$`````
M``P6`0``````(!8!```````P%@$``````#P6`0``````1!8!``````!0%@$`
M`````%@6`0``````9!8!``````!P%@$``````'P6`0``````A!8!``````",
M%@$``````)@6`0``````I!8!``````"P%@$``````+P6`0``````Q!8!````
M``#,%@$``````-@6`0``````[!8!````````%P$``````!@7`0``````(!<!
M```````L%P$``````#@7`0``````1!<!``````!,%P$``````%@7`0``````
M8!<!``````!L%P$``````'@7`0``````A!<!``````"0%P$``````)P7`0``
M````J!<!``````"T%P$``````,`7`0````````````````#,%P$``````-P7
M`0``````[!<!````````&`$``````!@8`0``````,!@!``````!`&`$`````
M`%08`0``````9!@!``````!T&`$``````(@8`0``````H!@!``````"P&`$`
M`````,08`0``````V!@!``````#H&`$``````/P8`0``````#!D!```````@
M&0$``````#`9`0``````/!D!``````!,&0$`````````````````6!D!````
M``````````````\`7U]C>&%?871E>&ET```6`%]?97)R;F\````C`%]?9V5T
M<F5E;G0`````+@!?7VUA:6X`````,`!?7VUE;6-P>5]C:&L``$P`7U]S<')I
M;G1F7V-H:P!/`%]?<W1A8VM?8VAK7V9A:6P``%$`7U]S=&%C:U]C:&M?9W5A
M<F0`;`!?9&QL7V-R=#``<P!?:6UP=7)E7W!T<@```(,`86-C97-S``````(!
M8V%L;&]C`````$,!8VQO<V4`1`%C;&]S961I<@``?`%C>6=W:6Y?9&5T86-H
M7V1L;`!^`6-Y9W=I;E]I;G1E<FYA;````)H!9&QL7V1L;&-R=#````"W`65N
M=FER;VX```#4`65X:70``#X"9G!R:6YT9@```$H"9G)E90``L`)G971P:60`
M````NP)G971P=W5I9```R0)G971U:60`````C0-L<V5E:P"/`VQS=&%T`)0#
M;6%L;&]C`````*,#;65M8VAR`````*4#;65M8W!Y`````*<#;65M;6]V90``
M`*L#;6MD:7(`[0-O<&5N``#Q`V]P96YD:7(````$!'!O<VEX7VUE;6%L:6=N
M`````"D$<'1H<F5A9%]A=&9O<FL`````7P1P=&AR96%D7V=E='-P96-I9FEC
M````MP1R96%D``"X!')E861D:7(```"Z!')E861L:6YK``"]!')E86QL;V,`
M``#K!')M9&ER`(`%<W!R:6YT9@```(@%<W1A=```DP5S=')C:'(`````E05S
M=')C;7``````F@5S=')D=7``````I`5S=')L96X`````J05S=')N8VUP````
ML05S=')R8VAR````M05S=')S='(`````NP5S=')T;VL`````+09U;FQI;FL`
M````%P!03%]D;U]U;F1U;7```&T`4$Q?=&AR7VME>0````!K`5!E<FQ?871F
M;W)K7VQO8VL``&P!4&5R;%]A=&9O<FM?=6YL;V-K`````.4!4&5R;%]C<F]A
M:U]N;V-O;G1E>'0``'@"4&5R;%]G971?<W8```"G`E!E<FQ?9W9?9F5T8VAP
M=@```&D#4&5R;%]M9U]S970```#?`U!E<FQ?;F5W6%,`````X0-097)L7VYE
M=UA37V9L86=S``"G!5!E<FQ?<V%F97-Y<VUA;&QO8P````"&!E!E<FQ?<W9?
M<V5T<'8`CP9097)L7W-V7W-E='!V;@````"[!E!E<FQ?<WES7VEN:70S````
M`+P&4&5R;%]S>7-?=&5R;0`Y!V)O;W1?1'EN84QO861E<@```$H'<&5R;%]A
M;&QO8P````!,!W!E<FQ?8V]N<W1R=6-T`````$T'<&5R;%]D97-T<G5C=`!.
M!W!E<FQ?9G)E90!/!W!E<FQ?<&%R<V4`````4`=P97)L7W)U;@``BP)'971-
M;V1U;&5(86YD;&5!````$`$``!`!```0`0``$`$``!`!```0`0``$`$``!`!
M```0`0``$`$``!`!```0`0``$`$``!`!```0`0``$`$``!`!```0`0``$`$`
M`!`!```0`0``$`$``!`!```0`0``$`$``!`!```0`0``$`$``!`!```0`0``
M$`$``!`!```0`0``$`$``!`!```0`0``$`$``!`!```0`0``$`$``!`!```0
M`0``$`$``!`!```0`0``$`$``!`!```0`0``$`$``!`!```0`0``$`$`8WEG
M=VEN,2YD;&P`%!`!`!00`0`4$`$`%!`!`!00`0`4$`$`%!`!`!00`0`4$`$`
M%!`!`!00`0`4$`$`%!`!`!00`0`4$`$`%!`!`!00`0`4$`$`%!`!`!00`0`4
M$`$`%!`!`&-Y9W!E<FPU7S,R+F1L;``H$`$`2T523D5,,S(N9&QL````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````````````````0``P```#```(`.````P```@!````#P``"`&````"`!
M`(````````````````````0``0```&```(`"````>```@`,```"0``"`!```
M`*@``(````````````````````$`"00``%`!``````````````````````$`
M"00``&`!``````````````````````$`"00``'`!````````````````````
M``$`"00``(`!```````````````````!````P`$`@-@``(``````````````
M``````$`"00``)`!``````````````````````$``0````@!`(``````````
M``````````$`"00``*`!``````````````````````$``0```#@!`(``````
M``````````````$`"00``+`!``#0(0$`:`4`````````````."<!`*@(````
M`````````.`O`0!H!0````````````!(-0$`J`@`````````````\#T!`#X`
M`````````````#`^`0!8`@````````````"(0`$`V@4`````````````!@!7
M`$D`3@!%`%@`10```"@````0````(`````$`"`````````$`````````````
M``$````!```&!@8`!`0$``("`@``````0D)"`)V=G0!N;FX`@8&!`+&QL0#_
M__\`<W-S`)^?GP!86%@`LK*R`/W]_0!K:VL`_/S\`-?7UP`6%A8`VMK:`):6
ME@".CHX`1D9&`,O+RP`+"PL`H*"@`/[^_@#)R<D`P<'!``,#`P`1$1$`DY.3
M`+.SLP#N[NX`]/3T`./CXP#EY>4`HJ*B`+JZN@"OKZ\`!P<'`'%Q<0"UM;4`
M]?7U`(F)B0"*BHH`-#0T`.CHZ``U-34`6EI:`+>WMP#8V-@`T-#0`!T='0`,
M#`P`CX^/`+Z^O@!M;6T`"0D)`)N;FP`O+R\`S<W-`-G9V0`9&1D`34U-`$1$
M1`!04%``/#P\`&1D9`#"PL(`,C(R``@("``-#0T`#P\/````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M`````````````````````````````````````````````````````````4%!
M04%!04%!04%!04%!`0$!-4`C!4$Q0T%!1$4N`0$!`0$!`04!/!<7`0$3`0$!
M`0$!`0$5`0$F%0$!#P$!`0$!`0$!$P$1`04U`28!`0$!`0$!`2(2$#$/!0$@
M`0$!`0$!`0$0"B`B%14N(2X!`0$!`1\@(1`B$",C)28G`0$!`240$!`0$!`0
M$!`710$!`4$0$!`0$!`0$!`0$!<!`0$/$!`5$!`0$!`0$!`!`0$!$A`0$Q\0
M$!`0$!`'`0$!`2`0$!`!#Q`0$!`1`0$!`0$'(!`*`0$%$!`0,0$!`0$!`0$!
M`0$!`4$%#P$!`0$!`0$!`0$!`0$!`0$!`0$!`8`!``#``0``X`,``/`#``#P
M`P``\`,``.`#``#``0``@`$``(`!``"``P``@`,````'````!P``C@\``/\?
M```H````(````$`````!``@````````$```````````````!`````0``````
M``$!`0`"`@(``P,#``0$!``%!04`!@8&``<'!P`("`@`"0D)``H*"@`+"PL`
M#`P,``T-#0`.#@X`#P\/`!`0$``1$1$`$A(2`!,3$P`4%!0`%145`!86%@`7
M%Q<`&!@8`!D9&0`:&AH`&QL;`!P<'``='1T`'AX>`!\?'P`@("``(2$A`"(B
M(@`C(R,`)"0D`"4E)0`F)B8`)R<G`"@H*``I*2D`*BHJ`"LK*P`L+"P`+2TM
M`"XN+@`O+R\`,#`P`#$Q,0`R,C(`,S,S`#0T-``U-34`-C8V`#<W-P`X.#@`
M.3DY`#HZ.@`[.SL`/#P\`#T]/0`^/CX`/S\_`$!`0`!!04$`0D)"`$-#0P!$
M1$0`145%`$9&1@!'1T<`2$A(`$E)20!*2DH`2TM+`$Q,3`!-34T`3DY.`$]/
M3P!04%``45%1`%)24@!34U,`5%14`%5550!65E8`5U=7`%A86`!965D`6EI:
M`%M;6P!<7%P`75U=`%Y>7@!?7U\`8&!@`&%A80!B8F(`8V-C`&1D9`!E964`
M9F9F`&=G9P!H:&@`:6EI`&IJ:@!K:VL`;&QL`&UM;0!N;FX`;V]O`'!P<`!Q
M<7$`<G)R`'-S<P!T='0`=75U`'9V=@!W=W<`>'AX`'EY>0!Z>GH`>WM[`'Q\
M?`!]?7T`?GY^`']_?P"`@(``@8&!`(*"@@"#@X,`A(2$`(6%A0"&AH8`AX>'
M`(B(B`")B8D`BHJ*`(N+BP",C(P`C8V-`(Z.C@"/CX\`D)"0`)&1D0"2DI(`
MDY.3`)24E`"5E94`EI:6`)>7EP"8F)@`F9F9`)J:F@";FYL`G)R<`)V=G0">
MGIX`GY^?`*"@H`"AH:$`HJ*B`*.CHP"DI*0`I:6E`*:FI@"GIZ<`J*BH`*FI
MJ0"JJJH`JZNK`*RLK`"MK:T`KJZN`*^OKP"PL+``L;&Q`+*RL@"SL[,`M+2T
M`+6UM0"VMK8`M[>W`+BXN`"YN;D`NKJZ`+N[NP"\O+P`O;V]`+Z^O@"_O[\`
MP,#``,'!P0#"PL(`P\/#`,3$Q`#%Q<4`QL;&`,?'QP#(R,@`R<G)`,K*R@#+
MR\L`S,S,`,W-S0#.SLX`S\_/`-#0T`#1T=$`TM+2`-/3TP#4U-0`U=75`-;6
MU@#7U]<`V-C8`-G9V0#:VMH`V]O;`-S<W`#=W=T`WM[>`-_?WP#@X.``X>'A
M`.+BX@#CX^,`Y.3D`.7EY0#FYN8`Y^?G`.CHZ`#IZ>D`ZNKJ`.OKZP#L[.P`
M[>WM`.[N[@#O[^\`\/#P`/'Q\0#R\O(`\_/S`/3T]`#U]?4`]O;V`/?W]P#X
M^/@`^?GY`/KZ^@#[^_L`_/S\`/W]_0#^_OX`____`#0T-#P\/#P\/#P\/#P\
M/#P\/#P\/#P\/"P,!`0$!`0$7+34U-34U-34U-34W-34U-34U-34U-34M#11
M!`0$!`1DS.ST]/3\]/ST]/3\]/ST_/3\_/ST].S$-%%1!`0$!&3$Y.ST]/3T
M]/3T]/3T]/3T]/3T]/3L[+PT45%1!`0$9,3D[.SLY.SL[.3DY.3L[.3LY.SL
M[.SDO#1145%1!`1DO.3L[*R,K-3$E)2<E(R,E)24G)R<[-R\-%%145%1!&2\
MW.3LY,11E*QDY*1,-"Q<Q,Q1M-3DW+PT45%145%19+S<Y.3LY.11A'3DC&RD
MG&R,O'2LU.3<O#0$!`0$!`1DO-SDY.SDY+Q1C-34C*S,=,S<I%'$Y-R\-`0$
M!`0$!&2\W.3DY.3DW(Q\G+1\A)QLS.2\=,3DW+0T!`0$!`0$7+S<W.3DY.34
MG*1<E(S,/)1DS+R,4=S4M#0$!`0$!`1DM-3<W-SDW(R,K"R\M,Q:Q&2<;)2D
MW-2T-`0$!`0$!%RTU-S<Y-S<=)343*3TC#S4=)1TG$S<S+0T!`0$!`0$9+34
MW-S<Y-QT4?1TA.S,9.1\;*2T;*34M#0$!`0$!`1DM,S<W,Q\C(2T_,S$_/2T
M_+QDS*R$4;RL-`0$!`0$!%RTS-3,C*3<W.S\_/S\_/3\[+3LK'R44:PT!`0$
M!`0$7*S,S&R<]/S\_/S\_/S\_/S\_/S$E(Q1K#0$!`0$!`1<K,2\?.3\_/S\
M_/S\_/S\_/S\_.S,K%&L-`0$!`0$!%RLM$S$_/S\[/3\_/S\_/S\_/S\]+Q1
MQ*PT!`0$!`0$7*RT3-3\_/2TW/S\_/S\_/S\_/S,4<S$I#0$!`0$!`1<K,2D
M4?S\_+2<]/S\_/S\_/S\])11S,2D+`0$!`0$!%RDI%&$[/S\S&2T[/S\_/S\
M_/S4;*S,O*0T!`0$!`0$7*1D=+ST_/S$482<Y/S\_/S\Y(Q1S,S$I#0$!`0$
M!`1<I&R,Y/3\_)Q1S%&4[/S\_/R\=,S,Q+R<+`0$!`0$!%RDK&1<9*2D4:S,
MQ'24Y/3\Y%&\S,S$O)PT!`0$!`0$7*2TO+R,45'$Q,S,K%%\?)1L?+S,Q,2\
MG"P$!`0$!`1<G+S$Q,3$Q,3$Q,3$Q%%145&\Q,3$Q+2<-`0$!`0$!%2<M,3$
MQ,3$Q,3$Q,3$Q,3$Q,3$Q,2\M)PL!`0$!`0$5)R\Q,2\O+R\O+R\O+R\O+R\
MO+R\O+RTG"P$!`0$!`1<G+3$O+2TM+2TM+2TM+2TM+2TM+2TM*R<+`0$!`0$
M!%2$E)R<E)24E)24E)24E)24E)24E)24E(0D!`0$!`0$1%1<7%Q<7%Q<7%Q<
M7%Q<7%Q<7%Q<7%Q<5#0$!`0$!`0````_````'P````\````'`````P````$`
M````````/P```#\````_````/P```#\````_````/P```#\````_````/P``
M`#\````_````/P```#\````_````/P```#\````_````/P```#\````_````
M/P```#\````_````/R@````0````(`````$`"`````````$`````````````
M``$````!```&!@8`!`0$``("`@``````0D)"`)V=G0!N;FX`@8&!`+&QL0#_
M__\`<W-S`)^?GP!86%@`LK*R`/W]_0!K:VL`_/S\`-?7UP`6%A8`VMK:`):6
ME@".CHX`1D9&`,O+RP`+"PL`H*"@`/[^_@#)R<D`P<'!``,#`P`1$1$`DY.3
M`+.SLP#N[NX`]/3T`./CXP#EY>4`HJ*B`+JZN@"OKZ\`!P<'`'%Q<0"UM;4`
M]?7U`(F)B0"*BHH`-#0T`.CHZ``U-34`6EI:`+>WMP#8V-@`T-#0`!T='0`,
M#`P`CX^/`+Z^O@!M;6T`"0D)`)N;FP`O+R\`S<W-`-G9V0`9&1D`34U-`$1$
M1`!04%``/#P\`&1D9`#"PL(`,C(R``@("``-#0T`#P\/````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M`````````````````````````````````````````````````````````4%!
M04%!04%!04%!04%!`0$!-4`C!4$Q0T%!1$4N`0$!`0$!`04!/!<7`0$3`0$!
M`0$!`0$5`0$F%0$!#P$!`0$!`0$!$P$1`04U`28!`0$!`0$!`2(2$#$/!0$@
M`0$!`0$!`0$0"B`B%14N(2X!`0$!`1\@(1`B$",C)28G`0$!`240$!`0$!`0
M$!`710$!`4$0$!`0$!`0$!`0$!<!`0$/$!`5$!`0$!`0$!`!`0$!$A`0$Q\0
M$!`0$!`'`0$!`2`0$!`!#Q`0$!`1`0$!`0$'(!`*`0$%$!`0,0$!`0$!`0$!
M`0$!`4$%#P$!`0$!`0$!`0$!`0$!`0$!`0$!`8`!``#``0``X`,``/`#``#P
M`P``\`,``.`#``#``0``@`$``(`!``"``P``@`,````'````!P``C@\``/\?
M```H````(````$`````!``@````````$```````````````!`````0``````
M``$!`0`"`@(``P,#``0$!``%!04`!@8&``<'!P`("`@`"0D)``H*"@`+"PL`
M#`P,``T-#0`.#@X`#P\/`!`0$``1$1$`$A(2`!,3$P`4%!0`%145`!86%@`7
M%Q<`&!@8`!D9&0`:&AH`&QL;`!P<'``='1T`'AX>`!\?'P`@("``(2$A`"(B
M(@`C(R,`)"0D`"4E)0`F)B8`)R<G`"@H*``I*2D`*BHJ`"LK*P`L+"P`+2TM
M`"XN+@`O+R\`,#`P`#$Q,0`R,C(`,S,S`#0T-``U-34`-C8V`#<W-P`X.#@`
M.3DY`#HZ.@`[.SL`/#P\`#T]/0`^/CX`/S\_`$!`0`!!04$`0D)"`$-#0P!$
M1$0`145%`$9&1@!'1T<`2$A(`$E)20!*2DH`2TM+`$Q,3`!-34T`3DY.`$]/
M3P!04%``45%1`%)24@!34U,`5%14`%5550!65E8`5U=7`%A86`!965D`6EI:
M`%M;6P!<7%P`75U=`%Y>7@!?7U\`8&!@`&%A80!B8F(`8V-C`&1D9`!E964`
M9F9F`&=G9P!H:&@`:6EI`&IJ:@!K:VL`;&QL`&UM;0!N;FX`;V]O`'!P<`!Q
M<7$`<G)R`'-S<P!T='0`=75U`'9V=@!W=W<`>'AX`'EY>0!Z>GH`>WM[`'Q\
M?`!]?7T`?GY^`']_?P"`@(``@8&!`(*"@@"#@X,`A(2$`(6%A0"&AH8`AX>'
M`(B(B`")B8D`BHJ*`(N+BP",C(P`C8V-`(Z.C@"/CX\`D)"0`)&1D0"2DI(`
MDY.3`)24E`"5E94`EI:6`)>7EP"8F)@`F9F9`)J:F@";FYL`G)R<`)V=G0">
MGIX`GY^?`*"@H`"AH:$`HJ*B`*.CHP"DI*0`I:6E`*:FI@"GIZ<`J*BH`*FI
MJ0"JJJH`JZNK`*RLK`"MK:T`KJZN`*^OKP"PL+``L;&Q`+*RL@"SL[,`M+2T
M`+6UM0"VMK8`M[>W`+BXN`"YN;D`NKJZ`+N[NP"\O+P`O;V]`+Z^O@"_O[\`
MP,#``,'!P0#"PL(`P\/#`,3$Q`#%Q<4`QL;&`,?'QP#(R,@`R<G)`,K*R@#+
MR\L`S,S,`,W-S0#.SLX`S\_/`-#0T`#1T=$`TM+2`-/3TP#4U-0`U=75`-;6
MU@#7U]<`V-C8`-G9V0#:VMH`V]O;`-S<W`#=W=T`WM[>`-_?WP#@X.``X>'A
M`.+BX@#CX^,`Y.3D`.7EY0#FYN8`Y^?G`.CHZ`#IZ>D`ZNKJ`.OKZP#L[.P`
M[>WM`.[N[@#O[^\`\/#P`/'Q\0#R\O(`\_/S`/3T]`#U]?4`]O;V`/?W]P#X
M^/@`^?GY`/KZ^@#[^_L`_/S\`/W]_0#^_OX`____`#0T-#P\/#P\/#P\/#P\
M/#P\/#P\/#P\/"P,!`0$!`0$7+34U-34U-34U-34W-34U-34U-34U-34M#11
M!`0$!`1DS.ST]/3\]/ST]/3\]/ST_/3\_/ST].S$-%%1!`0$!&3$Y.ST]/3T
M]/3T]/3T]/3T]/3T]/3L[+PT45%1!`0$9,3D[.SLY.SL[.3DY.3L[.3LY.SL
M[.SDO#1145%1!`1DO.3L[*R,K-3$E)2<E(R,E)24G)R<[-R\-%%145%1!&2\
MW.3LY,11E*QDY*1,-"Q<Q,Q1M-3DW+PT45%145%19+S<Y.3LY.11A'3DC&RD
MG&R,O'2LU.3<O#0$!`0$!`1DO-SDY.SDY+Q1C-34C*S,=,S<I%'$Y-R\-`0$
M!`0$!&2\W.3DY.3DW(Q\G+1\A)QLS.2\=,3DW+0T!`0$!`0$7+S<W.3DY.34
MG*1<E(S,/)1DS+R,4=S4M#0$!`0$!`1DM-3<W-SDW(R,K"R\M,Q:Q&2<;)2D
MW-2T-`0$!`0$!%RTU-S<Y-S<=)343*3TC#S4=)1TG$S<S+0T!`0$!`0$9+34
MW-S<Y-QT4?1TA.S,9.1\;*2T;*34M#0$!`0$!`1DM,S<W,Q\C(2T_,S$_/2T
M_+QDS*R$4;RL-`0$!`0$!%RTS-3,C*3<W.S\_/S\_/3\[+3LK'R44:PT!`0$
M!`0$7*S,S&R<]/S\_/S\_/S\_/S\_/S$E(Q1K#0$!`0$!`1<K,2\?.3\_/S\
M_/S\_/S\_/S\_.S,K%&L-`0$!`0$!%RLM$S$_/S\[/3\_/S\_/S\_/S\]+Q1
MQ*PT!`0$!`0$7*RT3-3\_/2TW/S\_/S\_/S\_/S,4<S$I#0$!`0$!`1<K,2D
M4?S\_+2<]/S\_/S\_/S\])11S,2D+`0$!`0$!%RDI%&$[/S\S&2T[/S\_/S\
M_/S4;*S,O*0T!`0$!`0$7*1D=+ST_/S$482<Y/S\_/S\Y(Q1S,S$I#0$!`0$
M!`1<I&R,Y/3\_)Q1S%&4[/S\_/R\=,S,Q+R<+`0$!`0$!%RDK&1<9*2D4:S,
MQ'24Y/3\Y%&\S,S$O)PT!`0$!`0$7*2TO+R,45'$Q,S,K%%\?)1L?+S,Q,2\
MG"P$!`0$!`1<G+S$Q,3$Q,3$Q,3$Q%%145&\Q,3$Q+2<-`0$!`0$!%2<M,3$
MQ,3$Q,3$Q,3$Q,3$Q,3$Q,2\M)PL!`0$!`0$5)R\Q,2\O+R\O+R\O+R\O+R\
MO+R\O+RTG"P$!`0$!`1<G+3$O+2TM+2TM+2TM+2TM+2TM+2TM*R<+`0$!`0$
M!%2$E)R<E)24E)24E)24E)24E)24E)24E(0D!`0$!`0$1%1<7%Q<7%Q<7%Q<
M7%Q<7%Q<7%Q<7%Q<5#0$!`0$!`0````_````'P````\````'`````P````$`
M````````/P```#\````_````/P```#\````_````/P```#\````_````/P``
M`#\````_````/P```#\````_````/P```#\````_````/P```#\````_````
M/P```#\````_````/P```0`$`!`0```!``@`:`4```$`("````$`"`"H"```
M`@`0$````0`(`&@%```#`"`@```!``@`J`@```0```!8`C0```!6`%,`7P!6
M`$4`4@!3`$D`3P!.`%\`20!.`$8`3P``````O03O_@```0``````````````
M````````/P`````````$``0``0```````````````````+@!```!`%,`=`!R
M`&D`;@!G`$8`:0!L`&4`20!N`&8`;P```)0!```!`#``,``P`#``,``T`$(`
M,````"0``@`!`$,`;P!M`'``80!N`'D`3@!A`&T`90``````(````"P``@`!
M`$8`:0!L`&4`1`!E`',`8P!R`&D`<`!T`&D`;P!N```````@````,``(``$`
M1@!I`&P`90!6`&4`<@!S`&D`;P!N```````P`"X`,``N`#``+@`P````)``"
M``$`20!N`'0`90!R`&X`80!L`$X`80!M`&4````@````*``"``$`3`!E`&<`
M80!L`$,`;P!P`'D`<@!I`&<`:`!T````(````"P``@`!`$P`90!G`&$`;`!4
M`'(`80!D`&4`;0!A`'(`:P!S```````@````+``"``$`3P!R`&D`9P!I`&X`
M80!L`$8`:0!L`&4`;@!A`&T`90```"`````D``(``0!0`'(`;P!D`'4`8P!T
M`$X`80!M`&4``````"`````T``@``0!0`'(`;P!D`'4`8P!T`%8`90!R`',`
M:0!O`&X````P`"X`,``N`#``+@`P````1`````$`5@!A`'(`1@!I`&P`90!)
M`&X`9@!O```````D``0```!4`'(`80!N`',`;`!A`'0`:0!O`&X`````````
ML`0\/WAM;"!V97)S:6]N/2(Q+C`B(&5N8V]D:6YG/2)55$8M."(@<W1A;F1A
M;&]N93TB>65S(C\^#0H\87-S96UB;'D@>&UL;G,](G5R;CIS8VAE;6%S+6UI
M8W)O<V]F="UC;VTZ87-M+G8Q(B!M86YI9F5S=%9E<G-I;VX](C$N,"(^#0H@
M("`@/&%S<V5M8FQY261E;G1I='D@=F5R<VEO;CTB,"XP+C`N,"(@='EP93TB
M=VEN,S(B(&YA;64](E!!4BU086-K97(N07!P;&EC871I;VXB("\^#0H@("`@
M/&1E<V-R:7!T:6]N/E!!4BU086-K97(@07!P;&EC871I;VX\+V1E<V-R:7!T
M:6]N/@T*("`@(#QD97!E;F1E;F-Y/@T*("`@("`@("`\9&5P96YD96YT07-S
M96UB;'D^#0H@("`@("`@("`@("`\87-S96UB;'E)9&5N=&ET>2!T>7!E/2)W
M:6XS,B(@;F%M93TB36EC<F]S;V9T+E=I;F1O=W,N0V]M;6]N+4-O;G1R;VQS
M(B!V97)S:6]N/2(V+C`N,"XP(B!P=6)L:6-+97E4;VME;CTB-C4Y-6(V-#$T
M-&-C9C%D9B(@;&%N9W5A9V4](BHB('!R;V-E<W-O<D%R8VAI=&5C='5R93TB
M*B(@+SX-"B`@("`@("`@/"]D97!E;F1E;G1!<W-E;6)L>3X-"B`@("`\+V1E
M<&5N9&5N8WD^#0H@("`@/'1R=7-T26YF;R!X;6QN<STB=7)N.G-C:&5M87,M
M;6EC<F]S;V9T+6-O;3IA<VTN=C,B/@T*("`@("`@("`\<V5C=7)I='D^#0H@
M("`@("`@("`@("`\<F5Q=65S=&5D4')I=FEL96=E<SX-"B`@("`@("`@("`@
M("`@("`\<F5Q=65S=&5D17AE8W5T:6]N3&5V96P@;&5V96P](F%S26YV;VME
M<B(@=6E!8V-E<W,](F9A;'-E(B`O/@T*("`@("`@("`@("`@/"]R97%U97-T
M9610<FEV:6QE9V5S/@T*("`@("`@("`\+W-E8W5R:71Y/@T*("`@(#PO=')U
M<W1);F9O/@T*("`@(#QC;VUP871I8FEL:71Y('AM;&YS/2)U<FXZ<V-H96UA
M<RUM:6-R;W-O9G0M8V]M.F-O;7!A=&EB:6QI='DN=C$B/@T*("`@("`@("`\
M87!P;&EC871I;VX^#0H@("`@("`@("`@("`\(2TM(%=I;F1O=W,@."XQ("TM
M/@T*("`@("`@("`@("`@/'-U<'!O<G1E9$]3($ED/2)[,68V-S9C-S8M.#!E
M,2TT,C,Y+3DU8F(M.#-D,&8V9#!D83<X?2(O/@T*("`@("`@("`@("`@/"$M
M+2!7:6YD;W=S(%9I<W1A("TM/@T*("`@("`@("`@("`@/'-U<'!O<G1E9$]3
M($ED/2)[93(P,3$T-3<M,34T-BTT,V,U+6$U9F4M,#`X9&5E93-D,V8P?2(O
M/@T*("`@("`@("`@("`@/"$M+2!7:6YD;W=S(#<@+2T^#0H@("`@("`@("`@
M("`\<W5P<&]R=&5D3U,@260](GLS-3$S.&(Y82TU9#DV+31F8F0M.&4R9"UA
M,C0T,#(R-68Y,V%](B\^#0H@("`@("`@("`@("`\(2TM(%=I;F1O=W,@."`M
M+3X-"B`@("`@("`@("`@(#QS=7!P;W)T961/4R!)9#TB>S1A,F8R.&4S+34S
M8CDM-#0T,2UB83EC+60V.60T831A-F4S.'TB+SX-"B`@("`@("`@("`@(#PA
M+2T@5VEN9&]W<R`Q,"`M+3X-"B`@("`@("`@("`@(#QS=7!P;W)T961/4R!)
M9#TB>SAE,&8W83$R+6)F8C,M-&9E."UB.6$U+30X9F0U,&$Q-6$Y87TB+SX-
M"B`@("`@("`@/"]A<'!L:6-A=&EO;CX-"B`@("`\+V-O;7!A=&EB:6QI='D^
M#0H\+V%S<V5M8FQY/@T*````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````````````````!0```,````N*(```!@```,````&*````"P```8````
MX*#PH`"A$*$@H3"A0*$`````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
>````````````````````````````````````````
