package PAR::StrippedPARL::Dynamic;
use 5.008001;
use strict;
use warnings;
our $VERSION = '0.958';

use base 'PAR::StrippedPARL::Base';

our $Data_Pos = tell DATA;

=head1 NAME

PAR::StrippedPARL::Dynamic - Data package containing a dynamic PARL

=head1 SYNOPSIS

  # For details, see PAR::StrippedPARL::Base.
  PAR::StrippedPARL::Dynamic->write_parl($file) or die "Some error...";

=head1 DESCRIPTION

This class is internal to PAR. Do not use it outside of PAR.

This class is basically just a container for a dynamic binary PAR loader
which doesn't include the PAR code like the F<parl> or F<parl.exe>
you are used to. If you're really curious, I'll tell you it is
just a copy of the F<myldr/par> (or F<myldr/par.exe>) file.

The data is appended during the C<make> phase of the PAR build process,
but only if applicable: If you perl is static, you won't get the dynamic
loader.

If the binary data isn't appended during the build process, the class
methods will return the empty list.

=head1 CLASS METHODS

Inherits the methods from L<PAR::StrippedPARL::Base>.

=cut

sub _data_pos {
    my $class = shift;
    return $Data_Pos;
}

=head1 AUTHORS

Steffen Mueller E<lt>smueller@cpan.orgE<gt>,
Audrey Tang E<lt>cpan@audreyt.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2006-2009 by Steffen Mueller E<lt>smueller@cpan.orgE<gt>.

This program is free software; you can redistribute it and/or 
modify it under the same terms as Perl itself.

See F<LICENSE>.

=cut

$^V eq 5.32.1 
    or die sprintf("Perl (%s) version (%vd) doesn't match the version (5.32.1) ".
                   "that PAR::Packer was built with; please rebuild PAR::Packer", 
                   $^X, $^V);

1;

__DATA__
M35J0``,````$````__\``+@`````````0```````````````````````````
M````````````````````@`````X?N@X`M`G-(;@!3,TA5&AI<R!P<F]G<F%M
M(&-A;FYO="!B92!R=6X@:6X@1$]3(&UO9&4N#0T*)`````````!010``9(8+
M`*B>TF(``````````/``+@(+`@(F`$0```#,````$@```!`````0`````$``
M`0`````0`````@``!``````````%``(```````!P`0``!```AAP!``,``(``
M`"`````````0```````````0````````$``````````````0`````!`!`+T!
M````(`$`I`H````P`0!0*P```-```-0!``````````````!@`0`X`````,``
M`!P`````````````````````````````````````````````````````````
MN"(!`&@"```````````````````````````````````N=&5X=````'A"````
M$````$0````$``````````````````!@``!@+F1A=&$```!@`````&`````"
M````2```````````````````0```P"YR9&%T80``K$,```!P````1````$H`
M`````````````````$```$`N8G5I;&1I9#4`````P`````(```".````````
M``````````!```!`+G!D871A``#4`0```-`````"````D```````````````
M````0```0"YX9&%T80``P`$```#@`````@```)(``````````````````$``
M`$`N8G-S`````.`1````\`````````````````````````````"```#`+F5D
M871A``"]`0```!`!```"````E```````````````````0```0"YI9&%T80``
MI`H````@`0``#````)8``````````````````$```,`N<G-R8P```%`K````
M,`$`4"L```"B``````````````````!```#`+G)E;&]C```X`````&`!```"
M````S@``````````````````0```0@``````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M`````````````````````````````````````````````$B#["A(C0T%0```
MZ%`Z``!%,<`QTC')Z&0Z``!%,<`QTC')Z&@Z``!%,<`QTC')Z&PZ``!%,<`Q
MTC')2(/$*.EL.@``D)"0D)"0D)"0D)"03(T%J4\``#'22(T-$````.F+-P``
M9F8N#Q^$``````##D)"0D)"0D)"0D)"0D)"0055!5%575E-(@^PH28G-2(G7
M2(7)#X2U````2(L%7I\``$B+*$B%[0^$H@````^V`83`=`0\/757,?9,BV4`
M387D=#)(8_9(B>MF#Q]$``!)B?!,B>I,B>'H0CD``(7`=0=!@#PT/71'3(MC
M"$B#PPA-A>1UVDR)X$B#Q"A;7E]=05Q!7<-F+@\?A```````2(G.#[9&`4B#
MQ@&$P'0$/#UU\$0I[NN3#Q^$``````!(*>M-C60T`4C!^P-,B>")'TB#Q"A;
M7E]=05Q!7<-FD$4QY$R)X$B#Q"A;7E]=05Q!7<-F9BX/'X0``````&:0059!
M54%455=64TB![+````!(BQU8G@``2(L#2(F$)*@````QP$F)S$B%R70%@#D`
M=29(BX0DJ````$@K`P^%'`$``$B!Q+````!;7E]=05Q!74%>PP\?`.B3-P``
M28G%2(7`=,U(C2TD7@``2(U\)"!,B>GHES<``$B%P`^$G@```(!X&"Y(C7`8
M=0R`?@$`=-]F#Q]$``"`>!@N#X26````3(GAZ/8W``!(B?%)B<;HZS<``$&-
M3`8"2&/)Z,XV``!)B?%-B>!(B>I)B<9(B<'H>C<``$R)\4B)^NA_-P``3(GQ
M@_C_=!"+1"0P)0#P```]`$```'1?Z.(W``!,B?'H*C8``$R)Z>@"-P``2(7`
M#X5K____9@\?A```````3(GIZ-@U``!,B>'H$#<``.G]_O__#Q\`@'X!+@^%
M8/___X!^`@`/A"7____I4?___P\?@`````#HH_[__^N?Z%PU``"09F8N#Q^$
M``````!!5%-(@>RX````2(L=[YP``$B+`TB)A"2H````,<!(C50D($F)S.CE
M-0``A<!U,8M4)#"!X@#P``"!^@!```!T"('Z`*```'49N@(```!,B>'H&C4`
M`(7`#Y3`#[;`ZP(QP$B+E"2H````2"L3=0M(@<2X````6T%<P^C1-```D$%4
MN#`0``!55U93Z$`T``!(*<1(BSUFG```0;C_#P``2(L'2(F$)"@0```QP$B-
M="0@28G,2(G32(T-9UP``$B)\NCI-0``A<!^38U0_TACZH!\+"``=7B%TG0\
M28GHNB\```!(B?'H1#4``$B%P'0G2(L%V)L``$B+".C`,P``28GI28GP3(GB
M2(G!Z#<S``#K(@\?1```2(L%L9L``$B+".B9,P``28G83(GB2(G!Z!LS``!(
MBX0D*!```$@K!W432('$,!```%M>7UU!7,-(8^CKA^CV,P``D`\?1```05=!
M5D%505155U932(/L.$B+'8F;``!(BP-(B40D*#'`,<"`.CU(B=8/E,!(B<](
M`<9(B?'HEC4``$B-5"0D2(GY2(G%Z/;[__])B<1(A<`/A/H```!(B<'H<C4`
M`$@YQ0^&R0```$QC?"0D3(LM39L``$G!YP,/M@>$P`^$C@$``#P]#X2&`0``
M2(GX#Q\`#[90`4B#P`&$TG0%@/H]=>](*?A(F$V+90!(C4PH`N@,-```30'\
M28D$)$B%P'1028M%`$B-3P%*BQ0X#[8'B`*$P'4<ZQYF#Q^$```````/M@%(
M@\(!2(/!`8@"A,!T!#P]=>O&`CTQP&8/'T0```^V#`:(3`(!2(/``83)=?!(
MBT0D*$@K`P^%!@$``$B#Q#A;7E]=05Q!74%>05_#D#'`9@\?1```#[84!D&(
M%`1(@\`!A-)U\.O&9@\?1```3(LM:9H``$V+=0!)@SX`#X2R````3(GP13'D
M#Q]$``!(@\`(1(GB08/$`4B#.`!U[X/"`TACTDC!X@.+!2_J``"%P'53QP4A
MZ@```0```$B)T>@9,P``2(G!2(7`#X1=____36/\3(GR2<'G`TV)^.@K,P``
M28E%`$B)P4&-1"0!1(ED)"1(F$C'!,$`````Z9+^__\/'P!,B?'H<#,``$F)
M10!(B<%(A<`/A!#___]-8_Q)P><#Z\,/'X``````,<#IC_[__[H0````13'D
MZ6?____HQS$``)!F#Q]$``!32(/L,$B+'329``!(BPOH'#$``$B+"^@4,0``
M3(T--5P``$R-!78P``!(C15T60``2(G!Z)\P``!(BPOH[S```$R-#7E9``#'
M1"0H`````$R-!>D1``!,B4PD($B)P4B-%5Q9``#H93```)!(@\0P6\-F9BX/
M'X0```````\?`$B#["A(BP6UF```2(L(Z)TP``!,C0V^6P``3(T%_R\``$B)
MP4B-%?I8``!(@\0HZ20P``!F9BX/'X0``````)`QP,-F9BX/'X0``````&:0
M05>XV!```$%6055!5%575E/H6C```$@IQ$B+/8"8``!(BP=(B80DR!```#'`
M2(UL)$!(B<Y)B=5(C0W*6```2(GJZ/GX__^Z+P```$B)\4B)P^A),@``2(7`
M=$1(B?)(C0VL6```Z)7\__](B?!(BS4KF```2(N4),@0``!(*Q8/A6<!``!(
M@<38$```6UY?74%<05U!7D%?PV8/'T0``$B)\4B-/758``#H(3(``$R)Z4&)
MQ.@&,@``2(GZ2(G!Z#LR``!)B<9(A<!TCT&`/@!,C2U06```#X2R````9I!(
MA=MT#TR)\DB)V>C`,0``A<!T?$R)\>C4,0``ZP5FD,8"`(G!@^@!2&/03`'R
M@#HO=.U"C40A`3W_#P``#X\Y____2(ET)#`QTDR-O"3`````3(T-Z%<``$R)
M;"0H3(GY0;@`$```3(ET)"#HCR\``$B)ZDR)^>@T,0``A<!U$(M$)%`E`/``
M`#T`@```=%!(B?HQR>B&,0``28G&2(7`#X36_O__08`^``^%4/___TB%VW0%
M@#LN=!9,C35F5P``N0$```#I:/___P\?1```@'L!`'2ZZ^(/'X0``````+H!
M````3(GYZ$,O``"%P'6?3(GZ2(T--U<``.@@^___3(GYZ-@P``#I@?[__^C^
M+@``D&9F+@\?A```````9I`/M@%)B<B$P'079@\?1```2(/!`3PO#[8!3`]$
MP83`=>],B<##9F8N#Q^$``````!!5$B#["!)B<Q(A<D/A+X```"`.0`/A+4`
M``#H@#```$F-1`3_23G$<@_ILP```)!(@^@!23G$="6`."]T\DDYQ`^#FP``
M`$B#Z`%).<1T#H`X+W0P2(/H`4DYQ'7R08`\)"]T:TB-#8Q6``!(@\0@05SI
M&3```&8/'X0``````(`X+W4)2(/H`4DYQ'+R3"G@13'`2(U0`DB!^@`0``!W
M'$R-0`%!N0`0``!,B>)(C0W;U0``Z!8N``!)B<!,B<!(@\0@05S#9@\?A```
M````2(T-`%8``$B#Q"!!7.FN+P``23G$=9[I<O___P\?0`!!5T%6055!5%57
M5E-(@^PX2(LM:94``$B+-8*5``!(C1WE50``2(M%`$B)1"0H,<!,C60D)$R)
MXDB)V>C@]?__2(7`=#M(8U0D)$B+!DB-!-!(BU`(2(D02(72=-@/'P!(BU`0
M2(/`"$B)$$B%TG7P3(GB2(G9Z*7U__](A<!UQ4B+-1F5``!(C1V%50``3(GB
M2(G9Z(?U__](A<!T0DAC5"0D2(L&2(T$T$B+4`A(B1!(A=)TV&8N#Q^$````
M``!(BU`02(/`"$B)$$B%TG7P3(GB2(G9Z$7U__](A<!UODB+-;F4``!(C1TM
M50``3(GB2(G9Z"?U__](A<!T0DAC5"0D2(L&2(T$T$B+4`A(B1!(A=)TV&8N
M#Q^$``````!(BU`02(/`"$B)$$B%TG7P3(GB2(G9Z.7T__](A<!UODB+-5F4
M``!(C1W65```3(GB2(G9Z,?T__](A<!T0DAC5"0D2(L&2(T$T$B+4`A(B1!(
MA=)TV&8N#Q^$``````!(BU`02(/`"$B)$$B%TG7P3(GB2(G9Z(7T__](A<!U
MODB+'?F3``!,C2U]5```3(GB3(GIZ&?T__](A<!T0DAC5"0D2(L#2(T$T$B+
M4`A(B1!(A=)TV&8N#Q^$``````!(BU`02(/`"$B)$$B%TG7P3(GB3(GIZ"7T
M__](A<!UODB+-9F3``!(C1TM5```3(GB2(G9Z`?T__](A<!T0DAC5"0D2(L&
M2(T$T$B+4`A(B1!(A=)TV&8N#Q^$``````!(BU`02(/`"$B)$$B%TG7P3(GB
M2(G9Z,7S__](A<!UODB+'3F3``!,C35[4P``3(GB3(GQZ*?S__](A<!T0DAC
M5"0D2(L#2(T$T$B+4`A(B1!(A=)TV&8N#Q^$``````!(BU`02(/`"$B)$$B%
MTG7P3(GB3(GQZ&7S__](A<!UODB+-=F2``!(C1UY4P``3(GB2(G9Z$?S__](
MA<!T0DAC5"0D2(L&2(T$T$B+4`A(B1!(A=)TV&8N#Q^$``````!(BU`02(/`
M"$B)$$B%TG7P3(GB2(G9Z`7S__](A<!UODB+-7F2``!,C3TC4P``3(GB3(GY
MZ.?R__](A<!T0DAC5"0D2(L&2(T$T$B+4`A(B1!(A=)TV&8N#Q^$``````!(
MBU`02(/`"$B)$$B%TG7P3(GB3(GYZ*7R__](A<!UODB+/1F2``!(C37-4@``
M3(GB2(GQZ(?R__](A<!T0DAC5"0D2(L'2(T$T$B+4`A(B1!(A=)TV&8N#Q^$
M``````!(BU`02(/`"$B)$$B%TG7P3(GB2(GQZ$7R__](A<!UODB+/;F1``!(
MC34$4@``3(GB2(GQZ"?R__](A<!T0DAC5"0D2(L'2(T$T$B+4`A(B1!(A=)T
MV&8N#Q^$``````!(BU`02(/`"$B)$$B%TG7P3(GB2(GQZ.7Q__](A<!UODR)
MXDB-#1M2``#HT?'__TB)PDB%P'0(3(GYZ('U__],B>)(C0T-4@``Z++Q__](
MB<)(A<!T#$B-#0M2``#H7O7__TR)XDB-#0=2``#HC_'__TB)PDB%P'0R3(GQ
MZ#_U__](BT0D*$@K10!U0$B-%0%2``!,B>E(@\0X6UY?74%<05U!7D%?Z17U
M__],B>)(C0W.40``Z$;Q__](B<)(A<!TOTB)V>CV]/__Z[7HWR@``)!F9BX/
M'X0```````\?`%-(@^PP2(L==)```$B+`TB)1"0HBP660```@_C_=!%(BU0D
M*$@K$W4\2(/$,%O#D$B-5"0D2(T-"U$``.C?\/__2(G",<!(A=)T$@^V`CPP
M#Y7"A,`/E<`/ML`AT(D%34```.NZZ&8H``"0#Q]$``!(@^PHZ/<H``")P>C@
M*```2(7`=`-(BP!(@\0HPP\?`$%6055!5%932(/L0$B+-=V/``!,C2T,40``
M2(L&2(E$)#@QP$F)S$B-5"0T3(GIZ%SP__](B<-(A<!T!8`X`'4O2(M$)#A(
M*P8/A:$```!,B>),B>E(@\1`6UY!7$%=05[IZO/__V8N#Q^$``````!,B>)(
MB<'HQ2D``$B%P'0@2(M$)#A(*P9U9DB#Q$!;7D%<05U!7L-F#Q^$``````!,
MB>'H>"D``$B)V4F)QNAM*0``28U,!@+H4R@``$B)7"0@38G@3(T-HT\``$B-
M%9Y/``!(B<%)B<;H\B@``$B+1"0X2"L&=0A,B?+I8O___^A+)P``D&8N#Q^$
M``````!!5[BH@0``059!54%455=64^BJ)@``2"G$2(L]T(X``$R--3)/``!(
MBP=(B80DF($``$B-!1Y/``!(B<Y,C6PD4$R)\4C'A"3@`````````$R)ZDC'
MA"0X`0```````$C'A"0``0```````$B)A"38````2(T%PD\``$B)A"08`0``
M2(T%ND\``$B)A"0@`0``2(T%LT\``$B)A"0H`0``2(T%J4\``$B)A"0P`0``
M2(T%GD\``$B)A"3X````Z,WN__])B<1(A<!T"8`X``^%7`0``.@')P``B<'H
M\"8``$B%P`^$QP,``$B+*$B%[0^$NP,``$B)Z>@C*```2(U,``'H"2<``$0/
MMD4`2(G#183`="U)B=],C24_3P``9I!%#[;`3(GY3(GB2(/%`>B=)P``1`^V
M10!)@\<"183`==](C:PD$`$``$B-#:A.``#K%V8/'T0``$B+30A(@\4(2(7)
M#X0'`0``3(GJZ!_N__])B<1(A<!TWX`X`'3:2(G!Z&KP__^%P'3.3(GAZ'XG
M``!(B<=(A<`/A-(```!FD$B)^>AX)P``2(G92(G%Z&TG``!$C:0%!`0``$UC
MY$R)X>A*)@``2(E<)"A(C1603@``28GX28G'2(T%C$X``$R-#9)-``!(B40D
M($R)^>C=)@``NL`!``!,B?GH4"8``(/X_P^$7P(``$R)ZDR)^>CL)0``B<.%
MP`^$P@```$B+'C'MZ&@D``!-B?E(C15^3@``2(M(&$F)V.AB)0``2(LUJXP`
M`$B+A"28@0``2"L&#X55!0``2(GH2('$J($``%M>7UU!7$%=05Y!7\,/'P`Q
M[4R-);Y-``#K'`\?1```2&/%3(NDQ-````!-A>1T0$&`/"0`=#E,B>&#Q0'H
M3N___X7`=-I,B>'H8B8``$B)QTB%P`^%YO[__TACQ4R+I,30````387D=<9F
M#Q]$```Q_^G)_O__9@\?A```````BT0D8"4`\```/0!````/A2K___^+;"1H
MZ.,D```YQ0^%&?___XM$)&`E_P$``#W``0``#X4%____3(GAZ.\D``!,B>I(
MC0W[3```2(G%Z%WL__](B<)(A<`/A#$"``!(BP[H"?/__TB)QTB%P`^$'0(`
M`$B)^>C%)0``28G$@_@'?C5(F$B-%9=-``!(C4P'^.B*)0``A<!U'D&#_`@/
MA'@#``!!C40D]TB8@#P'+P^$9P,```\?`(L%@CL``(/X_P^$$0,``$B-#0E,
M``"%P`^$S`$``$B-%6=,``!(C35]2P``Z(?O__],C2VH2P``3(TE5TT``.CD
M(P``2(ET)"A-B?A(B>F)1"0@3(T-ADL``$B-%2=-``#HTB0``.LPZ`LC``"#
M.!%U..BQ(P``@\,!2(ET)#!-B>F)7"0H38GX3(GB2(GIB40D(.B@)```NL`!
M``!(B>GH$R0``(/X_W2^3(GYZ&8C``!,B?%(B>KH^^[__TB)Z>BC^O__Z=S]
M__]F#Q]$``#HHR(``(,X$0^$D_W__^B5(@``2(L>BRCH"R(``$V)^4F)V$B-
M%=Y+``!(BT@8B6PD(#'MZ/\B``#IF/W__V8N#Q^$```````QVTB-#7)+``#K
M%0\?1```2&/#2(N,Q/````!(A<ET.4R)ZH/#`>BUZO__2(7`=."`.`!TVTB)
MP>@C)```2(G%2(7`#X7\^___2&/#2(N,Q/````!(A<EUQ[D-````2(TM&TL`
M`.CV(@``0;A3````2(G#Z>S[__\/'X0``````$B)P>C(^?__2(L]L8D``$B+
MA"28@0``2"L'#X5;`@``3(GA2('$J($``%M>7UU!7$%=05Y!7^FE(P``#Q]$
M``!(BS[IV_W__\<%GCD```````!(B?FZ```!`.C1(@``2(T-(4H``$&)Q(7`
M#X05_O__0;@"````2,?"[O___XG!Z#LB``!(C9PDD`$``$&X!@```$2)X4B)
MVNBR(@``@+PDD`$```!U#H"\))$!``!##X11`0``13'`,=)$B>'H_2$``.C8
M&P``28G%ZQ$/'P!!B<!(B=I,B>GH(AP``$&X`(```$B)VD2)X>AA(@``A<!_
MW42)X4B-G"1``0``2(V\)&`!``#H-2$``$R)ZDB)V4F)W>AW'0``2(G[2(VT
M)(@!``!,C24"2@``#Q]$``!%#[9%`$B)V4R)XDB#PP))@\4!Z%@B``!(.=YU
MXTB-!;U(``#&A"2(`0```$B)1"0H2(E\)"!,C0W;2```38GX2(T5:DH``$B)
MZ>@A(@``Z8[]__\/'T``2(T-^$@``$R)ZDB)3"1(Z,3H__](A<`/A*/^__\/
MM@"$P`^$F/[__SPP#X20_O__QP4N.````0```$B+3"1(Z;K\__](BW8(2(7V
M#X2/_/__2(GQZ`\B``!(@_@##X9]_/__2(U,!OQ(C17E20``Z-0A``"%P$@/
M1/[I8?S__X"\))(!``!!#X6A_O__@+PDDP$``$,/A9/^__^`O"24`0``2`^%
MA?[__X"\))4!``!%#X5W_O__0;@"````2,?"QO___T2)X>AL(```0;@I````
M2(G:1(GAZ.L@``!(C06E1P``2(E<)"!(B40D*.GK_O__Z)`?``"09F8N#Q^$
M```````/'T``059!54%45U932(/L.$B+'>R&``!(BST5AP``2(L+2(L'2(E$
M)"@QP$R-9"0DZ+X>``!(BPM(B<;HLQX```^VEK@```!(C0V01P``B)"Y````
M3(GBZ'?G__](A<!T"8`X,0^$F0$``.A$\?__2(L+Z'P>``!!N0,```!!N`$`
M``!(C17X2```2(G!Z!D>``!(B<9(A<!T,TB+0!!(BPM,BRCH2AX``$R-!=1(
M``!(B<%,B>KHR!T``$B+1A!,BRA!]D4.0`^%=@$``$B+"^@>'@``0;D#````
M0;@!````2(T5H4@``$B)P>B['0``2(G&2(7`=#A(C0VF1@``3(GBZ,SF__](
MBPM)B<5(A<`/A/T```!(BT803(LPZ-$=``!-B>A(B<%,B?+H4QT``$B+"^B[
M'0``0;@!````2(T51D@``$B)P>AF'0``2(L+28G%Z)L=``!,C05!2```3(GJ
M2(G!Z!D=``!(BPOH@1T``$B-#2-&``!,B>+&@+D`````Z$OF__](B07,U0``
M2(G!2(7`=#RZP`$``.CR'@``C5`!@_H!=@F#^!$/A:D```!(BT0D*$@K!P^%
MNP```$B#Q#A;7E]!7$%=05[##Q]$``!(BPV)U0``Z&3V__](B05UU0``2(G!
M2(7`=:E(C0VV1P``Z,$<``!F#Q^$``````"`>`$`#X1B_O__Z5C^__^02(L%
M2=4``$R+,$B+1A!,BRCHRAP``$R)\DR)Z>A_Z/__Z??^__]F+@\?A```````
M2(L+Z*@<``!,B>I(B<'H31P``.ER_O__08G!2(L%_M0``$R+!>_4``!(C0U@
M1P``2(L0Z$`<``#H`QT``)!FD$%505164TB#[#A(BS6?A```2(L&2(E$)"@Q
MP$F)S>@=[O__#[8028G$A-)T%0\?`$B#P`&`^B\/MA!,#T3@A-)U[HL%F#0`
M`$V%[0^5PX/X_W1MA<`/E<`APX3;=2)(BT0D*$@K!@^%DP```$B#Q#A;7D%<
M05W#9@\?A```````08!]``!TUT&X!````$B-%9=%``!,B>'H5!X``(7`=;Y(
MBT0D*$@K!G553(GI2(/$.%M>05Q!7>F$Y?__#Q]``$B-5"0D2(T-JT0``.A_
MY/__2(7`=`L/M@"$P'0$/#!U#\<%]3,```````#I;O___\<%YC,```$```#I
M6____^C\&P``D)"0D)"0D)"0D)"005=!5D%505155U932('LN`(``$B+/8:#
M``!(BP=(B80DJ`(``#'`2(G*3(U9.$B-3"0@2(V<)*````!)B<IFD$&+`TV+
M"TF#PA!)@\,(#\B)P$V)R$F)0O!,B<A)P>@@2,'H*$G!Z3@E`/\``$P)R$V)
MP4G!X`A)P>$808'@``#_`$6)R4P)R$P)P$F)0OA).=IUK$R-B0`"```/'T0`
M`$B+06A(,T%`2(/!"$@S00A(,T'X3(T$`$C!Z!],"<"#X/](B4%X3#G)==5,
MBQI(BW(02(M:($B+>AA,B=A,B=E(BVH(3(M4)"A(P>D;2,'@!4@)R`-$)"!.
MC907F7F"6HV,&)EY@EI(B?!(,?A((>A(,?A$C0P!2(GI2(GH2,'H`DC!X1Y,
MB<M("<%(B?!(P>L;08G(3#'`3"'82#'P3`'038G*2<'B!4D)VD:-)!!,B=A-
MB=I)P>H"2,'@'DV)Y4P)T$R+5"0P2<'M&T@QP8G#3"')3HV4%IEY@EI,,<%,
M`=%-B>))P>(%30GJ3(ML)#A!`<I,B<E)P>D"2,'A'D6)TD^-A"B9>8):3`G)
M38G52#'(08G)3"'@2<'M&T@QV$P!P$V)T$G!X`5-">A,BVPD0$$!P$R)X$G!
M[`)(P>`>18G`2HV<*YEY@EI,">!-B<5(,<%)P>T;08G$3"'13#')2`'93(G#
M2,'C!4P)ZTR+;"1(`<M,B=%)P>H"2,'A'HG;3XV,*9EY@EI,"=%)B=U(,<A)
MP>T;08G*3"'`3#'@3`'(28G92<'A!4T)Z4R+;"1000'!3(G`2<'H`DC!X!Y%
MB<E/C:0LF7F"6DP)P$V)S4@QP4G![1M!B<!((=E,,=%,`>%-B<Q)P>0%30GL
M3(ML)%A!`<Q(B=E(P>L"2,'A'D6)Y$^-E"J9>8):2`G938GE2#'(2<'M&XG+
M3"'(3#'`3`'038GB2<'B!4T)ZDR+;"1@00'"3(G(2<'I`DC!X!Y%B=)/C80H
MF7F"6DP)R$V)U4@QP4G![1M!B<%,(>%(,=E,`<%-B=!)P>`%30GH3(ML)&A!
M`<A,B>%)P>P"2,'A'D6)P$J-G"N9>8):3`GA38G%2#'(2<'M&T&)S$PAT$PQ
MR$@!V$R)PTC!XP5,">M,BVPD<`'#3(G02<'J`DC!X!Z)VT^-C"F9>8):3`G0
M28G=2#'!2<'M&T&)PDPAP4PQX4P!R4F)V4G!X05-">E,BVPD>$$!R4R)P4G!
MZ`)(P>$>18G)3XVD+)EY@EI,"<%-B<U(,<A)P>T;08G(2"'83#'03`'@38G,
M2<'D!4T)[$R+K"2`````00'$2(G82,'@'DC!ZP)%B>1/C90JF7F"6D@)V$V)
MY4@QP4G![1N)PTPAR4PQP4P!T4V)XDG!X@5-">I,BZPDB````$$!RDR)R4G!
MZ0)(P>$>18G23XV$*)EY@EI,"<E-B=5(,<A)P>T;08G)3"'@2#'83`'`38G0
M2<'@!4T)Z$R+K"20````00'`3(G@2<'L`DC!X!Y%B<!*C9PKF7F"6DP)X$V)
MQ4@QP4G![1M!B<1,(=%,,<E(`=E,B<-(P>,%3`GK3(NL))@````!RTR)T4C!
MX1Y)P>H"B=M/C8PIF7F"6DP)T4F)W4@QR$G![1M!B<I,(<!,,>!,`<A)B=E)
MP>$%30GI3(NL)*````!!`<%,B<!)P>@"2,'@'D6)R4^-I"R9>8):3`G`38G-
M2#'!2<'M&T&)P$@AV4PQT4P!X4V)S$G!Y`5-">Q&C30A2(G92,'K`DC!X1Y(
M"=E(BYPDJ````$@QR$&)S$PAR$V-E!J9>8):3(GS3#'`2,'K&TP!T$V)\DG!
MX@5)"=I(BYPDL````$$!PDR)R$C!X!Y)P>D"18G238V$&)EY@EI,"<A,B=-(
M,<%(P>L;08G!3"'Q3#'A20'(3(G12,'A!4@)V4R)\TG![@)(P>,>B<E,"?-)
M`<A(BXPDN````$@QV$6)Q4PAT$F-C`R9>8):38GL3#'(2<'L&T@!R$R)Z4C!
MX05,">&)R4@!R$R)T4G!Z@)(P>$>08G$3`G128G:B=M),<I%,=!,BY0DP```
M`$^-C!&AZ]EN38GB30'!38G@2<'J&TG!X`5-"=!%B<!-`<%-B>A)P>`>2<'M
M`D6)RD@#G"3(````30GH28G-B<E(`XPDT````$TQQ40QZ$V)U4B-G`.AZ]EN
M3(G02<'M&TC!X`5,">B)P$@!PTR)X$G![`)(P>`>08G=3`G@38G$18G`3`.$
M)-@```!),<1%,>%-B>Q.C8P)H>O9;DR)Z4G![!M(P>$%3`GAB<E)`<E,B=%)
MP>H"2,'A'D6)S$P)T4F)PHG`23'*03':3(GC3XV4$*'KV6Y-B>!(P>L;2<'@
M!4D)V$6)P$T!PDV)Z$G![0))P>`>1(G330GH28G-2`.$).````")R4TQQ4@#
MC"3H````13'I28G=3HV,"*'KV6Y(B=A)P>T;2,'@!4P)Z(G`20'!3(G@2<'L
M`DC!X!Y%B<U,">!-B<1%B<!,`X0D\````$DQQ$4QXDV)[$Z-E!&AZ]EN3(GI
M2<'L&TC!X05,">&)R4D!RDB)V4C!ZP)(P>$>18G42`G92(G#B<!(`X0D^```
M`$@QRT$QV4R)XT^-C`BAZ]EN38G@2,'K&TG!X`5)"=A%B<!-`<%-B>A)P>T"
M2<'@'D2)RTT)Z$F)S8G)33'%2`.,)``!``!%,>I)B=U.C900H>O9;DB)V$G!
M[1M(P>`%3`GHB<!)`<),B>!)P>P"2,'@'D6)U4P)X$V)Q$6)P$P#A"0(`0``
M23'$13'A38GL3HV,":'KV6Y,B>E)P>P;2,'A!4P)X8G)20')2(G92,'K`DC!
MX1Y%B<Q("=E(B<.)P$@#A"00`0``2#'+03':3(GC3XV4$*'KV6Y-B>!(P>L;
M2<'@!4D)V$6)P$T!PDV)Z$G![0))P>`>1(G330GH28G-B<E-,<5%,>E)B=U.
MC8P(H>O9;DB)V$C!X`5)P>T;2`.,)!@!``!,">B)P$D!P4R)X$G![`)(P>`>
M18G-3`G@38G$18G`3`.$)"`!``!),<1%,>)-B>Q.C901H>O9;DR)Z4G![!M(
MP>$%3`GAB<E)`<I(B=E(P>L"2,'A'D6)U$@)V4B)PXG`2`.$)"@!``!(,<M!
M,=E,B>-/C8P(H>O9;DV)X$C!ZQM)P>`%20G818G`30'!38GH2<'M`DG!X!Y$
MB<M-">A)B<V)R4TQQ44QZDF)W4Z-E!"AZ]EN2(G82<'M&TC!X`5,">A(`XPD
M,`$``(G`20'"3(G@2<'L`DC!X!Y%B=5,">!-B<1%B<!,`X0D.`$``$DQQ$4Q
MX4V)[$Z-C`FAZ]EN3(GI2<'L&TC!X05,">&)R4D!R4B)V4C!ZP)(P>$>18G,
M2`G92(G#B<!(`X0D0`$``$@QRT$QVDR)XT^-E!"AZ]EN38G@2,'K&TG!X`5)
M"=A%B<!-`<)-B>A)P>T"2<'@'D2)TTT)Z$F)S8G)33'%13'I28G=3HV,"*'K
MV6Y(B=A)P>T;2,'@!4P)Z(G`20'!3(G@2,'@'DG![`)%B<U(`XPD2`$``$P)
MX$V)Q$6)P$P#A"10`0``23'$13'B3HVD$:'KV6Y,B>E-B>I)P>H;2,'A!4P)
MT8G)20',2(G92,'K`DC!X1Y%B>)("=E(B<.)P$@#A"18`0``2#'+03'93(G3
M3XV,"*'KV6Y-B=!(P>L;2<'@!4D)V$R)ZTG![0)(P>,>10'!28G(B<E,">M%
MB<E),=A)B=Y!B=U%,>!-B<Q.C80`H>O9;DR)R$G![!M(P>`%3`G@18TD`$V)
MT$G!Z@*XW+P;CTG!X!Y-"=!-"<9%B<),(=--(<Y)"=Y(BYPD8`$``$@!PT@!
MV4R)XTP!\4V)YDC!XP5)P>X;3`GS`<M,B<E)P>D"2,'A'HG;3`G)20G(28G.
M08G)32'@32'630G&3(N$)&@!``!)`<!-`>A)B=U-`?!)B=Y)P>4%2<'N&TT)
M]4F)SD4!Q4V)X$G![`))P>`>18GM30G@18G$30G&22'>3"'A3`GQ3(NT)'`!
M``!)`<9-`=9-B>I,`?%-B>Y)P>(%2<'N&TT)\D$!RDB)V4C!ZP)(P>$>18G2
M2`G920G(28G.B<M-(>A-(>9-"<9,BX0D>`$``$D!P$T!R$V)T4T!\$V)UDG!
MX05)P>X;30GQ28G.10'!38GH2<'M`DG!X!Y%B<E-">A%B<5-"<9-(=9,(>E,
M"?%,B[0D@`$``$D!QDT!YDV)S$P!\4V)SDG!Y`5)P>X;30GT00',3(G12<'J
M`DC!X1Y%B>1,"=%)"<A)B<Y!B<I-(<A-(>Y-"<9,BX0DB`$``$D!P$D!V$R)
MXTT!\$V)YDC!XP5)P>X;3`GS28G.1`'#38G(2<'I`DG!X!Z)VTT)R$6)P4T)
MQDTAYDPAR4P)\4R+M"20`0``20'&30'N28G=3`'Q28G>2<'E!4G![AM-"?5!
M`<U,B>%(P>$>2<'L`D6)[4P)X4D)R$F)SD&)S$DAV$TASDT)QDR+A"28`0``
M20'`30'038GJ30'P38GN2<'B!4G![AM-"?))B<Y%`<))B=A(P>L"2<'@'D6)
MTDD)V$2)PTT)QDTA[D@AV4P)\4R+M"2@`0``20'&30'.38G13`'Q38G62<'A
M!4G![AM-"?%!`<E,B>E)P>T"2,'A'D6)R4P)Z4D)R$F)SD&)S4TAT$DAWDT)
MQDR+A"2H`0``20'`30'@38G,30'P38G.2<'D!4G![AM-"?1)B<Y%`<1-B=!)
MP>H"2<'@'D6)Y$T)T$6)PDT)QDTASDPAT4P)\4R+M"2P`0``20'&20'>3(GC
M3`'Q38GF2,'C!4G![AM,"?,!RTR)R4G!Z0)(P>$>B=M,"<E)"<A)B<Y!B<E-
M(>!-(=9-"<9,BX0DN`$``$D!P$T!Z$F)W4T!\$F)WDG!Y05)P>X;30GU28G.
M10'%38G@2<'L`DG!X!Y%B>U-">!%B<1-"<9)(=Y,(>%,"?%,B[0DP`$``$D!
MQDT!UDV)ZDP!\4V)[DG!X@5)P>X;30GR00'*2(G92,'K`DC!X1Y%B=)("=E)
M"<A)B<Z)RTTAZ$TAYDT)QDR+A"3(`0``20'`30'(38G130'P38G62<'A!4G!
M[AM-"?%)B<Y%`<%-B>A)P>T"2<'@'D6)R4T)Z$6)Q4T)QDTAUDPAZ4P)\4R+
MM"30`0``20'&30'F38G,3`'Q38G.2<'D!4G![AM-"?1!`<Q,B=%)P>H"2,'A
M'D6)Y$P)T4D)R$F)SD&)RDTAR$TA[DT)QDR+A"38`0``20'`20'83(GC30'P
M38GF2,'C!4G![AM,"?-)B<Y$`<--B<A)P>D"2<'@'HG;30G(30G&18G!32'F
M3"')3`GQ3(NT).`!``!)`<9-`>Y)B=U,`?%)B=Y)P>4%2<'N&TT)]4$!S4R)
MX4G![`)(P>$>18GM3`GA20G(28G,08G/22'832',30G$3(N$).@!``!)`<!-
M`=!-B>I-`>!-B>Q)P>(%2<'L&TT)XD4!PDF)V$C!ZP))P>`>18G220G82(G+
M18G$3`G#3"'K3"'A2`G92(N<)/`!``!(`<-(`X0D^`$``$D!V4R)TTP!^$R+
MO"08`@``20')3(G12,'K&TC!X05("=F)R4D!R4R)Z4G![0)(P>$>1(G+3`GI
M20G(28G-08G.32'032'E30G%38U$!0!(B=A)B=U)P>T;2,'@!4P)Z(G`20'`
M3(G02<'J`DC!X!Y%B<5)"<*XUL%BRDPQT40QR4R+C"0``@``20'!30'A38GL
M3`')38GI2<'L&TG!X05-">%)B=Q%B<E,`<E)P>0>20''2,'K`D&)R4P)XTV)
MU$6)TDDQW$4QQ$R+A"0(`@``20'`30'P38G.30'@38G,2<'N&TG!Y`5-"?1-
MB>Y)P>T"2<'F'D6)Y$T)]4R+M"00`@``30'@18G$20'&30'R28G>33'N03'.
M3(GA30'R38GF2,'A!4G![AM,"?&)R4D!RDR)R4G!Z0)(P>$>18G620G)B=E,
MB>M,,<M,`?E,B[PD(`(``$0QPTR-!!E,B?%,B?-(P>$%2,'K&TD!QT@)V8G)
M20'(3(GA2<'L`DC!X1Y$B<-,">%%B>Q-B<U%B<E),<U-`?Q,B[PD.`(``$4Q
MU4^-%"Q)B=Q)B=U)P>T;2<'D!4T)[$V)]4G![@))P>4>18GD30GN3(NL)"@"
M``!-`>)%B=1)`<5-`>E)B<V)R4TQ]44QQ4V)X$T!Z4V)Y4G!X`5)P>T;30GH
M18G`30'!28G82,'K`DG!X!Y%B<U)"=A(BYPD,`(``$@!PT@!V4R)\TPQPT0Q
MTTD!QTR-%!E,B>E,B>M(P>L;2,'A!4@)V8G)20'*3(GA2<'L`DC!X1Y$B=-)
M"<Q$B?%-B<9%B<!-,>9,`?E,B[PD2`(``$4QSDZ-##%(B=E)B=Y)`<=)P>X;
M2,'A!4P)\8G)20')3(GI2<'M`DC!X1Y%B<Y,">E,BZPD0`(``$D!Q4T!Z$V)
MY4DQS44QU4^-%"A-B?!-B?5)P>T;2<'@!4T)Z$6)P$T!PDF)V$C!ZP))P>`>
M18G53`G#18G@28G,B<E-`?A),=Q,B[PD6`(``$4QS$^-#"!-B>A-B>Q)`<=)
MP>P;2<'@!4T)X$6)P$T!P4V)\$G![@))P>`>18G,30GP3(NT)%`"``!)`<9,
M`?%)B=Y-,<9%,=9.C10Q3(GA38GF2<'N&TC!X05,"?&)R4D!RDR)Z4G![0)(
MP>$>18G620G-B=E,B<-%B<!,,>M,`?E,B[PD:`(``$0QRTR-#!E,B?%,B?-(
MP>L;2,'A!4@)V8G)20')3(GA2<'L`DC!X1Y$B<M,">%,BZ0D8`(``$D!Q$T!
MX$V)[$D!QTDQS$4QU$^-%"!)B=A)B=Q)P>P;2<'@!4T)X$6)P$T!PDV)\$G!
M[@))P>`>18G430G&18GH28G-B<E-,?5-`?A,B[PD>`(``$4QS4^-#"A-B>!-
MB>5)P>T;2<'@!4T)Z$6)P$T!P4F)V$C!ZP))P>`>18G-20G82(N<)'`"``!(
M`<-(`=E,B?-,,<-$,=-,C1093(GI3(GK2,'K&TC!X05("=F)R4D!RDR)X4G!
M[`)(P>$>1(G320G,1(GQ38G&20''33'F3`'Y18G`3(N\)(@"``!%,<Y)B=E)
M`<Y(B=E)P>D;20''2,'A!4P)R8G)20'.3(GI2<'M`DC!X1Y%B?%,">E,BZPD
M@`(``$D!Q4T!Z$V)Y4DQS44QU4^-%"A-B<A-B<U)P>T;2<'@!4T)Z$6)P$T!
MPDF)V$C!ZP))P>`>18G53`G#18G@28G,B<E),=Q-`?A%,?1-B>Y-`>!-B>Q)
MP>X;2<'D!4T)]$V)SD6)Y$T!X$G!YAY)P>D"18G$30GQ3(NT))`"``!)`<9(
M`X0DF`(``$P!\4F)WDP!V(G;33'.2`'813'638GB3`'Q38GF2<'B!4G![AM-
M"?)$`=%-B>I)P>T"2<'B'HG)30G538G*1`-*($TQZD$!_4R)2B!%,=!)B<I,
MB6H83`'`28G(2<'J&P'I2<'@!4B)2@A-"=!$`<!(B0),B>!)P>P"2,'@'DP)
MX`'P2(LU6&T``$B)0A!(BX0DJ`(``$@K!G442('$N`(``%M>7UU!7$%=05Y!
M7\/H?@4``)!F9BX/'X0``````&:02(/L*+F`````Z#(&``"ZB:O-[[G^W+J8
M2(E0"$B!ZIG)^BM(QP`!(T5G2(E($$C'0!AV5#(02(E0($C'0"@`````2,=`
M,`````#'0'@`````2(/$*,-F+@\?A```````055!5%575E-(@^PH26/`3(M!
M*$B)STB)TTB+5S!)B<1(C7<X08T,P$PYP4B)3RA(8T]X2(/2`$C!Z!U(`=!(
MB4<PA<ET2+U`````2(G:*<U$.>5!#T_L2`'Q3&/M38GHZ)L%``"+1W@!Z(E'
M>(/X0'0.2(/$*%M>7UU!7$%=PY!(B?E!*>Q,`>OHDNC__T&#_#\/CM@```!!
M@^Q`18GE0<'M!D&-;0%(P>4&2`'=9I#S#V\#2(GY2(/#0`\1!O,/;TO0#Q%.
M$/,/;U/@#Q%6(/,/;UOP#Q%>,.A"Z/__2#GK=<U!P>4&12GL26/$2(/X"')=
M2(M5`$B-3@A(@^'X2(E7.$B+5`7X2(E4!OA(*<Y(`?!(*?5(@^#X2(/X"'(8
M2(/@^#'23(M$%0!,B0012(/""$@YPG+N1(EG>$B#Q"A;7E]=05Q!7<,/'X``
M````J`1U)$B%P'3?#[95`(A7.*@"=-0/MU0%_F:)5`;^Z\A(B=WI<O___XM5
M`(E7.(M4!?R)5`;\Z[`/'X0``````$%455=64TB#["!(BT(HBW(HBWHP2,'H
M`P_.@^`_#\](B<N-2`%(F$F)U,9$`CB`2(UJ.$ACP4@!Z(/Y.`^.3@$``$&X
M0````$$IR'00,=*)T8/"`<8$"`!$.<)R\DR)X>@KY___2<=$)#@`````2<=$
M)$``````2,=%$`````!(QT48`````$C'12``````2,=%*`````!(QT4P````
M`$&)?"1P3(GA08ET)'3HW^;__TF+!"1,B>%(P>@8B`-)BP0D2,'H$(A#`4F+
M!"2(8P))BP0DB$,#28M$)`A(P>@8B$,$28M$)`A(P>@0B$,%28M$)`B(8P9)
MBT0D"(A#!TF+1"002,'H&(A#"$F+1"002,'H$(A#"4F+1"00B&,*28M$)!"(
M0PM)BT0D&$C!Z!B(0PQ)BT0D&$C!Z!"(0PU)BT0D&(AC#DF+1"08B$,/28M$
M)"!(P>@8B$,028M$)"!(P>@0B$,128M$)""(8Q))BT0D((A#$TB#Q"!;7E]=
M05SI:@(``&8N#Q^$``````"Z.````$4QR2G*@_H(<DF)T4C'``````!(QT0(
M^`````!(C4@(2(/A^$@IR`'"@^+X@_H(#X+5_O__@^+X,<!!B<"#P`A.B0P!
M.=!R\NF]_O__9@\?1```]L($=2.%T@^$JO[__\8``/;"`@^$GO[__S')9HE,
M$/[IDO[__P\?`,<``````,=$$/P`````Z7S^__^0D)"0D)"0D)"0D)"0_R4:
MW@``D)#_)0K>``"0D/\E^MT``)"0_R7JW0``D)#_)=K=``"0D/\ERMT``)"0
M_R6ZW0``D)#_):K=``"0D/\EFMT``)"0_R6*W0``D)#_)7K=``"0D/\E:MT`
M`)"0_R5:W0``D)#_)4K=``"0D/\E.MT``)"0_R4JW0``D)#_)1K=``"0D/\E
M"MT``)"0_R7ZW```D)#_)>K<``"0D/\E0ML``)"0D)"0D)"0D)#_)3+<``"0
MD)"0D)"0D)"0_R4JW```D)"0D)"0D)"0D%%02#T`$```2(U,)!AR&4B!Z0`0
M``!(@PD`2"T`$```2#T`$```=^=(*<%(@PD`6%G#D)"0D)"0D)"0D)"0D)#_
M)<+:``"0D)"0D)"0D)"0_R6ZV@``D)"0D)"0D)"0D/\ENMH``)"0D)"0D)"0
MD)#_);+:``"0D)"0D)"0D)"0_R6JV@``D)"0D)"0D)"0D/\EJMH``)"0D)"0
MD)"0D)#_);+:``"0D)"0D)"0D)"0_R6RV@``D)"0D)"0D)"0D/\EJMH``)"0
MD)"0D)"0D)#_)<+:``"0D)"0D)"0D)"0_R6ZV@``D)"0D)"0D)"0D/\ELMH`
M`)"0D)"0D)"0D)#_):K:``"0D)"0D)"0D)"0_R6BV@``D)"0D)"0D)"0D/\E
MFMH``)"0D)"0D)"0D)#_)9+:``"0D)"0D)"0D)"0_R6*V@``D)"0D)"0D)"0
MD/\E@MH``)"0D)"0D)"0D)#_)7K:``"0D)"0D)"0D)"0_R5RV@``D)"0D)"0
MD)"0D/\E:MH``)"0D)"0D)"0D)#_)6+:``"0D)"0D)"0D)"0_R5:V@``D)"0
MD)"0D)"0D/\E4MH``)"0D)"0D)"0D)#_)6+:``"0D)"0D)"0D)"0_R5:V@``
MD)"0D)"0D)"0D/\E4MH``)"0D)"0D)"0D)#_)4K:``"0D)"0D)"0D)"0_R5"
MV@``D)"0D)"0D)"0D/\E.MH``)"0D)"0D)"0D)#_)3+:``"0D)"0D)"0D)"0
M_R4JV@``D)"0D)"0D)"0D/\E(MH``)"0D)"0D)"0D)#_)1K:``"0D)"0D)"0
MD)"0_R42V@``D)"0D)"0D)"0D/\E"MH``)"0D)"0D)"0D)#_)0+:``"0D)"0
MD)"0D)"0_R7ZV0``D)"0D)"0D)"0D/\E\MD``)"0D)"0D)"0D)!(@^PH,=+H
M50```/\5A]@``)"0D)"0D)"0D)"0D)"0D,.0D)"0D)"0D)"0D)"0D)##D)"0
MD)"0D)"0D)"0D)"0PY"0D)"0D)"0D)"0D)"0D,.0D)"0D)"0D)"0D)"0D)!6
M4TB#["A(B<Y(B=.Y"````.C*!```2(7;#X01`@``2+H`````50$``$B)<T!(
MC36,____2(T-50<``$B)D_````!(C15?!P``2(E34$B+%>S7``!(B;.`````
M2(TU;O___TB)2TA(C0US____2(E3&$B-%>BT``!(B;.(````2(TU:O___TB)
MBY````!(N4@!``"["P``2(FSF````$B--9O]__](B5,X2(T5,/W__TB)2PA(
MC0TE_O__2(ES($B--?H#``!(B5,H2(T5'P0``$C'`P````#'0Q`%````2(E+
M,$B)<WA(B9,8`0``2(7`#X2]````2(N`X````$B%P`^$K0```(L58[0``(72
M#X6?````2(,]0Q0````/A`T"``!(@ST]%`````^$;P(``$B#/3<4````#X11
M`@``2(,],10````/A#,"``!(@STK%`````^$%0(``$B#/244````#X3W`0``
M2(,]'Q0````/A-D!``!(@ST9%`````^$NP$``&8/;P73$P``#Q$`9@]O#=@3
M```/$4@09@]O%=P3```/$5`@9@]O'>`3```/$5@P,<GHA0,``#')2(F#Z```
M`$B-!743``!(B4-82(T%RA,``$B)0V!(C05?HP``2(E#:$B-!32U``!(B4-P
M2(T%N68``$B)@R`!``!(C07G9@``2(F#*`$``$B-!2VS__](B8,P`0``Z)$"
M``"X`0```$B#Q"A;7L,/'T0``$B)<$!(C36%_?__2+H`````50$``$B-#5P%
M``!(B;"`````2(TU=OW__TB)D/````!(C14H!0``2(FPB````$B--6K]__](
MB5!(2(L5O]4``$B)2%!(BXA``0``2(FPD````$B--5;]__](B;"8````2(TU
MN+(``$B)"DB-%8[[__](C0TG^___2(EP.$B^2`$``+L+``!(B7`(2(TU#OS_
M_TB)4"!(C17C`0``2(E(*$B-#0@"``!(QP``````QT`0!0```$B)<#!(B5!X
M2(F(&`$``$B%P`^$IO[__TB)PTB+@.````!(A<`/A>;]___ICO[__V8N#Q^$
M``````!(BQ!(B14F$@``Z>3]__^02(M0.$B)%4T2``#I-?[__TB+4#!(B14U
M$@``Z1?^__](BU`H2(D5'1(``.GY_?__2(M0($B)%042``#IV_W__TB+4!A(
MB17M$0``Z;W]__](BU`02(D5U1$``.F?_?__2(M0"$B)%;T1``#I@?W__TR+
M!<FQ``"+%<NQ``!(BPW,L0``Z=<````/'X``````2(/L.(/Z`G1W=U6%T@^$
M?0```#'`387`2(T5P?___TB)#9JQ```/E,!,B06`L0``QP5^L0```0```(D%
M8+$``.CK````2(/X_TB)!5BQ```/E<`/ML!(@\0XPP\?1```N`$```"#^@-U
M[+H#````2(/$..E8````#Q^$``````"Z`@```$B#Q#CI0@```&:0,=+H.0``
M`(7`=+M(BPT&L0``B40D+.A5````BT0D+$C'!>ZP``#_____ZYH/'T``Z3O_
M__^0D)"0D)"0D)"0D+@!````PY"0D)"0D)"0D)##D)"0D)"0D)"0D)"0D)"0
M_R6*TP``D)"0D)"0D)"0D/\EDM,``)"0D)"0D)"0D)#_)8K3``"0D)"0D)"0
MD)"0_R4*U```D)"0D)"0D)"0D$%5051(@^PH3(TMD;```$F)S$B)T4R)ZNCC
M^O__3(GJ3(GA2(/$*$%<05WI`````/\E0M,``)"0D)"0D)"0D)#_)0K5``"0
MD`\?A```````055!5%932(/L6(F,)(````!(B50D.$R)1"0PZ.#W__](C50D
M.$B+':1?``!,C40D,$B-C"2`````2(L#2(E$)$@QP.C(]O__2(L5<5\``$B+
M#5I?``!,C04#]___Z![W__](BP4G7P``2(LU,%\``(`X`'4SZ'[V__](B06'
MKP``2(G!2(7`#X2,`0``Z%[V__](BS4'7P``2(L.Z._V___&@"$*````2(L.
MZ.#V__](BPZ`B'($```"Z-'V__^`B'($```!BX0D@````(U(!(UP`TACR4C!
MX0/H2?;__TR+5"0X28G!2(D%$J\``$F+`DF)`4B-!;8A``!)B4$(2(L%`@\`
M`$F)01!(C06C(0``28E!&(N$)(````"#^`$/CNH```!)@\((@\`#N@0````/
M'X``````38L"2&/*@\(!28/""$V)!,DYT'7K2)A(BPVRK@``08GP2(T5`,7_
M_TG'!,$`````2,=$)"``````Z&+U__]!B<2%P'1.2(L-A*X``$R-;"1$Z%KU
M__],B>I(C0WS'@``Z-.^__](A<!T.DB+#5^N``#H,O7__^A5]?__2(M$)$A(
M*P-U9D2)X$B#Q%A;7D%<05W#2(L--JX``.CY]/__08G$ZZ%(BPT5K@``2(7)
M=`?H,]G__^NS2(T-0QX``$R)ZNAROO__2(D%\ZT``$B)P4B%P'25Z]FX!```
M`.DT____N0$```#H3?;__^CX]?__D)"0D)"0D)#I"[[__Y"0D)"0D)"0D)"0
M__________]`4D```0``````````````__________\`````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M`````````````````````````````````````````````````````````/__
M__\`````V')```$`````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M```````````````````````````````````````````````````````````E
M<R\E<P`O<')O8R]S96QF+V5X90!$>6YA3&]A9&5R.CIB;V]T7T1Y;F%,;V%D
M97(``$EN=&5R;F%L<SHZ4$%2.CI"3T]4`"X`4$%27U1%35``4$%27U!23T=.
M04U%`#H`)7,E<R5S`"\`4$523#5,24(`4$523$Q)0@!015),-4]05`!015),
M24\`4$%27TE.251)04Q)6D5$`%!!4E]34$%73D5$`%!!4E]#3$5!3@!005)?
M1$5"54<`4$%27T-!0TA%`%!!4E]'3$]"04Q?1$5"54<`4$%27T=,3T)!3%]4
M35!$25(`4$%27U1-4$1)4@!005)?1TQ/0D%,7U1%35``4$%27T=,3T)!3%]#
M3$5!3@`Q`%!!5$@`55-%4@!365-414T`+W1M<`!435!$25(`5$5-4$1)4@!4
M14U0`%1-4`!54T523D%-10`E,#)X`"5S)7,E<R5S`'!A<BT`)7,Z(&-R96%T
M:6]N(&]F('!R:79A=&4@<W5B9&ER96-T;W)Y("5S(&9A:6QE9"`H97)R;F\]
M)6DI"@```````"5S.B!P<FEV871E('-U8F1I<F5C=&]R>2`E<R!I<R!U;G-A
M9F4@*'!L96%S92!R96UO=F4@:70@86YD(')E=')Y('EO=7(@;W!E<F%T:6]N
M*0H`<&%R;"YE>&4`+G!A<@`E<R5S8V%C:&4M)7,E<P`E<R5S=&5M<"TE=25S
M`"5S)7-T96UP+25U+25U)7,`&`!P97)L`#``4$%2.CI086-K97(Z.E9%4E-)
M3TX`,2XP-34```````!5;F%B;&4@=&\@8W)E871E(&-A8VAE(&1I<F5C=&]R
M>0``````````)7,Z(&-R96%T:6]N(&]F('!R:79A=&4@8V%C:&4@<W5B9&ER
M96-T;W)Y("5S(&9A:6QE9"`H97)R;F\])6DI"@`M90`M+0``<&5R;'AS:2YC
M`````````&5V86P@)V5X96,@+W5S<B]B:6XO<&5R;"`@+5,@)#`@)'LQ*R(D
M0")])PII9B`P.R`C(&YO="!R=6YN:6YG('5N9&5R('-O;64@<VAE;&P*<&%C
M:V%G92!?7W!A<E]P;#L*;7D@*"1005)?34%'24,L("1P87)?=&5M<"P@)'!R
M;V=N86UE+"!`=&UP9FEL92P@)4UO9'5L94-A8VAE*3L*14Y$('L@:68@*"1%
M3E9[4$%27T-,14%.?2D@>PIR97%U:7)E($9I;&4Z.E1E;7`["G)E<75I<F4@
M1FEL93HZ0F%S96YA;64["G)E<75I<F4@1FEL93HZ4W!E8SL*;7D@)'1O<&1I
M<B`]($9I;&4Z.D)A<V5N86UE.CID:7)N86UE*"1P87)?=&5M<"D["F]U=',H
M<7%;4F5M;W9I;F<@9FEL97,@:6X@(B1P87)?=&5M<")=*3L*1FEL93HZ1FEN
M9#HZ9FEN9&1E<'1H*'-U8B![("@@+60@*2`_(')M9&ER(#H@=6YL:6YK('TL
M("1P87)?=&5M<"D["G)M9&ER("1P87)?=&5M<#L*:68@*"UD("1P87)?=&5M
M<"`F)B`D7D\@;F4@)TU35VEN,S(G*2!["FUY("1T;7`@/2!N97<@1FEL93HZ
M5&5M<"@*5$5-4$Q!5$4@/3X@)W1M<%A86%A8)RP*1$E2(#T^($9I;&4Z.D)A
M<V5N86UE.CID:7)N86UE*"1T;W!D:7(I+`I3549&25@@/3X@)RYC;60G+`I5
M3DQ)3DL@/3X@,"P**3L*;7D@)&9I;&5N86UE(#T@)'1M<"T^9FEL96YA;64[
M"G!R:6YT("1T;7`@/#PB+BXN(CL*>#TQ.R!W:&EL92!;(%PD>"`M;'0@,3`@
M73L@9&\*<FT@+7)F("<D<&%R7W1E;7`G"FEF(%L@7"$@+60@)R1P87)?=&5M
M<"<@73L@=&AE;@IB<F5A:PIF:0IS;&5E<"`Q"G@]8&5X<'(@7"1X("L@,6`*
M9&]N90IR;2`G)&9I;&5N86UE)PHN+BX*8VQO<V4@)'1M<#L*8VAM;V0@,#<P
M,"P@)&9I;&5N86UE.PIM>2`D8VUD(#T@(B1F:6QE;F%M92`^+V1E=B]N=6QL
M(#(^)C$@)B(["G-Y<W1E;2@D8VUD*3L*;W5T<RAQ<5M3<&%W;F5D(&)A8VMG
M<F]U;F0@<')O8V5S<R!T;R!P97)F;W)M(&-L96%N=7`Z("1F:6QE;F%M95TI
M.PI]"GT@?0I"14=)3B!["B1005)?34%'24,@/2`B7&Y005(N<&U<;B(["DEN
M=&5R;F%L<SHZ4$%2.CI"3T]4*"D@:68@9&5F:6YE9"`F26YT97)N86QS.CI0
M05(Z.D)/3U0["F5V86P@>PI?<&%R7VEN:71?96YV*"D["FUY("1Q=6EE="`]
M("$D14Y6>U!!4E]$14)51WT["FUY("5S>7,@/2`H"G!A=&A?<V5P("`@(#T^
M("@D7D\@/7X@+UY-4U=I;B\@/R`G.R<@.B`G.B<I+`I?97AE("`@("`@("`]
M/B`H)%Y/(#U^("]>*#\Z35-7:6Y\3U,R?&-Y9W=I;BDO(#\@)RYE>&4G(#H@
M)R<I+`I?9&5L:6T@("`@("`]/B`H)%Y/(#U^("]>35-7:6Y\3U,R+R`_("=<
M7"<@.B`G+R<I+`HI.PI?<V5T7W!R;V=N86UE*"D["F]U=',H<7%;7"1P<F]G
M;F%M92`]("(D<')O9VYA;64B72D["E]S971?<&%R7W1E;7`H*3L*;W5T<RAQ
M<5M<)$5.5GM005)?5$5-4'T@/2`B)$5.5GM005)?5$5-4'TB72D["FUY("@D
M<W1A<G1?<&]S+"`D9&%T85]P;W,I.PI-04=)0SH@>PIL;V-A;"`D4TE'>U]?
M5T%23E]??2`]('-U8B![?3L*=6YL97-S("AO<&5N(%]&2"P@)SPZ<F%W)RP@
M)'!R;V=N86UE*2!["F]U=',H<7%;0V%N)W0@<F5A9"!F<F]M(&9I;&4@(B1P
M<F]G;F%M92)=*3L@(",@9&]N)W0@=7-E("0A(&AE<F4@87,@:70@<F5Q=6ER
M97,@17)R;F\N<&T*;&%S="!-04=)0SL*?0IM>2`D8VAU;FM?<VEZ92`](#8T
M("H@,3`R-#L*;7D@)&)U9CL*;7D@)'-I>F4@/2`M<R!?1D@["FUY("1M86=I
M8U]P;W,@/2`D<VEZ92`M("1S:7IE("4@)&-H=6YK7W-I>F4[(",@3D]413H@
M)&UA9VEC7W!O<R!I<R!A(&UU;'1I<&QE(&]F("1C:'5N:U]S:7IE"G=H:6QE
M("@D;6%G:6-?<&]S(#X](#`I('L*<V5E:R!?1D@L("1M86=I8U]P;W,L(#`[
M"G)E860@7T9(+"`D8G5F+"`D8VAU;FM?<VEZ92`K(&QE;F=T:"@D4$%27TU!
M1TE#*3L*:68@*"AM>2`D:2`](')I;F1E>"@D8G5F+"`D4$%27TU!1TE#*2D@
M/CT@,"D@>PHD;6%G:6-?<&]S("L]("1I.PIL87-T.PI]"B1M86=I8U]P;W,@
M+3T@)&-H=6YK7W-I>F4["GT*:68@*"1M86=I8U]P;W,@/"`P*2!["F]U=',H
M<7%;0V%N)W0@9FEN9"!M86=I8R!S=')I;F<@(B1005)?34%'24,B(&EN(&9I
M;&4@(B1P<F]G;F%M92)=*3L*;&%S="!-04=)0SL*?0IO=71S*")&;W5N9"!0
M05(@;6%G:6,@870@<&]S:71I;VX@)&UA9VEC7W!O<R(I.PIS965K(%]&2"P@
M)&UA9VEC7W!O<R`M(#0L(#`["G)E860@7T9(+"`D8G5F+"`T.PIM>2`D;V9F
M<V5T(#T@=6YP86-K*").(BP@)&)U9BD["F]U=',H(D]F9G-E="!F<F]M('-T
M87)T(&]F($9)3$5S(&ES("1O9F9S970B*3L*<V5E:R!?1D@L("1M86=I8U]P
M;W,@+2`T("T@)&]F9G-E="P@,#L*)&1A=&%?<&]S(#T@=&5L;"!?1D@["FUY
M("5R97%U:7)E7VQI<W0["G)E860@7T9(+"`D8G5F+"`T.R`@("`@("`@("`@
M("`@("`@("`@("`@("`@(",@<F5A9"!T:&4@9FER<W0@(D9)3$4B"G=H:6QE
M("@D8G5F(&5Q(")&24Q%(BD@>PIR96%D(%]&2"P@)&)U9BP@-#L*<F5A9"!?
M1D@L("1B=68L('5N<&%C:R@B3B(L("1B=68I.PIM>2`D9G5L;&YA;64@/2`D
M8G5F.PIO=71S*'%Q6U5N<&%C:VEN9R!&24Q%("(D9G5L;&YA;64B+BXN72D[
M"FUY("1C<F,@/2`H("1F=6QL;F%M92`]?B!S?%XH6V$M9EQD77LX?2DO?'P@
M*2`_("0Q(#H@=6YD968["FUY("@D8F%S96YA;64L("1E>'0I(#T@*"1B=68@
M/7X@;7PH/SHN*B\I/R@N*BDH7"XN*BE\*3L*<F5A9"!?1D@L("1B=68L(#0[
M"G)E860@7T9(+"`D8G5F+"!U;G!A8VLH(DXB+"`D8G5F*3L*:68@*&1E9FEN
M960H)&5X="D@86YD("1E>'0@(7X@+UPN*#\Z<&U\<&Q\:7A\86PI)"]I*2![
M"FUY("1F:6QE;F%M92`](%]S879E7V%S*"(D8W)C)&5X="(L("1B=68L(#`W
M-34I.PHD4$%2.CI(96%V>3HZ1G5L;$-A8VAE>R1F=6QL;F%M97T@/2`D9FEL
M96YA;64["B1005(Z.DAE879Y.CI&=6QL0V%C:&5[)&9I;&5N86UE?2`]("1F
M=6QL;F%M93L*?0IE;'-I9B`H("1F=6QL;F%M92`]?B!M?%XO/W-H;&EB+WP@
M86YD(&1E9FEN960@)$5.5GM005)?5$5-4'T@*2!["FUY("1F:6QE;F%M92`]
M(%]S879E7V%S*"(D8F%S96YA;64D97AT(BP@)&)U9BP@,#<U-2D["F]U=',H
M(E-(3$E".B`D9FEL96YA;65<;B(I.PI]"F5L<V4@>PHD36]D=6QE0V%C:&5[
M)&9U;&QN86UE?2`]('L*8G5F(#T^("1B=68L"F-R8R`]/B`D8W)C+`IN86UE
M(#T^("1F=6QL;F%M92P*?3L*)')E<75I<F5?;&ES='LD9G5L;&YA;65]*RL[
M"GT*<F5A9"!?1D@L("1B=68L(#0["GT*;&]C86P@0$E.0R`]("AS=6(@>PIM
M>2`H)'-E;&8L("1M;V1U;&4I(#T@0%\["G)E='5R;B!I9B!R968@)&UO9'5L
M92!O<B`A)&UO9'5L93L*;7D@)&EN9F\@/2`D36]D=6QE0V%C:&5[)&UO9'5L
M97T["G)E='5R;B!U;FQE<W,@)&EN9F\["F1E;&5T92`D<F5Q=6ER95]L:7-T
M>R1M;V1U;&5].PHD24Y#>R1M;V1U;&5](#T@(B]L;V%D97(O)&EN9F\O)&UO
M9'5L92(["FEF("@D14Y6>U!!4E]#3$5!3GT@86YD(&1E9FEN960H)DE/.CI&
M:6QE.CIN97<I*2!["FUY("1F:"`]($E/.CI&:6QE+3YN97=?=&UP9FEL92!O
M<B!D:64@(D-A;B=T(&-R96%T92!T96UP(&9I;&4Z("0A(CL*)&9H+3YB:6YM
M;V1E*"D["B1F:"T^<')I;G0H)&EN9F\M/GMB=69]*3L*)&9H+3YS965K*#`L
M(#`I.PIR971U<FX@)&9H.PI]"F5L<V4@>PIM>2`D9FEL96YA;64@/2!?<V%V
M95]A<R@B)&EN9F\M/GMC<F-]+G!M(BP@)&EN9F\M/GMB=69]*3L*)&EN9F\M
M/GMF:6QE?2`]("1F:6QE;F%M93L*;W!E;B!M>2`D9F@L("<\.G)A=R<L("1F
M:6QE;F%M92!O<B!D:64@<7%;0V%N)W0@<F5A9"`B)&9I;&5N86UE(CH@)"%=
M.PIR971U<FX@)&9H.PI]"F1I92`B0F]O='-T<F%P<&EN9R!F86EL960Z(&-A
M;B=T(&9I;F0@;6]D=6QE("1M;V1U;&4A(CL*?2P@0$E.0RD["G)E<75I<F4@
M6%-,;V%D97(["G)E<75I<F4@4$%2.CI(96%V>3L*<F5Q=6ER92!#87)P.CI(
M96%V>3L*<F5Q=6ER92!%>'!O<G1E<CHZ2&5A=GD["E!!4CHZ2&5A=GDZ.E]I
M;FET7V1Y;F%L;V%D97(H*3L*<F5Q=6ER92!)3SHZ1FEL93L*=VAI;&4@*&UY
M("1F:6QE;F%M92`]("AS;W)T(&ME>7,@)7)E<75I<F5?;&ES="E;,%TI('L*
M=6YL97-S("@D24Y#>R1F:6QE;F%M97T@;W(@)&9I;&5N86UE(#U^("]"4T10
M04XO*2!["FEF("@D9FEL96YA;64@/7X@+UPN<&UC/R0O:2D@>PIR97%U:7)E
M("1F:6QE;F%M93L*?0IE;'-E('L*9&\@)&9I;&5N86UE('5N;&5S<R`D9FEL
M96YA;64@/7X@+W-I=&5C=7-T;VUI>F5<+G!L)"\["GT*?0ID96QE=&4@)')E
M<75I<F5?;&ES='LD9FEL96YA;65].PI]"G5N;&5S<R`H)&)U9B!E<2`B4$M<
M,#`S7#`P-"(I('L*;W5T<RAQ<5M.;R!Z:7`@9F]U;F0@869T97(@1DE,12!S
M96-T:6]N(&EN(&9I;&4@(B1P<F]G;F%M92)=*3L*;&%S="!-04=)0R`["GT*
M)'-T87)T7W!O<R`]("AT96QL(%]&2"D@+2`T.R`@("`@("`@("`@("`@("`C
M('-T87)T(&]F('II<`I]"FUY($!P87)?87)G<SL*;7D@*"1O=70L("1B=6YD
M;&4L("1L;V=F:"P@)&-A8VAE7VYA;64I.PID96QE=&4@)$5.5GM005)?05!0
M7U)%55-%?3L@(R!S86YI=&EZ92`H4D554T4@;6%Y(&)E(&$@<V5C=7)I='D@
M<')O8FQE;2D*)'%U:65T(#T@,"!U;FQE<W,@)$5.5GM005)?1$5"54=].PII
M9B`H(21S=&%R=%]P;W,@;W(@*"1!4D=66S!=(&5Q("<M+7!A<BUO<'1I;VYS
M)R`F)B!S:&EF="DI('L*;7D@)61I<W1?8VUD(#T@<7<H"G`@("!B;&EB7W1O
M7W!A<@II("`@:6YS=&%L;%]P87(*=2`@('5N:6YS=&%L;%]P87(*<R`@('-I
M9VY?<&%R"G8@("!V97)I9GE?<&%R"BD["FEF("A`05)'5B!A;F0@)$%21U9;
M,%T@97$@)RTM<F5U<V4G*2!["G-H:69T($!!4D=6.PHD14Y6>U!!4E]!4%!?
M4D554T5](#T@<VAI9G0@0$%21U8["GT*96QS92![(",@;F]R;6%L('!A<FP@
M8F5H879I;W5R"FUY($!A9&1?=&]?:6YC.PIW:&EL92`H0$%21U8I('L*)$%2
M1U9;,%T@/7X@+UXM*%M!24U/0DQB<7!I=7-4=ETI*"XJ*2\@;W(@;&%S=#L*
M:68@*"0Q(&5Q("=))RD@>PIP=7-H($!A9&1?=&]?:6YC+"`D,CL*?0IE;'-I
M9B`H)#$@97$@)TTG*2!["F5V86P@(G5S92`D,B(["GT*96QS:68@*"0Q(&5Q
M("=!)RD@>PIU;G-H:69T($!P87)?87)G<RP@)#(["GT*96QS:68@*"0Q(&5Q
M("=/)RD@>PHD;W5T(#T@)#(["GT*96QS:68@*"0Q(&5Q("=B)RD@>PHD8G5N
M9&QE(#T@)W-I=&4G.PI]"F5L<VEF("@D,2!E<2`G0B<I('L*)&)U;F1L92`]
M("=A;&PG.PI]"F5L<VEF("@D,2!E<2`G<2<I('L*)'%U:65T(#T@,3L*?0IE
M;'-I9B`H)#$@97$@)TPG*2!["F]P96X@)&QO9V9H+"`B/CXB+"`D,B!O<B!D
M:64@<7%;0V%N)W0@;W!E;B!L;V<@9FEL92`B)#(B.B`D(5T["GT*96QS:68@
M*"0Q(&5Q("=4)RD@>PHD8V%C:&5?;F%M92`]("0R.PI]"G-H:69T*$!!4D=6
M*3L*:68@*&UY("1C;60@/2`D9&ES=%]C;61[)#%]*2!["F1E;&5T92`D14Y6
M>R=005)?5$5-4"=].PII;FET7VEN8R@I.PIR97%U:7)E(%!!4CHZ1&ES=#L*
M)GLB4$%2.CI$:7-T.CHD8VUD(GTH*2!U;FQE<W,@0$%21U8["B9[(E!!4CHZ
M1&ES=#HZ)&-M9")]*"1?*2!F;W(@0$%21U8["F5X:70["GT*?0IU;G-H:69T
M($!)3D,L($!A9&1?=&]?:6YC.PI]"GT*:68@*"1O=70I('L*>PIR97%U:7)E
M($E/.CI&:6QE.PIR97%U:7)E($%R8VAI=F4Z.EII<#L*<F5Q=6ER92!$:6=E
M<W0Z.E-(03L*?0IM>2`D<&%R(#T@<VAI9G0H0$%21U8I.PIM>2`D>FEP.PII
M9B`H9&5F:6YE9"`D<&%R*2!["F]P96X@;7D@)&9H+"`G/#IR87<G+"`D<&%R
M(&]R(&1I92!Q<5M#86XG="!F:6YD('!A<B!F:6QE("(D<&%R(CH@)"%=.PIB
M;&5S<R@D9F@L("=)3SHZ1FEL92<I.PHD>FEP(#T@07)C:&EV93HZ6FEP+3YN
M97<["B@@)'II<"T^<F5A9$9R;VU&:6QE2&%N9&QE*"1F:"P@)'!A<BD@/3T@
M07)C:&EV93HZ6FEP.CI!6E]/2R@I("D*;W(@9&EE('%Q6T5R<F]R(')E861I
M;F<@>FEP(&%R8VAI=F4@(B1P87(B73L*?0II9B`H9&5F:6YE9"`D<&%R*2![
M"F]P96X@;7D@)'!H+"`G/#IR87<G+"`D<&%R(&]R(&1I92!Q<5M#86XG="!R
M96%D('!A<B!F:6QE("(D<&%R(CH@)"%=.PIM>2`D8G5F.PIR96%D("1P:"P@
M)&)U9BP@-#L*9&EE('%Q6R(D<&%R(B!I<R!N;W0@82!P87(@9FEL95T@=6YL
M97-S("1B=68@97$@(E!+7#`P,UPP,#0B.PIC;&]S92`D<&@["GT*0W)E871E
M4&%T:"@D;W5T*2`["FUY("1F:"`]($E/.CI&:6QE+3YN97<H"B1O=70L"DE/
M.CI&:6QE.CI/7T-214%4*"D@?"!)3SHZ1FEL93HZ3U]74D].3%DH*2!\($E/
M.CI&:6QE.CI/7U1254Y#*"DL"C`W-S<L"BD@;W(@9&EE('%Q6T-A;B=T(&-R
M96%T92!F:6QE("(D;W5T(CH@)"%=.PHD9F@M/F)I;FUO9&4H*3L*<V5E:R!?
M1D@L(#`L(#`["FUY("1L;V%D97(["FEF("AD969I;F5D("1D871A7W!O<RD@
M>PIR96%D(%]&2"P@)&QO861E<BP@)&1A=&%?<&]S.PI](&5L<V4@>PIL;V-A
M;"`D+R`]('5N9&5F.PHD;&]A9&5R(#T@/%]&2#X["GT*:68@*"$D14Y6>U!!
M4E]615)"051)37T@86YD("1L;V%D97(@/7X@+UXH/SHC(7Q<0')E;2DO*2![
M"G)E<75I<F4@4$%2.CI&:6QT97(Z.E!O9%-T<FEP.PI005(Z.D9I;'1E<CHZ
M4&]D4W1R:7`M/F%P<&QY*%PD;&]A9&5R+"`D,"D["GT*)&9H+3YP<FEN="@D
M;&]A9&5R*0IO<B!D:64@<7%;17)R;W(@=W)I=&EN9R!L;V%D97(@=&\@(B1O
M=70B.B`D(5T["FEF("@D8G5N9&QE*2!["G)E<75I<F4@4$%2.CI(96%V>3L*
M4$%2.CI(96%V>3HZ7VEN:71?9'EN86QO861E<B@I.PII;FET7VEN8R@I.PIR
M97%U:7)E7VUO9'5L97,H*3L*;7D@0&EN8R`](&=R97`@>R`A+T)31%!!3B\@
M?0IG<F5P('L**"1B=6YD;&4@;F4@)W-I=&4G*2!O<@HH)%\@;F4@)$-O;F9I
M9SHZ0V]N9FEG>V%R8VAL:6)E>'!](&%N9`HD7R!N92`D0V]N9FEG.CI#;VYF
M:6=[<')I=FQI8F5X<'TI.PI]($!)3D,["G-\+RM\+WQG+"!S?"\D?'P@9F]R
M96%C:"!`:6YC.PIM>2`E9FEL97,["B1F:6QE<WLD7WTK*R!F;W(@0$1Y;F%,
M;V%D97(Z.F1L7W-H87)E9%]O8FIE8W1S+"!V86QU97,@)4E.0SL*;7D@)&QI
M8E]E>'0@/2`D0V]N9FEG.CI#;VYF:6=[;&EB7V5X='T[("`@("`@("`@(R!8
M6%@@;&EB7V5X="!V<R!D;&5X="`_"FUY("5W<FET=&5N.PIF;W)E86-H(&UY
M("1K97D@*'-O<G0@:V5Y<R`E9FEL97,I('L*;7D@*"1F:6QE+"`D;F%M92D[
M"FEF("AD969I;F5D*&UY("1F8R`]("1005(Z.DAE879Y.CI&=6QL0V%C:&5[
M)&ME>7TI*2!["B@D9FEL92P@)&YA;64I(#T@*"1K97DL("1F8RD["GT*96QS
M92!["F9O<F5A8V@@;7D@)&1I<B`H0&EN8RD@>PII9B`H)&ME>2`]?B!M?%Y<
M421D:7)<12\H+BHI)'QI*2!["B@D9FEL92P@)&YA;64I(#T@*"1K97DL("0Q
M*3L*;&%S=#L*?0II9B`H)&ME>2`]?B!M?%XO;&]A9&5R+UM>+UTK+R@N*BDD
M?"D@>PII9B`H;7D@)')E9B`]("1-;V1U;&5#86-H97LD,7TI('L**"1F:6QE
M+"`D;F%M92D@/2`H)')E9BP@)#$I.PIL87-T.PI]"FEF("@M9B`B)&1I<B\D
M,2(I('L**"1F:6QE+"`D;F%M92D@/2`H(B1D:7(O)#$B+"`D,2D["FQA<W0[
M"GT*?0I]"GT*;F5X="!U;FQE<W,@9&5F:6YE9"`D;F%M93L*;F5X="!I9B`D
M=W)I='1E;GLD;F%M97TK*SL*;F5X="!I9B`A<F5F*"1F:6QE*2!A;F0@)&9I
M;&4@/7X@+UPN7%$D;&EB7V5X=%Q%)"]I.PIO=71S*'-P<FEN=&8H<7%;4&%C
M:VEN9R!&24Q%("(E<R(N+BY=+"!R968@)&9I;&4@/R`D9FEL92T^>VYA;65]
M(#H@)&9I;&4I*3L*;7D@)&-O;G1E;G0["FEF("AR968H)&9I;&4I*2!["B1C
M;VYT96YT(#T@)&9I;&4M/GMB=69].PI]"F5L<V4@>PIL;V-A;"`D+R`]('5N
M9&5F.PIO<&5N(&UY("1T:"P@)SPZ<F%W)RP@)&9I;&4@;W(@9&EE('%Q6T-A
M;B=T(')E860@(B1F:6QE(CH@)"%=.PHD8V]N=&5N="`](#PD=&@^.PIC;&]S
M92`D=&@["E!!4CHZ1FEL=&5R.CI0;V13=')I<"T^87!P;'DH7"1C;VYT96YT
M+"`B/&5M8F5D9&5D/B\D;F%M92(I"FEF("$D14Y6>U!!4E]615)"051)37T@
M86YD("1N86UE(#U^("]<+B@_.G!M?&EX?&%L*20O:3L*4$%2.CI&:6QT97(Z
M.E!A=&-H0V]N=&5N="T^;F5W+3YA<'!L>2A<)&-O;G1E;G0L("1F:6QE+"`D
M;F%M92D["GT*)&9H+3YP<FEN="@B1DE,12(L"G!A8VLH)TXG+"!L96YG=&@H
M)&YA;64I("L@.2DL"G-P<FEN=&8H(B4P.'@O)7,B+"!!<F-H:79E.CI::7`Z
M.F-O;7!U=&5#4D,S,B@D8V]N=&5N="DL("1N86UE*2P*<&%C:R@G3B<L(&QE
M;F=T:"@D8V]N=&5N="DI+`HD8V]N=&5N="D*;W(@9&EE('%Q6T5R<F]R('=R
M:71I;F<@96UB961D960@1DE,12!T;R`B)&]U="(Z("0A73L*;W5T<RAQ<5M7
M<FET=&5N(&%S("(D;F%M92)=*3L*?0I]"FEF("@D>FEP*2!["B1Z:7`M/G=R
M:71E5&]&:6QE2&%N9&QE*"1F:"D@/3T@07)C:&EV93HZ6FEP.CI!6E]/2PIO
M<B!D:64@<7%;17)R;W(@=W)I=&EN9R!Z:7`@<&%R="!O9B`B)&]U=")=.PI]
M"B1C86-H95]N86UE(#T@<W5B<W1R("1C86-H95]N86UE+"`P+"`T,#L*:68@
M*"$D8V%C:&5?;F%M92!A;F0@;7D@)&UT:6UE(#T@*'-T870H)&]U="DI6SE=
M*2!["FUY("1C='@@/2!$:6=E<W0Z.E-(02T^;F5W*#$I.PIO<&5N(&UY("1T
M:"P@(CPZ<F%W(BP@)&]U=#L*)&-T>"T^861D9FEL92@D=&@I.PIC;&]S92`D
M=&@["B1C86-H95]N86UE(#T@)&-T>"T^:&5X9&EG97-T.PI]"B1C86-H95]N
M86UE("X](")<,"(@>"`H-#`@+2!L96YG=&@@)&-A8VAE7VYA;64I.PHD8V%C
M:&5?;F%M92`N/2`B7#!#04-(12(["FUY("1O9F9S970@/2`D9F@M/G1E;&P@
M*R!L96YG=&@H)&-A8VAE7VYA;64I("T@;&5N9W1H*"1L;V%D97(I.PHD9F@M
M/G!R:6YT*"1C86-H95]N86UE+`IP86-K*"=.)RP@)&]F9G-E="DL"B1005)?
M34%'24,I"B8F("1F:"T^8VQO<V4*;W(@9&EE('%Q6T5R<F]R('=R:71I;F<@
M=')A:6QE<B!O9B`B)&]U="(Z("0A73L*8VAM;V0@,#<U-2P@)&]U=#L*97AI
M=#L*?0I["FQA<W0@=6YL97-S(&1E9FEN960@)'-T87)T7W!O<SL*7V9I>%]P
M<F]G;F%M92@I.PIR97%U:7)E(%!!4CL*4$%2.CI(96%V>3HZ7VEN:71?9'EN
M86QO861E<B@I.PI["G)E<75I<F4@1FEL93HZ1FEN9#L*<F5Q=6ER92!!<F-H
M:79E.CI::7`["GT*;7D@)&9H(#T@24\Z.D9I;&4M/FYE=SL@("`@("`@("`@
M("`@("`@("`@("`@("`@("`@(",@07)C:&EV93HZ6FEP(&]P97)A=&5S(&]N
M(&%N($E/.CI(86YD;&4*)&9H+3YF9&]P96XH9FEL96YO*%]&2"DL("=R)RD@
M;W(@9&EE('%Q6V9D;W!E;B@I(&9A:6QE9#H@)"%=.PI!<F-H:79E.CI::7`Z
M.G-E=$-H=6YK4VEZ92@M<R!?1D@I.PIM>2`D>FEP(#T@07)C:&EV93HZ6FEP
M+3YN97<["B@D>FEP+3YR96%D1G)O;49I;&5(86YD;&4H)&9H+"`D<')O9VYA
M;64I(#T]($%R8VAI=F4Z.EII<#HZ05I?3TLH*2D*;W(@9&EE('%Q6T5R<F]R
M(')E861I;F<@>FEP(&%R8VAI=F4@(B1P<F]G;F%M92)=.PI!<F-H:79E.CI:
M:7`Z.G-E=$-H=6YK4VEZ92@V-"`J(#$P,C0I.PHD<75I970@/2`A)$5.5GM0
M05)?1$5"54=].PIO=71S*")296%D:6YG($U%5$$N>6UL+BXN(BD["FEF("AM
M>2`D;65T82`]("1Z:7`M/F-O;G1E;G1S*"=-151!+GEM;"<I*2!["B1M971A
M(#U^(',O+BI><&%R.EQS*B0O+VUS.PHD;65T82`]?B!S+UY<4RXJ+R]M<SL*
M:68@*&UY("@D8VQE86XI(#T@)&UE=&$@/7X@+UY<<RMC;&5A;CH@*"XJ*20O
M;2D@>PHD8VQE86X@/7X@+UY<<RI\7',J)"]G.PHD14Y6>U!!4E]#3$5!3GT@
M/2`Q(&EF("1C;&5A;CL*?0I]"G!U<V@@0%!!4CHZ3&EB0V%C:&4L("1Z:7`[
M"B1005(Z.DQI8D-A8VAE>R1P<F]G;F%M97T@/2`D>FEP.PIO=71S*")%>'1R
M86-T:6YG('II<"XN+B(I.PII9B`H9&5F:6YE9"`D14Y6>U!!4E]414U0?2D@
M>R`C('-H;W5L9"!B92!S970@870@=&AI<R!P;VEN="$*9F]R96%C:"!M>2`D
M;65M8F5R("@@)'II<"T^;65M8F5R<R`I('L*;F5X="!I9B`D;65M8F5R+3YI
M<T1I<F5C=&]R>3L*;7D@)&UE;6)E<E]N86UE(#T@)&UE;6)E<BT^9FEL94YA
M;64["FYE>'0@=6YL97-S("1M96UB97)?;F%M92`]?B!M>PI>"B\_<VAL:6(O
M"B@_.B1#;VYF:6<Z.D-O;F9I9WMV97)S:6]N?2\I/PHH/SHD0V]N9FEG.CI#
M;VYF:6=[87)C:&YA;65]+RD_"BA;7B]=*RD*)`I]>#L*;7D@)&5X=')A8W1?
M;F%M92`]("0Q.PIM>2`D9&5S=%]N86UE(#T@1FEL93HZ4W!E8RT^8V%T9FEL
M92@D14Y6>U!!4E]414U0?2P@)&5X=')A8W1?;F%M92D["FEF("@M9B`D9&5S
M=%]N86UE("8F("US(%\@/3T@)&UE;6)E<BT^=6YC;VUP<F5S<V5D4VEZ92@I
M*2!["F]U=',H<7%;4VMI<'!I;F<@(B1M96UB97)?;F%M92(@<VEN8V4@:70@
M86QR96%D>2!E>&ES=',@870@(B1D97-T7VYA;64B72D["GT@96QS92!["F]U
M=',H<7%;17AT<F%C=&EN9R`B)&UE;6)E<E]N86UE(B!T;R`B)&1E<W1?;F%M
M92(N+BY=*3L**"1M96UB97(M/F5X=')A8W14;T9I;&5.86UE9"@D9&5S=%]N
M86UE*2`]/2!!<F-H:79E.CI::7`Z.D%:7T]+*"DI"F]R(&1I92!Q<5M%<G)O
M<B!E>'1R86-T:6YG('II<"!M96UB97(@=&\@(B1D97-T7VYA;64B73L*8VAM
M;V0H,#4U-2P@)&1E<W1?;F%M92D@:68@)%Y/(&5Q(")H<'5X(CL*?0I]"GT*
M?0IU;FQE<W,@*"1005(Z.DQI8D-A8VAE>R1P<F]G;F%M97TI('L*9&EE(#P\
M("(N(B!U;FQE<W,@0$%21U8["E5S86=E.B`D,"!;("U!;&EB+G!A<B!=(%L@
M+4ED:7(@72!;("U-;6]D=6QE(%T@6R!S<F,N<&%R(%T@6R!P<F]G<F%M+G!L
M(%T*)#`@6R`M0GPM8B!=(%LM3V]U=&9I;&5=('-R8RYP87(*+@HD14Y6>U!!
M4E]04D]'3D%-17T@/2`D<')O9VYA;64@/2`D,"`]('-H:69T*$!!4D=6*3L*
M?0IS=6(@0W)E871E4&%T:"!["FUY("@D;F%M92D@/2!`7SL*<F5Q=6ER92!&
M:6QE.CI"87-E;F%M93L*;7D@*"1B87-E;F%M92P@)'!A=&@L("1E>'0I(#T@
M1FEL93HZ0F%S96YA;64Z.F9I;&5P87)S92@D;F%M92P@*"=<+BXJ)RDI.PIR
M97%U:7)E($9I;&4Z.E!A=&@["D9I;&4Z.E!A=&@Z.FUK<&%T:"@D<&%T:"D@
M=6YL97-S*"UE("1P871H*3L@(R!M:W!A=&@@9&EE<R!W:71H(&5R<F]R"GT*
M<W5B(')E<75I<F5?;6]D=6QE<R!["G)E<75I<F4@;&EB.PIR97%U:7)E($1Y
M;F%,;V%D97(["G)E<75I<F4@:6YT96=E<CL*<F5Q=6ER92!S=')I8W0["G)E
M<75I<F4@=V%R;FEN9W,["G)E<75I<F4@=F%R<SL*<F5Q=6ER92!#87)P.PIR
M97%U:7)E($-A<G`Z.DAE879Y.PIR97%U:7)E($5R<FYO.PIR97%U:7)E($5X
M<&]R=&5R.CI(96%V>3L*<F5Q=6ER92!%>'!O<G1E<CL*<F5Q=6ER92!&8VYT
M;#L*<F5Q=6ER92!&:6QE.CI496UP.PIR97%U:7)E($9I;&4Z.E-P96,["G)E
M<75I<F4@6%-,;V%D97(["G)E<75I<F4@0V]N9FEG.PIR97%U:7)E($E/.CI(
M86YD;&4["G)E<75I<F4@24\Z.D9I;&4["G)E<75I<F4@0V]M<')E<W,Z.EIL
M:6(["G)E<75I<F4@07)C:&EV93HZ6FEP.PIR97%U:7)E($1I9V5S=#HZ4TA!
M.PIR97%U:7)E(%!!4CL*<F5Q=6ER92!005(Z.DAE879Y.PIR97%U:7)E(%!!
M4CHZ1&ES=#L*<F5Q=6ER92!005(Z.D9I;'1E<CHZ4&]D4W1R:7`["G)E<75I
M<F4@4$%2.CI&:6QT97(Z.E!A=&-H0V]N=&5N=#L*<F5Q=6ER92!A='1R:6)U
M=&5S.PIE=F%L('L@<F5Q=6ER92!#=V0@?3L*979A;"![(')E<75I<F4@5VEN
M,S(@?3L*979A;"![(')E<75I<F4@4V-A;&%R.CI5=&EL('T["F5V86P@>R!R
M97%U:7)E($%R8VAI=F4Z.E5N>FEP.CI"=7)S="!].PIE=F%L('L@<F5Q=6ER
M92!4:64Z.DAA<V@Z.DYA;65D0V%P='5R92!].PIE=F%L('L@<F5Q=6ER92!0
M97)L24\[(')E<75I<F4@4&5R;$E/.CIS8V%L87(@?3L*979A;"![(')E<75I
M<F4@=71F."!].PI]"G-U8B!?<V5T7W!A<E]T96UP('L*:68@*&1E9FEN960@
M)$5.5GM005)?5$5-4'T@86YD("1%3E9[4$%27U1%35!](#U^("\H+BLI+RD@
M>PHD<&%R7W1E;7`@/2`D,3L*<F5T=7)N.PI]"F9O<F5A8V@@;7D@)'!A=&@@
M*`HH;6%P("1%3E9[)%]]+"!Q=R@@4$%27U1-4$1)4B!435!$25(@5$5-4$1)
M4B!414U0(%1-4"`I*2P*<7<H($,Z7%Q414U0("]T;7`@+B`I"BD@>PIN97AT
M('5N;&5S<R!D969I;F5D("1P871H(&%N9"`M9"`D<&%T:"!A;F0@+7<@)'!A
M=&@["FUY("1U<V5R;F%M93L*;7D@)'!W=6ED.PIE=F%L('LH)'!W=6ED*2`]
M(&=E='!W=6ED*"0^*2!I9B!D969I;F5D("0^.WT["FEF("@@9&5F:6YE9"@F
M5VEN,S(Z.DQO9VEN3F%M92D@*2!["B1U<V5R;F%M92`]("97:6XS,CHZ3&]G
M:6Y.86UE.PI]"F5L<VEF("AD969I;F5D("1P=W5I9"D@>PHD=7-E<FYA;64@
M/2`D<'=U:60["GT*96QS92!["B1U<V5R;F%M92`]("1%3E9[55-%4DY!345]
M('Q\("1%3E9[55-%4GT@?'P@)U-94U1%32<["GT*)'5S97)N86UE(#U^(',O
M7%<O7R]G.PIM>2`D<W1M<&1I<B`]("(D<&%T:"1S>7-[7V1E;&EM?7!A<BTB
M+G5N<&%C:R@B2"HB+"`D=7-E<FYA;64I.PIM:V1I<B`D<W1M<&1I<BP@,#<U
M-3L*:68@*"$D14Y6>U!!4E]#3$5!3GT@86YD(&UY("1M=&EM92`]("AS=&%T
M*"1P<F]G;F%M92DI6SE=*2!["F]P96X@;7D@)&9H+"`B/#IR87<B+"`D<')O
M9VYA;64@;W(@9&EE('%Q6T-A;B=T(')E860@(B1P<F]G;F%M92(Z("0A73L*
M<V5E:R`D9F@L("TQ."P@,CL*;7D@)&)U9CL*<F5A9"`D9F@L("1B=68L(#8[
M"FEF("@D8G5F(&5Q(")<,$-!0TA%(BD@>PIS965K("1F:"P@+34X+"`R.PIR
M96%D("1F:"P@)&)U9BP@-#$["B1B=68@/7X@<R]<,"\O9SL*)'-T;7!D:7(@
M+CT@(B1S>7-[7V1E;&EM?6-A8VAE+21B=68B.PI]"F5L<V4@>PIM>2`D9&EG
M97-T(#T@979A;`I["G)E<75I<F4@1&EG97-T.CI32$$["FUY("1C='@@/2!$
M:6=E<W0Z.E-(02T^;F5W*#$I.PIO<&5N(&UY("1F:"P@(CPZ<F%W(BP@)'!R
M;V=N86UE(&]R(&1I92!Q<5M#86XG="!R96%D("(D<')O9VYA;64B.B`D(5T[
M"B1C='@M/F%D9&9I;&4H)&9H*3L*8VQO<V4H)&9H*3L*)&-T>"T^:&5X9&EG
M97-T.PI]('Q\("1M=&EM93L*)'-T;7!D:7(@+CT@(B1S>7-[7V1E;&EM?6-A
M8VAE+21D:6=E<W0B.PI]"F-L;W-E*"1F:"D["GT*96QS92!["B1%3E9[4$%2
M7T-,14%.?2`](#$["B1S=&UP9&ER("X]("(D<WES>U]D96QI;7UT96UP+20D
M(CL*?0HD14Y6>U!!4E]414U0?2`]("1S=&UP9&ER.PIM:V1I<B`D<W1M<&1I
M<BP@,#<U-3L*;&%S=#L*?0HD<&%R7W1E;7`@/2`D,2!I9B`D14Y6>U!!4E]4
M14U0?2!A;F0@)$5.5GM005)?5$5-4'T@/7X@+R@N*RDO.PI]"G-U8B!?<V%V
M95]A<R!["FUY("@D;F%M92P@)&-O;G1E;G1S+"`D;6]D92D@/2!`7SL*;7D@
M)&9U;&QN86UE(#T@(B1P87)?=&5M<"\D;F%M92(["G5N;&5S<R`H+64@)&9U
M;&QN86UE*2!["FUY("1T96UP;F%M92`]("(D9G5L;&YA;64N)"0B.PIM>2`D
M9F@["BAO<&5N("1F:"P@)SXZ<F%W)RP@)'1E;7!N86UE*0HF)B`H<')I;G0@
M)&9H("1C;VYT96YT<RD*)B8@*&-L;W-E("1F:"D*;W(@9&EE('%Q6T5R<F]R
M('=R:71I;F<@(B1T96UP;F%M92(Z("0A73L*8VAM;V0@)&UO9&4L("1T96UP
M;F%M92!I9B!D969I;F5D("1M;V1E.PIR96YA;64H)'1E;7!N86UE+"`D9G5L
M;&YA;64I(&]R('5N;&EN:R@D=&5M<&YA;64I.PI]"G)E='5R;B`D9G5L;&YA
M;64["GT*<W5B(%]S971?<')O9VYA;64@>PII9B`H9&5F:6YE9"`D14Y6>U!!
M4E]04D]'3D%-17T@86YD("1%3E9[4$%27U!23T=.04U%?2`]?B`O*"XK*2\I
M('L*)'!R;V=N86UE(#T@)#$["GT*)'!R;V=N86UE('Q\/2`D,#L*:68@*"1%
M3E9[4$%27U1%35!](&%N9"!I;F1E>"@D<')O9VYA;64L("1%3E9[4$%27U1%
M35!]*2`^/2`P*2!["B1P<F]G;F%M92`]('-U8G-T<B@D<')O9VYA;64L(')I
M;F1E>"@D<')O9VYA;64L("1S>7-[7V1E;&EM?2D@*R`Q*3L*?0II9B`H(21%
M3E9[4$%27U!23T=.04U%?2!O<B!I;F1E>"@D<')O9VYA;64L("1S>7-[7V1E
M;&EM?2D@/CT@,"D@>PII9B`H;W!E;B!M>2`D9F@L("<\)RP@)'!R;V=N86UE
M*2!["G)E='5R;B!I9B`M<R`D9F@["GT*:68@*"US("(D<')O9VYA;64D<WES
M>U]E>&5](BD@>PHD<')O9VYA;64@+CT@)'-Y<WM?97AE?3L*<F5T=7)N.PI]
M"GT*9F]R96%C:"!M>2`D9&ER("AS<&QI="`O7%$D<WES>W!A=&A?<V5P?5Q%
M+RP@)$5.5GM0051(?2D@>PIN97AT(&EF(&5X:7-T<R`D14Y6>U!!4E]414U0
M?2!A;F0@)&1I<B!E<2`D14Y6>U!!4E]414U0?3L*)&1I<B`]?B!S+UQ1)'-Y
M<WM?9&5L:6U]7$4D+R\["B@H)'!R;V=N86UE(#T@(B1D:7(D<WES>U]D96QI
M;7TD<')O9VYA;64D<WES>U]E>&5](BDL(&QA<W0I"FEF("US("(D9&ER)'-Y
M<WM?9&5L:6U])'!R;V=N86UE)'-Y<WM?97AE?2(["B@H)'!R;V=N86UE(#T@
M(B1D:7(D<WES>U]D96QI;7TD<')O9VYA;64B*2P@;&%S="D*:68@+7,@(B1D
M:7(D<WES>U]D96QI;7TD<')O9VYA;64B.PI]"GT*<W5B(%]F:7A?<')O9VYA
M;64@>PHD,"`]("1P<F]G;F%M92!\?#T@)$5.5GM005)?4%)/1TY!345].PII
M9B`H:6YD97@H)'!R;V=N86UE+"`D<WES>U]D96QI;7TI(#P@,"D@>PHD<')O
M9VYA;64@/2`B+B1S>7-[7V1E;&EM?21P<F]G;F%M92(["GT*;7D@)'!W9"`]
M("AD969I;F5D("9#=V0Z.F=E=&-W9"D@/R!#=V0Z.F=E=&-W9"@I"CH@*"AD
M969I;F5D("97:6XS,CHZ1V5T0W=D*2`_(%=I;C,R.CI'971#=V0H*2`Z(&!P
M=V1@*3L*8VAO;7`H)'!W9"D["B1P<F]G;F%M92`]?B!S+UXH/SU<+EPN/UQ1
M)'-Y<WM?9&5L:6U]7$4I+R1P=V0D<WES>U]D96QI;7TO.PHD14Y6>U!!4E]0
M4D]'3D%-17T@/2`D<')O9VYA;64["GT*<W5B(%]P87)?:6YI=%]E;G8@>PII
M9B`H("1%3E9[4$%27TE.251)04Q)6D5$?2LK(#T](#$@*2!["G)E='5R;CL*
M?2!E;'-E('L*)$5.5GM005)?24Y)5$E!3$E:141](#T@,CL*?0IF;W(@*'%W
M*"!34$%73D5$(%1%35`@0TQ%04X@1$5"54<@0T%#2$4@4%)/1TY!344@*2`I
M('L*9&5L971E("1%3E9[(E!!4E\D7R)].PI]"F9O<B`H<7<O(%1-4$1)4B!4
M14U0($-,14%.($1%0E5'("\I('L*)$5.5GLB4$%27R1?(GT@/2`D14Y6>R)0
M05)?1TQ/0D%,7R1?(GT@:68@97AI<W1S("1%3E9[(E!!4E]'3$]"04Q?)%\B
M?3L*?0II9B`H)$5.5GM005)?5$5-4'TI('L*9&5L971E("1%3E9[4$%27T-,
M14%.?3L*?0I]"G-U8B!O=71S('L*<F5T=7)N(&EF("1Q=6EE=#L*:68@*"1L
M;V=F:"D@>PIP<FEN="`D;&]G9F@@(D!?7&XB.PI]"F5L<V4@>PIP<FEN="`B
M0%]<;B(["GT*?0IS=6(@:6YI=%]I;F,@>PIR97%U:7)E($-O;F9I9SL*<'5S
M:"!`24Y#+"!G<F5P(&1E9FEN960L(&UA<"`D0V]N9FEG.CI#;VYF:6=[)%]]
M+"!Q=R@*87)C:&QI8F5X<"!P<FEV;&EB97AP('-I=&5A<F-H97AP('-I=&5L
M:6)E>'`*=F5N9&]R87)C:&5X<"!V96YD;W)L:6)E>'`**3L*?0IP86-K86=E
M(&UA:6X["G)E<75I<F4@4$%2.PIU;G-H:69T($!)3D,L(%PF4$%2.CIF:6YD
M7W!A<CL*4$%2+3YI;7!O<G0H0'!A<E]A<F=S*3L*9&EE('%Q6W!A<BYP;#H@
M0V%N)W0@;W!E;B!P97)L('-C<FEP="`B)'!R;V=N86UE(CH@3F\@<W5C:"!F
M:6QE(&]R(&1I<F5C=&]R>5QN70IU;FQE<W,@+64@)'!R;V=N86UE.PID;R`D
M<')O9VYA;64["D-/4D4Z.F5X:70H)#$I(&EF("@D0"`]?B]>7U1+7T58251?
M7"@H7&0K*5PI+RD["F1I92`D0"!I9B`D0#L*?3L*)#HZ7U]%4E)/4B`]("1`
M(&EF("1`.PI]"D-/4D4Z.F5X:70H)#$I(&EF("@D.CI?7T524D]2(#U^+UY?
M5$M?15A)5%]<*"A<9"LI7"DO*3L*9&EE("0Z.E]?15)23U(@:68@)#HZ7U]%
M4E)/4CL*,3L*7U]%3D1?7PH```````````!P3P``````````````````6"1!
M``$``````````````&`D00`!``````````````!X1T```0``````````````
M<$=```$``````````````.@B00`!```````````````01T```0``````````
M````0"-!``$``````````````$=#0SH@*$=.52D@,3$N,BXP(#(P,C$P-S(X
M("A&961O<F$@0WEG=VEN(#$Q+C(N,"TR*0````````````````!'0T,Z("A'
M3E4I(#$Q+C,N,````````````````````$=#0SH@*$=.52D@,3$N,RXP````
M````````````````1T-#.B`H1TY5*2`Q,2XS+C````````````````````!'
M0T,Z("A'3E4I(#$Q+C(N,"`R,#(Q,#<R."`H1F5D;W)A($-Y9W=I;B`Q,2XR
M+C`M,BD`````````````````1T-#.B`H1TY5*2`Q,2XR+C`@,C`R,3`W,C@@
M*$9E9&]R82!#>6=W:6X@,3$N,BXP+3(I`````````````````$=#0SH@*$=.
M52D@,3$N,BXP(#(P,C$P-S(X("A&961O<F$@0WEG=VEN(#$Q+C(N,"TR*0``
M``````````````!'0T,Z("A'3E4I(#$Q+C(N,"`R,#(Q,#<R."`H1F5D;W)A
M($-Y9W=I;B`Q,2XR+C`M,BD`````````````````1T-#.B`H1TY5*2`Q,2XR
M+C`@,C`R,3`W,C@@*$9E9&]R82!#>6=W:6X@,3$N,BXP+3(I````````````
M`````$=#0SH@*$=.52D@,3$N,BXP(#(P,C$P-S(X("A&961O<F$@0WEG=VEN
M(#$Q+C(N,"TR*0````````````````!'0T,Z("A'3E4I(#$Q+C(N,"`R,#(Q
M,#<R."`H1F5D;W)A($-Y9W=I;B`Q,2XR+C`M,BD`````````````````1T-#
M.B`H1TY5*2`Q,2XR+C`@,C`R,3`W,C@@*$9E9&]R82!#>6=W:6X@,3$N,BXP
M+3(I`````````````````$=#0SH@*$=.52D@,3$N,BXP(#(P,C$P-S(X("A&
M961O<F$@0WEG=VEN(#$Q+C(N,"TR*0````````````````!'0T,Z("A'3E4I
M(#$Q+C(N,"`R,#(Q,#<R."`H1F5D;W)A($-Y9W=I;B`Q,2XR+C`M,BD`````
M````````````1T-#.B`H1TY5*2`Q,2XS+C````````````````````!'0T,Z
M("A'3E4I(#$Q+C,N,````````````````````````````````0```%@D`0"@
MKP``0````&`D`0"PKP``0````.@B`0#@KP``0````$`C`0``L```0```````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````(````9````',```!R.``!24T13?O/$,>*^U0#6W0`XUM-1]@$`````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M`````````````!```$00````X```4!```&40```(X```<!```'$0```,X```
M@!```&,1```0X```<!$``-42```DX```X!(``&`3```\X```8!,``#L4``!(
MX```0!0``&H6``!<X```<!8``.(6``!TX```\!8``"07``!\X```,!<``#,7
M``"$X```0!<``#,9``"(X```0!D``&49``"@X```<!D``%P:``"DX```8!H`
M`%(?``"LX```8!\``,L?``#$X```T!\``.T?``#,X```\!\``.8@``#4X```
M\"```*$H``#DX```L"@``"XK``#\X```,"L``#4L```0X0``0"P``+-"```P
MX0``P$(``!9#``!(X0``($,``*A$``!0X0``L$0``--&``!DX0``8$H``')*
M``!TX0``@$H``(%*``!\X0``D$H``)%*``"`X0``H$H``*%*``"$X0``L$H`
M`+%*``"(X0``P$H``'!.``",X0``<$X``(E.``"8X0``D$X``$Q/``"<X0``
M4$\``%5/``"DX0``8$\``&9/``"HX0``<$\``'%/``"LX0``P$\``/!/``"P
MX0``$%```#E2```@X0``0%(``$52``"\X0``````````````````````````
M```````````````````````````````````!!`$`!$(```$````!`````0P'
M``Q""#`'8`9P!5`$P`+0```!$0D`$0$6``HP"6`(<`=0!L`$T`+@```!"@0`
M"@$7``,P`L`!$P<`$P$&`@LP"F`)<`A0`L````$0"0`08@PP"V`*<`E0",`&
MT`3@`O````$%`@`%4@$P`00!``1"```!`````1D*`!D!&P(1,!!@#W`.4`W`
M"]`)X`+P`0````$&`@`&,@+``1`)`!!B##`+8`IP"5`(P`;0!.`"\````04"
M``52`3`!!`$`!$(```$,!@`,<@@P!V`&P`30`N`!&0H`&0$U$!$P$&`/<`Y0
M#<`+T`G@`O`!#0<`#6(),`A@!W`&P`30`N````$*!0`*8@8P!6`$P`+0```!
M"@4`"J(&,`5@!,`"T````1,*`!,!5P`,,`M@"G`)4`C`!M`$X`+P`00!``1"
M```!#`<`#$((,`=@!G`%4`3``M````$*!@`*,@8P!6`$<`-0`L`!!`$`!$(`
M``$````!`````0````$````!!@,`!D(",`%@```!`````00!``1B```!````
M`0````$````!"`,`"$($P`+0```!````````````````````````````````
M````````````````````````````````````````````````````````````
M``"GGM)B`````-(0`0`!````$0```!$````H$`$`;!`!`+`0`0"P*```T!\`
M`!!0```88```0!D``#`K```P%P``<!D``&`?``!`%P``8!H``/`@``#P'P``
ML$0``,!"```@0P``\!8``-H0`0#P$`$`"Q$!`!`1`0`:$0$`)Q$!`#,1`0!$
M$0$`4!$!`%X1`0!K$0$`>!$!`(41`0"7$0$`H1$!`*H1`0"U$0$````!``(`
M`P`$``4`!@`'``@`"0`*``L`#``-``X`#P`0`'!A<BYE>&4`6%-?26YT97)N
M86QS7U!!4E]"3T]4`&=E=%]U<V5R;F%M95]F<F]M7V=E='!W=6ED`&UA:6X`
M;7E?<&%R7W!L`'!A<E]B87-E;F%M90!P87)?8VQE86YU<`!P87)?8W5R<F5N
M=%]E>&5C`'!A<E]D:7)N86UE`'!A<E]E;G9?8VQE86X`<&%R7V9I;F1P<F]G
M`'!A<E]I;FET7V5N=@!P87)?;6MT;7!D:7(`<&%R7W-E='5P7VQI8G!A=&@`
M<VAA7V9I;F%L`'-H85]I;FET`'-H85]U<&1A=&4`>'-?:6YI=```````````
M````````````````````````````````````````````````````````````
M````````````````````4"`!````````````'"H!`+@B`0#P(0$`````````
M``"`*@$`6"0!`*@B`0```````````)0J`0`0)0$`````````````````````
M```````@)0$``````#`E`0``````/"4!``````!,)0$``````%@E`0``````
M:"4!``````!\)0$``````)`E`0``````H"4!``````"L)0$``````+PE`0``
M````R"4!``````#4)0$``````-PE`0``````Z"4!``````#\)0$``````!`F
M`0``````("8!```````L)@$``````#0F`0``````0"8!``````!()@$`````
M`%0F`0``````8"8!``````!L)@$``````'0F`0``````?"8!``````"()@$`
M`````)0F`0``````H"8!``````"L)@$``````+0F`0``````O"8!``````#(
M)@$``````-PF`0``````\"8!```````()P$``````!`G`0``````'"<!````
M```H)P$``````#0G`0``````/"<!``````!()P$``````%`G`0``````7"<!
M``````!H)P$``````'0G`0``````@"<!``````",)P$``````)@G`0``````
MI"<!`````````````````+`G`0``````P"<!``````#0)P$``````.0G`0``
M````_"<!```````4*`$``````"0H`0``````."@!``````!(*`$``````%@H
M`0``````;"@!``````"$*`$``````)0H`0``````J"@!``````"\*`$`````
M`,PH`0``````X"@!``````#P*`$```````0I`0``````%"D!```````@*0$`
M`````#`I`0`````````````````\*0$`````````````````("4!```````P
M)0$``````#PE`0``````3"4!``````!8)0$``````&@E`0``````?"4!````
M``"0)0$``````*`E`0``````K"4!``````"\)0$``````,@E`0``````U"4!
M``````#<)0$``````.@E`0``````_"4!```````0)@$``````"`F`0``````
M+"8!```````T)@$``````$`F`0``````2"8!``````!4)@$``````&`F`0``
M````;"8!``````!T)@$``````'PF`0``````B"8!``````"4)@$``````*`F
M`0``````K"8!``````"T)@$``````+PF`0``````R"8!``````#<)@$`````
M`/`F`0``````""<!```````0)P$``````!PG`0``````*"<!```````T)P$`
M`````#PG`0``````2"<!``````!0)P$``````%PG`0``````:"<!``````!T
M)P$``````(`G`0``````C"<!``````"8)P$``````*0G`0``````````````
M``"P)P$``````,`G`0``````T"<!``````#D)P$``````/PG`0``````%"@!
M```````D*`$``````#@H`0``````2"@!``````!8*`$``````&PH`0``````
MA"@!``````"4*`$``````*@H`0``````O"@!``````#,*`$``````.`H`0``
M````\"@!```````$*0$``````!0I`0``````("D!```````P*0$`````````
M````````/"D!``````````````````\`7U]C>&%?871E>&ET```6`%]?97)R
M;F\````C`%]?9V5T<F5E;G0`````+@!?7VUA:6X`````3`!?7W-P<FEN=&9?
M8VAK`$\`7U]S=&%C:U]C:&M?9F%I;```40!?7W-T86-K7V-H:U]G=6%R9`!7
M`%]?<W1R;F-P>5]C:&L`;`!?9&QL7V-R=#``<P!?:6UP=7)E7W!T<@```(,`
M86-C97-S``````(!8V%L;&]C`````$,!8VQO<V4`1`%C;&]S961I<@``?`%C
M>6=W:6Y?9&5T86-H7V1L;`!^`6-Y9W=I;E]I;G1E<FYA;````)H!9&QL7V1L
M;&-R=#````"W`65N=FER;VX```#4`65X:70``#X"9G!R:6YT9@```$H"9G)E
M90``L`)G971P:60`````NP)G971P=W5I9```R0)G971U:60`````C0-L<V5E
M:P"/`VQS=&%T`)0#;6%L;&]C`````*,#;65M8VAR`````*4#;65M8W!Y````
M`*<#;65M;6]V90```*L#;6MD:7(`[0-O<&5N``#Q`V]P96YD:7(````$!'!O
M<VEX7VUE;6%L:6=N`````"D$<'1H<F5A9%]A=&9O<FL`````7P1P=&AR96%D
M7V=E='-P96-I9FEC````MP1R96%D``"X!')E861D:7(```"Z!')E861L:6YK
M``"]!')E86QL;V,```#K!')M9&ER`(`%<W!R:6YT9@```(@%<W1A=```DP5S
M=')C:'(`````E05S=')C;7``````F@5S=')D=7``````I`5S=')L96X`````
MJ05S=')N8VUP````M05S=')S='(`````NP5S=')T;VL`````+09U;FQI;FL`
M````%P!03%]D;U]U;F1U;7```&T`4$Q?=&AR7VME>0````!K`5!E<FQ?871F
M;W)K7VQO8VL``&P!4&5R;%]A=&9O<FM?=6YL;V-K`````.4!4&5R;%]C<F]A
M:U]N;V-O;G1E>'0``'@"4&5R;%]G971?<W8```"G`E!E<FQ?9W9?9F5T8VAP
M=@```&D#4&5R;%]M9U]S970```#?`U!E<FQ?;F5W6%,`````X0-097)L7VYE
M=UA37V9L86=S``"G!5!E<FQ?<V%F97-Y<VUA;&QO8P````"&!E!E<FQ?<W9?
M<V5T<'8`CP9097)L7W-V7W-E='!V;@````"[!E!E<FQ?<WES7VEN:70S````
M`+P&4&5R;%]S>7-?=&5R;0`Y!V)O;W1?1'EN84QO861E<@```$H'<&5R;%]A
M;&QO8P````!,!W!E<FQ?8V]N<W1R=6-T`````$T'<&5R;%]D97-T<G5C=`!.
M!W!E<FQ?9G)E90!/!W!E<FQ?<&%R<V4`````4`=P97)L7W)U;@``BP)'971-
M;V1U;&5(86YD;&5!````(`$``"`!```@`0``(`$``"`!```@`0``(`$``"`!
M```@`0``(`$``"`!```@`0``(`$``"`!```@`0``(`$``"`!```@`0``(`$`
M`"`!```@`0``(`$``"`!```@`0``(`$``"`!```@`0``(`$``"`!```@`0``
M(`$``"`!```@`0``(`$``"`!```@`0``(`$``"`!```@`0``(`$``"`!```@
M`0``(`$``"`!```@`0``(`$``"`!```@`0``(`$``"`!```@`0!C>6=W:6XQ
M+F1L;``4(`$`%"`!`!0@`0`4(`$`%"`!`!0@`0`4(`$`%"`!`!0@`0`4(`$`
M%"`!`!0@`0`4(`$`%"`!`!0@`0`4(`$`%"`!`!0@`0`4(`$`%"`!`!0@`0`4
M(`$`8WEG<&5R;#5?,S(N9&QL`"@@`0!+15).14PS,BYD;&P`````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M```````````````````````````````````````````````````````$``,`
M```P``"`#@```,```(`0````\```@!@````@`0"````````````````````$
M``$```!@``"``@```'@``(`#````D```@`0```"H``"`````````````````
M```!``D$``!0`0`````````````````````!``D$``!@`0``````````````
M```````!``D$``!P`0`````````````````````!``D$``"``0``````````
M`````````0```,`!`(#8``"````````````````````!``D$``"0`0``````
M```````````````!``$````(`0"````````````````````!``D$``"@`0``
M```````````````````!``$````X`0"````````````````````!``D$``"P
M`0``T#$!`&@%`````````````#@W`0"H"`````````````#@/P$`:`4`````
M````````2$4!`*@(`````````````/!-`0`^```````````````P3@$`6`(`
M````````````B%`!`-H%``````````````8`5P!)`$X`10!8`$4````H````
M$````"`````!``@````````!```````````````!`````0``!@8&``0$!``"
M`@(``````$)"0@"=G9T`;FYN`(&!@0"QL;$`____`'-S<P"?GY\`6%A8`+*R
ML@#]_?T`:VMK`/S\_`#7U]<`%A86`-K:V@"6EI8`CHZ.`$9&1@#+R\L`"PL+
M`*"@H`#^_OX`R<G)`,'!P0`#`P,`$1$1`).3DP"SL[,`[N[N`/3T]`#CX^,`
MY>7E`**BH@"ZNKH`KZ^O``<'!P!Q<7$`M;6U`/7U]0")B8D`BHJ*`#0T-`#H
MZ.@`-34U`%I:6@"WM[<`V-C8`-#0T``='1T`#`P,`(^/CP"^OKX`;6UM``D)
M"0";FYL`+R\O`,W-S0#9V=D`&1D9`$U-30!$1$0`4%!0`#P\/`!D9&0`PL+"
M`#(R,@`("`@`#0T-``\/#P``````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````````````````````````````````%!04%!04%!04%!04%!00$!`35`
M(P5!,4-!041%+@$!`0$!`0$%`3P7%P$!$P$!`0$!`0$!%0$!)A4!`0\!`0$!
M`0$!`1,!$0$%-0$F`0$!`0$!`0$B$A`Q#P4!(`$!`0$!`0$!$`H@(A45+B$N
M`0$!`0$?("$0(A`C(R4F)P$!`0$E$!`0$!`0$!`0%T4!`0%!$!`0$!`0$!`0
M$!`7`0$!#Q`0%1`0$!`0$!`0`0$!`1(0$!,?$!`0$!`0!P$!`0$@$!`0`0\0
M$!`0$0$!`0$!!R`0"@$!!1`0$#$!`0$!`0$!`0$!`0%!!0\!`0$!`0$!`0$!
M`0$!`0$!`0$!`0&``0``P`$``.`#``#P`P``\`,``/`#``#@`P``P`$``(`!
M``"``0``@`,``(`#````!P````<``(X/``#_'P``*````"````!``````0`(
M````````!````````````````0````$````````!`0$``@("``,#`P`$!`0`
M!04%``8&!@`'!P<`"`@(``D)"0`*"@H`"PL+``P,#``-#0T`#@X.``\/#P`0
M$!``$1$1`!(2$@`3$Q,`%!04`!45%0`6%A8`%Q<7`!@8&``9&1D`&AH:`!L;
M&P`<'!P`'1T=`!X>'@`?'Q\`("`@`"$A(0`B(B(`(R,C`"0D)``E)24`)B8F
M`"<G)P`H*"@`*2DI`"HJ*@`K*RL`+"PL`"TM+0`N+BX`+R\O`#`P,``Q,3$`
M,C(R`#,S,P`T-#0`-34U`#8V-@`W-S<`.#@X`#DY.0`Z.CH`.SL[`#P\/``]
M/3T`/CX^`#\_/P!`0$``04%!`$)"0@!#0T,`1$1$`$5%10!&1D8`1T='`$A(
M2`!)24D`2DI*`$M+2P!,3$P`34U-`$Y.3@!/3T\`4%!0`%%140!24E(`4U-3
M`%145`!5554`5E96`%=75P!86%@`65E9`%I:6@!;6UL`7%Q<`%U=70!>7EX`
M7U]?`&!@8`!A86$`8F)B`&-C8P!D9&0`965E`&9F9@!G9V<`:&AH`&EI:0!J
M:FH`:VMK`&QL;`!M;6T`;FYN`&]O;P!P<'``<7%Q`')R<@!S<W,`='1T`'5U
M=0!V=G8`=W=W`'AX>`!Y>7D`>GIZ`'M[>P!\?'P`?7U]`'Y^?@!_?W\`@("`
M`(&!@0""@H(`@X.#`(2$A`"%A84`AH:&`(>'AP"(B(@`B8F)`(J*B@"+BXL`
MC(R,`(V-C0".CHX`CX^/`)"0D`"1D9$`DI*2`).3DP"4E)0`E965`):6E@"7
MEY<`F)B8`)F9F0":FIH`FYN;`)R<G`"=G9T`GIZ>`)^?GP"@H*``H:&A`**B
MH@"CHZ,`I*2D`*6EI0"FIJ8`IZ>G`*BHJ`"IJ:D`JJJJ`*NKJP"LK*P`K:VM
M`*ZNK@"OKZ\`L+"P`+&QL0"RLK(`L[.S`+2TM`"UM;4`MK:V`+>WMP"XN+@`
MN;FY`+JZN@"[N[L`O+R\`+V]O0"^OKX`O[^_`,#`P`#!P<$`PL+"`,/#PP#$
MQ,0`Q<7%`,;&Q@#'Q\<`R,C(`,G)R0#*RLH`R\O+`,S,S`#-S<T`SL[.`,_/
MSP#0T-``T='1`-+2T@#3T],`U-34`-75U0#6UM8`U]?7`-C8V`#9V=D`VMK:
M`-O;VP#<W-P`W=W=`-[>W@#?W]\`X.#@`.'AX0#BXN(`X^/C`.3DY`#EY>4`
MYN;F`.?GYP#HZ.@`Z>GI`.KJZ@#KZ^L`[.SL`.WM[0#N[NX`[^_O`/#P\`#Q
M\?$`\O+R`//S\P#T]/0`]?7U`/;V]@#W]_<`^/CX`/GY^0#Z^OH`^_O[`/S\
M_`#]_?T`_O[^`/___P`T-#0\/#P\/#P\/#P\/#P\/#P\/#P\/#PL#`0$!`0$
M!%RTU-34U-34U-34U-S4U-34U-34U-34U+0T400$!`0$9,SL]/3T_/3\]/3T
M_/3\]/ST_/S\]/3LQ#11400$!`1DQ.3L]/3T]/3T]/3T]/3T]/3T]/3T[.R\
M-%%1400$!&3$Y.SL[.3L[.SDY.3D[.SD[.3L[.SLY+PT45%1400$9+SD[.RL
MC*S4Q)24G)2,C)24E)R<G.S<O#1145%1401DO-SD[.3$492L9.2D3#0L7,3,
M4;34Y-R\-%%145%1462\W.3D[.3D481TY(QLI)QLC+QTK-3DW+PT!`0$!`0$
M9+S<Y.3LY.2\48S4U(RLS'3,W*11Q.3<O#0$!`0$!`1DO-SDY.3DY-R,?)RT
M?(2<;,SDO'3$Y-RT-`0$!`0$!%R\W-SDY.3DU)RD7)2,S#R49,R\C%'<U+0T
M!`0$!`0$9+34W-S<Y-R,C*PLO+3,6L1DG&R4I-S4M#0$!`0$!`1<M-3<W.3<
MW'24U$RD](P\U'24=)Q,W,RT-`0$!`0$!&2TU-S<W.3<=%'T=(3LS&3D?&RD
MM&RDU+0T!`0$!`0$9+3,W-S,?(R$M/S,Q/STM/R\9,RLA%&\K#0$!`0$!`1<
MM,S4S(RDW-SL_/S\_/ST_.RT[*Q\E%&L-`0$!`0$!%RLS,QLG/3\_/S\_/S\
M_/S\_/S\Q)2,4:PT!`0$!`0$7*S$O'SD_/S\_/S\_/S\_/S\_/SLS*Q1K#0$
M!`0$!`1<K+1,Q/S\_.ST_/S\_/S\_/S\_/2\4<2L-`0$!`0$!%RLM$S4_/ST
MM-S\_/S\_/S\_/S\S%',Q*0T!`0$!`0$7*S$I%'\_/RTG/3\_/S\_/S\_/24
M4<S$I"P$!`0$!`1<I*11A.S\_,QDM.S\_/S\_/S\U&RLS+RD-`0$!`0$!%RD
M9'2\]/S\Q%&$G.3\_/S\_.2,4<S,Q*0T!`0$!`0$7*1LC.3T_/R<4<Q1E.S\
M_/S\O'3,S,2\G"P$!`0$!`1<I*QD7&2DI%&LS,1TE.3T_.11O,S,Q+R<-`0$
M!`0$!%RDM+R\C%%1Q,3,S*Q1?'R4;'R\S,3$O)PL!`0$!`0$7)R\Q,3$Q,3$
MQ,3$Q,1145%1O,3$Q,2TG#0$!`0$!`14G+3$Q,3$Q,3$Q,3$Q,3$Q,3$Q,3$
MO+2<+`0$!`0$!%2<O,3$O+R\O+R\O+R\O+R\O+R\O+R\M)PL!`0$!`0$7)RT
MQ+RTM+2TM+2TM+2TM+2TM+2TM+2LG"P$!`0$!`14A)2<G)24E)24E)24E)24
ME)24E)24E)2$)`0$!`0$!$147%Q<7%Q<7%Q<7%Q<7%Q<7%Q<7%Q<7%0T!`0$
M!`0$````/P```!\````/````!P````,````!`````````#\````_````/P``
M`#\````_````/P```#\````_````/P```#\````_````/P```#\````_````
M/P```#\````_````/P```#\````_````/P```#\````_````/P```#\H````
M$````"`````!``@````````!```````````````!`````0``!@8&``0$!``"
M`@(``````$)"0@"=G9T`;FYN`(&!@0"QL;$`____`'-S<P"?GY\`6%A8`+*R
ML@#]_?T`:VMK`/S\_`#7U]<`%A86`-K:V@"6EI8`CHZ.`$9&1@#+R\L`"PL+
M`*"@H`#^_OX`R<G)`,'!P0`#`P,`$1$1`).3DP"SL[,`[N[N`/3T]`#CX^,`
MY>7E`**BH@"ZNKH`KZ^O``<'!P!Q<7$`M;6U`/7U]0")B8D`BHJ*`#0T-`#H
MZ.@`-34U`%I:6@"WM[<`V-C8`-#0T``='1T`#`P,`(^/CP"^OKX`;6UM``D)
M"0";FYL`+R\O`,W-S0#9V=D`&1D9`$U-30!$1$0`4%!0`#P\/`!D9&0`PL+"
M`#(R,@`("`@`#0T-``\/#P``````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````````````````````````````````%!04%!04%!04%!04%!00$!`35`
M(P5!,4-!041%+@$!`0$!`0$%`3P7%P$!$P$!`0$!`0$!%0$!)A4!`0\!`0$!
M`0$!`1,!$0$%-0$F`0$!`0$!`0$B$A`Q#P4!(`$!`0$!`0$!$`H@(A45+B$N
M`0$!`0$?("$0(A`C(R4F)P$!`0$E$!`0$!`0$!`0%T4!`0%!$!`0$!`0$!`0
M$!`7`0$!#Q`0%1`0$!`0$!`0`0$!`1(0$!,?$!`0$!`0!P$!`0$@$!`0`0\0
M$!`0$0$!`0$!!R`0"@$!!1`0$#$!`0$!`0$!`0$!`0%!!0\!`0$!`0$!`0$!
M`0$!`0$!`0$!`0&``0``P`$``.`#``#P`P``\`,``/`#``#@`P``P`$``(`!
M``"``0``@`,``(`#````!P````<``(X/``#_'P``*````"````!``````0`(
M````````!````````````````0````$````````!`0$``@("``,#`P`$!`0`
M!04%``8&!@`'!P<`"`@(``D)"0`*"@H`"PL+``P,#``-#0T`#@X.``\/#P`0
M$!``$1$1`!(2$@`3$Q,`%!04`!45%0`6%A8`%Q<7`!@8&``9&1D`&AH:`!L;
M&P`<'!P`'1T=`!X>'@`?'Q\`("`@`"$A(0`B(B(`(R,C`"0D)``E)24`)B8F
M`"<G)P`H*"@`*2DI`"HJ*@`K*RL`+"PL`"TM+0`N+BX`+R\O`#`P,``Q,3$`
M,C(R`#,S,P`T-#0`-34U`#8V-@`W-S<`.#@X`#DY.0`Z.CH`.SL[`#P\/``]
M/3T`/CX^`#\_/P!`0$``04%!`$)"0@!#0T,`1$1$`$5%10!&1D8`1T='`$A(
M2`!)24D`2DI*`$M+2P!,3$P`34U-`$Y.3@!/3T\`4%!0`%%140!24E(`4U-3
M`%145`!5554`5E96`%=75P!86%@`65E9`%I:6@!;6UL`7%Q<`%U=70!>7EX`
M7U]?`&!@8`!A86$`8F)B`&-C8P!D9&0`965E`&9F9@!G9V<`:&AH`&EI:0!J
M:FH`:VMK`&QL;`!M;6T`;FYN`&]O;P!P<'``<7%Q`')R<@!S<W,`='1T`'5U
M=0!V=G8`=W=W`'AX>`!Y>7D`>GIZ`'M[>P!\?'P`?7U]`'Y^?@!_?W\`@("`
M`(&!@0""@H(`@X.#`(2$A`"%A84`AH:&`(>'AP"(B(@`B8F)`(J*B@"+BXL`
MC(R,`(V-C0".CHX`CX^/`)"0D`"1D9$`DI*2`).3DP"4E)0`E965`):6E@"7
MEY<`F)B8`)F9F0":FIH`FYN;`)R<G`"=G9T`GIZ>`)^?GP"@H*``H:&A`**B
MH@"CHZ,`I*2D`*6EI0"FIJ8`IZ>G`*BHJ`"IJ:D`JJJJ`*NKJP"LK*P`K:VM
M`*ZNK@"OKZ\`L+"P`+&QL0"RLK(`L[.S`+2TM`"UM;4`MK:V`+>WMP"XN+@`
MN;FY`+JZN@"[N[L`O+R\`+V]O0"^OKX`O[^_`,#`P`#!P<$`PL+"`,/#PP#$
MQ,0`Q<7%`,;&Q@#'Q\<`R,C(`,G)R0#*RLH`R\O+`,S,S`#-S<T`SL[.`,_/
MSP#0T-``T='1`-+2T@#3T],`U-34`-75U0#6UM8`U]?7`-C8V`#9V=D`VMK:
M`-O;VP#<W-P`W=W=`-[>W@#?W]\`X.#@`.'AX0#BXN(`X^/C`.3DY`#EY>4`
MYN;F`.?GYP#HZ.@`Z>GI`.KJZ@#KZ^L`[.SL`.WM[0#N[NX`[^_O`/#P\`#Q
M\?$`\O+R`//S\P#T]/0`]?7U`/;V]@#W]_<`^/CX`/GY^0#Z^OH`^_O[`/S\
M_`#]_?T`_O[^`/___P`T-#0\/#P\/#P\/#P\/#P\/#P\/#P\/#PL#`0$!`0$
M!%RTU-34U-34U-34U-S4U-34U-34U-34U+0T400$!`0$9,SL]/3T_/3\]/3T
M_/3\]/ST_/S\]/3LQ#11400$!`1DQ.3L]/3T]/3T]/3T]/3T]/3T]/3T[.R\
M-%%1400$!&3$Y.SL[.3L[.SDY.3D[.SD[.3L[.SLY+PT45%1400$9+SD[.RL
MC*S4Q)24G)2,C)24E)R<G.S<O#1145%1401DO-SD[.3$492L9.2D3#0L7,3,
M4;34Y-R\-%%145%1462\W.3D[.3D481TY(QLI)QLC+QTK-3DW+PT!`0$!`0$
M9+S<Y.3LY.2\48S4U(RLS'3,W*11Q.3<O#0$!`0$!`1DO-SDY.3DY-R,?)RT
M?(2<;,SDO'3$Y-RT-`0$!`0$!%R\W-SDY.3DU)RD7)2,S#R49,R\C%'<U+0T
M!`0$!`0$9+34W-S<Y-R,C*PLO+3,6L1DG&R4I-S4M#0$!`0$!`1<M-3<W.3<
MW'24U$RD](P\U'24=)Q,W,RT-`0$!`0$!&2TU-S<W.3<=%'T=(3LS&3D?&RD
MM&RDU+0T!`0$!`0$9+3,W-S,?(R$M/S,Q/STM/R\9,RLA%&\K#0$!`0$!`1<
MM,S4S(RDW-SL_/S\_/ST_.RT[*Q\E%&L-`0$!`0$!%RLS,QLG/3\_/S\_/S\
M_/S\_/S\Q)2,4:PT!`0$!`0$7*S$O'SD_/S\_/S\_/S\_/S\_/SLS*Q1K#0$
M!`0$!`1<K+1,Q/S\_.ST_/S\_/S\_/S\_/2\4<2L-`0$!`0$!%RLM$S4_/ST
MM-S\_/S\_/S\_/S\S%',Q*0T!`0$!`0$7*S$I%'\_/RTG/3\_/S\_/S\_/24
M4<S$I"P$!`0$!`1<I*11A.S\_,QDM.S\_/S\_/S\U&RLS+RD-`0$!`0$!%RD
M9'2\]/S\Q%&$G.3\_/S\_.2,4<S,Q*0T!`0$!`0$7*1LC.3T_/R<4<Q1E.S\
M_/S\O'3,S,2\G"P$!`0$!`1<I*QD7&2DI%&LS,1TE.3T_.11O,S,Q+R<-`0$
M!`0$!%RDM+R\C%%1Q,3,S*Q1?'R4;'R\S,3$O)PL!`0$!`0$7)R\Q,3$Q,3$
MQ,3$Q,1145%1O,3$Q,2TG#0$!`0$!`14G+3$Q,3$Q,3$Q,3$Q,3$Q,3$Q,3$
MO+2<+`0$!`0$!%2<O,3$O+R\O+R\O+R\O+R\O+R\O+R\M)PL!`0$!`0$7)RT
MQ+RTM+2TM+2TM+2TM+2TM+2TM+2LG"P$!`0$!`14A)2<G)24E)24E)24E)24
ME)24E)24E)2$)`0$!`0$!$147%Q<7%Q<7%Q<7%Q<7%Q<7%Q<7%Q<7%0T!`0$
M!`0$````/P```!\````/````!P````,````!`````````#\````_````/P``
M`#\````_````/P```#\````_````/P```#\````_````/P```#\````_````
M/P```#\````_````/P```#\````_````/P```#\````_````/P```#\```$`
M!``0$````0`(`&@%```!`"`@```!``@`J`@```(`$!````$`"`!H!0```P`@
M(````0`(`*@(```$````6`(T````5@!3`%\`5@!%`%(`4P!)`$\`3@!?`$D`
M3@!&`$\``````+T$[_X```$``````````````````````#\`````````!``$
M``$```````````````````"X`0```0!3`'0`<@!I`&X`9P!&`&D`;`!E`$D`
M;@!F`&\```"4`0```0`P`#``,``P`#``-`!"`#`````D``(``0!#`&\`;0!P
M`&$`;@!Y`$X`80!M`&4``````"`````L``(``0!&`&D`;`!E`$0`90!S`&,`
M<@!I`'``=`!I`&\`;@``````(````#``"``!`$8`:0!L`&4`5@!E`'(`<P!I
M`&\`;@``````,``N`#``+@`P`"X`,````"0``@`!`$D`;@!T`&4`<@!N`&$`
M;`!.`&$`;0!E````(````"@``@`!`$P`90!G`&$`;`!#`&\`<`!Y`'(`:0!G
M`&@`=````"`````L``(``0!,`&4`9P!A`&P`5`!R`&$`9`!E`&T`80!R`&L`
M<P``````(````"P``@`!`$\`<@!I`&<`:0!N`&$`;`!&`&D`;`!E`&X`80!M
M`&4````@````)``"``$`4`!R`&\`9`!U`&,`=`!.`&$`;0!E```````@````
M-``(``$`4`!R`&\`9`!U`&,`=`!6`&4`<@!S`&D`;P!N````,``N`#``+@`P
M`"X`,````$0````!`%8`80!R`$8`:0!L`&4`20!N`&8`;P``````)``$````
M5`!R`&$`;@!S`&P`80!T`&D`;P!N`````````+`$/#]X;6P@=F5R<VEO;CTB
M,2XP(B!E;F-O9&EN9STB551&+3@B('-T86YD86QO;F4](GEE<R(_/@T*/&%S
M<V5M8FQY('AM;&YS/2)U<FXZ<V-H96UA<RUM:6-R;W-O9G0M8V]M.F%S;2YV
M,2(@;6%N:69E<W1697)S:6]N/2(Q+C`B/@T*("`@(#QA<W-E;6)L>4ED96YT
M:71Y('9E<G-I;VX](C`N,"XP+C`B('1Y<&4](G=I;C,R(B!N86UE/2)005(M
M4&%C:V5R+D%P<&QI8V%T:6]N(B`O/@T*("`@(#QD97-C<FEP=&EO;CY005(M
M4&%C:V5R($%P<&QI8V%T:6]N/"]D97-C<FEP=&EO;CX-"B`@("`\9&5P96YD
M96YC>3X-"B`@("`@("`@/&1E<&5N9&5N=$%S<V5M8FQY/@T*("`@("`@("`@
M("`@/&%S<V5M8FQY261E;G1I='D@='EP93TB=VEN,S(B(&YA;64](DUI8W)O
M<V]F="Y7:6YD;W=S+D-O;6UO;BU#;VYT<F]L<R(@=F5R<VEO;CTB-BXP+C`N
M,"(@<'5B;&EC2V5Y5&]K96X](C8U.35B-C0Q-#1C8V8Q9&8B(&QA;F=U86=E
M/2(J(B!P<F]C97-S;W)!<F-H:71E8W1U<F4](BHB("\^#0H@("`@("`@(#PO
M9&5P96YD96YT07-S96UB;'D^#0H@("`@/"]D97!E;F1E;F-Y/@T*("`@(#QT
M<G5S=$EN9F\@>&UL;G,](G5R;CIS8VAE;6%S+6UI8W)O<V]F="UC;VTZ87-M
M+G8S(CX-"B`@("`@("`@/'-E8W5R:71Y/@T*("`@("`@("`@("`@/')E<75E
M<W1E9%!R:79I;&5G97,^#0H@("`@("`@("`@("`@("`@/')E<75E<W1E9$5X
M96-U=&EO;DQE=F5L(&QE=F5L/2)A<TEN=F]K97(B('5I06-C97-S/2)F86QS
M92(@+SX-"B`@("`@("`@("`@(#PO<F5Q=65S=&5D4')I=FEL96=E<SX-"B`@
M("`@("`@/"]S96-U<FET>3X-"B`@("`\+W1R=7-T26YF;SX-"B`@("`\8V]M
M<&%T:6)I;&ET>2!X;6QN<STB=7)N.G-C:&5M87,M;6EC<F]S;V9T+6-O;3IC
M;VUP871I8FEL:71Y+G8Q(CX-"B`@("`@("`@/&%P<&QI8V%T:6]N/@T*("`@
M("`@("`@("`@/"$M+2!7:6YD;W=S(#@N,2`M+3X-"B`@("`@("`@("`@(#QS
M=7!P;W)T961/4R!)9#TB>S%F-C<V8S<V+3@P93$M-#(S.2TY-6)B+3@S9#!F
M-F0P9&$W.'TB+SX-"B`@("`@("`@("`@(#PA+2T@5VEN9&]W<R!6:7-T82`M
M+3X-"B`@("`@("`@("`@(#QS=7!P;W)T961/4R!)9#TB>V4R,#$Q-#4W+3$U
M-#8M-#-C-2UA-69E+3`P.&1E964S9#-F,'TB+SX-"B`@("`@("`@("`@(#PA
M+2T@5VEN9&]W<R`W("TM/@T*("`@("`@("`@("`@/'-U<'!O<G1E9$]3($ED
M/2)[,S4Q,SAB.6$M-60Y-BTT9F)D+3AE,F0M83(T-#`R,C5F.3-A?2(O/@T*
M("`@("`@("`@("`@/"$M+2!7:6YD;W=S(#@@+2T^#0H@("`@("`@("`@("`\
M<W5P<&]R=&5D3U,@260](GLT83)F,CAE,RTU,V(Y+30T-#$M8F$Y8RUD-CED
M-&$T839E,SA](B\^#0H@("`@("`@("`@("`\(2TM(%=I;F1O=W,@,3`@+2T^
M#0H@("`@("`@("`@("`\<W5P<&]R=&5D3U,@260](GLX93!F-V$Q,BUB9F(S
M+31F93@M8CEA-2TT.&9D-3!A,35A.6%](B\^#0H@("`@("`@(#PO87!P;&EC
M871I;VX^#0H@("`@/"]C;VUP871I8FEL:71Y/@T*/"]A<W-E;6)L>3X-"@``
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````4```
M#````%BB````8```#````!B@````H```%````*"OL*_`K]"OX*_PKP"P```,
M`````*``````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
-````````````````````
